/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

/* -- Note about comments and variable names --
   In versions 1.00 up to 2.08, lcrzo was only written in French (my
   mother tongue). Since version 3.00, lcrzo is written in English, but
   most of comments and variable names are still in French.
   In future versions, I'll try to translate everything into English,
   but as you may guess, that's a quite big work...
   So, for the moment, here are some word translation, if you want to
   be able to understand what's going on in the functions :
     afficher - print     ;  ajouter - add         ;  copie - copy
     dernier - last       ;  donnee - data         ;  ecrire - write
     entete - header      ;  envoye - sent         ;  ferme - close
     fichier - file       ;  lire - read           ;  masque - mask 
     nboct - byte number  ;  ouvre, ouvrir - open  ;  paq, paquet - packet
     plage - range        ;  premier - first       ;  recu - received
     reseau - network     ;  retour - return       ;  taille - size
     tunnel - pipe

*/

#include "lcrzo_priv.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <string.h>
 #include <fcntl.h>
 #include <time.h>
 #include <errno.h>
 #include <termios.h>
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <string.h>
 #include <fcntl.h>
 #include <time.h>
 #include <errno.h>
 #include <termios.h>
#elif defined LCRZODEF_SYSTEM_Solaris
 #include <stdio.h>
 #include <stdlib.h>
 #include <unistd.h>
 #include <string.h>
 #include <fcntl.h>
 #include <time.h>
 #include <errno.h>
 #include <termios.h>
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif

/*-------------------------------------------------------------*/
int lcrzo_priv_rand_init(lcrzo_bool forceinit,
			 lcrzo_uint32 seed);
int lcrzo_priv_rand_init(lcrzo_bool forceinit,
			 lcrzo_uint32 seed)
{ static lcrzo_bool initdone=LCRZO_FALSE;
  int fd, retour;
  char pourseed[4];

  /* if we do not want to force initialization, and it was already 
     done, we return */
  if ( !forceinit && initdone )
  { return(LCRZO_ERR_OK);
  }
  initdone=LCRZO_TRUE;

  if (seed!=0)
  { srand(seed);
    return(LCRZO_ERR_OK);
  }

  /*essaie avec "/dev/random"*/  
  fd=open("/dev/random", O_RDONLY);
  if ( fd != -1 )
  { retour=lcrzo_fd_block_set(fd, LCRZO_FALSE);
    if (retour==LCRZO_ERR_OK)
    { retour=read(fd, pourseed, 4);
      if (retour==4)
      { srand( (pourseed[0]<<24) | (pourseed[1]<<16) |
	       (pourseed[2]<<8)  | pourseed[3] );
        close(fd);
	errno=0;
	return(LCRZO_ERR_OK);
      }
    }
    close(fd);
  }

  /*essaie avec "/dev/urandom"*/  
  fd=open("/dev/urandom", O_RDONLY);
  if ( fd != -1 )
  { retour=lcrzo_fd_block_set(fd, LCRZO_FALSE);
    if (retour==LCRZO_ERR_OK)
    { retour=read(fd, pourseed, 4);
      if (retour==4)
      { srand( (pourseed[0]<<24) | (pourseed[1]<<16) |
	       (pourseed[2]<<8)  | pourseed[3] );
        close(fd);
	errno=0;
	return(LCRZO_ERR_OK);
      }
    }
    close(fd);
  }

  /*se sert de l'heure et des pid*/  
  srand( time(0) ^ ( (getpid()<<16) | getppid() ) );
  errno=0;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_rand_init(lcrzo_uint32 seed)
{ return(lcrzo_priv_rand_init(LCRZO_TRUE, seed));
}

/*-------------------------------------------------------------*/
int lcrzo_uint32_rand(lcrzo_uint32 min,
		      lcrzo_uint32 max, 
		      lcrzo_uint32 *prand)
{ lcrzo_uint32 alea;

  if (min>max) return(LCRZO_ERR_PAINFHIGHERSUP);

  /* eventually init random generator */
  lcrzo_er(lcrzo_priv_rand_init(LCRZO_FALSE, 0));

  /*genere l'alea demande*/
  alea = (rand()<<16) | rand();
  if (prand!=NULL)
  { if ((max-min)==0xFFFFFFFFU) *prand=alea;
    else *prand= min + alea%(max-min+1);
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_int32_rand(lcrzo_int32 min,
		     lcrzo_int32 max, 
		     lcrzo_int32 *pnumber)
{ lcrzo_int32 alea;

  if (min>max) return(LCRZO_ERR_PAINFHIGHERSUP);

  /* eventually init random generator */
  lcrzo_er(lcrzo_priv_rand_init(LCRZO_FALSE, 0));

  /*genere l'alea demande*/
  alea = (rand()<<16) | rand();
  if (pnumber!=NULL)
  { if ((max-min)==0xFFFFFFFFU) *pnumber=alea;
    else *pnumber= min + alea%(max-min+1);
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_uint16_rand(lcrzo_uint16 min,
		      lcrzo_uint16 max, 
		      lcrzo_uint16 *pnumber)
{ lcrzo_uint32 alea;

  lcrzo_er(lcrzo_uint32_rand(min, max, &alea));
  if (pnumber!=NULL) *pnumber=(lcrzo_uint16)alea;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_int16_rand(lcrzo_int16 min,
		     lcrzo_int16 max, 
		     lcrzo_int16 *pnumber)
{ lcrzo_int32 alea;

  lcrzo_er(lcrzo_int32_rand(min, max, &alea));
  if (pnumber!=NULL) *pnumber=(lcrzo_int16)alea;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_uint8_rand(lcrzo_uint8 min,
		     lcrzo_uint8 max, 
		     lcrzo_uint8 *pnumber)
{ lcrzo_uint32 alea;

  lcrzo_er(lcrzo_uint32_rand(min, max, &alea));
  if (pnumber!=NULL) *pnumber=(lcrzo_uint8)alea;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_int8_rand(lcrzo_int8 min,
		    lcrzo_int8 max, 
		    lcrzo_int8 *pnumber)
{ lcrzo_int32 alea;

  lcrzo_er(lcrzo_int32_rand(min, max, &alea));
  if (pnumber!=NULL) *pnumber=(lcrzo_int8)alea;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_stdin_flush(void)
{ int old_fc, new_fc, c;

  /*le getc n'est pas bloquant*/
  if ( (old_fc=fcntl(0, F_GETFL, 0)) == -1 )
  { return(LCRZO_ERR_FUFCNTL);
  }
  new_fc = old_fc | O_NONBLOCK;
  if ( fcntl(0, F_SETFL, new_fc) == -1 )
  { return(LCRZO_ERR_FUFCNTL);
  }
  do { c = getchar(); } while ( c!=EOF );
  /*redefinitions des caracteristiques*/
  if ( fcntl(0, F_SETFL, old_fc) == -1 )
  { return(LCRZO_ERR_FUFCNTL); 
  }
  errno=0;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_stdin_press(const char *message)
{ struct termios old_termios, new_termios;

  /*change de mode*/
  if (tcgetattr(STDIN_FILENO, &old_termios)) return(LCRZO_ERR_FUTCGETATTR);
  new_termios=old_termios;
  new_termios.c_lflag &= (~ICANON); /*desactive le mode canonique*/
  new_termios.c_cc[VMIN] = 1;       /*passe si au moins un caractere*/
  new_termios.c_cc[VTIME] = 0;      /*attend indefiniment l'appui*/
  new_termios.c_lflag &= ~ECHO;     /*caractere invisible*/
  if (tcsetattr(STDIN_FILENO, TCSANOW, &new_termios))
    return(LCRZO_ERR_FUTCSETATTR);

  /*attend la pression d'une touche*/
  if (message!=NULL) if (message[0]!='\0') printf("%s ", message);
  lcrzo_stdin_flush();
  fgetc(stdin);
  if (message!=NULL) if (message[0]!='\0') printf("\n");

  /*remet le mode d'origine*/
  if (tcsetattr(STDIN_FILENO, TCSANOW, &old_termios))
    return(LCRZO_ERR_FUTCSETATTR);

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_stdin_char(const char *message,
		     const lcrzo_uint8 caracautorises[],
		     lcrzo_uint8 caracdefaut, lcrzo_uint8 *pcarac)
{ char prompt;
  struct termios old_termios, new_termios;
  lcrzo_uint8 carac;

  /*parameters verification*/
  if (caracautorises==NULL) return(LCRZO_ERR_PANULLPTR);
  if (!strchr((const char*)caracautorises, caracdefaut)) 
    return(LCRZO_ERR_PADEFAULTNOTINRANGE);

  /*change de mode*/
  if (tcgetattr(STDIN_FILENO, &old_termios)) return(LCRZO_ERR_FUTCGETATTR);
  new_termios=old_termios;
  new_termios.c_lflag &= (~ICANON); /*desactive le mode canonique*/
  new_termios.c_cc[VMIN] = 1; /*passe si au moins un caractere*/
  new_termios.c_cc[VTIME] = 0; /*attend indefiniment l'appui*/
  if (tcsetattr(STDIN_FILENO, TCSANOW, &new_termios))
    return(LCRZO_ERR_FUTCSETATTR);

  /*boucle pour demander a l'utilisateur*/
  prompt=':';
  while(1)
  { if (message!=NULL) if (message[0]!='\0') 
    { printf((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	     "%s (touche %s)[%c]%c ":"%s (key %s)[%c]%c ", 
	     message, caracautorises, caracdefaut, prompt);
    }
    prompt='>'; /*si erreur, utilise ce prompt*/
    /*saisie du caractere de l'utilisateur*/
    lcrzo_stdin_flush();
    carac=(lcrzo_uint8)fgetc(stdin);
    /*si rien n'a ete entre, on prend le defaut*/
    if ( carac==0x0D || carac==0x0A )
    { carac=caracdefaut;
      break;
    }
    else
    { if (message!=NULL) if (message[0]!='\0') printf("\n");
    }

    /*on verifie que le cacactere soit autorise*/
    if (strchr((const lcrzo_int8 *)caracautorises, carac)) break;
  }

  /*remet le mode d'origine*/
  if (tcsetattr(STDIN_FILENO, TCSANOW, &old_termios))
    return(LCRZO_ERR_FUTCSETATTR);

  if (pcarac!=NULL) *pcarac=carac;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_uint32_stdin(const char *message,
		       lcrzo_uint32 min, lcrzo_uint32 max, 
		       lcrzo_uint32 nbdefaut, lcrzo_uint32 *pnb)
{ char *pptr, prompt;
  lcrzo_string usertext;
  lcrzo_uint32 nb;

  /*verification des parametres*/
  if (min>max) return(LCRZO_ERR_PAINFHIGHERSUP);
  if (nbdefaut>max) return(LCRZO_ERR_PADEFAULTNOTINRANGE);
  if (nbdefaut<min) return(LCRZO_ERR_PADEFAULTNOTINRANGE);

  /*boucle pour demander a l'utilisateur*/
  prompt=':';
  while(1)
  { if (message!=NULL) if (message[0]!='\0')
    { printf((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	     "%s (entre %lu et %lu)[%lu]%c ":
	     "%s (between %lu and %lu)[%lu]%c ", 
	     message, min, max, nbdefaut, prompt);
    }
    prompt='>'; /*si erreur, utilise ce prompt*/
    /*saisie du nombre de l'utilisateur*/
    lcrzo_er(lcrzo_priv_stdin(&usertext, NULL));
    /*si rien n'a ete entre, on prend le defaut*/
    if (strlen(usertext)==0)
    { lcrzo_string_free(usertext);
      nb=nbdefaut;
      break;
    }
    /*on convertit la chaine entree en long*/
    nb=strtoul(usertext, &pptr, 10);
    if ( !errno ) /*sinon le nombre est sup a 0xffffffff*/
    { if ( *pptr == '\0' )
      { if ( (nb>=min) && (nb<=max) )
        { lcrzo_string_free(usertext);
          break;
        }
      }
    }
    lcrzo_string_free(usertext);
  }

  if (pnb!=NULL) *pnb=nb;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_int32_stdin(const char *message,
		      lcrzo_int32 min, 
		      lcrzo_int32 max, 
		      lcrzo_int32 defaultnumber, 
		      lcrzo_int32 *pnumber)
{ char *pptr, prompt;
  lcrzo_string usertext;
  lcrzo_int32 nb;

  /*verification des parametres*/
  if (min>max) return(LCRZO_ERR_PAINFHIGHERSUP);
  if (defaultnumber>max) return(LCRZO_ERR_PADEFAULTNOTINRANGE);
  if (defaultnumber<min) return(LCRZO_ERR_PADEFAULTNOTINRANGE);

  /*boucle pour demander a l'utilisateur*/
  prompt=':';
  while(1)
  { if (message!=NULL) if (message[0]!='\0')
    { printf((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	     "%s (entre %ld et %ld)[%ld]%c ":
	     "%s (between %ld and %ld)[%ld]%c ", 
	     message, min, max, defaultnumber, prompt);
    }
    prompt='>'; /*si erreur, utilise ce prompt*/
    /*saisie du nombre de l'utilisateur*/
    lcrzo_er(lcrzo_priv_stdin(&usertext, NULL));
    /*si rien n'a ete entre, on prend le defaut*/
    if (strlen(usertext)==0)
    { lcrzo_string_free(usertext);
      nb=defaultnumber;
      break;
    }
    /*on convertit la chaine entree en long*/
    nb=strtol(usertext, &pptr, 10);
    if ( !errno ) /*sinon le nombre est sup a 0xffffffff*/
    { if ( *pptr == '\0' )
      { if ( (nb>=min) && (nb<=max) )
        { lcrzo_string_free(usertext);
          break;
        }
      }
    }
    lcrzo_string_free(usertext);
  }

  if (pnumber!=NULL) *pnumber=nb;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_uint16_stdin(const char *message,
		       lcrzo_uint16 min, 
		       lcrzo_uint16 max, 
		       lcrzo_uint16 defaultnumber, 
		       lcrzo_uint16 *pnumber)
{ lcrzo_uint32 n;
  lcrzo_er(lcrzo_uint32_stdin(message,
			      min, max, defaultnumber, 
			      &n));
  if (pnumber!=NULL) *pnumber=(lcrzo_uint16)n;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_int16_stdin(const char *message,
		      lcrzo_int16 min, 
		      lcrzo_int16 max, 
		      lcrzo_int16 defaultnumber, 
		      lcrzo_int16 *pnumber)
{ lcrzo_int32 n;
  lcrzo_er(lcrzo_int32_stdin(message,
			     min, max, defaultnumber, 
			     &n));
  if (pnumber!=NULL) *pnumber=(lcrzo_int16)n;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_uint8_stdin(const char *message,
		      lcrzo_uint8 min, 
		      lcrzo_uint8 max, 
		      lcrzo_uint8 defaultnumber, 
		      lcrzo_uint8 *pnumber)
{ lcrzo_uint32 n;
  lcrzo_er(lcrzo_uint32_stdin(message,
			      min, max, defaultnumber, 
			      &n));
  if (pnumber!=NULL) *pnumber=(lcrzo_uint8)n;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_int8_stdin(const char *message,
		     lcrzo_int8 min, 
		     lcrzo_int8 max, 
		     lcrzo_int8 defaultnumber, 
		     lcrzo_int8 *pnumber)
{ lcrzo_int32 n;
  lcrzo_er(lcrzo_int32_stdin(message,
			     min, max, defaultnumber, 
			     &n));
  if (pnumber!=NULL) *pnumber=(lcrzo_int8)n;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_libinfos_fprint(LCRZOFILE *pf)
{
  union {
    struct { lcrzo_uint8 a; lcrzo_uint8 b; 
             lcrzo_uint8 c; lcrzo_uint8 d; } un;
    lcrzo_uint32 deux;
  } u;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  /*affichage des entetes*/
  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
  { fprintf(pf, "Ce programme a ete compile avec la bibliotheque reseau ");
    fprintf(pf, "lcrzo version %s.\n", LCRZODEF_VERSION);
    fprintf(pf, "Compile pour %s, %s endian, %s libpcap.\n",
	    LCRZODEF_SYS,
#if LCRZODEF_ENDIANLITTLE==1
	    "little"
#elif LCRZODEF_ENDIANBIG==1
	    "big"
#else
            #error endiannes inconnue
#endif
	    , (LCRZODEF_LIBPCAPINSTALLED==1)?"avec":"sans");
    fprintf(pf, "lcrzo is free software, covered by the GNU ");
    fprintf(pf, "Lesser General Public License\n");
    fprintf(pf, "lcrzo peut etre obtenu a :\n");
    fprintf(pf, "  http://www.laurentconstantin.com/     ");
    fprintf(pf, "(serveur principal)\n");
    fprintf(pf, "  http://go.to/laurentconstantin/       ");
    fprintf(pf, "(serveur de secours)\n");
    fprintf(pf, "  http://laurentconstantin.est-la.com/  ");
    fprintf(pf, "(serveur de secours)\n");
  }
  else
  { fprintf(pf, "This program was compiled with the network library ");
    fprintf(pf, "lcrzo version %s.\n", LCRZODEF_VERSION);
    fprintf(pf, "Compiled for %s, %s endian, %s libpcap.\n",
	    LCRZODEF_SYS,
#if LCRZODEF_ENDIANLITTLE==1
	    "little"
#elif LCRZODEF_ENDIANBIG==1
	    "big"
#else
            #error endiannes inconnue
#endif
	    , (LCRZODEF_LIBPCAPINSTALLED==1)?"with":"without");
    fprintf(pf, "lcrzo is free software, covered by the GNU Lesser General Public License\n");
    fprintf(pf, "lcrzo can be obtained at :\n");
    fprintf(pf, "  http://www.laurentconstantin.com/     (main server)\n");
    fprintf(pf, "  http://go.to/laurentconstantin/       (backup server)\n");
    fprintf(pf, "  http://laurentconstantin.est-la.com/  (backup server)\n");
  }

  /*verification des tailles*/
  if (sizeof(lcrzo_int8)!=1)
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      fprintf(pf, "Erreur : lcrzo_int8 doit avoir une taille de 8 bits\n");
    else
      fprintf(pf, "Error : lcrzo_int8 must have an 8 bit size\n");
    return(LCRZO_ERR_IEBADSIZEOF);
  }
  if (sizeof(lcrzo_uint8)!=1)
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      fprintf(pf, "Erreur : lcrzo_uint8 doit avoir une taille de 8 bits\n");
    else
      fprintf(pf, "Error : lcrzo_uint8 must have an 8 bit size\n");
    return(LCRZO_ERR_IEBADSIZEOF);
  }
  if (sizeof(lcrzo_int16)!=2)
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      fprintf(pf, "Erreur : lcrzo_int16 doit avoir une taille de 16 bits\n");
    else
      fprintf(pf, "Error : lcrzo_int16 must have an 16 bit size\n");
    return(LCRZO_ERR_IEBADSIZEOF);
  }
  if (sizeof(lcrzo_uint16)!=2)
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      fprintf(pf, "Erreur : lcrzo_uint16 doit avoir une taille de 16 bits\n");
    else
      fprintf(pf, "Error : lcrzo_uint16 must have an 16 bit size\n");
    return(LCRZO_ERR_IEBADSIZEOF);
  }
  if (sizeof(lcrzo_int32)!=4)
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      fprintf(pf, "Erreur : lcrzo_int32 doit avoir une taille de 32 bits\n");
    else
      fprintf(pf, "Error : lcrzo_int32 must have an 32 bit size\n");
    return(LCRZO_ERR_IEBADSIZEOF);
  }
  if (sizeof(lcrzo_uint32)!=4)
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      fprintf(pf, "Erreur : lcrzo_uint32 doit avoir une taille de 32 bits\n");
    else
      fprintf(pf, "Error : lcrzo_uint32 must have an 32 bit size\n");
    return(LCRZO_ERR_IEBADSIZEOF);
  }

  /*verification de l'endianness*/
  u.deux=0x12345678;
  if (LCRZODEF_ENDIANLITTLE)
  { if ( u.un.a!=0x78 || u.un.b!=0x56 || u.un.c!=0x34 || u.un.d!=0x12 )
    { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
        fprintf(pf, "Erreur : la machine n'est pas little endian\n");
      else
        fprintf(pf, "Error : this computer is not a little endian\n");
      return(LCRZO_ERR_IEBADENDIAN);
    }
  }
  else if (LCRZODEF_ENDIANBIG)
  { if ( u.un.a!=0x12 || u.un.b!=0x34 || u.un.c!=0x56 || u.un.d!=0x78 )
    { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
        fprintf(pf, "Erreur : la machine n'est pas big endian\n");
      else
        fprintf(pf, "Error : this computer is not a big endian\n");
      return(LCRZO_ERR_IEBADENDIAN);
    }
  }
  else
  { return(LCRZO_ERR_IEINTERNALERROR);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzoex_title_print(lcrzoex_infos infos)
{ 
  printf("% 6ld -", infos.refvalue);
  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
  { printf(" %s\n", infos.title_fr);
  }
  else
  { printf(" %s\n", infos.title_us);
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzoex_infos_print(lcrzoex_infos infos,
			lcrzo_conststring progname)
{ 
  printf("Reference    : %s_%06ld, ", infos.refname, infos.refvalue);
  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
  { printf("version (de cet exemple)");
  }
  else
  { printf("version (of this example)");
  }
  printf(" %d.%02d\n", infos.majorversion, infos.minorversion);
  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
  { printf("Titre        : %s\n", infos.title_fr);
    if ( strlen(infos.description_fr) )
      printf("Description  : %s\n", infos.description_fr);
    if ( strlen(infos.author) )
      printf("Auteur       : %s\n", infos.author);
    printf("Usage        : %s %s\n", progname, infos.usage_fr);
    printf("Exemple      : %s %s\n", progname, infos.usageex_fr);
  }
  else
  { printf("Titre        : %s\n", infos.title_us);
    if ( strlen(infos.description_us) )
      printf("Description  : %s\n", infos.description_us);
    if ( strlen(infos.author) )
      printf("Author       : %s\n", infos.author);
    printf("Usage        : %s %s\n", progname, infos.usage_us);
    printf("Example      : %s %s\n", progname, infos.usageex_us);
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzoex_redirect_print(lcrzoex_infos old,
			   lcrzoex_infos new)
{ if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
  { printf("** L'exemple :\n");
  }
  else
  { printf("** The example :\n");
  }
  printf("**   Reference  : %s_%06ld\n", old.refname, old.refvalue);
  printf("**   Version    : %d.%02d\n", old.majorversion, old.minorversion);
  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
  { printf("** a ete remplace par :\n");
  }
  else
  { printf("** was superseded by :\n");
  }
  printf("**   Reference  : %s_%06ld\n", new.refname, new.refvalue);
  printf("**   Version    : %d.%02d\n", new.majorversion, new.minorversion);
  
  return(LCRZO_ERR_OK);
}


