/*us    Laurent Constantin's network library (lcrzo)
                 lcrzo_misc module

  Miscellanous functions.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_misc

  Les fonctions presentes dans ce module assurent des taches diverses.
*/

/*---------------------------------------------------------------*/
/*us Name : lcrzo_rand_init
   Description :
     Seed the random generator
   Input parameter(s) :
     seed : if 0, use /dev/random (or urandom, or time_pid)
            else use seed
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_rand_init
   Description :
     Initialise le generateur aleatoire
   Parametre(s) d'entree :
     seed : si 0, utilise /dev/random (ou urandom, ou time_pid)
            sinon utilise seed
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_rand_init(lcrzo_uint32 seed);

/*us Name : lcrzo_xintxx_rand
   Description :
     Generate a random number
   Input parameter(s) :
     min : minimum value
     max : maximum value
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumber : random number generated
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xintxx_rand
   Description :
     Genere un nombre aleatoire
   Parametre(s) d'entree :
     min : valeur minimale
     max : valeur maximale
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumber : alea genere
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_uint32_rand(lcrzo_uint32 min,
		      lcrzo_uint32 max, 
		      lcrzo_uint32 *pnumber);
int lcrzo_int32_rand(lcrzo_int32 min,
		     lcrzo_int32 max, 
		     lcrzo_int32 *pnumber);
int lcrzo_uint16_rand(lcrzo_uint16 min,
		      lcrzo_uint16 max, 
		      lcrzo_uint16 *pnumber);
int lcrzo_int16_rand(lcrzo_int16 min,
		     lcrzo_int16 max, 
		     lcrzo_int16 *pnumber);
int lcrzo_uint8_rand(lcrzo_uint8 min,
		     lcrzo_uint8 max, 
		     lcrzo_uint8 *pnumber);
int lcrzo_int8_rand(lcrzo_int8 min,
		    lcrzo_int8 max, 
		    lcrzo_int8 *pnumber);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_stdin_flush
   Description :
     Flush the stdin buffer
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stdin_flush
   Description :
     Vide le buffer d'entree
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_stdin_flush(void);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_stdin_press
   Description :
     Ask user to press a key
   Input parameter(s) :
     message : message to print before
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stdin_press
   Description :
     Demande a l'utilisateur de presser une touche
   Parametre(s) d'entree :
     message : message a afficher avant
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_stdin_press(const char *message);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_stdin_char
   Description :
     Ask user to enter a character
   Input parameter(s) :
     message : message to print before
     allowedchar : string containing allowed characters
                   For example : "aAbBrR"
     defaultchar : default character
   Input/output parameter(s) :
   Output parameter(s) :
     *pchar : character choosen
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stdin_char
   Description :
     Demande a l'utilisateur d'entrer un caractere
   Parametre(s) d'entree :
     message : message a afficher avant
     allowedchar : chaine contenant les caracteres autorises
                   Par exemple : "aAbBrR"
     defaultchar : caractere par defaut
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pchar : caractere choisi
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_stdin_char(const char *message,
		     const lcrzo_uint8 allowedchar[],
		     lcrzo_uint8 defaultchar,
		     lcrzo_uint8 *pchar);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_uint32_stdin
   Description :
     Ask user to enter an integer
   Input parameter(s) :
     message : message to print before
     min : minvalue which can be entered
     max : maxvalue which can be entered
     defaultnumber : default character
   Input/output parameter(s) :
   Output parameter(s) :
     *pnum : number choosen
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_uint32_stdin
   Description :
     Demande a l'utilisateur d'entrer un entier
   Parametre(s) d'entree :
     message : message a afficher avant
     min : valeur minimale
     max : valeur maximale
     defaultnumber : entier par defaut
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumber : entier choisi
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_uint32_stdin(const char *message,
		       lcrzo_uint32 min, 
		       lcrzo_uint32 max, 
		       lcrzo_uint32 defaultnumber, 
		       lcrzo_uint32 *pnumber);
int lcrzo_int32_stdin(const char *message,
		      lcrzo_int32 min, 
		      lcrzo_int32 max, 
		      lcrzo_int32 defaultnumber, 
		      lcrzo_int32 *pnumber);
int lcrzo_uint16_stdin(const char *message,
		       lcrzo_uint16 min, 
		       lcrzo_uint16 max, 
		       lcrzo_uint16 defaultnumber, 
		       lcrzo_uint16 *pnumber);
int lcrzo_int16_stdin(const char *message,
		      lcrzo_int16 min, 
		      lcrzo_int16 max, 
		      lcrzo_int16 defaultnumber, 
		      lcrzo_int16 *pnumber);
int lcrzo_uint8_stdin(const char *message,
		      lcrzo_uint8 min, 
		      lcrzo_uint8 max, 
		      lcrzo_uint8 defaultnumber, 
		      lcrzo_uint8 *pnumber);
int lcrzo_int8_stdin(const char *message,
		     lcrzo_int8 min, 
		     lcrzo_int8 max, 
		     lcrzo_int8 defaultnumber, 
		     lcrzo_int8 *pnumber);

/*---------------------------------------------------------------*/
/*us verify if a lcrzo_bool is 0 or 1 */
/*fr verifie que la valeur d'un lcrzo_bool soit 0 ou 1 */
#define lcrzo_bool_verifbof(i) if((i)!=0&&(i)!=1)return(LCRZO_ERR_PABOOLNOT01)

/*---------------------------------------------------------------*/
/*us the folowing defines convert between host and network
   representation of "short int" et "long int" :
     lcrzo_ntohl : network to host long
     lcrzo_ntohs : network to host short
     lcrzo_htonl : host to network long
     lcrzo_htons : host to network short
*/
/*fr les defines suivants assurent la conversion entre les
   representations machine et reseau des "short int" et "long int" :
     lcrzo_ntohl : network to host long
     lcrzo_ntohs : network to host short
     lcrzo_htonl : host to network long
     lcrzo_htons : host to network short
*/
#if LCRZODEF_ENDIANLITTLE==1
 /*i386*/
 #define lcrzo_ntohl(x) \
        ((lcrzo_uint32)((((lcrzo_uint32)(x) & 0x000000ffU) << 24) | \
		      (((lcrzo_uint32)(x) & 0x0000ff00U) <<  8) | \
                      (((lcrzo_uint32)(x) & 0x00ff0000U) >>  8) | \
                      (((lcrzo_uint32)(x) & 0xff000000U) >> 24)))
 #define lcrzo_ntohs(x) \
        ((lcrzo_uint16)((((lcrzo_uint16)(x) & 0x00ff) << 8) | \
                      (((lcrzo_uint16)(x) & 0xff00) >> 8)))
 #define lcrzo_htonl(x) lcrzo_ntohl(x)
 #define lcrzo_htons(x) lcrzo_ntohs(x)
#elif LCRZODEF_ENDIANBIG==1
 /*68000*/
 #define lcrzo_ntohl(x)        (x)
 #define lcrzo_ntohs(x)        ((lcrzo_uint16)(x))
 #define lcrzo_htonl(x)        (x)
 #define lcrzo_htons(x)        ((lcrzo_uint16)(x))
#else
 #error  "Complete Endian"
 #error  "Completer le cas de Endian"
#endif

/*---------------------------------------------------------------*/
/*us Name : lcrzo_libinfos_fprint
   Description :
     Print compilation parameters.
   Input parameter(s) :
     *pf : FILE where the function prints
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_libinfos_fprint
   Description :
     Affiche les conditions de compilation de la librarie
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
/**/
int lcrzo_libinfos_fprint(LCRZOFILE *pf);
#define lcrzo_libinfos_print() lcrzo_libinfos_fprint(stdout)

/*---------------------------------------------------------------*/
/*us the folowing type and functions deal with examples.
   You might want to use them if you want to create your own
   examples (cf lcrzoex.c).*/
/*fr les types et fonctions suivantes gerent les exemples.
   Vous pouvez les utiliser si vous creez vos propres exemples
   (cf lcrzoex.c) */
typedef struct
{ lcrzo_conststring refname;
  lcrzo_int32       refvalue;
  lcrzo_uint8       majorversion;
  lcrzo_uint8       minorversion;
  lcrzo_conststring title_us;
  lcrzo_conststring title_fr;
  lcrzo_conststring description_us;
  lcrzo_conststring description_fr;
  lcrzo_conststring usage_us;
  lcrzo_conststring usage_fr;
  lcrzo_conststring usageex_us;
  lcrzo_conststring usageex_fr;
  lcrzo_conststring author;
} lcrzoex_infos;
/*us print title of an example */
/*fr affiche le titre d'un exemple */
int lcrzoex_title_print(lcrzoex_infos infos);
/*us print information about an example */
/*fr affiche les informations sur un exemple */
int lcrzoex_infos_print(lcrzoex_infos infos,
			lcrzo_conststring progname);
/*us redirect an old example */
/*fr redirige un ancien exemple, vers un nouvel exemple*/
int lcrzoex_redirect_print(lcrzoex_infos old,
			   lcrzoex_infos new);
