/*us    Laurent Constantin's network library (lcrzo)
                 lcrzo_packet module

  Functions herein allow to create and decode packets.
*/
/*fr  Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_packet

  Les fonctions presentes dans ce module permettent de creer ou
  d'analyser des paquets
*/

/*---------------------------------------------------------------*/
/*US*********************************************************
 * Note about header fields modification :                  *
 * Some header fields (hdrlip.totlen, hdrludp.len, etc.)    *
 * might be computed by functions lcrzo_packet_init_xxx.    *
 *                                                          *
 * the function lcrzo_global_set_hdr_computefields allows   *
 * to define if lcrzo has to recompute fields :             *
 *  - lcrzo_global_set_hdr_computefields(LCRZO_FALSE) :     *
 *    do not recompute, so use user's values                *
 *  - lcrzo_global_set_hdr_computefields(LCRZO_TRUE) :      *
 *    recompute (we can refine computed fields with         *
 *    lcrzo_global_set_hdr_cf_xxx)                          *
 *                                                          *
 * If we choose to compute, fields are computed only if     *
 * they value is different from default value. Concerned    *
 * fields are (followed by they default value) :            *
 *     hdrleth.type    LCRZO_VALDEF_ETHTYPE                 *
 *     hdrlip.totlen   LCRZO_VALDEF_IPTOTLEN                *
 *     hdrlip.ihl      LCRZO_VALDEF_IPIHL                   *
 *     hdrlip.check    LCRZO_VALDEF_IPCHECK                 *
 *     hdrlip.protocol LCRZO_VALDEF_IPPROTOCOL              *
 *     hdrludp.len     LCRZO_VALDEF_UDPLEN                  *
 *     hdrludp.check   LCRZO_VALDEF_UDPCHECK                *
 *     hdrltcp.doff    LCRZO_VALDEF_TCPDOFF                 *
 *     hdrltcp.check   LCRZO_VALDEF_TCPCHECK                *
 *     hdrlicmp.check  LCRZO_VALDEF_ICMPCHECK               *
 * Other fields aren't computed by lcrzo.                   *
 ************************************************************/
/*FR*********************************************************
 * Note sur le calcul des champs d'entete :                 *
 * Certains champs des entetes (hdrlip.totlen, hdrludp.len, *
 * etc.) peuvent etre recalcules par les fonctions          *
 * lcrzo_packet_init_xxx.                                   *
 *                                                          *
 * La fonction lcrzo_global_set_hdr_computefields permet de *
 * dire si lcrzo doit recalculer les champs :               *
 *  - lcrzo_global_set_hdr_computefields(LCRZO_FALSE) :     *
 *    ne recalcule pas, c'est a dire emploie les valeurs    *
 *    passees par l'utilisateur                             *
 *  - lcrzo_global_set_hdr_computefields(LCRZO_TRUE) :      *
 *    recalcule (on peut affiner les champs calcules avec   *
 *    lcrzo_global_set_hdr_cf_xxx)                          *
 *                                                          *
 * Lorsque l'on recalcule, les champs ne sont recalcules    *
 * que si ils ont une valeur differente de la valeur par    *
 * defaut. Les champs concernes par ce calcul sont (suivis  *
 * de leur valeur par defaut) :                             *
 *     hdrleth.type    LCRZO_VALDEF_ETHTYPE                 *
 *     hdrlip.totlen   LCRZO_VALDEF_IPTOTLEN                *
 *     hdrlip.ihl      LCRZO_VALDEF_IPIHL                   *
 *     hdrlip.check    LCRZO_VALDEF_IPCHECK                 *
 *     hdrlip.protocol LCRZO_VALDEF_IPPROTOCOL              *
 *     hdrludp.len     LCRZO_VALDEF_UDPLEN                  *
 *     hdrludp.check   LCRZO_VALDEF_UDPCHECK                *
 *     hdrltcp.doff    LCRZO_VALDEF_TCPDOFF                 *
 *     hdrltcp.check   LCRZO_VALDEF_TCPCHECK                *
 *     hdrlicmp.check  LCRZO_VALDEF_ICMPCHECK               *
 * Les autres champs des entetes ne sont pas du tout        *
 * modifies par la bibliotheque : la valeur entree par      *
 * l'utilisateur est mise dans le packet.                   *
 ************************************************************/

/*us Name : lcrzo_packet_init_ipoptdata
   Description :
     Initialize a packet with {IP header, IP options, IP data}.
   Input parameter(s) :
     hdrlip : IP header
     ipopt : IP options
     ipoptsize : IP options size
     data : IP data
     datasize : IP data size
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_packet_init_ipoptdata
   Description :
     Initialise un paquet avec {entete IP, options IP, donnees IP}
   Parametre(s) d'entree :
     hdrlip : entete IP
     ipopt : options IP
     ipoptsize : taille des options IP
     data : donnees IP
     datasize : taille des donnees IP
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_packet_init_ipoptdata(lcrzo_hdrlip hdrlip,
				const lcrzo_ipopt ipopt,
				lcrzo_uint8 ipoptsize,
				lcrzo_constdata data, 
				lcrzo_uint16 datasize,
				lcrzo_int32 dataoutmaxsize,
				lcrzo_data dataout,
				lcrzo_int32 *pdataoutsize);
/*us idem, but allocate pdataout (user then need to call free())*/
/*fr idem, mais alloue pdataout (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_initm_ipoptdata(lcrzo_hdrlip hdrlip,
				 const lcrzo_ipopt ipopt,
				 lcrzo_uint8 ipoptsize,
				 lcrzo_constdata data, 
				 lcrzo_uint16 datasize,
				 lcrzo_data *pdataout,
				 lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_packet_init_ipoptudpdata
   Description :
     Initialize a packet with {IP header, IP options,
     UDP header, UDP data}.
   Input parameter(s) :
     hdrlip : IP header
     ipopt : IP options
     ipoptsize : IP options size
     hdrludp : UDP header
     data : UDP data
     datasize : UDP data size
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_packet_init_ipoptudpdata
   Description :
     Initialise un paquet avec {entete IP, options IP,
     entete UDP, donnees UDP}
   Parametre(s) d'entree :
     hdrlip : entete IP
     ipopt : options IP
     ipoptsize : taille des options IP
     hdrludp : entete UDP
     data : donnees UDP
     datasize : taille des donnees UDP
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_packet_init_ipoptudpdata(lcrzo_hdrlip hdrlip,
				   const lcrzo_ipopt ipopt,
				   lcrzo_uint8 ipoptsize,
				   lcrzo_hdrludp hdrludp,
				   lcrzo_constdata data, 
				   lcrzo_uint16 datasize,
				   lcrzo_int32 dataoutmaxsize,
				   lcrzo_data dataout,
				   lcrzo_int32 *pdataoutsize);
/*us idem, but allocate pdataout (user then need to call free())*/
/*fr idem, mais alloue pdataout (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_initm_ipoptudpdata(lcrzo_hdrlip hdrlip,
				    const lcrzo_ipopt ipopt,
				    lcrzo_uint8 ipoptsize,
				    lcrzo_hdrludp hdrludp,
				    lcrzo_constdata data, 
				    lcrzo_uint16 datasize,
				    lcrzo_data *pdataout,
				    lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_packet_init_ipopttcpoptdata
   Description :
     Initialize a packet with {IP header, IP options,
     TCP header, TCP options, TCP data}.
   Input parameter(s) :
     hdrlip : IP header
     ipopt : IP options
     ipoptsize : IP options size
     hdrltcp : TCP header
     tcpopt : TCP options
     tcpoptsize : TCP options size
     data : TCP data
     datasize : TCP data size
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_packet_init_ipopttcpoptdata
   Description :
     Initialise un paquet avec {entete IP, options IP,
     entete TCP, options TCP, donnees TCP}
   Parametre(s) d'entree :
     hdrlip : entete IP
     ipopt : options IP
     ipoptsize : taille des options IP
     hdrltcp : entete TCP
     tcpopt : options TCP
     tcpoptsize : taille des options TCP
     data : donnees TCP
     datasize : taille des donnees TCP
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_packet_init_ipopttcpoptdata(lcrzo_hdrlip hdrlip,
				      const lcrzo_ipopt ipopt,
				      lcrzo_uint8 ipoptsize,
				      lcrzo_hdrltcp hdrltcp,
				      const lcrzo_tcpopt tcpopt,
				      lcrzo_uint8 tcpoptsize,
				      lcrzo_constdata data, 
				      lcrzo_uint16 datasize,
				      lcrzo_int32 dataoutmaxsize,
				      lcrzo_data dataout,
				      lcrzo_int32 *pdataoutsize);
/*us idem, but allocate pdataout (user then need to call free())*/
/*fr idem, mais alloue pdataout (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_initm_ipopttcpoptdata(lcrzo_hdrlip hdrlip,
				       const lcrzo_ipopt ipopt,
				       lcrzo_uint8 ipoptsize,
				       lcrzo_hdrltcp hdrltcp,
				       const lcrzo_tcpopt tcpopt,
				       lcrzo_uint8 tcpoptsize,
				       lcrzo_constdata data, 
				       lcrzo_uint16 datasize,
				       lcrzo_data *pdataout,
				       lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_packet_init_ipopticmpdata
   Description :
     Initialize a packet with {IP header, IP options,
     ICMP header, ICMP data}.
   Input parameter(s) :
     hdrlip : IP header
     ipopt : IP options
     ipoptsize : IP options size
     hdrlicmp : ICMP header
     data : ICMP data
     datasize : ICMP data size
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_packet_init_ipoptudpdata
   Description :
     Initialise un paquet avec {entete IP, options IP,
     entete ICMP, donnees ICMP}
   Parametre(s) d'entree :
     hdrlip : entete IP
     ipopt : options IP
     ipoptsize : taille des options IP
     hdrlicmp : entete ICMP
     data : donnees ICMP
     datasize : taille des donnees ICMP
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_packet_init_ipopticmpdata(lcrzo_hdrlip hdrlip,
				    const lcrzo_ipopt ipopt,
				    lcrzo_uint8 ipoptsize,
				    lcrzo_hdrlicmp hdrlicmp,
				    lcrzo_constdata data, 
				    lcrzo_uint16 datasize,
				    lcrzo_int32 dataoutmaxsize,
				    lcrzo_data dataout,
				    lcrzo_int32 *pdataoutsize);
/*us idem, but allocate pdataout (user then need to call free())*/
/*fr idem, mais alloue pdataout (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_initm_ipopticmpdata(lcrzo_hdrlip hdrlip,
				     const lcrzo_ipopt ipopt,
				     lcrzo_uint8 ipoptsize,
				     lcrzo_hdrlicmp hdrlicmp,
				     lcrzo_constdata data, 
				     lcrzo_uint16 datasize,
				     lcrzo_data *pdataout,
				     lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_packet_init_ethdata
   Description :
     Initialize a packet with {Ethernet header, Ethernet data}.
   Input parameter(s) :
     hdrleth : Ethernet header
     data : Ethernet data
     datasize : Ethernet data size
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_packet_init_ethdata
   Description :
     Initialise un paquet avec {entete Ethernet, donnees Ethernet}
   Parametre(s) d'entree :
     hdrleth : entete Ethernet
     data : donnees Ethernet
     datasize : taille des donnees Ethernet
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_packet_init_ethdata(lcrzo_hdrleth hdrleth,
			      lcrzo_constdata data, 
			      lcrzo_uint16 datasize,
			      lcrzo_int32 dataoutmaxsize,
			      lcrzo_data dataout,
			      lcrzo_int32 *pdataoutsize);
/*us idem, but allocate pdataout (user then need to call free())*/
/*fr idem, mais alloue pdataout (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_initm_ethdata(lcrzo_hdrleth hdrleth,
			       lcrzo_constdata data, 
			       lcrzo_uint16 datasize,
			       lcrzo_data *pdataout,
			       lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_packet_init_ethipoptdata
   Description :
     Initialize a packet with {Ethernet header, IP header, 
     IP options, IP data}.
   Input parameter(s) :
     hdrleth : Ethernet header
     hdrlip : IP header
     ipopt : IP options
     ipoptsize : IP options size
     data : IP data
     datasize : IP data size
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_packet_init_ethipoptdata
   Description :
     Initialise un paquet avec {entete Ethernet, entete IP,
     options IP, donnees IP }
   Parametre(s) d'entree :
     hdrleth : entete Ethernet
     hdrlip : entete IP
     ipopt : options IP
     ipoptsize : taille des options IP
     data : donnees IP
     datasize : taille des donnees IP
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_packet_init_ethipoptdata(lcrzo_hdrleth hdrleth,
				   lcrzo_hdrlip hdrlip,
				   const lcrzo_ipopt ipopt,
				   lcrzo_uint8 ipoptsize,
				   lcrzo_constdata data, 
				   lcrzo_uint16 datasize,
				   lcrzo_int32 dataoutmaxsize,
				   lcrzo_data dataout,
				   lcrzo_int32 *pdataoutsize);
/*us idem, but allocate pdataout (user then need to call free())*/
/*fr idem, mais alloue pdataout (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_initm_ethipoptdata(lcrzo_hdrleth hdrleth,
				    lcrzo_hdrlip hdrlip,
				    const lcrzo_ipopt ipopt,
				    lcrzo_uint8 ipoptsize,
				    lcrzo_constdata data, 
				    lcrzo_uint16 datasize,
				    lcrzo_data *pdataout,
				    lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_packet_init_ethipoptudpdata
   Description :
     Initialize a packet with {Ethernet header, IP header, 
     IP options, UDP header, UDP data}.
   Input parameter(s) :
     hdrleth : Ethernet header
     hdrlip : IP header
     ipopt : IP options
     ipoptsize : IP options size
     hdrludp : UDP header
     data : UDP data
     datasize : UDP data size
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_packet_init_ethipoptudpdata
   Description :
     Initialise un paquet avec {entete Ethernet, entete IP,
     options IP, entete UDP, donnees UDP}
   Parametre(s) d'entree :
     hdrleth : entete Ethernet
     hdrlip : entete IP
     ipopt : options IP
     ipoptsize : taille des options IP
     hdrludp : entete UDP
     data : donnees UDP
     datasize : taille des donnees UDP
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_packet_init_ethipoptudpdata(lcrzo_hdrleth hdrleth,
				      lcrzo_hdrlip hdrlip,
				      const lcrzo_ipopt ipopt,
				      lcrzo_uint8 ipoptsize,
				      lcrzo_hdrludp hdrludp,
				      lcrzo_constdata data, 
				      lcrzo_uint16 datasize,
				      lcrzo_int32 dataoutmaxsize,
				      lcrzo_data dataout,
				      lcrzo_int32 *pdataoutsize);
/*us idem, but allocate pdataout (user then need to call free())*/
/*fr idem, mais alloue pdataout (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_initm_ethipoptudpdata(lcrzo_hdrleth hdrleth,
				       lcrzo_hdrlip hdrlip,
				       const lcrzo_ipopt ipopt,
				       lcrzo_uint8 ipoptsize,
				       lcrzo_hdrludp hdrludp,
				       lcrzo_constdata data, 
				       lcrzo_uint16 datasize,
				       lcrzo_data *pdataout,
				       lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_packet_init_ethipopttcpoptdata
   Description :
     Initialize a packet with {Ethernet header, IP header, 
     IP options, TCP header, TCP options, TCP data}.
   Input parameter(s) :
     hdrleth : Ethernet header
     hdrlip : IP header
     ipopt : IP options
     ipoptsize : IP options size
     hdrltcp : TCP header
     tcpopt : TCP options
     tcpoptsize : TCP options size
     data : TCP data
     datasize : TCP data size
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_packet_init_ethipopttcpoptdata
   Description :
     Initialise un paquet avec {entete Ethernet, entete IP,
     options IP, entete TCP, options TCP, donnees TCP}
   Parametre(s) d'entree :
     hdrleth : entete Ethernet
     hdrlip : entete IP
     ipopt : options IP
     ipoptsize : taille des options IP
     hdrltcp : entete TCP
     tcpopt : options TCP
     tcpoptsize : taille des options TCP
     data : donnees TCP
     datasize : taille des donnees TCP
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_packet_init_ethipopttcpoptdata(lcrzo_hdrleth hdrleth,
					 lcrzo_hdrlip hdrlip,
					 const lcrzo_ipopt ipopt,
					 lcrzo_uint8 ipoptsize,
					 lcrzo_hdrltcp hdrltcp,
					 const lcrzo_tcpopt tcpopt,
					 lcrzo_uint8 tcpoptsize,
					 lcrzo_constdata data, 
					 lcrzo_uint16 datasize,
					 lcrzo_int32 dataoutmaxsize,
					 lcrzo_data dataout,
					 lcrzo_int32 *pdataoutsize);
/*us idem, but allocate pdataout (user then need to call free())*/
/*fr idem, mais alloue pdataout (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_initm_ethipopttcpoptdata(lcrzo_hdrleth hdrleth,
					  lcrzo_hdrlip hdrlip,
					  const lcrzo_ipopt ipopt,
					  lcrzo_uint8 ipoptsize,
					  lcrzo_hdrltcp hdrltcp,
					  const lcrzo_tcpopt tcpopt,
					  lcrzo_uint8 tcpoptsize,
					  lcrzo_constdata data, 
					  lcrzo_uint16 datasize,
					  lcrzo_data *pdataout,
					  lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_packet_init_ethipopticmpdata
   Description :
     Initialize a packet with {Ethernet header, IP header, 
     IP options, ICMP header, ICMP data}.
   Input parameter(s) :
     hdrleth : Ethernet header
     hdrlip : IP header
     ipopt : IP options
     ipoptsize : IP options size
     hdrlicmp : ICMP header
     data : ICMP data
     datasize : ICMP data size
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_packet_init_ethipopticmpdata
   Description :
     Initialise un paquet avec {entete Ethernet, entete IP,
     options IP, entete ICMP, donnees ICMP}
   Parametre(s) d'entree :
     hdrleth : entete Ethernet
     hdrlip : entete IP
     ipopt : options IP
     ipoptsize : taille des options IP
     hdrlicmp : entete ICMP
     data : donnees ICMP
     datasize : taille des donnees ICMP
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_packet_init_ethipopticmpdata(lcrzo_hdrleth hdrleth,
				       lcrzo_hdrlip hdrlip,
				       const lcrzo_ipopt ipopt,
				       lcrzo_uint8 ipoptsize,
				       lcrzo_hdrlicmp hdrlicmp,
				       lcrzo_constdata data, 
				       lcrzo_uint16 datasize,
				       lcrzo_int32 dataoutmaxsize,
				       lcrzo_data dataout,
				       lcrzo_int32 *pdataoutsize);
/*us idem, but allocate pdataout (user then need to call free())*/
/*fr idem, mais alloue pdataout (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_initm_ethipopticmpdata(lcrzo_hdrleth hdrleth,
					lcrzo_hdrlip hdrlip,
					const lcrzo_ipopt ipopt,
					lcrzo_uint8 ipoptsize,
					lcrzo_hdrlicmp hdrlicmp,
					lcrzo_constdata data, 
					lcrzo_uint16 datasize,
					lcrzo_data *pdataout,
					lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_packet_init_etharp
   Description :
     Initialize a packet with {Ethernet header, ARP header}.
   Input parameter(s) :
     hdrleth : Ethernet header
     hdrlarp : ARP header
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_packet_init_etharp
   Description :
     Initialise un paquet avec {entete Ethernet, entete ARP}
   Parametre(s) d'entree :
     hdrleth : entete Ethernet
     hdrlarp : entete ARP
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_packet_init_etharp(lcrzo_hdrleth hdrleth,
			     lcrzo_hdrlarp hdrlarp,
			     lcrzo_int32 dataoutmaxsize,
			     lcrzo_data dataout,
			     lcrzo_int32 *pdataoutsize);
/*us idem, but allocate pdataout (user then need to call free())*/
/*fr idem, mais alloue pdataout (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_initm_etharp(lcrzo_hdrleth hdrleth,
			      lcrzo_hdrlarp hdrlarp,
			      lcrzo_data *pdataout,
			      lcrzo_int32 *pdataoutsize);


/*---------------------------------------------------------------*/
/*us Name : lcrzo_packet_decode_ipoptdata
   Description :
     Check if a packet contains {IP header, 
     IP options, IP data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrlip : IP header
     ipopt : IP options
     *pipoptsize : size of ipopt
     data : IP data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_ipoptdata
   Description :
     Verifie si un packet contient {entete IP,
     options IP, donnees IP }
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrlip : entete IP
     ipopt : options IP
     *pipoptsize : taille de ipopt
     data : donnees IP
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_ipoptdata(lcrzo_constdata packet,
				  lcrzo_int32 packetsize,
				  lcrzo_hdrlip *phdrlip,
				  lcrzo_ipopt ipopt,
				  lcrzo_uint8 *pipoptsize,
				  lcrzo_uint16 datamaxsize,
				  lcrzo_data data,
				  lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_ipoptdata(lcrzo_constdata packet,
				   lcrzo_int32 packetsize,
				   lcrzo_hdrlip *phdrlip,
				   lcrzo_ipopt ipopt,
				   lcrzo_uint8 *pipoptsize,
				   lcrzo_data *pdata,
				   lcrzo_uint16 *pdatasize);

/*us Name : lcrzo_packet_decode_udpdata
   Description :
     Check if a packet contains {UDP header, UDP data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrludp : UDP header
     data : UDP data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_udpdata
   Description :
     Verifie si un packet contient {entete UDP, donnees UDP}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrludp : entete UDP
     data : donnees UDP
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_udpdata(lcrzo_constdata packet,
				lcrzo_int32 packetsize,
				lcrzo_hdrludp *phdrludp,
				lcrzo_uint16 datamaxsize,
				lcrzo_data data,
				lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_udpdata(lcrzo_constdata packet,
				 lcrzo_int32 packetsize,
				 lcrzo_hdrludp *phdrludp,
				 lcrzo_data *pdata,
				 lcrzo_uint16 *pdatasize);

/*us Name : lcrzo_packet_decode_tcpoptdata
   Description :
     Check if a packet contains {TCP header, 
     TCP options, TCP data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrltcp : TCP header
     tcpopt : TCP options
     *ptcpoptsize : size of tcpopt
     data : TCP data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_tcpoptdata
   Description :
     Verifie si un packet contient {entete TCP,
     options TCP, donnees TCP}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrltcp : entete TCP
     tcpopt : options TCP
     *ptcpoptsize : taille de tcpopt
     data : donnees TCP
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_tcpoptdata(lcrzo_constdata packet, 
				   lcrzo_int32 packetsize,
				   lcrzo_hdrltcp *phdrltcp,
				   lcrzo_tcpopt tcpopt,
				   lcrzo_uint8 *ptcpoptsize,
				   lcrzo_uint16 datamaxsize,
				   lcrzo_data data,
				   lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_tcpoptdata(lcrzo_constdata packet, 
				    lcrzo_int32 packetsize,
				    lcrzo_hdrltcp *phdrltcp,
				    lcrzo_tcpopt tcpopt,
				    lcrzo_uint8 *ptcpoptsize,
				    lcrzo_data *pdata,
				    lcrzo_uint16 *pdatasize);

/*us Name : lcrzo_packet_decode_icmpdata
   Description :
     Check if a packet contains {ICMP header, ICMP data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrlicmp : ICMP header
     data : ICMP data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_icmpdata
   Description :
     Verifie si un packet contient {entete ICMP, donnees ICMP}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrlicmp : entete ICMP
     data : donnees ICMP
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_icmpdata(lcrzo_constdata packet,
				 lcrzo_int32 packetsize,
				 lcrzo_hdrlicmp *phdrlicmp,
				 lcrzo_uint16 datamaxsize,
				 lcrzo_data data,
				 lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_icmpdata(lcrzo_constdata packet,
				  lcrzo_int32 packetsize,
				  lcrzo_hdrlicmp *phdrlicmp,
				  lcrzo_data *pdata,
				  lcrzo_uint16 *pdatasize);

/*us Name : lcrzo_packet_decode_ethdata
   Description :
     Check if a packet contains {Ethernet header, Ethernet data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrleth : Ethernet header
     data : Ethernet data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_ethdata
   Description :
     Verifie si un packet contient {entete Ethernet, donnees Ethernet}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrleth : entete Ethernet
     data : donnees Ethernet
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_ethdata(lcrzo_constdata packet,
				lcrzo_int32 packetsize,
				lcrzo_hdrleth *phdrleth,
				lcrzo_uint16 datamaxsize,
				lcrzo_data data,
				lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_ethdata(lcrzo_constdata packet,
				 lcrzo_int32 packetsize,
				 lcrzo_hdrleth *phdrleth,
				 lcrzo_data *pdata,
				 lcrzo_uint16 *pdatasize);

/*us Name : lcrzo_packet_decode_arp
   Description :
     Check if a packet contains {ARP/RARP header}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrlarp : arp/rarp header
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
*/
/*fr Nom : lcrzo_packet_decode_arp
   Description :
     Verifie si un packet contient {entete arp/rarp}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrlarp : entete arp/rarp
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
*/
int lcrzo_packet_decode_arp(lcrzo_constdata packet,
			    lcrzo_int32 packetsize,
			    lcrzo_hdrlarp *phdrlarp);
	
/*us Name : lcrzo_packet_decode_ipoptudpdata
   Description :
     Check if a packet contains {IP header, 
     IP options, UDP header, UDP data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrlip : IP header
     ipopt : IP options
     *pipoptsize : size of ipopt
     *phdrludp : UDP header
     data : UDP data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_ipoptudpdata
   Description :
     Verifie si un packet contient {entete IP,
     options IP, entete UDP, donnees UDP}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrlip : entete IP
     ipopt : options IP
     *pipoptsize : taille de ipopt
     *phdrludp : entete UDP
     data : donnees UDP
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_ipoptudpdata(lcrzo_constdata packet, 
				     lcrzo_int32 packetsize,
				     lcrzo_hdrlip *phdrlip,
				     lcrzo_ipopt ipopt,
				     lcrzo_uint8 *pipoptsize,
				     lcrzo_hdrludp *phdrludp,
				     lcrzo_uint16 datamaxsize,
				     lcrzo_data data,
				     lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_ipoptudpdata(lcrzo_constdata packet, 
				      lcrzo_int32 packetsize,
				      lcrzo_hdrlip *phdrlip,
				      lcrzo_ipopt ipopt,
				      lcrzo_uint8 *pipoptsize,
				      lcrzo_hdrludp *phdrludp,
				      lcrzo_data *pdata,
				      lcrzo_uint16 *pdatasize);

/*us Name : lcrzo_packet_decode_ipopttcpoptdata
   Description :
     Check if a packet contains {IP header, 
     IP options, TCP header, TCP options, TCP data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrlip : IP header
     ipopt : IP options
     *pipoptsize : size of ipopt
     *phdrltcp : TCP header
     tcpopt : TCP options
     *ptcpoptsize : size of tcpopt
     data : TCP data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_ipopttcpoptdata
   Description :
     Verifie si un packet contient {entete IP,
     options IP, entete TCP, options TCP, donnees TCP}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrlip : entete IP
     ipopt : options IP
     *pipoptsize : taille de ipopt
     *phdrltcp : entete TCP
     tcpopt : options TCP
     *ptcpoptsize : taille de tcpopt
     data : donnees TCP
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_ipopttcpoptdata(lcrzo_constdata packet, 
					lcrzo_int32 packetsize,
					lcrzo_hdrlip *phdrlip,
					lcrzo_ipopt ipopt,
					lcrzo_uint8 *pipoptsize,
					lcrzo_hdrltcp *phdrltcp,
					lcrzo_tcpopt tcpopt,
					lcrzo_uint8 *ptcpoptsize,
					lcrzo_uint16 datamaxsize,
					lcrzo_data data,
					lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_ipopttcpoptdata(lcrzo_constdata packet, 
					 lcrzo_int32 packetsize,
					 lcrzo_hdrlip *phdrlip,
					 lcrzo_ipopt ipopt,
					 lcrzo_uint8 *pipoptsize,
					 lcrzo_hdrltcp *phdrltcp,
					 lcrzo_tcpopt tcpopt,
					 lcrzo_uint8 *ptcpoptsize,
					 lcrzo_data *pdata,
					 lcrzo_uint16 *pdatasize);

/*us Name : lcrzo_packet_decode_ipopticmpdata
   Description :
     Check if a packet contains {IP header, 
     IP options, ICMP header, ICMP data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrlip : IP header
     ipopt : IP options
     *pipoptsize : size of ipopt
     *phdrlicmp : ICMP header
     data : ICMP data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_ipopticmpdata
   Description :
     Verifie si un packet contient {entete IP,
     options IP, entete ICMP, donnees ICMP}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrlip : entete IP
     ipopt : options IP
     *pipoptsize : taille de ipopt
     *phdrlicmp : entete ICMP
     data : donnees ICMP
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_ipopticmpdata(lcrzo_constdata packet, 
				      lcrzo_int32 packetsize,
				      lcrzo_hdrlip *phdrlip,
				      lcrzo_ipopt ipopt,
				      lcrzo_uint8 *pipoptsize,
				      lcrzo_hdrlicmp *phdrlicmp,
				      lcrzo_uint16 datamaxsize,
				      lcrzo_data data,
				      lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_ipopticmpdata(lcrzo_constdata packet, 
				       lcrzo_int32 packetsize,
				       lcrzo_hdrlip *phdrlip,
				       lcrzo_ipopt ipopt,
				       lcrzo_uint8 *pipoptsize,
				       lcrzo_hdrlicmp *phdrlicmp,
				       lcrzo_data *pdata,
				       lcrzo_uint16 *pdatasize);

/*us Name : lcrzo_packet_decode_ethipoptdata
   Description :
     Check if a packet contains {Ethernet header, IP header, 
     IP options, IP data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrleth : Ethernet header
     *phdrlip : IP header
     ipopt : IP options
     *pipoptsize : size of ipopt
     data : IP data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_ethipoptdata
   Description :
     Verifie si un packet contient {entete Ethernet, entete IP,
     options IP, donnees IP}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrleth : entete Ethernet
     *phdrlip : entete IP
     ipopt : options IP
     *pipoptsize : taille de ipopt
     data : donnees IP
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_ethipoptdata(lcrzo_constdata packet, 
				     lcrzo_int32 packetsize,
				     lcrzo_hdrleth *phdrleth,
				     lcrzo_hdrlip *phdrlip,
				     lcrzo_ipopt ipopt,
				     lcrzo_uint8 *pipoptsize,
				     lcrzo_uint16 datamaxsize,
				     lcrzo_data data,
				     lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_ethipoptdata(lcrzo_constdata packet, 
				      lcrzo_int32 packetsize,
				      lcrzo_hdrleth *phdrleth,
				      lcrzo_hdrlip *phdrlip,
				      lcrzo_ipopt ipopt,
				      lcrzo_uint8 *pipoptsize,
				      lcrzo_data *pdata,
				      lcrzo_uint16 *pdatasize);

/*us Name : lcrzo_packet_decode_ethipoptudpdata
   Description :
     Check if a packet contains {Ethernet header, IP header, 
     IP options, UDP header, UDP data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrleth : Ethernet header
     *phdrlip : IP header
     ipopt : IP options
     *pipoptsize : size of ipopt
     *phdrludp : UDP header
     data : UDP data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_ethipoptudpdata
   Description :
     Verifie si un packet contient {entete Ethernet, entete IP,
     options IP, entete UDP, donnees UDP}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrleth : entete Ethernet
     *phdrlip : entete IP
     ipopt : options IP
     *pipoptsize : taille de ipopt
     *phdrludp : entete UDP
     data : donnees UDP
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_ethipoptudpdata(lcrzo_constdata packet,
					lcrzo_int32 packetsize,
					lcrzo_hdrleth *phdrleth,
					lcrzo_hdrlip *phdrlip,
					lcrzo_ipopt ipopt,
					lcrzo_uint8 *pipoptsize,
					lcrzo_hdrludp *phdrludp,
					lcrzo_uint16 datamaxsize,
					lcrzo_data data,
					lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_ethipoptudpdata(lcrzo_constdata packet,
					 lcrzo_int32 packetsize,
					 lcrzo_hdrleth *phdrleth,
					 lcrzo_hdrlip *phdrlip,
					 lcrzo_ipopt ipopt,
					 lcrzo_uint8 *pipoptsize,
					 lcrzo_hdrludp *phdrludp,
					 lcrzo_data *pdata,
					 lcrzo_uint16 *pdatasize);

/*us Name : lcrzo_packet_decode_ethipopttcpoptdata
   Description :
     Check if a packet contains {Ethernet header, IP header, 
     IP options, TCP header, TCP options, TCP data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrleth : Ethernet header
     *phdrlip : IP header
     ipopt : IP options
     *pipoptsize : size of ipopt
     *phdrltcp : TCP header
     tcpopt : TCP options
     *ptcpoptsize : size of tcpopt
     data : TCP data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_ethipopttcpoptdata
   Description :
     Verifie si un packet contient {entete Ethernet, entete IP,
     options IP, entete TCP, options TCP, donnees TCP}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrleth : entete Ethernet
     *phdrlip : entete IP
     ipopt : options IP
     *pipoptsize : taille de ipopt
     *phdrltcp : entete TCP
     tcpopt : options TCP
     *ptcpoptsize : taille de tcpopt
     data : donnees TCP
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_ethipopttcpoptdata(lcrzo_constdata packet,
					   lcrzo_int32 packetsize,
					   lcrzo_hdrleth *phdrleth,
					   lcrzo_hdrlip *phdrlip,
					   lcrzo_ipopt ipopt,
					   lcrzo_uint8 *pipoptsize,
					   lcrzo_hdrltcp *phdrltcp,
					   lcrzo_tcpopt tcpopt,
					   lcrzo_uint8 *ptcpoptsize,
					   lcrzo_uint16 datamaxsize,
					   lcrzo_data data,
					   lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_ethipopttcpoptdata(lcrzo_constdata packet,
					    lcrzo_int32 packetsize,
					    lcrzo_hdrleth *phdrleth,
					    lcrzo_hdrlip *phdrlip,
					    lcrzo_ipopt ipopt,
					    lcrzo_uint8 *pipoptsize,
					    lcrzo_hdrltcp *phdrltcp,
					    lcrzo_tcpopt tcpopt,
					    lcrzo_uint8 *ptcpoptsize,
					    lcrzo_data *pdata,
					    lcrzo_uint16 *pdatasize);
					  
/*us Name : lcrzo_packet_decode_ethipopticmpdata
   Description :
     Check if a packet contains {Ethernet header, IP header, 
     IP options, ICMP header, ICMP data}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrleth : Ethernet header
     *phdrlip : IP header
     ipopt : IP options
     *pipoptsize : size of ipopt
     *phdrlicmp : ICMP header
     data : ICMP data
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
     LCRZO_ERR_OKDATATRUNCATED : packet decoded, but datamaxsize
                                 reached before being able to store
				 all the needed data in data
*/
/*fr Nom : lcrzo_packet_decode_ethipopticmpdata
   Description :
     Verifie si un packet contient {entete Ethernet, entete IP,
     options IP, entete ICMP, donnees ICMP}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
     datamaxsize : taille maximale qui peut etre stockee
                   dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrleth : entete Ethernet
     *phdrlip : entete IP
     ipopt : options IP
     *pipoptsize : taille de ipopt
     *phdrlicmp : entete ICMP
     data : donnees ICMP
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
     LCRZO_ERR_OKDATATRUNCATED : paquet decode, mais datamaxsize
                                 a ete atteint avant d'avoir pu 
				 stocker toutes les donnees dans
				 data
*/
int lcrzo_packet_decode_ethipopticmpdata(lcrzo_constdata packet,
					 lcrzo_int32 packetsize,
					 lcrzo_hdrleth *phdrleth,
					 lcrzo_hdrlip *phdrlip,
					 lcrzo_ipopt ipopt,
					 lcrzo_uint8 *pipoptsize,
					 lcrzo_hdrlicmp *phdrlicmp,
					 lcrzo_uint16 datamaxsize,
					 lcrzo_data data,
					 lcrzo_uint16 *pdatasize);
/*us idem, but allocate pdata (user then need to call free())*/
/*fr idem, mais alloue pdata (l'utilisateur doit etre ensuite 
  liberer la memoire avec free())*/
int lcrzo_packet_decodem_ethipopticmpdata(lcrzo_constdata packet,
					  lcrzo_int32 packetsize,
					  lcrzo_hdrleth *phdrleth,
					  lcrzo_hdrlip *phdrlip,
					  lcrzo_ipopt ipopt,
					  lcrzo_uint8 *pipoptsize,
					  lcrzo_hdrlicmp *phdrlicmp,
					  lcrzo_data *pdata,
					  lcrzo_uint16 *pdatasize);

/*us Name : lcrzo_packet_decode_etharp
   Description :
     Check if a packet contains {Ethernet header, ARP header}.
   Input parameter(s) :
     packet : packet to check
     packetsize : size of packet
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrleth : Ethernet header
     *phdrlarp : arp/rarp header
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
*/
/*fr Nom : lcrzo_packet_decode_etharp
   Description :
     Verifie si un packet contient {entete Ethernet, ARP header}.
   Parametre(s) d'entree :
     packet : paquet a verifier
     packetsize : taille de packet
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrleth : entete Ethernet
     *phdrlarp : entete arp/rarp
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
*/
int lcrzo_packet_decode_etharp(lcrzo_constdata packet,
			       lcrzo_int32 packetsize,
			       lcrzo_hdrleth *phdrleth,
			       lcrzo_hdrlarp *phdrlarp);

