/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

/* -- Note about comments and variable names --
   In versions 1.00 up to 2.08, lcrzo was only written in French (my
   mother tongue). Since version 3.00, lcrzo is written in English, but
   most of comments and variable names are still in French.
   In future versions, I'll try to translate everything into English,
   but as you may guess, that's a quite big work...
   So, for the moment, here are some word translation, if you want to
   be able to understand what's going on in the functions :
     afficher - print     ;  ajouter - add         ;  copie - copy
     dernier - last       ;  donnee - data         ;  ecrire - write
     entete - header      ;  envoye - sent         ;  ferme - close
     fichier - file       ;  lire - read           ;  masque - mask 
     nboct - byte number  ;  ouvre, ouvrir - open  ;  paq, paquet - packet
     plage - range        ;  premier - first       ;  recu - received
     reseau - network     ;  retour - return       ;  taille - size
     tunnel - pipe

*/
 
#include "lcrzo_priv.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #include <stdlib.h>
 #include <stdio.h>
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #include <stdlib.h>
 #include <stdio.h>
#elif defined LCRZODEF_SYSTEM_Solaris
 #include <stdlib.h>
 #include <stdio.h>
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif


/*-------------------------------------------------------------*/
#define LCRZO_PRINTTYPE_C_ustxt    ", comment before"
#define LCRZO_PRINTTYPE_C_frtxt    ", commentaire avant"
#define LCRZO_PRINTTYPE_S_ustxt    ", short data lines"
#define LCRZO_PRINTTYPE_S_frtxt    ", lignes courtes"
#define LCRZO_PRINTTYPE_N_ustxt    ", newline after"
#define LCRZO_PRINTTYPE_N_frtxt    ", saut de ligne apres"
#define LCRZO_PRINTTYPE_I_ustxt    ""
#define LCRZO_PRINTTYPE_I_frtxt    ""

/*-------------------------------------------------------------*/
#define LCRZO_PRIV_HEXA       1
#define LCRZO_PRIV_DEC        2
#define LCRZO_PRIV_CAR        4
#define LCRZO_PRIV_HEXADEC    (LCRZO_PRIV_HEXA|LCRZO_PRIV_DEC)
#define LCRZO_PRIV_HEXACAR    (LCRZO_PRIV_HEXA|LCRZO_PRIV_CAR)
#define LCRZO_PRIV_DECCAR     (LCRZO_PRIV_DEC|LCRZO_PRIV_CAR)
#define LCRZO_PRIV_HEXADECCAR (LCRZO_PRIV_HEXADEC|LCRZO_PRIV_CAR)
int lcrzo_priv_ecrit_tab01(LCRZOFILE *pf, lcrzo_uint8 nb);
int lcrzo_priv_ecrit_tab01(LCRZOFILE *pf, lcrzo_uint8 nb)
{ fprintf(pf, "%1d_", nb);
  return(LCRZO_ERR_OK);
}
int lcrzo_priv_ecrit_tab04(LCRZOFILE *pf, lcrzo_uint8 nb, int type);
int lcrzo_priv_ecrit_tab04(LCRZOFILE *pf, lcrzo_uint8 nb, int type)
{ if (type==LCRZO_PRIV_HEXA)
    fprintf(pf, "___%1Xh___", nb);
  else if (type==LCRZO_PRIV_DEC)
    fprintf(pf, "___%2d___", nb);
  else
    return(LCRZO_ERR_IEINTERNALERROR);
  return(LCRZO_ERR_OK);
}
int lcrzo_priv_ecrit_tab08(LCRZOFILE *pf, lcrzo_uint8 nb, int type);
int lcrzo_priv_ecrit_tab08(LCRZOFILE *pf, lcrzo_uint8 nb, int type)
{/*             "0_1_2_3_.0_1_2_3_:0_1_2_3_.0_1_2_3_|..."*/
  if (type==LCRZO_PRIV_HEXA)
    fprintf(pf, "_______%02Xh_______", nb);
  else if (type==LCRZO_PRIV_DEC)
    fprintf(pf, "_______%3d_______", nb);
  else if (type==LCRZO_PRIV_CAR)
    fprintf(pf, "________%c________", (nb<31 || nb>126) ? '.' : nb);
  else if (type==LCRZO_PRIV_HEXADEC)
    fprintf(pf, "_____%02Xh=%3d_____", nb, nb);
  else if (type==LCRZO_PRIV_HEXACAR)
    fprintf(pf, "______%02Xh=%c______", nb, (nb<31 || nb>126) ? '.' : nb);
  else if (type==LCRZO_PRIV_DECCAR)
    fprintf(pf, "______%3d=%c______", nb, (nb<31 || nb>126) ? '.' : nb);
  else if (type==LCRZO_PRIV_HEXADECCAR)
    fprintf(pf, "____%02Xh=%3d=%c____", nb, nb,
	    (nb<31 || nb>126) ? '.' : nb);
  else
    return(LCRZO_ERR_IEINTERNALERROR);
  return(LCRZO_ERR_OK);
}
int lcrzo_priv_ecrit_tab16(LCRZOFILE *pf, lcrzo_uint16 nb, int type);
int lcrzo_priv_ecrit_tab16(LCRZOFILE *pf, lcrzo_uint16 nb, int type)
{/*             "0_1_2_3_.0_1_2_3_:0_1_2_3_.0_1_2_3_|..."*/
  if (type==LCRZO_PRIV_HEXA)
    fprintf(pf, "_______________%04Xh_______________", nb);
  else if (type==LCRZO_PRIV_DEC)
    fprintf(pf, "_______________%5d_______________", nb);
  else if (type==LCRZO_PRIV_HEXADEC)
    fprintf(pf, "____________%04Xh=%5d____________", nb, nb);
  else
    return(LCRZO_ERR_IEINTERNALERROR);
  return(LCRZO_ERR_OK);
}
int lcrzo_priv_ecrit_tab32(LCRZOFILE *pf, lcrzo_uint32 nb, int type);
int lcrzo_priv_ecrit_tab32(LCRZOFILE *pf, lcrzo_uint32 nb, int type)
{/*             "0_1_2_3_.0_1_2_3_:0_1_2_3_.0_1_2_3_|..."*/
  if (type==LCRZO_PRIV_HEXA)
    fprintf(pf, "_______________________________%08lXh_______________________________", nb);
  else if (type==LCRZO_PRIV_DEC)
    fprintf(pf, "______________________________%10lu_______________________________", nb);
  else if (type==LCRZO_PRIV_HEXADEC)
    fprintf(pf, "_________________________%08lXh=%10lu__________________________", nb, nb);
  else
    return(LCRZO_ERR_IEINTERNALERROR);
  return(LCRZO_ERR_OK);
}
/*
int lcrzo_priv_ecrit_tabxx_test(void)
{ int i;
  printf("0_1_2_3_.0_1_2_3_:0_1_2_3_.0_1_2_3_|");
  puts("0_1_2_3_.0_1_2_3_:0_1_2_3_.0_1_2_3_");
  lcrzo_priv_ecrit_tab01(stdout, 1);puts("ZZ");
  for ( i=1 ; i<=2 ; i++ )
  { lcrzo_priv_ecrit_tab04(stdout, 0xF, i);puts("ZZ");
  }
  for ( i=1 ; i<=8 ; i++ )
  { lcrzo_priv_ecrit_tab08(stdout, 0xFF, i);puts("ZZ");
  }
  for ( i=1 ; i<=8 ; i++ )
  { lcrzo_priv_ecrit_tab16(stdout, 0xFFFF, i);puts("ZZ");
  }
  for ( i=1 ; i<=8 ; i++ )
  { lcrzo_priv_ecrit_tab32(stdout, 0xFFffffFFU, i);puts("ZZ");
  }
  return(LCRZO_ERR_OK);
}
*/

/*-------------------------------------------------------------*/
int lcrzo_priv_printtype_printcomment(LCRZOFILE *pf, 
				      lcrzo_printtype printtype,
				      const char *uscom,
				      const char *frcom);
int lcrzo_priv_printtype_printcomment(LCRZOFILE *pf, 
				      lcrzo_printtype printtype,
				      const char *uscom,
				      const char *frcom)
{
  if ( printtype&LCRZO_PRINTTYPE_C_ )
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
    { if (frcom!=NULL) fprintf(pf, "#%s\n", frcom);
    }
    else
    { if (uscom!=NULL) fprintf(pf, "#%s\n", uscom);
    }
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_printtype_printnewline(LCRZOFILE *pf, 
				      lcrzo_printtype printtype);
int lcrzo_priv_printtype_printnewline(LCRZOFILE *pf, 
				      lcrzo_printtype printtype)
{
  if ( printtype&LCRZO_PRINTTYPE_N_ )
  { fprintf(pf, "\n");
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ecrittransition(LCRZOFILE *pf, 
			      lcrzo_printprofile *pprintprofile,
			      lcrzo_printtype nextprinttype);
int lcrzo_priv_ecrittransition(LCRZOFILE *pf, 
			      lcrzo_printprofile *pprintprofile,
			      lcrzo_printtype nextprinttype)
{
  /*fprintf(pf, "[Transition de %04X -> %04X]", pprintprofile->lastused,
    nextprinttype);*/

  /*rien a ecrire si on ne veut rien ecrire. Ne pas changer
    l'etat de pprintprofile->lastused*/
  if ( (nextprinttype&0xFF) == LCRZO_PRINTTYPE_NOTHING_ )
  { return(LCRZO_ERR_OK);
  }

  /*if the last used has a newline*/
  if ( (pprintprofile->lastused & (LCRZO_PRINTTYPE_N_|LCRZO_PRINTTYPE_I_)) ||
       (pprintprofile->lastused == LCRZO_PRINTTYPE_START) )
  { /*nothing to print*/
  }
  else if ( (pprintprofile->lastused&0xFF) == LCRZO_PRINTTYPE_NOTHING_ )
  { /*nothing to print*/
  }
  else
  { /*if the next must begin with an new line*/ 
    if ( (nextprinttype & ( LCRZO_PRINTTYPE_C_ | LCRZO_PRINTTYPE_I_ )) ||
       ( nextprinttype == LCRZO_PRINTTYPE_START ) )
    { fprintf(pf, "\n");
    }
    else
    { /*so now, we only have the case where the lastused doesn't end with
	an newline, and the next mustn't begin with an new line*/
      switch(pprintprofile->lastused) {
      case LCRZO_PRINTTYPE_SYNTH :
	switch(nextprinttype) {
	case LCRZO_PRINTTYPE_SYNTH :
	case LCRZO_PRINTTYPE_SYNTHN :
	case LCRZO_PRINTTYPE_TEXT :
	case LCRZO_PRINTTYPE_TEXTN :
	case LCRZO_PRINTTYPE_PTEXT :
	case LCRZO_PRINTTYPE_PTEXTN :
	case LCRZO_PRINTTYPE_BASE64 :
	case LCRZO_PRINTTYPE_BASE64N :
	case LCRZO_PRINTTYPE_HEXA :
	case LCRZO_PRINTTYPE_HEXAN :
	case LCRZO_PRINTTYPE_MIXED :
	case LCRZO_PRINTTYPE_MIXEDN :
	  fprintf(pf, " - ");
	  break;
	default :
	  return(LCRZO_ERR_IEINTERNALERROR);
	}
	break;
      case LCRZO_PRINTTYPE_TEXT :
      case LCRZO_PRINTTYPE_PTEXT :
	switch(nextprinttype) {
	case LCRZO_PRINTTYPE_SYNTH :
	case LCRZO_PRINTTYPE_SYNTHN :
	  fprintf(pf, " - ");
	  break;
	case LCRZO_PRINTTYPE_TEXT :
	case LCRZO_PRINTTYPE_TEXTN :
	case LCRZO_PRINTTYPE_PTEXT :
	case LCRZO_PRINTTYPE_PTEXTN :
	  break;
	case LCRZO_PRINTTYPE_BASE64 :
	case LCRZO_PRINTTYPE_BASE64N :
	case LCRZO_PRINTTYPE_HEXA :
	case LCRZO_PRINTTYPE_HEXAN :
	case LCRZO_PRINTTYPE_MIXED :
	case LCRZO_PRINTTYPE_MIXEDN :
	  break;
	default :
	  return(LCRZO_ERR_IEINTERNALERROR);
	}
	break;
      case LCRZO_PRINTTYPE_BASE64 :
      case LCRZO_PRINTTYPE_HEXA :
      case LCRZO_PRINTTYPE_MIXED :
	switch(nextprinttype) {
	case LCRZO_PRINTTYPE_SYNTH :
	case LCRZO_PRINTTYPE_SYNTHN :
	  fprintf(pf, " - ");
	  break;
	case LCRZO_PRINTTYPE_TEXT :
	case LCRZO_PRINTTYPE_TEXTN :
	case LCRZO_PRINTTYPE_PTEXT :
	case LCRZO_PRINTTYPE_PTEXTN :
	  break;
	case LCRZO_PRINTTYPE_BASE64 :
	case LCRZO_PRINTTYPE_BASE64N :
	case LCRZO_PRINTTYPE_HEXA :
	case LCRZO_PRINTTYPE_HEXAN :
	case LCRZO_PRINTTYPE_MIXED :
	case LCRZO_PRINTTYPE_MIXEDN :
	  fprintf(pf, " ");
	  break;
	default :
	  return(LCRZO_ERR_IEINTERNALERROR);
	}
	break;
      default :
	return(LCRZO_ERR_IEINTERNALERROR);
	break;
      }
    }
  }
  
  /*for the next function call*/
  pprintprofile->lastused = nextprinttype;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ecrittransitionfin(LCRZOFILE *pf, 
				 lcrzo_printprofile *pprintprofile);
int lcrzo_priv_ecrittransitionfin(LCRZOFILE *pf, 
				 lcrzo_printprofile *pprintprofile)
{ return(lcrzo_priv_ecrittransition(pf, pprintprofile, LCRZO_PRINTTYPE_START));
}

/*-------------------------------------------------------------*/
int lcrzo_priv_printprofile_tcdf(LCRZOFILE *pf, 
				 lcrzo_printprofile *pprintprofile,
				 lcrzo_printtype printtype,
				 const char *uscom,
				 const char *frcom,
				 lcrzo_constdata packet, 
				 lcrzo_int32 nboctpacket);
int lcrzo_priv_printprofile_tcdf(LCRZOFILE *pf, 
				 lcrzo_printprofile *pprintprofile,
				 lcrzo_printtype printtype,
				 const char *uscom,
				 const char *frcom,
				 lcrzo_constdata packet, 
				 lcrzo_int32 nboctpacket)
{ lcrzo_er(lcrzo_priv_ecrittransition(pf, pprintprofile, printtype));
  lcrzo_er(lcrzo_priv_printtype_printcomment(pf, printtype, uscom, frcom));
  lcrzo_er(lcrzo_data_fprint(pf, packet, nboctpacket, printtype));
  return(lcrzo_priv_ecrittransitionfin(pf, pprintprofile));
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ecritpacket_datadump(LCRZOFILE *pf, 
				    lcrzo_constdata packet, 
				    lcrzo_int32 nboctpacket);
int lcrzo_priv_ecritpacket_datadump(LCRZOFILE *pf, 
				    lcrzo_constdata packet, 
				    lcrzo_int32 nboctpacket)
{ char tabgauche[70], tabdroite[17], tmp[3], c;
  int i, j, poscour;
  
  /*parameters verification*/
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);

  for ( i=0 ; i<=(int)nboctpacket/16 ; i++)
  { poscour=0;
    lcrzo_string_init_text("                                                     ",
		      69, tabgauche);
    for ( j=0 ; j<16 ; j++ )
    { if ( (j+(i<<4)) >= (int)nboctpacket ) break;
      c=packet[j+(i<<4)];
      lcrzo_string_init_int((lcrzo_uint8)c, "%02X", 2, tmp);
      if (!(j%4) && (j!=0)) tabgauche[poscour++]=' ';
      tabgauche[poscour++]=' ';
      tabgauche[poscour++]=tmp[0];
      tabgauche[poscour++]=tmp[1];
      tabdroite[j] = (char)( (c<31 || c>126) ? '.' : c );
    }
    tabdroite[j]='\0';
    if (j) fprintf(pf, "%s   # %s\n", tabgauche, tabdroite);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ecritpacket_datahexmixs(LCRZOFILE *pf, 
				       lcrzo_constdata packet, 
				       lcrzo_int32 nboctpacket,
				       lcrzo_printtype printtypeichage);
int lcrzo_priv_ecritpacket_datahexmixs(LCRZOFILE *pf, 
				       lcrzo_constdata packet, 
				       lcrzo_int32 nboctpacket,
				       lcrzo_printtype printtypeichage)
{ char tabhexa[80], tabmixte[80], c;
  int i, j, guillemetecrit;

  /*parameters verification*/
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);

  for ( i=0 ; i<=(int)nboctpacket/16 ; i++)
  { lcrzo_string_init_text("", 79, tabhexa);
    lcrzo_string_init_text("", 79, tabmixte);
    guillemetecrit=0;
    for ( j=0 ; j<16 ; j++ )
    { if ( (j+(i<<4)) >= (int)nboctpacket ) break;
      c=packet[j+(i<<4)];
      /*ecriture de c sous forme hexa*/
      lcrzo_string_append_char(' ', 1, 79, tabhexa);
      lcrzo_string_append_int((lcrzo_uint8)c, "%02X", 79, tabhexa);
      /*ecriture de c sous forme de hexa/chaine_caractere*/
      if (c<32 || c>126)
      { if (guillemetecrit) 
	{ lcrzo_string_append_char('\'', 1, 79, tabmixte); guillemetecrit=0;
	}
        lcrzo_string_append_char(' ', 1, 79, tabmixte);
        lcrzo_string_append_int((lcrzo_uint8)c, "%02X", 79, tabmixte);
      }
      else
      { if (!guillemetecrit) 
        { lcrzo_string_append_text(" '", 79, tabmixte); guillemetecrit=1;
	}
        if (c=='\'')
	{ lcrzo_string_append_text("''", 79, tabmixte);
	}
	else
        { lcrzo_string_append_char(c, 1, 79, tabmixte);
	}
      }
    }
    if (guillemetecrit) lcrzo_string_append_char('\'', 1, 79, tabmixte);
    if (j)
    { switch(printtypeichage&0xFF)
      { case LCRZO_PRINTTYPE_MIXED_ :
	  fprintf(pf, " %s\n", tabmixte);
          break;
        case LCRZO_PRINTTYPE_CMH_ :
	  fprintf(pf, "#%s\n %s\n", tabmixte, tabhexa);
          break;
        case LCRZO_PRINTTYPE_CHM_ :
	  fprintf(pf, "#%s\n %s\n", tabhexa, tabmixte);
          break;
        default :
          return(LCRZO_ERR_IEINTERNALERROR);
      } 
    }
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ecritpacket_datamixed(LCRZOFILE *pf, 
				     lcrzo_constdata packet, 
				     lcrzo_int32 nboctpacket);
int lcrzo_priv_ecritpacket_datamixed(LCRZOFILE *pf, 
				     lcrzo_constdata packet, 
				     lcrzo_int32 nboctpacket)
{ int i;
  lcrzo_bool guillemetecrit, first;
  lcrzo_uint8 c;

  /*parameters verification*/
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);

  guillemetecrit=0;
  first=1;
  for ( i=0 ; i<nboctpacket ; i++)
  { c=packet[i];
    if (c<32 || c>126)
    { if (guillemetecrit)
      { fputc('\'', pf);
        guillemetecrit=0;
      }
      if (first)
	first=0;
      else
	fputc(' ', pf);
      fprintf(pf, "%02X", c);
    }
    else
    { if (!guillemetecrit) 
      { if (first)
	  first=0;
        else
	  fputc(' ', pf);
        fputc('\'', pf);
        guillemetecrit=1;
      }
      if (c=='\'')
      { fprintf(pf, "''");
      }
      else
      { fputc(c, pf);
      }
    }
  }
  if (guillemetecrit) fputc('\'', pf);

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
void lcrzo_priv_affcarbase64(LCRZOFILE *pf, int nb);
void lcrzo_priv_affcarbase64(LCRZOFILE *pf, int nb)
{ if (nb<26) fprintf(pf, "%c", nb+'A');
  else if (nb<52) fprintf(pf, "%c", (nb-26)+'a');
  else if (nb<62) fprintf(pf, "%c", (nb-52)+'0');
  else if (nb==62) fprintf(pf, "+");
  else if (nb==63) fprintf(pf, "/");
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ecrit_ipopttabipl(LCRZOFILE *pf,
				  lcrzo_uint8 nbiplaffectes,
				  lcrzo_ipl iplt[]);
int lcrzo_priv_ecrit_ipopttabipl(LCRZOFILE *pf,
				  lcrzo_uint8 nbiplaffectes,
				  lcrzo_ipl iplt[])
{ lcrzo_ipa ipa;
  int i;

  if (nbiplaffectes==0) return(LCRZO_ERR_OK);

  for ( i=0; i<nbiplaffectes ; i++ )
  { lcrzo_ipa_init_ipl(iplt[i], ipa);
    if ( i )
    { if ( !(i%4) ) fprintf(pf, "\n |    ");
      else fprintf(pf, ", ");
    }
    else
    { if ( !(i%4) ) fprintf(pf, " |    ");
    }
    lcrzo_ipa_fprint(pf, "", ipa, "");
  }
  fprintf(pf, "\n");
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ecrit_ipopttabtime(LCRZOFILE *pf,
				   lcrzo_uint8 nbiplaffectes,
				   lcrzo_ipl iplt[]);
int lcrzo_priv_ecrit_ipopttabtime(LCRZOFILE *pf,
				   lcrzo_uint8 nbiplaffectes,
				   lcrzo_ipl iplt[])
{ int i;

  if (nbiplaffectes==0) return(LCRZO_ERR_OK);

  for ( i=0; i<nbiplaffectes ; i++ )
  { if ( i )
    { if ( !(i%4) ) fprintf(pf, "\n |    ");
      else fprintf(pf, ", ");
    }
    else
    { if ( !(i%4) ) fprintf(pf, " |    ");
    }
    fprintf(pf, "%ld", iplt[i]);
  }
  fprintf(pf, "\n");
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ecrit_ipopttabiptime(LCRZOFILE *pf,
				     lcrzo_uint8 nbiplaffectes,
				     lcrzo_ipl iplt[]);
int lcrzo_priv_ecrit_ipopttabiptime(LCRZOFILE *pf,
				     lcrzo_uint8 nbiplaffectes,
				     lcrzo_ipl iplt[])
{ lcrzo_ipa ipa;
  int i;

  if (nbiplaffectes==0) return(LCRZO_ERR_OK);

  for ( i=0; i<nbiplaffectes ; i++ )
  { fprintf(pf, " |    ");
    lcrzo_ipa_init_ipl(iplt[2*i], ipa);
    lcrzo_ipa_fprint(pf, "", ipa, " : ");
    fprintf(pf, "%ld\n", iplt[2*i+1]);
  }
  return(LCRZO_ERR_OK);
}
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/



/*-------------------------------------------------------------*/
int lcrzo_printtype_fprint(LCRZOFILE *pf, lcrzo_printtype printtype)
{
  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  /*printf("[%04x]", printtype);*/

  switch(printtype&0xFF)
  { case LCRZO_PRINTTYPE_NOTHING_:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	fprintf(pf, "%s", LCRZO_PRINTTYPE_NOTHING_frtxt);
      else
	fprintf(pf, "%s", LCRZO_PRINTTYPE_NOTHING_ustxt);
      break;
    case LCRZO_PRINTTYPE_SYNTH_:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	fprintf(pf, "%s", LCRZO_PRINTTYPE_SYNTH_frtxt);
      else
	fprintf(pf, "%s", LCRZO_PRINTTYPE_SYNTH_ustxt);
      break;
    case LCRZO_PRINTTYPE_TEXT_:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	fprintf(pf, "%s", LCRZO_PRINTTYPE_TEXT_frtxt);
      else
	fprintf(pf, "%s", LCRZO_PRINTTYPE_TEXT_ustxt);
      break;
    case LCRZO_PRINTTYPE_PTEXT_:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	fprintf(pf, "%s", LCRZO_PRINTTYPE_PTEXT_frtxt);
      else
	fprintf(pf, "%s", LCRZO_PRINTTYPE_PTEXT_ustxt);
      break;
    case LCRZO_PRINTTYPE_BASE64_:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	fprintf(pf, "%s", LCRZO_PRINTTYPE_BASE64_frtxt);
      else
	fprintf(pf, "%s", LCRZO_PRINTTYPE_BASE64_ustxt);
      break;
    case LCRZO_PRINTTYPE_ARRAY_:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	fprintf(pf, "%s", LCRZO_PRINTTYPE_ARRAY_frtxt);
      else
	fprintf(pf, "%s", LCRZO_PRINTTYPE_ARRAY_ustxt);
      break;
    case LCRZO_PRINTTYPE_DUMP_:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	fprintf(pf, "%s", LCRZO_PRINTTYPE_DUMP_frtxt);
      else
	fprintf(pf, "%s", LCRZO_PRINTTYPE_DUMP_ustxt);
      break;
    case LCRZO_PRINTTYPE_HEXA_:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	fprintf(pf, "%s", LCRZO_PRINTTYPE_HEXA_frtxt);
      else
	fprintf(pf, "%s", LCRZO_PRINTTYPE_HEXA_ustxt);
      break;
    case LCRZO_PRINTTYPE_MIXED_:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	fprintf(pf, "%s", LCRZO_PRINTTYPE_MIXED_frtxt);
      else
	fprintf(pf, "%s", LCRZO_PRINTTYPE_MIXED_ustxt);
      break;
    case LCRZO_PRINTTYPE_CMH_:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	fprintf(pf, "%s", LCRZO_PRINTTYPE_CMH_frtxt);
      else
	fprintf(pf, "%s", LCRZO_PRINTTYPE_CMH_ustxt);
      break;
    case LCRZO_PRINTTYPE_CHM_:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	fprintf(pf, "%s", LCRZO_PRINTTYPE_CHM_frtxt);
      else
	fprintf(pf, "%s", LCRZO_PRINTTYPE_CHM_ustxt);
      break;
    default:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
        fprintf(pf, "%s", "Erreur : type d'affichage non reconnu");
      else
        fprintf(pf, "%s", "Error : unknown printtype");
      return(LCRZO_ERR_SPUNKNOWNTYPE);
      break;
  }

  /*error verification*/
  if ( (printtype&0xFF00) & ( ~(LCRZO_PRINTTYPE_C_|LCRZO_PRINTTYPE_S_|
				LCRZO_PRINTTYPE_N_|LCRZO_PRINTTYPE_I_) ) )
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      fprintf(pf, "%s", "Erreur : type d'affichage non reconnu");
    else
      fprintf(pf, "%s", "Error : unknown printtype");
    return(LCRZO_ERR_SPUNKNOWNTYPE);
  }

  if (printtype&LCRZO_PRINTTYPE_C_)
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      fprintf(pf, "%s", LCRZO_PRINTTYPE_C_frtxt);
    else
      fprintf(pf, "%s", LCRZO_PRINTTYPE_C_ustxt);
  }
  if (printtype&LCRZO_PRINTTYPE_S_)
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      fprintf(pf, "%s", LCRZO_PRINTTYPE_S_frtxt);
    else
      fprintf(pf, "%s", LCRZO_PRINTTYPE_S_ustxt);
  }
  if (printtype&LCRZO_PRINTTYPE_N_)
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      fprintf(pf, "%s", LCRZO_PRINTTYPE_N_frtxt);
    else
      fprintf(pf, "%s", LCRZO_PRINTTYPE_N_ustxt);
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_printtype_stdin(const char *message,
			  lcrzo_printtype defautprinttype,
			  lcrzo_printtype *pprinttype)
{ lcrzo_printtype ttypaff[LCRZO_PRIV_PRINTTYPE_COUNT+1];
  lcrzo_bool valeurdefauttrouvee;
  lcrzo_uint32 numeroprinttypechoisi, numeroprinttypedefaut;
  lcrzo_int32 count, i;

  /*initialization*/
  lcrzo_er(lcrzo_printtype_count(&count));
  valeurdefauttrouvee=0;
  if (message!=NULL) if (strlen(message)) printf("%s\n", message);
  numeroprinttypedefaut=ttypaff[0]; /*pour faire plaisir au compilateur*/
  /*affichage des choix possibles*/
  for ( i=1 ; i<=count ; i++ )
  { /*affichage*/
    printf(" %2ld - ", i);
    lcrzo_printtype_value_pos(i, &(ttypaff[i]));
    lcrzo_printtype_print(ttypaff[i]);
    printf("\n");
    /*a t'on trouve la valeur par defaut proposee ?*/
    if ( ttypaff[i] == defautprinttype )
    { valeurdefauttrouvee=1;
      numeroprinttypedefaut=i;
    }
  }

  /*si la valeur par defaut n'a pas ete trouvee*/
  if (!valeurdefauttrouvee)
  { return(LCRZO_ERR_PADEFAULTNOTINRANGE);
  }

  /*demande a entrer le choix voulu*/ 
  lcrzo_er(lcrzo_uint32_stdin((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
			      "Entrez votre choix":"Enter your choice", 
			      1, count, numeroprinttypedefaut,
			      &numeroprinttypechoisi));
  if (pprinttype!=NULL) *pprinttype=ttypaff[numeroprinttypechoisi];
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_printtype_count(lcrzo_int32 *pcount)
{ if (pcount!=NULL) *pcount=LCRZO_PRIV_PRINTTYPE_COUNT;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_printtype_value_pos(lcrzo_int32 pos,
			      lcrzo_printtype *pprinttype)
{ lcrzo_int32 pospos, count;
  lcrzo_printtype printtype;

  lcrzo_er(lcrzo_printtype_count(&count));
  lcrzo_er(lcrzo_priv_manageelempos(count, pos, 0, NULL, &pospos, NULL,NULL));

  switch((int)pospos)
  { case 1:
      printtype=LCRZO_PRINTTYPE_NOTHING;
      break;
    case 2:
      printtype=LCRZO_PRINTTYPE_SYNTH;
      break;
    case 3:
      printtype=LCRZO_PRINTTYPE_SYNTHN;
      break;
    case 4:
      printtype=LCRZO_PRINTTYPE_SYNTHC;
      break;
    case 5:
      printtype=LCRZO_PRINTTYPE_TEXT;
      break;
    case 6:
      printtype=LCRZO_PRINTTYPE_TEXTN;
      break;
    case 7:
      printtype=LCRZO_PRINTTYPE_TEXTC;
      break;
    case 8:
      printtype=LCRZO_PRINTTYPE_PTEXT;
      break;
    case 9:
      printtype=LCRZO_PRINTTYPE_PTEXTN;
      break;
    case 10:
      printtype=LCRZO_PRINTTYPE_PTEXTC;
      break;
    case 11:
      printtype=LCRZO_PRINTTYPE_BASE64;
      break;
    case 12:
      printtype=LCRZO_PRINTTYPE_BASE64N;
      break;
    case 13:
      printtype=LCRZO_PRINTTYPE_BASE64C;
      break;
    case 14:
      printtype=LCRZO_PRINTTYPE_ARRAY;
      break;
    case 15:
      printtype=LCRZO_PRINTTYPE_ARRAYC;
      break;
    case 16:
      printtype=LCRZO_PRINTTYPE_DUMP;
      break;
    case 17:
      printtype=LCRZO_PRINTTYPE_DUMPC;
      break;
    case 18:
      printtype=LCRZO_PRINTTYPE_HEXA;
      break;
    case 19:
      printtype=LCRZO_PRINTTYPE_HEXAN;
      break;
    case 20:
      printtype=LCRZO_PRINTTYPE_HEXAC;
      break;
    case 21:
      printtype=LCRZO_PRINTTYPE_HEXAS;
      break;
    case 22:
      printtype=LCRZO_PRINTTYPE_HEXACS;
      break;
    case 23:
      printtype=LCRZO_PRINTTYPE_MIXED;
      break;
    case 24:
      printtype=LCRZO_PRINTTYPE_MIXEDN;
      break;
    case 25:
      printtype=LCRZO_PRINTTYPE_MIXEDC;
      break;
    case 26:
      printtype=LCRZO_PRINTTYPE_MIXEDS;
      break;
    case 27:
      printtype=LCRZO_PRINTTYPE_MIXEDCS;
      break;
    case 28:
      printtype=LCRZO_PRINTTYPE_CMH;
      break;
    case 29:
      printtype=LCRZO_PRINTTYPE_CMHC;
      break;
    case 30:
      printtype=LCRZO_PRINTTYPE_CMHS;
      break;
    case 31:
      printtype=LCRZO_PRINTTYPE_CMHCS;
      break;
    case 32:
      printtype=LCRZO_PRINTTYPE_CHM;
      break;
    case 33:
      printtype=LCRZO_PRINTTYPE_CHMC;
      break;
    case 34:
      printtype=LCRZO_PRINTTYPE_CHMS;
      break;
    case 35:
      printtype=LCRZO_PRINTTYPE_CHMCS;
      break;
    default:
      return(LCRZO_ERR_POTOOBIG);
  }

  if (pprinttype!=NULL) *pprinttype=printtype;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_printtype_value_first(lcrzo_printtype *pprinttype)
{ return(lcrzo_printtype_value_pos(+1, pprinttype));
}

/*-------------------------------------------------------------*/
int lcrzo_printtype_value_last(lcrzo_printtype *pprinttype)
{ return(lcrzo_printtype_value_pos(-1, pprinttype));
}

/*-------------------------------------------------------------*/
int lcrzo_printtype_loop_range(lcrzo_int32 startpos,
			       lcrzo_int32 endpos,
			       int (*pfonc)(lcrzo_printtype printtype,
					    const void *pinfos),
			       const void *pinfos)
{ lcrzo_int32 count, suppos, infpos, i;
  lcrzo_printtype printtype;

  /*parameters verification*/
  lcrzo_er(lcrzo_printtype_count(&count));
  if (pfonc==NULL) return(LCRZO_ERR_PANULLPTR);
  if ( count==0 && startpos==+1 && endpos==-1 )
    return(LCRZO_ERR_OK);
  /*affectation des positions*/
  lcrzo_er(lcrzo_priv_manageelemrange(count, startpos, endpos,
				      NULL, &infpos, NULL, &suppos,
				      NULL,NULL,NULL,NULL));

  /*verification des parametres et affectation des positions*/
  if (suppos>=infpos)
  { /*loop*/
    for ( i=infpos ; i<=suppos ; i++ )
    { lcrzo_er(lcrzo_printtype_value_pos(i, &printtype));
      lcrzo_er(((*pfonc)(printtype, pinfos)));
    }
  }
  else
  { /*loop*/
    for ( i=infpos ; i>=suppos ; i-- )
    { lcrzo_er(lcrzo_printtype_value_pos(i, &printtype));
      lcrzo_er(((*pfonc)(printtype, pinfos)));
    }
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_printtype_loop_all(int (*pfonc)(lcrzo_printtype printtype,
					  const void *pinfos),
			     const void *pinfos)
{ return(lcrzo_printtype_loop_range(+1, -1, pfonc, pinfos));
}


/*-------------------------------------------------------------*/
lcrzo_printprofile lcrzo_printprofile_returnstruct(lcrzo_printtype hdr, lcrzo_printtype hdreth, lcrzo_printtype hdrip, lcrzo_printtype hdrudp, lcrzo_printtype hdrtcp, lcrzo_printtype hdricmp, lcrzo_printtype hdrarp, lcrzo_printtype data, lcrzo_printtype dataeth, lcrzo_printtype dataip, lcrzo_printtype dataudp, lcrzo_printtype datatcp, lcrzo_printtype dataicmp)
{ lcrzo_printprofile printprofile;

  printprofile.hdr     = hdr;
  printprofile.hdreth  = hdreth;
  printprofile.hdrip   = hdrip;
  printprofile.hdrudp  = hdrudp;
  printprofile.hdrtcp  = hdrtcp;
  printprofile.hdricmp = hdricmp;
  printprofile.hdrarp  = hdrarp;
  printprofile.data     = data;
  printprofile.dataeth  = dataeth;
  printprofile.dataip   = dataip;
  printprofile.dataudp  = dataudp;
  printprofile.datatcp  = datatcp;
  printprofile.dataicmp = dataicmp;
  printprofile.lastused = LCRZO_PRINTTYPE_START;
  return (printprofile);
}

/*-------------------------------------------------------------*/
int lcrzo_printprofile_init(lcrzo_printtype hdr,
			    lcrzo_printtype hdreth,
			    lcrzo_printtype hdrip, 
			    lcrzo_printtype hdrudp,
			    lcrzo_printtype hdrtcp,
			    lcrzo_printtype hdricmp,
			    lcrzo_printtype hdrarp, 
			    lcrzo_printtype data,
			    lcrzo_printtype dataeth,
			    lcrzo_printtype dataip,
			    lcrzo_printtype dataudp,
			    lcrzo_printtype datatcp,
			    lcrzo_printtype dataicmp,
			    lcrzo_printprofile *pprintprofile)
{ lcrzo_printprofile printprofile;

  printprofile.hdr     = hdr;
  printprofile.hdreth  = hdreth;
  printprofile.hdrip   = hdrip;
  printprofile.hdrudp  = hdrudp;
  printprofile.hdrtcp  = hdrtcp;
  printprofile.hdricmp = hdricmp;
  printprofile.hdrarp  = hdrarp;
  printprofile.data     = data;
  printprofile.dataeth  = dataeth;
  printprofile.dataip   = dataip;
  printprofile.dataudp  = dataudp;
  printprofile.datatcp  = datatcp;
  printprofile.dataicmp = dataicmp;
  printprofile.lastused = LCRZO_PRINTTYPE_START;
  if (pprintprofile!=NULL) *pprintprofile=printprofile;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_printprofile_fprint(LCRZOFILE *pf, lcrzo_printprofile printprofile)
{ char msg[20];
  lcrzo_int32 count, i;
  lcrzo_printprofile pp;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  /*initialization*/
  lcrzo_er(lcrzo_printprofile_count(&count));
  /*affichage*/
  printprofile.lastused = LCRZO_PRINTTYPE_START; /*to be able to compare*/
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_printprofile_value_pos(i, &pp);
    /*a t'on trouve la valeur par defaut proposee ?*/
    if ( !memcmp(&pp, &printprofile, sizeof(lcrzo_printprofile)) )
    { switch((int)i)
      { case 1:
          if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_SYNTHSYNTH_frtxt);
	  else
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_SYNTHSYNTH_ustxt);
	  break;
        case 2:
          if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_SYNTHESYNTH_frtxt);
	  else
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_SYNTHESYNTH_ustxt);
	  break;
        case 3:
          if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_ARRAYARRAY_frtxt);
	  else
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_ARRAYARRAY_ustxt);
	  break;
        case 4:
          if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_ARRAYDUMP_frtxt);
	  else
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_ARRAYDUMP_ustxt);
	  break;
        case 5:
          if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_ARRAYCHMS_frtxt);
	  else
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_ARRAYCHMS_ustxt);
	  break;
        case 6:
          if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_HEXANHEXAS_frtxt);
	  else
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_HEXANHEXAS_ustxt);
	  break;
        case 7:
          if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_HEXANDUMP_frtxt);
	  else
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_HEXANDUMP_ustxt);
	  break;
        case 8:
          if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_HEXANCHMS_frtxt);
	  else
	    fprintf(pf, "%s", LCRZO_PRINTPROFILE_HEXANCHMS_ustxt);
	  break;
        default:
          return(LCRZO_ERR_IEINTERNALERROR);
      }
      return(LCRZO_ERR_OK);
    }
  }

  /*if its not standard, print detailled information*/
  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
    lcrzo_string_init_text("Entete", 19, msg);
  else
    lcrzo_string_init_text("Header", 19, msg);
  fprintf(pf, "%ss : ", msg);
  lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.hdr));
  if (printprofile.hdr != printprofile.hdreth)
  { fprintf(pf, "\n%s ETH  : ", msg);
    lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.hdreth));
  }
  if (printprofile.hdr != printprofile.hdrip)
  { fprintf(pf, "\n%s IP   : ", msg);
    lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.hdrip));
  }
  if (printprofile.hdr != printprofile.hdrudp)
  { fprintf(pf, "\n%s UDP  : ", msg);
    lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.hdrudp));
  }
  if (printprofile.hdr != printprofile.hdrtcp)
  { fprintf(pf, "\n%s TCP  : ", msg);
    lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.hdrtcp));
  }
  if (printprofile.hdr != printprofile.hdricmp)
  { fprintf(pf, "\n%s ICMP : ", msg);
    lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.hdricmp));
  }
  if (printprofile.hdr != printprofile.hdrarp)
  { fprintf(pf, "\n%s ARP  : ", msg);
    lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.hdrarp));
  }
  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
    lcrzo_string_init_text("Donnees", 19, msg);
  else
    lcrzo_string_init_text("Data", 19, msg);
  fprintf(pf, "\n%s : ", msg);
  lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.data));
  if (printprofile.data != printprofile.dataeth)
  { fprintf(pf, "\n%s ETH  : ", msg);
    lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.dataeth));
  }
  if (printprofile.data != printprofile.dataip)
  { fprintf(pf, "\n%s IP   : ", msg);
    lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.dataip));
  }
  if (printprofile.data != printprofile.dataudp)
  { fprintf(pf, "\n%s UDP  : ", msg);
    lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.dataudp));
  }
  if (printprofile.data != printprofile.datatcp)
  { fprintf(pf, "\n%s TCP  : ", msg);
    lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.datatcp));
  }
  if (printprofile.data != printprofile.dataicmp)
  { fprintf(pf, "\n%s ICMP : ", msg);
    lcrzo_er(lcrzo_printtype_fprint(pf, printprofile.dataicmp));
  }
  fprintf(pf, "\n");

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_printprofile_stdin(const char *message,
			     lcrzo_printprofile defaultprintprofile,
			     lcrzo_printprofile *ppp)
{ lcrzo_printprofile tpp[LCRZO_PRIV_PRINTPROFILE_COUNT+1], printprofile;
  lcrzo_bool valeurdefauttrouvee;
  lcrzo_uint32 numeroprintprofilechoisi, numeroprintprofiledefaut;
  lcrzo_int32 count, i;
  char c;

  /*initialization*/
  lcrzo_er(lcrzo_printprofile_count(&count));
  valeurdefauttrouvee=0;
  if (message!=NULL) if (strlen(message)) printf("%s\n", message);
  numeroprintprofiledefaut=0; /*pour faire plaisir au compilateur*/

  /*affichage des choix possibles*/
  defaultprintprofile.lastused=LCRZO_PRINTTYPE_START;/*to be able to compare*/
  for ( i=1 ; i<=count ; i++ )
  { /*affichage*/
    printf(" %2ld - ", i);
    lcrzo_printprofile_value_pos(i, &(tpp[i]));
    lcrzo_printprofile_print(tpp[i]);
    printf("\n");
    /*a t'on trouve la valeur par defaut proposee ?*/
    if ( !memcmp(&(tpp[i]), &defaultprintprofile, sizeof(lcrzo_printprofile)) )
    { valeurdefauttrouvee=1;
      numeroprintprofiledefaut=i;
    }
  }

  /*si la valeur par defaut n'a pas ete trouvee*/
  if (!valeurdefauttrouvee)
  { return(LCRZO_ERR_PADEFAULTNOTINRANGE);
  }

  /*demande a entrer le choix voulu*/ 
  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
    printf(" %2ld - profil personnalise\n", count+1);
  else
    printf(" %2ld - personnalized profile\n", count+1);
  lcrzo_er(lcrzo_uint32_stdin((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
			      "Choisissez le profil voulu":
			      "Choose the profile",
			      1, count+1, numeroprintprofiledefaut,
			      &numeroprintprofilechoisi));

  /*standard profile*/
  if ((lcrzo_int32)numeroprintprofilechoisi<=count)
  { if (ppp!=NULL) *ppp=tpp[numeroprintprofilechoisi];
    return(LCRZO_ERR_OK);  
  }

  /*personalized profile*/
  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
  { lcrzo_printtype_stdin("Choisissez le type d'affichage pour les entetes",
			  LCRZO_PRINTTYPE_ARRAY, &printprofile.hdr);
    lcrzo_printtype_stdin("Choisissez le type d'affichage pour les donnees",
			  LCRZO_PRINTTYPE_DUMP, &printprofile.data);
    lcrzo_stdin_char("Voulez-vous affiner la definition du profil",
		     (const lcrzo_uint8 *)"oOnN", 'n', (lcrzo_uint8 *)&c);
  }
  else
  { lcrzo_printtype_stdin("Choose the printtype for headers",
			  LCRZO_PRINTTYPE_ARRAY, &printprofile.hdr);
    lcrzo_printtype_stdin("Choose the printtype for data",
			  LCRZO_PRINTTYPE_DUMP, &printprofile.data);
    lcrzo_stdin_char("Do you want to refine the profile",
		     (const lcrzo_uint8 *)"yYnN", 'n', (lcrzo_uint8 *)&c);
  }
  if ( c=='o' || c=='O' || c=='y' || c=='Y' )
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
    { lcrzo_printtype_stdin("Choisissez le type d'affichage pour les entetes ETH", printprofile.hdr, &printprofile.hdreth);
      lcrzo_printtype_stdin("Choisissez le type d'affichage pour les entetes IP", printprofile.hdr, &printprofile.hdrip);
      lcrzo_printtype_stdin("Choisissez le type d'affichage pour les entetes UDP", printprofile.hdr, &printprofile.hdrudp);
      lcrzo_printtype_stdin("Choisissez le type d'affichage pour les entetes TCP", printprofile.hdr, &printprofile.hdrtcp);
      lcrzo_printtype_stdin("Choisissez le type d'affichage pour les entetes ICMP", printprofile.hdr, &printprofile.hdricmp);
      lcrzo_printtype_stdin("Choisissez le type d'affichage pour les entetes ARP", printprofile.hdr, &printprofile.hdrarp);
      lcrzo_printtype_stdin("Choisissez le type d'affichage pour les donnees ETH", printprofile.data, &printprofile.dataeth);
      lcrzo_printtype_stdin("Choisissez le type d'affichage pour les donnees IP", printprofile.data, &printprofile.dataip);
      lcrzo_printtype_stdin("Choisissez le type d'affichage pour les donnees UDP", printprofile.data, &printprofile.dataudp);
      lcrzo_printtype_stdin("Choisissez le type d'affichage pour les donnees TCP", printprofile.data, &printprofile.datatcp);
      lcrzo_printtype_stdin("Choisissez le type d'affichage pour les donnees ICMP", printprofile.data, &printprofile.dataicmp);
    }
    else
    { lcrzo_printtype_stdin("Choose the printtype for headers ETH", 
			    printprofile.hdr, &printprofile.hdreth);
      lcrzo_printtype_stdin("Choose the printtype for headers IP", 
			    printprofile.hdr, &printprofile.hdrip);
      lcrzo_printtype_stdin("Choose the printtype for headers UDP",
			    printprofile.hdr, &printprofile.hdrudp);
      lcrzo_printtype_stdin("Choose the printtype for headers TCP",
			    printprofile.hdr, &printprofile.hdrtcp);
      lcrzo_printtype_stdin("Choose the printtype for headers ICMP", 
			    printprofile.hdr, &printprofile.hdricmp);
      lcrzo_printtype_stdin("Choose the printtype for headers ARP",
			    printprofile.hdr, &printprofile.hdrarp);
      lcrzo_printtype_stdin("Choose the printtype for data ETH",
			    printprofile.data, &printprofile.dataeth);
      lcrzo_printtype_stdin("Choose the printtype for data IP",
			    printprofile.data, &printprofile.dataip);
      lcrzo_printtype_stdin("Choose the printtype for data UDP",
			    printprofile.data, &printprofile.dataudp);
      lcrzo_printtype_stdin("Choose the printtype for data TCP",
			    printprofile.data, &printprofile.datatcp);
      lcrzo_printtype_stdin("Choose the printtype for data ICMP",
			    printprofile.data, &printprofile.dataicmp);
    }
  }
  else
  { printprofile.hdreth=printprofile.hdr;
    printprofile.hdrip=printprofile.hdr;
    printprofile.hdrudp=printprofile.hdr;
    printprofile.hdrtcp=printprofile.hdr;
    printprofile.hdricmp=printprofile.hdr;
    printprofile.hdrarp=printprofile.hdr;
    printprofile.dataeth=printprofile.data;
    printprofile.dataip=printprofile.data;
    printprofile.dataudp=printprofile.data;
    printprofile.datatcp=printprofile.data;
    printprofile.dataicmp=printprofile.data;
  }
  printprofile.lastused=LCRZO_PRINTTYPE_START;
  if (ppp!=NULL) *ppp=printprofile;
  return(LCRZO_ERR_OK);  
}

/*-------------------------------------------------------------*/
int lcrzo_printprofile_count(lcrzo_int32 *pcount)
{ if (pcount!=NULL) *pcount=LCRZO_PRIV_PRINTPROFILE_COUNT;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_printprofile_value_pos(lcrzo_int32 pos,
				 lcrzo_printprofile *pprintprofile)
{ lcrzo_int32 pospos, count;
  lcrzo_printprofile printprofile;

  lcrzo_er(lcrzo_printprofile_count(&count));
  lcrzo_er(lcrzo_priv_manageelempos(count, pos, 0, NULL, &pospos, NULL,NULL));

  switch((int)pospos)
  { case 1:
      printprofile=LCRZO_PRINTPROFILE_SYNTHSYNTH;
      break;
    case 2:
      printprofile=LCRZO_PRINTPROFILE_SYNTHESYNTH;
      break;
    case 3:
      printprofile=LCRZO_PRINTPROFILE_ARRAYARRAY;
      break;
    case 4:
      printprofile=LCRZO_PRINTPROFILE_ARRAYDUMP;
      break;
    case 5:
      printprofile=LCRZO_PRINTPROFILE_ARRAYCHMS;
      break;
    case 6:
      printprofile=LCRZO_PRINTPROFILE_HEXANHEXAS;
      break;
    case 7:
      printprofile=LCRZO_PRINTPROFILE_HEXANDUMP;
      break;
    case 8:
      printprofile=LCRZO_PRINTPROFILE_HEXANCHMS;
      break;
    default:
      return(LCRZO_ERR_POTOOBIG);
  }

  if (pprintprofile!=NULL) *pprintprofile=printprofile;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_printprofile_value_first(lcrzo_printprofile *pprintprofile)
{ return(lcrzo_printprofile_value_pos(+1, pprintprofile));
}

/*-------------------------------------------------------------*/
int lcrzo_printprofile_value_last(lcrzo_printprofile *pprintprofile)
{ return(lcrzo_printprofile_value_pos(-1, pprintprofile));
}

/*-------------------------------------------------------------*/
int lcrzo_printprofile_loop_range(lcrzo_int32 startpos,
				  lcrzo_int32 endpos,
				  int (*pfonc)(lcrzo_printprofile printprofile,
					       const void *pinfos),
				  const void *pinfos)
{ lcrzo_int32 count, suppos, infpos, i;
  lcrzo_printprofile printprofile;

  /*parameters verification*/
  if (pfonc==NULL) return(LCRZO_ERR_PANULLPTR);
  /*verification des parametres et affectation des positions*/
  lcrzo_er(lcrzo_printprofile_count(&count));
  if ( count==0 && startpos==+1 && endpos==-1 )
    return(LCRZO_ERR_OK);
  lcrzo_er(lcrzo_priv_manageelemrange(count, startpos, endpos,
				      NULL, &infpos, NULL, &suppos,
				      NULL,NULL,NULL,NULL));

  /*verification des parametres et affectation des positions*/
  if (suppos>=infpos)
  { /*loop*/
    for ( i=infpos ; i<=suppos ; i++ )
    { lcrzo_er(lcrzo_printprofile_value_pos(i, &printprofile));
      lcrzo_er(((*pfonc)(printprofile, pinfos)));
    }
  }
  else
  { /*loop*/
    for ( i=infpos ; i>=suppos ; i-- )
    { lcrzo_er(lcrzo_printprofile_value_pos(i, &printprofile));
      lcrzo_er(((*pfonc)(printprofile, pinfos)));
    }
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_printprofile_loop_all(int (*pfonc)(lcrzo_printprofile printprofile,
					  const void *pinfos),
			     const void *pinfos)
{ return(lcrzo_printprofile_loop_range(+1, -1, pfonc, pinfos));
}


/*-------------------------------------------------------------*/
int lcrzo_hs_fprint(LCRZOFILE *pf,
		    const char *deb, const lcrzo_hs hs, const char *fin)
{ lcrzo_hs hs_copie;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_er(lcrzo_string_init_coretext(hs, LCRZO_HS_MAXBYTES, hs_copie));
  if (deb!=NULL) fprintf(pf, "%s", deb);
  fprintf(pf, "%s", hs_copie);
  if (fin!=NULL) fprintf(pf, "%s", fin);
  return(LCRZO_ERR_OK);
}
int lcrzo_ips_fprint(LCRZOFILE *pf,
		     const char *deb, const lcrzo_ips ips, const char *fin)
{ lcrzo_ips ips_copie;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_string_init_coretext(ips, LCRZO_IPS_MAXBYTES, ips_copie));
  if (deb!=NULL) fprintf(pf, "%s", deb);
  fprintf(pf, "%s", ips_copie);
  if (fin!=NULL) fprintf(pf, "%s", fin); 
 return(LCRZO_ERR_OK);
}
int lcrzo_ipa_fprint(LCRZOFILE *pf,
		     const char *deb, const lcrzo_ipa ipa, const char *fin)
{
  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ipa==NULL) return(LCRZO_ERR_PANULLPTR);

  if (deb!=NULL) fprintf(pf, "%s", deb);
  fprintf(pf, "%d.%d.%d.%d", (lcrzo_uint8)ipa[0],
	  (lcrzo_uint8)ipa[1], (lcrzo_uint8)ipa[2], (lcrzo_uint8)ipa[3]);
  if (fin!=NULL) fprintf(pf, "%s", fin);
  return(LCRZO_ERR_OK);
}
int lcrzo_ipl_fprint(LCRZOFILE *pf,
		     const char *deb, lcrzo_ipl ipl, const char *fin)
{ 
  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  if (deb!=NULL) fprintf(pf, "%s", deb);
  fprintf(pf, "%d.%d.%d.%d(%08lx)",
	  (lcrzo_uint8)(ipl>>24),
	  (lcrzo_uint8)((ipl>>16)&0xFF),
	  (lcrzo_uint8)((ipl>>8)&0xFF),
	  (lcrzo_uint8)(ipl&0xFF),
	  ipl);
  if (fin!=NULL) fprintf(pf, "%s", fin);
  return(LCRZO_ERR_OK);
}
int lcrzo_eths_fprint(LCRZOFILE *pf,
		      const char *deb, const lcrzo_eths eths, const char *fin)
{ lcrzo_hs eths_copie;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_string_init_coretext(eths, LCRZO_HS_MAXBYTES, eths_copie));
  if (deb!=NULL) fprintf(pf, "%s", deb);
  fprintf(pf, "%s", eths_copie);
  if (fin!=NULL) fprintf(pf, "%s", fin);
  return(LCRZO_ERR_OK);
}
int lcrzo_etha_fprint(LCRZOFILE *pf,
		      const char *deb, const lcrzo_etha etha, const char *fin)
{
  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  if (deb!=NULL) fprintf(pf, "%s", deb);
  fprintf(pf, "%02X:%02X:%02X:%02X:%02X:%02X", (lcrzo_uint8)etha[0],
	  (lcrzo_uint8)etha[1], (lcrzo_uint8)etha[2],
	  (lcrzo_uint8)etha[3], (lcrzo_uint8)etha[4],
	  (lcrzo_uint8)etha[5]);
  if (fin!=NULL) fprintf(pf, "%s", fin);
 return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_device_fprint(LCRZOFILE *pf,
		   const char *deb, const lcrzo_device device, const char *fin)
{ lcrzo_device device_copie;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_string_init_coretext(device,
				  LCRZO_DEVICE_MAXBYTES, device_copie));
  if (deb!=NULL) fprintf(pf, "%s", deb);
  fprintf(pf, "%s", device_copie);
  if (fin!=NULL) fprintf(pf, "%s", fin);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hs_fprint_infos(LCRZOFILE *pf, const lcrzo_hs hs)
{ lcrzo_ipl ipl;
  lcrzo_eths eths;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_er(lcrzo_hs_fprint(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
			   "nom de machine :    ":"computer name :     ",
			   hs, "\n"));

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	            " adresse ip :       ":" ip address :       ");
  retour=lcrzo_ipl_init_hs(hs, &ipl);
  if (retour==LCRZO_ERR_OK)
    lcrzo_ipl_fprint(pf, "", ipl, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolue\n":"unresolved\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " adresse ethernet : ":" ethernet address : ");
  if (retour==LCRZO_ERR_OK) retour=lcrzo_eths_init_ipl(ipl, eths);
  if (retour==LCRZO_ERR_OK)
    lcrzo_eths_fprint(pf, "", eths, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolue\n":"unresolved\n");

  lcrzo_err_purge();
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ips_fprint_infos(LCRZOFILE *pf, const lcrzo_ips ips)
{ lcrzo_hs hs;
  lcrzo_ipl ipl;
  lcrzo_eths eths;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_er(lcrzo_ips_fprint(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
			    "adresse ip :        ":"ip address :        ",
			    ips, "\n"));

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " nom de machine :   ":" computer name :    ");
  retour=lcrzo_hs_init_ips(ips, hs);
  if (retour==LCRZO_ERR_OK)
    lcrzo_hs_fprint(pf, "", hs, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolu\n":"unresolved\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " adresse ip :       ":" ip address :       ");
  retour=lcrzo_ipl_init_ips(ips, &ipl);
  if (retour==LCRZO_ERR_OK)
    lcrzo_ipl_fprint(pf, "", ipl, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolue\n":"unresolved\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " adresse ethernet : ":" ethernet address : ");
  retour=lcrzo_eths_init_ips(ips, eths);
  if (retour==LCRZO_ERR_OK)
    lcrzo_eths_fprint(pf, "", eths, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolue\n":"unresolved\n");

  lcrzo_err_purge();
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipa_fprint_infos(LCRZOFILE *pf, const lcrzo_ipa ipa)
{ lcrzo_hs hs;
  lcrzo_ipl ipl;
  lcrzo_eths eths;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_ipa_fprint(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   "adresse ip :        ":"ip address :        ", 
		   ipa, "\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " nom de machine :   ":" computer name :    ");
  retour=lcrzo_hs_init_ipa(ipa, hs);
  if (retour==LCRZO_ERR_OK)
    lcrzo_hs_fprint(pf, "", hs, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolu\n":"unresolved\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " adresse ip :       ":" ip address :       ");
  retour=lcrzo_ipl_init_ipa(ipa, &ipl);
  if (retour==LCRZO_ERR_OK)
    lcrzo_ipl_fprint(pf, "", ipl, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolue\n":"unresolved\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " adresse ethernet : ":" ethernet address : ");
  retour=lcrzo_eths_init_ipa(ipa, eths);
  if (retour==LCRZO_ERR_OK)
    lcrzo_eths_fprint(pf, "", eths, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolue\n":"unresolved\n");

  lcrzo_err_purge();
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipl_fprint_infos(LCRZOFILE *pf, lcrzo_ipl ipl)
{ lcrzo_hs hs;
  lcrzo_eths eths;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_ipl_fprint(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   "adresse ip :        ":"ip address :        ", 
		   ipl, "\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " nom de machine :   ":" computer name :    ");
  retour=lcrzo_hs_init_ipl(ipl, hs);
  if (retour==LCRZO_ERR_OK)
    lcrzo_hs_fprint(pf, "", hs, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolu\n":"unresolved\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " adresse ethernet : ":" ethernet address : ");
  retour=lcrzo_eths_init_ipl(ipl, eths);
  if (retour==LCRZO_ERR_OK)
    lcrzo_eths_fprint(pf, "", eths, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolue\n":"unresolved\n");

  lcrzo_err_purge();
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_eths_fprint_infos(LCRZOFILE *pf, const lcrzo_eths eths)
{ lcrzo_hs hs;
  lcrzo_ipl ipl;
  int retour1, retour2;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_er(lcrzo_eths_fprint(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
			     "adresse ethernet :  ":"ethernet address :  ",
			     eths, "\n"));

  retour1=lcrzo_ipl_init_eths(eths, &ipl);
  if (retour1==LCRZO_ERR_OK) retour2=lcrzo_hs_init_ipl(ipl, hs);
  else retour2=retour1;

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " nom de machine :   ":" computer name :    ");
  if (retour2==LCRZO_ERR_OK)
    lcrzo_hs_fprint(pf, "", hs, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolu\n":"unresolved\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " adresse ip :       ":" ip address :       ");
  if (retour1==LCRZO_ERR_OK)
    lcrzo_ipl_fprint(pf, "", ipl, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolue\n":"unresolved\n");

  lcrzo_err_purge();
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_etha_fprint_infos(LCRZOFILE *pf, const lcrzo_etha etha)
{ lcrzo_hs hs;
  lcrzo_ipl ipl;
  int retour1, retour2;
	      
  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_etha_fprint(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		    "adresse ethernet :  ":"ethernet address :  ",
		    etha, "\n");

  retour1=lcrzo_ipl_init_etha(etha, &ipl);
  if (retour1==LCRZO_ERR_OK) retour2=lcrzo_hs_init_ipl(ipl, hs);
  else retour2=retour1;


  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " nom de machine :   ":" computer name :    ");
  if (retour2==LCRZO_ERR_OK)
    lcrzo_hs_fprint(pf, "", hs, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolu\n":"unresolved\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " adresse ip :       ":" ip address :       ");
  if (retour2==LCRZO_ERR_OK)
    lcrzo_ipl_fprint(pf, "", ipl, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolue\n":"unresolved\n");
  
  lcrzo_err_purge();
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_device_fprint_infos(LCRZOFILE *pf, const lcrzo_device device)
{ lcrzo_hs hs;
  lcrzo_ipl ipl, mask;
  lcrzo_etha etha;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_device_fprint(pf, "device :            ", device, "\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " nom de machine :   ":" computer name :    ");
  retour=lcrzo_hs_init_device(device, hs);
  if (retour==LCRZO_ERR_OK)
    lcrzo_hs_fprint(pf, "", hs, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolu\n":"unresolved\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " adresse ip :       ":" ip address :       ");
  retour=lcrzo_ipl_init_device(device, &ipl, &mask);
  if (retour==LCRZO_ERR_OK)
  { lcrzo_ipl_fprint(pf, "", ipl, "");
    lcrzo_ipl_fprint(pf, "/", mask, "\n");
  }
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolue\n":"unresolved\n");

  fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		   " adresse ethernet : ":" ethernet address : ");
  retour=lcrzo_etha_init_device(device, etha);
  if (retour==LCRZO_ERR_OK)
    lcrzo_etha_fprint(pf, "", etha, "\n");
  else
    fprintf(pf, "%s", (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	             "non resolue\n":"unresolved\n");
    
  lcrzo_err_purge();
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_data_fprint(LCRZOFILE *pf, 
		      lcrzo_constdata data,
		      lcrzo_int32 datasize,
		      lcrzo_printtype printtype)
{ lcrzo_int32 i, nbblocscomplets, nbcaracrestant;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  if (data==NULL && datasize) return(LCRZO_ERR_SPNULLPTRSIZE);

  switch( printtype & ( 0x00FF | LCRZO_PRINTTYPE_S_ ) )
  { case LCRZO_PRINTTYPE_NOTHING_ :
      break;
    case LCRZO_PRINTTYPE_SYNTH_ :
      fprintf(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
	      "%lu octets":"%lu bytes", datasize);
      break;
    case LCRZO_PRINTTYPE_TEXT_ :
      for ( i=0 ; i<datasize ; i++ ) fputc(data[i], pf);
      break;
    case LCRZO_PRINTTYPE_PTEXT_ :
      for ( i=0 ; i<datasize ; i++ ) 
	fputc((data[i]<31 || data[i]>126) ? '.' : data[i], pf);
      break;
    case LCRZO_PRINTTYPE_BASE64_ :
      nbblocscomplets=datasize/3;
      nbcaracrestant=datasize-3*nbblocscomplets;
      /*traite les blocs complets*/
      for ( i=0 ; i<nbblocscomplets ; i++ )
      { lcrzo_priv_affcarbase64(pf, data[3*i]>>2 );
	lcrzo_priv_affcarbase64(pf, ((data[3*i]&3)<<4) |
				 (data[3*i+1]>>4) );
	lcrzo_priv_affcarbase64(pf, ((data[3*i+1]&0xF)<<2) | 
				 (data[3*i+2]>>6) );
	lcrzo_priv_affcarbase64(pf, data[3*i+2]&0x3F );
      }
      /*traite selon le nombre de caracteres restants*/
      if (nbcaracrestant==1)
      { lcrzo_priv_affcarbase64(pf, data[3*i]>>2 );
	lcrzo_priv_affcarbase64(pf, ((data[3*i]&3)<<4) );
	fprintf(pf, "==");
      }
      else if (nbcaracrestant==2)
      { lcrzo_priv_affcarbase64(pf, data[3*i]>>2 );
	lcrzo_priv_affcarbase64(pf, ((data[3*i]&3)<<4) |
				 (data[3*i+1]>>4) );
	lcrzo_priv_affcarbase64(pf, ((data[3*i+1]&0xF)<<2) );
	fprintf(pf, "=");
      }
      break;
    case LCRZO_PRINTTYPE_ARRAY_ :
      if (datasize)
      { fprintf(pf, " .___________________________________");
	fprintf(pf, "____________________________________.\n");
      }
      for ( i=0 ; i<datasize/4 ; i++ )
      { fprintf(pf, " |");
        lcrzo_er(lcrzo_priv_ecrit_tab08(pf, data[4*i], 
				     LCRZO_PRIV_HEXADECCAR));
	fprintf(pf, "|");
        lcrzo_er(lcrzo_priv_ecrit_tab08(pf, data[4*i+1], 
				     LCRZO_PRIV_HEXADECCAR));
	fprintf(pf, "|");
        lcrzo_er(lcrzo_priv_ecrit_tab08(pf, data[4*i+2], 
				     LCRZO_PRIV_HEXADECCAR));
 	fprintf(pf, "|");
        lcrzo_er(lcrzo_priv_ecrit_tab08(pf, data[4*i+3], 
				     LCRZO_PRIV_HEXADECCAR));
        fprintf(pf, "|\n");
      }
      if (datasize%4 == 3)
      { fprintf(pf, " |");
        lcrzo_er(lcrzo_priv_ecrit_tab08(pf, data[4*i], 
				     LCRZO_PRIV_HEXADECCAR));
	fprintf(pf, "|");
        lcrzo_er(lcrzo_priv_ecrit_tab08(pf, data[4*i+1], 
				     LCRZO_PRIV_HEXADECCAR));
	fprintf(pf, "|");
        lcrzo_er(lcrzo_priv_ecrit_tab08(pf, data[4*i+2], 
				     LCRZO_PRIV_HEXADECCAR));
        fprintf(pf, "|\n");
      }
      else if (datasize%4 == 2)
      { fprintf(pf, " |");
        lcrzo_er(lcrzo_priv_ecrit_tab08(pf, data[4*i], 
				     LCRZO_PRIV_HEXADECCAR));
	fprintf(pf, "|");
        lcrzo_er(lcrzo_priv_ecrit_tab08(pf, data[4*i+1], 
				     LCRZO_PRIV_HEXADECCAR));
        fprintf(pf, "|\n");
      }
      else if (datasize%4 == 1)
      { fprintf(pf, " |");
        lcrzo_er(lcrzo_priv_ecrit_tab08(pf, data[4*i], 
				     LCRZO_PRIV_HEXADECCAR));
        fprintf(pf, "|\n");
      }
      break;
    case LCRZO_PRINTTYPE_DUMP_ :
      lcrzo_priv_ecritpacket_datadump(pf, data, datasize);
      break;
    case LCRZO_PRINTTYPE_HEXA_ :
      for ( i=0 ; i<datasize ; i++ ) fprintf(pf, "%02X", data[i]);
      break;
    case LCRZO_PRINTTYPE_HEXAS&(~LCRZO_PRINTTYPE_I_) :
      for ( i=0 ; i<datasize ; )
      { fprintf(pf, " %02X", data[i]);
        i++;
	if ( !(i%24) ) fprintf(pf, "\n");
      }
      if (i%24) fprintf(pf, "\n");
      break;
    case LCRZO_PRINTTYPE_MIXEDS&(~LCRZO_PRINTTYPE_I_) :
    case LCRZO_PRINTTYPE_CMHS&(~LCRZO_PRINTTYPE_I_) :
    case LCRZO_PRINTTYPE_CHMS&(~LCRZO_PRINTTYPE_I_) :
      lcrzo_er(lcrzo_priv_ecritpacket_datahexmixs(pf, data, datasize, 
						  printtype));
      break;
    case LCRZO_PRINTTYPE_MIXED_ :
      lcrzo_er(lcrzo_priv_ecritpacket_datamixed(pf, data, datasize));
      break;
    case LCRZO_PRINTTYPE_CMH_ :
      fprintf(pf, "# ");
      lcrzo_er(lcrzo_priv_ecritpacket_datamixed(pf, data, datasize));
      fprintf(pf, "\n  ");
      for ( i=0 ; i<datasize ; i++ ) fprintf(pf, "%02X", data[i]);
      fprintf(pf, "\n");
      break;
    case LCRZO_PRINTTYPE_CHM_ :
      fprintf(pf, "# ");
      for ( i=0 ; i<datasize ; i++ ) fprintf(pf, "%02X", data[i]);
      fprintf(pf, "\n ");
      lcrzo_er(lcrzo_priv_ecritpacket_datamixed(pf, data, datasize));
      fprintf(pf, "\n");
      break;
      break;
    default :
      return(LCRZO_ERR_IEINTERNALERROR);
      break;
  }

  /*eventually print a newline*/
  lcrzo_er(lcrzo_priv_printtype_printnewline(pf, printtype));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hdrleth_fprint(LCRZOFILE *pf,
			lcrzo_hdrleth hdrleth, lcrzo_printtype printtype)
{
  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  /*affichage eventuel du commentaire*/
  lcrzo_er(lcrzo_priv_printtype_printcomment(pf, printtype, "Ethernet header",
					     "Entete Ethernet"));

  /*affichage de l'entete*/
  switch(printtype&0xFF)
  { case LCRZO_PRINTTYPE_NOTHING_ :
      break;
    case LCRZO_PRINTTYPE_SYNTH_ :
      fprintf(pf, "%02x:%02x:%02x:%02x:%02x:%02x", 
	      (lcrzo_uint8)hdrleth.src[0],  (lcrzo_uint8)hdrleth.src[1],
	      (lcrzo_uint8)hdrleth.src[2], (lcrzo_uint8)hdrleth.src[3], 
	      (lcrzo_uint8)hdrleth.src[4], (lcrzo_uint8)hdrleth.src[5]);
      fprintf(pf, "->%02x:%02x:%02x:%02x:%02x:%02x",
	      (lcrzo_uint8)hdrleth.dst[0],  (lcrzo_uint8)hdrleth.dst[1],
	      (lcrzo_uint8)hdrleth.dst[2], (lcrzo_uint8)hdrleth.dst[3], 
	      (lcrzo_uint8)hdrleth.dst[4], (lcrzo_uint8)hdrleth.dst[5]);
      break;
    case LCRZO_PRINTTYPE_TEXT_ :
    case LCRZO_PRINTTYPE_PTEXT_ :
    case LCRZO_PRINTTYPE_BASE64_ :
    case LCRZO_PRINTTYPE_DUMP_ :
    case LCRZO_PRINTTYPE_HEXA_ :
    case LCRZO_PRINTTYPE_MIXED_ :
    case LCRZO_PRINTTYPE_CMH_ :
    case LCRZO_PRINTTYPE_CHM_ :
      { lcrzo_hdrpeth hdrpeth;
        lcrzo_er(lcrzo_hdrpeth_init_hdrleth(hdrleth, &hdrpeth));
	lcrzo_er(lcrzo_data_fprint(pf, (lcrzo_uint8 *)&hdrpeth,
					  sizeof(hdrpeth),
					  printtype));
	return(LCRZO_ERR_OK); /*return here to not print newline twice*/
      }
      break;
    case LCRZO_PRINTTYPE_ARRAY_ :
      fprintf(pf, " ETH________________________________________");
      fprintf(pf, "_____________________________.\n");
      fprintf(pf, " | %02x:%02x:%02x:%02x:%02x:%02x", 
	      (lcrzo_uint8)hdrleth.src[0],  (lcrzo_uint8)hdrleth.src[1],
	      (lcrzo_uint8)hdrleth.src[2], (lcrzo_uint8)hdrleth.src[3], 
	      (lcrzo_uint8)hdrleth.src[4], (lcrzo_uint8)hdrleth.src[5]);
      fprintf(pf, " vers %02x:%02x:%02x:%02x:%02x:%02x",
	      (lcrzo_uint8)hdrleth.dst[0],  (lcrzo_uint8)hdrleth.dst[1],
	      (lcrzo_uint8)hdrleth.dst[2], (lcrzo_uint8)hdrleth.dst[3], 
	      (lcrzo_uint8)hdrleth.dst[4], (lcrzo_uint8)hdrleth.dst[5]);
      fprintf(pf, "         type : 0x%04x        |\n", hdrleth.type);
      fprintf(pf, " |_______________________________________________");
      fprintf(pf, "________________________|\n");
      break;
    default :
      return(LCRZO_ERR_IEINTERNALERROR);
  }

  /*eventually print a newline*/
  lcrzo_er(lcrzo_priv_printtype_printnewline(pf, printtype));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hdrlip_fprint(LCRZOFILE *pf,
		       lcrzo_hdrlip hdrlip, lcrzo_printtype printtype)
{ lcrzo_ips ips;
  char tab[20]; /*va servir a tabuler les adresses ip*/
  int posgauche;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  /*affichage eventuel du commentaire*/
  lcrzo_er(lcrzo_priv_printtype_printcomment(pf, printtype,"IP header","Entete IP"));

  /*affichage de l'entete*/
  switch(printtype&0xFF)
  { case LCRZO_PRINTTYPE_NOTHING_ :
      break;
    case LCRZO_PRINTTYPE_SYNTH_ :
      lcrzo_er(lcrzo_ips_init_ipl(hdrlip.saddr, ips));
      fprintf(pf, "%s", ips);
      lcrzo_er(lcrzo_ips_init_ipl(hdrlip.daddr, ips));
      fprintf(pf, "->%s", ips);
      break;
    case LCRZO_PRINTTYPE_TEXT_ :
    case LCRZO_PRINTTYPE_PTEXT_ :
    case LCRZO_PRINTTYPE_BASE64_ :
    case LCRZO_PRINTTYPE_DUMP_ :
    case LCRZO_PRINTTYPE_HEXA_ :
    case LCRZO_PRINTTYPE_MIXED_ :
    case LCRZO_PRINTTYPE_CMH_ :
    case LCRZO_PRINTTYPE_CHM_ :
      { lcrzo_hdrpip hdrpip;
        lcrzo_er(lcrzo_hdrpip_init_hdrlip(hdrlip, &hdrpip));
	lcrzo_er(lcrzo_data_fprint(pf, (lcrzo_uint8 *)&hdrpip,
					sizeof(hdrpip),
					printtype));
        return(LCRZO_ERR_OK); /*return here to not print newline twice*/
      }
      break;
    case LCRZO_PRINTTYPE_ARRAY_ :
      fprintf(pf, " IP____________________________________________");
      fprintf(pf, "__________________________.\n");
      fprintf(pf, " |version |  ihl   |       tos       |");
      fprintf(pf, "              totlen               |\n");
      fprintf(pf, " |");
      lcrzo_er(lcrzo_priv_ecrit_tab04(pf, hdrlip.version,
				       LCRZO_PRIV_DEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab04(pf, hdrlip.ihl, LCRZO_PRIV_DEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab08(pf, hdrlip.tos, LCRZO_PRIV_DEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrlip.totlen,
				   LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|\n");
      fprintf(pf, " |                id                 |");
      fprintf(pf, "xxDfMf         fragoffset          |\n");
      fprintf(pf, " |");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrlip.id,
				   LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab01(pf, hdrlip.reserve));
      lcrzo_er(lcrzo_priv_ecrit_tab01(pf, hdrlip.dontfrag));
      lcrzo_er(lcrzo_priv_ecrit_tab01(pf, hdrlip.morefrag));
      fprintf(pf, "_________%04Xh=%5d_________|\n", hdrlip.offsetfrag,
	      hdrlip.offsetfrag);
      fprintf(pf, " |       ttl       |    protocol     |");
      fprintf(pf, "          header checksum          |\n");
      fprintf(pf, " |");
      lcrzo_er(lcrzo_priv_ecrit_tab08(pf, hdrlip.ttl,
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab08(pf, hdrlip.protocol,
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrlip.check, 
				   LCRZO_PRIV_HEXA));
      fprintf(pf, "|\n");
      fprintf(pf, " |                                source");
      fprintf(pf, "                                 |\n");
      lcrzo_er(lcrzo_ips_init_ipl(hdrlip.saddr, ips));
      posgauche=(15-strlen(ips))/2;
      lcrzo_string_init_text("_______________", posgauche, tab);
      lcrzo_string_append_text(ips, 19, tab);
      lcrzo_string_append_text("_____________", 19, tab);
      tab[15]=0;
      fprintf(pf, " |_____________________________%s", tab);
      fprintf(pf, "___________________________|\n");
      fprintf(pf, " |                              destination");
      fprintf(pf, "                              |\n");
      lcrzo_er(lcrzo_ips_init_ipl(hdrlip.daddr, ips));
      posgauche=(15-strlen(ips))/2;
      lcrzo_string_init_text("_______________", posgauche, tab);
      lcrzo_string_append_text(ips, 19, tab);
      lcrzo_string_append_text("_____________", 19, tab);
      tab[15]=0;
      fprintf(pf, " |_____________________________%s", tab);
      fprintf(pf, "___________________________|\n");
      break;
    default :
      return(LCRZO_ERR_IEINTERNALERROR);
  }

  /*eventually print a newline*/
  lcrzo_er(lcrzo_priv_printtype_printnewline(pf, printtype));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hdrludp_fprint(LCRZOFILE *pf, 
			lcrzo_hdrludp hdrludp, lcrzo_printtype printtype)
{
  /*parameters verification*/

  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  /*affichage eventuel du commentaire*/
  lcrzo_er(lcrzo_priv_printtype_printcomment(pf, printtype, "UDP header", 
				       "Entete UDP"));

  /*affichage de l'entete*/
  switch(printtype&0xFF)
  { case LCRZO_PRINTTYPE_NOTHING_ :
      break;
    case LCRZO_PRINTTYPE_SYNTH_ :
      fprintf(pf, "UDP%d->%d",
	      hdrludp.sport, hdrludp.dport);
      break;
    case LCRZO_PRINTTYPE_TEXT_ :
    case LCRZO_PRINTTYPE_PTEXT_ :
    case LCRZO_PRINTTYPE_BASE64_ :
    case LCRZO_PRINTTYPE_DUMP_ :
    case LCRZO_PRINTTYPE_HEXA_ :
    case LCRZO_PRINTTYPE_MIXED_ :
    case LCRZO_PRINTTYPE_CMH_ :
    case LCRZO_PRINTTYPE_CHM_ :
      { lcrzo_hdrpudp hdrpudp;
        lcrzo_er(lcrzo_hdrpudp_init_hdrludp(hdrludp, &hdrpudp));
	lcrzo_er(lcrzo_data_fprint(pf, (lcrzo_uint8 *)&hdrpudp,
					sizeof(hdrpudp),
					printtype));
        return(LCRZO_ERR_OK); /*return here to not print newline twice*/
      }
      break;
    case LCRZO_PRINTTYPE_ARRAY_ :
      fprintf(pf, " UDP_________________________________");
      fprintf(pf, "____________________________________.\n");
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      { fprintf(pf, " |            port source            |");
        fprintf(pf, "         port destination          |\n");
      }
      else
      { fprintf(pf, " |            source port            |");
        fprintf(pf, "         destination port          |\n");
      }
      fprintf(pf, " |");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrludp.sport, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrludp.dport, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|\n");
      fprintf(pf, " |              length               |");
      fprintf(pf, "             checksum              |\n");
      fprintf(pf, " |");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrludp.len, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrludp.check, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|\n");
      break;
    default :
      return(LCRZO_ERR_IEINTERNALERROR);
  }

  /*eventually print a newline*/
  lcrzo_er(lcrzo_priv_printtype_printnewline(pf, printtype));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hdrltcp_fprint(LCRZOFILE *pf, 
			lcrzo_hdrltcp hdrltcp, lcrzo_printtype printtype)
{
  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  /*affichage eventuel du commentaire*/
  lcrzo_er(lcrzo_priv_printtype_printcomment(pf, printtype, "TCP header",
				       "Entete TCP"));

  /*affichage de l'entete*/
  switch(printtype&0xFF)
  { case LCRZO_PRINTTYPE_NOTHING_ :
      break;
    case LCRZO_PRINTTYPE_SYNTH_ :
      fprintf(pf, "TCP%d->%d",
	      hdrltcp.sport, hdrltcp.dport);
      break;
    case LCRZO_PRINTTYPE_TEXT_ :
    case LCRZO_PRINTTYPE_PTEXT_ :
    case LCRZO_PRINTTYPE_BASE64_ :
    case LCRZO_PRINTTYPE_DUMP_ :
    case LCRZO_PRINTTYPE_HEXA_ :
    case LCRZO_PRINTTYPE_MIXED_ :
    case LCRZO_PRINTTYPE_CMH_ :
    case LCRZO_PRINTTYPE_CHM_ :
      { lcrzo_hdrptcp hdrptcp;
        lcrzo_er(lcrzo_hdrptcp_init_hdrltcp(hdrltcp, &hdrptcp));
	lcrzo_er(lcrzo_data_fprint(pf, (lcrzo_uint8 *)&hdrptcp,
					sizeof(hdrptcp),
					printtype));
        return(LCRZO_ERR_OK); /*return here to not print newline twice*/
      }
      break;
    case LCRZO_PRINTTYPE_ARRAY_ :
      fprintf(pf, " TCP______________________________");
      fprintf(pf, "_______________________________________.\n");
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
      { fprintf(pf, " |            port source            |");
        fprintf(pf, "         port destination          |\n");
      }
      else
      { fprintf(pf, " |            source port            |");
        fprintf(pf, "         destination port          |\n");
      }
      fprintf(pf, " |");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrltcp.sport, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrltcp.dport, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|\n");
      fprintf(pf, " |                                seq");
      fprintf(pf, " num                                |\n");
      fprintf(pf, " |");
      lcrzo_er(lcrzo_priv_ecrit_tab32(pf, hdrltcp.seqnum, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|\n");
      fprintf(pf, " |                                ack ");
      fprintf(pf, "num                                |\n");
      fprintf(pf, " |");
      lcrzo_er(lcrzo_priv_ecrit_tab32(pf, hdrltcp.acknum, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|\n");
      fprintf(pf, " |dataoff |         . . UrAk PuRsSyFi|");
      fprintf(pf, "               window              |\n");
      fprintf(pf, " |");
      lcrzo_er(lcrzo_priv_ecrit_tab04(pf, hdrltcp.doff, 
				       LCRZO_PRIV_DEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab04(pf, (lcrzo_uint8)(hdrltcp.reserve>>2), 
				       LCRZO_PRIV_DEC));
      fprintf(pf, "_");
      lcrzo_er(lcrzo_priv_ecrit_tab01(pf,
				       (lcrzo_uint8)(1&(hdrltcp.reserve>>1))));
      lcrzo_er(lcrzo_priv_ecrit_tab01(pf, (lcrzo_uint8)(1&hdrltcp.reserve)));
      lcrzo_er(lcrzo_priv_ecrit_tab01(pf, hdrltcp.urg));
      lcrzo_er(lcrzo_priv_ecrit_tab01(pf, hdrltcp.ack));
      fprintf(pf, "_");
      lcrzo_er(lcrzo_priv_ecrit_tab01(pf, hdrltcp.psh));
      lcrzo_er(lcrzo_priv_ecrit_tab01(pf, hdrltcp.rst));
      lcrzo_er(lcrzo_priv_ecrit_tab01(pf, hdrltcp.syn));
      lcrzo_er(lcrzo_priv_ecrit_tab01(pf, hdrltcp.fin));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrltcp.window, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|\n");
      fprintf(pf, " |             checksum              |");
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
        fprintf(pf, "          pointeur urgent          |\n");
      else
        fprintf(pf, "          urgent pointer           |\n");
      fprintf(pf, " |");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrltcp.check, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrltcp.urgptr, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|\n");
      break;
    default :
      return(LCRZO_ERR_IEINTERNALERROR);
  }

  /*eventually print a newline*/
  lcrzo_er(lcrzo_priv_printtype_printnewline(pf, printtype));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hdrlicmp_fprint(LCRZOFILE *pf, 
			 lcrzo_hdrlicmp hdrlicmp, lcrzo_printtype printtype)
{
  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  /*affichage eventuel du commentaire*/
  lcrzo_er(lcrzo_priv_printtype_printcomment(pf, printtype, "ICMP header",
				       "Entete ICMP"));

  /*affichage de l'entete*/
  switch(printtype&0xFF)
  { case LCRZO_PRINTTYPE_NOTHING_ :
      break;
    case LCRZO_PRINTTYPE_SYNTH_ :
      fprintf(pf, "ICMP%d",  hdrlicmp.type);
      break;
    case LCRZO_PRINTTYPE_TEXT_ :
    case LCRZO_PRINTTYPE_PTEXT_ :
    case LCRZO_PRINTTYPE_BASE64_ :
    case LCRZO_PRINTTYPE_DUMP_ :
    case LCRZO_PRINTTYPE_HEXA_ :
    case LCRZO_PRINTTYPE_MIXED_ :
    case LCRZO_PRINTTYPE_CMH_ :
    case LCRZO_PRINTTYPE_CHM_ :
      { lcrzo_hdrpicmp hdrpicmp;
        lcrzo_er(lcrzo_hdrpicmp_init_hdrlicmp(hdrlicmp, &hdrpicmp));
	lcrzo_er(lcrzo_data_fprint(pf, (lcrzo_uint8 *)&hdrpicmp,
					sizeof(hdrpicmp),
					printtype));
        return(LCRZO_ERR_OK); /*return here to not print newline twice*/
      }
      break;
    case LCRZO_PRINTTYPE_ARRAY_ :
      fprintf(pf, " ICMP_______________________________");
      fprintf(pf, "_____________________________________.\n");
      fprintf(pf, " |      type       |      code       |");
      fprintf(pf, "             checksum              |\n");
      fprintf(pf, " |");
      lcrzo_er(lcrzo_priv_ecrit_tab08(pf, hdrlicmp.type, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab08(pf, hdrlicmp.code, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|");
      lcrzo_er(lcrzo_priv_ecrit_tab16(pf, hdrlicmp.check, 
				       LCRZO_PRIV_HEXADEC));
      fprintf(pf, "|\n");
      break;
    default :
      return(LCRZO_ERR_IEINTERNALERROR);
  }

  /*eventually print a newline*/
  lcrzo_er(lcrzo_priv_printtype_printnewline(pf, printtype));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hdrlarp_fprint(LCRZOFILE *pf,
			lcrzo_hdrlarp hdrlarp, lcrzo_printtype printtype)
{ int i;
  char tab[40]; /* pour tabuler le | de droite*/
  char chpsrc[40], chpdst[40];

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  /*affichage eventuel du commentaire*/
  lcrzo_er(lcrzo_priv_printtype_printcomment(pf, printtype, "ARP header",
				       "Entete ARP"));

  /*affichage de l'entete*/
  switch(printtype&0xFF)
  { case LCRZO_PRINTTYPE_NOTHING_ :
      break;
    case LCRZO_PRINTTYPE_SYNTH_ :
      switch(hdrlarp.op)
      { case 1  : fprintf(pf, "ARPRequest"); 
                  lcrzo_ipa_fprint(pf, "_", hdrlarp.prot_dst, "\n");
		  break;
        case 2  : fprintf(pf, "ARPReply");
                  lcrzo_etha_fprint(pf, "_", hdrlarp.hw_src, "\n");
		  break;
        case 3  : fprintf(pf, "RARPRequest"); 
                  lcrzo_etha_fprint(pf, "_", hdrlarp.hw_dst, "\n");
		  break;
        case 4  : fprintf(pf, "RARPReply"); 
                  lcrzo_ipa_fprint(pf, "_", hdrlarp.prot_dst, "\n");
		  break;
        default : fprintf(pf, "ARP??");
		  break;
      }
      break;
    case LCRZO_PRINTTYPE_TEXT_ :
    case LCRZO_PRINTTYPE_PTEXT_ :
    case LCRZO_PRINTTYPE_BASE64_ :
    case LCRZO_PRINTTYPE_DUMP_ :
    case LCRZO_PRINTTYPE_HEXA_ :
    case LCRZO_PRINTTYPE_MIXED_ :
    case LCRZO_PRINTTYPE_CMH_ :
    case LCRZO_PRINTTYPE_CHM_ :
      { lcrzo_hdrparp hdrparp;
        lcrzo_er(lcrzo_hdrparp_init_hdrlarp(hdrlarp, &hdrparp));
	lcrzo_er(lcrzo_data_fprint(pf, (lcrzo_uint8 *)&hdrparp,
					sizeof(hdrparp),
					printtype));
        return(LCRZO_ERR_OK); /*return here to not print newline twice*/
      }
      break;
    case LCRZO_PRINTTYPE_ARRAY_ :
      switch(hdrlarp.op)
      { case 1  : fprintf(pf, " ARP Request_"); 
                  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
                  { lcrzo_string_init_text("demandeur :      ", 39, chpsrc);
		    lcrzo_string_init_text("info demandee :  ", 39, chpdst);
		  }
		  else
                  { lcrzo_string_init_text("this address :   ", 39, chpsrc);
		    lcrzo_string_init_text("asks that :      ", 39, chpdst);
		  }
		  break;
        case 2  : fprintf(pf, " ARP Reply___");
                  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
                  { lcrzo_string_init_text("info demandee :  ", 39, chpsrc);
		    lcrzo_string_init_text("pour qui c'est : ", 39, chpdst);
		  }
		  else
                  { lcrzo_string_init_text("that answer :    ", 39, chpsrc);
		    lcrzo_string_init_text("is for this :    ", 39, chpdst);
		  }
		  break;
        case 3  : fprintf(pf, " RARP Request"); 
	          if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
                  { lcrzo_string_init_text("demandeur :      ", 39, chpsrc);
		    lcrzo_string_init_text("info demandee :  ", 39, chpdst);
		  }
		  else
                  { lcrzo_string_init_text("this address :   ", 39, chpsrc);
		    lcrzo_string_init_text("asks that :      ", 39, chpdst);
		  }
		  break;
        case 4  : fprintf(pf, " RARP Reply__"); 
                  if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
                  { lcrzo_string_init_text("repondeur :      ", 39, chpsrc);
		    lcrzo_string_init_text("info repondue :  ", 39, chpdst);
		  }
		  else
                  { lcrzo_string_init_text("this address :   ", 39, chpsrc);
		    lcrzo_string_init_text("answered :       ", 39, chpdst);
		  }
		  break;
        default : fprintf(pf, " ARP ??______");
                  lcrzo_string_init_text("source :         ", 39, chpsrc);
		  lcrzo_string_init_text("destination :    ", 39, chpdst);
		  break;
      }
      fprintf(pf, "______________________________");
      fprintf(pf, "______________________________.\n");
      fprintf(pf, " | hw_type=%04xh, prot_type=%04xh,",
	      hdrlarp.hw_type, hdrlarp.prot_type);
      fprintf(pf, " hw_size=%02xh, prot_size=%02xh,",
	      hdrlarp.hw_size, hdrlarp.prot_size);
      fprintf(pf, " op=%04xh  |\n", hdrlarp.op);
      fprintf(pf, " | %s%02x:%02x:%02x:%02x:%02x:%02x", chpsrc,
	      (lcrzo_uint8)hdrlarp.hw_src[0],  (lcrzo_uint8)hdrlarp.hw_src[1],
	      (lcrzo_uint8)hdrlarp.hw_src[2], (lcrzo_uint8)hdrlarp.hw_src[3], 
	      (lcrzo_uint8)hdrlarp.hw_src[4], (lcrzo_uint8)hdrlarp.hw_src[5]);
      fprintf(pf, " - %d.%d.%d.%d %n",
	      (lcrzo_uint8)hdrlarp.prot_src[0],
	      (lcrzo_uint8)hdrlarp.prot_src[1],
	      (lcrzo_uint8)hdrlarp.prot_src[2], 
	      (lcrzo_uint8)hdrlarp.prot_src[3], &i);
      lcrzo_string_init_text("                                     ", 39, tab); 
      tab[30-i]='\0'; fprintf(pf, "%s      |\n", tab);
      fprintf(pf, " | %s%02x:%02x:%02x:%02x:%02x:%02x", chpdst,
	      (lcrzo_uint8)hdrlarp.hw_dst[0],  (lcrzo_uint8)hdrlarp.hw_dst[1],
	      (lcrzo_uint8)hdrlarp.hw_dst[2], (lcrzo_uint8)hdrlarp.hw_dst[3], 
	      (lcrzo_uint8)hdrlarp.hw_dst[4], (lcrzo_uint8)hdrlarp.hw_dst[5]);
      fprintf(pf, " - %d.%d.%d.%d %n",
	      (lcrzo_uint8)hdrlarp.prot_dst[0],
	      (lcrzo_uint8)hdrlarp.prot_dst[1],
	      (lcrzo_uint8)hdrlarp.prot_dst[2], 
	      (lcrzo_uint8)hdrlarp.prot_dst[3], &i);
      lcrzo_string_init_text("                                     ", 39, tab); 
      tab[30-i]='\0'; fprintf(pf, "%s      |\n", tab);
      fprintf(pf, " |______________________________________");
      fprintf(pf, "_________________________________|\n");
      break;
    default :
      return(LCRZO_ERR_IEINTERNALERROR);
  }

  /*eventually print a newline*/
  lcrzo_er(lcrzo_priv_printtype_printnewline(pf, printtype));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipopt_fprint(LCRZOFILE *pf,
		      const lcrzo_ipopt ipopt, 
		      lcrzo_uint8 nboctipopt,
		      lcrzo_printtype printtype)
{ int retour;
  lcrzo_uint8 pointeur, nboctnoop, overflow, flag;
  lcrzo_ipl iplt[9];
  lcrzo_ipopt optioninconnue;
  lcrzo_uint8 nboctoptioninconnue;
  lcrzo_int8 nbiplaffectes;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ipopt==NULL && nboctipopt) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*affichage eventuel du commentaire*/
  lcrzo_er(lcrzo_priv_printtype_printcomment(pf, printtype, "IP options",
				       "Options IP"));

  /*affichage de l'entete*/
  switch(printtype&0xFF)
  { case LCRZO_PRINTTYPE_NOTHING_ :
      break;
    case LCRZO_PRINTTYPE_SYNTH_ :
    case LCRZO_PRINTTYPE_TEXT_ :
    case LCRZO_PRINTTYPE_PTEXT_ :
    case LCRZO_PRINTTYPE_BASE64_ :
    case LCRZO_PRINTTYPE_DUMP_ :
    case LCRZO_PRINTTYPE_HEXA_ :
    case LCRZO_PRINTTYPE_MIXED_ :
    case LCRZO_PRINTTYPE_CMH_ :
    case LCRZO_PRINTTYPE_CHM_ :
      lcrzo_er(lcrzo_data_fprint(pf, ipopt, nboctipopt, printtype));
      return(LCRZO_ERR_OK); /*return here to not print newline twice*/
      break;
    case LCRZO_PRINTTYPE_ARRAY_ :
      fprintf(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?" OPTIONS IP":" IP OPTIONS");
      fprintf(pf, "____________________________________");
      fprintf(pf, "__________________________.\n");
      /*cherche lsrr*/
      retour=lcrzo_ipopt_decode_lsrr(ipopt, nboctipopt,
				    &nbiplaffectes, &pointeur,
				    &(iplt[0]), &(iplt[1]), &(iplt[2]), 
				    &(iplt[3]), &(iplt[4]), &(iplt[5]), 
				    &(iplt[6]), &(iplt[7]), &(iplt[8]));
      if (retour==LCRZO_ERR_OK)
      { fprintf(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		" | Loose source routing : nombre_de_routes %d,":
		" | Loose source routing : number_of_routes %d,", 
		nbiplaffectes);
        fprintf(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		"  pointeur %d                |\n":
		"  pointer %d                 |\n", pointeur);
	lcrzo_priv_ecrit_ipopttabipl(pf, nbiplaffectes, iplt);
      }
      /*cherche ssrr*/
      retour=lcrzo_ipopt_decode_ssrr(ipopt, nboctipopt,
				    &nbiplaffectes, &pointeur,
				    &(iplt[0]), &(iplt[1]), &(iplt[2]), 
				    &(iplt[3]), &(iplt[4]), &(iplt[5]), 
				    &(iplt[6]), &(iplt[7]), &(iplt[8]));
      if (retour==LCRZO_ERR_OK)
      { fprintf(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		" | Strict source routing : nombre_de_routes %d,":
		" | Strict source routing : number_of_routes %d,", 
		nbiplaffectes);
        fprintf(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		"  pointeur %d               |\n":
		"  pointer %d                |\n", pointeur);
	lcrzo_priv_ecrit_ipopttabipl(pf, nbiplaffectes, iplt);
      }
      /*cherche rr*/
      retour=lcrzo_ipopt_decode_rr(ipopt, nboctipopt,
				  &nbiplaffectes, &pointeur,
				  &(iplt[0]), &(iplt[1]), &(iplt[2]), 
				  &(iplt[3]), &(iplt[4]), &(iplt[5]), 
				  &(iplt[6]), &(iplt[7]), &(iplt[8]));
      if (retour==LCRZO_ERR_OK)
      { fprintf(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		" | Record route : nombre_de_routes %d, pointeur %d":
		" | Record route : number_of_routes %d, pointer %d ", 
		nbiplaffectes, pointeur);
        fprintf(pf, "                         |\n");
	lcrzo_priv_ecrit_ipopttabipl(pf, nbiplaffectes, iplt);
      }
      /*cherche timestamp*/ 
      retour=lcrzo_ipopt_decode_time(ipopt, nboctipopt,
				    &nbiplaffectes, &pointeur,
				    &overflow, &flag,
				    &(iplt[0]), &(iplt[1]), &(iplt[2]), 
				    &(iplt[3]), &(iplt[4]), &(iplt[5]), 
				    &(iplt[6]), &(iplt[7]), &(iplt[8]));
      if (retour==LCRZO_ERR_OK)
      { fprintf(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		" | Timestamp : nombre_de_timestamp %d, pointeur %d, ":
		" | Timestamp : number_of_routes %d, pointer %d,     ",
		nbiplaffectes, pointeur);
        fprintf(pf, "overflow %d, flag %d     |\n", overflow, flag);
	lcrzo_priv_ecrit_ipopttabtime(pf, nbiplaffectes, iplt);
      }
      /*cherche timestamp*/
      retour=lcrzo_ipopt_decode_timeip(ipopt, nboctipopt,
				      &nbiplaffectes, &pointeur,
				      &overflow, &flag,
				      &(iplt[0]), &(iplt[1]), &(iplt[2]), 
				      &(iplt[3]), &(iplt[4]), &(iplt[5]), 
				      &(iplt[6]), &(iplt[7]));
      if (retour==LCRZO_ERR_OK)
      { fprintf(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		" | Timestamp : nombre_de_timestamp %d, pointeur %d, ":
		" | Timestamp : number_of_timestamp %d, pointer %d,  ", 
		nbiplaffectes, pointeur);
        fprintf(pf, "overflow %d, flag %d     |\n", overflow, flag);
	lcrzo_priv_ecrit_ipopttabiptime(pf, nbiplaffectes, iplt);
      }
      /*cherche noop*/
      retour=lcrzo_ipopt_decode_noop(ipopt, nboctipopt, &nboctnoop);
      if (retour==LCRZO_ERR_OK)
      { fprintf(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		" | Nombre d'octets NOOP : %d\n":
		" | Number of NOOP bytes: %d\n", nboctnoop);
      }
      /*cherche inconnu*/
      retour=lcrzo_ipopt_decode_unknown(ipopt, nboctipopt,
					optioninconnue, &nboctoptioninconnue);
      if (retour==LCRZO_ERR_OK)
      { fprintf(pf, (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		" | Option non reconnue : \n":
		" | Unrecognized option : \n");
        lcrzo_er(lcrzo_data_fprint(pf, optioninconnue, 
					nboctoptioninconnue,
					LCRZO_PRINTTYPE_HEXAS));
      }
      fprintf(pf, " |____________________________________________");
      fprintf(pf, "___________________________|\n");
      break;
    default :
      return(LCRZO_ERR_IEINTERNALERROR);
  }

  /*eventually print a newline*/
  lcrzo_er(lcrzo_priv_printtype_printnewline(pf, printtype));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_packet_fprint_eth(LCRZOFILE *pf, 
			    lcrzo_constdata packet, 
			    lcrzo_int32 nboctpacket,
			    lcrzo_printprofile printprofile)
{ lcrzo_data data;
  lcrzo_hdrleth hdrleth;
  lcrzo_uint16 nboctdata;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*tentative de decodage*/
  retour=lcrzo_packet_decodem_ethdata(packet, nboctpacket,
				      &hdrleth, &data, &nboctdata);
  if ( retour!=LCRZO_ERR_OK )
  { return(lcrzo_priv_printprofile_tcdf(pf, &printprofile, printprofile.data, 
					"Data", "Donnees",
					packet, nboctpacket));
  }

  /*affichage de l'entete*/
  lcrzo_efr(lcrzo_priv_ecrittransition(pf, &printprofile, printprofile.hdreth),
	    lcrzo_data_free(data));
  lcrzo_efr(lcrzo_hdrleth_fprint(pf, hdrleth, printprofile.hdreth),
	    lcrzo_data_free(data));

  /*affichage des donnees ethernet*/
  switch(hdrleth.type)
    {
    case LCRZO_HDRLETH_TYPE_IP:
      retour=lcrzo_packet_fprint_ip(pf, data, nboctdata, printprofile);
      break;
    case LCRZO_HDRLETH_TYPE_ARP:
    case LCRZO_HDRLETH_TYPE_RARP:
      retour=lcrzo_packet_fprint_arp(pf, data, nboctdata, printprofile);
      break;
    default:
      retour=lcrzo_priv_printprofile_tcdf(pf, &printprofile,
					  printprofile.dataeth, 
					  "ETH data", "Donnees ETH",
					  data, nboctdata);
      break;
    }
  lcrzo_data_free(data);
  return(retour);
}

/*-------------------------------------------------------------*/
int lcrzo_packet_fprint_ip(LCRZOFILE *pf, 
			   lcrzo_constdata packet,
			   lcrzo_int32 nboctpacket,
			   lcrzo_printprofile printprofile)
{ lcrzo_data data;
  lcrzo_ipopt opt;
  lcrzo_hdrlip hdrlip;
  lcrzo_uint8 nboctopt;
  lcrzo_uint16 nboctdata;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*tentative de decodage*/
  retour=lcrzo_packet_decodem_ipoptdata(packet, nboctpacket, &hdrlip,
					opt, &nboctopt, &data, &nboctdata);
  if ( retour!=LCRZO_ERR_OK )
  { return(lcrzo_priv_printprofile_tcdf(pf, &printprofile,
					printprofile.dataeth, 
					"ETH data", "Donnees ETH",
					packet, nboctpacket));
  }

  /*affichage de l'entete*/
  lcrzo_efr(lcrzo_priv_ecrittransition(pf, &printprofile, printprofile.hdrip),
	    lcrzo_data_free(data));
  lcrzo_efr(lcrzo_hdrlip_fprint(pf, hdrlip, printprofile.hdrip),
	    lcrzo_data_free(data));
  if ( nboctopt )
  { lcrzo_efr(lcrzo_priv_ecrittransition(pf, &printprofile,printprofile.hdrip),
	      lcrzo_data_free(data));
    lcrzo_efr(lcrzo_ipopt_fprint(pf, opt, nboctopt, printprofile.hdrip),
	      lcrzo_data_free(data));
  }

  if ( hdrlip.offsetfrag )
  { /*ne sait pas decoder les fragments*/
    retour=lcrzo_priv_printprofile_tcdf(pf, &printprofile,
					printprofile.dataip, 
					"IP data", "Donnees IP",
					data, nboctdata);
    lcrzo_data_free(data);
    return(retour);
  }

  /*affichage des donnees IP*/
  switch(hdrlip.protocol)  
    {
    case LCRZO_HDRLIP_PROTOCOL_UDP:
      retour=lcrzo_packet_fprint_udp(pf, data, nboctdata, printprofile);
      break;
    case LCRZO_HDRLIP_PROTOCOL_TCP:
      retour=lcrzo_packet_fprint_tcp(pf, data, nboctdata, printprofile);
      break;
    case LCRZO_HDRLIP_PROTOCOL_ICMP:
      retour=lcrzo_packet_fprint_icmp(pf, data, nboctdata, printprofile);
      break;
    default:
      retour=lcrzo_priv_printprofile_tcdf(pf, &printprofile,
					  printprofile.dataip, 
					  "IP data", "Donnees IP",
					  data, nboctdata);
      break;
    }

  lcrzo_data_free(data);
  return(retour);
}

/*-------------------------------------------------------------*/
int lcrzo_packet_fprint_udp(LCRZOFILE *pf, 
			    lcrzo_constdata packet,
			    lcrzo_int32 nboctpacket,
			    lcrzo_printprofile printprofile)
{ lcrzo_data data;
  lcrzo_hdrludp hdrludp;
  lcrzo_uint16 nboctdata;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*tentative de decodage*/
  retour=lcrzo_packet_decodem_udpdata(packet, nboctpacket, &hdrludp,
                                      &data, &nboctdata);
  if ( retour!=LCRZO_ERR_OK )
  { return(lcrzo_priv_printprofile_tcdf(pf, &printprofile,
					printprofile.dataip, 
					"IP data", "Donnees IP",
					packet, nboctpacket));
  }

  /*affichage de l'entete*/
  lcrzo_efr(lcrzo_priv_ecrittransition(pf, &printprofile, printprofile.hdrudp),
	    lcrzo_data_free(data));
  lcrzo_efr(lcrzo_hdrludp_fprint(pf, hdrludp, printprofile.hdrudp),
	    lcrzo_data_free(data));

  /*affichage des donnees UDP*/
  retour=lcrzo_priv_printprofile_tcdf(pf, &printprofile,
				      printprofile.dataudp, 
				      "UDP data", "Donnees UDP",
				      data, nboctdata);
  lcrzo_data_free(data);
  return(retour);
}

/*-------------------------------------------------------------*/
int lcrzo_packet_fprint_tcp(LCRZOFILE *pf, 
			    lcrzo_constdata packet,
			    lcrzo_int32 nboctpacket,
			    lcrzo_printprofile printprofile)
{ lcrzo_data data;
  lcrzo_tcpopt tcpopt;
  lcrzo_hdrltcp hdrltcp;
  lcrzo_uint16 nboctdata;
  lcrzo_uint8 nbocttcpopt;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*tentative de decodage*/
  retour=lcrzo_packet_decodem_tcpoptdata(packet, nboctpacket, 
					 &hdrltcp, tcpopt, &nbocttcpopt,
					 &data, &nboctdata);
  if ( retour!=LCRZO_ERR_OK )
  { return(lcrzo_priv_printprofile_tcdf(pf, &printprofile,
					printprofile.dataip, 
					"IP data", "Donnees IP",
					packet, nboctpacket));
  }

  /*affichage de l'entete*/
  lcrzo_efr(lcrzo_priv_ecrittransition(pf, &printprofile, printprofile.hdrtcp),
	    lcrzo_data_free(data));
  lcrzo_efr(lcrzo_hdrltcp_fprint(pf, hdrltcp, printprofile.hdrtcp),
	    lcrzo_data_free(data));
  if ( nbocttcpopt )
  { lcrzo_efr(lcrzo_priv_ecrittransition(pf, &printprofile,
					 printprofile.hdrtcp),
	      lcrzo_data_free(data));
    lcrzo_efr(lcrzo_priv_printtype_printcomment(pf, printprofile.hdrtcp,
						"TCP options", "Options TCP"),
	      lcrzo_data_free(data));
    lcrzo_efr(lcrzo_data_fprint(pf, tcpopt, nbocttcpopt, printprofile.hdrtcp),
	      lcrzo_data_free(data));
  }

  /*affichage des donnees TCP*/
  retour=lcrzo_priv_printprofile_tcdf(pf, &printprofile,
				      printprofile.datatcp, 
				      "TCP data", "Donnees TCP",
				      data, nboctdata);
  lcrzo_data_free(data);
  return(retour);
}

/*-------------------------------------------------------------*/
int lcrzo_packet_fprint_icmp(LCRZOFILE *pf, 
			     lcrzo_constdata packet,
			     lcrzo_int32 nboctpacket,
			     lcrzo_printprofile printprofile)
{ lcrzo_data data;
  lcrzo_hdrlicmp hdrlicmp;
  lcrzo_uint16 nboctdata;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*tentative de decodage*/
  retour=lcrzo_packet_decodem_icmpdata(packet, nboctpacket, &hdrlicmp,
				       &data, &nboctdata);
  if ( retour!=LCRZO_ERR_OK )
  { return(lcrzo_priv_printprofile_tcdf(pf, &printprofile,
					printprofile.dataip, 
					"IP data", "Donnees IP",
					packet, nboctpacket));
  }

  /*affichage de l'entete*/
  lcrzo_efr(lcrzo_priv_ecrittransition(pf, &printprofile,printprofile.hdricmp),
	    lcrzo_data_free(data));
  lcrzo_efr(lcrzo_hdrlicmp_fprint(pf, hdrlicmp, printprofile.hdricmp),
	    lcrzo_data_free(data));

  /*affichage des donnees ICMP*/
  retour=lcrzo_priv_printprofile_tcdf(pf, &printprofile,
				      printprofile.dataicmp, 
				      "ICMP data", "Donnees ICMP",
				      data, nboctdata);
  lcrzo_data_free(data);
  return(retour);
}

/*-------------------------------------------------------------*/
int lcrzo_packet_fprint_arp(LCRZOFILE *pf, 
			    lcrzo_constdata packet,
			    lcrzo_int32 nboctpacket,
			    lcrzo_printprofile printprofile)
{ lcrzo_hdrlarp hdrlarp;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*tentative de decodage*/
  retour=lcrzo_packet_decode_arp(packet, nboctpacket, &hdrlarp);
  if ( retour!=LCRZO_ERR_OK )
  { return(lcrzo_priv_printprofile_tcdf(pf, &printprofile,
					printprofile.dataeth,
					"Ethernet data", "Donnees Ethernet", 
					packet, nboctpacket));
  }

  /*affichage de l'entete*/
  lcrzo_er(lcrzo_priv_ecrittransition(pf, &printprofile, printprofile.hdrarp));
  lcrzo_er(lcrzo_hdrlarp_fprint(pf, hdrlarp, printprofile.hdrarp));
  lcrzo_er(lcrzo_priv_ecrittransitionfin(pf, &printprofile));

  return(LCRZO_ERR_OK);
}


