/*us    Laurent Constantin's network library (lcrzo)
                 lcrzo_print module

  Functions herein allow to write headers, packets or datas.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_print

  Les fonctions presentes dans ce module permettent d'ecrire 
  des entetes, paquets ou des donnees.
*/


/*---------------------------------------------------------------*/
/*US                          Print types                        */
/*FR                        Type d'affichage                     */
/*---------------------------------------------------------------*/
/*us a lcrzo_printtype defines how to write headers or data */
/*fr un lcrzo_printtype definit comment ecrire des entetes ou
   des donnees*/
typedef lcrzo_uint16 lcrzo_printtype;

/*****************************************************************/
/*us you should not use these defines */
/*fr vous ne devriez pas utiliser ces defines*/
#define LCRZO_PRINTTYPE_C_         0x8000
#define LCRZO_PRINTTYPE_S_         0x4000
#define LCRZO_PRINTTYPE_N_         0x2000
#define LCRZO_PRINTTYPE_I_         0x1000
#define LCRZO_PRINTTYPE_C          (LCRZO_PRINTTYPE_C_|LCRZO_PRINTTYPE_N_)
#define LCRZO_PRINTTYPE_S          (LCRZO_PRINTTYPE_S_|LCRZO_PRINTTYPE_I_)
#define LCRZO_PRINTTYPE_N          (LCRZO_PRINTTYPE_N_)
#define LCRZO_PRINTTYPE_I          (LCRZO_PRINTTYPE_I_)
#define LCRZO_PRINTTYPE_CS         (LCRZO_PRINTTYPE_C_|LCRZO_PRINTTYPE_S)
#define LCRZO_PRINTTYPE_START      0x0000
#define LCRZO_PRINTTYPE_NOTHING_   0x0001
#define LCRZO_PRINTTYPE_SYNTH_     0x0002
#define LCRZO_PRINTTYPE_TEXT_      0x0003
#define LCRZO_PRINTTYPE_PTEXT_     0x0004 
#define LCRZO_PRINTTYPE_BASE64_    0x0005
#define LCRZO_PRINTTYPE_ARRAY_     0x0006 
#define LCRZO_PRINTTYPE_DUMP_      0x0007
#define LCRZO_PRINTTYPE_HEXA_      0x0008
#define LCRZO_PRINTTYPE_MIXED_     0x0009
#define LCRZO_PRINTTYPE_CMH_       0x000A
#define LCRZO_PRINTTYPE_CHM_       0x000B
/*us end of unusable defines */
/*fr fin des defines non utilisables */
/*****************************************************************/

/*---------------------------------------------------------------*/
/*us here are the lcrzo_printtype you can use (defines ending
   with _ustxt or _frtxt are explanations of these printypes)*/
/*fr voici les lcrzo_printtype que vous pouvez utiliser (les
   defines se terminant par _ustxt ou _frtxt expliquent ces 
   printtypes*/
#define LCRZO_PRINTTYPE_NOTHING_ustxt   "print nothing"
#define LCRZO_PRINTTYPE_NOTHING_frtxt   "ne rien afficher"
#define LCRZO_PRINTTYPE_NOTHING    LCRZO_PRINTTYPE_NOTHING_

#define LCRZO_PRINTTYPE_SYNTH_ustxt     "synthetic presentation"
#define LCRZO_PRINTTYPE_SYNTH_frtxt     "forme synthetique"
#define LCRZO_PRINTTYPE_SYNTH      LCRZO_PRINTTYPE_SYNTH_
#define LCRZO_PRINTTYPE_SYNTHN     (LCRZO_PRINTTYPE_SYNTH|LCRZO_PRINTTYPE_N)
#define LCRZO_PRINTTYPE_SYNTHC     (LCRZO_PRINTTYPE_SYNTH|LCRZO_PRINTTYPE_C)

#define LCRZO_PRINTTYPE_TEXT_ustxt      "text"
#define LCRZO_PRINTTYPE_TEXT_frtxt      "texte"
#define LCRZO_PRINTTYPE_TEXT       LCRZO_PRINTTYPE_TEXT_
#define LCRZO_PRINTTYPE_TEXTN      (LCRZO_PRINTTYPE_TEXT|LCRZO_PRINTTYPE_N)
#define LCRZO_PRINTTYPE_TEXTC      (LCRZO_PRINTTYPE_TEXT|LCRZO_PRINTTYPE_C)

#define LCRZO_PRINTTYPE_PTEXT_ustxt     "printable text"
#define LCRZO_PRINTTYPE_PTEXT_frtxt     "texte affichable"
#define LCRZO_PRINTTYPE_PTEXT      LCRZO_PRINTTYPE_PTEXT_
#define LCRZO_PRINTTYPE_PTEXTN     (LCRZO_PRINTTYPE_PTEXT|LCRZO_PRINTTYPE_N)
#define LCRZO_PRINTTYPE_PTEXTC     (LCRZO_PRINTTYPE_PTEXT|LCRZO_PRINTTYPE_C)

#define LCRZO_PRINTTYPE_BASE64_ustxt    "base 64"
#define LCRZO_PRINTTYPE_BASE64_frtxt    "base 64"
#define LCRZO_PRINTTYPE_BASE64     LCRZO_PRINTTYPE_BASE64_
#define LCRZO_PRINTTYPE_BASE64N    (LCRZO_PRINTTYPE_BASE64|LCRZO_PRINTTYPE_N)
#define LCRZO_PRINTTYPE_BASE64C    (LCRZO_PRINTTYPE_BASE64|LCRZO_PRINTTYPE_C)

#define LCRZO_PRINTTYPE_ARRAY_ustxt     "array"
#define LCRZO_PRINTTYPE_ARRAY_frtxt     "tableau"
#define LCRZO_PRINTTYPE_ARRAY      (LCRZO_PRINTTYPE_ARRAY_|LCRZO_PRINTTYPE_I)
#define LCRZO_PRINTTYPE_ARRAYC     (LCRZO_PRINTTYPE_ARRAY|LCRZO_PRINTTYPE_C_)

#define LCRZO_PRINTTYPE_DUMP_ustxt      "dump"
#define LCRZO_PRINTTYPE_DUMP_frtxt      "dump"
#define LCRZO_PRINTTYPE_DUMP       (LCRZO_PRINTTYPE_DUMP_|LCRZO_PRINTTYPE_I)
#define LCRZO_PRINTTYPE_DUMPC      (LCRZO_PRINTTYPE_DUMP|LCRZO_PRINTTYPE_C_)

#define LCRZO_PRINTTYPE_HEXA_ustxt      "hexadecimal"
#define LCRZO_PRINTTYPE_HEXA_frtxt      "hexadecimal"
#define LCRZO_PRINTTYPE_HEXA       LCRZO_PRINTTYPE_HEXA_ 
#define LCRZO_PRINTTYPE_HEXAN      (LCRZO_PRINTTYPE_HEXA|LCRZO_PRINTTYPE_N)
#define LCRZO_PRINTTYPE_HEXAC      (LCRZO_PRINTTYPE_HEXA|LCRZO_PRINTTYPE_C)
#define LCRZO_PRINTTYPE_HEXAS      (LCRZO_PRINTTYPE_HEXA|LCRZO_PRINTTYPE_S)
#define LCRZO_PRINTTYPE_HEXACS     (LCRZO_PRINTTYPE_HEXA|LCRZO_PRINTTYPE_CS)

#define LCRZO_PRINTTYPE_MIXED_ustxt     "mixed (hexa+string)"
#define LCRZO_PRINTTYPE_MIXED_frtxt     "mixte (hexa+chaine)"
#define LCRZO_PRINTTYPE_MIXED      LCRZO_PRINTTYPE_MIXED_
#define LCRZO_PRINTTYPE_MIXEDN     (LCRZO_PRINTTYPE_MIXED|LCRZO_PRINTTYPE_N)
#define LCRZO_PRINTTYPE_MIXEDC     (LCRZO_PRINTTYPE_MIXED|LCRZO_PRINTTYPE_C)
#define LCRZO_PRINTTYPE_MIXEDS     (LCRZO_PRINTTYPE_MIXED|LCRZO_PRINTTYPE_S)
#define LCRZO_PRINTTYPE_MIXEDCS    (LCRZO_PRINTTYPE_MIXED|LCRZO_PRINTTYPE_CS)

#define LCRZO_PRINTTYPE_CMH_ustxt       "commented mixed then hexa"
#define LCRZO_PRINTTYPE_CMH_frtxt       "mixte commente puis hexa"
#define LCRZO_PRINTTYPE_CMH        (LCRZO_PRINTTYPE_CMH_|LCRZO_PRINTTYPE_I)
#define LCRZO_PRINTTYPE_CMHC       (LCRZO_PRINTTYPE_CMH|LCRZO_PRINTTYPE_C_)
#define LCRZO_PRINTTYPE_CMHS       (LCRZO_PRINTTYPE_CMH|LCRZO_PRINTTYPE_S)
#define LCRZO_PRINTTYPE_CMHCS      (LCRZO_PRINTTYPE_CMH|LCRZO_PRINTTYPE_CS)

#define LCRZO_PRINTTYPE_CHM_ustxt       "commented hexa then mixed"
#define LCRZO_PRINTTYPE_CHM_frtxt       "hexa commente puis mixte"
#define LCRZO_PRINTTYPE_CHM        (LCRZO_PRINTTYPE_CHM_|LCRZO_PRINTTYPE_I)
#define LCRZO_PRINTTYPE_CHMC       (LCRZO_PRINTTYPE_CHM|LCRZO_PRINTTYPE_C_)
#define LCRZO_PRINTTYPE_CHMS       (LCRZO_PRINTTYPE_CHM|LCRZO_PRINTTYPE_S)
#define LCRZO_PRINTTYPE_CHMCS      (LCRZO_PRINTTYPE_CHM|LCRZO_PRINTTYPE_CS)

/*---------------------------------------------------------------*/
/*us Name : lcrzo_printtype_fprint
   Description :
     Print a lcrzo_printtype.
   Input parameter(s) :
     *pf : FILE where the function prints
     printtype : lcrzo_printtype to print
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printtype_fprint
   Description :
     Ecrit un lcrzo_printtype.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     printtype : lcrzo_printtype a ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printtype_fprint(LCRZOFILE *pf,
			   lcrzo_printtype printtype);

/*us Name : lcrzo_printtype_stdin
   Description :
     Ask user to choose a lcrzo_printtype
   Input parameter(s) :
     message : message to print before
     defaultprinttype : default lcrzo_printtype to use if
                        user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     *pprinttype : lcrzo_printype choosen
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printtype_stdin
   Description :
     Demande a l'utilisateur de choisir un lcrzo_printtype
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaultprinttype : lcrzo_printtype par defaut a utiliser
                        si l'utilisateur n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pprinttype : lcrzo_printype choisi
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printtype_stdin(const char *message,
			  lcrzo_printtype defaultprinttype,
			  lcrzo_printtype *pprinttype);

/*us Name : lcrzo_printtype_count
   Description :
     Count the number of printtype
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pcount : number of lcrzo_printtype
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printtype_count
   Description :
     Compte le nombre de type d'affichage
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcount : nombre de type d'affichage
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printtype_count(lcrzo_int32 *pcount);

/*US*******************************************
 * Note about positions :                     *
 * Warning : 0 is an invalid value            *
 * Here are possible values for a position :  *
 *  +1 : first item                           *
 *  +2 : second item                          *
 *  ...                                       *
 * Moreover, we can count from the end :      *
 *  -1 : last item                            *
 *  -2 : last but one item                    *
 *  ...                                       *
 **********************************************/
/*FR*******************************************
 * Note sur les positions :                   *
 * Attention : 0 n'est pas une valeur valide  *
 * Voici des valeurs de position possibles :  *
 *  +1 : premier element                      *
 *  +2 : deuxieme element                     *
 *  ...                                       *
 * De plus, on peut partir de la fin :        *
 *  -1 : dernier element                      *
 *  -2 : avant dernier element                *
 *  ...                                       *
 **********************************************/

/*us Name : lcrzo_printtype_value_pos
   Description :
     Get a printtype
   Input parameter(s) :
     pos : position of the wanted printtype (cf. the above frame to
           know possible values)
   Input/output parameter(s) :
   Output parameter(s) :
     *pprinttype : lcrzo_printtype number 'pos'
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printtype_value_pos
   Description :
     Fournit le type d'affichage situe a la position voulue
   Parametre(s) d'entree :
     pos : position du type d'affichage cherche (cf. le cadre
            ci-dessus pour connaitre les valeurs valides)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pprinttype : lcrzo_printtype numero 'pos'
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printtype_value_pos(lcrzo_int32 pos,
			      lcrzo_printtype *pprinttype);

/*us Name : lcrzo_printtype_value_first
   Description :
     Get the first printtype
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pprinttype : first lcrzo_printtype
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printtype_value_first
   Description :
     Fournit le premier type d'affichage
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pprinttype : premier lcrzo_printtype
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printtype_value_first(lcrzo_printtype *pprinttype);

/*us Name : lcrzo_printtype_value_last
   Description :
     Get the last printtype
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pprinttype : last lcrzo_printtype
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printtype_value_last
   Description :
     Fournit le dernier type d'affichage
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pprinttype : dernier lcrzo_printtype
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printtype_value_last(lcrzo_printtype *pprinttype);

/*us Name : lcrzo_printtype_loop_range
   Description :
     Call a function for each lcrzo_printtype in a
     given position range.
   Input parameter(s) :
     startpos : start position for the loop (cf. the above frame to
                know possible values)
     endpos : end position for the loop (cf. the above frame to
              know possible values)
     pfunc : memory address of the function which will be called
             for each lcrzo_printtype in the range [startpos->endpos].
             For each call, the first parameter ('printtype') will
             be set with the printtype, and the second parameter
             ('pinfos') will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printtype_loop_range
   Description :
     Appelle une fonction pour les types d'affichage
     situes dans un intervalle donne.
   Parametre(s) d'entree :
     startpos : position de debut de la boucle (cf. le cadre
                ci-dessus pour connaitre les valeurs valides)
     endpos : position de fin de la boucle (cf. le cadre
              ci-dessus pour connaitre les valeurs valides)
     pfunc : adresse memoire de la fonction appelee pour chaque
             printtype de l'intervalle [startpos->endpos]. Lors de
             chaque appel, le premier parametre ('printtype') est 
             affecte avec le printtype courant, et le deuxieme
             parametre ('pinfos') est affecte avec la valeur 
             de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printtype_loop_range(lcrzo_int32 startpos,
			       lcrzo_int32 endpos,
			       int (*pfunc)(lcrzo_printtype printtype,
					    const void *pinfos),
			       const void *pinfos);

/*us Name : lcrzo_printtype_loop_all
   Description :
     Call a function for every lcrzo_printtype.
   Input parameter(s) :
     pfunc : memory address of the function which will be called
             for every lcrzo_printtype.
             For each call, the first parameter ('printtype') will
             be set with the printtype, and the second parameter
             ('pinfos') will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printtype_loop_all
   Description :
     Appelle une fonction pour tous les types d'affichage.
   Parametre(s) d'entree :
     pfunc : adresse memoire de la fonction appelee pour tous 
             les printtype. Lors de
             chaque appel, le premier parametre ('printtype') est 
             affecte avec le printtype courant, et le deuxieme
             parametre ('pinfos') est affecte avec la valeur 
             de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printtype_loop_all(int (*pfunc)(lcrzo_printtype printtype,
					  const void *pinfos),
			     const void *pinfos);


/*---------------------------------------------------------------*/
/*US                         Print profile                       */
/*FR                       Profil d'affichage                    */
/*---------------------------------------------------------------*/
/*us a lcrzo_printprofile defines how to print a packet */
/*fr un lcrzo_printprofile definit comment ecrire un paquet */

/*us the definition of a lcrzo_printprofile has to be public
   (because the compiler needs to know its size), 
   but you should not use its fields directly. */
/*fr la definition d'un lcrzo_printprofile doit etre publique
   (car le compilateur doit connaitre sa taille), mais vous
   ne devriez pas utiliser directement ses champs. */
typedef struct {
  lcrzo_printtype hdr; 
  lcrzo_printtype hdreth;
  lcrzo_printtype hdrip;
  lcrzo_printtype hdrudp;
  lcrzo_printtype hdrtcp;
  lcrzo_printtype hdricmp;
  lcrzo_printtype hdrarp;
  lcrzo_printtype data;
  lcrzo_printtype dataeth;
  lcrzo_printtype dataip;
  lcrzo_printtype dataudp;
  lcrzo_printtype datatcp;
  lcrzo_printtype dataicmp;
  lcrzo_printtype lastused;
} lcrzo_printprofile;

/*us this function is usefull for the folowing defines, but you
   should not use it (use lcrzo_printprofile_init instead) */
/*fr cette fonction est utile pour les defines qui suivent, mais
   vous ne devriez pas l'utiliser (employez lcrzo_printprofile_init) */
lcrzo_printprofile lcrzo_printprofile_returnstruct(lcrzo_printtype hdr, lcrzo_printtype hdreth, lcrzo_printtype hdrip, lcrzo_printtype hdrudp, lcrzo_printtype hdrtcp, lcrzo_printtype hdricmp, lcrzo_printtype hdrarp, lcrzo_printtype data, lcrzo_printtype dataeth, lcrzo_printtype dataip, lcrzo_printtype dataudp, lcrzo_printtype datatcp, lcrzo_printtype dataicmp);

/*us here are some lcrzo_printprofile you can use (defines ending
   with _ustxt or _frtxt are explanations of these printprofile).
   You can create your own lcrzo_printprofile with lcrzo_printprofile_init */
/*fr voici quelques lcrzo_printprofile que vous pouvez utiliser (les
   defines se terminant par _ustxt ou _frtxt expliquent ces 
   printprofile). Vous pouvez creer vos lcrzo_printprofile 
   personnalises avec lcrzo_printprofile_init */
#define LCRZO_PRINTPROFILE_SYNTHSYNTH lcrzo_printprofile_returnstruct(LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH)
#define LCRZO_PRINTPROFILE_SYNTHSYNTH_ustxt "header and data in synthetic aspect"
#define LCRZO_PRINTPROFILE_SYNTHSYNTH_frtxt "entetes et donnees sous forme synthetique"

#define LCRZO_PRINTPROFILE_SYNTHESYNTH lcrzo_printprofile_returnstruct(LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_NOTHING, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH, LCRZO_PRINTTYPE_SYNTH)
#define LCRZO_PRINTPROFILE_SYNTHESYNTH_ustxt "header (without ethernet) and data in synthetic aspect"
#define LCRZO_PRINTPROFILE_SYNTHESYNTH_frtxt "entetes (sauf ethernet) et donnees sous forme synthetique"

#define LCRZO_PRINTPROFILE_ARRAYARRAY lcrzo_printprofile_returnstruct(LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY)
#define LCRZO_PRINTPROFILE_ARRAYARRAY_ustxt "header and data in array aspect"
#define LCRZO_PRINTPROFILE_ARRAYARRAY_frtxt "entetes et donnees sous forme de tableau"

#define LCRZO_PRINTPROFILE_ARRAYDUMP lcrzo_printprofile_returnstruct(LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_DUMP, LCRZO_PRINTTYPE_DUMP, LCRZO_PRINTTYPE_DUMP, LCRZO_PRINTTYPE_DUMP, LCRZO_PRINTTYPE_DUMP, LCRZO_PRINTTYPE_DUMP)
#define LCRZO_PRINTPROFILE_ARRAYDUMP_ustxt "header in array aspect and data in dump"
#define LCRZO_PRINTPROFILE_ARRAYDUMP_frtxt "entetes sous forme de tableau et donnees en dump"

#define LCRZO_PRINTPROFILE_ARRAYCHMS lcrzo_printprofile_returnstruct(LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_ARRAY, LCRZO_PRINTTYPE_CHMS, LCRZO_PRINTTYPE_CHMS, LCRZO_PRINTTYPE_CHMS, LCRZO_PRINTTYPE_CHMS, LCRZO_PRINTTYPE_CHMS, LCRZO_PRINTTYPE_CHMS)
#define LCRZO_PRINTPROFILE_ARRAYCHMS_ustxt "header in array aspect and data in mixed"
#define LCRZO_PRINTPROFILE_ARRAYCHMS_frtxt "entetes sous forme de tableau et donnees en mixte"

#define LCRZO_PRINTPROFILE_HEXANHEXAS lcrzo_printprofile_returnstruct(LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAS, LCRZO_PRINTTYPE_HEXAS, LCRZO_PRINTTYPE_HEXAS, LCRZO_PRINTTYPE_HEXAS, LCRZO_PRINTTYPE_HEXAS, LCRZO_PRINTTYPE_HEXAS)
#define LCRZO_PRINTPROFILE_HEXANHEXAS_ustxt "header and data in hexa aspect"
#define LCRZO_PRINTPROFILE_HEXANHEXAS_frtxt "entetes et donnees sous forme hexa"

#define LCRZO_PRINTPROFILE_HEXANDUMP lcrzo_printprofile_returnstruct(LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_DUMP, LCRZO_PRINTTYPE_DUMP, LCRZO_PRINTTYPE_DUMP, LCRZO_PRINTTYPE_DUMP, LCRZO_PRINTTYPE_DUMP, LCRZO_PRINTTYPE_DUMP)
#define LCRZO_PRINTPROFILE_HEXANDUMP_ustxt "header in hexa aspect and data in dump"
#define LCRZO_PRINTPROFILE_HEXANDUMP_frtxt "entetes sous forme d'hexa et donnees en dump"

#define LCRZO_PRINTPROFILE_HEXANCHMS lcrzo_printprofile_returnstruct(LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_HEXAN, LCRZO_PRINTTYPE_CHMS, LCRZO_PRINTTYPE_CHMS, LCRZO_PRINTTYPE_CHMS, LCRZO_PRINTTYPE_CHMS, LCRZO_PRINTTYPE_CHMS, LCRZO_PRINTTYPE_CHMS)
#define LCRZO_PRINTPROFILE_HEXANCHMS_ustxt "header in hexa aspect and data in mixed"
#define LCRZO_PRINTPROFILE_HEXANCHMS_frtxt "entetes sous forme d'hexa et donnees en mixte"

/*us Name : lcrzo_printprofile_init
   Description :
     Create a custom lcrzo_printprofile.
   Input parameter(s) :
     hdr : default lcrzo_printtype for headers
     hdreth : lcrzo_printtype for Ethernet headers
     hdrip : lcrzo_printtype for IP headers
     hdrudp : lcrzo_printtype for UDP headers
     hdrtcp : lcrzo_printtype for TCP headers
     hdricmp : lcrzo_printtype for ICMP headers
     hdrarp : lcrzo_printtype for ARP/RARP headers
     data : default lcrzo_printtype for data
     dataeth : lcrzo_printtype for Ethernet data
     dataip : lcrzo_printtype for IP data
     dataudp : lcrzo_printtype for UDP data
     datatcp : lcrzo_printtype for TCP data
     dataicmp : lcrzo_printtype for ICMP data
   Input/output parameter(s) :
   Output parameter(s) :
     *pprintprofile : lcrzo_printprofile created
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printprofile_init
   Description :
     Cree un lcrzo_printprofile personnalise.
   Parametre(s) d'entree :
     hdr : lcrzo_printtype par default pour les entetes
     hdreth : lcrzo_printtype pour les entetes Ethernet
     hdrip : lcrzo_printtype pour les entetes IP
     hdrudp : lcrzo_printtype pour les entetes UDP
     hdrtcp : lcrzo_printtype pour les entetes TCP
     hdricmp : lcrzo_printtype pour les entetes ICMP
     hdrarp : lcrzo_printtype pour les entetes ARP/RARP
     data : lcrzo_printtype pour les donnees 
     dataeth : lcrzo_printtype pour les donnees Ethernet
     dataip : lcrzo_printtype pour les donnees IP
     dataudp : lcrzo_printtype pour les donnees UDP
     datatcp : lcrzo_printtype pour les donnees TCP
     dataicmp : lcrzo_printtype pour les donnees ICMP
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pprintprofile : lcrzo_printprofile cree
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printprofile_init(lcrzo_printtype hdr,
			    lcrzo_printtype hdreth,
			    lcrzo_printtype hdrip, 
			    lcrzo_printtype hdrudp,
			    lcrzo_printtype hdrtcp,
			    lcrzo_printtype hdricmp,
			    lcrzo_printtype hdrarp, 
			    lcrzo_printtype data,
			    lcrzo_printtype dataeth,
			    lcrzo_printtype dataip,
			    lcrzo_printtype dataudp,
			    lcrzo_printtype datatcp,
			    lcrzo_printtype dataicmp,
			    lcrzo_printprofile *pprintprofile);

/*us Name : lcrzo_printprofile_fprint
   Description :
     Print a lcrzo_printprofile.
   Input parameter(s) :
     *pf : FILE where the function prints
     printprofile : lcrzo_printprofile to print
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printprofile_fprint
   Description :
     Ecrit un lcrzo_printype.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     printprofile : lcrzo_printprofile a ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printprofile_fprint(LCRZOFILE *pf, 
			      lcrzo_printprofile printprofile);

/*us Name : lcrzo_printprofile_stdin
   Description :
     Ask user to choose a lcrzo_printprofile
   Input parameter(s) :
     message : message to print before
     defaultprintprofile : default lcrzo_printprofile to use if
                        user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     *pprintprofile : lcrzo_printype choosen
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printprofile_stdin
   Description :
     Demande a l'utilisateur de choisir un lcrzo_printprofile
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaultprintprofile : lcrzo_printprofile par defaut a utiliser
                        si l'utilisateur n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pprintprofile : lcrzo_printype choisi
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printprofile_stdin(const char *message,
			     lcrzo_printprofile defaultprintprofile,
			     lcrzo_printprofile *pprintprofile);

/*us Name : lcrzo_printprofile_count
   Description :
     Count the number of printprofile
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pcount : number of lcrzo_printprofile
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printprofile_count
   Description :
     Compte le nombre de profils d'affichage
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcount : nombre de profils d'affichage
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printprofile_count(lcrzo_int32 *pcount);

/*US*******************************************
 * Note about positions :                     *
 * Warning : 0 is an invalid value            *
 * Here are possible values for a position :  *
 *  +1 : first item                           *
 *  +2 : second item                          *
 *  ...                                       *
 * Moreover, we can count from the end :      *
 *  -1 : last item                            *
 *  -2 : last but one item                    *
 *  ...                                       *
 **********************************************/
/*FR*******************************************
 * Note sur les positions :                   *
 * Attention : 0 n'est pas une valeur valide  *
 * Voici des valeurs de position possibles :  *
 *  +1 : premier element                      *
 *  +2 : deuxieme element                     *
 *  ...                                       *
 * De plus, on peut partir de la fin :        *
 *  -1 : dernier element                      *
 *  -2 : avant dernier element                *
 *  ...                                       *
 **********************************************/

/*us Name : lcrzo_printprofile_value_pos
   Description :
     Get a printprofile
   Input parameter(s) :
     pos : position of the wanted printprofile (cf. the above frame to
           know possible values)
   Input/output parameter(s) :
   Output parameter(s) :
     *pprintprofile : lcrzo_printprofile number 'pos'
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printprofile_value_pos
   Description :
     Fournit le profil d'affichage situe a la position voulue
   Parametre(s) d'entree :
     pos : position du profil d'affichage cherche (cf. le cadre
            ci-dessus pour connaitre les valeurs valides)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pprintprofile : lcrzo_printprofile numero 'pos'
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printprofile_value_pos(lcrzo_int32 pos,
				 lcrzo_printprofile *pprintprofile);

/*us Name : lcrzo_printprofile_value_first
   Description :
     Get the first printprofile
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pprintprofile : first lcrzo_printprofile
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printprofile_value_first
   Description :
     Fournit le premier profil d'affichage
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pprintprofile : premier lcrzo_printprofile
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printprofile_value_first(lcrzo_printprofile *pprintprofile);

/*us Name : lcrzo_printprofile_value_last
   Description :
     Get the last printprofile
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pprintprofile : last lcrzo_printprofile
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printprofile_value_last
   Description :
     Fournit le dernier profil d'affichage
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pprintprofile : dernier lcrzo_printprofile
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printprofile_value_last(lcrzo_printprofile *pprintprofile);

/*us Name : lcrzo_printprofile_loop_range
   Description :
     Call a function for each lcrzo_printprofile in a
     given position range.
   Input parameter(s) :
     startpos : start position for the loop (cf. the above frame to
                know possible values)
     endpos : end position for the loop (cf. the above frame to
              know possible values)
     pfunc : memory address of the function which will be called
             for each lcrzo_printprofile in the range [startpos->endpos].
             For each call, the first parameter ('printprofile') will
             be set with the printprofile, and the second parameter
             ('pinfos') will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printprofile_loop_range
   Description :
     Appelle une fonction pour les profils d'affichage
     situes dans un intervalle donne.
   Parametre(s) d'entree :
     startpos : position de debut de la boucle (cf. le cadre
                ci-dessus pour connaitre les valeurs valides)
     endpos : position de fin de la boucle (cf. le cadre
              ci-dessus pour connaitre les valeurs valides)
     pfunc : adresse memoire de la fonction appelee pour chaque
             printprofile de l'intervalle [startpos->endpos]. Lors de
             chaque appel, le premier parametre ('printprofile') est 
             affecte avec le printprofile courant, et le deuxieme
             parametre ('pinfos') est affecte avec la valeur 
             de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printprofile_loop_range(lcrzo_int32 startpos,
				  lcrzo_int32 endpos,
				  int (*pfunc)(lcrzo_printprofile printprofile,
					       const void *pinfos),
				  const void *pinfos);

/*us Name : lcrzo_printprofile_loop_all
   Description :
     Call a function for every lcrzo_printprofile.
   Input parameter(s) :
     pfunc : memory address of the function which will be called
             for every lcrzo_printprofile.
             For each call, the first parameter ('printprofile') will
             be set with the printprofile, and the second parameter
             ('pinfos') will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printprofile_loop_all
   Description :
     Appelle une fonction pour tous les profils d'affichage.
   Parametre(s) d'entree :
     pfunc : adresse memoire de la fonction appelee pour tous 
             les printprofile. Lors de
             chaque appel, le premier parametre ('printprofile') est 
             affecte avec le printprofile courant, et le deuxieme
             parametre ('pinfos') est affecte avec la valeur 
             de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printprofile_loop_all(int (*pfunc)(lcrzo_printprofile printprofile,
					     const void *pinfos),
				const void *pinfos);


/*---------------------------------------------------------------*/
/*US                     Printing functions                      */
/*FR                    Fonctions d'affichage                    */
/*---------------------------------------------------------------*/

/*---------------------------------------------------------------*/
/*us Name : lcrzo_hs_fprint
   Description :
     Print a lcrzo_hs.
   Input parameter(s) :
     *pf : FILE where the function prints
     startstring : string to print before the lcrzo_hs
     hs : lcrzo_hs to print
     endstring : string to print after the lcrzo_hs
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hs_fprint
   Description :
     Ecrit un lcrzo_hs.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     startstring : chaine a ecrire avant le lcrzo_hs
     hs : lcrzo_hs a ecrire
     endstring : chaine a ecrire apres le lcrzo_hs
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hs_fprint(LCRZOFILE *pf,
		    const char *startstring,
		    const lcrzo_hs hs,
		    const char *endstring);

/*us Name : lcrzo_ips_fprint
   Description :
     Print a lcrzo_ips.
   Input parameter(s) :
     *pf : FILE where the function prints
     startstring : string to print before the lcrzo_ips
     ips : lcrzo_ips to print
     endstring : string to print after the lcrzo_ips
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ips_fprint
   Description :
     Ecrit un lcrzo_ips.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     startstring : chaine a ecrire avant le lcrzo_ips
     ips : lcrzo_ips a ecrire
     endstring : chaine a ecrire apres le lcrzo_ips
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ips_fprint(LCRZOFILE *pf,
		     const char *startstring,
		     const lcrzo_ips ips,
		     const char *endstring);

/*us Name : lcrzo_ipa_fprint
   Description :
     Print a lcrzo_ipa.
   Input parameter(s) :
     *pf : FILE where the function prints
     startstring : string to print before the lcrzo_ipa
     ipa : lcrzo_ipa to print
     endstring : string to print after the lcrzo_ipa
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipa_fprint
   Description :
     Ecrit un lcrzo_ipa.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     startstring : chaine a ecrire avant le lcrzo_ipa
     ipa : lcrzo_ipa a ecrire
     endstring : chaine a ecrire apres le lcrzo_ipa
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipa_fprint(LCRZOFILE *pf,
		     const char *startstring,
		     const lcrzo_ipa ipa, 
		     const char *endstring);

/*us Name : lcrzo_ipl_fprint
   Description :
     Print a lcrzo_ipl.
   Input parameter(s) :
     *pf : FILE where the function prints
     startstring : string to print before the lcrzo_ipl
     ipl : lcrzo_ipl to print
     endstring : string to print after the lcrzo_ipl
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipl_fprint
   Description :
     Ecrit un lcrzo_ipl.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     startstring : chaine a ecrire avant le lcrzo_ipl
     ipl : lcrzo_ipl a ecrire
     endstring : chaine a ecrire apres le lcrzo_ipl
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipl_fprint(LCRZOFILE *pf,
		     const char *startstring,
		     lcrzo_ipl ipl,
		     const char *endstring);

/*us Name : lcrzo_eths_fprint
   Description :
     Print a lcrzo_eths.
   Input parameter(s) :
     *pf : FILE where the function prints
     startstring : string to print before the lcrzo_eths
     eths : lcrzo_eths to print
     endstring : string to print after the lcrzo_eths
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_eths_fprint
   Description :
     Ecrit un lcrzo_eths.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     startstring : chaine a ecrire avant le lcrzo_eths
     eths : lcrzo_eths a ecrire
     endstring : chaine a ecrire apres le lcrzo_eths
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_eths_fprint(LCRZOFILE *pf,
		      const char *startstring,
		      const lcrzo_eths eths,
		      const char *endstring);

/*us Name : lcrzo_etha_fprint
   Description :
     Print a lcrzo_etha.
   Input parameter(s) :
     *pf : FILE where the function prints
     startstring : string to print before the lcrzo_etha
     etha : lcrzo_etha to print
     endstring : string to print after the lcrzo_etha
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_etha_fprint
   Description :
     Ecrit un lcrzo_etha.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     startstring : chaine a ecrire avant le lcrzo_etha
     etha : lcrzo_etha a ecrire
     endstring : chaine a ecrire apres le lcrzo_etha
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_etha_fprint(LCRZOFILE *pf,
		      const char *startstring,
		      const lcrzo_etha teth,
		      const char *endstring);

/*us Name : lcrzo_device_fprint
   Description :
     Print a lcrzo_device.
   Input parameter(s) :
     *pf : FILE where the function prints
     startstring : string to print before the lcrzo_device
     device : lcrzo_device to print
     endstring : string to print after the lcrzo_device
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_device_fprint
   Description :
     Ecrit un lcrzo_printype.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     startstring : chaine a ecrire avant le lcrzo_device
     device : lcrzo_device a ecrire
     endstring : chaine a ecrire apres le lcrzo_device
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_device_fprint(LCRZOFILE *pf,
			const char *startstring,
			const lcrzo_device device,
			const char *endstring);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_hs_fprint_infos
   Description :
     Print a lot of informations about a lcrzo_hs.
   Input parameter(s) :
     *pf : FILE where the function prints
     hs : lcrzo_hs whose informations are printed
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hs_fprint_infos
   Description :
     Ecrit toutes les informations sur un lcrzo_hs.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     hs : lcrzo_hs dont on affiche les informations
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hs_fprint_infos(LCRZOFILE *pf, 
			  const lcrzo_hs hs);

/*us Name : lcrzo_ips_fprint_infos
   Description :
     Print a lot of informations about a lcrzo_ips.
   Input parameter(s) :
     *pf : FILE where the function prints
     ips : lcrzo_ips whose informations are printed
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ips_fprint_infos
   Description :
     Ecrit toutes les informations sur un lcrzo_ips.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     ips : lcrzo_ips dont on affiche les informations
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ips_fprint_infos(LCRZOFILE *pf, 
			   const lcrzo_ips ips);

/*us Name : lcrzo_ipa_fprint_infos
   Description :
     Print a lot of informations about a lcrzo_ipa.
   Input parameter(s) :
     *pf : FILE where the function prints
     ipa : lcrzo_ipa whose informations are printed
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipa_fprint_infos
   Description :
     Ecrit toutes les informations sur un lcrzo_ipa.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     ipa : lcrzo_ipa dont on affiche les informations
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipa_fprint_infos(LCRZOFILE *pf, 
			   const lcrzo_ipa ipa);

/*us Name : lcrzo_ipl_fprint_infos
   Description :
     Print a lot of informations about a lcrzo_ipl.
   Input parameter(s) :
     *pf : FILE where the function prints
     ipl : lcrzo_ipl whose informations are printed
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipl_fprint_infos
   Description :
     Ecrit toutes les informations sur un lcrzo_ipl.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     ipl : lcrzo_ipl dont on affiche les informations
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipl_fprint_infos(LCRZOFILE *pf, 
			   lcrzo_ipl ipl);

/*us Name : lcrzo_eths_fprint_infos
   Description :
     Print a lot of informations about a lcrzo_eths.
   Input parameter(s) :
     *pf : FILE where the function prints
     eths : lcrzo_eths whose informations are printed
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_eths_fprint_infos
   Description :
     Ecrit toutes les informations sur un lcrzo_eths.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     eths : lcrzo_eths dont on affiche les informations
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_eths_fprint_infos(LCRZOFILE *pf,
			    const lcrzo_eths eths);

/*us Name : lcrzo_etha_fprint_infos
   Description :
     Print a lot of informations about a lcrzo_etha.
   Input parameter(s) :
     *pf : FILE where the function prints
     etha : lcrzo_etha whose informations are printed
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_etha_fprint_infos
   Description :
     Ecrit toutes les informations sur un lcrzo_etha.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     etha : lcrzo_etha dont on affiche les informations
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_etha_fprint_infos(LCRZOFILE *pf, 
			    const lcrzo_etha etha);

/*us Name : lcrzo_device_fprint_infos
   Description :
     Print a lot of informations about a lcrzo_device.
   Input parameter(s) :
     *pf : FILE where the function prints
     device : lcrzo_device whose informations are printed
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_device_fprint_infos
   Description :
     Ecrit toutes les informations sur un lcrzo_device.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     device : lcrzo_device dont on affiche les informations
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_device_fprint_infos(LCRZOFILE *pf,
			      const lcrzo_device device);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_data_fprint
   Description :
     Print data.
   Input parameter(s) :
     *pf : FILE where the function prints
     data : data to print
     datasize : size of data
     printtype : lcrzo_printtype to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_data_fprint
   Description :
     Ecrit des donnees.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     data : donnees a ecrire
     datasize : taille de data
     printtype : lcrzo_printtype a employer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_data_fprint(LCRZOFILE *pf, 
		      lcrzo_constdata data,
		      lcrzo_int32 datasize,
		      lcrzo_printtype printtype);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_hdrleth_fprint
   Description :
     Print a lcrzo_hdrleth.
   Input parameter(s) :
     *pf : FILE where the function prints
     hdrleth : lcrzo_hdrleth to print
     printtype : lcrzo_printtype to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hdrleth_fprint
   Description :
     Ecrit des donnees.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     hdrleth : lcrzo_hdrleth a ecrire
     printtype : lcrzo_printtype a employer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hdrleth_fprint(LCRZOFILE *pf,
			 lcrzo_hdrleth hdrleth, 
			 lcrzo_printtype printtype);

/*us Name : lcrzo_hdrlip_fprint
   Description :
     Print a lcrzo_hdrlip.
   Input parameter(s) :
     *pf : FILE where the function prints
     hdrlip : lcrzo_hdrlip to print
     printtype : lcrzo_printtype to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hdrlip_fprint
   Description :
     Ecrit des donnees.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     hdrlip : lcrzo_hdrlip a ecrire
     printtype : lcrzo_printtype a employer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hdrlip_fprint(LCRZOFILE *pf,
			lcrzo_hdrlip hdrlip,
			lcrzo_printtype printtype);

/*us Name : lcrzo_hdrludp_fprint
   Description :
     Print a lcrzo_hdrludp.
   Input parameter(s) :
     *pf : FILE where the function prints
     hdrludp : lcrzo_hdrludp to print
     printtype : lcrzo_printtype to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hdrludp_fprint
   Description :
     Ecrit des donnees.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     hdrludp : lcrzo_hdrludp a ecrire
     printtype : lcrzo_printtype a employer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hdrludp_fprint(LCRZOFILE *pf,
			 lcrzo_hdrludp hdrludp, 
			 lcrzo_printtype printtype);

/*us Name : lcrzo_hdrltcp_fprint
   Description :
     Print a lcrzo_hdrltcp.
   Input parameter(s) :
     *pf : FILE where the function prints
     hdrltcp : lcrzo_hdrltcp to print
     printtype : lcrzo_printtype to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hdrltcp_fprint
   Description :
     Ecrit des donnees.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     hdrltcp : lcrzo_hdrltcp a ecrire
     printtype : lcrzo_printtype a employer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hdrltcp_fprint(LCRZOFILE *pf,
			 lcrzo_hdrltcp hdrltcp, 
			 lcrzo_printtype printtype);

/*us Name : lcrzo_hdrlicmp_fprint
   Description :
     Print a lcrzo_hdrlicmp.
   Input parameter(s) :
     *pf : FILE where the function prints
     hdrlicmp : lcrzo_hdrlicmp to print
     printtype : lcrzo_printtype to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hdrlicmp_fprint
   Description :
     Ecrit des donnees.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     hdrlicmp : lcrzo_hdrlicmp a ecrire
     printtype : lcrzo_printtype a employer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hdrlicmp_fprint(LCRZOFILE *pf,
			  lcrzo_hdrlicmp hdrlicmp,
			  lcrzo_printtype printtype);

/*us Name : lcrzo_hdrlarp_fprint
   Description :
     Print a lcrzo_hdrlarp.
   Input parameter(s) :
     *pf : FILE where the function prints
     hdrlarp : lcrzo_hdrlarp to print
     printtype : lcrzo_printtype to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hdrlarp_fprint
   Description :
     Ecrit des donnees.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     hdrlarp : lcrzo_hdrlarp a ecrire
     printtype : lcrzo_printtype a employer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hdrlarp_fprint(LCRZOFILE *pf,
			 lcrzo_hdrlarp hdrlarp,
			 lcrzo_printtype printtype);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_ipopt_fprint
   Description :
     Print a lcrzo_ipopt.
   Input parameter(s) :
     *pf : FILE where the function prints
     ipopt : lcrzo_ipopt to print
     ipoptsize : size of ipopt
     printtype : lcrzo_printtype to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipopt_fprint
   Description :
     Ecrit des donnees.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     ipopt : lcrzo_ipopt a ecrire
     ipoptsize : taille de ipopt
     printtype : lcrzo_printtype a employer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipopt_fprint(LCRZOFILE *pf,
		       const lcrzo_ipopt ipopt, 
		       lcrzo_uint8 ipoptsize,
		       lcrzo_printtype printtype);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_packet_fprint_eth
   Description :
     Print a packet containing Ethernet.
   Input parameter(s) :
     *pf : FILE where the function prints
     packet : packet to print
     packetsize : size of packet
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_packet_fprint_eth
   Description :
     Ecrit un paquet contenant de l'Ethernet.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     packet : paquet a afficher
     packetsize : taille de packet
     printprofile : lcrzo_printprofile a utiliser
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_packet_fprint_eth(LCRZOFILE *pf, 
			    lcrzo_constdata packet,
			    lcrzo_int32 packetsize,
			    lcrzo_printprofile printprofile);

/*us Name : lcrzo_packet_fprint_ip
   Description :
     Print a packet containing IP.
   Input parameter(s) :
     *pf : FILE where the function prints
     packet : packet to print
     packetsize : size of packet
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_packet_fprint_ip
   Description :
     Ecrit un paquet contenant de l'IP.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     packet : paquet a afficher
     packetsize : taille de packet
     printprofile : lcrzo_printprofile a utiliser
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_packet_fprint_ip(LCRZOFILE *pf,
			   lcrzo_constdata packet,
			   lcrzo_int32 packetsize,
			   lcrzo_printprofile printprofile);

/*us Name : lcrzo_packet_fprint_udp
   Description :
     Print a packet containing UDP.
   Input parameter(s) :
     *pf : FILE where the function prints
     packet : packet to print
     packetsize : size of packet
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_packet_fprint_udp
   Description :
     Ecrit un paquet contenant de l'UDP.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     packet : paquet a afficher
     packetsize : taille de packet
     printprofile : lcrzo_printprofile a utiliser
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_packet_fprint_udp(LCRZOFILE *pf, 
			    lcrzo_constdata packet,
			    lcrzo_int32 packetsize,
			    lcrzo_printprofile printprofile);

/*us Name : lcrzo_packet_fprint_tcp
   Description :
     Print a packet containing TCP.
   Input parameter(s) :
     *pf : FILE where the function prints
     packet : packet to print
     packetsize : size of packet
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_packet_fprint_tcp
   Description :
     Ecrit un paquet contenant du TCP.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     packet : paquet a afficher
     packetsize : taille de packet
     printprofile : lcrzo_printprofile a utiliser
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_packet_fprint_tcp(LCRZOFILE *pf,
			    lcrzo_constdata packet,
			    lcrzo_int32 packetsize,
			    lcrzo_printprofile printprofile);

/*us Name : lcrzo_packet_fprint_icmp
   Description :
     Print a packet containing ICMP.
   Input parameter(s) :
     *pf : FILE where the function prints
     packet : packet to print
     packetsize : size of packet
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_packet_fprint_icmp
   Description :
     Ecrit un paquet contenant de l'ICMP.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     packet : paquet a afficher
     packetsize : taille de packet
     printprofile : lcrzo_printprofile a utiliser
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_packet_fprint_icmp(LCRZOFILE *pf,
			     lcrzo_constdata packet,
			      lcrzo_int32 packetsize,
			      lcrzo_printprofile printprofile);

/*us Name : lcrzo_packet_fprint_arp
   Description :
     Print a packet containing ARP.
   Input parameter(s) :
     *pf : FILE where the function prints
     packet : packet to print
     packetsize : size of packet
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_packet_fprint_arp
   Description :
     Ecrit un paquet contenant de l'ARP.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     packet : paquet a afficher
     packetsize : taille de packet
     printprofile : lcrzo_printprofile a utiliser
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_packet_fprint_arp(LCRZOFILE *pf,
			    lcrzo_constdata packet,
			    lcrzo_int32 packetsize,
			    lcrzo_printprofile printprofile);

/*---------------------------------------------------------------*/
/*us idem, but on stdout */
/*fr idem, mais sur stdout */
#define lcrzo_printtype_print(a) lcrzo_printtype_fprint(stdout,a)
#define lcrzo_printprofile_print(a) lcrzo_printprofile_fprint(stdout,a)

#define lcrzo_hs_print(a, b, c) lcrzo_hs_fprint(stdout, a, b, c)
#define lcrzo_ips_print(a, b, c) lcrzo_ips_fprint(stdout, a, b, c)
#define lcrzo_ipa_print(a, b, c) lcrzo_ipa_fprint(stdout, a, b, c)
#define lcrzo_ipl_print(a, b, c) lcrzo_ipl_fprint(stdout, a, b, c)
#define lcrzo_eths_print(a, b, c) lcrzo_eths_fprint(stdout, a, b, c)
#define lcrzo_etha_print(a, b, c) lcrzo_etha_fprint(stdout, a, b, c)
#define lcrzo_device_print(a, b, c) lcrzo_device_fprint(stdout, a, b, c)

#define lcrzo_hs_print_infos(a) lcrzo_hs_fprint_infos(stdout, a)
#define lcrzo_ips_print_infos(a) lcrzo_ips_fprint_infos(stdout, a)
#define lcrzo_ipa_print_infos(a) lcrzo_ipa_fprint_infos(stdout, a)
#define lcrzo_ipl_print_infos(a) lcrzo_ipl_fprint_infos(stdout, a)
#define lcrzo_eths_print_infos(a) lcrzo_eths_fprint_infos(stdout, a)
#define lcrzo_etha_print_infos(a) lcrzo_etha_fprint_infos(stdout, a)
#define lcrzo_device_print_infos(a) lcrzo_device_fprint_infos(stdout, a)

#define lcrzo_data_print(p,t,a) lcrzo_data_fprint(stdout,p,t,a)

#define lcrzo_hdrleth_print(a,b) lcrzo_hdrleth_fprint(stdout,a,b)
#define lcrzo_hdrlip_print(a,b) lcrzo_hdrlip_fprint(stdout,a,b)
#define lcrzo_hdrludp_print(a,b) lcrzo_hdrludp_fprint(stdout,a,b)
#define lcrzo_hdrltcp_print(a,b) lcrzo_hdrltcp_fprint(stdout,a,b)
#define lcrzo_hdrlicmp_print(a,b) lcrzo_hdrlicmp_fprint(stdout,a,b)
#define lcrzo_hdrlarp_print(a,b) lcrzo_hdrlarp_fprint(stdout,a,b)

#define lcrzo_ipopt_print(p,t,a) lcrzo_ipopt_fprint(stdout,p,t,a)

#define lcrzo_packet_print_eth(p,t,a) lcrzo_packet_fprint_eth(stdout,p,t,a) 
#define lcrzo_packet_print_ip(p,t,a) lcrzo_packet_fprint_ip(stdout,p,t,a) 
#define lcrzo_packet_print_udp(p,t,a) lcrzo_packet_fprint_udp(stdout,p,t,a) 
#define lcrzo_packet_print_tcp(p,t,a) lcrzo_packet_fprint_tcp(stdout,p,t,a) 
#define lcrzo_packet_print_icmp(p,t,a) lcrzo_packet_fprint_icmp(stdout,p,t,a) 
#define lcrzo_packet_print_arp(p,t,a) lcrzo_packet_fprint_arp(stdout,p,t,a) 
