/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

/* -- Note about comments and variable names --
   In versions 1.00 up to 2.08, lcrzo was only written in French (my
   mother tongue). Since version 3.00, lcrzo is written in English, but
   most of comments and variable names are still in French.
   In future versions, I'll try to translate everything into English,
   but as you may guess, that's a quite big work...
   So, for the moment, here are some word translation, if you want to
   be able to understand what's going on in the functions :
     afficher - print     ;  ajouter - add         ;  copie - copy
     dernier - last       ;  donnee - data         ;  ecrire - write
     entete - header      ;  envoye - sent         ;  ferme - close
     fichier - file       ;  lire - read           ;  masque - mask 
     nboct - byte number  ;  ouvre, ouvrir - open  ;  paq, paquet - packet
     plage - range        ;  premier - first       ;  recu - received
     reseau - network     ;  retour - return       ;  taille - size
     tunnel - pipe

*/

#include "lcrzo_priv.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #define _GNU_SOURCE
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <string.h>
 #include <ctype.h>
 #include <fcntl.h>
 #include <errno.h>
 #include <termios.h>
 #include <sys/types.h>
 #include <sys/stat.h>      
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <string.h>
 #include <ctype.h>
 #include <fcntl.h>
 #include <errno.h>
 #include <termios.h>
 #include <sys/types.h>
 #include <sys/stat.h>      
#elif defined LCRZODEF_SYSTEM_Solaris
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <string.h>
 #include <ctype.h>
 #include <fcntl.h>
 #include <errno.h>
 #include <termios.h>
 #include <sys/types.h>
 #include <sys/stat.h>      
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif

/*---------------------------------------------------------------*/
int lcrzo_priv_etha_and(lcrzo_etha a, lcrzo_etha b, lcrzo_etha resu)
{ int i;

  /*parameters verification*/
  if (a==NULL) return(LCRZO_ERR_PANULLPTR);
  if (b==NULL) return(LCRZO_ERR_PANULLPTR);
  
  if (resu!=NULL)
  { for ( i=0 ; i<LCRZO_ETHA_MAXBYTES ; i++ )
    { resu[i]=(lcrzo_uint8)(a[i]&b[i]);
    }
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_etha_or(lcrzo_etha a, lcrzo_etha b, lcrzo_etha resu)
{ int i;
  
  /*parameters verification*/
  if (a==NULL) return(LCRZO_ERR_PANULLPTR);
  if (b==NULL) return(LCRZO_ERR_PANULLPTR);
  
  if (resu!=NULL)
  { for ( i=0 ; i<LCRZO_ETHA_MAXBYTES ; i++ )
    { resu[i]=(lcrzo_uint8)(a[i]|b[i]);
    }
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_etha_xor(lcrzo_etha a, lcrzo_etha b, lcrzo_etha resu)
{ int i;

  /*parameters verification*/
  if (a==NULL) return(LCRZO_ERR_PANULLPTR);
  if (b==NULL) return(LCRZO_ERR_PANULLPTR);
  
  if (resu!=NULL)
  { for ( i=0 ; i<LCRZO_ETHA_MAXBYTES ; i++ )
    { resu[i]=(lcrzo_uint8)(a[i]^b[i]);
    }
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_etha_not(lcrzo_etha a, lcrzo_etha resu)
{ int i;

  /*parameters verification*/
  if (a==NULL) return(LCRZO_ERR_PANULLPTR);
  
  if (resu!=NULL)
  { for ( i=0 ; i<LCRZO_ETHA_MAXBYTES ; i++ )
    { resu[i]=(lcrzo_uint8)(~a[i]);
    }
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_etha_inc(lcrzo_etha a, lcrzo_etha resu)
{ int i;
  lcrzo_uint8 retenue;

  /*parameters verification*/
  if (a==NULL) return(LCRZO_ERR_PANULLPTR);
  
  if (resu!=NULL)
  { retenue=1;
    for ( i=LCRZO_ETHA_MAXBYTES-1 ; i>=0 ; i-- )
    { if ( a[i]==0xFF && retenue ) /*il va y avoir debordement*/
      { resu[i]=0;
      }
      else
      { resu[i]=(lcrzo_uint8)(a[i]+retenue);
        retenue=0; /*plus de retenue a propager*/
      }
    }
  }
  /*on ignore le debordement de ff:ff:ff:ff:ff:ff*/
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_etha_dec(lcrzo_etha a, lcrzo_etha resu)
{ int i;
  lcrzo_uint8 retenue;

  /*parameters verification*/
  if (a==NULL) return(LCRZO_ERR_PANULLPTR);
  
  if (resu!=NULL)
  { retenue=1;
    for ( i=LCRZO_ETHA_MAXBYTES-1 ; i>=0 ; i-- )
    { if ( a[i]==0 && retenue ) /*il va y avoir debordement*/
      { resu[i]=0xFF;
      }
      else
      { resu[i]=(lcrzo_uint8)(a[i]-retenue);
        retenue=0; /*plus de retenue a propager*/
      }
    }
  }
  /*on ignore le debordement de 0:0:0:0:0:0*/
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_etha_shl(lcrzo_etha a, lcrzo_etha resu)
{ int i;
  lcrzo_uint8 retenue, r;

  /*parameters verification*/
  if (a==NULL) return(LCRZO_ERR_PANULLPTR);
  
  if (resu!=NULL)
  { retenue=0;
    for ( i=LCRZO_ETHA_MAXBYTES-1 ; i>=0 ; i-- )
    { r=(lcrzo_uint8)(a[i]&0x80);
      resu[i]=(lcrzo_uint8)((a[i]<<1)|retenue);
      retenue=(lcrzo_uint8)(r>>7);
    }
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_etha_shr(lcrzo_etha a, lcrzo_etha resu)
{ int i;
  lcrzo_uint8 retenue, r;

  /*parameters verification*/
  if (a==NULL) return(LCRZO_ERR_PANULLPTR);
  
  if (resu!=NULL)
  { retenue=0;
    for ( i=0 ; i<LCRZO_ETHA_MAXBYTES ; i++ )
    { r=(lcrzo_uint8)(a[i]&1);
      resu[i]=(lcrzo_uint8)((a[i]>>1)|(retenue<<7));
      retenue=r;
    }
  }
  return(LCRZO_ERR_OK);
}


/*---------------------------------------------------------------*/
int lcrzo_priv_hdrleth_equal_addr(lcrzo_hdrleth hdrleth,
				  const lcrzo_etha ethasrc,
				  const lcrzo_etha ethadst)
{ return(lcrzo_etha_equal(hdrleth.src, ethasrc) &&
	 lcrzo_etha_equal(hdrleth.dst, ethadst));
}

/*---------------------------------------------------------------*/
int lcrzo_priv_hdrlip_equal_addr(lcrzo_hdrlip hdrlip,
				 const lcrzo_ipl iplsrc,
				 const lcrzo_ipl ipldst)
{ return(lcrzo_ipl_equal(hdrlip.saddr, iplsrc) &&
	 lcrzo_ipl_equal(hdrlip.daddr, ipldst));
}

/*---------------------------------------------------------------*/
int lcrzo_priv_hdrludp_equal_port(lcrzo_hdrludp hdrludp,
				  lcrzo_uint16 portsrc,
				  lcrzo_uint16 portdst)
{ return( (hdrludp.sport==portsrc) && (hdrludp.dport==portdst) );
}

/*---------------------------------------------------------------*/
int lcrzo_priv_hdrltcp_equal_port(lcrzo_hdrltcp hdrltcp,
				  lcrzo_uint16 portsrc,
				  lcrzo_uint16 portdst)
{ return( (hdrltcp.sport==portsrc) && (hdrltcp.dport==portdst) );
}

/*-------------------------------------------------------------*/
int lcrzo_priv_stdin(lcrzo_string *puserstring,
		     lcrzo_int32 *puserstringsize)
{ char *ptr;
  lcrzo_int32 fgetstep, ptrallocsize, userstringsize;
  int c;

  fgetstep=3;
  ptrallocsize=fgetstep;
  lcrzo_er(lcrzo_string_alloc(ptrallocsize, &ptr));  

  lcrzo_stdin_flush();

  userstringsize=0;
  c=getc(stdin);  
  while ( c!=0x0D && c!=0x0A && c!=0x00 && c!=EOF )
  { ptr[userstringsize]=(char)c;
    userstringsize++;
    c=getc(stdin);
    if (userstringsize==ptrallocsize)
    { ptrallocsize+=fgetstep;
      lcrzo_er(lcrzo_string_realloc(ptrallocsize, &ptr));  
    }
  }
  ptr[userstringsize]='\0';

  if (puserstring!=NULL)
  { *puserstring=ptr;
  }
  else
  { lcrzo_string_free(ptr);
  }
  if (puserstringsize!=NULL) *puserstringsize=userstringsize;

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_passwd(lcrzo_string *puserstring,
		      lcrzo_int32 *puserstringsize)
{ char *ptr;
  lcrzo_int32 fgetstep, ptrallocsize, userstringsize;
  int c;
  struct termios tios;

  fgetstep=3;
  ptrallocsize=fgetstep;
  lcrzo_er(lcrzo_string_alloc(ptrallocsize, &ptr));  

  lcrzo_stdin_flush();

  if (tcgetattr(STDIN_FILENO, &tios)) return(LCRZO_ERR_FUTCGETATTR);
  tios.c_lflag &= ~ECHO;
  if (tcsetattr(STDIN_FILENO, TCSANOW,  &tios)) return(LCRZO_ERR_FUTCSETATTR);
  userstringsize=0;
  c=getc(stdin);  
  while ( c!=0x0D && c!=0x0A && c!=0x00 && c!=EOF )
  { ptr[userstringsize]=(char)c;
    userstringsize++;
    c=getc(stdin);
    if (userstringsize==ptrallocsize)
    { ptrallocsize+=fgetstep;
      lcrzo_er(lcrzo_string_realloc(ptrallocsize, &ptr));  
    }
  }
  ptr[userstringsize]='\0';
  tios.c_lflag |= ECHO;
  if (tcsetattr(STDIN_FILENO, TCSANOW,  &tios)) return(LCRZO_ERR_FUTCSETATTR);
  puts("");

  if (puserstring!=NULL)
  { *puserstring=ptr;
  }
  else
  { lcrzo_string_free(ptr);
  }
  if (puserstringsize!=NULL) *puserstringsize=userstringsize;

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_manageelempos(lcrzo_int32 elemcount,
			     lcrzo_int32 position,
			     lcrzo_bool wanttoinsertelem,
			     lcrzo_int32 *pposnegative,
			     lcrzo_int32 *ppospositive,
			     int *pdirection, 
			     lcrzo_int32 *pdistance)
{ int sens;
  lcrzo_int32 distance, pospositif, posnegatif;

  /*verification des parametres*/
  lcrzo_bool_verifbof(wanttoinsertelem);

  /*verification et calcul de la position positivee*/
  if (position==0)
  { return(LCRZO_ERR_PONOZERO);
  }
  else if (position>0)
  { if ( position > elemcount+(wanttoinsertelem?1:0) )
    { return(LCRZO_ERR_POTOOBIG);
    }
    posnegatif=position-elemcount-(wanttoinsertelem?2:1);
    pospositif=position;
  }
  else
  { if ( position < -(elemcount+(wanttoinsertelem?1:0)) )
    { return(LCRZO_ERR_POTOOLOW);
    }
    posnegatif=position;
    pospositif=elemcount+(wanttoinsertelem?2:1)+position;
  }

  /*regarde quel sens prendre*/
  if ( pospositif <= -posnegatif )
  { sens=LCRZO_PRIV_INCREASE;
    distance=pospositif;
  }
  else
  { sens=LCRZO_PRIV_DECREASE;
    distance=-posnegatif;
  }

  if (pdirection!=NULL) *pdirection=sens;
  if (pdistance!=NULL) *pdistance=distance;
  if (pposnegative!=NULL) *pposnegative=posnegatif;
  if (ppospositive!=NULL) *ppospositive=pospositif;

  /*  printf("cnt=%ld pos=%ld ins=%d ==> dir %d, distance %ld [%ld,%ld]\n",
	 elemcount, position, wanttoinsertelem,
	 sens, distance, posnegatif, pospositif);*/
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_manageelemrange(lcrzo_int32 elemcount,
			       lcrzo_int32 infposition,
			       lcrzo_int32 supposition,
			       lcrzo_int32 *pinfposnegative,
			       lcrzo_int32 *pinfpospositive,
			       lcrzo_int32 *psupposnegative,
			       lcrzo_int32 *psuppospositive,
			       int *pinfdirection, 
			       lcrzo_int32 *pinfdistance,
			       int *pinfsupdirection, 
			       lcrzo_int32 *pinfsupdistance)
{ lcrzo_int32 infpospositive, suppospositive, infsupdistance;
  int infsupdirection;

  lcrzo_er(lcrzo_priv_manageelempos(elemcount, infposition, 0,
				    pinfposnegative, &infpospositive,
				    pinfdirection, pinfdistance));
  lcrzo_er(lcrzo_priv_manageelempos(elemcount, supposition, 0,
				    psupposnegative, &suppospositive,
				    NULL, NULL));

  /*on calcule le nombre d'elements a voir et le sens*/
  if (suppospositive>=infpospositive)
  { infsupdirection=LCRZO_PRIV_INCREASE;
    infsupdistance=suppospositive-infpospositive;
  }
  else
  { infsupdirection=LCRZO_PRIV_DECREASE;
    infsupdistance=infpospositive-suppospositive;
  }

  if (pinfpospositive!=NULL) *pinfpospositive=infpospositive;
  if (psuppospositive!=NULL) *psuppospositive=suppospositive;
  if (pinfsupdirection!=NULL) *pinfsupdirection=infsupdirection;
  if (pinfsupdistance!=NULL) *pinfsupdistance=infsupdistance;

  /*  printf("pos=%ld*%ld[%ld*%ld] ==> dir %d, distance %ld\n",
	 infposition, supposition, infpospositive, suppospositive, 
	 infsupdirection, infsupdistance);
  */
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_manageelemrangefast(lcrzo_int32 elemcount,
				   lcrzo_int32 infposition,
				   lcrzo_int32 supposition,
				   lcrzo_int32 *pinfposnegative,
				   lcrzo_int32 *pinfpospositive,
				   lcrzo_int32 *psupposnegative,
				   lcrzo_int32 *psuppospositive,
				   int *pinfdirection, 
				   lcrzo_int32 *pinfdistance,
				   int *pinfsupdirection, 
				   lcrzo_int32 *pinfsupdistance)
{ lcrzo_int32 infdist, supdist;

  lcrzo_er(lcrzo_priv_manageelempos(elemcount, infposition, 0, NULL, NULL,
				    NULL, &infdist));
  lcrzo_er(lcrzo_priv_manageelempos(elemcount, supposition, 0, NULL, NULL,
				    NULL, &supdist));

  if (infdist<=supdist)
  { lcrzo_er(lcrzo_priv_manageelemrange(elemcount,
					infposition, supposition,
					pinfposnegative, pinfpospositive,
					psupposnegative, psuppospositive,
					pinfdirection, pinfdistance,
					pinfsupdirection, pinfsupdistance));
  }
  else
  { lcrzo_er(lcrzo_priv_manageelemrange(elemcount,
					supposition, infposition,
					pinfposnegative, pinfpospositive,
					psupposnegative, psuppospositive,
					pinfdirection, pinfdistance,
					pinfsupdirection, pinfsupdistance));
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_managedataposneg(lcrzo_int32 datasize,
				lcrzo_int32 negative,
				lcrzo_int32 *pnegative,
				lcrzo_int32 *ppositive)
{
  if (negative==0) return(LCRZO_ERR_PONOZERO);
  if (negative<datasize-1)return(LCRZO_ERR_POTOOBIG);
  
  if (pnegative!=NULL) *pnegative=negative;
  if (ppositive!=NULL) *ppositive=negative+datasize+1;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_managedatapospos(lcrzo_int32 datasize,
				lcrzo_int32 positive,
				lcrzo_int32 *pnegative,
				lcrzo_int32 *ppositive)
{
  if (positive==0) return(LCRZO_ERR_PONOZERO);
  if (positive>datasize+1)return(LCRZO_ERR_POTOOBIG);
  
  if (pnegative!=NULL) *pnegative=positive-datasize-1;
  if (ppositive!=NULL) *ppositive=positive;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_managedatarange(lcrzo_int32 datasize,
			       lcrzo_int32 infposition,
			       lcrzo_int32 supposition,
			       lcrzo_int32 *pinfposnegative,
			       lcrzo_int32 *pinfpospositive,
			       lcrzo_int32 *psupposnegative,
			       lcrzo_int32 *psuppospositive,
			       int *pinfdirection, 
			       lcrzo_int32 *pinfdistance,
			       int *pinfsupdirection, 
			       lcrzo_int32 *pinfsupdistance)
{ lcrzo_int32 infposnegative, infpospositive, supposnegative;
  lcrzo_int32 suppospositive, infdistance, infsupdistance;
  int infdirection, infsupdirection;

  if (infposition==0) return(LCRZO_ERR_PONOZERO);
  if (supposition==0) return(LCRZO_ERR_PONOZERO);

  /*check infposition*/
  if (infposition>0)
  { if (infposition>datasize+1)return(LCRZO_ERR_POTOOBIG);
    infposnegative=infposition-datasize-2;
    infpospositive=infposition;
  }
  else
  { if (-infposition>datasize+1)return(LCRZO_ERR_POTOOLOW);
    infposnegative=infposition;
    infpospositive=datasize+2+infposition;
  }

  /*check supposition*/
  if (supposition>0)
  { if (supposition>datasize+1)return(LCRZO_ERR_POTOOBIG);
    supposnegative=supposition-datasize-2;
    suppospositive=supposition;
  }
  else
  { if (-supposition>datasize+1)return(LCRZO_ERR_POTOOLOW);
    supposnegative=supposition;
    suppospositive=datasize+2+supposition;
  }

  /*calculate infdirection and distance*/
  if (infpospositive>(datasize+1)/2)
  { infdirection=LCRZO_PRIV_DECREASE;
    infdistance=-infposnegative-1;
  }
  else
  { infdirection=LCRZO_PRIV_INCREASE;
    infdistance=infpospositive-1;
  }

  /*on calcule le nombre d'elements a voir et le sens*/
  if (suppospositive>=infpospositive)
  { infsupdirection=LCRZO_PRIV_INCREASE;
    infsupdistance=suppospositive-infpospositive;
  }
  else
  { infsupdirection=LCRZO_PRIV_DECREASE;
    infsupdistance=infpospositive-suppospositive;
  }

  if (pinfposnegative!=NULL) *pinfposnegative=infposnegative;
  if (psupposnegative!=NULL) *psupposnegative=supposnegative;
  if (pinfpospositive!=NULL) *pinfpospositive=infpospositive;
  if (psuppospositive!=NULL) *psuppospositive=suppospositive;
  if (pinfdirection!=NULL) *pinfdirection=infdirection;
  if (pinfdistance!=NULL) *pinfdistance=infdistance;
  if (pinfsupdirection!=NULL) *pinfsupdirection=infsupdirection;
  if (pinfsupdistance!=NULL) *pinfsupdistance=infsupdistance;

  /*
  printf("%ld pos=%ld*%ld[%ld*%ld][%ld*%ld] ==> %d_%ld %d_%ld\n", datasize,
	 infposition, supposition, 
	 infposnegative, supposnegative, infpospositive, suppospositive, 
	 infdirection, infdistance, infsupdirection, infsupdistance);
  */
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_priv_data_u16_init_data(lcrzo_constdata datain,
				  lcrzo_uint16 datainsize,
				  lcrzo_uint16 dataoutmaxsize,
				  lcrzo_data dataout,
				  lcrzo_uint16 *pdataoutsize)
{ return(lcrzo_priv_data_u16_append_data(datain, datainsize, 0,
				    dataoutmaxsize, dataout, pdataoutsize));
}

/*-------------------------------------------------------------*/
int lcrzo_priv_data_u16_initm_data(lcrzo_constdata datain,
				   lcrzo_uint16 datainsize,
				   lcrzo_data *pdataout,
				   lcrzo_uint16 *pdataoutsize)
{ lcrzo_data ptr;

  ptr=NULL;
  lcrzo_er(lcrzo_priv_data_u16_appendm_data(datain, datainsize, 0,
					    &ptr, pdataoutsize));
  if (pdataout!=NULL) *pdataout=ptr;
  else lcrzo_data_free(ptr);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_data_u16_append_data(lcrzo_constdata datain,
				    lcrzo_uint16 datainsize,
				    lcrzo_uint16 dataoutcurrentsize,
				    lcrzo_uint16 dataoutmaxsize,
				    lcrzo_data dataout,
				    lcrzo_uint16 *pdataoutsize)
{
  /*parameters verification*/
  if (datain==NULL && datainsize) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (dataoutcurrentsize>dataoutmaxsize) return(LCRZO_ERR_PADATACURSUPMAX);

  if (dataoutcurrentsize+datainsize<=dataoutmaxsize)
  { if (dataout!=NULL)
    { memcpy(dataout+dataoutcurrentsize, datain, datainsize);
    }
    if (pdataoutsize!=NULL) 
    { *pdataoutsize=(lcrzo_uint16)(dataoutcurrentsize+datainsize);
    }
    return(LCRZO_ERR_OK);
  }
  else
  { if (dataout!=NULL)
    { memcpy(dataout+dataoutcurrentsize, datain,
             dataoutmaxsize-dataoutcurrentsize);
    }
    if (pdataoutsize!=NULL) *pdataoutsize=dataoutmaxsize;
    return(LCRZO_ERR_OKDATATRUNCATED);
  }
 
  /*should never be here*/
  return(LCRZO_ERR_IEINTERNALERROR);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_data_u16_appendm_data(lcrzo_constdata datain,
				     lcrzo_uint16 datainsize,
				     lcrzo_uint16 dataoutcurrentsize,
				     lcrzo_data *pdataout,
				     lcrzo_uint16 *pdataoutsize)
{ lcrzo_uint16 dataoutsize;

  /*parameters verification*/
  if (datain==NULL && datainsize) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*compute the output size*/
  dataoutsize=(lcrzo_uint16)(dataoutcurrentsize+datainsize);
  if (pdataoutsize!=NULL) 
  { *pdataoutsize=(lcrzo_uint16)dataoutsize;
  }

  /*init pdataout*/
  if (pdataout!=NULL)
  { lcrzo_er(lcrzo_data_realloc(dataoutsize, pdataout));
    memcpy(*pdataout+dataoutcurrentsize, datain, datainsize);
  }

  return(LCRZO_ERR_OK);
}
