/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

/* -- Note about comments and variable names --
   In versions 1.00 up to 2.08, lcrzo was only written in French (my
   mother tongue). Since version 3.00, lcrzo is written in English, but
   most of comments and variable names are still in French.
   In future versions, I'll try to translate everything into English,
   but as you may guess, that's a quite big work...
   So, for the moment, here are some word translation, if you want to
   be able to understand what's going on in the functions :
     afficher - print     ;  ajouter - add         ;  copie - copy
     dernier - last       ;  donnee - data         ;  ecrire - write
     entete - header      ;  envoye - sent         ;  ferme - close
     fichier - file       ;  lire - read           ;  masque - mask 
     nboct - byte number  ;  ouvre, ouvrir - open  ;  paq, paquet - packet
     plage - range        ;  premier - first       ;  recu - received
     reseau - network     ;  retour - return       ;  taille - size
     tunnel - pipe

*/

#include "lcrzo_priv.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #if LCRZODEF_LIBPCAPINSTALLED==1
  #define _GNU_SOURCE
  #include <stdlib.h>
  #include <string.h>
  #include <pcap.h>
  #include <net/if.h>
 #else
  #include <stdlib.h>
  #include <string.h>
  #include <stdio.h>
 #endif
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #if LCRZODEF_LIBPCAPINSTALLED==1
  #include <stdlib.h>
  #include <string.h>
  #include <pcap.h>
  #include <sys/socket.h>
  #include <net/if.h>
 #else
  #include <stdlib.h>
  #include <string.h>
  #include <stdio.h>
 #endif
#elif defined LCRZODEF_SYSTEM_Solaris
 #if LCRZODEF_LIBPCAPINSTALLED==1
  #include <stdlib.h>
  #include <string.h>
  #include <pcap.h>
  #include <sys/socket.h>
  #include <net/if.h>
 #else
  #include <stdlib.h>
  #include <string.h>
  #include <stdio.h>
 #endif
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif


/*-------------------------------------------------------------*/
typedef struct {
  lcrzo_time heureajout;
  lcrzo_ipl ipls;
  lcrzo_ipl ipld;
  lcrzo_uint16 id;
  lcrzo_uint8 protocol;
} lcrzo_priv_paqipvus;
typedef struct {
  lcrzo_time heurepremierfrag;
  lcrzo_hdrleth hdreth;
  lcrzo_hdrlip hdrip;
  lcrzo_uint8 hdripcorrect;
  lcrzo_ipopt ipopt;
  lcrzo_uint8 nboctipopt;
  lcrzo_array64k dataip;
  lcrzo_uint16 nboctdataip;
  lcrzo_uint8 datasauvees[8192+1]; /*car on va se servir de string*/
} lcrzo_priv_paqipattente;
typedef struct {
  lcrzo_time heuredernieremaj;
  lcrzo_ipl ipls;
  lcrzo_ipl ipld;
  lcrzo_uint16 ports;
  lcrzo_uint16 portd;
  lcrzo_uint32 octetattendu;
} lcrzo_priv_paqtcpvus;
typedef struct {
  lcrzo_time heureajout;
  lcrzo_ipl ipls;
  lcrzo_ipl ipld;
  lcrzo_uint16 ports;
  lcrzo_uint16 portd;
  lcrzo_uint32 seqnum;
  lcrzo_uint8 *pdata;
  lcrzo_uint16 nboctdata;
  lcrzo_uint32 nboctdonneestcpsynfin; /*nboct + 1 si syn ou fin*/
} lcrzo_priv_paqtcpattente;
void lcrzo_priv_effacepaqtcpattente(void *pelem);
void lcrzo_priv_effacepaqtcpattente(void *pelem)
{ lcrzo_priv_paqtcpattente *ppta;
  ppta=(lcrzo_priv_paqtcpattente*)pelem;
  lcrzo_data_free(ppta->pdata);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_affpaqipvus(const void *pelem, const void *pinfos);
int lcrzo_priv_affpaqipvus(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqipvus *ppiv;
  ppiv=(const lcrzo_priv_paqipvus *)pelem;
  printf("%08lu; %lX->%lX id%d proto%d\n", ppiv->heureajout.sec,
	 ppiv->ipls, ppiv->ipld, ppiv->id, ppiv->protocol);
  return(LCRZO_ERR_OK);
}
int lcrzo_priv_affpaqipattente(const void *pelem, const void *pinfos);
int lcrzo_priv_affpaqipattente(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqipattente *ppia;
  int taille;
  ppia=(const lcrzo_priv_paqipattente *)pelem;
  if (ppia->datasauvees[8192]) taille=-1;
  else taille=strlen((const char*)ppia->datasauvees);
  printf("%08lu; %lX->%lX(%d) nboctdataip%u strlen%d\n", 
	 ppia->heurepremierfrag.sec,
	 ppia->hdrip.saddr, ppia->hdrip.daddr, ppia->hdripcorrect,
	 ppia->nboctdataip, taille);
  return(LCRZO_ERR_OK);
}
int lcrzo_priv_affpaqtcpvus(const void *pelem, const void *pinfos);
int lcrzo_priv_affpaqtcpvus(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqtcpvus *pptv;
  pptv=(const lcrzo_priv_paqtcpvus *)pelem;
  printf("%08lu; %lX:%d->%lX:%d attend%lu\n", pptv->heuredernieremaj.sec,
	 pptv->ipls, pptv->ports, pptv->ipld, pptv->portd, pptv->octetattendu);
  return(LCRZO_ERR_OK);
}
int lcrzo_priv_affpaqtcpattente(const void *pelem, const void *pinfos);
int lcrzo_priv_affpaqtcpattente(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqtcpattente *ppta;
  ppta=(const lcrzo_priv_paqtcpattente *)pelem;
  printf("%08lu; %lX:%d->%lX:%d seqnum%lu data%u tcp+synfin%lu\n",
	 ppta->heureajout.sec,
	 ppta->ipls, ppta->ports, ppta->ipld, ppta->portd, ppta->seqnum,
	 ppta->nboctdata, ppta->nboctdonneestcpsynfin);
  return(LCRZO_ERR_OK);
}
int lcrzo_priv_affdebugsniff(lcrzo_sniff cs);
int lcrzo_priv_affdebugsniff(lcrzo_sniff cs)
{ printf("lcrzo_priv_paqipvus:\n");
  lcrzo_epr(lcrzo_list_loop_all(cs.seenippackets,
				&lcrzo_priv_affpaqipvus, NULL));
  printf("lcrzo_priv_waitingippackets:\n");
  lcrzo_epr(lcrzo_list_loop_all(cs.waitingippackets,
				&lcrzo_priv_affpaqipattente, NULL));
  printf("lcrzo_priv_paqtcpvus:\n");
  lcrzo_epr(lcrzo_list_loop_all(cs.seentcppackets,
				&lcrzo_priv_affpaqtcpvus, NULL));
  printf("lcrzo_priv_waitingtcppackets:\n");
  lcrzo_epr(lcrzo_list_loop_all(cs.waitingtcppackets,
				&lcrzo_priv_affpaqtcpattente, NULL));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
#if LCRZODEF_LIBPCAPINSTALLED==1
int lcrzo_sniff_init(const lcrzo_device device,
		     lcrzo_int32 nbmaxoct,
		     const char *filtre, 
		     lcrzo_sniff *pcs)
{ int retour, pcapfileno;
  bpf_u_int32 localnet, netmask;
  struct bpf_program fcode;
  lcrzo_device dev;
  lcrzo_string filter2;

  /*parameters verification*/
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_device_verifbof(device);
  if (pcs==NULL) return(LCRZO_ERR_PANULLPTR);
  if (nbmaxoct<=0) return(LCRZO_ERR_PATOOLOW);

  /*oblige car pcap ne travaille pas avec des const*/
  lcrzo_string_init_text(device, LCRZO_DEVICE_MAXBYTES, dev);
  pcs->ppcapt = pcap_open_live(dev, nbmaxoct+sizeof(lcrzo_hdrpeth),
			       1, 500, lcrzo_global.errmsgsys);
  if ( pcs->ppcapt == NULL ) return(LCRZO_ERR_FUPCAPOPEN);

  /* get the netmask */
  retour=pcap_lookupnet(dev, &localnet, &netmask, lcrzo_global.errmsgsys);
  if ( retour ) 
  { return(LCRZO_ERR_FUPCAPLOOKUPNET);
  }

  /* in all cases, we use a filter. We should not need it, but
     the libpcap package modified by RedHat requires it... */
  if ( filtre==NULL )
  { lcrzo_er(lcrzo_string_initm_text("", &filter2));
  }
  else
  { lcrzo_er(lcrzo_string_initm_coretext(filtre, &filter2));
  }
  retour=pcap_compile(pcs->ppcapt, &fcode, filter2, 0, netmask);
  lcrzo_string_free(filter2);
  if ( retour )
  { lcrzo_string_init_text(pcap_geterr(pcs->ppcapt),
			   LCRZO_GLOERRMSG_MAXBYTES, 
			   lcrzo_global.errmsgsys);
    return(LCRZO_ERR_FUPCAPCOMPILE);
  }
  retour=pcap_setfilter(pcs->ppcapt, &fcode);
  if ( retour )
  { lcrzo_string_init_text(pcap_geterr(pcs->ppcapt),
			   LCRZO_GLOERRMSG_MAXBYTES, 
			   lcrzo_global.errmsgsys);
    return(LCRZO_ERR_FUPCAPSETFILTER);
  }

  /*par defaut, on bloque la lecture*/
  pcs->beblocking=LCRZO_TRUE;
  pcapfileno=pcap_fileno((pcap_t*)(pcs->ppcapt));
  lcrzo_er(lcrzo_fd_block_set(pcapfileno, LCRZO_TRUE));

  /*initialise les listes des packets fragmentes et tcp*/
  lcrzo_er(lcrzo_list_init(&(pcs->waitingippackets),
			   sizeof(lcrzo_priv_paqipattente), NULL));
  lcrzo_er(lcrzo_list_init(&(pcs->seenippackets),
			   sizeof(lcrzo_priv_paqipvus), NULL));
  lcrzo_er(lcrzo_list_init(&(pcs->waitingtcppackets),
			   sizeof(lcrzo_priv_paqtcpattente),
			   &lcrzo_priv_effacepaqtcpattente));
  lcrzo_er(lcrzo_list_init(&(pcs->seentcppackets),
			   sizeof(lcrzo_priv_paqtcpvus), NULL));

  /*initialize the variables*/
  pcs->para_ipreastimeout1=lcrzo_global.sniff_ipreastimeout1;
  pcs->para_ipreastimeout2=lcrzo_global.sniff_ipreastimeout2;
  pcs->para_tcpreortimeout1=lcrzo_global.sniff_tcpreortimeout1;
  pcs->para_tcpreortimeout2=lcrzo_global.sniff_tcpreortimeout2;

  return(LCRZO_ERR_OK);
}
#else
int lcrzo_sniff_init(const lcrzo_device device,
		     lcrzo_int32 nbmaxoct,
		     const char *filtre, 
		     lcrzo_sniff *pcs)
{ if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
  { lcrzo_string_init_text("Fonction lcrzo_sniff_init non implementee car ",
		       LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    lcrzo_string_append_text("libpcap n'est pas installe. Vous devez installer ",
			 LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    lcrzo_string_append_text("libpcap et recompiler liblcrzo.",
			 LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
  }
  else 
  { lcrzo_string_init_text("Function lcrzo_sniff_init unimplemented because ",
		       LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    lcrzo_string_append_text("libpcap is not installed. You should install ",
			 LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    lcrzo_string_append_text("libpcap and recompile liblcrzo.",
			 LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
  }
  return (LCRZO_ERR_IEUNIMPLEMENTED);
}
#endif

/*-------------------------------------------------------------*/
#if LCRZODEF_LIBPCAPINSTALLED==1
/*******/
int lcrzo_priv_sniff_nextmnormal(lcrzo_sniff *pcs,
				 lcrzo_bool beblocking,
				 lcrzo_data *pdata,
				 lcrzo_int32 *pdatasize);
int lcrzo_priv_sniff_nextmnormal(lcrzo_sniff *pcs,
				 lcrzo_bool beblocking,
				 lcrzo_data *pdata,
				 lcrzo_int32 *pdatasize)
{ const lcrzo_uint8 *pptr;
  struct pcap_pkthdr hdr[3];
  int pcapfileno;

  /*faut-il changer le bloquage ?*/
  if ( pcs->beblocking != beblocking )
  { pcapfileno=pcap_fileno((pcap_t*)(pcs->ppcapt));
    lcrzo_er(lcrzo_fd_block_set(pcapfileno, beblocking));
  } 

  /* In this function, we use an array for hdr, even if it
     should not be necessary. Indeed, there is an overflow 
     in libpcap package modified by RedHat, so we have
     to use this trick ... (note : there is no problem
     with tcpdump because they only use pcap_loop) */

  /*si on est bloquant*/
  if (beblocking)
  { pptr = NULL;
    while( pptr==NULL ) 
    { memset((char *)hdr, 0, 3*sizeof(struct pcap_pkthdr));
      pptr = pcap_next(pcs->ppcapt, &(hdr[1]));
    }
    lcrzo_er(lcrzo_data_initm_data(pptr, hdr[1].caplen, pdata, pdatasize));
  }
  /*sinon*/
  else
  { memset(&hdr, 0, 3*sizeof(struct pcap_pkthdr));
    pptr = pcap_next(pcs->ppcapt, &(hdr[1]));
    if (pptr==NULL) /*rien de present*/
    { if (pdatasize!=NULL) *pdatasize=0;
      return(LCRZO_ERR_OKTEMPDATAEND);
    }
    else /*capture un packet*/
    { lcrzo_er(lcrzo_data_initm_data(pptr, hdr[1].caplen, pdata, pdatasize));
    }
  }
  return(LCRZO_ERR_OK);
}

/*******/
int lcrzo_priv_purgpaqipvuscrit(const void *pelem, const void *pinfos);
int lcrzo_priv_purgpaqipvuscrit(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqipvus *pfi;
  lcrzo_uint32 secmini;

  pfi=(const lcrzo_priv_paqipvus *)pelem;
  secmini=*(const lcrzo_uint32 *)pinfos;
  if ( pfi->heureajout.sec < secmini ) return(1);
  return(0);
}
int lcrzo_priv_cherpaqipvuscrit(const void *pelem, const void *pinfos);
int lcrzo_priv_cherpaqipvuscrit(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqipvus *pfi1;
  const lcrzo_priv_paqipvus *pfi2;

  pfi1=(const lcrzo_priv_paqipvus *)pelem;
  pfi2=(const lcrzo_priv_paqipvus *)pinfos;
  if ( pfi1->ipls==pfi2->ipls &&
       pfi1->ipld==pfi2->ipld &&
       pfi1->protocol==pfi2->protocol &&
       pfi1->id==pfi2->id ) return(1);
  return(0);
}
int lcrzo_priv_cherpaqipvus(lcrzo_sniff sniff,
			    lcrzo_list *pliste,
			    lcrzo_ipl ipls,
			    lcrzo_ipl ipld,
			    lcrzo_uint8 protocol,
			    lcrzo_uint16 id,
			    lcrzo_int32 *ppos);
int lcrzo_priv_cherpaqipvus(lcrzo_sniff sniff,
			    lcrzo_list *pliste,
			    lcrzo_ipl ipls,
			    lcrzo_ipl ipld,
			    lcrzo_uint8 protocol,
			    lcrzo_uint16 id,
			    lcrzo_int32 *ppos)
{ static int i=0;
  lcrzo_time heure;
  lcrzo_uint32 secmini;
  lcrzo_int32 retour;
  lcrzo_priv_paqipvus piv;

  /*on commence par purger*/ 
  i++;
  if (!(i%5)) /*pour ne pas le faire a chaque passage*/
  { /*on timeout apres xx secondes*/
    lcrzo_er(lcrzo_time_init(&heure));
    secmini=heure.sec-sniff.para_ipreastimeout1;
    /*on purge*/
    lcrzo_er(lcrzo_list_remove_criteria_all(pliste,
					    &lcrzo_priv_purgpaqipvuscrit,
					    &secmini));
    i-=5;
  }

  /*on se sert d'un piv pour faire la recherche*/
  piv.ipls=ipls;
  piv.ipld=ipld;
  piv.protocol=protocol;
  piv.id=id;
  retour=lcrzo_list_search_all(*pliste, &lcrzo_priv_cherpaqipvuscrit,
			       &piv, ppos, NULL);
  if (retour==LCRZO_ERR_OKSEARCHNOTFOUND) {*ppos=0; retour=LCRZO_ERR_OK;}
  if (retour!=LCRZO_ERR_OK) return(retour);

  return(LCRZO_ERR_OK);
}
/*--*/
int lcrzo_priv_purgpaqipattentecrit(const void *pelem, const void *pinfos);
int lcrzo_priv_purgpaqipattentecrit(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqipattente *pfi;
  lcrzo_uint32 secmini;

  pfi=(const lcrzo_priv_paqipattente *)pelem;
  secmini=*(const lcrzo_uint32 *)pinfos;
  if ( pfi->heurepremierfrag.sec < secmini ) return(1);
  return(0);
}
int lcrzo_priv_cherpaqipattentecrit(const void *pelem, const void *pinfos);
int lcrzo_priv_cherpaqipattentecrit(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqipattente *pfi1;
  const lcrzo_priv_paqipvus *pfi2;

  pfi1=(const lcrzo_priv_paqipattente *)pelem;
  pfi2=(const lcrzo_priv_paqipvus *)pinfos;
  if ( pfi1->hdrip.saddr==pfi2->ipls &&
       pfi1->hdrip.daddr==pfi2->ipld &&
       pfi1->hdrip.protocol==pfi2->protocol &&
       pfi1->hdrip.id==pfi2->id ) return(1);
  return(0);
}
int lcrzo_priv_cherpaqipattente(lcrzo_sniff sniff,
				lcrzo_list *pliste,
				lcrzo_ipl ipls,
				lcrzo_ipl ipld,
				lcrzo_uint8 protocol,
				lcrzo_uint16 id,
				lcrzo_int32 *ppos,
				lcrzo_priv_paqipattente *pfi);
int lcrzo_priv_cherpaqipattente(lcrzo_sniff sniff,
				lcrzo_list *pliste,
				lcrzo_ipl ipls,
				lcrzo_ipl ipld,
				lcrzo_uint8 protocol,
				lcrzo_uint16 id,
				lcrzo_int32 *ppos,
				lcrzo_priv_paqipattente *pfi)
{ static int i=0;
  lcrzo_time heure;
  lcrzo_uint32 secmini;
  lcrzo_int32 retour;
  lcrzo_priv_paqipvus piv;

  /*on commence par purger*/ 
  i++;
  if (!(i%5)) /*pour ne pas le faire a chaque passage*/
  { /*on timeout apres xx secondes*/
    lcrzo_er(lcrzo_time_init(&heure));
    secmini=heure.sec-sniff.para_ipreastimeout2;
    /*on purge*/
    lcrzo_er(lcrzo_list_remove_criteria_all(pliste,
					    &lcrzo_priv_purgpaqipattentecrit,
					    &secmini));
    i-=5;
  }

  /*on se sert d'un piv pour faire la recherche*/
  piv.ipls=ipls;
  piv.ipld=ipld;
  piv.protocol=protocol;
  piv.id=id;
  retour=lcrzo_list_search_all(*pliste, &lcrzo_priv_cherpaqipattentecrit,
			       &piv, ppos, pfi);
  if (retour==LCRZO_ERR_OKSEARCHNOTFOUND) {*ppos=0; retour=LCRZO_ERR_OK;}
  if (retour!=LCRZO_ERR_OK) return(retour);

  return(LCRZO_ERR_OK);
}
int lcrzo_priv_sniff_nextmreasip(lcrzo_sniff *pcs,
				 lcrzo_bool beblocking,
				 lcrzo_data *pdata,
				 lcrzo_int32 *pdatasize);
int lcrzo_priv_sniff_nextmreasip(lcrzo_sniff *pcs,
				 lcrzo_bool beblocking,
				 lcrzo_data *pdata,
				 lcrzo_int32 *pdatasize)
{ int retour;
  lcrzo_hdrleth hdreth;
  lcrzo_hdrlip hdrip;
  lcrzo_ipopt ipopt;
  lcrzo_uint8 nboctipopt;
  lcrzo_data dataip;
  lcrzo_uint16 nboctdataip;
  lcrzo_int32 poselem;
  lcrzo_priv_paqipattente pia;
  lcrzo_priv_paqipvus piv;
  lcrzo_data sniffdata;
  lcrzo_int32 sniffdatasize;

  while(1) /*boucle infinie dont on sort eventuellement en cours*/
  { /*affichage de debug*/
    /*lcrzo_er(lcrzo_priv_affdebugsniff(*pcs));*/
    /*sniffe et sort sur erreur*/
    lcrzo_er(lcrzo_priv_sniff_nextmnormal(pcs, beblocking,
					  &sniffdata, &sniffdatasize));
    /*si on arrive ici, on a retourne LCRZO_ERR_OK, donc on a des donnees
      quelle que soit la valeur de beblocking*/
    /*si ce n'est pas de l'IP, ou que le packet n'est pas fragmente*/
    retour=lcrzo_packet_decodem_ethipoptdata(sniffdata, sniffdatasize,
					     &hdreth, &hdrip,
					     ipopt, &nboctipopt, 
					     &dataip, &nboctdataip);
    if (retour!=LCRZO_ERR_OK)
    { if (pdatasize!=NULL) *pdatasize=sniffdatasize;
      if (pdata!=NULL) *pdata=sniffdata;
      else lcrzo_data_free(sniffdata);
      return(LCRZO_ERR_OK);
    }
    /*si l'on arrive ici, ce packet est de l'ip*/
    /*on cherche a savoir si on a deja vu ce packet*/
    lcrzo_er(lcrzo_priv_cherpaqipvus(*pcs, &(pcs->seenippackets),
				      hdrip.saddr, hdrip.daddr,
				      hdrip.protocol, hdrip.id, &poselem));
    if (poselem!=0) /*trouve dans la liste : c'est un doublon*/
    { /*on n'a rien a fournir a l'appelant*/
      lcrzo_data_free(sniffdata);
      lcrzo_data_free(dataip);
      if (!beblocking) return(LCRZO_ERR_OKTEMPDATAEND);
      /*sinon continue la boucle infinie pour etre bloquant*/
      continue;
    }
    if ( hdrip.morefrag==0 && hdrip.offsetfrag==0 )
    { /*ce packet n'est pas fragmente : ajoute dans la liste des packets vus*/
      lcrzo_er(lcrzo_time_init(&(piv.heureajout)));
      piv.ipls=hdrip.saddr;
      piv.ipld=hdrip.daddr;
      piv.id=hdrip.id;
      piv.protocol=hdrip.protocol;
      lcrzo_efr(lcrzo_list_add_first(&(pcs->seenippackets), &piv),
		lcrzo_data_free(sniffdata));
      /*quitte : passe ce packet a l'appelant*/
      if (pdatasize!=NULL) *pdatasize=sniffdatasize;
      if (pdata!=NULL) *pdata=sniffdata;
      else lcrzo_data_free(sniffdata);
      lcrzo_data_free(dataip);
      return(LCRZO_ERR_OK);
    }
    lcrzo_data_free(sniffdata);
    /*si on arrive ici, ce packet est un fragment, on tente de le
      reassembler*/

    /*on regarde si ce packet a deja ete mis dans la liste*/
    lcrzo_efr(lcrzo_priv_cherpaqipattente(*pcs, &(pcs->waitingippackets),
					  hdrip.saddr, hdrip.daddr,
					  hdrip.protocol, hdrip.id, 
					  &poselem, &pia),
	      lcrzo_data_free(dataip));
    if (poselem==0) /*pas trouve dans la liste*/
    { /*on va l'ajouter dans la liste*/
      lcrzo_er(lcrzo_time_init(&(pia.heurepremierfrag)));
      /*cette taille ne sera connue que lors de la reception du dernier*/
      pia.nboctdataip=0;
      /*l'hdrip correct vient du premier fragment*/
      pia.hdripcorrect=0;
      /*initialise les fragments recus*/
      memset(pia.datasauvees, 0, 8192+1); /*car on va se servir de string*/
    }

    /*toutes les entetes sont copiees en provenance du premier fragment*/
    if ( hdrip.offsetfrag==0 )
    { memcpy(&(pia.hdreth), &(hdreth), sizeof(lcrzo_hdrleth));
      memcpy(&(pia.hdrip), &(hdrip), sizeof(lcrzo_hdrlip));
      memcpy(pia.ipopt, ipopt, nboctipopt);
      pia.nboctipopt=nboctipopt;
      pia.hdripcorrect=1;
    }
    /*on a besoin de l'entete, meme si on n'a pas encore recu le premier frag*/
    if ( ! (pia.hdripcorrect) )
    { memcpy(&(pia.hdrip), &(hdrip), sizeof(lcrzo_hdrlip));
    }
    /*la taille des donnees est calculee a partir du dernier fragment*/
    if ( hdrip.morefrag==0 )
    { pia.nboctdataip=(lcrzo_uint16)(nboctdataip+8*hdrip.offsetfrag);
    }
    /*mise a jour des donnees*/
    memcpy(pia.dataip+8*hdrip.offsetfrag, dataip, nboctdataip);
    lcrzo_data_free(dataip);
    memset(pia.datasauvees+hdrip.offsetfrag, 1, (nboctdataip+7)/8);

    if (poselem==0)
    { /*on l'ajoute dans la liste*/
      lcrzo_er(lcrzo_list_add_first(&(pcs->waitingippackets), &pia));
    }
    else
    { /*on remplace le vieux par le nouveau*/
      lcrzo_er(lcrzo_list_replace_pos(&(pcs->waitingippackets),poselem, &pia));
    }
 
    /*si on a toutes les donnees du packet, il ne doit plus y avoir
      de '\0' (terminateur de string) dans datasauvees*/
    if ( pia.nboctdataip &&
	 strlen((const char *)pia.datasauvees)==
	 (lcrzo_int32)((pia.nboctdataip+7)/8) )
    { /*on reconstruit ce packet*/
      pia.hdrip.totlen=(lcrzo_uint16)( sizeof(lcrzo_hdrpip)+pia.nboctipopt+
				       pia.nboctdataip );
      pia.hdrip.offsetfrag=0;
      pia.hdrip.morefrag=0;
      pia.hdrip.check=0;
      lcrzo_er(lcrzo_packet_initm_ethipoptdata(pia.hdreth, pia.hdrip,
					       pia.ipopt, pia.nboctipopt,
					       pia.dataip, pia.nboctdataip,
					       pdata, pdatasize));
      /*ajoute ce packet dans la liste des packets vus*/
      lcrzo_er(lcrzo_time_init(&(piv.heureajout)));
      piv.ipls=pia.hdrip.saddr;
      piv.ipld=pia.hdrip.daddr;
      piv.id=pia.hdrip.id;
      piv.protocol=pia.hdrip.protocol;
      lcrzo_er(lcrzo_list_add_first(&(pcs->seenippackets), &piv));
      /*on efface l'element de la liste*/
      lcrzo_er(lcrzo_list_remove_pos(&(pcs->waitingippackets), poselem));
      /*on quitte ici*/
      return(LCRZO_ERR_OK);
    }

    /*on n'a rien a fournir a l'appelant*/
    if (!beblocking) return(LCRZO_ERR_OKTEMPDATAEND);
    /*sinon continue la boucle infinie pour etre bloquant*/
  }
  /*jamais atteint, mais pour faire plaisir au compilateur*/
  return(LCRZO_ERR_IEINTERNALERROR);
}
/*******/
int lcrzo_priv_peutenvoyertcpattcrit(const void *pelem, const void *pinfos);
int lcrzo_priv_peutenvoyertcpattcrit(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqtcpvus *pptv1;
  const lcrzo_priv_paqtcpvus *pptv2;

  pptv1=(const lcrzo_priv_paqtcpvus *)pelem;
  pptv2=(const lcrzo_priv_paqtcpvus *)pinfos;
  if ( pptv1->ipls==pptv2->ipls &&
       pptv1->ipld==pptv2->ipld &&
       pptv1->ports==pptv2->ports &&
       pptv1->portd==pptv2->portd &&
       pptv1->octetattendu==pptv2->octetattendu ) return(1);
  return(0);
}
int lcrzo_priv_peutenvoyertcpattente(lcrzo_priv_paqtcpattente pta,
				      lcrzo_list *plptv);
int lcrzo_priv_peutenvoyertcpattente(lcrzo_priv_paqtcpattente pta,
				      lcrzo_list *plptv)
{ int retour;
  lcrzo_priv_paqtcpvus ptvtrouve, ptvcherche;
  lcrzo_int32 poselem;

  /*on se sert d'un ptv pour la recherche*/
  ptvcherche.ipls=pta.ipls;
  ptvcherche.ipld=pta.ipld;
  ptvcherche.ports=pta.ports;
  ptvcherche.portd=pta.portd;
  ptvcherche.octetattendu=pta.seqnum;
  /*on cherche*/
  retour=lcrzo_list_search_all(*plptv, &lcrzo_priv_peutenvoyertcpattcrit,
			       &ptvcherche, &poselem, &ptvtrouve);
  if (retour!=LCRZO_ERR_OK) return(0);
  /*on l'a trouve, on met a jour octetattendu*/
  ptvtrouve.octetattendu+=pta.nboctdonneestcpsynfin;
  lcrzo_er(lcrzo_time_init(&(ptvtrouve.heuredernieremaj)));
  lcrzo_er(lcrzo_list_replace_pos(plptv, poselem, &ptvtrouve));
  return(1);
}
int lcrzo_priv_purgpaqtcpattcrit(const void *pelem, const void *pinfos);
int lcrzo_priv_purgpaqtcpattcrit(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqtcpattente *ppta;
  lcrzo_uint32 secmini;

  ppta=(const lcrzo_priv_paqtcpattente *)pelem;
  secmini=*(const lcrzo_uint32 *)pinfos;
  if ( ppta->heureajout.sec < secmini ) return(1);
  return(0);
}
int lcrzo_priv_purgpaqtcpvuscrit(const void *pelem, const void *pinfos);
int lcrzo_priv_purgpaqtcpvuscrit(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqtcpvus *pptv;
  lcrzo_uint32 secmini;

  pptv=(const lcrzo_priv_paqtcpvus *)pelem;
  secmini=*(const lcrzo_uint32 *)pinfos;
  if ( pptv->heuredernieremaj.sec < secmini ) return(1);
  return(0);
}
int lcrzo_priv_purgepaqtcpvusattente(lcrzo_sniff sniff,
				     lcrzo_list *plistevus,
				     lcrzo_list *plisteattente);
int lcrzo_priv_purgepaqtcpvusattente(lcrzo_sniff sniff,
				     lcrzo_list *plistevus,
				     lcrzo_list *plisteattente)
{ static int i=0;
  lcrzo_time heure;
  lcrzo_uint32 secmini;

  i++;
  if (!(i%5)) /*pour ne pas le faire a chaque passage*/
  { lcrzo_er(lcrzo_time_init(&heure));
    /*on timeout apres xx secondes la liste d'attente*/
    secmini=heure.sec-sniff.para_tcpreortimeout2;
    /*on purge*/
    lcrzo_er(lcrzo_list_remove_criteria_all(plisteattente,
					    &lcrzo_priv_purgpaqtcpattcrit,
					    &secmini));
    /*on timeout apres xx secondes la liste des vues*/
    secmini=heure.sec-sniff.para_tcpreortimeout1;
    /*on purge*/
    lcrzo_er(lcrzo_list_remove_criteria_all(plistevus,
					    &lcrzo_priv_purgpaqtcpvuscrit,
					    &secmini));
    i-=5;
  }
  return(LCRZO_ERR_OK);
}
int lcrzo_priv_cherpaqtcpvuscrit(const void *pelem, const void *pinfos);
int lcrzo_priv_cherpaqtcpvuscrit(const void *pelem, const void *pinfos)
{ const lcrzo_priv_paqtcpvus *pptv1;
  const lcrzo_priv_paqtcpvus *pptv2;

  pptv1=(const lcrzo_priv_paqtcpvus *)pelem;
  pptv2=(const lcrzo_priv_paqtcpvus *)pinfos;
  if ( pptv1->ipls==pptv2->ipls &&
       pptv1->ipld==pptv2->ipld &&
       pptv1->ports==pptv2->ports &&
       pptv1->portd==pptv2->portd ) return(1);
  return(0);
}
int lcrzo_priv_cherpaqtcpvus(lcrzo_list *pliste,
			      lcrzo_ipl ipls,
			      lcrzo_ipl ipld,
			      lcrzo_uint16 ports,
			      lcrzo_uint16 portd,
			      lcrzo_int32 *ppos,
			      lcrzo_priv_paqtcpvus *pptv);
int lcrzo_priv_cherpaqtcpvus(lcrzo_list *pliste,
			      lcrzo_ipl ipls,
			      lcrzo_ipl ipld,
			      lcrzo_uint16 ports,
			      lcrzo_uint16 portd,
			      lcrzo_int32 *ppos,
			      lcrzo_priv_paqtcpvus *pptv)
{ lcrzo_int32 retour;
  lcrzo_priv_paqtcpvus ptv;

  ptv.ipls=ipls;
  ptv.ipld=ipld;
  ptv.ports=ports;
  ptv.portd=portd;
  retour=lcrzo_list_search_all(*pliste, &lcrzo_priv_cherpaqtcpvuscrit,
			       &ptv, ppos, pptv);
  if (retour==LCRZO_ERR_OKSEARCHNOTFOUND) {*ppos=0; retour=LCRZO_ERR_OK;}
  if (retour!=LCRZO_ERR_OK) return(retour);

  return(LCRZO_ERR_OK);
}
int lcrzo_priv_sniff_nextmreortcp(lcrzo_sniff *pcs,
				  lcrzo_bool beblocking,
				  lcrzo_data *pdata,
				  lcrzo_int32 *pdatasize);
int lcrzo_priv_sniff_nextmreortcp(lcrzo_sniff *pcs,
				  lcrzo_bool beblocking,
				  lcrzo_data *pdata,
				  lcrzo_int32 *pdatasize)
{ lcrzo_uint32 tole;
  lcrzo_int32 poselem, nbelems, i;
  lcrzo_uint16 nboctdonneestcp;
  lcrzo_priv_paqtcpattente pta;
  lcrzo_priv_paqtcpvus ptv;
  lcrzo_hdrlip hdrip;
  lcrzo_hdrltcp hdrtcp;
  int retour;
  lcrzo_data sniffdata;
  lcrzo_int32 sniffdatasize;

  /*lcrzo_er(lcrzo_priv_affdebugsniff(*pcs));*/
  
  /*regarde si un packet en attente peut etre envoye*/
  lcrzo_er(lcrzo_list_count(pcs->waitingtcppackets, &nbelems));
  for ( i=1 ; i<=nbelems ; i++ )
  { lcrzo_er(lcrzo_list_value_pos(pcs->waitingtcppackets, i, &pta));
    /*si le packet est bon a envoyer*/
    if ( lcrzo_priv_peutenvoyertcpattente(pta, &(pcs->seentcppackets)) )
    { /*prepare les donnees a sortir*/
      lcrzo_er(lcrzo_data_initm_data(pta.pdata, pta.nboctdata,
				     pdata, pdatasize));
      /*efface le pta*/
      lcrzo_er(lcrzo_list_remove_pos(&(pcs->waitingtcppackets), i));
      /*le lcrzo_priv_paqtcpvus a ete mis a jour dans peutenvoyertcpattente*/
      /*on peut donc quitter*/
      return(LCRZO_ERR_OK);
    }
  }

  /*on purge les listes*/
  lcrzo_er(lcrzo_priv_purgepaqtcpvusattente(*pcs, &(pcs->seentcppackets),
					    &(pcs->waitingtcppackets)));

  /*on sniffe maintenant le reseau*/
  while(1) /*boucle infinie dont on sort eventuellement en cours*/
  { /*sniffe et sort sur erreur*/
    lcrzo_er(lcrzo_priv_sniff_nextmreasip(pcs, beblocking,
					  &sniffdata, &sniffdatasize));

    /*si on arrive ici, on a retourne LCRZO_ERR_OK, donc on a des donnees
      quelle que soit la valeur de beblocking*/
    /*si ce n'est pas du TCP, on sort*/
    retour=lcrzo_packet_decodem_ethipopttcpoptdata(sniffdata, sniffdatasize,
						   NULL, &hdrip, NULL, NULL,
						   &hdrtcp, NULL, NULL,
						   NULL, &nboctdonneestcp);
    if (retour!=LCRZO_ERR_OK)
    { if (pdatasize!=NULL) *pdatasize=sniffdatasize;
      if (pdata!=NULL) *pdata=sniffdata;
      else lcrzo_data_free(sniffdata);
      return(LCRZO_ERR_OK);
    }
    /*si l'on arrive ici, ce packet est du tcp*/
    /*on cherche a savoir si on a deja vu ce packet*/
    lcrzo_efr(lcrzo_priv_cherpaqtcpvus(&(pcs->seentcppackets),
				       hdrip.saddr, hdrip.daddr,
				       hdrtcp.sport, hdrtcp.dport,
				       &poselem, &ptv),
	      lcrzo_data_free(sniffdata));
    if (poselem==0) /*non trouve dans la liste*/
    { /*on l'y ajoute*/
      lcrzo_er(lcrzo_time_init(&(ptv.heuredernieremaj)));
      ptv.ipls=hdrip.saddr;
      ptv.ipld=hdrip.daddr;
      ptv.ports=hdrtcp.sport;
      ptv.portd=hdrtcp.dport;
      ptv.octetattendu=hdrtcp.seqnum + nboctdonneestcp
	               + ((hdrtcp.syn|hdrtcp.fin)?1:0);
      lcrzo_er(lcrzo_list_add_first(&(pcs->seentcppackets), &ptv));
      if (pdatasize!=NULL) *pdatasize=sniffdatasize;
      if (pdata!=NULL) *pdata=sniffdata;
      else lcrzo_data_free(sniffdata);
      return(LCRZO_ERR_OK);
    }

    /*on cherche maintenant a savoir si il faut l'envoyer de suite ou non*/
    if ( ptv.octetattendu==hdrtcp.seqnum )    
    { /*on peut l'envoyer, donc on fait les mises a jour et on quitte*/
      ptv.octetattendu+=nboctdonneestcp + ((hdrtcp.syn|hdrtcp.fin)?1:0);
      lcrzo_er(lcrzo_time_init(&(ptv.heuredernieremaj)));
      lcrzo_er(lcrzo_list_replace_pos(&(pcs->seentcppackets), poselem, &ptv));
      if (pdatasize!=NULL) *pdatasize=sniffdatasize;
      if (pdata!=NULL) *pdata=sniffdata;
      else lcrzo_data_free(sniffdata);
      return(LCRZO_ERR_OK);
    }

    /*si ce packet est un vieux, ou est totalement desynchronise (on
      se donne une tolerance de 100000), on l'ignore.*/
    tole=100000;
    if ( ptv.octetattendu <= 0xFFFFFFFFu-tole )
    { if ( hdrtcp.seqnum > ptv.octetattendu+tole ||
	   hdrtcp.seqnum <= ptv.octetattendu )
      { lcrzo_data_free(sniffdata);
        continue;
      }
    }
    else
    { if ( hdrtcp.seqnum > ptv.octetattendu+tole/*deborde*/ &&
	   hdrtcp.seqnum <= ptv.octetattendu )
      { lcrzo_data_free(sniffdata);
	continue;
      }
    }

    /*ce packet est arrive trop tot, on le met dans la liste d'attente*/
    lcrzo_efr(lcrzo_time_init(&(pta.heureajout)), lcrzo_data_free(sniffdata));
    pta.ipls=hdrip.saddr;
    pta.ipld=hdrip.daddr;
    pta.ports=hdrtcp.sport;
    pta.portd=hdrtcp.dport;
    pta.seqnum=hdrtcp.seqnum;
    if ( sniffdatasize > 0xFFFF )
    { lcrzo_data_free(sniffdata);
      return(LCRZO_ERR_IEINTERNALERROR);
    }
    pta.nboctdata=(lcrzo_uint16)sniffdatasize;
    lcrzo_efr(lcrzo_data_alloc(pta.nboctdata, &(pta.pdata)),
	      lcrzo_data_free(sniffdata));
    memcpy(pta.pdata, sniffdata, pta.nboctdata);
    lcrzo_data_free(sniffdata);
    pta.nboctdonneestcpsynfin=nboctdonneestcp + ((hdrtcp.syn|hdrtcp.fin)?1:0);
    lcrzo_er(lcrzo_list_add_first(&(pcs->waitingtcppackets), &pta));
  }
  /*jamais atteint, mais pour faire plaisir au compilateur*/
  return(LCRZO_ERR_IEINTERNALERROR);
}
#endif

/*-------------------------------------------------------------*/
int lcrzo_sniff_next(lcrzo_sniff *pcs,
		     lcrzo_bool beblocking,
		     int typesniff,
		     lcrzo_int32 datamaxsize,
		     lcrzo_data data,
		     lcrzo_int32 *pdatasize)
#if LCRZODEF_LIBPCAPINSTALLED==1
{ lcrzo_data ptr;
  lcrzo_int32 ptrsize;
  int ret;

  lcrzo_er(lcrzo_sniff_nextm(pcs, beblocking, typesniff, &ptr, &ptrsize));
  ret=lcrzo_data_init_data(ptr, ptrsize, datamaxsize, data, pdatasize);
  lcrzo_data_free(ptr);
  return(ret);
}
#else
{ return (LCRZO_ERR_IEUNIMPLEMENTED);
}
#endif

/*-------------------------------------------------------------*/
int lcrzo_sniff_nextm(lcrzo_sniff *pcs,
		      lcrzo_bool beblocking,
		      int typesniff,
		      lcrzo_data *pdata,
		      lcrzo_int32 *pdatasize)
#if LCRZODEF_LIBPCAPINSTALLED==1
{
  /*parameters verification*/
  if (pcs==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_bool_verifbof(beblocking);

  switch(typesniff)
  { case LCRZO_SNIFF_TYPE_NORMAL :
      return(lcrzo_priv_sniff_nextmnormal(pcs, beblocking,
					  pdata, pdatasize));
      break;
    case LCRZO_SNIFF_TYPE_IPREAS :
      return(lcrzo_priv_sniff_nextmreasip(pcs, beblocking,
					  pdata, pdatasize));
      break;
    case LCRZO_SNIFF_TYPE_TCPREOR :
      return(lcrzo_priv_sniff_nextmreortcp(pcs, beblocking,
					   pdata, pdatasize));
      break;
  }
  /*ne doit pas arriver ici*/
  return (LCRZO_ERR_IEINTERNALERROR);
}
#else
{ return (LCRZO_ERR_IEUNIMPLEMENTED);
}
#endif

/*-------------------------------------------------------------*/
#if LCRZODEF_LIBPCAPINSTALLED==1
int lcrzo_sniff_close(lcrzo_sniff *pcs)
{ 
  /*parameters verification*/
  if (pcs==NULL) return(LCRZO_ERR_PANULLPTR);

  /*ferme les listes des packets fragmentes et TCP*/
  lcrzo_er(lcrzo_list_close(&(pcs->waitingippackets)));
  lcrzo_er(lcrzo_list_close(&(pcs->seenippackets)));
  lcrzo_er(lcrzo_list_close(&(pcs->waitingtcppackets)));
  lcrzo_er(lcrzo_list_close(&(pcs->seentcppackets)));

  /*ferme pcap*/
  pcap_close(pcs->ppcapt);
  return (LCRZO_ERR_OK);
}
#else
int lcrzo_sniff_close(lcrzo_sniff *pcs)
{ return (LCRZO_ERR_IEUNIMPLEMENTED);
}
#endif

/*-------------------------------------------------------------*/
#if LCRZODEF_LIBPCAPINSTALLED==1
int lcrzo_sniff_loop(const lcrzo_device device,
		     lcrzo_int32 nbmaxoct,
		     const char *filter, 
		     lcrzo_bool beblocking,
		     int typesniff,
		     int (*pfonc)(lcrzo_constdata tab,
				  lcrzo_int32 nboct,
				  const void *pinfos),
		     const void *pinfos)
{ lcrzo_sniff cs;
  int retour;
  lcrzo_data ptr;
  lcrzo_int32 ptrsize;

  /*parameters verification*/
  lcrzo_bool_verifbof(beblocking);
  if (pfonc==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_sniff_init(device, nbmaxoct, filter, &cs));
  retour=lcrzo_sniff_nextm(&cs, beblocking, typesniff, &ptr, &ptrsize);
  /*lorsque l'on est bloquant, on ne doit sortir qu'avec LCRZO_ERR_OK*/
  while( retour==LCRZO_ERR_OK || 
	 ((!beblocking)&&(retour==LCRZO_ERR_OKTEMPDATAEND)) )
  { retour=(*pfonc)(ptr, ptrsize, pinfos);
    lcrzo_data_free(ptr);
    if ( retour!=LCRZO_ERR_OK )
    { lcrzo_er(lcrzo_sniff_close(&cs));
      return(retour);
    }
    retour=lcrzo_sniff_nextm(&cs, beblocking, typesniff, &ptr, &ptrsize);
  }
  /*atteint en cas d'erreur*/
  lcrzo_er(lcrzo_sniff_close(&cs));
  return (retour);
}
#else
int lcrzo_sniff_loop(const lcrzo_device device,
		     lcrzo_int32 nbmaxoct,
		     const char *filter, 
		     lcrzo_bool beblocking,
		     int typesniff,
		     int (*pfonc)(lcrzo_constdata tab,
				  lcrzo_int32 nboct,
				  const void *pinfos),
		     const void *pinfos)
{ if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
  { lcrzo_string_init_text("Fonction lcrzo_sniff_loop non implementee car ",
		       LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    lcrzo_string_append_text("libpcap n'est pas installe. Vous devez installer ",
			 LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    lcrzo_string_append_text("libpcap et recompiler liblcrzo.",
			 LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
  }
  else 
  { lcrzo_string_init_text("Function lcrzo_sniff_loop unimplemented ",
		       LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    lcrzo_string_append_text("because libpcap is not installed. You should ",
			 LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    lcrzo_string_append_text("install libpcap and recompile liblcrzo.",
			 LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
  }
  return (LCRZO_ERR_IEUNIMPLEMENTED);
}
#endif


/*-------------------------------------------------------------*/
int lcrzo_sniff_set_ipreas1timeout(lcrzo_sniff *psniff,
				   lcrzo_uint16 timeout)
{ if (psniff==NULL) return(LCRZO_ERR_PANULLPTR);
  psniff->para_ipreastimeout1=timeout;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sniff_set_ipreas2timeout(lcrzo_sniff *psniff,
				   lcrzo_uint16 timeout)
{ if (psniff==NULL) return(LCRZO_ERR_PANULLPTR);
  psniff->para_ipreastimeout2=timeout;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sniff_set_tcpreor1timeout(lcrzo_sniff *psniff,
				    lcrzo_uint16 timeout)
{ if (psniff==NULL) return(LCRZO_ERR_PANULLPTR);
  psniff->para_tcpreortimeout1=timeout;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sniff_set_tcpreor2timeout(lcrzo_sniff *psniff,
				    lcrzo_uint16 timeout)
{ if (psniff==NULL) return(LCRZO_ERR_PANULLPTR);
  psniff->para_tcpreortimeout2=timeout;
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_sniff_get_ipreas1timeout(lcrzo_sniff sniff,
				   lcrzo_uint16 *ptimeout)
{ if (ptimeout!=NULL) *ptimeout=sniff.para_ipreastimeout1;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sniff_get_ipreas2timeout(lcrzo_sniff sniff,
				   lcrzo_uint16 *ptimeout)
{ if (ptimeout!=NULL) *ptimeout=sniff.para_ipreastimeout2;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sniff_get_tcpreor1timeout(lcrzo_sniff sniff,
				    lcrzo_uint16 *ptimeout)
{ if (ptimeout!=NULL) *ptimeout=sniff.para_tcpreortimeout1;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sniff_get_tcpreor2timeout(lcrzo_sniff sniff,
				    lcrzo_uint16 *ptimeout)
{ if (ptimeout!=NULL) *ptimeout=sniff.para_tcpreortimeout2;
  return(LCRZO_ERR_OK);
}

