/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

#include "lcrzo_priv.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #define _GNU_SOURCE
 #if ! defined _BSD_SOURCE
  #define _BSD_SOURCE
 #endif
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <string.h>
 #include <ctype.h>
 #include <fcntl.h>
 #include <errno.h>
 #include <termios.h>
 #include <sys/types.h>
 #include <sys/stat.h>
 #include <sys/ioctl.h>
 #include <sys/socket.h>
 #if LCRZODEF_HAVEFONC_ETHER_NTOHOST==1 || LCRZODEF_HAVEFONC_ETHER_HOSTTON==1
  #include <net/ethernet.h>
  #include <netinet/ether.h>
 #endif
 #include <net/if.h>
 #include <net/if_arp.h>
 #include <net/route.h>
 #include <netinet/in.h>
 #include <netdb.h>
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #include <stdlib.h>
 #include <stdio.h>
 #include <string.h>
 #include <unistd.h>
 #include <netdb.h>
 #include <signal.h>
 #include <fcntl.h>
 #include <errno.h>
 #include <sys/types.h>
 #include <sys/socket.h>
 #include <sys/wait.h>
 #include <sys/time.h>
 #include <sys/sysctl.h>
 #include <sys/ioctl.h>
 #include <net/ethernet.h>
 #include <net/if.h>
 #include <net/if_types.h>
 #include <net/if_arp.h>
 #include <net/if_dl.h>
 #include <net/route.h>
 #include <net/bpf.h>
 #include <netinet/in.h>
 #include <netinet/if_ether.h>
#elif defined LCRZODEF_SYSTEM_Solaris
 #define __EXTENSIONS__
 #include <stdlib.h>
 #include <stdio.h>
 #include <string.h>
 #include <unistd.h>
 #include <netdb.h>
 #include <ctype.h>
 #include <signal.h>
 #include <stropts.h>
 #include <fcntl.h> 
 #include <sys/types.h>
 #include <sys/socket.h>
 #include <sys/sockio.h>
 #include <sys/wait.h>
 #include <sys/dlpi.h>
 #include <net/if.h>
 #include <net/if_arp.h>
 #if defined LCRZODEF_SYSVER_Solaris_27
  #include <net/if_dl.h>
 #endif
 #include <net/route.h>
 #include <netinet/in.h>
 #include <netinet/if_ether.h>
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif


/*-------------------------------------------------------------*/
#ifdef LCRZODEF_SYSTEM_Linux
char * lcrzo_priv_skipfield(char *p);
char * lcrzo_priv_skipfield(char *p)
{ while (isspace( (int)(*p) )) p++;
  while (*p && !isspace( (int)(*p) )) p++;
  return(p);
}
#elif defined LCRZODEF_SYSTEM_FreeBSD
/* values stored in sysctl are padded in a 4 bytes boundary, so
   we use this define to acces items */
#define FB_ROUNDUP(a) ((a)>0?(1+(((a)-1)|(sizeof(long)-1))):sizeof(long))

int lcrzo_priv_conf_device_init_ifnumber(lcrzo_int32 ifnumber,
				         lcrzo_device device);
int lcrzo_priv_conf_device_init_ifnumber(lcrzo_int32 ifnumber,
				         lcrzo_device device)
{ int mib[6];
  size_t len;
  char *buf, *lim, *next;
  struct  if_msghdr *ifm;
  struct  sockaddr_dl *sdl;

  mib[0] = CTL_NET;         /*reseau*/
  mib[1] = PF_ROUTE;        /*famille de protocole*/
  mib[2] = 0;               /*numero de protocole*/
  mib[3] = AF_INET;         /*famille d'adresse*/
  mib[4] = NET_RT_IFLIST;   /*liste des interfaces*/
  mib[5] = 0;

  /*on fait d'abord un premier appel pour connaitre la taille a allouer*/
  if (sysctl(mib, 6, NULL, &len, NULL, 0) < 0)
  { return (LCRZO_ERR_FUSYSCTL);
  }
  
  /*on alloue la taille necessaire*/
  lcrzo_er(lcrzo_string_alloc(len, &buf));

  /*on fait le vrai appel*/
  if (sysctl(mib, 6, buf, &len, NULL, 0) < 0)
  { lcrzo_string_free(buf);
    return (LCRZO_ERR_FUSYSCTL);
  }
  lim = buf + len;

  /*on parcours les elements de la liste trouvee*/
  next = buf;
  while (next < lim)
  { ifm = (struct if_msghdr *)next;
    next += ifm->ifm_msglen;

    /* a new device */
    if (ifm->ifm_type == RTM_IFINFO)
    { if ( ifm->ifm_index == ifnumber )
      { sdl = (struct sockaddr_dl *)(ifm + 1);
        /* sdl->sdl_nlen contains the device name len, not terminated by 0 */
        if ( sdl->sdl_nlen>LCRZO_DEVICE_MAXBYTES )
          return(LCRZO_ERR_PAOVERFLOW);
        lcrzo_string_init_text(sdl->sdl_data, LCRZO_DEVICE_MAXBYTES, device);
        device[sdl->sdl_nlen]='\0';
        lcrzo_string_free(buf);
        return(LCRZO_ERR_OK);
      }
    }
  }

  lcrzo_string_free(buf);
  lcrzo_device_init("", device);
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_Solaris
/*-------------------------------------------------------------*/
int lcrzo_priv_dlpi_putmsg(int fd, char *ptr, int len, int flags);
int lcrzo_priv_dlpi_putmsg(int fd, char *ptr, int len, int flags)
{ struct strbuf ctl;
  int retour;

  ctl.maxlen = 0;
  ctl.len = len;
  ctl.buf = ptr;

  retour=putmsg(fd, &ctl, (struct strbuf *) NULL, flags);
  if ( retour < 0 )
  { return(LCRZO_ERR_FUPUTMSG);
  }
  return(LCRZO_ERR_OK);
}
/*-------------------------------------------------------------*/
int lcrzo_priv_dlpi_getmsg(int fd, int size, char *bufp);
int lcrzo_priv_dlpi_getmsg(int fd, int size, char *bufp)
{ union DL_primitives dlp;
  struct strbuf ctl;
  int flags;
  int retour;

  ctl.maxlen = 8192;
  ctl.len = 0;
  ctl.buf = bufp;

  flags = 0;
  retour=getmsg(fd, &ctl, (struct strbuf*)NULL, &flags);
  if ( retour < 0 )
  { return(LCRZO_ERR_FUGETMSG);
  }

  /* on verifie le code de retour */
  memcpy(&dlp, ctl.buf, sizeof(union DL_primitives));
  /* avant j'avais dlp = (union DL_primitives *)ctl.buf;*/
  if ( dlp.dl_primitive == DL_ERROR_ACK )
  { return(LCRZO_ERR_FUGETMSG);
  }
  /* on verifie la taille attendue */
  if ( ctl.len < size )
  { return(LCRZO_ERR_FUGETMSG);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_dlpi_open(const lcrzo_device device, int *pfd);
int lcrzo_priv_dlpi_open(const lcrzo_device device, int *pfd)
{ int retour, i;
  dl_info_req_t req;
  dl_bind_req_t	bindreq;
  struct strioctl str;
  register dl_info_ack_t *infop;
  unsigned long buf[8192];
  int ppa, fd;
  lcrzo_device devicepurge, devdevice, devdevicesanschiffre;

  /*parameters verification*/
  lcrzo_er(lcrzo_device_init(device, devicepurge));

  /*creation des devices sous forme de chaine*/
  lcrzo_string_init_text("/dev/", LCRZO_DEVICE_MAXBYTES, devdevice);
  lcrzo_string_append_text(devicepurge, LCRZO_DEVICE_MAXBYTES, devdevice);
  lcrzo_device_init(devdevice, devdevicesanschiffre);
  i=0;
  ppa=0;
  while( devdevicesanschiffre[i] != '\0' )
  { if ( isdigit((int)devdevicesanschiffre[i]) )
    { ppa=devdevicesanschiffre[i]-'0';
      devdevicesanschiffre[i]='\0';
      break;
    }
    i++;
  }

  /* on tente d'ouvrir les devices */
  fd = open(devdevicesanschiffre, O_RDWR);
  if ( fd < 0 )
  { fd = open(devdevice, O_RDWR);
    if ( fd < 0 )
    { return(LCRZO_ERR_FUOPEN);
    }
  }

  /* on attache que si l'on a un style2 */
  /* on pose la question */
  req.dl_primitive = DL_INFO_REQ;
  lcrzo_er(lcrzo_priv_dlpi_putmsg(fd, (char *)&req, sizeof(req), RS_HIPRI));
  lcrzo_er(lcrzo_priv_dlpi_getmsg(fd, DL_INFO_ACK_SIZE, (char *)buf));
  infop = &((union DL_primitives *)buf)->info_ack;
  /* c'est un style 2 */
  if ( infop->dl_provider_style == DL_STYLE2 )
  { dl_attach_req_t attachreq;
    attachreq.dl_primitive = DL_ATTACH_REQ;
    attachreq.dl_ppa       = ppa;
    lcrzo_er(lcrzo_priv_dlpi_putmsg(fd, (char *)&attachreq,
				   sizeof(attachreq), 0));
    lcrzo_er(lcrzo_priv_dlpi_getmsg(fd, DL_OK_ACK_SIZE,  (char *)buf));
  }

  /* on fait un bind */
  memset((char *)&bindreq, 0, sizeof(bindreq));
  bindreq.dl_primitive = DL_BIND_REQ;
  bindreq.dl_sap = 0;
  bindreq.dl_service_mode = DL_CLDLS;
  lcrzo_er(lcrzo_priv_dlpi_putmsg(fd, (char *)&bindreq,
				    sizeof(bindreq), 0));
  lcrzo_er(lcrzo_priv_dlpi_getmsg(fd, DL_BIND_ACK_SIZE, (char *)buf));

  /* appel d'ioctl pour dire que l'on envoie en raw*/
  str.ic_cmd    = DLIOCRAW;
  str.ic_timout = -1;
  str.ic_len    = 0;
  str.ic_dp     = NULL;
  retour = ioctl(fd, I_STR, &str);
  if ( retour < 0 )
  { return(LCRZO_ERR_FUIOCTL);
  }

  if ( pfd==NULL )
  { retour=close(fd);
    if ( retour < 0 ) return(LCRZO_ERR_FUCLOSE);
  }
  else
  { *pfd=fd;
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_sysdep_etha_init_device(const lcrzo_device device,
  	                               lcrzo_etha etha);
int lcrzo_priv_sysdep_etha_init_device(const lcrzo_device device,
  	                               lcrzo_etha etha)
{ int fd;
  dl_phys_addr_req_t pareq;
  char buf[2048];
  lcrzo_uint8 *petha;

  if (lcrzo_device_equal(device, "lo0"))
  { return(lcrzo_etha_init(0,0,0,0,0,0, etha));
  }

  lcrzo_er(lcrzo_priv_dlpi_open(device, &fd));
  pareq.dl_primitive=DL_PHYS_ADDR_REQ;
  pareq.dl_addr_type=DL_CURR_PHYS_ADDR;

  lcrzo_er(lcrzo_priv_dlpi_putmsg(fd, (char *)&pareq, sizeof(pareq), 0));
  lcrzo_er(lcrzo_priv_dlpi_getmsg(fd, DL_PHYS_ADDR_ACK_SIZE, (char*)buf));

  petha=buf+((dl_phys_addr_ack_t*)buf)->dl_addr_offset;
  memcpy(etha, petha, LCRZO_ETHA_MAXBYTES);

  close(fd);
  return(LCRZO_ERR_OK);
}
#endif

/*-------------------------------------------------------------*/
int lcrzo_sysdep_conf_init_dev(void)
#if defined LCRZODEF_SYSTEM_Linux || defined LCRZODEF_SYSTEM_Solaris
{ int fd, i, nbinterface, ret;
  struct ifconf ifc;
  struct ifreq ireqbuf[100];
  struct sockaddr_in soin;
  lcrzo_bool isup, isanalias;
  lcrzo_device device;
  struct ifreq ifr;
  lcrzo_etha etha;
  lcrzo_ipl ipl, netmask;
  lcrzo_uint32 mtu;

  /*on ouvre une socket car c'est la dessus que l'on fait un ioctl*/
  fd = socket(AF_INET, SOCK_DGRAM, 0);
  if (fd<0) return(LCRZO_ERR_FUSOCKET);

  /*le buffer va comporter toutes les interfaces*/
  ifc.ifc_len = sizeof(ireqbuf); /*taille totale*/
  ifc.ifc_ifcu.ifcu_req = ireqbuf; /*position de depart*/
  memset((char *)ireqbuf, 0, sizeof(ireqbuf));

  /*met dans ifc la config locale*/
  if ( ioctl(fd, SIOCGIFCONF, (char *)&ifc) < 0 )
  { close(fd);
    return(LCRZO_ERR_FUIOCTL);
  }

  nbinterface = ifc.ifc_len / sizeof(struct ifreq);

  /*parcours la boucle*/
  for ( i=0 ; i<nbinterface ; i++ )
  { lcrzo_efr(lcrzo_device_init(ireqbuf[i].ifr_name, device), close(fd));
    /* get the info from the device */
    memset(&ifr, 0, sizeof(ifr));
    lcrzo_string_init_text(device, sizeof(ifr.ifr_name)-1, ifr.ifr_name);
    ifr.ifr_addr.sa_family = AF_INET;
    if (ioctl(fd, SIOCGIFFLAGS, (char*)&ifr) < 0)
    { close(fd);
      return(LCRZO_ERR_FUIOCTL);
    }
    /* check if up */
    if ( ifr.ifr_flags&IFF_UP )
      isup=1;
    else
      isup=0;
    /* check if it's an alias */
    ret=lcrzo_string_search_char(device, 0, +1, -1, ':', NULL, NULL);
    if (ret==LCRZO_ERR_OKSEARCHNOTFOUND)
      isanalias=0;
    else
      isanalias=1;
    /* get the Ethernet address */
#if defined LCRZODEF_SYSTEM_Solaris
    lcrzo_er(lcrzo_priv_sysdep_etha_init_device(device, etha));
#else
    memset(&ifr, 0, sizeof(ifr));
    lcrzo_string_init_text(device, sizeof(ifr.ifr_name)-1, ifr.ifr_name);
    ifr.ifr_addr.sa_family = AF_INET;
    if (ioctl(fd, SIOCGIFHWADDR, (char*)&ifr) < 0)
    { close(fd);
      return(LCRZO_ERR_FUIOCTL);
    }
    memcpy(etha, &ifr.ifr_hwaddr.sa_data, LCRZO_ETHA_MAXBYTES);
#endif
    /* get the IP address */
    memset(&ifr, 0, sizeof(ifr));
    lcrzo_string_init_text(device, sizeof(ifr.ifr_name)-1, ifr.ifr_name);
    ifr.ifr_addr.sa_family = AF_INET;
    if (ioctl(fd, SIOCGIFADDR, (char*)&ifr) < 0)
    { close(fd);
      return(LCRZO_ERR_FUIOCTL);
    }
    memcpy(&soin, &(ifr.ifr_addr), sizeof(struct sockaddr_in));
    ipl=lcrzo_ntohl(soin.sin_addr.s_addr);
    /* get the netmask */
    memset(&ifr, 0, sizeof(ifr));
    lcrzo_string_init_text(device, sizeof(ifr.ifr_name)-1, ifr.ifr_name);
    ifr.ifr_addr.sa_family = AF_INET;
    if (ioctl(fd, SIOCGIFNETMASK, (char*)&ifr) < 0)
    { close(fd);
      return(LCRZO_ERR_FUIOCTL);
    }
    memcpy(&soin, &(ifr.ifr_addr), sizeof(struct sockaddr_in));
    netmask=lcrzo_ntohl(soin.sin_addr.s_addr);
    /* get the mtu */
    memset(&ifr, 0, sizeof(ifr));
    lcrzo_string_init_text(device, sizeof(ifr.ifr_name)-1, ifr.ifr_name);
    ifr.ifr_addr.sa_family = AF_INET;
    if (ioctl(fd, SIOCGIFMTU, (char*)&ifr) < 0)
    { close(fd);
      return(LCRZO_ERR_FUIOCTL);
    }
#if defined LCRZODEF_SYSTEM_Solaris
    mtu=ifr.ifr_metric;
#else
    mtu=ifr.ifr_mtu;
#endif
    /* save it in the configuration */
    lcrzo_efr(lcrzo_conf_devices_add(device, etha, ipl, netmask, mtu,
				     isup, isanalias),
	      close(fd));
  }
          
  ret=close(fd);
  if ( ret == -1 ) return(LCRZO_ERR_FUCLOSE);
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_FreeBSD
{ int mib[6], flags;
  size_t len;
  char *buf, *lim, *next;
  struct  if_msghdr *ifm;
  struct  sockaddr_dl *sdl;
  lcrzo_device device;
  lcrzo_etha etha;
  lcrzo_bool isaninterestingdevice, isup, isanalias;
  lcrzo_uint32 mtu=0;
  lcrzo_ipl ipl, netmask;

  mib[0] = CTL_NET;         /*reseau*/
  mib[1] = PF_ROUTE;        /*famille de protocole*/
  mib[2] = 0;               /*numero de protocole*/
  mib[3] = AF_INET;         /*famille d'adresse*/
  mib[4] = NET_RT_IFLIST;   /*liste des interfaces*/
  mib[5] = 0;

  /*on fait d'abord un premier appel pour connaitre la taille a allouer*/
  if (sysctl(mib, 6, NULL, &len, NULL, 0) < 0)
  { return (LCRZO_ERR_FUSYSCTL);
  }
  
  /*on alloue la taille necessaire*/
  lcrzo_er(lcrzo_string_alloc(len, &buf));

  /*on fait le vrai appel*/
  if (sysctl(mib, 6, buf, &len, NULL, 0) < 0)
  { lcrzo_string_free(buf);
    return (LCRZO_ERR_FUSYSCTL);
  }
  lim = buf + len;

  /*on parcours les elements de la liste trouvee*/
  isaninterestingdevice=LCRZO_FALSE;
  isup=LCRZO_FALSE;
  isanalias=LCRZO_FALSE;
  next = buf;
  while (next < lim)
  { ifm = (struct if_msghdr *)next;
    next += ifm->ifm_msglen;

    /* a new device */
    if (ifm->ifm_type == RTM_IFINFO)
    { sdl = (struct sockaddr_dl *)(ifm + 1);
      if ( sdl->sdl_type==IFT_ETHER || sdl->sdl_type==IFT_LOOP )
      { /* sdl->sdl_nlen contains the device name len, not terminated by 0 */
        if ( sdl->sdl_nlen>LCRZO_DEVICE_MAXBYTES )
	  return(LCRZO_ERR_PAOVERFLOW);
        lcrzo_string_init_text(sdl->sdl_data, LCRZO_DEVICE_MAXBYTES, device);
        device[sdl->sdl_nlen]='\0';
        /* get the Ethernet address */
        memcpy(etha, LLADDR(sdl), LCRZO_ETHA_MAXBYTES);
        /* get the mtu */
        mtu=(ifm->ifm_data).ifi_mtu;
	/* get the up flag */
        flags=ifm->ifm_flags;
        if (flags&IFF_UP) 
          isup=LCRZO_TRUE;
	else 
          isup=LCRZO_FALSE;
        /* its an interesting device */
        isaninterestingdevice=LCRZO_TRUE;
        isanalias=LCRZO_FALSE;
      }
      else
      { isaninterestingdevice=LCRZO_FALSE;
      }
    }

    /* if it's an interesting device */
    if (isaninterestingdevice)
    { /* new addresses for a device */
      if (ifm->ifm_type == RTM_NEWADDR)
      { struct ifa_msghdr *ifam;
        struct sockaddr *psa;
        struct sockaddr_in *psai;
        int ifamaddr, i;
        lcrzo_bool netmaskset, iplset;

        /* it contains ifa_msghdr followed by sockaddrs */
        ifam = (struct ifa_msghdr *)ifm;
        /* ifamaddr contains a bitwise set of sockaddrs types 
           which are in ifm */
        ifamaddr=ifam->ifam_addrs;
        /* initialize the pointer to the first sockaddr */
        psa=(struct sockaddr *)((char*)ifm + sizeof(struct ifa_msghdr));
        /* we look if one of the sockaddr contains ipl or netmask */
	netmaskset=LCRZO_FALSE;
	iplset=LCRZO_FALSE;
        netmask=0; /* just for the compiler */
        ipl=0; /* just for the compiler */
        for ( i=0 ; i<RTAX_MAX ; i++ )
        { if ( ifamaddr & (1<<i) )
          { /* we are only interrested by NETMASK and IFA */
            if ( i==RTAX_NETMASK || i==RTAX_IFA )
            { psai=(struct sockaddr_in *)psa;
              if ( i==RTAX_NETMASK )
	      { netmask=lcrzo_ntohl(psai->sin_addr.s_addr);
	        netmaskset=LCRZO_TRUE;
              }
              else if ( i==RTAX_IFA )
              { ipl=lcrzo_ntohl(psai->sin_addr.s_addr);
	        iplset=LCRZO_TRUE;
              }
            }
            /* psa points to the next item */
            psa=(struct sockaddr *)(((char *)psa) + FB_ROUNDUP(psa->sa_len));
          }
        }
        /* we add the device */
        if ( netmaskset && iplset )
        { lcrzo_conf_devices_add(device, etha, ipl, netmask, mtu,
          isup, isanalias);
	  isanalias=LCRZO_TRUE;
        }
      }
    }
  }

  lcrzo_string_free(buf);
  return(LCRZO_ERR_OK);
}
#else
 #error "Traiter le cas de LCRZODEF_SYS"
#endif

/*-------------------------------------------------------------*/
int lcrzo_sysdep_conf_init_arpcache(void)
#if defined LCRZODEF_SYSTEM_Linux
{ int ret, c, hwtype, flags;
  FILE *pf;
  char mask[21], device[21];
  lcrzo_ips ips;
  lcrzo_ipl ipl;
  lcrzo_eths eths;
  lcrzo_etha etha;
  lcrzo_device purgeddevice;
  lcrzo_bool ispermanent;

  /*ouverture du procfs*/
  pf=fopen("/proc/net/arp", "r");
  if (pf==NULL) return (LCRZO_ERR_FUFOPEN);

  /*on passe l'entete*/
  c=fgetc(pf);
  while ( c!=0x0A && c!=EOF ) c=fgetc(pf);

  /*on lit les entrees*/
  while(!feof(pf))
  { ret=fscanf(pf, "%16s 0x%4x 0x%4x %20s %20s %20s\n", ips,
	       &hwtype, &flags, eths, mask, device);
    if (ret<6 )
    { /*on n'en a pas lu assez, on ignore et passe a la ligne suivante*/
      c=fgetc(pf);
      while ( c!=0x0A && c!=EOF ) c=fgetc(pf);
    }
    else
    { if ( hwtype!=ARPHRD_ETHER ) continue;
      ispermanent=flags&ATF_PERM;      
      ret=lcrzo_device_init(device, purgeddevice);
      if ( ret!=LCRZO_ERR_OK ) continue;
      ret=lcrzo_etha_init_eths(eths, etha);
      if ( ret!=LCRZO_ERR_OK ) continue;
      ret=lcrzo_ipl_init_ips(ips, &ipl);
      if ( ret!=LCRZO_ERR_OK ) continue;
      lcrzo_efr(lcrzo_conf_arp_add(purgeddevice, etha, ipl, ispermanent), 
		fclose(pf));
    }
  }
  ret=fclose(pf);
  if ( ret == -1 ) return(LCRZO_ERR_FUFCLOSE);
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_FreeBSD
{ int mib[6];
  size_t len;
  char *buf, *next, *end;
  struct rt_msghdr *rtm;
  struct sockaddr_inarp *sin2;
  struct sockaddr_dl *sdl;
  lcrzo_device device;
  lcrzo_etha etha;
  lcrzo_ipl ipl;
 
  /*detaille dans "man sysctl" et "socket.h"*/
  mib[0] = CTL_NET;        /*reseau*/
  mib[1] = PF_ROUTE;       /*famille de protocole*/
  mib[2] = 0;              /*numero de protocole*/
  mib[3] = AF_INET;        /*famille d'adresse*/
  mib[4] = NET_RT_FLAGS;   /*type d'information*/
  mib[5] = RTF_LLINFO;     /*flags*/
 
  /*on fait d'abord un premier appel pour connaitre la taille a allouer*/ 
  if (sysctl(mib, 6, NULL, &len, NULL, 0) < 0)
  { return(LCRZO_ERR_FUSYSCTL);
  }

  /*on alloue la taille necessaire*/
  lcrzo_er(lcrzo_string_alloc(len, &buf));

  /*on fait le vrai appel*/
  if (sysctl(mib, 6, buf, &len, NULL, 0) < 0) 
  { lcrzo_string_free(buf);
    return(LCRZO_ERR_FUSYSCTL);
  }
  end = buf + len;
  
  /*on parcours les elements de la liste trouvee*/
  for (next = buf ; next < end ; next += rtm->rtm_msglen) 
  { rtm = (struct rt_msghdr *)next;
    sin2 = (struct sockaddr_inarp *)(rtm + 1);
    sdl = (struct sockaddr_dl *)(sin2 + 1);
    /* get the device */
    if ( sdl->sdl_nlen>LCRZO_DEVICE_MAXBYTES )
      return(LCRZO_ERR_PAOVERFLOW);
    lcrzo_string_init_text(sdl->sdl_data, LCRZO_DEVICE_MAXBYTES, device);
    device[sdl->sdl_nlen]='\0';
    /* get the Ethernet address */
    memcpy(etha, LLADDR(sdl), ETHER_ADDR_LEN);
    /* get the IP address */
    ipl=lcrzo_ntohl(sin2->sin_addr.s_addr);
    lcrzo_conf_arp_add(device, etha, ipl, LCRZO_FALSE); 
  }

  lcrzo_string_free(buf);
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_Solaris
{ int ret, c, nbchamps;
  FILE *pf;
  char mask[21], device[21], ligne[120];
  lcrzo_ips ipsa;
  lcrzo_ipl ipl;
  lcrzo_eths ethsa[2];
  lcrzo_etha etha;

  /*ouverture du procfs*/
  pf=popen("/bin/netstat -pn", "r");
  if (pf==NULL) return(LCRZO_ERR_FUPOPEN);

  /*on passe l'entete : 3 lignes*/
  c=fgetc(pf); while ( c!=0x0A && c!=EOF ) c=fgetc(pf);
  c=fgetc(pf); while ( c!=0x0A && c!=EOF ) c=fgetc(pf);
  c=fgetc(pf); while ( c!=0x0A && c!=EOF ) c=fgetc(pf);

  /*on lit les entrees*/
  while(!feof(pf))
  { fgets(ligne, 119, pf);
    nbchamps=sscanf(ligne, "%20s %16s %16s %20s %20s",
		    device, ipsa, mask, ethsa[0], ethsa[1]);
    if (nbchamps<4 )
    { /*on n'en a pas lu assez, on ignore et passe a la ligne suivante*/
      c=fgetc(pf);
      while ( c!=0x0A && c!=EOF ) c=fgetc(pf);
    }
    else
    { ret=lcrzo_ipl_init_ips(ipsa, &ipl);
      if ( ret!=LCRZO_ERR_OK ) continue;
      ret=lcrzo_etha_init_eths(ethsa[nbchamps-4], etha);
      if ( ret!=LCRZO_ERR_OK ) continue;
      lcrzo_efr(lcrzo_conf_arp_add(device, etha, ipl, LCRZO_FALSE), 
		pclose(pf));
    }
  }

  ret=pclose(pf);
  if ( ret == -1 ) return(LCRZO_ERR_FUPCLOSE);
  lcrzo_err_purge(); /*because Solaris set errno*/
  return (LCRZO_ERR_OK);
} 
#else
 #error "Traiter le cas de LCRZODEF_SYS"
#endif

/*-------------------------------------------------------------*/
int lcrzo_sysdep_conf_init_rarpcache(void)
#if defined LCRZODEF_SYSTEM_Linux
{ int ret, c;
  FILE *pf;
  lcrzo_ips ips;
  lcrzo_ipl ipl;
  lcrzo_etha etha;

  /*ouverture du procfs*/
  pf=fopen("/proc/net/rarp", "r");
  if (pf==NULL) 
  { /* because rarp may not exist on the system */
    lcrzo_err_purge();
    return (LCRZO_ERR_OK);
  }

  /*on passe l'entete*/
  c=fgetc(pf);
  while ( c!=0x0A && c!=EOF ) c=fgetc(pf);

  /*on lit les entrees*/
  while(!feof(pf))
  { lcrzo_eths ethsat[5];
    int nb;
    nb=fscanf(pf, "%16s %20s %20s %20s %20s %20s\n", ips,
	      ethsat[0], ethsat[1], ethsat[2], ethsat[3], ethsat[4]);
    ret=lcrzo_etha_init_eths(ethsat[nb-2], etha);
    if ( ret!=LCRZO_ERR_OK ) continue;
    ret=lcrzo_ipl_init_ips(ips, &ipl);
    if ( ret!=LCRZO_ERR_OK ) continue;
    lcrzo_efr(lcrzo_conf_arp_add("", etha, ipl, LCRZO_FALSE), 
	      fclose(pf));
  }
  ret=fclose(pf);
  if ( ret == -1 ) return(LCRZO_ERR_FUFCLOSE);
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_FreeBSD || defined LCRZODEF_SYSTEM_Solaris
{ return(LCRZO_ERR_OK);
} 
#else
 #error "Traiter le cas de LCRZODEF_SYS"
#endif

/*-------------------------------------------------------------*/
int lcrzo_sysdep_conf_init_routes(void)
#if defined LCRZODEF_SYSTEM_Linux
{ int ret, c, fieldnumber;
  FILE *pf;
  lcrzo_device device;
  lcrzo_ipl dest, gw, mask;
  lcrzo_bool isup;
  int flags, refcnt, use, metric, mtu, window, irrt;

  /*ouverture du procfs*/
  pf=fopen("/proc/net/route", "r");
  if (pf==NULL) 
  { /* because route may not exist on the system */
    lcrzo_err_purge();
    return (LCRZO_ERR_OK);
  }

  /*on passe l'entete*/
  c=fgetc(pf);
  while ( c!=0x0A && c!=EOF ) c=fgetc(pf);

  /*on lit les entrees*/
  while(!feof(pf))
  { fieldnumber=fscanf(pf, "%16s %lX %lX %X %d %d %d %lX %d %d %d\n",
		       device, &dest, &gw, &flags, &refcnt, &use, &metric,
		       &mask, &mtu, &window, &irrt);
    if (fieldnumber<8 )
    { /*on n'en a pas lu assez, on ignore et passe a la ligne suivante*/
      c=fgetc(pf);
      while ( c!=0x0A && c!=EOF ) c=fgetc(pf);
    }
    else   
    { isup = flags&RTF_UP;
      if ( ! (flags&RTF_GATEWAY) )
      { /* it is not a gateway, so we put zero */
        gw=0;
      }
      /* add it */
      lcrzo_efr(lcrzo_conf_routes_add(device,
				      lcrzo_ntohl(dest),
				      lcrzo_ntohl(mask),
				      0, lcrzo_ntohl(gw),
				      isup), 
		fclose(pf));
    }
  }
  ret=fclose(pf);
  if ( ret == -1 ) return(LCRZO_ERR_FUFCLOSE);
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_FreeBSD
{ int mib[6], flags;
  size_t len;
  char *buf, *lim, *next;
  struct rt_msghdr *rtm;
  lcrzo_device device;
  lcrzo_ipl dest, gw, mask, ipsource;
  lcrzo_bool isup, ishost, hasgateway;
  lcrzo_bool destset, gwset, maskset;
  lcrzo_int32 ifnumber;

  mib[0] = CTL_NET;         /*reseau*/
  mib[1] = PF_ROUTE;        /*famille de protocole*/
  mib[2] = 0;               /*numero de protocole*/
  mib[3] = AF_INET;         /*famille d'adresse*/
  mib[4] = NET_RT_DUMP;     /*list of routes*/
  mib[5] = 0;

  /*on fait d'abord un premier appel pour connaitre la taille a allouer*/
  if (sysctl(mib, 6, NULL, &len, NULL, 0) < 0)
  { return (LCRZO_ERR_FUSYSCTL);
  }
  
  /*on alloue la taille necessaire*/
  lcrzo_er(lcrzo_string_alloc(len, &buf));

  /*on fait le vrai appel*/
  if (sysctl(mib, 6, buf, &len, NULL, 0) < 0)
  { lcrzo_string_free(buf);
    return (LCRZO_ERR_FUSYSCTL);
  }
  lim = buf + len;

  /*on parcours les elements de la liste trouvee*/
  for ( next=buf ; next<lim ; next += rtm->rtm_msglen )
  { int rtmaddr, i;
    struct sockaddr *psa;
    struct sockaddr_in *psai;

    /* the stored item is a rtm */
    rtm = (struct rt_msghdr *)next;
    /* get flags */
    flags=rtm->rtm_flags;
    if ( flags & RTF_UP )
      isup=LCRZO_TRUE;
    else
      isup=LCRZO_FALSE;
    if ( flags & RTF_GATEWAY )
      hasgateway=LCRZO_TRUE;
    else
      hasgateway=LCRZO_FALSE;
    if ( flags & RTF_HOST )
      ishost=LCRZO_TRUE;
    else
      ishost=LCRZO_FALSE;
    /* get bitmask of stored sockaddr types */
    rtmaddr=rtm->rtm_addrs;
    /* save the number of this interface */
    ifnumber=rtm->rtm_index;

  /*lcrzo_data_print((char*)rtm, rtm->rtm_msglen, LCRZO_PRINTTYPE_DUMP);*/

    /* initialize the pointer to the first sockaddr */
    psa=(struct sockaddr *)((char*)(rtm + 1));
    /* we look each sockaddr */
    destset=LCRZO_FALSE;
    gwset=LCRZO_FALSE;
    maskset=LCRZO_FALSE;
    dest=0; /* just for the compiler */
    gw=0; /* just for the compiler */
    mask=0; /* just for the compiler */
    for ( i=0 ; i<RTAX_MAX ; i++ )
    { if ( rtmaddr & (1<<i) )
      { /* we are only interrested by DST, GATEWAY and NETMASK */
        if ( i==RTAX_DST || i==RTAX_GATEWAY || i==RTAX_NETMASK  )
        { psai=(struct sockaddr_in *)psa;
          if ( i==RTAX_DST && psai->sin_family==AF_INET )
          { if ( psai->sin_len )
            { dest=lcrzo_ntohl(psai->sin_addr.s_addr);
              destset=LCRZO_TRUE;
            }
          }
          else if ( i==RTAX_GATEWAY )
          { if ( psai->sin_len && psai->sin_family==AF_INET )
            { gw=lcrzo_ntohl(psai->sin_addr.s_addr);
              gwset=LCRZO_TRUE;
            }
          }
          else if ( i==RTAX_NETMASK )
          { if ( psai->sin_len )
            { mask=lcrzo_ntohl(psai->sin_addr.s_addr);
              maskset=LCRZO_TRUE;
            }
          }
        }
       
        /* psa points to the next item */
        psa=(struct sockaddr *)(((char *)psa) + FB_ROUNDUP(psa->sa_len));
      }
    }
    /* we add the route */
    if ( destset )
    { /* ignore route cache entries*/
      if ( !maskset && !gwset ) continue;
      /* set the mask */
      if ( !maskset )
      { if (ishost) mask=0xFFFFFFFFu;
        else mask=0;
      }
      /* set the gateway */
      if ( !gwset ) gw=0;
      /* set ipsource */
      if ( hasgateway )
      { ipsource=0;
      }
      else
      { ipsource=gw;
        gw=0;
      }
      /* add it */
      lcrzo_efr(lcrzo_priv_conf_device_init_ifnumber(ifnumber, device),
	        lcrzo_string_free(buf));
      lcrzo_efr(lcrzo_conf_routes_add(device, dest, mask, ipsource, gw, isup),
                lcrzo_string_free(buf));
    }
  }
  lcrzo_string_free(buf);
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_Solaris
{ int ret, c, nbchamps;
  FILE *pf;
  char ligne[120], dest[17], mask[17], gateway[17], device[20], mxfrg[11];
  char flags[11];
  int rtt, ref, out, infwd;
  lcrzo_ipl ipdest, ipmask, ipgateway, ipsource;
  lcrzo_bool isup;

  /*ouverture du procfs*/
  pf=popen("/bin/netstat -rvn", "r");
  if (pf==NULL) return(LCRZO_ERR_FUPOPEN);

  /*on passe l'entete */
  c=fgetc(pf); while ( c!=0x0A && c!=EOF ) c=fgetc(pf);

  /*on lit les entrees*/
  while(!feof(pf))
  { fgets(ligne, 119, pf);
    nbchamps=sscanf(ligne, "%16s %16s %16s %20s %10s %d %d %10s %d %d\n",
		    dest, mask, gateway, device, mxfrg, &rtt, &ref, flags,
		    &out, &infwd);
    if (nbchamps<8 )
    { nbchamps=sscanf(ligne, "%16s %16s %16s %10s %d %d %10s %d %d\n",
		      dest, mask, gateway, mxfrg, &rtt, &ref, flags,
		      &out, &infwd);
      device[0]=0;
      if (nbchamps<8 )
      { /*on n'en a pas lu assez, on ignore et passe a la ligne suivante*/
        c=fgetc(pf);
        while ( c!=0x0A && c!=EOF ) c=fgetc(pf);
	continue;
      }
    }
    if ( strcmp(dest, "default") )
    { ret=lcrzo_ipl_init_ips(dest, &ipdest);
      if ( ret!=LCRZO_ERR_OK ) continue;
    }
    else
    { ipdest=0;
    }
    ret=lcrzo_ipl_init_ips(mask, &ipmask);
    if ( ret!=LCRZO_ERR_OK ) continue;
    ret=lcrzo_ipl_init_ips(gateway, &ipgateway);
    if ( ret!=LCRZO_ERR_OK ) continue;
    /* check if flag up 'U' is present */
    ret=lcrzo_string_search_char(flags, LCRZO_FALSE, +1, -1, 'U', NULL, NULL);
    if ( ret==LCRZO_ERR_OK )
      isup=LCRZO_TRUE;
    else
      isup=LCRZO_FALSE;
    /* check if flag gateway 'G' is present */
    ret=lcrzo_string_search_char(flags, LCRZO_FALSE, +1, -1, 'G', NULL, NULL);
    if ( ret!=LCRZO_ERR_OK )
    { ipsource=ipgateway;
      ipgateway=0;
    }
    else
    { ipsource=0;
    }
    /* add it */    
    lcrzo_er(lcrzo_conf_routes_add(device, ipdest, ipmask, 
				   ipsource, ipgateway, isup));
  }

  ret=pclose(pf);
  if ( ret == -1 ) return(LCRZO_ERR_FUPCLOSE);
  lcrzo_err_purge(); /*because Solaris set errno*/
  return (LCRZO_ERR_OK);
}
#else
 #error "Traiter le cas de LCRZODEF_SYS"
#endif


/*-------------------------------------------------------------*/
int lcrzo_sysdep_etha_init_ipa_ioctlarp(const lcrzo_ipa ipa, lcrzo_etha etha)
/* cette fonction interroge la table ARP locale (arp -a)*/
#if defined LCRZODEF_SYSTEM_Linux || defined LCRZODEF_SYSTEM_Solaris
{ int sfd, retour, ret;
  struct arpreq	ar;
  struct sockaddr_in *soin;
  lcrzo_ipl ipl;

  memset(&ar, 0, sizeof(ar));
  ar.arp_pa.sa_family = AF_INET;
  soin = (struct sockaddr_in *)&ar.arp_pa;
  soin->sin_family = AF_INET;
  memcpy(&soin->sin_addr.s_addr, ipa, LCRZO_IPA_MAXBYTES);

  sfd=socket(AF_INET, SOCK_DGRAM, 0);
  if ( sfd == -1 ) return(LCRZO_ERR_FUSOCKET);

  retour=ioctl(sfd, SIOCGARP, (char *)&ar);
  ret=close(sfd);
  if ( retour == -1 ) return(LCRZO_ERR_FUIOCTL);
  if ( ret == -1 ) return(LCRZO_ERR_FUCLOSE);

  lcrzo_er(lcrzo_ipl_init_ipa(ipa, &ipl));
  lcrzo_er(lcrzo_conf_arp_add("", ar.arp_ha.sa_data, ipl, LCRZO_FALSE));
  if (etha!=NULL) memcpy(etha, ar.arp_ha.sa_data, LCRZO_ETHA_MAXBYTES);
  lcrzo_err_purge();
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_FreeBSD
{ return(LCRZO_ERR_OKUNRESOLVED);
}
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif

/*-------------------------------------------------------------*/
int lcrzo_sysdep_etha_init_ipa_etcethers(const lcrzo_ipa ipa, lcrzo_etha etha)
#if LCRZODEF_HAVEFONC_ETHER_HOSTTON==1
{ struct ether_addr ea;
  int retour;
  lcrzo_ips ips;
  lcrzo_ipl ipl;
  lcrzo_etha localetha; 

  lcrzo_er(lcrzo_ips_init_ipa(ipa, ips));
  retour=ether_hostton(ips, &ea);
  if ( !retour )
  { 
#ifdef LCRZODEF_SYSTEM_FreeBSD
    memcpy(localetha, ea.octet, LCRZO_ETHA_MAXBYTES);
#else
    memcpy(localetha, ea.ether_addr_octet, LCRZO_ETHA_MAXBYTES);
#endif
    lcrzo_er(lcrzo_ipl_init_ipa(ipa, &ipl));
    lcrzo_er(lcrzo_conf_arp_add("", localetha, ipl, LCRZO_FALSE));
    if (etha!=NULL) memcpy(etha, localetha, LCRZO_ETHA_MAXBYTES);
    return(LCRZO_ERR_OK);
  }
  lcrzo_err_purge();
  return(LCRZO_ERR_OKUNRESOLVED);
}
#else
{ return(LCRZO_ERR_OKUNRESOLVED);
}
#endif

/*-------------------------------------------------------------*/
/* cette fonction interroge la table RARP locale (rarp -a)*/
int lcrzo_sysdep_ipa_init_etha_ioctlrarp(const lcrzo_etha etha, lcrzo_ipa ipa)
#if defined LCRZODEF_SYSTEM_Linux
{ int sfd, retour, ret;
  struct arpreq	ar;
  lcrzo_ipl ipl;

  memset(&ar, 0, sizeof(ar));
  ar.arp_pa.sa_family = AF_INET;
  memcpy(ar.arp_ha.sa_data, etha, LCRZO_ETHA_MAXBYTES);

  sfd=socket(AF_INET, SOCK_DGRAM, 0);
  if ( sfd == -1 ) return(LCRZO_ERR_FUSOCKET);
  
  retour=ioctl(sfd, SIOCGRARP, (char *)&ar);
  ret=close(sfd);
  if ( retour == -1 ) return(LCRZO_ERR_FUIOCTL);
  if ( ret == -1 ) return(LCRZO_ERR_FUCLOSE);

  if (ipa!=NULL) memcpy(ipa, ar.arp_pa.sa_data+2, LCRZO_IPA_MAXBYTES);

  lcrzo_er(lcrzo_ipl_init_ipa(ar.arp_pa.sa_data+2, &ipl));
  lcrzo_er(lcrzo_conf_arp_add("", etha, ipl, LCRZO_FALSE));
  lcrzo_err_purge();
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_Solaris || defined LCRZODEF_SYSTEM_FreeBSD
{ return(LCRZO_ERR_IEUNIMPLEMENTED);
}
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif


/*-------------------------------------------------------------*/
int lcrzo_sysdep_ipa_init_etha_etcethers(const lcrzo_etha etha, lcrzo_ipa ipa)
#if LCRZODEF_HAVEFONC_ETHER_NTOHOST==1
{ struct ether_addr ea;
  int retour;
  lcrzo_ips ips;
  lcrzo_ipl ipl;

#ifdef LCRZODEF_SYSTEM_FreeBSD
  memcpy(ea.octet, etha, LCRZO_ETHA_MAXBYTES);
#else
  memcpy(ea.ether_addr_octet, etha, LCRZO_ETHA_MAXBYTES);
#endif
  retour=ether_ntohost(ips, &ea);
  if ( ! retour )
  { lcrzo_er(lcrzo_ipl_init_ips(ips, &ipl));
    lcrzo_er(lcrzo_conf_arp_add("", etha, ipl, LCRZO_FALSE));
    return(lcrzo_ipa_init_ips(ips, ipa));
  }
  /*sinon l'adresse n'a pas ete resolue*/
  lcrzo_err_purge();
  return(LCRZO_ERR_OKUNRESOLVED);
}
#else
{ return(LCRZO_ERR_OKUNRESOLVED);
}
#endif


/*-------------------------------------------------------------*/
#if LCRZODEF_LIBNETINSTALLED==1
#include <libnet.h>
int lcrzo_sysdep_spoof_init_eth(lcrzo_spoof *pspoof)
{ /*because we can only init when we know the device*/
  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_NONE;
  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_eth(lcrzo_spoof *pspoof,
			   const lcrzo_device device, 
			   lcrzo_constdata packet, 
			   lcrzo_int32 nboctpacket)
{ int retour;
  lcrzo_data paq;
  char err_buf[LIBNET_ERRBUF_SIZE];
  lcrzo_device devicepurge;

  /*need to purge it*/
  lcrzo_er(lcrzo_string_init_coretext(device, LCRZO_DEVICE_MAXBYTES,
				      devicepurge));
  
  /*eventually close the old device*/
  if ( pspoof->spooftype==LCRZO_PRIV_SPOOF_TYPE_ETH )
  { if ( ! lcrzo_device_equal(devicepurge, pspoof->device) )
    { lcrzo_er(lcrzo_sysdep_spoof_close_eth(pspoof));
    }
  }

  /*ouverture du link interface*/
  if ( pspoof->spooftype==LCRZO_PRIV_SPOOF_TYPE_NONE )
  { pspoof->plli=libnet_open_link_interface(devicepurge, err_buf);
    if ( pspoof->plli==NULL )
    { return(LCRZO_ERR_FULIBNETOPENLINKINT);
    }
    pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_ETH;
    lcrzo_er(lcrzo_string_init_text(devicepurge, LCRZO_DEVICE_MAXBYTES,
				    pspoof->device));
  }

  /*ecriture*/
  /*copy because car libnet n'est pas const*/
  lcrzo_er(lcrzo_data_initm_data(packet, nboctpacket, &paq, NULL));
  retour=libnet_write_link_layer(pspoof->plli, (lcrzo_uint8*)devicepurge,
				 paq, nboctpacket);
  lcrzo_data_free(paq);
  if ( retour == -1 )
  { return(LCRZO_ERR_FULIBNETWRITELL);
  }

  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_close_eth(lcrzo_spoof *pspoof)
{ int retour;
  
  /*fermeture du lli*/
  if ( pspoof->spooftype==LCRZO_PRIV_SPOOF_TYPE_ETH )
  { retour=libnet_close_link_interface(pspoof->plli);
    if ( retour == -1 )
    { return(LCRZO_ERR_FULIBNETCLOSELINKINT);
    }
    pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_NONE;
  }
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_Linux
int lcrzo_sysdep_spoof_init_eth(lcrzo_spoof *pspoof)
{ /*ouverture de la socket*/ 
  pspoof->fd=socket(AF_INET, SOCK_PACKET, lcrzo_htons(ETH_P_ALL));
  if(pspoof->fd<0) return(LCRZO_ERR_FUSOCKET);

  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_ETH;
  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_eth(lcrzo_spoof *pspoof,
			 const lcrzo_device device, 
			 lcrzo_constdata packet, 
			 lcrzo_int32 nboctpacket)
{ int retour;
  struct sockaddr sa;
 
  /*uniquement pour faire plaisir a sendto (send() ne marche pas)*/
  memset(&sa, 0, sizeof(struct sockaddr));
  lcrzo_er(lcrzo_string_init_coretext(device, 
				      sizeof(sa.sa_data)-1, sa.sa_data));

  /*envoi*/
  retour=sendto(pspoof->fd, packet, nboctpacket, 0, &sa, sizeof(sa));
  if(retour<0) return(LCRZO_ERR_FUSENDTO);

  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_close_eth(lcrzo_spoof *pspoof)
{ int retour;
  
  retour=close(pspoof->fd);
  if(retour<0) return(LCRZO_ERR_FUCLOSE);

  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_NONE;
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_FreeBSD
int lcrzo_sysdep_spoof_init_eth(lcrzo_spoof *pspoof)
{ int i;
  char nombpf[11]; /*taille de /dev/bpf*/

  /*on tente d'ouvrir tous les bpf definis*/
  for ( i=0; i<1000 ; i++)
  { lcrzo_string_init_text("/dev/bpf", 10, nombpf);
    lcrzo_string_append_int(i, "%d", 10, nombpf);
    pspoof->fd=open(nombpf, O_RDWR);
    if ( pspoof->fd==-1 )
    { if ( errno==EBUSY ) continue; /*erreur non grave*/
      return(LCRZO_ERR_FUOPEN); /*on est arrive a dernier : erreur*/
    }
    else
    { break; /*fd contient un descripteur valide*/
    }
  }

#if LCRZODEF_HAVEFONC_IOCTL_BIOCSHDRCMPLT==1
  /*on dit que bpf ne doit pas modifier les entetes ethernet*/
  { int retour;
    u_int un;
    un=1;
    retour=ioctl(pspoof->fd, BIOCSHDRCMPLT, (caddr_t)&un);
    if ( retour==-1 )
    { close(pspoof->fd);
      return(LCRZO_ERR_FUIOCTL);
    }
  }
#endif

  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_ETH;
  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_eth(lcrzo_spoof *pspoof,
			 const lcrzo_device device, 
			 lcrzo_constdata packet, 
			 lcrzo_int32 nboctpacket)
{ int retour;
  lcrzo_device devicepurge;
  struct ifreq ifr;

  /*on dit quelle interface va spoofer*/ 
  lcrzo_er(lcrzo_string_init_coretext(device, LCRZO_DEVICE_MAXBYTES,
				   devicepurge));
  lcrzo_string_init_text(devicepurge, sizeof(ifr.ifr_name)-1, ifr.ifr_name);
  retour=ioctl(pspoof->fd, BIOCSETIF, (caddr_t)&ifr);
  if ( retour==-1 )
  { return(LCRZO_ERR_FUIOCTL);
  }

  /*on ecrit*/
  lcrzo_er(lcrzo_fd_write(pspoof->fd, packet, nboctpacket));

  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_close_eth(lcrzo_spoof *pspoof)
{ int retour;
  
  retour=close(pspoof->fd);
  if(retour<0) return(LCRZO_ERR_FUCLOSE);

  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_NONE;
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_Solaris
int lcrzo_sysdep_spoof_init_eth(lcrzo_spoof *pspoof)
{ /*because we can only init when we know the device*/
  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_NONE;
  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_eth(lcrzo_spoof *pspoof,
			 const lcrzo_device device, 
			 lcrzo_constdata packet, 
			 lcrzo_int32 nboctpacket)
{ int retour;
  struct strbuf data;
  lcrzo_device devicepurge;
  lcrzo_data paq;

  /*parameters verification*/
  lcrzo_er(lcrzo_device_init(device, devicepurge));

  /*eventually close the old device*/
  if ( pspoof->spooftype==LCRZO_PRIV_SPOOF_TYPE_ETH )
  { if ( ! lcrzo_device_equal(devicepurge, pspoof->device) )
    { lcrzo_er(lcrzo_sysdep_spoof_close_eth(pspoof));
    }
  }

  /*ouverture du link interface*/
  if ( pspoof->spooftype==LCRZO_PRIV_SPOOF_TYPE_NONE )
  { lcrzo_er(lcrzo_priv_dlpi_open(devicepurge, &(pspoof->fd)));
    /*save the actual state*/
    pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_ETH;
    lcrzo_er(lcrzo_device_init(devicepurge, pspoof->device));
  }

  /* ecriture du packet */
  data.maxlen = nboctpacket;
  data.len    = nboctpacket;
  lcrzo_er(lcrzo_data_initm_data(packet, nboctpacket, &paq, NULL));
  data.buf   =(lcrzo_int8*)paq;
  retour = putmsg(pspoof->fd, NULL, &data, 0);
  lcrzo_data_free(paq);
  if ( retour < -1 )
  { return(LCRZO_ERR_FUPUTMSG);
  }

  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_close_eth(lcrzo_spoof *pspoof)
{ int retour;
  
  if ( pspoof->spooftype==LCRZO_PRIV_SPOOF_TYPE_ETH )
  { /* fermeture */
    retour=close(pspoof->fd);
    if (retour==-1) return(LCRZO_ERR_FUCLOSE);
  }
  return(LCRZO_ERR_OK);
}
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif


/*-------------------------------------------------------------*/
#if LCRZODEF_LIBNETINSTALLED==1
int lcrzo_sysdep_spoof_init_ip(lcrzo_spoof *pspoof)
{
  /*ouverture*/
  pspoof->fd=libnet_open_raw_sock(IPPROTO_RAW);
  if ( pspoof->fd == -1 )
  { return(LCRZO_ERR_FULIBNETOPENRAWSOCK);
  }
  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_IP;
  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_ip(lcrzo_spoof *pspoof,
			lcrzo_constdata packet, 
			lcrzo_int32 nboctpacket)
{ int retour;
  lcrzo_data paq;

  /*parameters verification*/
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (nboctpacket==0) return(LCRZO_ERR_OK);

  /*ecriture*/
  lcrzo_er(lcrzo_data_initm_data(packet, nboctpacket, &paq, NULL));
  retour=libnet_write_ip(pspoof->fd, paq, nboctpacket);
  lcrzo_data_free(paq);
  if ( retour == -1 )
  { return(LCRZO_ERR_FULIBNETWRITEIP);
  }
  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_close_ip(lcrzo_spoof *pspoof)
{ int retour;
  
  /*fermeture*/
  retour=libnet_close_raw_sock(pspoof->fd);
  if ( retour == -1 )
  { return(LCRZO_ERR_FULIBNETCLOSERAWSOCK);
  }

  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_NONE;
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_Solaris
int lcrzo_sysdep_spoof_init_ip(lcrzo_spoof *pspoof)
{
  /*ouverture de la socket*/ 
  pspoof->fd=socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
  if(pspoof->fd<0) return(LCRZO_ERR_FUSOCKET);

  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_IP;
  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_ip(lcrzo_spoof *pspoof,
			lcrzo_constdata packet, 
			lcrzo_int32 nboctpacket)
{ int retour;
  struct sockaddr_in soin;
  lcrzo_ipl ipl;
  lcrzo_data paq;

  /*parameters verification*/
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (nboctpacket==0) return(LCRZO_ERR_OK);

  /*uniquement pour faire plaisir a sendto (send() ne marche pas)*/
  memset(&soin, 0, sizeof(struct sockaddr_in));
  soin.sin_family = AF_INET;
  if ( nboctpacket>19 ) /*sinon essaie sans initialiser, mais ca peut ne pas
                       marcher. Selon mes tests, ca depend des machines ...*/
  { lcrzo_ipl_init(packet[16], packet[17], packet[18], packet[19], &ipl);
    soin.sin_addr.s_addr = lcrzo_htonl(ipl);
  }

  /*a partir de maintenant, on travaille sur la copie*/
  lcrzo_er(lcrzo_data_initm_data(packet, nboctpacket, &paq, NULL));

  /*on n'utilise pas lcrzo_packet_decode_... qui est trop "gros" pour
    ce que l'on veut faire*/
  if ( nboctpacket>sizeof(lcrzo_hdrpip) )
  { lcrzo_uint8 ip_protocol;
    lcrzo_uint8 ip_ihl, ip_ihl4;

    ip_ihl=(lcrzo_uint8)(0x0F&paq[0]);
    ip_ihl4=(lcrzo_uint8)(ip_ihl*4);
    if ( ip_ihl>=5 ) /*sinon packet incorrect : touche pas*/
    { ip_protocol=paq[9];
#if defined LCRZODEF_SYSVER_Solaris_24 || defined LCRZODEF_SYSVER_Solaris_25
      /*a cause d'une erreur du noyau Solaris, il faut mettre :
        hdrtcp.check=lcrzo_htons(hdrtcp.doff<<2);
        hdrudp.check=hdrudp.len;
	Si on met une valeur differente de celles calculees ci-dessus, le
	noyau Solaris envoie un checksum incorrect sur le reseau.
	Par contre, en lui envoyant la valeur ci-dessus, il calcule 
	correctement le checksum.
	C'est un bug assez ennuyeux car il nous empeche d'envoyer un
	checksum falsifie : il faut utiliser pour cela lcrzo_spoof_eth.
      */
      /*correction bug checksum tcp*/
      if ( ip_protocol==LCRZO_HDRLIP_PROTOCOL_TCP )
      { if ( nboctpacket>=(lcrzo_int32)(ip_ihl4+sizeof(lcrzo_hdrptcp)) )
	{ lcrzo_uint8 tcp_dataoff;
	  lcrzo_uint16 check;
	  tcp_dataoff=(lcrzo_uint8)((paq[ip_ihl4+12])>>4);
          /*affecte hdrtcp.check*/
          check=lcrzo_htons(tcp_dataoff<<2);
          paq[ip_ihl4+16]=(lcrzo_uint8)(check>>8);
          paq[ip_ihl4+17]=(lcrzo_uint8)(check&0xFF);
	}
      }
      /*correction bug checksum udp*/
      if ( ip_protocol==LCRZO_HDRLIP_PROTOCOL_UDP )
      { if ( nboctpacket>=(lcrzo_int32)(ip_ihl4+sizeof(lcrzo_hdrpudp)) )
	{ /*affecte hdrudp.check*/
          paq[ip_ihl4+6]=paq[ip_ihl4+4];
          paq[ip_ihl4+7]=paq[ip_ihl4+5];
	}
      }
#endif /*LCRZODEF_SYSVER_Solaris_24 ou LCRZODEF_SYSVER_Solaris_25*/
      /*changement de place des options : les options comprises dans
        le packet ne sont pas reconnues, il faut les affecter comme cela*/
      if ( ip_ihl>5 )
      { if ( nboctpacket>=ip_ihl4 ) /*toutes les opt sont presentes*/
	{ lcrzo_uint16 nboctopt;
	  nboctopt=(lcrzo_uint8)((ip_ihl-5)*4);
          /*affecte les options*/
          retour = setsockopt(pspoof->fd, IPPROTO_IP, IP_OPTIONS,
			      paq+sizeof(lcrzo_hdrpip), nboctopt);
          if (retour)
          { lcrzo_data_free(paq); 
	    return(LCRZO_ERR_FUSETSOCKOPT); 
	  }
          /*les supprime du packet a envoyer*/  
          memcpy(paq+sizeof(lcrzo_hdrpip), paq+ip_ihl4, nboctpacket-ip_ihl4);
	  nboctpacket -= nboctopt;
	  /*modifie ihl*/
	  paq[0]= (lcrzo_uint8)( (paq[0]&0xF0) | 5 );
#if defined LCRZODEF_SYSVER_Solaris_27
	  { lcrzo_uint16 ip_totlen;
	    /*modifie totlen*/
	    ip_totlen = (lcrzo_uint8)( (paq[2]<<8) | paq[3] );
	    if ( ip_totlen > nboctopt )
	    { ip_totlen = (lcrzo_uint16)( ip_totlen - nboctopt );
	      paq[2] = (lcrzo_uint8)( ip_totlen << 8 );
	      paq[3] = (lcrzo_uint8)( ip_totlen & 0xFF );
	    }
	  }
#endif /*LCRZODEF_SYSVER_Solaris_27*/
          /*solaris recalcule le checksum IP pour nous*/
	}
      }
    }
  }

  /*envoi*/
  retour=sendto(pspoof->fd, paq, nboctpacket, 0,
		(struct sockaddr*)&soin, sizeof(struct sockaddr));
  lcrzo_data_free(paq);
  if(retour<0) return(LCRZO_ERR_FUSENDTO);

  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_close_ip(lcrzo_spoof *pspoof)
{ int retour;
  
  /* fermeture */
  retour=close(pspoof->fd);
  if (retour==-1) return(LCRZO_ERR_FUCLOSE);
  return(LCRZO_ERR_OK);

  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_NONE;
  return(LCRZO_ERR_OK);
}
#elif defined LCRZODEF_SYSTEM_FreeBSD
int lcrzo_sysdep_spoof_init_ip(lcrzo_spoof *pspoof)
{
  /*ouverture de la socket*/ 
  pspoof->fd=socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
  if(pspoof->fd<0) return(LCRZO_ERR_FUSOCKET);

  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_IP;
  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_ip(lcrzo_spoof *pspoof,
			lcrzo_constdata packet, 
			lcrzo_int32 nboctpacket)
{ int retour, un;
  struct sockaddr_in soin;
  lcrzo_ipl ipl;
  lcrzo_data paq;

  /*parameters verification*/
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (nboctpacket==0) return(LCRZO_ERR_OK);


  /*uniquement pour faire plaisir a sendto (send() ne marche pas)*/
  memset(&soin, 0, sizeof(struct sockaddr_in));
  soin.sin_family = AF_INET;
  if ( nboctpacket>19 ) /*sinon essaie sans initialiser, mais ca peut ne pas
                       marcher. Selon mes tests, ca depend des machines ...*/
  { lcrzo_ipl_init(packet[16], packet[17], packet[18], packet[19], &ipl);
    soin.sin_addr.s_addr = lcrzo_htonl(ipl);
  }

  /*l'entete IP est inclus*/
  un=1;
  retour = setsockopt(pspoof->fd, IPPROTO_IP, IP_HDRINCL, 
		      (void*)&un, sizeof(un));
  if (retour) return(LCRZO_ERR_FUSETSOCKOPT);

  /*autorise a envoyer des packets broadcast*/
  un=1;
  retour = setsockopt(pspoof->fd, SOL_SOCKET, SO_BROADCAST,
		      (void*)&un, sizeof(un));
  if (retour) return(LCRZO_ERR_FUSETSOCKOPT);

  /*a partir de maintenant, on travaille sur la copie*/
  lcrzo_er(lcrzo_data_initm_data(packet, nboctpacket, &paq, NULL));
  /*a cause d'un bug BSD, on est oblige d'inverser la taille et l'offset*/
  if (nboctpacket>=4)
  { paq[2]=packet[3];
    paq[3]=packet[2];
  }
  if (nboctpacket>=8)
  { paq[6]=packet[7];
    paq[7]=packet[6];
  }

  /*envoi*/
  retour=sendto(pspoof->fd, paq, nboctpacket, 0,
		(struct sockaddr*)&soin, sizeof(struct sockaddr));
  lcrzo_data_free(paq);
  if(retour<0) return(LCRZO_ERR_FUSENDTO);

  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_close_ip(lcrzo_spoof *pspoof)
{ int retour;
  
  /* fermeture */
  retour=close(pspoof->fd);
  if (retour==-1) return(LCRZO_ERR_FUCLOSE);
  return(LCRZO_ERR_OK);

  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_NONE;
  return(LCRZO_ERR_OK);
}
#else
int lcrzo_sysdep_spoof_init_ip(lcrzo_spoof *pspoof)
{ int i;

  /*ouverture de la socket*/ 
  pspoof->fd=socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
  i=0;
  while( pspoof->fd<0 && i<500 )
  { if (errno!=ENOBUFS) return(LCRZO_ERR_FUSOCKET);
    /*sinon, attend 500ms. En effet, sur des machines puissantes, le
      programme sature le systeme*/
    i++;
    lcrzo_er(lcrzo_time_sleep(1000));
    pspoof->fd=socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
  }
  /*on est toujours sature*/
  if(pspoof->fd<0) return(LCRZO_ERR_FUSOCKET);

  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_IP;
  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_ip(lcrzo_spoof *pspoof,
			lcrzo_constdata packet, 
			lcrzo_int32 nboctpacket)
{ int retour, un;
  struct sockaddr_in soin;
  lcrzo_ipl ipl;

  /*parameters verification*/
  if (packet==NULL && nboctpacket) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (nboctpacket==0) return(LCRZO_ERR_OK);

  /*uniquement pour faire plaisir a sendto (send() ne marche pas)*/
  memset(&soin, 0, sizeof(struct sockaddr_in));
  soin.sin_family = AF_INET;
  if ( nboctpacket>19 ) /*sinon essaie sans initialiser, mais ca peut ne pas
                       marcher. Selon mes tests, ca depend des machines ...*/
  { lcrzo_ipl_init(packet[16], packet[17], packet[18], packet[19], &ipl);
    soin.sin_addr.s_addr = lcrzo_htonl(ipl);
  }

  /*l'entete IP est inclus*/
  un=1;
  retour = setsockopt(pspoof->fd, IPPROTO_IP, IP_HDRINCL,
		      (void*)&un, sizeof(un));
  if (retour) return(LCRZO_ERR_FUSETSOCKOPT);

  /*autorise a envoyer des packets broadcast*/
  un=1;
  retour = setsockopt(pspoof->fd, SOL_SOCKET, SO_BROADCAST,
		      (void*)&un, sizeof(un));
  if (retour) return(LCRZO_ERR_FUSETSOCKOPT);

  /*envoi*/
  retour=sendto(pspoof->fd, packet, nboctpacket, 0,
		(struct sockaddr*)&soin, sizeof(struct sockaddr));
  if(retour<0) return(LCRZO_ERR_FUSENDTO);

  return(LCRZO_ERR_OK);
}
int lcrzo_sysdep_spoof_close_ip(lcrzo_spoof *pspoof)
{ int retour;
  
  /* fermeture */
  retour=close(pspoof->fd);
  if (retour==-1) return(LCRZO_ERR_FUCLOSE);
  return(LCRZO_ERR_OK);

  pspoof->spooftype=LCRZO_PRIV_SPOOF_TYPE_NONE;
  return(LCRZO_ERR_OK);
}
#endif

/*-------------------------------------------------------------*/
int lcrzo_sysdep_print_cpuall(void)
#ifdef LCRZODEF_SYSTEM_Linux
{ static unsigned long cpu_user=0, cpu_system=0, cpu_nice=0, cpu_idle=0;
  unsigned long cpu_user_, cpu_system_, cpu_nice_, cpu_idle_;
  float total;
  char buf[100];
  int fd, nboct, retour, afficher;
  char *p;

  if ( !cpu_user && !cpu_system && !cpu_nice && !cpu_idle )
  { printf("CPUall: Le premier appel est toujours peu significatif.\n");
    afficher=0;
  }
  else
  { afficher=1;
  }

  /*on stocke dans buf la premiere ligne*/
  fd=open("/proc/stat", O_RDONLY);
  if (fd<0) return(LCRZO_ERR_FUOPEN);
  nboct=read(fd, buf, sizeof(buf)-1);
  if (nboct<0) return(LCRZO_ERR_FUREAD);
  buf[nboct] = '\0';
  retour=close(fd);
  if (retour<0) return(LCRZO_ERR_FUCLOSE);

  /*la premiere ligne contient normalement :
    cpu  60754 425 35548 720580
  */
  p = lcrzo_priv_skipfield(buf);
  cpu_user_   = strtoul(p, &p, 0) - cpu_user;
  cpu_system_ = strtoul(p, &p, 0) - cpu_system;
  cpu_nice_   = strtoul(p, &p, 0) - cpu_nice;
  cpu_idle_   = strtoul(p, &p, 0) - cpu_idle;  
  total=(float)0.01*(cpu_user_+cpu_system_+cpu_nice_+cpu_idle_);
  if (total==0.0)
  { if (afficher)
    { printf("CPUall: user ??.???%%, system ??.???%%,");
      printf("nice ??.???%%, idle ??.???%%\n");
    }
  }
  else
  { if (afficher)
    { printf("CPUall: user %3.3f%%, system %3.3f%%, ",
	     (float)cpu_user_/total, (float)cpu_system_/total);
      printf("nice %3.3f%%, idle %3.3f%%\n",
	     (float)cpu_nice_/total, (float)cpu_idle_/total);
    }
    cpu_user+=cpu_user_;
    cpu_system+=cpu_system_;
    cpu_nice+=cpu_nice_;
    cpu_idle+=cpu_idle_;
  }
  return(LCRZO_ERR_OK);
}
#else
{ return (LCRZO_ERR_IEUNIMPLEMENTED);
}
#endif

/*-------------------------------------------------------------*/
int lcrzo_sysdep_print_memall(void)
#ifdef LCRZODEF_SYSTEM_Linux
{ unsigned long mem_total, mem_used, mem_free;
  char buf[200];
  int fd, nboct, retour;
  char *p;

  /*on stocke dans buf la premiere ligne*/
  fd=open("/proc/meminfo", O_RDONLY);
  if (fd<0) return(LCRZO_ERR_FUOPEN);
  nboct=read(fd, buf, sizeof(buf)-1);
  if (nboct<0) return(LCRZO_ERR_FUREAD);
  buf[nboct] = '\0';
  retour=close(fd);
  if (retour<0) return(LCRZO_ERR_FUCLOSE);

  /*les premieres lignes contiennent normalement :
           total:    used:    free:  shared: buffers:  cached:
    Mem:  64688128 63479808  1208320 16080896 31932416 15171584
  */
  p = (char*)strchr(buf, '\n');
  p = lcrzo_priv_skipfield(p);
  mem_total=strtoul(p, &p, 0);
  mem_used=strtoul(p, &p, 0);
  mem_free=strtoul(p, &p, 0);
  printf("MEMall: total %10ld, utilise %10ld, libre %10ld\n", mem_total,
	 mem_used, mem_free);

  return(LCRZO_ERR_OK);
}
#else
{ return (LCRZO_ERR_IEUNIMPLEMENTED);
}
#endif

/*-------------------------------------------------------------*/
int lcrzo_sysdep_print_cpupid(int pid)
#ifdef LCRZODEF_SYSTEM_Linux
{ static unsigned long cpu_user=0, cpu_system=0, cpu_nice=0, cpu_idle=0;
  static unsigned long cpu_utime=0, cpu_stime=0;
  unsigned long cpu_user_, cpu_system_, cpu_nice_, cpu_idle_;
  unsigned long cpu_utime_, cpu_stime_;
  int fd, nboct, retour, afficher, i;
  char buf[400];
  char *p;
  float total;

  /*on lit le cpu utilise par le processus*/
  lcrzo_er(lcrzo_string_init_text("/proc/", 399, buf));
  lcrzo_er(lcrzo_string_append_int(pid, "%d", 399, buf));
  lcrzo_er(lcrzo_string_append_text("/stat", 399, buf));

  /*on stocke dans buf la premiere ligne*/
  fd=open(buf, O_RDONLY);
  if (fd<0) return(LCRZO_ERR_FUOPEN);
  nboct=read(fd, buf, sizeof(buf)-1);
  if (nboct<0) return(LCRZO_ERR_FUREAD);
  buf[nboct] = '\0';
  retour=close(fd);
  if (retour<0) return(LCRZO_ERR_FUCLOSE);

  /*analyse de la premiere ligne :
    1948 (lcrzoex) R 685 1948 685 768 1948 1048832 126 0 117 0
    805 42
    ...*/
  p = lcrzo_priv_skipfield(buf);
  for ( i=0 ; i<12 ; i++ )
  { p = lcrzo_priv_skipfield(p);
  }
  cpu_utime_=strtoul(p, &p, 0)-cpu_utime;
  cpu_stime_=strtoul(p, &p, 0)-cpu_stime;
  cpu_utime+=cpu_utime_;
  cpu_stime+=cpu_stime_;

  /*on lit le cpu total utilise*/
  if ( !cpu_user && !cpu_system && !cpu_nice && !cpu_idle )
  { printf("CPU%d: Le premier appel est toujours peu significatif.\n", pid);
    afficher=0;
  }
  else
  { afficher=1;
  }

  /*on stocke dans buf la premiere ligne*/
  fd=open("/proc/stat", O_RDONLY);
  if (fd<0) return(LCRZO_ERR_FUOPEN);
  nboct=read(fd, buf, sizeof(buf)-1);
  if (nboct<0) return(LCRZO_ERR_FUREAD);
  buf[nboct] = '\0';
  retour=close(fd);
  if (retour<0) return(LCRZO_ERR_FUCLOSE);

  /*la premiere ligne contient normalement :
    cpu  60754 425 35548 720580
  */
  p = lcrzo_priv_skipfield(buf);
  cpu_user_   = strtoul(p, &p, 0) - cpu_user;
  cpu_system_ = strtoul(p, &p, 0) - cpu_system;
  cpu_nice_   = strtoul(p, &p, 0) - cpu_nice;
  cpu_idle_   = strtoul(p, &p, 0) - cpu_idle;  
  total=(float)0.01*(cpu_user_+cpu_system_+cpu_nice_+cpu_idle_);
  if (total==0.0)
  { if (afficher)
      printf("CPU%d: utilisateur ??.???%%, noyau ??.???%%\n", pid);
  }
  else
  { if (afficher)
      printf("CPU%d: utilisateur %3.3f%%, noyau %3.3f%%\n", pid,
	     (float)cpu_utime_/total, (float)cpu_stime_/total);
    cpu_user+=cpu_user_;
    cpu_system+=cpu_system_;
    cpu_nice+=cpu_nice_;
    cpu_idle+=cpu_idle_;
  }
  return(LCRZO_ERR_OK);
}
#else
{ return (LCRZO_ERR_IEUNIMPLEMENTED);
}
#endif

/*-------------------------------------------------------------*/
int lcrzo_sysdep_print_mempid(int pid)
#ifdef LCRZODEF_SYSTEM_Linux
{ unsigned long mem_total, mem;
  char buf[200];
  int fd, nboct, retour;
  char *p;

  /*on stocke dans buf la premiere ligne*/
  fd=open("/proc/meminfo", O_RDONLY);
  if (fd<0) return(LCRZO_ERR_FUOPEN);
  nboct=read(fd, buf, sizeof(buf)-1);
  if (nboct<0) return(LCRZO_ERR_FUREAD);
  buf[nboct] = '\0';
  retour=close(fd);
  if (retour<0) return(LCRZO_ERR_FUCLOSE);

  /*les premieres lignes contiennent normalement :
           total:    used:    free:  shared: buffers:  cached:
    Mem:  64688128 63479808  1208320 16080896 31932416 15171584
  */
  p = (char*)strchr(buf, '\n');
  p = lcrzo_priv_skipfield(p);
  mem_total=strtoul(p, &p, 0);

  /*on recupere le nb de pages occupees par le processus*/
  lcrzo_er(lcrzo_string_init_text("/proc/", 399, buf));
  lcrzo_er(lcrzo_string_append_int(pid, "%d", 399, buf));
  lcrzo_er(lcrzo_string_append_text("/statm", 399, buf));

  /*on stocke dans buf la premiere ligne*/
  fd=open(buf, O_RDONLY);
  if (fd<0) return(LCRZO_ERR_FUOPEN);
  nboct=read(fd, buf, sizeof(buf)-1);
  if (nboct<0) return(LCRZO_ERR_FUREAD);
  buf[nboct] = '\0';
  retour=close(fd);
  if (retour<0) return(LCRZO_ERR_FUCLOSE);

  /*analyse de la premiere ligne :
    106 106 87 21 0 85 19
  */
  p = lcrzo_priv_skipfield(buf);
  mem=strtoul(p, &p, 0)*4096;

  /*on affiche*/
  printf("MEM%d: %ld octets, %3.3f%%\n", pid, mem,
	 100.0*(float)mem/(float)mem_total); 
 
  return(LCRZO_ERR_OK);
}
#else
{ return (LCRZO_ERR_IEUNIMPLEMENTED);
}
#endif

/*-------------------------------------------------------------*/
int lcrzo_sysdep_print_mempidcurrent(void)
{ return(lcrzo_sysdep_print_mempid(getpid()));
}

/*-------------------------------------------------------------*/
int lcrzo_sysdep_print_cpupidcurrent(void)
{ return(lcrzo_sysdep_print_cpupid(getpid()));
}

/*-------------------------------------------------------------*/
int lcrzo_sysdep_print_memcpuall(void)
{ lcrzo_er(lcrzo_sysdep_print_memall());
  return(lcrzo_sysdep_print_cpuall());
}

/*-------------------------------------------------------------*/
int lcrzo_sysdep_print_memcpupid(int pid)
{ lcrzo_er(lcrzo_sysdep_print_mempid(pid));
  return(lcrzo_sysdep_print_cpupid(pid));
}

/*-------------------------------------------------------------*/
int lcrzo_sysdep_print_memcpupidcurrent(void)
{ int pid;
  pid=getpid();
  lcrzo_er(lcrzo_sysdep_print_mempid(pid));
  return(lcrzo_sysdep_print_cpupid(pid));
}

/*-------------------------------------------------------------*/
int lcrzo_sysdep_print_memcpuallpid(int pid)
{ lcrzo_er(lcrzo_sysdep_print_memall());
  lcrzo_er(lcrzo_sysdep_print_cpuall());
  lcrzo_er(lcrzo_sysdep_print_mempid(pid));
  return(lcrzo_sysdep_print_cpupid(pid));
}

/*-------------------------------------------------------------*/
int lcrzo_sysdep_print_memcpuallpidcurrent(void)
{ int pid;
  pid=getpid();
  lcrzo_er(lcrzo_sysdep_print_memall());
  lcrzo_er(lcrzo_sysdep_print_cpuall());
  lcrzo_er(lcrzo_sysdep_print_mempid(pid));
  return(lcrzo_sysdep_print_cpupid(pid));
}
