/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

/* -- Note about comments and variable names --
   In versions 1.00 up to 2.08, lcrzo was only written in French (my
   mother tongue). Since version 3.00, lcrzo is written in English, but
   most of comments and variable names are still in French.
   In future versions, I'll try to translate everything into English,
   but as you may guess, that's a quite big work...
   So, for the moment, here are some word translation, if you want to
   be able to understand what's going on in the functions :
     afficher - print     ;  ajouter - add         ;  copie - copy
     dernier - last       ;  donnee - data         ;  ecrire - write
     entete - header      ;  envoye - sent         ;  ferme - close
     fichier - file       ;  lire - read           ;  masque - mask 
     nboct - byte number  ;  ouvre, ouvrir - open  ;  paq, paquet - packet
     plage - range        ;  premier - first       ;  recu - received
     reseau - network     ;  retour - return       ;  taille - size
     tunnel - pipe

*/

#include "lcrzo.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #define _GNU_SOURCE
 #define _BSD_SOURCE
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <errno.h>
 #include <sys/resource.h>
 #include <sys/time.h>
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <errno.h>
 #include <sys/time.h>
 #include <sys/resource.h>
#elif defined LCRZODEF_SYSTEM_Solaris
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <errno.h>
 #include <sys/resource.h>
 #include <sys/time.h>
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif

/*---------------------------------------------------------------*/
int lcrzo_time_fprint(LCRZOFILE *pf,
		     const char *deb,
		     lcrzo_time heure,
                     const char *fin)
{ 
  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  if (deb!=NULL) fprintf(pf, "%s", deb);
  fprintf(pf, "%lus:%06luus", heure.sec, heure.usec);
  if (fin!=NULL) fprintf(pf, "%s", fin);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_time_init(lcrzo_time *pheure)
{ struct timeval tv;
  int retour;

  /*parameters verification*/
  if (pheure==NULL) return(LCRZO_ERR_PANULLPTR);

  retour=gettimeofday(&tv, NULL);
  if (retour)
  { return(LCRZO_ERR_FUGETTIMEOFDAY);
  }
  /*on stocke les valeurs*/
  pheure->sec=(lcrzo_uint32)tv.tv_sec;
  pheure->usec=(lcrzo_uint32)tv.tv_usec;
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
/*cette fonction n'utilise pas lcrzo_time_ecrit et lcrzo_time_init
  afin d'etre plus rapide*/
int lcrzo_time_initfprint(LCRZOFILE *pf,
			   const char *deb,
			   const char *fin,
			   lcrzo_time *pheure)
{ struct timeval tv;
  int retour;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  retour=gettimeofday(&tv, NULL);
  if (retour)
  { return(LCRZO_ERR_FUGETTIMEOFDAY);
  }
  /*on affiche*/
  if (deb!=NULL) fprintf(pf, "%s", deb);
  fprintf(pf, "%lus:%06luus", (lcrzo_uint32)tv.tv_sec, 
	  (lcrzo_uint32)tv.tv_usec);
  if (fin!=NULL) fprintf(pf, "%s", fin);

  /*on stocke les valeurs*/
  if (pheure!=NULL)
  { pheure->sec=(lcrzo_uint32)tv.tv_sec;
    pheure->usec=(lcrzo_uint32)tv.tv_usec;
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_time_diff(lcrzo_time heure,
		    lcrzo_uint32 *psec,
		    lcrzo_uint32 *pusec)
{ struct timeval tv;
  int retour;
  lcrzo_uint32 sec, usec;

  /*on recupere l'heure courante*/
  retour=gettimeofday(&tv, NULL);
  if (retour)
  { if (psec!=NULL) *psec=0;
    if (pusec!=NULL) *pusec=0;
    return(LCRZO_ERR_FUGETTIMEOFDAY);
  }

  /*si l'heure courante est inferieure a l'heure sauvee, probleme*/
  if ( tv.tv_sec < heure.sec ) return(LCRZO_ERR_IETIMEHIGHER);
  if ( tv.tv_sec == heure.sec &&  tv.tv_usec < heure.usec ) 
    return(LCRZO_ERR_IETIMEHIGHER);

  /*calcule le resultat*/  
  if ( tv.tv_usec > heure.usec )
  { sec=tv.tv_sec-heure.sec;
    usec=tv.tv_usec-heure.usec;
  }
  else
  { sec=tv.tv_sec-heure.sec-1;
    usec=tv.tv_usec-heure.usec+1000000;
  }
  if (psec!=NULL) *psec=sec;
  if (pusec!=NULL) *pusec=1000000*sec+usec;

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
/*cette fonction n'utilise pas lcrzo_time_ecrit et lcrzo_time_init
  afin d'etre plus rapide*/
int lcrzo_time_difffprint(LCRZOFILE *pf,
			   const char *deb,
			   lcrzo_time heure,
			   const char *fin,
			   lcrzo_uint32 *psec,
			   lcrzo_uint32 *pusec)
{ struct timeval tv;
  int retour;
  lcrzo_uint32 sec, usec;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  /*on recupere l'heure courante*/
  retour=gettimeofday(&tv, NULL);
  if (retour)
  { if (psec!=NULL) *psec=0;
    if (pusec!=NULL) *pusec=0;
    return(LCRZO_ERR_FUGETTIMEOFDAY);
  }

  /*si l'heure courante est inferieure a l'heure sauvee, probleme*/
  if ( tv.tv_sec < heure.sec ) return(LCRZO_ERR_IETIMEHIGHER);
  if ( tv.tv_sec == heure.sec &&  tv.tv_usec < heure.usec ) 
    return(LCRZO_ERR_IETIMEHIGHER);

  /*calcule le resultat*/
  if ( tv.tv_usec > heure.usec )
  { sec=tv.tv_sec-heure.sec;
    usec=tv.tv_usec-heure.usec;
  }
  else
  { sec=tv.tv_sec-heure.sec-1;
    usec=tv.tv_usec-heure.usec+1000000;
  }

  /*on affiche*/
  if (deb!=NULL) fprintf(pf, "%s", deb);
  if (sec<1000)
    fprintf(pf, "%luus", 1000000*sec+usec);
  else
    fprintf(pf, "%lus", sec);
  if (fin!=NULL) fprintf(pf, "%s", fin);

  /*on stocke les valeurs*/
  if (psec!=NULL) *psec=sec;
  if (pusec!=NULL) *pusec=1000000*sec+usec;

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_time_sleep(lcrzo_uint32 usec)
{ struct timeval tv1, tv2;
  int retour, oldprio;

  if (usec<lcrzo_global.time_sleep_threshold1)
  { retour=gettimeofday(&tv1, NULL);
    if (retour) return(LCRZO_ERR_FUGETTIMEOFDAY);
    tv2=tv1;
    /*on boucle en attendant l'heure*/
    while( (tv2.tv_sec-tv1.tv_sec)*1000000+(tv2.tv_usec-tv1.tv_usec) < usec )
    { retour=gettimeofday(&tv2, NULL);
      if (retour) return(LCRZO_ERR_FUGETTIMEOFDAY);
    }
  }
  else if (usec<lcrzo_global.time_sleep_threshold2)
  { retour=gettimeofday(&tv1, NULL);
    if (retour) return(LCRZO_ERR_FUGETTIMEOFDAY);
    tv2=tv1;
    /*pour consommer moins de ressources*/
    oldprio=getpriority(PRIO_PROCESS, 0);
    setpriority(PRIO_PROCESS, 0, +20);
    /*on boucle en attendant l'heure*/
    while( (tv2.tv_sec-tv1.tv_sec)*1000000+(tv2.tv_usec-tv1.tv_usec) < usec )
    { retour=gettimeofday(&tv2, NULL);
      if (retour) return(LCRZO_ERR_FUGETTIMEOFDAY);
    }
    setpriority(PRIO_PROCESS, 0, oldprio);
  }
  else if (usec<lcrzo_global.time_sleep_threshold3)
  { /*utilise usleep*/
    usleep(usec);
  }
  else
  { /*utilise sleep*/
    sleep(usec/1000000);
  }

  errno=0; /*because Solaris sets errno to 4*/
  return(LCRZO_ERR_OK);
}
