/***************************************************************************
 *   Copyright (C) 2005 Meni Livne <livne@kde.org>                         *
 *   Copyright (C) 2005 Boaz Anin <boazanin@gmail.com>                     *
 *   Copyright (C) 2005 Shahar Karin <shaharka@post.tau.ac.il>             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef __PHISH_H
#define __PHISH_H


#ifdef __cplusplus
extern "C" {
#endif


#define PHISH_LIB_NAME "libphish"
#define PHISH_LIB_VERSION "0.1.0"


#include <sys/types.h>

/** Operation was successful */
#define PHISH_SUCCESS 0
/** Library hasn't been initialised */
#define PHISH_ERR_NOT_INITIALISED 1
/** Memory allocation error */
#define PHISH_ERR_MEMORY 2
/** Error in settings */
#define PHISH_ERR_SETTINGS 3

/** A malformed URL was given as parameter */
#define PHISH_ERR_MALFORMED_URL 11

/** Error initiating TCP connection */
#define PHISH_ERR_TCP_CONNECT 21
/** Error reading from socket */
#define PHISH_ERR_SOCK_READ 22
/** Error writing to socket */
#define PHISH_ERR_SOCK_WRITE 23
/** Error resolving a hostname */
#define PHISH_ERR_RESOLVE 24
/** An unexpected HTTP header format was encountered when reading an HTTP
 *  reply from a server */
#define PHISH_ERR_HTTP_BAD_HEADER 25
/** Unexpected HTTP status code received in reply from HTTP server */
#define PHISH_ERR_HTTP_BAD_STATUS 26

/** Error opening or reading from file */
#define PHISH_ERR_FILE 31
/** Error writing to file */
#define PHISH_ERR_FILE_WRITE 32

/** Error reading from XML file */
#define PHISH_XML_READ_ERROR 41
/** Error parsing XML file */
#define PHISH_XML_PARSE_ERROR 42

/** Unexpected reply from OPDB server */
#define PHISH_ERR_OPDB_BAD_REPLY 51

/** Sites XML file on server wasn't modified */
#define PHISH_XML_NOT_MODIFIED 61


#define PHISH_URL_LENGTH_LIMIT 4192


/** All library functions return this type, which indicates whether the
 *  function succeeded or the type of error that occurred.
 */
typedef int phish_result_t;

/** Possible running modes for checking URLs. */
typedef enum { PHISH_ONLINE_MODE, PHISH_OFFLINE_MODE } phish_mode_t;

/** Risk level indication. */
typedef enum { PHISH_RISK_UNKNOWN, PHISH_RISK_NONE,
               PHISH_RISK_LOW, PHISH_RISK_MEDIUM,
               PHISH_RISK_HIGH } phish_risk_t;

/** Results structure returned by the URL checking functions with data about
 *  the URL given.
 */
typedef struct
{
  /** Risk level of URL */
  phish_risk_t risk_level;

  /** Whether the server part of the URL was found to be in the database */
  int server;

  /** Whether the IP address of the server was found to be in the database */
  int ip;

  /** Whether the path part of the URL was found to be in the database */
  int path;

  /** Whether the domain of the URL was found to be in the database */
  int domain;

  /** Two level country code of country where site is probably located, or "--"
   *  if the country is unknown
   */
  char country[3];

  /** Length of comments about URL, 0 if no comments */
  size_t comments_length;

  /** Comments about URL, NULL if no comments */
  char *comments;

  /** Whether URL scheme contains a username, i.e. protocol://user@host... */
  int user_scheme;

  /** Whether the server part of the URL contains characters which are not
   *  normally part of a server name (e.g. non-latin characters in
   *  international domain names)
   */
  int suspicious_host;
  
} phish_url_data_t;


/** The safe list */
typedef struct phish_safelist_t phish_safe_list_t;

/** An entry in the safe list */
typedef struct phish_safelist_entry_t phish_safe_list_entry_t;


/** Initialise the library. This function must be called before any other
 *  library function can be called. Functions will return
 *  PHISH_ERR_NOT_INITIALISED if called without having the library initialised.
 *  @param user_agent user agent string of client (browser) used to identify
 *                    itself
 *  @param client_version version string of client (e.g. anti-phishing
 *                        toolbar version
 */
phish_result_t phish_init(const char *user_agent,
                          const char *client_version);

/** Saves settings and shuts the library down. This function should be called
 *  when you're finished using the library.
 */
phish_result_t phish_shutdown();

/** Checks a URL. The checking resource (i.e. server or local XML file) is
 *  determined according to the settings.
 *  @param url complete URL to check
 *  @param results pointer to results structure which will be filled with
 *                 results of checking the URL
 */
phish_result_t phish_checkURL(const char *url, phish_url_data_t *results);

/** Checks only the country of a site. Useful for checking a site's country
 *  in offline mode, where the other URL data is obtained from the local
 *  XML file which was previsouly downloaded.
 *  @param url complete URL to check
 *  @param results pointer to results structure whose country field will be
 *                 filled with the country of the given URL
 */
phish_result_t phish_checkCountry(const char *url, phish_url_data_t *results);

/** Deallocates memory associated with URL data structure.
 *  @param url_data pointer to URL data structure to free.
 */
phish_result_t phish_deleteURLData(phish_url_data_t *url_data);

/** Checks if a URL is in the safe list.
 *  @param url complete URL to check
 *  @param reply pointer to integer which will be assigned 1 if the URL is in
 *               the safe list, 0 otherwise
 */
phish_result_t phish_checkSafeList(const char *url, int *reply);

/** Download the sites database as an XML file from the server. If the
 *  database hasn't been modified since the last download, no download will
 *  happen.
 *  @return PHISH_SUCCESS if the database was updated and the XML file was
 *          downloaded and saved successfully, PHISH_XML_NOT_MODIFIED if the
 *          database hasn't been modified.
 */
phish_result_t phish_downloadDBAsXML();

/** Given a URL of a site to report, return a URL that the client should go
 *  to which contains the data required to submit a site report. Use this when
 *  you want to report a site by pointing browser to returned URL.
 *  @param url URL of site to report
 *  @param result pointer to string which will be allocated and filled with
 *                URL to point client to in order to submit the report for
 *                this site
 */
phish_result_t phish_getReportingURL(const char *url, char **result);

/** Returns the first entry in the safe list.
 *  @param entry will be assigned with pointer to first entry in safe list
 */
phish_result_t phish_getSafeListFirst(phish_safe_list_entry_t **entry);

/** Returns the next entry in the safe list.
 *  @param entry entry before the entry that will be returned
 *  @param next will be assigned with pointer to entry after entry given above
 */
phish_result_t phish_getSafeListNext(phish_safe_list_entry_t *entry,
                                     phish_safe_list_entry_t **next);

/** Returns the URL associated with a safe list entry.
 *  @param entry entry whose URL is requested
 *  @param url will be assigned with URL of safe list entry
 */
phish_result_t phish_getSafeListData(phish_safe_list_entry_t *entry,
                                     const char **url);

/** Adds a URL to the currently used safe list.
 *  @param url URL to add
 */
phish_result_t phish_addToCurrentSafeList(const char *url);

/** Creates a new, empty safe list.
 *  @param list will be assigned with newly created safe list
 */
phish_result_t phish_newSafeList(phish_safe_list_t **list);

/** Adds a URL to a safe list.
 *  @param list safe list to add URL to
 *  @param url URL to add
 */
phish_result_t phish_addToSafeList(phish_safe_list_t *list, const char *url);

/** Sets the safe list to a new one.
 *  @param list new safe list to use
 */
phish_result_t phish_setSafeList(phish_safe_list_t *list);

/** Saves the safe list to the safe list file.
 */
phish_result_t phish_saveSafeList();

/** Saves the settings to the settings file.
 */
phish_result_t phish_saveSettings();


/** Returns the running mode.
 *  @param result will be assigned with current running mode
 */
phish_result_t phish_runningMode(phish_mode_t *result);

/** Returns the URL used for site queries on the server.
 *  @param result will be assigned with current site query URL
 */
phish_result_t phish_siteQueryURL(const char **result);

/** Returns the URL used for country queries on the server.
 *  @param result will be assigned with current country query URL
 */
phish_result_t phish_countryQueryURL(const char **result);

/** Returns the URL used for reporting sites on the server.
 *  @param result will be assigned with current site reporting URL
 */
phish_result_t phish_reportSiteURL(const char **result);

/** Returns the URL to the XML file to download on the server
 *  @param result will be assigned with current URL of the XML file
 */
phish_result_t phish_remoteXMLURL(const char **result);

/** Returns the name of the local XML file.
 *  @param result will be assigned with current name of the local XML file
 */
phish_result_t phish_localXMLFile(const char **result);

/** Returns the name of the safe list file.
 *  @param result will be assigned with current name of the safe list file
 */
phish_result_t phish_safeListFile(const char **result);

/** Sets the running mode.
 *  @param mode new mode to use
 */
phish_result_t phish_setRunningMode(phish_mode_t mode);

/** Sets the site queries URL.
 *  @param url new URL to use
 */
phish_result_t phish_setSiteQueryURL(const char *url);

/** Sets the country queries URL.
 *  @param url new URL to use
 */
phish_result_t phish_setCountryQueryURL(const char *url);

/** Sets the site reporting URL.
 *  @param url new URL to use
 */
phish_result_t phish_setReportSiteURL(const char *url);

/** Sets the remote XML file URL.
 *  @param url new URL to use
 */
phish_result_t phish_setRemoteXMLURL(const char *url);

/** Sets the local XML file name.
 *  @param path new file name to use
 */
phish_result_t phish_setLocalXMLFile(const char *path);

/** Sets the safe list file name.
 *  @param path new file name to use
 */
phish_result_t phish_setSafeListFile(const char *path);


#ifdef __cplusplus
}
#endif

#endif /* __PHISH_H */

