/***************************************************************************
 *   Copyright (C) 2005 Meni Livne <livne@kde.org>                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#ifndef __PHISH_OPDB_SERVER_H
#define __PHISH_OPDB_SERVER_H

#include "phish.h"
#include "phish_util_url.h"

/** Queries the server about a URL.
 *  @param query_url URL of query script on server
 *  @param url complete URL to check
 *  @param ip ip address of URL
 *  @param user_agent user agent to send to server, or NULL if not to send
 *  @param results pointer to results structure which will be filled with
 *                 results of checking the URL
 */
phish_result_t phish_opdbserver_checkURL(phish_util_url_t *query_url,
                                         phish_util_url_t *url,
                                         const char *ip,
                                         const char *user_agent,
                                         phish_url_data_t *results);

/** Queries the server about the country of an IP address.
 *  @param country_url URL of country query script on server
 *  @param ip ip address of URL
 *  @param user_agent user agent to send to server, or NULL if not to send
 *  @param results pointer to results structure which whose country field will
 *                 be filled with the country corresponding to the given IP
 *                 address
 */
phish_result_t phish_opdbserver_checkCountry(phish_util_url_t *country_url,
                                             const char *ip,
                                             const char *user_agent,
                                             phish_url_data_t *results);

/** Download the sites database as an XML file from the server. If the
 *  database hasn't been modified since the last download, no download will
 *  happen.
 *  @param download_url URL of XML file to download
 *  @param user_agent user agent to send to server, or NULL if not to send
 *  @param local_file path to file on disk to download to
 *  @param etag pointer to null-terminated string representing ETag of last
 *              version downloaded.
 *  @param new_etag pointer to pointer to string which will be allocated and
 *                  filled with ETag of new file if file was modified on server
 *  @return PHISH_SUCCESS if the database was updated and the XML file was
 *          downloaded and saved successfully, PHISH_XML_NOT_MODIFIED if the
 *          database hasn't been modified.
 */
phish_result_t phish_opdbserver_downloadDBAsXML(phish_util_url_t *download_url,
                                                const char *user_agent,
                                                const char *local_file,
                                                const char *etag,
                                                char **new_etag);

/** Given a URL of a site to report, return a URL that the client should go
 *  to which contains the data required to submit a site report.
 *  @param reporting_url URL of reporting script on server
 *  @param client_ua User agent string of client (e.g. anti-phishing toolbar)
 *                   that will be sent in the query to the server
 *  @param url URL of site to report
 *  @param result pointer to string which will be allocated and filled with
 *                URL to point client to in order to submit the report for
 *                this site
 */
phish_result_t phish_opdbserver_getReportingURL(phish_util_url_t *reporting_url,
                                                const char *client_ua,
                                                const char *url,
                                                char **result);


#endif /* __PHISH_OPDB_SERVER_H */

