/***************************************************************************
 *   Copyright (C) 2005 Meni Livne <livne@kde.org>                         *
 *   Copyright (C) 2005 Boaz Anin <boazanin@gmail.com>                     *
 *   Copyright (C) 2005 Shahar Karin <shaharka@post.tau.ac.il>             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "phish_safelist.h"


phish_result_t phish_safeList_new(phish_safelist_t **list)
{
  *list = phish_util_newList();

  if (*list == NULL)
    return PHISH_ERR_MEMORY;
  else
    return PHISH_SUCCESS;
}

phish_result_t phish_safelist_open(const char *path,
                                  phish_safelist_t **list)
{
  FILE *safelist_file;
  char *url;

  *list = phish_util_newList();
  if (*list == NULL)
    return PHISH_ERR_MEMORY;
  
  safelist_file = fopen(path, "r");
  if (safelist_file == NULL)
    return PHISH_ERR_FILE;
  
  while (!feof(safelist_file))
  {
    url = malloc(PHISH_URL_LENGTH_LIMIT);
    if (url == NULL)
    {
      fclose(safelist_file);
      return PHISH_ERR_MEMORY;
    }
    
    if (fgets(url, PHISH_URL_LENGTH_LIMIT, safelist_file) == NULL)
    {
      free(url);
      
      if (feof(safelist_file))
        break;
      else
      {
        fclose(safelist_file);
        return PHISH_ERR_FILE;
      }
    }
    else
    {
      char *nl = strchr(url, '\n');
      if (nl != NULL)
        *nl = '\0'; /* remove newline character */
      
      phish_util_addToList(*list, url);
    }
  }
  
  fclose(safelist_file);
  
  return PHISH_SUCCESS;
}

void phish_safelist_close(phish_safelist_t *list)
{
  phish_util_deepDeleteList(list);
}

phish_result_t phish_safelist_write(const char *path,
                                   phish_safelist_t *list)
{
  FILE *safelist_file;
  char *url;
  phish_safelist_entry_t *entry;
  
  if (list == NULL)
    return PHISH_SUCCESS;
  
  safelist_file = fopen(path, "w");
  if (safelist_file == NULL)
    return PHISH_ERR_FILE;
  
  entry = phish_util_listHead(list);
  while (entry != NULL)
  {
    url = phish_util_listData(entry);
    fprintf(safelist_file, "%s\n", url);

    entry = phish_util_listNext(entry);
  }
  
  fclose(safelist_file);
  return PHISH_SUCCESS;
}

int phish_safelist_checkURL(const char *url, phish_safelist_t *list)
{
  phish_safelist_entry_t *entry;
  
  if (list == NULL)
    return 0;
  
  entry = phish_util_listHead(list);
  while (entry != NULL)
  {
    char *current_url = phish_util_listData(entry);
    int current_url_len = strlen(current_url);
    
    if (current_url_len > 0 && strlen(url) >= current_url_len)
      if (strncmp(current_url, url, current_url_len) == 0)
        return 1;
    
    entry = phish_util_listNext(entry);
  }
  
  return 0;
}

phish_result_t phish_safelist_add(const char *url, phish_safelist_t *list)
{
  char *url_copy;

  /* avoid adding a URL already covered by safe list */
  if (phish_safelist_checkURL(url, list) == 1)
    return PHISH_SUCCESS;

  url_copy = strdup(url);
  if (url_copy == NULL)
    return PHISH_ERR_MEMORY;
  
  phish_util_addToList(list, url_copy);
  return PHISH_SUCCESS;
}

phish_safelist_entry_t *phish_safelist_first(phish_safelist_t *list)
{
  return phish_util_listHead(list);
}

phish_safelist_entry_t *phish_safelist_next(phish_safelist_entry_t *entry)
{
  return phish_util_listNext(entry);
}

const char *phish_safelist_entryData(phish_safelist_entry_t *entry)
{
  return phish_util_listData(entry);
}

