/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
#include "libvg.h"
#include "vg_digest.h"

struct hash_operations *get_hash_op(vg_uint type)
{
	vg_uchar i = 0;
	while(hop_tbl[i].hash_type) {
		if (hop_tbl[i].hash_type == type)
			return (&hop_tbl[i]);
		++i;
	}
	return (NULL);
}

/*
 * 	this transforms the digest produced by one of the 
 * 	algorithms on a printable hexa string
 */
vg_char	*digest2hextring(vg_uint type, vg_uchar *val, vg_uchar *digest)
{
	const vg_char hexa[]="0123456789abcdef";
	vg_int i;

	memset(digest, 0, 40+1);	
	for(i=0;i<MAX_DIGEST_SZ;i++) {
		digest[i+i] = hexa[val[i]>>4];
		digest[i+i+1] = hexa[val[i] & 0x0f];
	}
	switch(type) {
	case MD5:
		digest[32] = '\0';
		break;
	case SHA1:
	case RMD160:
		digest[40] = '\0';
		break;
	}
	return (digest);
}

/*
 * vg_digest_buf = compute the message digest for a chunk of data 
 * of @size bytes
 *
 * @htype = type of hash algorithm
 * @data = data pointer
 * @size = data size
 * @digest = 40 bytes buffer where the produced digest will be stored
 * @fd = file descriptor to dump the digest, optional
 */
vg_int	vg_digest_data(vg_uint htype, vg_uchar *data, vg_size size, 
			vg_uchar *digest, vg_int fd)
{
	struct hash_operations	*hop;
	union all_ctx		ctx;
	vg_uchar		val[MAX_DIGEST_SZ];
	
	/* check */
	if (!htype || !data || !size || (!digest && fd<=0))
		VGERR(VG_DIGEST_DATA, NULL, E_INVAL, ERR);

	/* get hash operations */
	if (!(hop = get_hash_op(htype))) 
		VGERR(VG_DIGEST_DATA, "Incorrect hash algorithm", 0, ERR);

	/* make digest */
	hop->init(&ctx);
	hop->update(&ctx, data, size);
	hop->final(val, &ctx);

	/* transform val to hex string */
	digest2hextring(htype, val, digest);

	/* write to disk */
	if (fd)
		if (write(fd, digest, MAX_DIGEST_SZ) != MAX_DIGEST_SZ)
			VGERR(VG_DIGEST_DATA, INTERNAL_ERR, 0, ERR);

	return (OK);
}

/*
 * vg_digest_obj = compute the message digest for an element of an
 * object: segments, elf sections, symbols, etc.
 * 
 * @obj: object structure
 * @what: object 
 */
vg_int	vg_digest_obj(vg_uint htype, vg_object *obj, vg_int what, 
				vg_uchar *digest, vg_int fd)
{
	if (!htype || !what || !obj || (!digest && fd<=0)) 
		VGERR(VG_DIGEST_OBJ, NULL, E_INVAL, ERR);
	
	switch(what) {
	case VG_OBJ_HDR:
		if (!vg_obj_get_hdr(obj) || 
			vg_digest_data(htype, (char*)obj->elf.hdr.data, 
				 obj->elf.hdr.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_PHDR:
		if (!vg_obj_get_phdr(obj, 0) ||
			vg_digest_data(htype, (char*)obj->elf.phdr.data, 
				obj->elf.phdr.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_INTERP:
		if (!vg_obj_get_interp(obj) ||
			vg_digest_data(htype, (char*)obj->elf.interp.data,
				obj->elf.interp.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_DYN:
		if (!vg_obj_get_dyn(obj, 0) ||
			vg_digest_data(htype, (char*)obj->elf.dyn.data, 
				obj->elf.dyn.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_DYNSYM:
		if (!vg_obj_get_dynsym(obj, 0) ||
			vg_digest_data(htype, (char*)obj->elf.dynsym.data, 
				obj->elf.dynsym.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_DYNSTR:
		if (!vg_obj_get_dynstr(obj) ||
			vg_digest_data(htype, (char*)obj->elf.dynstr.data, 
				obj->elf.dynstr.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_RELPLT:
		if (!vg_obj_get_relplt(obj, 0) ||
			vg_digest_data(htype, (char*)obj->elf.relplt.data, 
				obj->elf.relplt.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_RELDYN:
		if (!vg_obj_get_reldyn(obj, 0) ||
			vg_digest_data(htype, (char*)obj->elf.reldyn.data, 
				obj->elf.reldyn.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_HASH:
		if (!vg_obj_get_hash(obj) ||
			vg_digest_data(htype, (char*)obj->elf.hash.data, 
				obj->elf.hash.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_GOT:
		if (!vg_obj_get_got(obj, 0) ||
			vg_digest_data(htype, (char*)obj->elf.got.data,
				obj->elf.got.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_PLT:
		if (!vg_obj_get_plt(obj, 0) ||
			vg_digest_data(htype, (char*)obj->elf.plt.data,
				obj->elf.plt.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_NOTE:
		if (!vg_obj_get_note(obj) ||
			vg_digest_data(htype, (char*)obj->elf.note.data,
				obj->elf.note.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_BSS:
		if (!vg_obj_get_bss(obj) ||
			vg_digest_data(htype, (char*)obj->elf.bss.data,
				obj->elf.bss.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_TEXT_SEG:
		if (vg_digest_data(htype, (char*)obj->text.ptr, 
				obj->text.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_DATA_SEG:
		if (vg_digest_data(htype, (char*)obj->data.ptr, 
				obj->data.size, digest, fd) == -1)
			return (ERR);
		break;
	case VG_OBJ_HEAP_SEG:
		if (vg_digest_data(htype, (char*)obj->heap.ptr, 
				obj->heap.size, digest, fd) == -1)
			return (ERR);
		break;
	default:
		VGERR(VG_DIGEST_OBJ, "Incorrect request\n", 0,ERR);
		break;
	}

	return (OK);
}

/* 
 * DIGEST ALGORITHMS: I HATE USE OTHERS LIBRARIES ;) 
 *
 * Note: RFC1321, FIPS-180, openssl, libcrypt.... 
 */
/*
  ################################
  ###   MD5 DIGEST ALGORITHM   ###
  ################################
*/

static vg_uint8 MD5_PADDING[MD5_BLOCK_LENGTH] = {
	0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

#define	B64_TO_LE(c, n)		\
do {				\
	(c)[7] = (n)>>56;	\
	(c)[6] = (n)>>48;	\
	(c)[5] = (n)>>40;	\
	(c)[4] = (n)>>32;	\
	(c)[3] = (n)>>24;	\
	(c)[2] = (n)>>16;	\
	(c)[1] = (n)>>8;	\
	(c)[0] = (n);		\
} while(0)

#define	B32_TO_LE(c, n)		\
do {				\
	(c)[3] = (n)>>24;	\
	(c)[2] = (n)>>16;	\
	(c)[1] = (n)>>8;	\
	(c)[0] = (n);		\
} while(0)

#define	FA(x,y,z)	( z ^ (x & (y ^ z)))
#define	FB(x,y,z)	FA(z,x,y)
#define FC(x,y,z)	(x ^ y ^ z)
#define	FD(x,y,z)	(y ^(x | ~z))

#define	FSTEP(f, w, x, y, z, data, s)	\
	( w+=f(x,y,z) + data, w = w<<s | w>>(32-s),  w += x )


vg_void VG_MD5_Init(VG_MD5_CTX *ctx)
{
	ctx->count = 0;
	ctx->state[0] = 0x67452301;
	ctx->state[1] = 0xefcdab89;
	ctx->state[2] = 0x98badcfe;
	ctx->state[3] = 0x10325476;
}

vg_void VG_MD5_Update(VG_MD5_CTX *ctx, const vg_uchar *data, vg_uint size)
{
	vg_size a, b;
	a = (vg_size)((ctx->count >> 3) & (MD5_BLOCK_LENGTH - 1));
	b = MD5_BLOCK_LENGTH - a;
	
	ctx->count += (u_int64_t)size << 3;
	
	if (size>=b) {
		if (a!=0) {
			bcopy(data, ctx->buf+a, b);
			VG_MD5_Transform(ctx->state, ctx->buf);
			data += b;
			size -= b;
			a = 0;
		}
		while(size >= MD5_BLOCK_LENGTH) {
			VG_MD5_Transform(ctx->state, data);
			data += MD5_BLOCK_LENGTH;
			size -= MD5_BLOCK_LENGTH;
		}	
	}
	
	if (size)
		bcopy(data, ctx->buf+a, size);
}

vg_void VG_MD5_Final(u_char digest[16], VG_MD5_CTX *ctx)
{
	vg_uchar bits[8];
	vg_uint idx, padlen;
	
	B64_TO_LE(bits, ctx->count);
	padlen = MD5_BLOCK_LENGTH - ((ctx->count >> 3) & (MD5_BLOCK_LENGTH - 1));
	if (padlen  < 9)	
		padlen += MD5_BLOCK_LENGTH;
	VG_MD5_Update(ctx, MD5_PADDING, padlen - 8);
	VG_MD5_Update(ctx, bits, 8);
	if (digest) {
		for(idx=0;idx<4;idx++) 
			B32_TO_LE(digest+idx*4, ctx->state[idx]);
	}
	memset(ctx, 0, sizeof(VG_MD5_CTX));
}

vg_void VG_MD5_Transform(u_int state[4], const u_char block[MD5_BLOCK_LENGTH])
{
	u_int a, b, c, d, x[16];

#if BYTE_ORDER == LITTLE_ENDIAN
	bcopy(block, x, sizeof(x));
#else
	for (a = 0; a < MD5_BLOCK_LENGTH / 4; a++) {
		x[a] = (u_int32_t)(
		    (u_int32_t)(block[a * 4 + 0]) |
		    (u_int32_t)(block[a * 4 + 1]) <<  8 |
		    (u_int32_t)(block[a * 4 + 2]) << 16 |
		    (u_int32_t)(block[a * 4 + 3]) << 24);
	}
#endif
	a = state[0];
	b = state[1];
	c = state[2];
	d = state[3];

	FSTEP(FA, a, b, c, d, x[ 0] + 0xd76aa478,  7);
	FSTEP(FA, d, a, b, c, x[ 1] + 0xe8c7b756, 12);
	FSTEP(FA, c, d, a, b, x[ 2] + 0x242070db, 17);
	FSTEP(FA, b, c, d, a, x[ 3] + 0xc1bdceee, 22);
	FSTEP(FA, a, b, c, d, x[ 4] + 0xf57c0faf,  7);
	FSTEP(FA, d, a, b, c, x[ 5] + 0x4787c62a, 12);
	FSTEP(FA, c, d, a, b, x[ 6] + 0xa8304613, 17);
	FSTEP(FA, b, c, d, a, x[ 7] + 0xfd469501, 22);
	FSTEP(FA, a, b, c, d, x[ 8] + 0x698098d8,  7);
	FSTEP(FA, d, a, b, c, x[ 9] + 0x8b44f7af, 12);
	FSTEP(FA, c, d, a, b, x[10] + 0xffff5bb1, 17);
	FSTEP(FA, b, c, d, a, x[11] + 0x895cd7be, 22);
	FSTEP(FA, a, b, c, d, x[12] + 0x6b901122,  7);
	FSTEP(FA, d, a, b, c, x[13] + 0xfd987193, 12);
	FSTEP(FA, c, d, a, b, x[14] + 0xa679438e, 17);
	FSTEP(FA, b, c, d, a, x[15] + 0x49b40821, 22);

	FSTEP(FB, a, b, c, d, x[ 1] + 0xf61e2562,  5);
	FSTEP(FB, d, a, b, c, x[ 6] + 0xc040b340,  9);
	FSTEP(FB, c, d, a, b, x[11] + 0x265e5a51, 14);
	FSTEP(FB, b, c, d, a, x[ 0] + 0xe9b6c7aa, 20);
	FSTEP(FB, a, b, c, d, x[ 5] + 0xd62f105d,  5);
	FSTEP(FB, d, a, b, c, x[10] + 0x02441453,  9);
	FSTEP(FB, c, d, a, b, x[15] + 0xd8a1e681, 14);
	FSTEP(FB, b, c, d, a, x[ 4] + 0xe7d3fbc8, 20);
	FSTEP(FB, a, b, c, d, x[ 9] + 0x21e1cde6,  5);
	FSTEP(FB, d, a, b, c, x[14] + 0xc33707d6,  9);
	FSTEP(FB, c, d, a, b, x[ 3] + 0xf4d50d87, 14);
	FSTEP(FB, b, c, d, a, x[ 8] + 0x455a14ed, 20);
	FSTEP(FB, a, b, c, d, x[13] + 0xa9e3e905,  5);
	FSTEP(FB, d, a, b, c, x[ 2] + 0xfcefa3f8,  9);
	FSTEP(FB, c, d, a, b, x[ 7] + 0x676f02d9, 14);
	FSTEP(FB, b, c, d, a, x[12] + 0x8d2a4c8a, 20);

	FSTEP(FC, a, b, c, d, x[ 5] + 0xfffa3942,  4);
	FSTEP(FC, d, a, b, c, x[ 8] + 0x8771f681, 11);
	FSTEP(FC, c, d, a, b, x[11] + 0x6d9d6122, 16);
	FSTEP(FC, b, c, d, a, x[14] + 0xfde5380c, 23);
	FSTEP(FC, a, b, c, d, x[ 1] + 0xa4beea44,  4);
	FSTEP(FC, d, a, b, c, x[ 4] + 0x4bdecfa9, 11);
	FSTEP(FC, c, d, a, b, x[ 7] + 0xf6bb4b60, 16);
	FSTEP(FC, b, c, d, a, x[10] + 0xbebfbc70, 23);
	FSTEP(FC, a, b, c, d, x[13] + 0x289b7ec6,  4);
	FSTEP(FC, d, a, b, c, x[ 0] + 0xeaa127fa, 11);
	FSTEP(FC, c, d, a, b, x[ 3] + 0xd4ef3085, 16);
	FSTEP(FC, b, c, d, a, x[ 6] + 0x04881d05, 23);
	FSTEP(FC, a, b, c, d, x[ 9] + 0xd9d4d039,  4);
	FSTEP(FC, d, a, b, c, x[12] + 0xe6db99e5, 11);
	FSTEP(FC, c, d, a, b, x[15] + 0x1fa27cf8, 16);
	FSTEP(FC, b, c, d, a, x[ 2] + 0xc4ac5665, 23);

	FSTEP(FD, a, b, c, d, x[ 0] + 0xf4292244,  6);
	FSTEP(FD, d, a, b, c, x[ 7] + 0x432aff97, 10);
	FSTEP(FD, c, d, a, b, x[14] + 0xab9423a7, 15);
	FSTEP(FD, b, c, d, a, x[ 5] + 0xfc93a039, 21);
	FSTEP(FD, a, b, c, d, x[12] + 0x655b59c3,  6);
	FSTEP(FD, d, a, b, c, x[ 3] + 0x8f0ccc92, 10);
	FSTEP(FD, c, d, a, b, x[10] + 0xffeff47d, 15);
	FSTEP(FD, b, c, d, a, x[ 1] + 0x85845dd1, 21);
	FSTEP(FD, a, b, c, d, x[ 8] + 0x6fa87e4f,  6);
	FSTEP(FD, d, a, b, c, x[15] + 0xfe2ce6e0, 10);
	FSTEP(FD, c, d, a, b, x[ 6] + 0xa3014314, 15);
	FSTEP(FD, b, c, d, a, x[13] + 0x4e0811a1, 21);
	FSTEP(FD, a, b, c, d, x[ 4] + 0xf7537e82,  6);
	FSTEP(FD, d, a, b, c, x[11] + 0xbd3af235, 10);
	FSTEP(FD, c, d, a, b, x[ 2] + 0x2ad7d2bb, 15);
	FSTEP(FD, b, c, d, a, x[ 9] + 0xeb86d391, 21);
	
	state[0] += a;
	state[1] += b;
	state[2] += c;
	state[3] += d;
}

/*
  ################
  ###   SHA1   ###
  ################
*/

#define SHA1HANDSOFF

#define rol(value, bits) (((value) << (bits)) | ((value) >> (32 - (bits))))

#if BYTE_ORDER == LITTLE_ENDIAN
#define blk0(i) (block->l[i] = (rol(block->l[i],24)&0xFF00FF00) \
    |(rol(block->l[i],8)&0x00FF00FF))
#else
#define blk0(i) block->l[i]
#endif
#define blk(i) (block->l[i&15] = rol(block->l[(i+13)&15]^block->l[(i+8)&15] \
    ^block->l[(i+2)&15]^block->l[i&15],1))

#define R0(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk0(i)+0x5A827999+rol(v,5);w=rol(w,30);
#define R1(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk(i)+0x5A827999+rol(v,5);w=rol(w,30);
#define R2(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0x6ED9EBA1+rol(v,5);w=rol(w,30);
#define R3(v,w,x,y,z,i) z+=(((w|x)&y)|(w&x))+blk(i)+0x8F1BBCDC+rol(v,5);w=rol(w,30);
#define R4(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0xCA62C1D6+rol(v,5);w=rol(w,30);

void	VG_SHA1_Init(VG_SHA1_CTX *ctx)
{ 
	ctx->count = 0;
	ctx->state[0] = 0x67452301;
	ctx->state[1] = 0xEFCDAB89;
	ctx->state[2] = 0x98BADCFE; 
	ctx->state[3] = 0x10325476;
	ctx->state[4] = 0xC3D2E1F0;
}

void	VG_SHA1_Update(VG_SHA1_CTX *ctx, u_char *data, u_int len) 
{ 
	u_int i,j;
	j = (u_int)((ctx->count >> 3) & (SHA1_BLOCK_LENGTH - 1));
	ctx->count += (len<<3);
	if ((j+len) > (SHA1_BLOCK_LENGTH - 1)) {
		bcopy(data, &ctx->buf[j], (i=SHA1_BLOCK_LENGTH-j));
		VG_SHA1_Transform(ctx->state, ctx->buf);
		for(;i+63<len; i+=SHA1_BLOCK_LENGTH) {
			VG_SHA1_Transform(ctx->state, &data[i]);	
		}
		j = 0;
	}
	else 
		i = 0;
	bcopy(&data[i], &ctx->buf[j], len - i);
}

void    VG_SHA1_Final(u_char *digest, VG_SHA1_CTX *ctx) 
{
	u_char bits[8];
	u_int i;

	for (i = 0; i < 8; i++) {
		bits[i] = (u_char)((ctx->count >> 
			((7 - (i & 7)) * 8)) & 255);
	}

	VG_SHA1_Update(ctx, (u_char*)"\200", 1);
	while ((ctx->count & 504) != 448) {
		VG_SHA1_Update(ctx, (u_char*)"\0", 1);
	}
	VG_SHA1_Update(ctx, bits, 8);
	if (digest) {
		for (i = 0; i < SHA1_DIGEST_LENGTH; i++) {
			digest[i] = (u_char)((ctx->state[i >> 2] >> 
					((3 - (i & 3)) * 8)) & 255);
		}
	}
}

void	VG_SHA1_Transform(u_int state[5], u_char buf[SHA1_BLOCK_LENGTH])
{
	u_int a, b, c, d, e;
	typedef union {
		u_char	c[64];
		u_int	l[16];
	} CHAR64LONG16;

	CHAR64LONG16* block;
#ifdef SHA1HANDSOFF
	static unsigned char workspace[SHA1_BLOCK_LENGTH];
	block = (CHAR64LONG16 *)workspace;
	bcopy(buf, block, SHA1_BLOCK_LENGTH);
#else
	block = (CHAR64LONG16 *)buf;
#endif	
	a = state[0];
	b = state[1];
	c = state[2];
	d = state[3];
	e = state[4];

	R0(a,b,c,d,e, 0); R0(e,a,b,c,d, 1); R0(d,e,a,b,c, 2); R0(c,d,e,a,b, 3);
	R0(b,c,d,e,a, 4); R0(a,b,c,d,e, 5); R0(e,a,b,c,d, 6); R0(d,e,a,b,c, 7);
	R0(c,d,e,a,b, 8); R0(b,c,d,e,a, 9); R0(a,b,c,d,e,10); R0(e,a,b,c,d,11);
	R0(d,e,a,b,c,12); R0(c,d,e,a,b,13); R0(b,c,d,e,a,14); R0(a,b,c,d,e,15);

	R1(e,a,b,c,d,16); R1(d,e,a,b,c,17); R1(c,d,e,a,b,18); R1(b,c,d,e,a,19);

	R2(a,b,c,d,e,20); R2(e,a,b,c,d,21); R2(d,e,a,b,c,22); R2(c,d,e,a,b,23);
	R2(b,c,d,e,a,24); R2(a,b,c,d,e,25); R2(e,a,b,c,d,26); R2(d,e,a,b,c,27);
	R2(c,d,e,a,b,28); R2(b,c,d,e,a,29); R2(a,b,c,d,e,30); R2(e,a,b,c,d,31);
	R2(d,e,a,b,c,32); R2(c,d,e,a,b,33); R2(b,c,d,e,a,34); R2(a,b,c,d,e,35);
 	R2(e,a,b,c,d,36); R2(d,e,a,b,c,37); R2(c,d,e,a,b,38); R2(b,c,d,e,a,39);

	R3(a,b,c,d,e,40); R3(e,a,b,c,d,41); R3(d,e,a,b,c,42); R3(c,d,e,a,b,43);
	R3(b,c,d,e,a,44); R3(a,b,c,d,e,45); R3(e,a,b,c,d,46); R3(d,e,a,b,c,47);
	R3(c,d,e,a,b,48); R3(b,c,d,e,a,49); R3(a,b,c,d,e,50); R3(e,a,b,c,d,51);
	R3(d,e,a,b,c,52); R3(c,d,e,a,b,53); R3(b,c,d,e,a,54); R3(a,b,c,d,e,55);
	R3(e,a,b,c,d,56); R3(d,e,a,b,c,57); R3(c,d,e,a,b,58); R3(b,c,d,e,a,59);

	R4(a,b,c,d,e,60); R4(e,a,b,c,d,61); R4(d,e,a,b,c,62); R4(c,d,e,a,b,63);
	R4(b,c,d,e,a,64); R4(a,b,c,d,e,65); R4(e,a,b,c,d,66); R4(d,e,a,b,c,67);
	R4(c,d,e,a,b,68); R4(b,c,d,e,a,69); R4(a,b,c,d,e,70); R4(e,a,b,c,d,71);
	R4(d,e,a,b,c,72); R4(c,d,e,a,b,73); R4(b,c,d,e,a,74); R4(a,b,c,d,e,75);
	R4(e,a,b,c,d,76); R4(d,e,a,b,c,77); R4(c,d,e,a,b,78); R4(b,c,d,e,a,79);

	state[0] += a;
	state[1] += b;
	state[2] += c;
	state[3] += d;
	state[4] += e;

	a = b = c = d = e = 0;
}

