/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
#include "libvg.h"


/*
 * vg_dmp_obj_hdr = dump the elf header of an object specificed by @obj
 *
 * @obj: a valid object structure
 * @fd: file descriptor
 * @buf: local buffer
 * #ret: dumped data size
 */
vg_int	vg_dump_obj_hdr(vg_object *obj, vg_int fd, vg_char *buf)
{
	/* check args */
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_HDR, NULL, E_INVAL, ERR);

	/* get things that we need */
	if (!vg_obj_get_hdr(obj))
		return (ERR);

	/* write data: disk or memory */
	if (fd) 
		if (write(fd, obj->elf.hdr.data, obj->elf.hdr.size) !=
						 obj->elf.hdr.size)
			VGERR(VG_DUMP_OBJ_HDR, INTERNAL_ERR, 0, ERR);
	if (buf) 
		memcpy(buf, obj->elf.hdr.data, obj->elf.hdr.size);
	return (obj->elf.hdr.size);
}

/*
 * vg_dump_obj_phdr = dump the program header table of an object specificed
 * by @obj
 *
 */
vg_int	vg_dump_obj_phdr(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf)) 
		VGERR(VG_DUMP_OBJ_PHDR, NULL, E_INVAL, ERR);

	if (!vg_obj_get_phdr(obj, 0))
		return (ERR);

	if (fd)
		if (write(fd, obj->elf.phdr.data, obj->elf.phdr.size) !=
						  obj->elf.phdr.size)
			VGERR(VG_DUMP_OBJ_PHDR, INTERNAL_ERR, 0, ERR);
	if (buf) 
		memcpy(buf, obj->elf.phdr.data, obj->elf.phdr.size);
	return(obj->elf.phdr.size);
}

/*
 * vg_dump_obj_dynamic = dump the dynamic table of an object specified
 * by @obj
 *
 */
vg_int	vg_dump_obj_dyn(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_DYN, NULL, E_INVAL, ERR);

	if (!vg_obj_get_dyn(obj,0))
		return (ERR);

	if (fd)
		if (write(fd, obj->elf.dyn.data, obj->elf.dyn.size) != 
						 obj->elf.dyn.size)
			VGERR(VG_DUMP_OBJ_DYN, INTERNAL_ERR, 0, ERR);
	if (buf)
		memcpy(buf, obj->elf.dyn.data, obj->elf.dyn.size);	
	return(obj->elf.dyn.size);
}

/* 
 * vg_dump_obj_dynsym = dump the dynamic symbol table of an object
 * specified by @obj
 *
 */
vg_int	vg_dump_obj_dynsym(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_DYNSYM, NULL, E_INVAL, ERR);

	if (!vg_obj_get_dynsym(obj,0))
		return (ERR);

	if (fd)
		if (write(fd, obj->elf.dynsym.data, obj->elf.dynsym.size) !=
						    obj->elf.dynsym.size) 
			VGERR(VG_DUMP_OBJ_DYNSYM, INTERNAL_ERR, 0, ERR);
	if (buf) 
		memcpy(buf, obj->elf.dynsym.data, obj->elf.dynsym.size);
	return (obj->elf.dynsym.size);
}

/*
 * vg_dump_obj_dynstr = dump the dynamic string table of an object
 * specified by @obj
 *
 */
vg_int	vg_dump_obj_dynstr(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_DYNSTR, NULL, E_INVAL, ERR);

	if (!vg_obj_get_dynstr(obj))
		return (ERR);

	if (fd)
		if (write(fd, obj->elf.dynstr.data, obj->elf.dynstr.size) != 
						    obj->elf.dynstr.size) 
			VGERR(VG_DUMP_OBJ_DYNSTR, INTERNAL_ERR, 0, ERR);
	if (buf)
		memcpy(buf, obj->elf.dynstr.data, obj->elf.dynstr.size);
	return (obj->elf.dynstr.size);
}

/*
 * vg_dump_obj_reldyn = 
 *
 */
vg_int	vg_dump_obj_reldyn(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_RELDYN, NULL, E_INVAL, ERR);

	if (!vg_obj_get_reldyn(obj,0))
		return (ERR);

	if (fd)
		if (write(fd, obj->elf.reldyn.data, obj->elf.reldyn.size)!= 
						    obj->elf.reldyn.size)
			VGERR(VG_DUMP_OBJ_RELDYN, INTERNAL_ERR, 0, ERR);
	if (buf)
		memcpy(buf, obj->elf.reldyn.data, obj->elf.reldyn.size);
	return (obj->elf.reldyn.size);
}

/*
 * vg_dump_obj_relplt = 
 *
 */
vg_int	vg_dump_obj_relplt(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_RELPLT, NULL, E_INVAL, ERR);

	if (!vg_obj_get_relplt(obj,0))
		return (ERR);

	if (fd)
		if (write(fd, obj->elf.relplt.data, obj->elf.relplt.size) != 
						    obj->elf.relplt.size)
			VGERR(VG_DUMP_OBJ_RELPLT, INTERNAL_ERR, 0, ERR);
	if (buf)
		memcpy(buf, obj->elf.relplt.data, obj->elf.relplt.size);
	return (obj->elf.relplt.size);
}

/* 
 * vg_dump_obj_hash =
 *
 */
vg_int	vg_dump_obj_hash(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_HASH, NULL, E_INVAL, ERR);

	if (!vg_obj_get_hash(obj))
		return (ERR);

	if (fd)
		if (write(fd, obj->elf.hash.data, obj->elf.hash.size) != 
						  obj->elf.hash.size)
			VGERR(VG_DUMP_OBJ_HASH, INTERNAL_ERR, 0, ERR);
	if (buf)
		memcpy(buf, obj->elf.hash.data, obj->elf.hash.size);
        return (obj->elf.hash.size);
}

/*
 * vg_dump_obj_plt = dump the procedure linkage table of an object
 * specified by @obj
 *
 */
vg_int	vg_dump_obj_plt(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_PLT, NULL, E_INVAL, ERR);

	if (!vg_obj_get_plt(obj, 0))
		return (ERR);

	if (fd)
		if (write(fd, obj->elf.plt.data, obj->elf.plt.size) !=
						 obj->elf.plt.size)
			VGERR(VG_DUMP_OBJ_PLT, INTERNAL_ERR, 0, ERR);
	if (buf)
		memcpy(buf, obj->elf.plt.data, obj->elf.plt.size);
	return (obj->elf.plt.size);
}

/*
 * vg_dump_obj_got =  dump the global offset table of an object
 * specified by @obj
 *
 */
vg_int	vg_dump_obj_got(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf)) 
		VGERR(VG_DUMP_OBJ_GOT, NULL, E_INVAL, ERR);

	if (!vg_obj_get_got(obj,0))
		return (ERR);

	if (fd)
		if (write(fd, obj->elf.got.data, obj->elf.got.size) !=
						 obj->elf.got.size)
			VGERR(VG_DUMP_OBJ_GOT, INTERNAL_ERR, 0, ERR);
	if (buf)
		memcpy(buf, obj->elf.got.data, obj->elf.got.size);
	return (obj->elf.got.size);
}

/*
 * vg_dump_obj_interp = dump the interpreter string of an object 
 * specified by @obj
 *
 */
vg_int	vg_dump_obj_interp(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_INTERP, NULL, E_INVAL, ERR);

	if (!vg_obj_get_interp(obj))
		return (ERR);
	
	if (fd)
		if (write(fd, obj->elf.interp.data, obj->elf.interp.size) !=
						    obj->elf.interp.size)
			VGERR(VG_DUMP_OBJ_INTERP, INTERNAL_ERR, 0, ERR);
	if (buf)
		memcpy(buf, obj->elf.interp.data, obj->elf.interp.size);
	return (obj->elf.interp.size);
}

/*
 * vg_dump_obj_note = dump the note section of an object specified by
 * @obj
 *
 */
vg_int	vg_dump_obj_note(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_NOTE, NULL, E_INVAL, ERR);

	if (!vg_obj_get_note(obj))
		return (ERR);

	if (fd)
		if (write(fd, obj->elf.note.data, obj->elf.note.size) !=
						  obj->elf.note.size)
			VGERR(VG_DUMP_OBJ_NOTE, INTERNAL_ERR, 0, ERR);		
	if (buf)
		memcpy(buf, obj->elf.note.data, obj->elf.note.size);
	return (obj->elf.note.size);
}

/*
 * vg_dump_obj_bss = dump the bss section of an object specified by
 * @obj
 *
 */
vg_int	vg_dump_obj_bss(vg_object *obj, vg_int fd, vg_char *buf)
{
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_BSS, NULL, E_INVAL, ERR);

	if (!vg_obj_get_bss(obj))
		return (ERR);

	if (fd)
		if (write(fd, obj->elf.bss.data, obj->elf.bss.size) !=
						 obj->elf.bss.size)
			VGERR(VG_DUMP_OBJ_BSS, INTERNAL_ERR, 0, ERR);
	if (buf)
		memcpy(buf, obj->elf.bss.data, obj->elf.bss.size);
	return (obj->elf.bss.size);	
}

/*
 * vg_dump_obj_seg = dump segment
 *
 * @obj:
 * @fd:
 * @buf:
 * @type:
 * #ret:
 */
vg_int	vg_dump_obj_seg(vg_object *obj, vg_int fd, vg_char *buf, vg_uint type)
{
	vg_char	*data;
	vg_int	sz = 0;

        /* check args */
	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ_SEG, NULL, E_INVAL, ERR);

	if (OBJ_IS_MAPPED(obj)) {		/* local */
		/* check type */
        	switch(type) {
        	case VG_TEXT_SEG:
			if (fd)
				if (write(fd, obj->text.ptr, obj->text.size) != 
							     obj->text.size) 
					VGERR(VG_DUMP_OBJ_SEG, INTERNAL_ERR,0, ERR);
			if (buf)
				memcpy(buf, obj->text.ptr, obj->text.size);
			sz = obj->text.size;
			break;
        	case VG_DATA_SEG:
			if (fd)
        	        	if (write(fd, obj->data.ptr, obj->data.size) !=
						 	     obj->data.size)
					VGERR(VG_DUMP_OBJ_SEG, INTERNAL_ERR, 0,ERR);
			if (buf)
				memcpy(buf, obj->data.ptr, obj->data.size);
			sz = obj->data.size;
        	        break;
		case VG_HEAP_SEG:
			if (fd)
				if (write(fd, obj->heap.ptr, obj->heap.size) !=
							     obj->heap.size)
					VGERR(VG_DUMP_OBJ_SEG, INTERNAL_ERR, 0,ERR);
			if (buf)
				memcpy(buf, obj->heap.ptr, obj->heap.size);
			sz = obj->heap.size;
			break;
       		default:
			VGERR(VG_DUMP_OBJ_SEG, "Incorrect segment type" , 0, ERR);
			break;
        	}
	}
	else {					/* remote */
		/* get data directly from proc */
		switch(type) {
		case VG_TEXT_SEG:
			S_ALLOC(data, obj->text.size, ERR, VG_DUMP_OBJ_SEG, CLEAN);
			if (vg_dread(data, obj->text.start, obj->text.size) != 
							    obj->text.size)
				VGERR2(VG_DUMP_OBJ_SEG, 0,0,ERR,data); 
			if (fd)
				if (write(fd, data, obj->text.size)!= obj->text.size)
					VGERR2(VG_DUMP_OBJ_SEG, INTERNAL_ERR, 0, ERR, data);
			if (buf)
				memcpy(buf, data, obj->text.size);
			sz = obj->text.size;
			free(data);
			break;
		case VG_DATA_SEG:
			S_ALLOC(data, obj->data.size, ERR, VG_DUMP_OBJ_SEG, CLEAN);
 			if (vg_dread(data, obj->data.start, obj->data.size) != 
							    obj->data.size)
				VGERR2(VG_DUMP_OBJ_SEG, 0, 0, ERR, data);
			if (fd) 
				if (write(fd, data, obj->data.size)!=obj->data.size)
					VGERR2(VG_DUMP_OBJ_SEG, INTERNAL_ERR,0, ERR, data);
			if (buf)
				memcpy(buf, data, obj->data.size);
			sz = obj->data.size;
			free(data);
			break;
		case VG_HEAP_SEG:
			S_ALLOC(data, obj->heap.size, ERR, VG_DUMP_OBJ_SEG, CLEAN);
			if (vg_dread(data, obj->heap.start, obj->heap.size) !=	
							    obj->heap.size)
				VGERR2(VG_DUMP_OBJ_SEG, 0, 0, ERR, data);
			if (fd)
				if (write(fd, data, obj->heap.size)!=obj->heap.size)
					VGERR2(VG_DUMP_OBJ_SEG, INTERNAL_ERR,0, ERR, data);
			if (buf)
				memcpy(buf, data, obj->heap.size);
			sz = obj->heap.size;
			free(data);
			break;	
		default:
			VGERR(VG_DUMP_OBJ_SEG, "Incorrect segment type", 0, ERR);
		}
	}
        return(sz);
}

/*
 *	[A]	DUMP WHOLE OBJECT
 */
vg_int	vg_dump_obj(vg_object *obj, vg_int fd, vg_char *buf)
{
	vg_char *data;
	vg_int	sz;

	if (!obj || (fd<0 && !buf))
		VGERR(VG_DUMP_OBJ, NULL, E_INVAL, ERR);

	if (OBJ_IS_MAPPED(obj)) {		/* local */
		/* text size + data size, addr -> obj->text.ptr */
		if (fd)
			if (write(fd, obj->text.ptr, (obj->text.size + obj->data.size)) 
					 != (obj->text.size + obj->data.size))
				VGERR(VG_DUMP_OBJ,INTERNAL_ERR, 0, ERR);
		if (buf)
			memcpy(buf, obj->text.ptr, (obj->text.size + obj->data.size));
		sz = obj->text.size + obj->data.size;
	}		
	else {					/* remote */
		/* allocate mem for data, text and heap */
		S_ALLOC(data, obj->text.size+obj->data.size+obj->heap.size, ERR, 
							VG_DUMP_OBJ, CLEAN);	
		
		/* first get text */
		if (vg_dread(data, obj->text.start, obj->text.size) !=
						    obj->text.size) 
			VGERR2(VG_DUMP_OBJ, 0, 0, ERR, data);

		/* second get data */
		if (vg_dread((char*)data+obj->text.size, obj->data.start,
				obj->data.size) != obj->data.size) 
			VGERR2(VG_DUMP_OBJ, 0, 0, ERR, data);

		/* then dump it */
		if (fd) 
			if (write(fd, data, obj->text.size + obj->data.size) != 
					(obj->text.size + obj->data.size))
				VGERR2(VG_DUMP_OBJ, INTERNAL_ERR, 0, ERR, data);
		if (buf)
			memcpy(buf, data, obj->text.size + obj->data.size);
		sz = obj->text.size + obj->data.size;
		free(data);
	}
	return (sz);
}

/*
 * vg_dump_stack = dump stack segment 
 *
 */
vg_int	vg_dump_stack(vg_int fd, vg_char *buf)
{
	vg_char *data;

	if (fd<0 && !buf)
		VGERR(VG_DUMP_STACK, NULL, E_INVAL, ERR);

	S_ALLOC(data, cur_proc->stack.csize, ERR, VG_DUMP_STACK, CLEAN);
	if (vg_dread(data, cur_proc->stack.start ,cur_proc->stack.csize) != 
						  cur_proc->stack.csize)
		VGERR2(VG_DUMP_STACK, 0,0, ERR, data);

	if (fd)
		if (write(fd, data, cur_proc->stack.csize) !=
				    cur_proc->stack.csize)
			VGERR2(VG_DUMP_STACK ,INTERNAL_ERR, ERR, 0, data);
	if (buf)
		memcpy(buf, data, cur_proc->stack.csize);
	free(data);
	return (cur_proc->stack.csize);
}

/*
 * vg_dump_data = dump some data
 */
vg_int	vg_dump_data(vg_vaddr start, vg_vaddr end, vg_int fd, vg_char *buf)
{
	vg_char	*data;

	/* check correct args */
	if ((!start || !end || (fd<0 && !buf)) || (end <= start))
		VGERR(VG_DUMP_DATA, NULL, E_INVAL, ERR);

	/* get mem */
	S_ALLOC(data, end - start, ERR, VG_DUMP_DATA, CLEAN);

	/* get data from local or remote, vg_read */
	if (vg_read(data, start, (end-start)) != (end-start))
		VGERR2(VG_DUMP_DATA, 0, 0, ERR, data);

	/* write */
	if (fd)
		if (write(fd, data, (end-start)) != (end-start))
			VGERR2(VG_DUMP_DATA, INTERNAL_ERR, 0, ERR, data);
	if (buf)
		memcpy(buf, data, (end-start));
	free(data);
	return (end-start);
}

/* 
 * vg_dump_sym = 
 *
 * @sym
 * @name
 */
vg_int	vg_dump_sym(vg_int fd, vg_char *buf, vg_sym *sym, vg_char *name)
{
	vg_sym	s;

	if ((fd<0 && !buf) || (!sym && !name))
		VGERR(VG_DUMP_SYM, NULL, E_INVAL, ERR);

	if (name) {
		if (vg_sym_func_local_by_name(0, &s, name)<0)
			return (ERR);
		sym = &s;
	}

	if (!sym->size)	
		VGERR(VG_DUMP_SYM, "Incorrect symbol size", 0, ERR);

	vg_dump_data(sym->vaddr, sym->vaddr+sym->size, fd, buf);

	return (sym->size);
}


