/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
#include "libvg.h"

/*
 * vg_obj_get_dyn = load dynamic section
 */
Elf32_Dyn *vg_obj_get_dyn(vg_object *obj, vg_uint *entnum)
{
	Elf32_Phdr	*phdr;

	CHECKDYN(VG_OBJ_GET_DYN, NULL);

	/* check args */
	if (!obj)
		VGERR(VG_OBJ_GET_DYN, NULL, E_INVAL, NULL);

	/* si ia sta salir */
	if (obj->elf.dyn.data)
		goto end;

	/* we need phdr */
	if (!vg_obj_get_phdr(obj, NULL))
		return (NULL);

	/* get phdr of PT_DYNAMIC */
	if (!(phdr = vg_obj_phentry_by_type(obj, PT_DYNAMIC, 0)))
		VGERR(VG_OBJ_GET_DYN, 0, 0, NULL);

	/* some systems set p->memsz = 0, like solaris, so in
	   this cases we will use p->filesz
	*/
	if (phdr->p_memsz)
		obj->elf.dyn.size = phdr->p_memsz;
	else if (phdr->p_filesz) 
		obj->elf.dyn.size = phdr->p_filesz;
	else 
		VGERR(VG_OBJ_GET_DYN, "Incorrect section size", 0, NULL);

	/* xeck object */
	if (OBJ_IS_MAPPED(obj)) {
		obj->elf.dyn.data = (Elf32_Dyn*)RTOL((u_long)obj->od->od_dyn, obj);
	}
	else {
		S_ALLOC(obj->elf.dyn.data, obj->elf.dyn.size, NULL, 
						VG_OBJ_GET_DYN, CLEAN);
		if (vg_dread(obj->elf.dyn.data, (u_long)obj->od->od_dyn, 
					obj->elf.dyn.size) != obj->elf.dyn.size)
			VGERR2(VG_OBJ_GET_DYN, 0, 0, NULL, obj->elf.dyn.data);
	}

	obj->elf.dyn.vaddr = phdr->p_vaddr;
	obj->elf.dyn.offset = obj->elf.dyn.vaddr - obj->text.start;
	obj->elf.dyn.entsize = sizeof(Elf32_Dyn);
	obj->elf.dyn.entnum = obj->elf.dyn.size / obj->elf.dyn.entsize;
end:
	if (entnum)
		*entnum = obj->elf.dyn.entnum;
	return (obj->elf.dyn.data);	
}

/*
 * vg_obj_dyn_get/set_vaddr = 
 */
vg_vaddr vg_obj_dyn_get_vaddr(vg_object *obj)
{
	CHECKDYN(VG_OBJ_DYN_GET_VADDR, ERR);
	if (!obj || !obj->elf.dyn.data)
		VGERR(VG_OBJ_DYN_GET_VADDR, NULL, E_INVAL, ERR);
	return (obj->elf.dyn.vaddr);
}

vg_vaddr vg_obj_dyn_set_vaddr(vg_object *obj, vg_vaddr vaddr)
{
	CHECKDYN(VG_OBJ_DYN_SET_VADDR, ERR);
	if (!obj || !obj->elf.dyn.data)
		VGERR(VG_OBJ_DYN_SET_VADDR, NULL, E_INVAL, ERR);
	obj->elf.dyn.vaddr = vaddr;
	return (obj->elf.dyn.vaddr);
}

/*
 * vg_obj_dyn_get/set_size = 
 */
vg_size	vg_obj_dyn_get_size(vg_object *obj)
{
	CHECKDYN(VG_OBJ_DYN_GET_SIZE, ERR);
	if (!obj || !obj->elf.dyn.data)
		VGERR(VG_OBJ_DYN_GET_SIZE, NULL, E_INVAL, ERR);
	return (obj->elf.dyn.size);
}

vg_size	vg_obj_dyn_set_size(vg_object *obj, vg_size size)
{
	CHECKDYN(VG_OBJ_DYN_SET_SIZE, ERR);
	if (!obj || !obj->elf.dyn.data)
		VGERR(VG_OBJ_DYN_SET_SIZE, NULL, E_INVAL, ERR);
	obj->elf.dyn.size = size;
	return (obj->elf.dyn.size);
}

/*
 * vg_obj_dyn_get/set_offset = 
 */
vg_off	vg_obj_dyn_get_offset(vg_object *obj)
{
	CHECKDYN(VG_OBJ_DYN_GET_OFFSET, ERR);
	if (!obj || !obj->elf.dyn.data)
		VGERR(VG_OBJ_DYN_GET_OFFSET, NULL, E_INVAL, ERR);
	return (obj->elf.dyn.offset);	
}

vg_off	vg_obj_dyn_set_offset(vg_object *obj, vg_off offset)
{
	CHECKDYN(VG_OBJ_DYN_SET_OFFSET, ERR);
	if (!obj || !obj->elf.dyn.data)
		VGERR(VG_OBJ_DYN_SET_OFFSET, NULL, E_INVAL, ERR);
	obj->elf.dyn.offset = offset;
	return (obj->elf.dyn.offset);
}

/*
 * vg_obj_dyn_get/set_entsize = 
 */
vg_size vg_obj_dyn_get_entsize(vg_object *obj)
{
	CHECKDYN(VG_OBJ_DYN_GET_ENTSIZE, ERR);
	if (!obj || !obj->elf.dyn.data)
		VGERR(VG_OBJ_DYN_GET_ENTSIZE, NULL, E_INVAL, ERR);
	return (obj->elf.dyn.entsize);
}	

vg_size	vg_obj_dyn_set_entsize(vg_object *obj, vg_size entsize)
{
	CHECKDYN(VG_OBJ_DYN_SET_ENTSIZE, ERR);
	if (!obj || !obj->elf.dyn.data)
		VGERR(VG_OBJ_DYN_SET_ENTSIZE, NULL, E_INVAL, ERR);
	obj->elf.dyn.entsize = entsize;
	return (obj->elf.dyn.entsize);
}

/*
 * vg_obj_dyn_get/set_entnum = 
 */
vg_idx	vg_obj_dyn_get_entnum(vg_object *obj)
{
	CHECKDYN(VG_OBJ_DYN_GET_ENTNUM, ERR);
	if (!obj || !obj->elf.dyn.data)
		VGERR(VG_OBJ_DYN_GET_ENTNUM, NULL, E_INVAL, ERR);
	return (obj->elf.dyn.entnum);
}

vg_idx	vg_obj_dyn_set_entnum(vg_object *obj, vg_idx entnum)
{
	CHECKDYN(VG_OBJ_DYN_SET_ENTNUM, ERR);
	if (!obj || !obj->elf.dyn.data)
		VGERR(VG_OBJ_DYN_SET_ENTNUM, NULL, E_INVAL, ERR);
	obj->elf.dyn.entnum = entnum;
	return (obj->elf.dyn.entnum);
}


/*
 * vg_obj_dentry_by_type = return a dynamic entry giving its type (d_tag)
 */
Elf32_Dyn *vg_obj_dentry_by_type(vg_object *obj, vg_int type, vg_idx entidx)
{
	Elf32_Dyn	*dentry;

	CHECKDYN(VG_OBJ_DENTRY_BY_TYPE, NULL);

	if (!obj)
		VGERR(VG_OBJ_DENTRY_BY_TYPE, NULL, E_INVAL, NULL);

	if (!vg_obj_get_dyn(obj, 0))
		return (NULL);

	for (dentry = obj->elf.dyn.data; dentry->d_tag != DT_NULL; dentry++) {
		if (dentry->d_tag == type) {
			entidx -= (entidx?1:0);
			if (!entidx)
				return (dentry);
		}
	}
	return (NULL);
}

/*
 * vg_obj_dentry_by_val = return a dynamic entry giving its value
 */
Elf32_Dyn *vg_obj_dentry_by_val(vg_object *obj, vg_vaddr val, vg_idx entidx)
{
	Elf32_Dyn       *dentry;

	CHECKDYN(VG_OBJ_DENTRY_BY_VAL, NULL);

	if (!obj)
		VGERR(VG_OBJ_DENTRY_BY_VAL, NULL, E_INVAL, NULL);

	if (!vg_obj_get_dyn(obj, 0))
		return (NULL);

	for(dentry = obj->elf.dyn.data; dentry->d_tag != DT_NULL; dentry++) {
		if (dentry->d_un.d_val == val) {
			entidx -= (entidx?1:0);
			if (!entidx)
				return (dentry);
		}
	}
	return (NULL);
}

/*
 * vg_obj_dentry_by_idx = return a dynamic entry giving its index into the
 * dynamic table
 */
Elf32_Dyn *vg_obj_dentry_by_idx(vg_object *obj, vg_idx idx)
{
	CHECKDYN(VG_OBJ_DENTRY_BY_IDX, NULL);
	if (!obj)
		VGERR(VG_OBJ_DENTRY_BY_IDX, NULL, E_INVAL, NULL);

	if (!vg_obj_get_dyn(obj,0)) 
		return (NULL);

	if (idx < 0 || idx > obj->elf.dyn.entnum)
		VGERR(VG_OBJ_DENTRY_BY_IDX, "Incorrect index", 0, NULL);

	return ((idx? obj->elf.dyn.data+idx: obj->elf.dyn.data));	
}


/*
 * vg_obj_dentry_get/set_tag = return/change tag field of a dynamic
 * entry
 */
vg_int	vg_obj_dentry_get_tag(vg_object *obj, Elf32_Dyn *dentry)
{
	CHECKDYN(VG_OBJ_DENTRY_GET_TAG, ERR);
	if (!obj || !obj->elf.dyn.data || !dentry)
		VGERR(VG_OBJ_DENTRY_GET_TAG, NULL, E_INVAL, ERR);
	return (dentry->d_tag);
}

vg_int	vg_obj_dentry_set_tag(vg_object *obj, Elf32_Dyn *dentry, vg_int tag)
{
	CHECKDYN(VG_OBJ_DENTRY_SET_TAG, ERR);
	if (!obj || !obj->elf.dyn.data || !dentry)
		VGERR(VG_OBJ_DENTRY_SET_TAG, NULL, E_INVAL, ERR);
	dentry->d_tag = tag;
	SETOBJCH(obj);
	return (dentry->d_tag);
}

/*
 * vg_obj_dentry_get/set_ptr = return/change the value field of a dynamic
 * entry
 */
vg_vaddr vg_obj_dentry_get_ptr(vg_object *obj, Elf32_Dyn *dentry)
{
	CHECKDYN(VG_OBJ_DENTRY_GET_PTR, ERR);
	if (!obj || !obj->elf.dyn.data || !dentry)
		VGERR(VG_OBJ_DENTRY_GET_PTR, NULL, E_INVAL, ERR);
	return (dentry->d_un.d_ptr);
}

vg_int	vg_obj_dentry_set_ptr(vg_object *obj, Elf32_Dyn *dentry, vg_vaddr vaddr)
{
	CHECKDYN(VG_OBJ_DENTRY_SET_PTR, ERR);
	if (!obj || !obj->elf.dyn.data || !dentry)
		VGERR(VG_OBJ_DENTRY_SET_PTR, NULL, E_INVAL, ERR);
	dentry->d_un.d_ptr = vaddr;
	SETOBJCH(obj);
	return (dentry->d_un.d_ptr);
}


/*
 * Some dynamic entries as DT_NEDEED or DT_SONAME, have an index into the
 * dynstr section.
 * 
 * vg_obj_dentry_get/set_str = return/change the string
 */
vg_char	*vg_obj_dentry_get_str(vg_object *obj, Elf32_Dyn *dentry)
{
	CHECKDYN(VG_OBJ_DENTRY_GET_STR, NULL);
	if (!obj || !obj->elf.dyn.data || !dentry)
		VGERR(VG_OBJ_DENTRY_GET_STR, NULL, E_INVAL, NULL);

	/* get dynstr if not loaded */
	if (!vg_obj_get_dynstr(obj))
		return (NULL);

	/* only these */
	switch (dentry->d_tag) {
	case DT_NEEDED:	/* string table offset of needed lib 		*/
	case DT_SONAME:	/* string table offset of shared obj 		*/
	case DT_RPATH: 	/* string table offset of library search path 	*/
		return (obj->elf.dynstr.data+dentry->d_un.d_ptr);
		break;
	default:
		VGERR(VG_OBJ_DENTRY_GET_STR, "Incorrect d_tag value", 0, NULL);
	}

	return (NULL);
}

vg_char	*vg_obj_dentry_set_str(vg_object *obj, Elf32_Dyn *dentry, vg_char *str)
{
	CHECKDYN(VG_OBJ_DENTRY_SET_STR, NULL);
	if (!obj || !obj->elf.dyn.data || !dentry || !str) 
		VGERR(VG_OBJ_DENTRY_SET_STR, NULL, E_INVAL, NULL);

	if (!vg_obj_get_dynstr(obj))
		return (NULL);

	switch(dentry->d_tag) {
	case DT_NEEDED:	/* string table offset of needed lib		*/
	case DT_SONAME:	/* string table offset of shared obj		*/
	case DT_RPATH:	/* string table offset of library search path	*/
		/* mm, i haven't set any limit here, be careful :) */
		memcpy(obj->elf.dynstr.data+dentry->d_un.d_ptr, str, strlen(str));
		*(obj->elf.dynstr.data+dentry->d_un.d_ptr+strlen(str)+1) = '\0';
		SETOBJCH(obj);
		return (obj->elf.dynstr.data+dentry->d_un.d_ptr);
		break;
	default:
		VGERR(VG_OBJ_DENTRY_SET_STR, "Incorrect d_tag value", 0, NULL);
	}	

	return (NULL);
}

