/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
#include "libvg.h"

/*
 * vg_obj_get_dynsym = load dynsym section
 */
Elf32_Sym *vg_obj_get_dynsym(vg_object *obj, vg_uint *entnum)
{
	Elf32_Dyn 	*dentry = NULL;
	Elf32_Sym 	sentry, *tmp = NULL;
	vg_vaddr	s_addr = 0;
	vg_int		idx = 0, idy = 0;
	u_int num = 0;

	CHECKDYN(VG_OBJ_GET_DYNSYM, NULL);

	if (!obj)
		VGERR(VG_OBJ_GET_DYNSYM, NULL, E_INVAL, NULL);

	if (obj->elf.dynsym.data)
		goto end;

	if (!vg_obj_get_dyn(obj, &num))
		return (NULL);

	/* get symtab addr */
	if (!(dentry = vg_obj_dentry_by_type(obj, DT_SYMTAB, 0)))
		VGERR(VG_OBJ_GET_DYNSYM, "Dynamic symbol table not found", 0, NULL);

	/* check object */
	SET_OBJECT_VADDR(dentry->d_un.d_ptr, obj);

	if (OBJ_IS_MAPPED(obj)) {	/* local */
		obj->elf.dynsym.data = (Elf32_Sym*)RTOL((u_long)dentry->d_un.d_ptr, obj);
		tmp = obj->elf.dynsym.data;
		do {
			++idx;
			++tmp;
		} while(!(tmp->st_name&0xff0000));
	}
	else {
		/* this sucks ... */
		s_addr = dentry->d_un.d_ptr;
		do {
			vg_dread(&sentry, s_addr, sizeof(Elf32_Sym));
			++idx;
			s_addr+=sizeof(Elf32_Sym);
		} while(!(sentry.st_name&0xff0000));
		idx -= 1;
		obj->elf.dynsym.data = (Elf32_Sym*)calloc(idx, sizeof(Elf32_Sym));
		tmp = obj->elf.dynsym.data;
		
		idy = idx;
		s_addr = dentry->d_un.d_ptr;
		do {
			vg_dread(tmp, s_addr, sizeof(Elf32_Sym));
			s_addr +=sizeof(Elf32_Sym);
			++tmp;
		} while (--idy);
	}

	obj->elf.dynsym.vaddr = dentry->d_un.d_ptr;
	obj->elf.dynsym.offset = obj->elf.dynsym.vaddr - obj->text.start;
	obj->elf.dynsym.size = idx * sizeof(Elf32_Sym);
	obj->elf.dynsym.entsize = sizeof(Elf32_Sym);
	obj->elf.dynsym.entnum = obj->elf.dynsym.size / obj->elf.dynsym.entsize;
end:
	if (entnum) 
		*entnum = obj->elf.dynsym.entnum;
	return (obj->elf.dynsym.data);
}

/*
 * vg_obj_dynsym_get/set_vaddr = 
 */
vg_vaddr vg_obj_dynsym_get_vaddr(vg_object *obj)
{
	CHECKDYN(VG_OBJ_DYNSYM_GET_VADDR, ERR);
	if (!obj || !obj->elf.dynsym.data)
		VGERR(VG_OBJ_DYNSYM_GET_VADDR, NULL, E_INVAL, ERR);
	return (obj->elf.dynsym.vaddr);
}

vg_vaddr vg_obj_dynsym_set_vaddr(vg_object *obj, vg_vaddr vaddr)
{
	CHECKDYN(VG_OBJ_DYNSYM_SET_VADDR, ERR);
	if (!obj || !obj->elf.dynsym.data)
		VGERR(VG_OBJ_DYNSYM_SET_VADDR, NULL, E_INVAL, ERR);
	obj->elf.dynsym.vaddr = vaddr;
	return (obj->elf.dynsym.vaddr);
}

/*
 * vg_obj_dynsym_get/set_size = 
 */
vg_size	vg_obj_dynsym_get_size(vg_object *obj)
{
	CHECKDYN(VG_OBJ_DYNSYM_GET_SIZE, ERR);
	if (!obj || !obj->elf.dynsym.data)
		VGERR(VG_OBJ_DYNSYM_GET_SIZE, NULL, E_INVAL, ERR);
	return (obj->elf.dynsym.size);
}

vg_size	vg_obj_dynsym_set_size(vg_object *obj, vg_size size)
{
	CHECKDYN(VG_OBJ_DYNSYM_SET_SIZE, ERR);
	if (!obj || !obj->elf.dynsym.data)
		VGERR(VG_OBJ_DYNSYM_SET_SIZE, NULL, E_INVAL, ERR);
	obj->elf.dynsym.size = size;
	return (obj->elf.dynsym.size);
}

/*
 * vg_obj_dynsym_get/set_offset = 
 */
vg_off	vg_obj_dynsym_get_offset(vg_object *obj)
{
	CHECKDYN(VG_OBJ_DYNSYM_GET_OFFSET, ERR);
	if (!obj || !obj->elf.dynsym.data)
		VGERR(VG_OBJ_DYNSYM_GET_OFFSET, NULL, E_INVAL, ERR);
	return (obj->elf.dynsym.offset);	
}
	
vg_off	vg_obj_dynsym_set_offset(vg_object *obj, vg_off offset)
{
	CHECKDYN(VG_OBJ_DYNSYM_SET_OFFSET, ERR);
	if (!obj || !obj->elf.dynsym.data)
		VGERR(VG_OBJ_DYNSYM_SET_OFFSET, NULL, E_INVAL, ERR);
	obj->elf.dynsym.offset = offset;
	return (obj->elf.dynsym.offset);
}

/*
 * vg_obj_dynsym_get/set_entsize = 
 */
vg_size	vg_obj_dynsym_get_entsize(vg_object *obj)
{
	CHECKDYN(VG_OBJ_DYNSYM_GET_ENTSIZE, ERR);
	if (!obj || !obj->elf.dynsym.data)
		VGERR(VG_OBJ_DYNSYM_GET_ENTSIZE, NULL, E_INVAL, ERR);
	return (obj->elf.dynsym.entsize);
}

vg_size	vg_obj_dynsym_set_entsize(vg_object *obj, vg_size entsize)
{
	CHECKDYN(VG_OBJ_DYNSYM_SET_ENTSIZE, ERR);
	if (!obj || !obj->elf.dynsym.data)
		VGERR(VG_OBJ_DYNSYM_SET_ENTSIZE, NULL, E_INVAL, ERR);
	obj->elf.dynsym.entsize = entsize;
	return (obj->elf.dynsym.entsize);
}

/*
 * vg_obj_dynsym_get/set_entnum = 
 */
vg_idx	vg_obj_dynsym_get_entnum(vg_object *obj)
{
	CHECKDYN(VG_OBJ_DYNSYM_GET_ENTNUM, ERR);
	if (!obj || !obj->elf.dynsym.data)
		VGERR(VG_OBJ_DYNSYM_GET_ENTNUM, NULL, E_INVAL, ERR);
	return (obj->elf.dynsym.entnum);
}

vg_idx	vg_obj_dynsym_set_entnum(vg_object *obj, vg_idx entnum)
{
	CHECKDYN(VG_OBJ_DYNSYM_SET_ENTNUM, ERR);
	if (!obj || !obj->elf.dynsym.data)
		VGERR(VG_OBJ_DYNSYM_SET_ENTNUM, NULL, E_INVAL, ERR);
	obj->elf.dynsym.entnum = entnum;
	return (obj->elf.dynsym.entnum);
}

/*
 * vg_obj_dsym_by_idx = return a symbol entry giving its index into
 * the dynamic symbol table
 */
Elf32_Sym *vg_obj_dsym_by_idx(vg_object *obj, vg_idx idx)
{
	CHECKDYN(VG_OBJ_DSYM_BY_IDX, NULL);
	if (!obj)	
		VGERR(VG_OBJ_DSYM_BY_IDX, NULL, E_INVAL, NULL);

	if (!vg_obj_get_dynsym(obj, 0))
		return(NULL);

	if (idx<0 || idx > obj->elf.dynsym.entnum)
		VGERR(VG_OBJ_DSYM_BY_IDX, "Incorrect index", 0, NULL);

	return ((idx? obj->elf.dynsym.data+idx: obj->elf.dynsym.data));
}

/*
 * vg_obj_dsym_by_idxname = return a symbol entry giving its name string idx 
 * into the dynamic string table
 */
Elf32_Sym *vg_obj_dsym_by_idxname(vg_object *obj, vg_idx idxname)
{
	Elf32_Sym	*sym;
	vg_idx		idx;

	CHECKDYN(VG_OBJ_DSYM_BY_IDXNAME, NULL);
	if (!obj || idxname<0)
		VGERR(VG_OBJ_DSYM_BY_IDXNAME, NULL, E_INVAL, NULL);

	if (!(sym = vg_obj_get_dynsym(obj, 0)))
		return(NULL);

	for(idx = 0; idx < vg_obj_dynsym_get_entnum(obj); idx++, sym++) {
		if (sym->st_name == idxname)
			return (sym);
	}

	return (NULL);	
}

/*
 * vg_obj_dsym_by_name = return a symbol entry giving its name
 */
Elf32_Sym *vg_obj_dsym_by_name(vg_object *obj, vg_char *name)
{
	Elf32_Sym	*sym;
	vg_idx 		idx;

	CHECKDYN(VG_OBJ_DSYM_BY_NAME, NULL);
	if (!obj || !name) 
		VGERR(VG_OBJ_DSYM_BY_NAME, NULL, E_INVAL, NULL);

	if (!(sym = vg_obj_get_dynsym(obj, 0)))
		return (NULL);

	for (idx = 0; idx < vg_obj_dynsym_get_entnum(obj); idx++, sym++) {
		if (!strcmp(name, vg_obj_dsym_get_strname(obj, sym)))
			return (sym);
	}
	return (NULL);
}	

/* 
 * vg_obj_dsym_by_value = return a symbol entry giving its value
 */
Elf32_Sym *vg_obj_dsym_by_value(vg_object *obj, vg_vaddr value)
{
	Elf32_Sym	*sym;
	vg_idx 		idx;

	CHECKDYN(VG_OBJ_DSYM_BY_VALUE, NULL);
	if (!obj || !value)
		VGERR(VG_OBJ_DSYM_BY_VALUE, NULL, E_INVAL, NULL);

	if (!(sym = vg_obj_get_dynsym(obj, 0)))
		return (NULL);

	for (idx = 0; idx < vg_obj_dynsym_get_entnum(obj); idx++, sym++) {
		if (sym->st_value == value)
			return (sym);
	}
	return (NULL);
}


/*
 * vg_obj_dsym_get/set_idxname = 
 */
Elf32_Word vg_obj_dsym_get_idxname(vg_object *obj, Elf32_Sym *sym)
{
	CHECKDYN(VG_OBJ_DSYM_GET_IDXNAME, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_GET_IDXNAME, NULL, E_INVAL, ERR);
	return (sym->st_name);
}

Elf32_Word vg_obj_dsym_set_idxname(vg_object *obj, Elf32_Sym *sym, Elf32_Word idxname)
{
	CHECKDYN(VG_OBJ_DSYM_SET_IDXNAME, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_SET_IDXNAME, NULL, E_INVAL, ERR);
	sym->st_name = idxname;
	SETOBJCH(obj);
	return (sym->st_name);
}

/*
 * vg_obj_dsym_get/set_strname = ptr to symbol name into string table
 */
vg_char	*vg_obj_dsym_get_strname(vg_object *obj, Elf32_Sym *sym)
{
	CHECKDYN(VG_OBJ_DSYM_GET_STRNAME, NULL);
	if (!obj || !sym)
		VGERR(VG_OBJ_DSYM_GET_STRNAME, NULL, E_INVAL, NULL);
	if (!sym->st_name)
		VGERR(VG_OBJ_DSYM_GET_STRNAME, "Incorrect name index", 0, NULL);
	if (!vg_obj_get_dynstr(obj)) 
		return (NULL);
	return (obj->elf.dynstr.data + sym->st_name);
}

vg_char	*vg_obj_dsym_set_strname(vg_object *obj, Elf32_Sym *sym, vg_char *strname)
{
	CHECKDYN(VG_OBJ_DSYM_SET_STRNAME, NULL);
	if (!obj || !sym || !strname) 
		VGERR(VG_OBJ_DSYM_SET_STRNAME, NULL, E_INVAL, NULL);
	if (!vg_obj_get_dynsym(obj,0))
		return (NULL);
	if (!vg_obj_get_dynstr(obj))
		return (NULL);				    /* no limits */
	strncpy(obj->elf.dynstr.data+sym->st_name, strname, strlen(strname)+1);
	SETOBJCH(obj);
	return (obj->elf.dynstr.data + sym->st_name);
}

/*
 * vg_obj_dsym_get/set_value = 
 */
Elf32_Addr vg_obj_dsym_get_value(vg_object *obj, Elf32_Sym *sym)
{
	CHECKDYN(VG_OBJ_DSYM_GET_VALUE, ERR);
	if (!obj || !obj->elf.dynsym.data|| !sym)
		VGERR(VG_OBJ_DSYM_GET_VALUE, NULL, E_INVAL, ERR);
	return (sym->st_value);
}

Elf32_Addr vg_obj_dsym_set_value(vg_object *obj, Elf32_Sym *sym, Elf32_Addr value)
{
	CHECKDYN(VG_OBJ_DSYM_SET_VALUE, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_SET_VALUE, NULL, E_INVAL, ERR);
	sym->st_value = value;
	SETOBJCH(obj);
	return (sym->st_value);
}

/*
 * vg_obj_dsym_get/set_size = 
 */
Elf32_Word vg_obj_dsym_get_size(vg_object *obj, Elf32_Sym *sym)
{
	CHECKDYN(VG_OBJ_DSYM_GET_SIZE, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_GET_SIZE, NULL, E_INVAL, ERR);
	return (sym->st_size);
}

Elf32_Word vg_obj_dsym_set_size(vg_object *obj, Elf32_Sym *sym, Elf32_Word size)
{
	CHECKDYN(VG_OBJ_DSYM_SET_SIZE, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_SET_SIZE, NULL, E_INVAL, ERR);
	sym->st_size = size;
	SETOBJCH(obj);
	return (sym->st_size);
}

/*	
 * vg_obj_dsym_get/set_info = 
 */
vg_uchar vg_obj_dsym_get_info(vg_object *obj, Elf32_Sym *sym)
{
	CHECKDYN(VG_OBJ_DSYM_GET_INFO, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_GET_INFO, NULL, E_INVAL, ERR);
	return (sym->st_info);
}

vg_uchar vg_obj_dsym_set_info(vg_object *obj, Elf32_Sym *sym, vg_uchar info)
{
	CHECKDYN(VG_OBJ_DSYM_SET_INFO, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_SET_INFO, NULL, E_INVAL, ERR);
	sym->st_info = info;
	SETOBJCH(obj);
	return (sym->st_info);
}

vg_uchar vg_obj_dsym_get_info_bind(vg_object *obj, Elf32_Sym *sym)
{
	CHECKDYN(VG_OBJ_DSYM_GET_INFO_BIND, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_GET_INFO_BIND, NULL, E_INVAL, ERR);
	return (ELF32_ST_BIND(sym->st_info));
}

vg_uchar vg_obj_dsym_set_info_bind(vg_object *obj, Elf32_Sym *sym,  vg_uchar bind)
{
	CHECKDYN(VG_OBJ_DSYM_SET_INFO_BIND, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_SET_INFO_BIND, NULL, E_INVAL, ERR);
	sym->st_info = VG_SET_ST_BIND(sym->st_info, bind);
	SETOBJCH(obj);
	return (ELF32_ST_BIND(sym->st_info));
}

vg_uchar vg_obj_dsym_get_info_type(vg_object *obj, Elf32_Sym *sym)
{
	CHECKDYN(VG_OBJ_DSYM_GET_INFO_TYPE, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_GET_INFO_TYPE, NULL, E_INVAL, ERR);
	return (ELF32_ST_TYPE(sym->st_info));
}

vg_uchar vg_obj_dsym_set_info_type(vg_object *obj, Elf32_Sym *sym, vg_uchar type)
{	
	CHECKDYN(VG_OBJ_DSYM_SET_INFO_TYPE, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_SET_INFO_TYPE, NULL, E_INVAL, ERR);
	sym->st_info = VG_SET_ST_TYPE(sym->st_info, type);
	SETOBJCH(obj);
	return (ELF32_ST_TYPE(sym->st_info));
}

/*
 * vg_obj_dsym_get/set_other = 
 */
vg_uchar vg_obj_dsym_get_other(vg_object *obj, Elf32_Sym *sym)
{
	CHECKDYN(VG_OBJ_DSYM_GET_OTHER, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_GET_OTHER, NULL, E_INVAL, ERR);
	return (sym->st_other);
}

vg_uchar vg_obj_dsym_set_other(vg_object *obj, Elf32_Sym *sym, vg_uchar other)
{
	CHECKDYN(VG_OBJ_DSYM_SET_OTHER, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_SET_OTHER, NULL, E_INVAL, ERR);
	sym->st_other = other;
	SETOBJCH(obj);
	return (sym->st_other);
}

/*
 * vg_obj_dsym_get/set_shndx = 
 */
Elf32_Half vg_obj_dsym_get_shndx(vg_object *obj, Elf32_Sym *sym)
{
	CHECKDYN(VG_OBJ_DSYM_GET_SHNDX, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_GET_SHNDX, NULL, E_INVAL, ERR);
	return (sym->st_shndx);
}

Elf32_Half vg_obj_dsym_set_shndx(vg_object *obj, Elf32_Sym *sym, Elf32_Half shndx)
{
	CHECKDYN(VG_OBJ_DSYM_SET_SHNDX, ERR);
	if (!obj || !obj->elf.dynsym.data || !sym)
		VGERR(VG_OBJ_DSYM_SET_SHNDX, NULL, E_INVAL, ERR);
	sym->st_shndx = shndx;
	SETOBJCH(obj);
	return (sym->st_shndx);
}

