/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
#include "libvg.h"

/*
 * vg_obj_get_got = load got section
 */
vg_vaddr *vg_obj_get_got(vg_object *obj, vg_int *entnum)
{
	Elf32_Dyn	*got;
	Elf32_Rel	*rentry;
	vg_idx		idx;

	CHECKDYN(VG_OBJ_GET_GOT, NULL);
	if (!obj)
		VGERR(VG_OBJ_GET_GOT, NULL, E_INVAL, NULL);

	if (obj->elf.got.data)
		goto end;

	if (!vg_obj_get_reldyn(obj, NULL))
		return (NULL);

	if (!vg_obj_get_relplt(obj, NULL))
		return (NULL);

	if (!(got = vg_obj_dentry_by_type(obj, DT_PLTGOT, 0))) 
		VGERR(VG_OBJ_GET_GOT, "GOT section not found", 0, NULL);

	/* got size:
	   x86 		[ R_386_GLOB_DATA + R_386_JMP_SLOT ] entries, for now
	*/
#if defined(__i386__)
	for (idx = 0; idx < vg_obj_reldyn_get_entnum(obj); idx++) {
		rentry = vg_obj_reldyn_rentry_by_idx(obj, idx);
		if (rentry && vg_obj_rentry_get_info_type(obj, rentry) == R_386_GLOB_DAT)
			obj->elf.got.size += 1;
	}
	obj->elf.got.size = (vg_obj_relplt_get_entnum(obj) + 3) * sizeof(Elf32_Addr);

#if defined(__OpenBSD__)
	if (obj->flags & OBJ_SO) got->d_un.d_ptr -= 0x20000000;
#endif
#endif

	SET_OBJECT_VADDR(got->d_un.d_ptr, obj);

	if (obj->flags & OBJ_MAPPED) {
		obj->elf.got.data = (u_long*)RTOL((u_long)got->d_un.d_ptr, obj);
	}
	else {
		S_ALLOC(obj->elf.got.data, obj->elf.got.size, NULL, VG_OBJ_GET_GOT, CLEAN);
		if (vg_dread(obj->elf.got.data, got->d_un.d_ptr,
				obj->elf.got.size) != obj->elf.got.size)
			return (NULL);
	}
	obj->elf.got.vaddr = got->d_un.d_ptr;
	obj->elf.got.offset = 0x0;
	obj->elf.got.entsize = sizeof(Elf32_Addr);
	obj->elf.got.entnum = obj->elf.got.size / sizeof(Elf32_Addr);
end:
	if (entnum)
		*entnum = obj->elf.got.entnum;
	return (obj->elf.got.data);
}

/*
 * vg_obj_got_get/set_vaddr = get/set the starting address of the 
 * got section
 */
vg_vaddr vg_obj_got_get_vaddr(vg_object *obj)
{
	CHECKDYN(VG_OBJ_GOT_GET_VADDR, ERR);
	if (!obj || !obj->elf.got.data)
		VGERR(VG_OBJ_GOT_GET_VADDR, NULL, E_INVAL, ERR);
	return (obj->elf.got.vaddr);	
}

vg_vaddr vg_obj_get_set_vaddr(vg_object *obj, vg_vaddr vaddr)
{
	CHECKDYN(VG_OBJ_GOT_SET_VADDR, ERR);
	if (!obj || !obj->elf.got.data)
		VGERR(VG_OBJ_GOT_SET_VADDR, NULL, E_INVAL, ERR);
	obj->elf.got.vaddr = vaddr;
	return (obj->elf.got.vaddr);
}

/*
 * vg_obj_got_get/set_offset = get/set the offset of the got 
 * section into the object
 */
vg_off	vg_obj_got_get_offset(vg_object *obj)
{
	CHECKDYN(VG_OBJ_GOT_GET_OFFSET, ERR);
	if (!obj || !obj->elf.got.data)
		VGERR(VG_OBJ_GOT_GET_OFFSET, NULL, E_INVAL, ERR);
	return (obj->elf.got.offset);
}

vg_off	vg_obj_got_set_offset(vg_object *obj, vg_off offset)
{
	CHECKDYN(VG_OBJ_GOT_SET_OFFSET, ERR);
	if (!obj || !obj->elf.got.data)
		VGERR(VG_OBJ_GOT_SET_OFFSET, NULL, E_INVAL, ERR);
	obj->elf.got.offset = offset;
	return (obj->elf.got.offset);
}

/*
 * vg_obj_got_get/set_size = get/set the got size
 */
vg_size	vg_obj_got_get_size(vg_object *obj)
{
	CHECKDYN(VG_OBJ_GOT_GET_SIZE, ERR);
	if (!obj || !obj->elf.got.data)
		VGERR(VG_OBJ_GOT_GET_SIZE, NULL, E_INVAL, ERR);
	return (obj->elf.got.size);
}

vg_size	vg_obj_got_set_size(vg_object *obj, vg_size size)
{
	CHECKDYN(VG_OBJ_GOT_SET_SIZE, ERR);
	if (!obj || !obj->elf.got.data)
		VGERR(VG_OBJ_GOT_SET_SIZE, NULL, E_INVAL, ERR);
	obj->elf.got.size = size;
	return (obj->elf.got.size);
}

/*
 * vg_obj_got_get/set_entsize = get/set a got entry size
 */
vg_size	vg_obj_got_get_entsize(vg_object *obj)
{
	CHECKDYN(VG_OBJ_GOT_GET_ENTSIZE, ERR);
	if (!obj || !obj->elf.got.data)
		VGERR(VG_OBJ_GOT_GET_ENTSIZE, NULL, E_INVAL, ERR);
	return (obj->elf.got.entsize);
}

vg_size	vg_obj_got_set_entsize(vg_object *obj, vg_size entsize)
{
	CHECKDYN(VG_OBJ_GOT_SET_ENTSIZE, ERR);
	if (!obj || !obj->elf.got.data)
		VGERR(VG_OBJ_GOT_SET_ENTSIZE, NULL, E_INVAL, ERR);
	obj->elf.got.entsize = entsize;
	return (obj->elf.got.entsize);
}

/*
 * vg_obj_got_get/set_entnum = get/set the number of entries into
 * the got 
 */
vg_idx	vg_obj_got_get_entnum(vg_object *obj)
{
	CHECKDYN(VG_OBJ_GOT_GET_ENTNUM, ERR);
	if (!obj || !obj->elf.got.data) 
		VGERR(VG_OBJ_GOT_GET_ENTNUM, NULL, E_INVAL, ERR);
	return (obj->elf.got.entnum);
}
	
vg_idx	vg_obj_got_set_entnum(vg_object *obj, vg_idx entnum)
{
	CHECKDYN(VG_OBJ_GOT_SET_ENTNUM, ERR);
	if (!obj || !obj->elf.got.data)
		VGERR(VG_OBJ_GOT_SET_ENTNUM, NULL, E_INVAL, ERR);
	obj->elf.got.entnum = entnum;
	return (obj->elf.got.entnum);
}

/*
 * vg_obj_gentry_by_idx = return a gotentry giving its index
 * into the table.
 */
vg_int	vg_obj_gentry_by_idx(vg_object *obj, vg_idx idx, vg_gotentry *gent)
{
	CHECKDYN(VG_OBJ_GENTRY_BY_IDX, ERR);
	if (!obj || !gent)
		VGERR(VG_OBJ_GENTRY_BY_IDX, NULL, E_INVAL, ERR);

	if (!vg_obj_get_got(obj, 0))
		return (ERR);

	if (idx < 0 || idx > obj->elf.got.entnum)
		VGERR(VG_OBJ_GENTRY_BY_IDX, "Incorrect index", 0, ERR);

	gent->vaddr = (idx? obj->elf.got.vaddr+(sizeof(vg_vaddr)*idx): obj->elf.got.vaddr);
	vg_read(&gent->value, gent->vaddr, sizeof(vg_vaddr));

	return (OK);
}

/*
 * vg_obj_gentry_by_sym = get a gotentry giving a symbol name.
 * It check all relplt entries in search of the correct dynamic symbol
 * entry. If a match is found (names are equal), gotentry will be 
 * filled with the starting address of the got entry and its value.
 */
vg_int	vg_obj_gentry_by_sym(vg_object *obj, vg_char *sname, vg_gotentry *gent)
{	
	Elf32_Sym	*dsym;
	Elf32_Rel	*rentry;
	vg_idx		idx;

	if (!obj || !sname || !gent)
		VGERR(VG_OBJ_GENTRY_BY_SYM, NULL, E_INVAL, ERR);

	for (idx = 0; idx < vg_obj_relplt_get_entnum(obj); idx++) {
		rentry = vg_obj_relplt_rentry_by_idx(obj, idx);
		if (!rentry)
			continue;

		dsym = vg_obj_dsym_by_idx(obj,
			vg_obj_rentry_get_info_symidx(obj, rentry));
		if (!dsym)
			continue;

		if (strcmp(sname, vg_obj_dsym_get_strname(obj, dsym)))
			continue;
	
		gent->vaddr = vg_obj_rentry_get_offset(obj, rentry);
		SET_OBJECT_VADDR(gent->vaddr, obj);
		vg_read(&gent->value, gent->vaddr, sizeof(vg_vaddr));

		return (OK);
	}

	return (ERR);
}


