/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
#include "libvg.h"

/*
 * vg_obj_get_hash = load hash table
 *
 * @obj:
 * #ret:
 */
vg_vaddr *vg_obj_get_hash(vg_object *obj)
{
	Elf32_Dyn	*hash_table;

	CHECKDYN(VG_OBJ_GET_HASH, NULL);
	if (!obj)
		VGERR(VG_OBJ_GET_HASH, NULL, E_INVAL, NULL);

	if (obj->elf.hash.data)
		goto end;

	if (!vg_obj_get_dyn(obj, NULL))
		return (NULL);

	if (!(hash_table = vg_obj_dentry_by_type(obj, DT_HASH, 0)))
		VGERR(VG_OBJ_GET_HASH, "HASH section not found", 0, NULL);

	SET_OBJECT_VADDR(hash_table->d_un.d_ptr, obj);

	/* get bucket and chains vectors size */
	if (vg_dread(&obj->elf.hash.nbuckets, hash_table->d_un.d_ptr,
				sizeof(Elf32_Addr)) != sizeof(Elf32_Addr))
		return (NULL);
	
	if (vg_dread(&obj->elf.hash.nchains, hash_table->d_un.d_ptr+4, 
				sizeof(Elf32_Addr)) != sizeof(Elf32_Addr))
		return (NULL);

	/* tamao => numero entradas chains * int (4bytes) + n entradas bu * int */
	obj->elf.hash.size = (obj->elf.hash.nchains+obj->elf.hash.nbuckets+2) * sizeof(Elf32_Addr);
	obj->elf.hash.vaddr = (u_long)hash_table->d_un.d_ptr;

	if (OBJ_IS_MAPPED(obj)) {
		obj->elf.hash.data = (u_long*)RTOL((u_long)hash_table->d_un.d_ptr, obj);
	}
	else {
		S_ALLOC(obj->elf.hash.data, obj->elf.hash.size, NULL, VG_OBJ_GET_HASH, CLEAN);
		if (vg_dread(obj->elf.hash.data, hash_table->d_un.d_ptr, 
					obj->elf.hash.size) != obj->elf.hash.size)
			VGERR2(VG_OBJ_GET_HASH, 0,0, NULL, obj->elf.hash.data);			
	}
	obj->elf.hash.buckets = obj->elf.hash.data+(sizeof(Elf32_Addr)*2);
	obj->elf.hash.chains = (obj->elf.hash.data+(sizeof(Elf32_Addr)*2))+ 
				(obj->elf.hash.nbuckets*sizeof(Elf32_Addr));
end:
	return (obj->elf.hash.data);
}

/*
 * vg_obj_hash_get/set_vaddr = get/set the starting address of
 * the hash table
 *
 * @obj:
 * @vaddr:
 */
vg_vaddr vg_obj_hash_get_vaddr(vg_object *obj)
{
	CHECKDYN(VG_OBJ_HASH_GET_VADDR, ERR);
	if (!obj || !obj->elf.hash.data)
		VGERR(VG_OBJ_HASH_GET_VADDR, NULL, E_INVAL, ERR);
	return (obj->elf.hash.vaddr);
}
	
vg_vaddr vg_obj_hash_set_vaddr(vg_object *obj, vg_vaddr vaddr)
{
	CHECKDYN(VG_OBJ_HASH_SET_VADDR, ERR);
	if (!obj || !obj->elf.hash.data)
		VGERR(VG_OBJ_HASH_GET_VADDR, NULL, E_INVAL, ERR);
	obj->elf.hash.vaddr = vaddr;
	return (obj->elf.hash.vaddr);
}

/*
 * vg_obj_hash_get/set_offset = get/set the valid offset to the
 * hash table into an object
 *
 * @obj:
 * @offset:
 */
vg_off	vg_obj_hash_get_offset(vg_object *obj)
{	
	CHECKDYN(VG_OBJ_HASH_GET_OFFSET, ERR);
	if (!obj || !obj->elf.hash.data)
		VGERR(VG_OBJ_HASH_GET_OFFSET, NULL, E_INVAL, ERR);
	return (obj->elf.hash.offset);
}

vg_off	vg_obj_hash_set_offset(vg_object *obj, vg_off offset)
{
	CHECKDYN(VG_OBJ_HASH_SET_OFFSET, ERR);
	if (!obj || !obj->elf.hash.data)
		VGERR(VG_OBJ_HASH_SET_OFFSET, NULL, E_INVAL, ERR);
	obj->elf.hash.offset = offset;
	return (obj->elf.hash.offset);
}

/*
 * vg_obj_hash_get/set_size = get/set the size of the hash
 * table 
 *
 * @obj:
 * @size: new size
 */
vg_size	vg_obj_hash_get_size(vg_object *obj)
{
	CHECKDYN(VG_OBJ_HASH_GET_SIZE, ERR);
	if (!obj || !obj->elf.hash.data)
		VGERR(VG_OBJ_HASH_GET_SIZE, NULL, E_INVAL, ERR);
	return (obj->elf.hash.size);
}

vg_size	vg_obj_hash_set_size(vg_object *obj, vg_size size)
{
	CHECKDYN(VG_OBJ_HASH_SET_SIZE, ERR);
	if (!obj || !obj->elf.hash.data)
		VGERR(VG_OBJ_HASH_SET_SIZE, NULL, E_INVAL, ERR);
	obj->elf.hash.size = size;
	return (obj->elf.hash.size);
}


/*
 * vg_get_sym_hash = elf specification hashing function
 *
 * @name: symbol name
 * #ret: index
 */
vg_vaddr vg_symbol_hash(const char *name)
{
	u_long  h = 0, g;

	while (*name) {
		h = (h << 4) + *name++;
		if ((g = h & 0xf0000000))
			h ^= g >> 24;
		h &= ~g;
	}
	return (h);
}

/*
 * get_sym_in_hash = locate a symbol into the hash table of an object 
 */
Elf32_Sym *get_sym_in_hash(vg_object *obj, vg_char *name, vg_idx *sym_idx, vg_idx hidx)
{
	Elf32_Sym	*s;
        vg_idx          ndx;

	if (!vg_obj_get_hash(obj))
		return (NULL);

	if (!vg_obj_get_dynsym(obj, 0))
		return (NULL);

	if (!vg_obj_get_dynstr(obj))
		return (NULL);

	for (ndx = *(u_long*)(obj->elf.hash.buckets + ( (hidx % obj->elf.hash.nbuckets)*4) );
				ndx; ndx = *(u_long*)(obj->elf.hash.chains+ndx*4)) {
		/* get sym */
		s = (Elf32_Sym*)(obj->elf.dynsym.data+ndx);
		/* check symbols with string only */
		if (s->st_name && !strncmp(obj->elf.dynstr.data+s->st_name, name, strlen(name))) {
			*sym_idx = ndx;
			return (s);
		}
	}

	return (NULL);
}


