/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE ON YOUR OWN RISK

*/
#ifndef	__LIBVG_MACROS_H
#define	__LIBVG_MACROS_H

extern	vg_vaddr proc_base;
extern	vg_proc	 *cur_proc;

char	vg_err_msg[256];
char	*tmp_base;

#define INTERNAL_ERR		strerror(errno)



/***************************************************************************
 *		Alignments, offsets, vaddrs...				   *
 ***************************************************************************/


	/*
	(x) vaddr to be aligned
	(a) page/other size
	*/
#define ROUND_DOWN(x,a)		((x) & (~((a)-1)))
#define ROUND_UP(x,a)		(((x) + ((a)-1)) & (~((a)-1)))
#define	ALIGN(x,a)		ROUND_DOWN(x,a)
#define PAGE_ALIGN(x)		ALIGN(x, PAGE_SIZE)


	/* 
	(o) offset
	(b) base 
	*/
#define	SET_VADDR(o,b)		(o += (o<b? b: 0))
#define	GET_VADDR(o,b)		((o<b? o+b: o))


	/*
	(b) base
	(v) vaddr
	*/
#define	SET_OFFSET(v,b)		(v -= (v>b? b: 0))
#define	GET_OFFSET(v,b)		((v>b? v-b: v))


	/*
	(o  ) offset 
	(obj) vg_object ptr
	*/
#define	SET_TEXTSEG_VADDR(o, obj)	SET_VADDR(o, obj->text.start)
#define	SET_DATASEG_VADDR(o, obj)	SET_VADDR(o, obj->data.start)
#define	SET_HEAPSEG_VADDR(o, obj)	SET_VADDR(o, obj->heap.start)

#define GET_TEXTSEG_VADDR(o, obj)	GET_VADDR(o, obj->text.start)
#define GET_DATASEG_VADDR(o, obj)	GET_VADDR(o, obj->data.start)
#define GET_HEAPSEG_VADDR(o, obj)	GET_VADDR(o, obj->heap.start)


	/*
	(v  ) vaddr
	(obj) vg_object ptr
	*/
#define	SET_TEXTSEG_OFFSET(v, obj)	SET_OFFSET(v, obj->text.start);
#define	SET_DATASEG_OFFSET(v, obj)	SET_OFFSET(v, obj->data.start);
#define	SET_HEAPSEG_OFFSET(v, obj)	SET_OFFSET(v, obj->heap.start);

#define GET_TEXTSEG_OFFSET(v, obj)	GET_OFFSET(v, obj->text.start);
#define GET_DATASEG_OFFSET(v, obj)	GET_OFFSET(v, obj->data.start);
#define GET_HEAPSEG_OFFSET(v, obj)	GET_OFFSET(v, obj->heap.start);


	/*
	(v  ) vaddr
	(obj) vg_object ptr
	*/
#define	CHECK_TEXTSEG_VADDR(v, obj)	\
	((v >= obj->text.start && v <= obj->text.end)? 1: 0)

#define	CHECK_DATASEG_VADDR(v, obj)	\
	((v >= obj->data.start && v <= obj->data.end)? 2: 0)

#define	CHECK_HEAPSEG_VADDR(v, obj)	\
	((v >= obj->heap.start && v <= obj->heap.end)? 3: 0)


	/*
	(o  ) offset
	(obj) vg_object ptr
	*/
#define	CHECK_TEXTSEG_OFFSET(o, obj)	\
	(CHECK_TEXTSEG_VADDR(GET_TEXTSEG_VADDR(o),obj))

#define	CHECK_DATASEG_OFFSET(o, obj)	\
	(CHECK_DATASEG_VADDR(GET_DATASEG_VADDR(o),obj))

#define CHECK_HEAPSEG_OFFSET(o, obj)	\
	(CHECK_HEAPSEG_VADDR(GET_HEAPSEG_VADDR(o),obj))


	/*
	(o  ) offset/vaddr (value)
	(obj) object 
	*/
#define	SET_OBJECT_VADDR(o, obj)	\
do {	\
	if (CHECK_TEXTSEG_VADDR(GET_TEXTSEG_VADDR(o,obj),obj))	\
		SET_TEXTSEG_VADDR(o, obj);	\
	if (CHECK_DATASEG_VADDR(GET_DATASEG_VADDR(o,obj),obj))	\
		SET_DATASEG_VADDR(o, obj);	\
	if (CHECK_HEAPSEG_VADDR(GET_HEAPSEG_VADDR(o,obj),obj))	\
		SET_HEAPSEG_VADDR(o, obj);	\
} while(0)

/*

#define CHECK_OBJECT_OFFSET(v, obj)	\
		((CHECK_TEXTSEG_VADDR(v,obj) ||	\
		  CHECK_DATASEG_VADDR(v,obj) ||	\
		  CHECK_HEAPSEG_VADDR(v,obj))? 1: 0)
*/


#define	CHECK_OBJECT_VADDR(o, obj)	\
		((CHECK_TEXTSEG_VADDR(GET_TEXTSEG_VADDR(o),obj) ||  \
	 	  CHECK_DATASEG_VADDR(GET_DATASEG_VADDR(o),obj) ||  \
		  CHECK_HEAPSEG_VADDR(GET_HEAPSEG_VADDR(o),obj))? 1: 0)


/***************************************************************************
 *                                                                         *
 ***************************************************************************/

	/*
	 Check if process (a) has been dinamically linked
 	
	(i) function id
	(e) err code
	*/
#define CHECKDYN(i, e) \
do {	\
	if (!((cur_proc)->flags & PROC_WITH_RTLD))	\
		VGERR(i, "Statically linked process", 0, e);	\
} while(0)


//  	set object (x) as changed

#define	SETOBJCH(x)	((x->flags & OBJ_MAPPED)?x->flags|=OBJ_CHANGED:0)

//	set object descriptor (x) as changed

#define	SETOBJDCH(x)	\
 	((x->flags & PROC_OBJD_CHANGED)? 0: (x->flags |= PROC_OBJD_CHANGED))

#define OBJ_IS_MAPPED(x)	((x)->flags & OBJ_MAPPED)
#define	OBJ_IS_MO(x)		((x)->flags & OBJ_MO)
#define	OBJ_IS_SO(x)		((x)->flags & OBJ_SO)

#define	is_text_segment(x)	\
	(((x)->p_type == PT_LOAD) && 	\
	(((x)->p_offset == 0) || ((x)->p_flags == (PF_X|PF_R))))

#define	is_data_segment(x)	\
	(((x)->p_type == PT_LOAD && \
	(((x)->p_flags == (PF_R+PF_W+PF_W)) || ((x)->p_flags == PF_R)))


/***************************************************************************
 *      Macros for virtual address translation                             *
 ***************************************************************************/


	/* translate a local vaddr to a remote vaddr */
	/*
	(o) vg_object ptr
	(v) value to be translated = can be an offset or a vaddr
	*/
#define	T_LTOR(v,o)	\
	(o->text.start + ((char*)o->text.ptr - (unsigned long)v)

#define	D_LTOR(v,o)	\
	(o->data.start + ((char*)o->data.ptr - (unsigned long)v)

#define	H_LTOR(v,o)	\
	(o->heap.start + ((char*)o->heap.ptr - (unsigned long)v)


	/* translate a remote vaddr to a local vaddr */
	/*
	(o) vg_object ptr
	(v) value = offset/vaddr
	*/
#define T_RTOL(v,o)	\
	((char*)o->text.ptr + ((unsigned long)v - (o->text.start)))

#define	D_RTOL(v,o)	\
	((char*)o->data.ptr + ((unsigned long)v - (o->data.start)))

#define	H_RTOL(v,o)	\
	((char*)o->heap.ptr + ((unsigned long)v - (o->heap.start)))


	/* segment-independent translator : remote to local */
#define	RTOL(rval, obj)	\
(CHECK_TEXTSEG_VADDR(GET_TEXTSEG_VADDR(rval, obj), obj)? T_RTOL(rval, obj):	\
(CHECK_DATASEG_VADDR(GET_DATASEG_VADDR(rval, obj), obj)? D_RTOL(rval, obj):	\
(CHECK_HEAPSEG_VADDR(GET_HEAPSEG_VADDR(rval, obj), obj)? H_RTOL(rval, obj):	\
	0)))

	/* segment-independent translator : local to remote */
#define	LTOR(lval, obj)	\
(CHECK_TEXTSEG_VADDR(GET_TEXTSEG_VADDR(rval, obj), obj)? T_LTOR(rval, obj):	\
(CHECK_DATASEG_VADDR(GET_DATASEG_VADDR(rval, obj), obj)? D_LTOR(rval, obj):	\
(CHECK_HEAPSEG_VADDR(GET_HEAPSEG_VADDR(rval, obj), obj)? H_LTOR(rval, obj):	\
	0)))

	/*
	(x) variable, normally pltentry->got_vaddr
	(y) plt entry, normally pltentry->code
	*/
#define	PLT_GET_GOT_VADDR(x, y)		(x = *(unsigned long*)(y+2))
#define	PLT_SET_GOT_VADDR(x, y)		(*(unsigned long*)(y+2) = (unsigned long)x)

#define	PLT_GET_GOT_IDX(x, y)		(x = *(unsigned long*)(y+7))
#define	PLT_SET_GOT_IDX(x, y)		(*(unsigned long*)(y+7) = (unsigned long)x)



/***************************************************************************	
 *	Secure memory allocators			   		   *
 ***************************************************************************/

	/*
	(x) ptr		(e) err code
	(y) size	(c) clean flag 
			(i) function id
	*/
#define	S_ALLOC(x,y, e,i,c)	\
do {	\
	if (y == 0)	\
		VGERR(i, "(S_ALLOC) Incorrect size", 0, e);	\
	if (!(x = malloc(y)))	\
		VGERR(i, INTERNAL_ERR, 0, e);		\
	if (c == CLEAN)		\
		memset(x, 0, y);	\
} while(0)

	/*
	(x) ptr		(e) err code				
	(y) num		(c) clean flag
	(z) size	(i) function id
	*/
#define	S_CALLOC(x,y,z, e,i,c)	\
do {	\
	if (y == 0 || z == 0)	\
		VGERR(i, "(S_CALLOC) Incorrect size", 0, e);	\
	if (!(x = calloc(y, z)))	\
		VGERR(i, INTERNAL_ERR, 0, e);	\
} while(0)

	/*
	(x) ptr		(e) err code
	(y) size	(c) clean flag
			(i) function id
	*/
#define	S_REALLOC(x,y, e,i,c)	\
do {	\
	if (y == 0)	\
		VGERR(i, "(S_REALLOC) Incorrect size", 0, e);	\
	if (!(x = realloc(x, y)))	\
		VGERR(i, INTERNAL_ERR, 0, e);	\
	if (c == CLEAN)	\
		memset(x, 0, y);	\
} while(0)

	/*
	(x) ptr		(c) clean flag
	(y) size
	*/
#define	S_FREE(x,y, c)	\
do {	\
	if (c == CLEAN && y != 0)	\
		memset(x, 0, y);	\
	free(x);	\
} while(0)


/***************************************************************************
 *	Secure anonymous memory mapping                    		   *
 ***************************************************************************/

	/*
	(x) mem ptr	(e) err code	
	(y) mem size	(i) function id
	*/
#define S_MAP(x,y, e,i)	\
do {	\
	if (y == 0)	\
		VGERR(i, "(S_MAP) Incorrect size", 0, e);	\
	x = mmap(0, y, PROT_EXEC|PROT_READ|PROT_WRITE,	\
			MAP_PRIVATE|MAP_ANON,-1,0);	\
	if (x == NULL)	\
		VGERR(i, INTERNAL_ERR, 0, e);	\
} while(0)

	/*
	(x) mem ptr	(e) err code
	(y) mem size	(i) function id
	(z) fd
	*/
#define S_MAPF(x,y,z, e,i)	\
do {	\
	if (y == 0)	\
		VGERR(i, "(S_MAPF) Incorrect size", 0, e);	\
	x = mmap(0,x,PROT_EXEC|PROT_READ|PROT_WRITE,	\
			MAP_PRIVATE|MAP_FILE,z,0);	\
	if (x == NULL)	\
		VGERR(i, INTERNAL_ERR, 0, e);	\
} while(0)

	/*
	(x) mem ptr	(e) err code
	(y) mem size	(i) function id
	*/
#define S_UNMAP(x,y, e,i)	\
do {	\
	if (y == 0)	\
		VGERR(i, "(S_UNMAP) Incorrect size", 0, e);	\
	if (munmap(x,y) == -1)	\
                VGERR(i, INTERNAL_ERR, 0, e);	\
} while(0)


/***************************************************************************
 *	Macros for errors                                  		   *
 ***************************************************************************/

	/*
	(x) first err msg  (func name)
	(y) second err msg (description)	

	(f) flags
	(e) err code
	
	flags:
	KEEP_LAST	do not change the current vg_err_msg value	
	NO_DESC		description string is not present
	NO_FUNC		the function name string is not present
	NO_RET		return nothing
				
	*/
#define	VGERR(x,y, f,e)	\
do {	\
	set_error_msg(x, y, f);	\
	if (!(f & NO_RET))	\
		return (e);	\
} while(0)

	/*
	(x) first err msg  (func name)
	(y) second err msg (descript )
	
	(f) flags	
	(e) error code

	(d) data
	*/
#define	VGERR2(x,y, f,e, d)	\
do {	\
	set_error_msg(x, y, f);	\
	if (d)	\
		free(d);	\
	if (!(f & NO_RET))	\
		return (e);	\
} while(0)


/***************************************************************************
 *	Misc				   				   *
 ***************************************************************************/

	/*
	Free a structure vg_elf 

	(x) elf structure pointer
	*/
#define ELFREE(x)	\
do {    \
	if ((x).hdr.data)	{ free((x).hdr.data);		}	\
	if ((x).phdr.data) 	{ free((x).phdr.data); 		}	\
	if ((x).dyn.data)	{ free((x).dyn.data); 		}	\
	if ((x).dynsym.data)	{ free((x).dynsym.data); 	}	\
	if ((x).dynstr.data)	{ free((x).dynstr.data); 	}	\
	if ((x).reldyn.data)	{ free((x).reldyn.data); 	}	\
	if ((x).relplt.data)	{ free((x).relplt.data); 	}	\
	if ((x).hash.data)	{ free((x).hash.data);		}	\
	if ((x).interp.data)	{ free((x).interp.data);	}	\
	if ((x).note.data)	{ free((x).note.data);		}	\
	if ((x).bss.data)	{ free((x).bss.data);		}	\
	if ((x).got.data) 	{ free((x).got.data);		}	\
	if ((x).plt.data) 	{ free((x).plt.data);		}	\
} while(0)


	/*
	set relocation info: symidx and type
	*/
#define VG_SET_RSYM(x,y)	((y << 8) | (x & 0xff))
#define VG_SET_RTYPE(x,y)	((y & 0xff) | (x & 0xffffff00))

#define	VG_SET_ST_BIND(x, y)	((y << 4) | (x & 0x80))
#define	VG_SET_ST_TYPE(x, y)	((y & 0x80) | (x & 0xffffff7f))


	/*
	convert (x) byte into a printable character
	(a)
	*/
#define ASCII(a)	(((a)>0x1f && (a)<0x7f)?(a):'.')

	/*
	get the correct formatted size of (x) used in vg_format_data()
	*/
#define	VGFMT_SZ		(HEX_ROW_SZ*3+HEX_ROW_SZ+1)
#define	VGFMT_OUTPUT_SZ(x)	((((x)/HEX_ROW_SZ)+1) * VGFMT_SZ)


	/*
	Simply Shellcode Patcher 
	
	(code) shellcode ptr
	(a   ) byte position
	(b   ) argument
	(c   ) default argument
	*/	
#define	PATCH_SC(code,a,b,c)	(*(u_long*)&code[(a)] = ((b)?(b):(c)) )

#define for_each_bkpt(a,b)      for(a = b; a->next; a = a->next)



/***************************************************************************
 *      macros to get the process base addr                                *
 ***************************************************************************/

/* 
 * IMPORTANT: GET_BASE macro must be before vg_load_proc(),
 * this is because we need get base addr of main object. 
 * It isn't the best method but works well on diferent systems.
 *
 * NOTE: was coded by the grugq, it has been adapted
 * to libvg. 
 */
#if defined(__i386__)
#define	GET_BASE	\
do {	\
	__asm__(	\
		"call .ret\n"	\
		".ret:\n"	\
		"popl %%eax\n"	\
		:"=a"(tmp_base)	\
		:	\
		);	\
	tmp_base = (void*)PAGE_ALIGN((long)tmp_base);	\
while(1) {					\
	if (!memcmp(tmp_base, ELFMAG, SELFMAG)) {	\
		proc_base = (vg_vaddr)tmp_base;	\
		break;	\
	}	\
	else	\
		tmp_base -= PAGE_SIZE;	\
}	\
}while(0)

#elif defined(sun)		/* thx sp4rk ;) */

#define GET_BASE	\
do {	\
	__asm__(	\
		"bn,a .-4\n"	\
		"bn,a .\n"	\
		"call .+4\n"	\
		"add %%o7,32,%%o0\n"	\
		"mov %%i7,%0\n"	\
		:"=r"(tmp_base)	\
		:	\
		);	\
	tmp_base = (void*)PAGE_ALIGN((long)tmp_base);	\
while(1) {	\
	if (!memcmp(tmp_base, ELFMAG, SELFMAG)) {	\
		proc_base = (void*)tmp_base;	\
		break;	\
		\
	}	\
	else	\
		tmp_base -= PAGE_SIZE;	\
}	\
}while(0)
#endif


#endif	/* __LIBVG_MACROS_H */

