/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.2 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE ON YOUR OWN RISK

*/
#ifndef	__LIBVG_STRUCTURES_H
#define	__LIBVG_STRUCTURES_H


typedef struct lmap {
        caddr_t         l_addr;         /* Base address of library */
        const char      *l_name;        /* Absolute path to library */
        void            *l_ld;          /* pointer to _DYNAMIC */
        struct link_map *l_next;
        struct link_map *l_prev;
} lm_entry;
/* freebsd object reference */
#if defined(__FreeBSD__)
typedef struct	fbsd_obj_descriptor {
	Elf32_Word	magic;	
	Elf32_Word 	version;	
	struct fbsd_obj_descriptor *next;		/* next	object				*/
	char 		*path;				/* path (name) shared objcet 		*/
	char 		*origin_path;			/* Directory path of origin file 	*/
	int 		refcount;
	int		dl_refcount; 		
	caddr_t		mapbase; 			/* base addr				 */
	size_t 		mapsize;		
	size_t 		textsize;
	Elf32_Addr 	vaddrbase; 
	caddr_t 	relocbase;    
	Elf32_Dyn 	*dynamic; 			/* dynamic section			 */
	caddr_t 	entry; 
}	bsd_obj_ref;
	/* netbsd object refence */
#elif defined(__NetBSD__)
typedef	struct	nbsd_obj_descriptor {
	Elf32_Word	magic;
	Elf32_Word	version;
	struct nbsd_obj_descriptor *next;
	char		*path;
	int		refcount;
	int		dl_refcount;
        caddr_t         mapbase;                
        size_t          mapsize;
        size_t          textsize;
        Elf32_Addr      vaddrbase;
        caddr_t         relocbase;
        Elf32_Dyn       *dynamic;                   
        caddr_t         entry;
}	bsd_obj_ref;
//#else
//typedef	vg_obj_ref	bsd_obj_ref;
#endif


/* breakpoint */
typedef	struct	breakpoint {
	char		*soname;			/* so where bkpt was set 		*/
	vg_vaddr	vaddr;				/* 1) vaddr for bkpt 			*/
	vg_char		*sname;				/* 2) symbol name for bkpt 		*/
	u_long		orig_code;			/* orig code 				*/
	vg_uchar	stat;				/* status of bkpt: enable or disable    */
	struct	breakpoint	*next;
}	vg_bkpt;


/* stack segment */
typedef struct stack_seg {
	vg_char         *ptr;
	vg_flags	flags;

	vg_vaddr        start;          /* top */
	vg_vaddr        end;            /* bottom */
	vg_size		csize;		/* current size */
	vg_size		msize;		/* max size */

	struct stack_operations {
		vg_vaddr (*get)();
		vg_vaddr (*push)(void *, vg_size);
		vg_vaddr (*pop)(void *, vg_size);
	} op;

}	vg_stack;


typedef struct registers {

#if defined(__FreeBSD__)                                /* FreeBSD 5.2 */
	u_int   fs, es;
	u_int   ds, edi;
	u_int   esi, ebp;
        u_int   isp, ebx;
        u_int   edx, ecx;
        u_int   eax, trapno;
        u_int   err, eip;
        u_int   cs, eflags;
        u_int   esp, ss, gs;

#elif defined(__OpenBSD__) || defined(__NetBSD__)       /* OpenBSD 3.4 & NetBSD */
        int     eax, ecx;
        int     edx, ebx;
        int     esp, ebp;
        int     esi, edi;
        int     eip, eflags;
        int     cs, ss;
        int     ds, es;
        int     fs, gs;

#elif defined(__linux__)                                /* linux 2.4.22 */
        long    ebx, ecx;
        long    edx, esi;
        long    edi, ebp, eax;
        u_short ds, __ds;
        u_short es, __es;
        u_short fs, __fs;
        u_short gs, __gs;
        long    orig_eax, eip;
        u_short cs, __cs;
        long    eflags, esp;
        u_short ss, __ss;
#endif

}       vg_regs;
#if defined(sun)                                      /* SunOS */
typedef prgregset_t vg_regs;
#endif


/* traced process state */
typedef	struct	process_state {

#if defined(sun)
        int             procfd_ctl;
        int             procfd_status;
        int             procfd_as;

        u_int           single_request;

        struct request1 {
                u_long          func;
                prgregset_t     regs;
        } setregs_req;

        struct request2 {
                u_long          func;
                sigset_t        signals;
        } attach_req;

        struct request3 {
                u_long          func;
                u_long          args;
        } control_req;
	pstatus_t       stat;

#elif defined(__i386__)
	vg_int	status;					/* current attached process stat	*/
	vg_int	wstatus;				/* process information given by wait()  */
	vg_int	signal;					/* last signal sent to attached process */
	vg_int	wopts;					/* options for wait()			*/
	vg_int	exit_val;				/* value for exit			*/
	vg_int	handler;				/* check_pstate() handler flags 	*/
	vg_int	ignore[NUM_SIG];			/* list of signals to be ignored	*/
	u_char	step;					/* stepping flag			*/
#endif

	vg_int	rpsm;

}	vg_pstate;


/* 
	VG Structures for elf sections

	Each structure define a valid elf section
	of an object and has the following info:
	
		data 	=>
		offset	=>
		vaddr	=>
		size	=>
		entsize	=>
		entnum 	=>
*/

/* elf header */
typedef struct hdr {
	Elf32_Ehdr	*data;
	vg_off		offset;
	vg_vaddr	vaddr;
	vg_size		size;
} 	vg_hdr;


/* program header table */
typedef struct phdr {
	Elf32_Phdr	*data;
	vg_off		offset;
	vg_vaddr	vaddr;
	vg_size		size;
	vg_size		entsize;
	vg_idx		entnum;	
} 	vg_phdr;


/* section header table */
typedef struct shdr {
	Elf32_Shdr	*data;
	vg_off		offset;	
	vg_vaddr	vadddr;
	vg_size		size;	
	vg_size		entsize;
	vg_idx		entnum;	
} 	vg_shdr;


/* interp section, interpreter */
typedef	struct interp {	
	char		*data;
	vg_off		offset;
	vg_vaddr	vaddr;
	vg_size		size;
} 	vg_interp;


/* notes section */
typedef	struct note {
	char		*data;
	vg_off		offset;
	vg_vaddr	vaddr;
	vg_size		size;
}	vg_note;


/* dynamic info */
typedef struct dyn {
	Elf32_Dyn	*data;
	vg_off		offset;	
	vg_vaddr	vaddr;
	vg_size		size;	
	vg_size		entsize;
	vg_idx		entnum;	
} 	vg_dyn;


/* dynamic symbol table */
typedef struct dynsym {
	Elf32_Sym	*data;
	vg_off		offset;
	vg_vaddr	vaddr;
	vg_size		size;	
	vg_size		entsize;
	vg_idx		entnum;
} 	vg_dynsym;
	

/* dynamic string table */
typedef struct dynstr {
	vg_char		*data;
	vg_off		offset;
	vg_vaddr	vaddr;
	vg_size		size;
} 	vg_dynstr;


/* reldyn table */
typedef struct reldyn {
	Elf32_Rel	*data;
	vg_off		offset;
	vg_vaddr	vaddr;
	vg_size		size;
	vg_size		entsize;
	vg_idx		entnum;
} 	vg_reldyn;			


/* relplt table */
typedef struct relplt {		
	Elf32_Rel	*data;
	vg_off		offset;
	vg_vaddr	vaddr;
	vg_size		size;
	vg_size		entsize;
	vg_idx		entnum;
} 	vg_relplt;		


/* hash table */
typedef struct hash {	
	void		*data;
	void		*buckets;
	void		*chains;
	vg_off		offset;	
	vg_vaddr	vaddr;
	vg_size		size;
	vg_idx		nbuckets;
	vg_idx		nchains;
} 	vg_hash;				


/* global offset table */
typedef struct got {
	vg_vaddr	*data;
	vg_off		offset;
	vg_vaddr	vaddr;
	vg_size		size;
	vg_size		entsize;	/* got entry size, typically integer size */
	vg_idx		entnum;		/* number of got entries */
}	vg_got;


/* procedure linkage table */
typedef struct plt {
	vg_char		*data;
	vg_off		offset;
	vg_vaddr	vaddr;	
	vg_size		size;
	vg_size		entsize;
	vg_idx		entnum;
}	vg_plt;	


/* bss */
typedef	struct bss {
	vg_char		*data;
	vg_off		offset;
	vg_vaddr	vaddr;
	vg_size		size;
}	vg_bss;


/* init */
typedef struct init {
	vg_vaddr	vaddr;
	vg_off		offset;
}	vg_init;


/* fini */
typedef struct fini {
	vg_vaddr	vaddr;
	vg_off		offset;
}	vg_fini;


/* elf sections of an object */
typedef	struct elf_info {	
	vg_flags	flags;

	/* elf headers*/
	vg_hdr		hdr;
	vg_phdr		phdr;
	vg_shdr		shdr;

	/* elf sections/structures */
	vg_interp	interp;
	vg_note		note;
	vg_dyn		dyn;
	vg_dynsym	dynsym;	
	vg_dynstr	dynstr;
	vg_reldyn	reldyn;
	vg_relplt	relplt;	
	vg_hash		hash;
	vg_got		got;
	vg_plt		plt;
	vg_bss		bss;
	vg_init		init;	
	vg_fini		fini;	

} 	vg_elfref;		


/* segment */
typedef struct obj_segment {
	vg_vaddr	start;		/* the segment start address */	
	vg_vaddr	end;		/* the segment end address */
	vg_size		size;		/* segment size in bytes */
	vg_vaddr	*ptr;		/* mapped segment pointer */
	vg_flags	flags;		/* some internal flags for segment */
}	vg_segment;		


/* object descriptor */
typedef struct obj_desc {
	vg_idx 		od_idx;		/* idx on descriptor table */
	vg_vaddr	od_base;	/* base addr of shared object */
	vg_vaddr	od_name;	/* vaddr of string name	*/
	vg_char		*od_str;	/* name of shared object */
	void 		*od_dyn;	/* dynamic section addr	of object */
	struct obj_desc	*od_next;	/* next entry */
	struct obj_desc	*od_prev;	/* previouse entry */
} 	vg_obj_desc;			


/* object */
typedef struct object {	
	vg_idx		idx;		/* shared object idx into object table */
	vg_char		*name;		/* shared object name */
	vg_uchar	flags;		/* object flags	*/

	vg_obj_desc	*od;		/* ptr to his descriptor */
	vg_elfref	elf;		/* elf tables and info */

	vg_segment	text;		/* the object text segment */
	vg_segment	data;		/* the object data segment */
	vg_segment	heap;		/* the object heap segment */

	struct object_operations {
		vg_int	(*map)(struct object *);
		vg_int	(*unmap)(struct object *);
		vg_int	(*insert)();
		vg_int	(*remove)();
			
		vg_idx		(*get_idx)(struct object *);
		vg_char		*(*get_name)(struct object *);

		struct object	*(*by_name)(char *);
		struct object	*(*by_base)(vg_vaddr);
		struct object	*(*by_dyn)(vg_vaddr);
		struct object	*(*by_idx)(vg_idx);
		struct object	*(*by_vaddr)(vg_vaddr, vg_uint *);
	} op;	

} 	vg_object;


/* high level symbol structure */
typedef struct sym_info {
        vg_char         *soname;      	/* object where symbol is */
        vg_char         *syname;      	/* symbol name */

	vg_object	*obj;
        vg_vaddr        vaddr;
        vg_off          offset;       	/* text segment offset */
        vg_size         size;         	/* symbol size */

        vg_idx		stridx;       	/* idx into dynstr */
        vg_idx		symtab_idx;   	/* idx into dynsym */
        vg_uchar	info;         	/* symbol type (var|func) etc */

	vg_vaddr	got;
	vg_vaddr	plt;

        struct sym_info *next;		/* next */
}	vg_sym;

typedef	struct got_entry {
	vg_vaddr	vaddr;	
	vg_vaddr	value;
}	vg_gotentry;

typedef struct plt_entry {
        vg_flags        flags;          /* flags */
        vg_type         type;           /* type of plt entry */
        vg_idx          idx;            /* index on plt section */
        vg_off          offset;         /* offset */
        vg_vaddr        vaddr;          /* virtual address */
        vg_char         *table;         /* ptr to first entry */
        vg_object       *obj;           /* object where the plt entry is */
        vg_char         *code;          /* pointer to the local copy, the code */
        vg_vaddr        got_vaddr;
        vg_idx          got_idx;
}       vg_pltentry;


/* process context structure */
typedef	struct	proc_context {
	vg_id 		id;				/* traced process id */
	vg_pid		pid;				/* pid */
	vg_flags	flags;				/* flags */
	vg_path		*execpath;			/* process string id (exec) */

	vg_object	*obj_table[MAX_OBJECTS];	/* objects table */
	vg_idx		objnum;

	vg_obj_desc	*obj_desc_table[MAX_OBJECTS];	/* object descriptor table */
	vg_vaddr	reflist_addr;
	
	vg_pstate	pstate;				/* traced process state info */
	vg_stack	stack;				/* stack segment */
	vg_regs		regs, orig_regs;
	vg_bkpt		*init_bkpt;			/* bkpts list */
	
	/* process operations */
	struct proc_operations {
		/* basic ops */
		vg_int	(*attach)(vg_pid);	
		vg_int	(*detach)(vg_pid);
		vg_size	(*read)(void *, vg_vaddr, vg_size);
		vg_size	(*write)(void *, vg_vaddr, vg_size);
		vg_regs	*(*gregs)(struct proc_context *);
		vg_regs	*(*sregs)(struct proc_context *, vg_regs *);
		vg_int	(*stop)(struct proc_context *);
		vg_int	(*cont)(struct proc_context *);
		vg_int	(*step)(struct proc_context *);
	} op;

	/* next and prev */
	struct proc_context *next;
	struct proc_context *prev;

}	vg_proc;


#endif /* __LIBVG_STRUCTURES_H */

