/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
#include "libvg.h"

extern vg_proc	*cur_proc;
extern vg_pid	cur_pid;

/*
 * Get the correct virtual address where the code will be injected. 
 * @where can be:
 *		
 *	1) a valid virtual address 
 *	2) a tag:
 *		2.1) next instruction tag (str is null)
 *		2.2) stack tag (str is null)
 *		2.3) symbol tag (str is a symbol name)
 *		2.4) object tag (str is an object name)
 *	
 */
vg_vaddr get_vaddr(vg_vaddr *where, vg_char *str)
{
	vg_object	*obj = 0;
	vg_sym		sym;

	/* 1) nothing else is needed */
	if (*where > LAST_TAG)
		goto end;

	/* 2.1) */
	if (VG_NEXT_INSTRUCTION == *where) {
#if defined(__i386__)
		*where = cur_proc->regs.eip;
#endif
		goto end;
	}

	/* 2.2 */	
	if (VG_STACK_SEG == *where) {
#if defined(__i386__)
		*where = cur_proc->regs.esp;
#endif
		goto end;
	}

	/* 2.3 */	
	switch(*where) {
	case VG_FUNC:
		if (vg_sym_func_local_by_name(0, &sym, str)<0)
			return (ERR);
		*where = sym.vaddr;
		goto end;
		break;
	case VG_VAR:
		if (vg_sym_var_local_by_name(0, &sym, str)<0)
			return (ERR);
		*where = sym.vaddr;
		goto end;
		break;
	}

	/* 2.4 */
	if (!(obj = vg_obj_by_name(str))) 
		return (ERR);

	switch(*where) {
	case VG_OBJ_HDR:
		if (!vg_obj_get_hdr(obj))
			return (ERR);
		*where = obj->elf.hdr.vaddr;
		break;
	case VG_OBJ_PHDR:
		if (!vg_obj_get_phdr(obj,0))
			return (ERR);
		*where = obj->elf.phdr.vaddr;
		break;
	case VG_OBJ_DYN:
		if (!vg_obj_get_dyn(obj,0))
			return (ERR);
		*where = obj->elf.dyn.vaddr;
		break;
	case VG_OBJ_DYNSYM:
		if (!vg_obj_get_dynsym(obj,0))
			return (ERR);
		*where = obj->elf.dynsym.vaddr;
		break;
	case VG_OBJ_DYNSTR:
		if (!vg_obj_get_dynstr(obj))
			return (ERR);
		*where = obj->elf.dynstr.vaddr;
		break;
	case VG_OBJ_RELPLT:
		if (!vg_obj_get_relplt(obj, 0))
			return (ERR);
		*where = obj->elf.relplt.vaddr;
		break;
	case VG_OBJ_RELDYN:
		if (!vg_obj_get_reldyn(obj,0))
			return (ERR);
		*where = obj->elf.reldyn.vaddr;
		break;
	case VG_OBJ_HASH:	
		if (!vg_obj_get_hash(obj))
			return (ERR);
		*where = obj->elf.hash.vaddr;
		break;
	case VG_OBJ_GOT:
		if (!vg_obj_get_got(obj,0))
			return (ERR);
		*where = obj->elf.got.vaddr;
		break;
	case VG_OBJ_PLT:
		if (!vg_obj_get_plt(obj,0))
			return (ERR);
		*where = obj->elf.plt.vaddr;
		break;
	case VG_OBJ_BSS:
		if (!vg_obj_get_bss(obj))
			return (ERR);
		*where = obj->elf.bss.vaddr;
		break;
	case VG_OBJ_NOTE:
		if (!vg_obj_get_note(obj))
			return (ERR);
		*where = obj->elf.note.vaddr;
		break;
	case VG_OBJ_INTERP:
		if (!vg_obj_get_interp(obj))
			return (ERR);
		*where = obj->elf.interp.vaddr;
		break;
	case VG_OBJ_INIT:
		if (!vg_obj_get_init(obj))
			return (ERR);
		*where = obj->elf.init.vaddr;
		break;
	case VG_OBJ_FINI:
		if (!vg_obj_get_fini(obj))
			return (ERR);
		*where = obj->elf.fini.vaddr;
		break;
	case VG_OBJ_TEXT_SEG:
		*where = obj->text.start;
		break;
	case VG_OBJ_DATA_SEG:
		*where = obj->data.start;
		break;
	case VG_OBJ_HEAP_SEG:
		*where = obj->heap.start;
		break;
	default:
		return (ERR);
	}

end:
	return (OK);
}

/*  
 * EXT_M: we get the code from an external source
 *
 * This function load a shared object file, it must be a
 * valid asm file compiled as shared object, and then  
 * search the init tag (VG_HEAD) and the fini tag (VG_TAIL)
 * in order to know the correct code size. 
 * 
 */
vg_int	m_external(vg_int to, vg_char *file)
{
	vg_vaddr *handle, *start, *end;

	/* locate code function  */
	if (!(handle = dlopen(file, RTLD_LAZY)))
		VGERR(0, INTERNAL_ERR, 0, ERR);

	/* get start */
	if (!(start = dlsym(handle, VG_HEAD)))
		VGERR(VG_INJECT, "VG_HEAD tag not found", 0, ERR);

	/* get end */
	if (!(end = dlsym(handle, VG_TAIL)))
		VGERR(VG_INJECT, "VG_TAIL tag not found", 0, ERR);

	/* write the code */
	return (vg_write(start, to, (vg_vaddr)end-(vg_vaddr)start));
}

/* 
 * INT_M: we get the code from our address space
 *
 * This one, write the code directly into the process
 */
vg_int	m_internal(vg_vaddr to, void *start, vg_size sz)
{
	return (vg_write(start, to, sz));
}

/*
 * vg_inject = this function inject code into the process's address
 * space, several methods can be used.
 *
 *
 */
vg_int	vg_inject(vg_vaddr where, void *start, void *end, vg_size size, 
					vg_char *str1, vg_char *str2)
{
	vg_int	method = 0;

	/* check the correct args for each method */
	if (where && str1)
		method = EXTERNAL_CODE;
	else if (where && (start && (size || end)))
		method = INTERNAL_CODE;
	else
		VGERR(VG_INJECT, NULL, E_INVAL, ERR);

	/* get tarjet vaddr */
	if (get_vaddr(&where, str2)<0)
		return (ERR);

	/* now inject the code */
	if (method == EXTERNAL_CODE) 
		return (m_external(where, str1));
	if (method == INTERNAL_CODE) 
		return (m_internal(where, start, (size?size:end-start)));
	
	return (ERR);
}

