/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
#include "libvg.h"

extern  vg_vaddr	proc_base;
extern  vg_proc 	*cur_proc;


vg_int	setup_object_operations(vg_object *obj)
{
	obj->op.map		= vg_map_object;
	obj->op.unmap		= vg_unmap_object;
	obj->op.insert		= 0;
	obj->op.remove		= 0;

	obj->op.get_idx		= vg_obj_get_idx;
	obj->op.get_name	= vg_obj_get_name;

	obj->op.by_name		= vg_obj_by_name;
	obj->op.by_base		= vg_obj_by_base;
	obj->op.by_dyn		= vg_obj_by_dyn;
	obj->op.by_idx		= vg_obj_by_idx;
	obj->op.by_vaddr	= vg_obj_by_vaddr;

	return (OK);
}


/*
 * set_text_segment_range = get object's segment code memory range
 *
 * @obj
 * #ret
 */
vg_int  set_text_segment_range(vg_object *obj)
{
        Elf32_Phdr      *phentry;
        u_long          start = 0;
        u_long          align = DEFAULT_ALIGN;
        u_int           idx, size;

        for (idx = 0, size = 0, phentry = obj->elf.phdr.data;
                idx < obj->elf.hdr.data->e_phnum;
                        idx++, phentry++) {

                if (is_text_segment(phentry)) {
                        /* get correct align given by phdr */
                        if (phentry->p_align)
                                align = phentry->p_align;

                        /* shared objects maybe need relocation */
                        if (obj->flags & OBJ_MO)
                                start = phentry->p_vaddr;
                        else
                                start = phentry->p_vaddr + obj->text.start;


                        start = ROUND_DOWN(start, align);
                        size += phentry->p_filesz;
                }
        }
        /* set range */
        obj->text.start = start;
        obj->text.end = start + ROUND_UP(size, align);
        obj->text.size = obj->text.end - obj->text.start;
        return (0);
}

/*
 * set_data_segment_range =
 *
 * @obj:
 * #ret:
 */
vg_int  set_data_segment_range(vg_object *obj)
{
        Elf32_Phdr      *entry = obj->elf.phdr.data;
        u_long          align = DEFAULT_ALIGN;
        u_long          start = 0;
        int             size, idx;
        int             first = 1;
        /* some systems like openbsd has more than one PT_LOAD entry at phdr
           for the data segment so we need information about them in order to
           get the correct size of the segment at memory
        */
        for (idx = 0, size = 0; idx < obj->elf.hdr.data->e_phnum; idx++, entry++) {
                /* PT_LOAD of data */
                if (entry->p_type == PT_LOAD &&
                        ( entry->p_flags == (PF_R+PF_W+PF_X) ||
                          entry->p_flags == (PF_R+PF_W) ||
                          entry->p_flags == PF_R)) {
                        if (entry->p_align)
                                align = entry->p_align;
                        if (first) {
                        /* esto me da el inicio, asi vale */
                                if (obj->flags & OBJ_MO)        /* main object */
                                        start = entry->p_vaddr;
                                else                            /* shared object */
                                        start = entry->p_vaddr + obj->text.start;
                                start = ROUND_DOWN(start,entry->p_align);
                                first = 0;
                        }
                        /* inicial - vaddractual */
                        if (obj->flags & OBJ_MO)
                                size = SET_VADDR(entry->p_vaddr, obj->text.start) - start;
                        else
                                size = (entry->p_vaddr + obj->text.start) - start;
                        /* offset actual + tamao en file */
                        size += entry->p_filesz;
                }
        }
        /* set range */
        obj->data.start = start;
        obj->data.end = start + ROUND_UP(size, align);
        obj->data.size = obj->data.end - obj->data.start;
        return (0);
}

/*
 * set_heap_segment_range =
 *
 * @obj:
 * #ret:
 */
vg_int  set_heap_segment_range(vg_object *obj)
{
        return (0);
}


/*
 * load_objects_table =   LOAD PROCESS'S OBJECTS TABLE
 *
 * @otbl
 * @objd_tbl
 */
vg_int	load_objects_table(vg_object **otbl, vg_obj_desc **objd_tbl)
{
	vg_idx	obj_idx, objd_idx;

	/* if no rtld:	one entry only, main object */
	/* if rtld: 	make entry for each object */
	for (obj_idx = 0, objd_idx = 0;
		objd_tbl[objd_idx] && objd_idx<MAX_OBJECTS;
			++objd_idx, ++obj_idx) {

		/* dont' use incorrect entries */
		if (objd_tbl[objd_idx]->od_base == INCORRECT_ENTRY) {
			--obj_idx;
			continue;
		}

		/* allocate next struct object */
		S_ALLOC(otbl[obj_idx], sizeof(vg_object), ERR, 0, CLEAN);

		/* setup operations */
		setup_object_operations(otbl[obj_idx]);

		/* set descriptro for this object */
		otbl[obj_idx]->od = objd_tbl[objd_idx];

		/* set so name to ref entry name */
		otbl[obj_idx]->name = objd_tbl[objd_idx]->od_str;

		/* set idx */
		otbl[obj_idx]->idx = obj_idx;

		/* set minimal mapp infor => text_base */
		otbl[obj_idx]->text.start = otbl[obj_idx]->od->od_base;		

		/* get minimal elf infor: hdr, phdr and dynamic */
		if (!vg_obj_get_hdr(otbl[obj_idx]))
			return (ERR);

		if (!vg_obj_get_phdr(otbl[obj_idx], 0))
			return (ERR);

		if (cur_proc->flags & PROC_WITH_RTLD) {
			if (!vg_obj_get_dyn(otbl[obj_idx], 0))
				return (ERR);
		}
		
		/* set main object or shared object flag */
		if (!objd_idx)
			otbl[obj_idx]->flags |= OBJ_MO;
		else
			otbl[obj_idx]->flags |= OBJ_SO;	

		/* get maps for text and data segments */
		set_text_segment_range(otbl[obj_idx]);
		set_data_segment_range(otbl[obj_idx]);
		set_heap_segment_range(otbl[obj_idx]);
	}

	cur_proc->objnum = obj_idx;

	return (OK);
}	

/*
 * unload_objects_table = Unload process's objects table
 * 
 * @otbl
 */
vg_int	unload_objects_table(vg_object **otbl)
{
	vg_idx	idx;

	for (idx = 0; otbl[idx]; idx++) {
		/* freeing each object */
		if (otbl[idx]->flags & OBJ_MAPPED)
			vg_unmap_object(otbl[idx]);
		/* free */
		ELFREE(otbl[idx]->elf);
		free(otbl[idx]);
	}
	return (OK);
}

/*
 * vg_map_object = Map object: copy object's segments from remote
 * process to local anonymous mapped memory, this set the object as mapped.
 *
 * @obj: object to map 
 * #ret: 0
 */
vg_int	vg_map_object(vg_object *obj)
{
	vg_size		map_size;
	vg_vaddr	*map_addr = NULL;
	
	if (!obj) 
		VGERR(VG_MAP_OBJECT, NULL, E_INVAL, ERR);

	/* already mapped */
	if (OBJ_IS_MAPPED(obj)) 
		return (OK);

	/* free elf structures */
	ELFREE(obj->elf);
	memset(&obj->elf, 0, sizeof(vg_elfref));

	/* map anonymous memory for text, data and heap segments */
	map_size = obj->text.size + obj->data.size + obj->heap.size + PAGE_SIZE;
	S_MAP(map_addr, map_size, ERR, VG_MAP_OBJECT);

	/* copy segments */
	if (vg_dread(map_addr, obj->text.start, obj->text.size) !=
			obj->text.size) {
		S_UNMAP((void*)obj->text.size, map_size, ERR, VG_MAP_OBJECT);
		return (ERR);
	}

	if (vg_dread((vg_char*)map_addr+obj->text.size, obj->data.start,
			obj->data.size) != obj->data.size) {
		S_UNMAP((void*)obj->text.size, map_size, ERR, VG_MAP_OBJECT);
		return (ERR);
	}
	
	/* update segments ptrs */
	obj->text.ptr = map_addr;	
	obj->data.ptr = (u_long*)((vg_char*)map_addr+obj->text.size);

	/* set mapped */
	obj->flags |= OBJ_MAPPED;

	/* update some elf ptrs */
	vg_obj_get_hdr(obj);
	vg_obj_get_phdr(obj, 0);
	vg_obj_get_dyn(obj, 0);

	return (OK);
}

/*
 * vg_unmap_object = Unset object mapped, unmap a previusly
 * mapped zone
 *
 * @obj = object to unmap
 * #ret = 0
 */
vg_int	vg_unmap_object(vg_object *obj)
{	
	vg_size	map_size; 

	if (!obj) 
		VGERR(VG_UNMAP_OBJECT, NULL, E_INVAL, ERR);

	/* mapped only */
	if (!OBJ_IS_MAPPED(obj))
		return (0);

	/* free mapped mem */
	map_size = obj->text.size + obj->data.size + obj->heap.size + PAGE_SIZE;
	S_UNMAP((void*)obj->text.ptr, map_size, ERR, VG_UNMAP_OBJECT);

	/* clean elf pointers */
	memset(&obj->elf, 0, sizeof(vg_elfref));

	/* set unmapped */
	obj->flags &= ~OBJ_MAPPED;
	
	/* get again necesesary sections and headers */
	vg_obj_get_hdr(obj);
	vg_obj_get_phdr(obj, 0);
	vg_obj_get_dyn(obj, 0);

	return (0);
}

/*
 * vg_obj_by_name = return an object giving its name
 * @name: correct object name
 */
vg_object *vg_obj_by_name(vg_char *name)
{
	vg_idx	  idx;

	CHECKDYN(VG_OBJ_BY_NAME, cur_proc->obj_table[0]);

	if (!name)
		VGERR(VG_OBJ_BY_NAME, NULL, E_INVAL, NULL);

	if (strlen(name) > MAX_BUF)
		VGERR(VG_OBJ_BY_NAME, "Object name too long", 0, NULL);

	for (idx = 0; idx < cur_proc->objnum; idx++) {
		if (!strcmp(cur_proc->obj_table[idx]->od->od_str, name))
			return (cur_proc->obj_table[idx]);
	}

	VGERR(VG_OBJ_BY_NAME, "Incorrect object name", 0, NULL);
}


/*
 * vg_obj_by_base = return an object giving the starting address
 * of its text segment
 *
 * @base: vaddr
 */
vg_object *vg_obj_by_base(vg_vaddr base)
{
	vg_int	  idx;

	CHECKDYN(VG_OBJ_BY_BASE, cur_proc->obj_table[0]);

	if (!base)	
		VGERR(VG_OBJ_BY_BASE, NULL, E_INVAL, NULL);

	for (idx = 0; idx < cur_proc->objnum; idx++) {
		if (base == cur_proc->obj_table[idx]->od->od_base)
			return (cur_proc->obj_table[idx]);
	}

	VGERR(VG_OBJ_BY_BASE, "Incorrect object base", 0, NULL);
}

/*
 * vg_obj_by_dyn = return an object giving the starting address
 * of its dynamic section
 * 
 * @dyn = vaddr
 */
vg_object *vg_obj_by_dyn(vg_vaddr dyn)
{
	vg_int	  idx;

	CHECKDYN(VG_OBJ_BY_DYN, cur_proc->obj_table[0]);

	if (!dyn)
		VGERR(VG_OBJ_BY_DYN, NULL, E_INVAL, NULL);

	for (idx = 0; idx < cur_proc->objnum; idx++) {
		if (cur_proc->obj_table[idx]->od->od_dyn == (vg_vaddr*)dyn)
			return (cur_proc->obj_table[idx]);
	}

	VGERR(VG_OBJ_BY_DYN, "Incorrect object dynamic section vaddr", 0, NULL);
}

/*
 * vg_obj_by_idx = return an object giving its index into the
 * objects table
 *
 */
vg_object *vg_obj_by_idx(vg_idx idx)
{
	CHECKDYN(VG_OBJ_BY_IDX, cur_proc->obj_table[0]);

	if (idx < 0 || idx > cur_proc->objnum)
		VGERR(VG_OBJ_BY_IDX, NULL, E_INVAL, NULL);

	return (cur_proc->obj_table[idx]);
}

/*
 * vg_obj_by_vaddr = 
 *
 * @vaddr:
 * @seg:
 */
vg_object *vg_obj_by_vaddr(vg_vaddr vaddr, vg_uint *seg)
{
	vg_idx	   idx;

	if (vaddr <= 0) 
		VGERR(VG_OBJ_BY_VADDR, NULL, E_INVAL, NULL);

	for(idx = 0; idx < cur_proc->objnum; idx++) {
		/* check with the text segment */
		if (CHECK_TEXTSEG_VADDR(vaddr, cur_proc->obj_table[idx])) {
			if (seg) *seg = VG_TEXT_SEG;
			return (cur_proc->obj_table[idx]);
		}
		/* check with the data segment */
		if (CHECK_DATASEG_VADDR(vaddr, cur_proc->obj_table[idx])) {
			if (seg) *seg = VG_DATA_SEG;
			return (cur_proc->obj_table[idx]);
		}
		/* check with the heap segment */
		if (CHECK_HEAPSEG_VADDR(vaddr, cur_proc->obj_table[idx])) {
			if (seg) *seg = VG_HEAP_SEG;
			return (cur_proc->obj_table[idx]);
		}
	}
	return (NULL);
}

/*
 * Get/Set segment information:
 *
 * vg_obj_get_X_start = get the starting address of the segment
 * vg_obj_set_X_start = change it 
 *
 * vg_obj_get_X_end = get the end of the segment 
 * vg_obj_get_X_end = change it
 *
 * vg_obj_get_X_size = get the size of the segment
 * vg_obj_set_X_size = change it
 */

/* text segment */
vg_vaddr vg_obj_get_text_start(vg_object *obj)
{
	if (!obj || !obj->text.start)
		VGERR(VG_OBJ_GET_TEXT_START, NULL, E_INVAL, ERR);
	return (obj->text.start);
}

vg_vaddr vg_obj_set_text_start(vg_object *obj, vg_vaddr start)
{
	if (!obj || !start || !obj->text.start) 
		VGERR(VG_OBJ_SET_TEXT_START, NULL, E_INVAL, ERR);
	obj->text.start = start;
	return (obj->text.start);
}

vg_vaddr vg_obj_get_text_end(vg_object *obj)
{
	if (!obj || !obj->text.end)
		VGERR(VG_OBJ_GET_TEXT_END, NULL, E_INVAL, ERR);
	return (obj->text.end);
}

vg_vaddr vg_obj_set_text_end(vg_object *obj, vg_vaddr end)
{
	if (!obj || !end || !obj->text.end)
		VGERR(VG_OBJ_SET_TEXT_END, NULL, E_INVAL, ERR);
	obj->text.end = end;
	return (obj->text.end);
}

vg_size	vg_obj_get_text_size(vg_object *obj)
{
	if (!obj || !obj->text.size)
		VGERR(VG_OBJ_GET_TEXT_SIZE, NULL, E_INVAL, ERR);
	return (obj->text.size);
}

vg_size	vg_obj_set_text_size(vg_object *obj, vg_size size)
{
	if (!obj || !size || !obj->text.size)
		VGERR(VG_OBJ_SET_TEXT_SIZE, NULL, E_INVAL, ERR);
	obj->text.size = size;
	return (obj->text.size);
}

/* data segment */
vg_vaddr vg_obj_get_data_start(vg_object *obj)
{
	if (!obj || !obj->data.start)
		VGERR(VG_OBJ_GET_DATA_START, NULL, E_INVAL, ERR);
	return (obj->data.start);
}

vg_vaddr vg_obj_set_data_start(vg_object *obj, vg_vaddr start)
{
	if (!obj || !start || !obj->data.start)
		VGERR(VG_OBJ_SET_DATA_START, NULL, E_INVAL, ERR);
	obj->data.start = start;
	return (obj->data.start);
}

vg_vaddr vg_obj_get_data_end(vg_object *obj)
{
	if (!obj || !obj->data.end)
		VGERR(VG_OBJ_GET_DATA_END, NULL, E_INVAL, ERR);
	return (obj->data.end);
}

vg_vaddr vg_obj_set_data_end(vg_object *obj, vg_vaddr end)
{
	if (!obj || !end || !obj->data.end) 
		VGERR(VG_OBJ_SET_DATA_END, NULL, E_INVAL, ERR);
	obj->data.end = end;
	return (obj->data.end);
}

vg_size	vg_obj_get_data_size(vg_object *obj)
{
	if (!obj || !obj->data.size)
		VGERR(VG_OBJ_GET_DATA_SIZE, NULL, E_INVAL, ERR);
	return (obj->data.size);
}

vg_size	vg_obj_set_data_size(vg_object *obj, vg_size size)
{
	if (!obj || !size || !obj->data.size)
		VGERR(VG_OBJ_SET_DATA_SIZE, NULL, E_INVAL, ERR);
	obj->data.size = size;
	return (obj->data.size);
}

/* heap segment */
vg_vaddr vg_obj_get_heap_start(vg_object *obj)
{
	if (!obj || !obj->heap.start)
		VGERR(VG_OBJ_GET_HEAP_START, NULL, E_INVAL, ERR);
	return (obj->heap.start);
}

vg_vaddr vg_obj_set_heap_start(vg_object *obj, vg_vaddr start)
{
	if (!obj || !start || !obj->heap.start)
		VGERR(VG_OBJ_SET_HEAP_START, NULL, E_INVAL, ERR);
	obj->heap.start = start;
	return (obj->heap.start);
}

vg_vaddr vg_obj_get_heap_end(vg_object *obj)
{
	if (!obj || !obj->heap.end)
		VGERR(VG_OBJ_GET_HEAP_END, NULL, E_INVAL, ERR);
	return (obj->heap.end);
}

vg_vaddr vg_obj_set_heap_end(vg_object *obj, vg_vaddr end)
{
	if (!obj || !end || !obj->heap.end)
		VGERR(VG_OBJ_SET_HEAP_END, NULL, E_INVAL, ERR);
	obj->heap.end = end;
	return (obj->heap.end);
}

vg_size	vg_obj_get_heap_size(vg_object *obj)
{
	if (!obj || !obj->heap.size)
		VGERR(VG_OBJ_GET_HEAP_SIZE, NULL, E_INVAL, ERR);
	return (obj->heap.size);
}

vg_size	vg_obj_set_heap_size(vg_object *obj, vg_size size)
{
	if (!obj || !size || !obj->heap.size)
		VGERR(VG_OBJ_SET_HEAP_SIZE, NULL, E_INVAL, ERR);
	obj->heap.size = size;
	return (obj->heap.size);
}

/*
 * vg_obj_get_name = get object name
 * @obj: 
 */
vg_char	*vg_obj_get_name(vg_object *obj)
{
	if (!obj)
		VGERR(VG_OBJ_GET_NAME, NULL, E_INVAL, NULL);
	return (obj->name);
}

vg_char	*vg_obj_set_name(vg_object *obj, vg_char *name)
{
	if (!obj || !name)
		VGERR(VG_OBJ_SET_NAME, NULL, E_INVAL, NULL);
	if (obj->name)
		free(obj->name);
	obj->name = strdup(name);
	return (obj->name);
}

/*
 * vg_obj_get_idx =
 * @obj: 
 */
vg_idx	vg_obj_get_idx(vg_object *obj)
{
	if (!obj)
		VGERR(VG_OBJ_GET_IDX, NULL, E_INVAL, ERR);
	return (obj->idx);
}


