/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
/*

 ############################################################################
 ##                    PROCESS ACCESS LAYER    II                          ##
 ############################################################################

*/
#include "libvg.h"

extern	pid_t	cur_pid;
extern	vg_proc	*cur_proc;

#if defined(__BSD__)
#define	PT_ARG3		(caddr_t)1
#else
#define	PT_ARG3		NULL
#endif


/*
 *
 * [ Ptrace API ]
 * -----------------------------------------------------------------------+
 *                                                                        |
 *    1 attach   4 write   7 stop                                         | 
 *    2 detach   5 gregs   8 cont                                         |
 *    3 read     6 sregs   9 step                                         |
 *                                                                        |
 * Supported: Linux and BSD                                               |
 *                                                                        |
 * -----------------------------------------------------------------------+
 */

vg_int	ptrace_attach(vg_pid);
vg_int	ptrace_detach(vg_pid);
vg_size	ptrace_read(void *, vg_vaddr, vg_size);
vg_size	ptrace_write(void *, vg_vaddr, vg_size);
vg_regs *ptrace_gregs(vg_proc *);
vg_regs	*ptrace_sregs(vg_proc *, vg_regs *);
vg_int	ptrace_cont(vg_proc *);
vg_int	ptrace_stop(vg_proc *);
vg_int	ptrace_step(vg_proc *);

vg_int	setup_ptrace_operations(vg_proc *p)
{
	/* initialize basic operations */
	p->op.attach	= ptrace_attach;
	p->op.detach	= ptrace_detach;
	p->op.read	= ptrace_read;
	p->op.write	= ptrace_write;
	p->op.gregs	= ptrace_gregs;
	p->op.sregs	= ptrace_sregs;
	p->op.stop 	= ptrace_stop;
	p->op.cont	= ptrace_cont;
	p->op.step	= ptrace_step;

	return (OK);
}

vg_int	ptrace_attach(vg_pid pid)
{
	if (pid <= 0)
		VGERR(__PROC_ATTACH, "Incorrect pid", 0, ERR);

	/* stop rpsm */
	disable_rpsm();

	if (ptrace(VG_ATTACH, pid, 0, 0)<0)
		VGERR(0, INTERNAL_ERR, 0, ERR);

	/* wait for sigchld signal */
	waitpid(cur_pid, &cur_proc->pstate.wstatus, cur_proc->pstate.wopts);
	cur_proc->pstate.status = PS_STOPPED;

	/* restart rpsm */
	enable_rpsm();

	return (OK);
}

vg_int	ptrace_detach(vg_pid pid)
{
	if (pid <= 0)
		VGERR(__PROC_DETACH, "Incorrect pid", 0, ERR);

	if (ptrace_stop(cur_proc)<0)
		VGERR(0, INTERNAL_ERR, 0, ERR);

	if (ptrace(VG_DETACH, pid, PT_ARG3, 0)<0)
		VGERR(0, INTERNAL_ERR, 0, ERR);

	return (OK);
}

vg_size	ptrace_read(void *buf, vg_vaddr vaddr, vg_size size)
{
	vg_vaddr *p;
        vg_uint  read;

	if (!buf || vaddr<=0 || size<=0)
		VGERR(VG_DREAD, NULL, E_INVAL, ERR);

#if defined(__linux__) || defined(__BSD__)
	for (read = 0; read < size; read += 4) {
		p = (u_long*)(buf+read);
#if defined(__linux__)
		*p = ptrace(PT_READ_I, cur_pid, vaddr, 0);
#elif defined(__BSD__)
		*p = ptrace(PT_READ_I, cur_pid, (caddr_t)vaddr, 0);
#endif
		vaddr += 4;
	}
#endif
	return (size);
}

vg_size	ptrace_write(void *buf, vg_vaddr vaddr, vg_size size)
{
	vg_uint write;

	if (!buf || vaddr<=0 || size<=0)
		VGERR(VG_DWRITE, NULL, E_INVAL, ERR);

#if defined(__linux__) || defined(__BSD__)
	for(write = 0; write < size; write += 4) {
		if (ptrace(VG_WRITE_I, cur_pid, (void*)vaddr, *(u_long*)(buf+write))<0)
			VGERR(0, INTERNAL_ERR, 0, ERR);
		vaddr += 4;
	}
#endif
	return (size);
}

vg_regs *ptrace_gregs(vg_proc *p)
{
	if (!p)
		VGERR(VG_GREGS, NULL, E_INVAL, NULL);

#if defined(__linux__)
	if (ptrace(VG_GET_REGS, p->pid, 0, &p->regs)<0)
#elif defined(__BSD__)
	if (ptrace(VG_GET_REGS, p->pid, (caddr_t)&p->regs, 0)<0)
#endif
		VGERR(0, INTERNAL_ERR, 0, NULL);
        /* make a copy of these orig regs */
	memcpy(&p->orig_regs, &p->regs, sizeof(vg_regs));
	return (&p->regs);
}

vg_regs	*ptrace_sregs(vg_proc *p, vg_regs *regs)
{
	if (!p || !p->pid || !regs)
		VGERR(VG_SREGS, NULL, E_INVAL, NULL);

#if defined(__linux__)
	if (ptrace(VG_SET_REGS, p->pid, 0, regs)<0)
#elif defined(__BSD__)
	if (ptrace(VG_SET_REGS, p->pid, (caddr_t)regs, 0)<0)
#endif
		VGERR(0, INTERNAL_ERR, 0, NULL);

	p->flags |= PROC_RG_CHANGED;
	memcpy(&p->regs, regs, sizeof(vg_regs));
	return (&p->regs);
}

vg_int	ptrace_step(vg_proc *p)
{
	if (!p || !p->pid)
		VGERR(VG_PROC_STEP, NULL, E_INVAL, ERR);

#if defined(__linux__) /* not stepping on OPENBSD */	
	if (ptrace(VG_STEP, p->pid, 0, 0)<0)
#elif defined(__FreeBSD__) ||  defined(__NetBSD__)
	if (ptrace(VG_STEP, p->pid, (caddr_t)1, 0)<0)
#endif
		VGERR(0, INTERNAL_ERR, 0,ERR);

	return (OK);
}

vg_int	ptrace_stop(vg_proc *p)
{
	if (!p && !p->pid)
		VGERR(VG_PROC_STOP, "Incorrect pid", 0, ERR);

#if defined(__linux__) || defined(__BSD__)
	if (cur_proc->pstate.status == PS_RUNNING) {
		disable_rpsm();
		/* send stop signal */
		kill(p->pid, SIGSTOP);
		waitpid(cur_pid, &cur_proc->pstate.wstatus, cur_proc->pstate.wopts);
		cur_proc->pstate.status = PS_STOPPED;
		enable_rpsm();
	}
#endif
	return (OK);
}

vg_int	ptrace_cont(vg_proc *p)
{
	if (!p || !p->pid)
		VGERR(VG_PROC_CONT, "Incorrect pid", 0, ERR);

#if defined(__linux__) || defined(__BSD__)
	/* if step flag is set, resume execution on stepping mode */
	if (cur_proc->pstate.step) {
		return (vg_proc_step(p));
	}
	if (cur_proc->pstate.status != PS_RUNNING) {
		cur_proc->pstate.status = PS_RUNNING;
		if (ptrace(PT_CONTINUE, p->pid, PT_ARG3, SIGCONT)<0)
			VGERR(0, INTERNAL_ERR, 0, ERR);
	}
#endif
	return (OK);
}



/*
 * [ Procfs API ]
 * ------------------------------------------------------------------------+
 *                                                                         |
 *	1 attach  4 write  7 cont                                          |
 *	2 detach  5 gregs  8 stop                                          |
 *	3 read    6 sregs  9 step                                          |
 *                                                                         |
 * Supported: Solaris (not finished)                                       |
 *                                                                         |
 * ------------------------------------------------------------------------+
 */

vg_int	procfs_attach(vg_pid);
vg_int	procfs_detach(vg_pid);
vg_size	procfs_read(void *, vg_vaddr, vg_size);
vg_size	procfs_write(void *, vg_vaddr, vg_size);
vg_regs	*procfs_gregs(vg_proc *);
vg_regs	*procfs_sregs(vg_proc *, vg_regs *);
vg_int	procfs_cont(vg_proc *);
vg_int	procfs_stop(vg_proc *);
vg_int	procfs_step(vg_proc *);

vg_int	setup_procfs_operations(vg_proc *p)
{
	/* initialize basic operations */
	p->op.attach	= procfs_attach;
	p->op.detach	= procfs_detach;
	p->op.read	= procfs_read;
	p->op.write	= procfs_write;
	p->op.gregs	= procfs_gregs;
	p->op.sregs	= procfs_sregs;
	p->op.cont	= procfs_cont;
	p->op.stop	= procfs_stop;
	p->op.step	= procfs_step;

	return (OK);
}

vg_int	procfs_attach(vg_pid pid)
{
	if (pid <= 0)
		VGERR(__PROC_ATTACH, "Incorrect pid", 0, ERR);

	disable_rpsm();

#if defined(__Solaris__)
	vg_char	*fname;

	/* first open devices */
	if (!(fname = (char*)malloc(MAX_BUF+1)))
		VGERR(0, INTERNAL_ERR, 0, ERR);

	memset(fname, 0, MAX_BUF+1);
	snprintf(fname, MAX_BUF, "/proc/%d/ctl", (vg_int)pid);
	if ((cur_proc->spstate.procfd_ctl = open(fname, O_WRONLY))<0)
		VGERR2(0, INTERNAL_ERR, 0, ERR, fname);

	memset(fname, 0, MAX_BUF+1);
	snprintf(fname, MAX_BUF, "/proc/%d/as", (vg_int)pid);
	if ((cur_proc->spstate.procfd_as = open(fname, O_RDONLY))<0)
		VGERR2(0, INTERNAL_ERR, 0, ERR, fname);

	memset(fname, 0, MAX_BUF+1);
	snprintf(fname, MAX_BUF, "/proc/%d/status", (vg_int)pid);
	if ((cur_proc->spstate.procfd_status = open(fname, O_RDWR))<0)
		VGERR2(0, INTERNAL_ERR,0, ERR, fname);
	free(fname);

	/* stop process */
	cur_proc->spstate.single_request = PCDSTOP;
	if (write(cur_proc->spstate.procfd_ctl, &cur_proc->spstate.single_request,
			sizeof(cur_proc->spstate.single_request))
			!= sizeof(cur_proc->spstate.single_request))
		VGERR(__PROC_ATTACH, INTERNAL_ERR, 0, ERR);

	/* set process to be traced */
	cur_proc->spstate.attach_req.func = PCSTRACE;
	prfillset(&cur_proc->spstate.attach_req.signals);
        if (write(cur_proc->spstate.procfd_ctl, &cur_proc->spstate.attach_req,
			sizeof(cur_proc->spstate.attach_req))
			!= sizeof(cur_proc->spstate.attach_req))
		VGERR(__PROC_ATTACH, INTERNAL_ERR, 0, ERR);
#endif

	/* wait for sigchld signal */
	waitpid(cur_pid, &cur_proc->pstate.wstatus, cur_proc->pstate.wopts);
	cur_proc->pstate.status = PS_STOPPED;

	enable_rpsm();

        return (OK);
}

vg_int	procfs_detach(vg_pid pid)
{
	if (pid <= 0)
		VGERR(__PROC_DETACH, "Incorrect pid", 0, ERR);

	return (OK);
}

vg_size	procfs_read(void *buf, vg_vaddr vaddr, vg_size size)
{
	if (!buf || vaddr<=0 || size<=0)
		VGERR(VG_DREAD, NULL, E_INVAL, ERR);

#if defined(__SOLARIS__)
        if (pread(cur_proc->spstate.procfd_as, buf, size, vaddr) != size)
                VGERR(0, INTERNAL_ERR, 0, ERR);
        }
#endif
        return (size);
}

vg_size	procfs_write(void *buf, vg_vaddr vaddr, vg_size size)
{
	if (!buf || vaddr<=0 || size<=0)
                VGERR(VG_DWRITE, NULL, E_INVAL, ERR);

#if defined(__SOLARIS__)
        if (pwrite(cur_proc->spstate.procfd_as, buf, size, vaddr) != size)
                VGERR(0, INTERNAL_ERR, 0, ERR);
        }
#endif
        return (size);
}

vg_regs	*procfs_gregs(vg_proc *p)
{
	if (!p)
		VGERR(VG_GREGS, NULL, E_INVAL, NULL);

#if defined(__SOLARIS__)
        /* read all status */
        if (pread(cur_proc->spstate.procfd_status, &cur_proc->spstate.stat,
                                        sizeof(pstatus_t), 0x0)<0)
                VGERR(0, INTERNAL_ERR, 0, NULL);
        return (&p->spstate.stat.pr_lwp.pr_reg);
#endif
	return (NULL);
}

vg_regs	*procfs_sregs(vg_proc *p, vg_regs *new_regs)
{
	if (!p || !new_regs)
		VGERR(VG_SREGS, NULL, E_INVAL, NULL);

#if defined(__SOLARIS__)
        memcpy(&p->spstate.stat.pr_lwp.pr_reg, new_regs, sizeof(new_regs));
	return (&p->spstate.stat.pr_lwp.pr_reg);
#endif
	return (NULL);
}

vg_int	procfs_cont(vg_proc *p)
{
        if (p->pid <= 0)
		VGERR(VG_PROC_CONT, "Incorrect pid", 0, ERR);

#if defined(__SOLARIS__)
        /* make continue request */
        cur_proc->spstate.control_req.func = PCRUN;
        cur_proc->spstate.control_req.args = PRSABORT;
        cur_proc->pstate.status = PS_RUNNING;
        if (pwrite(cur_proc->spstate.procfd_ctl, &cur_proc->spstate.control_req,
                                 sizeof(cur_proc->spstate.control_req), 0x0) !=
                                        sizeof(cur_proc->spstate.control_req))
                VGERR(0, INTERNAL_ERR, 0, ERR);
#endif
	return (OK);
}

vg_int	procfs_step(vg_proc *p)
{
       if (p->pid <= 0)
                VGERR(VG_PROC_STEP, NULL, E_INVAL, ERR);

#if defined(__SOLARIS__)
        cur_proc->spstate.control_req.func = PCRUN;
        cur_proc->spstate.control_req.args = PRSTEP;
        cur_proc->pstate.status = PS_RUNNING;
        if (pwrite(cur_proc->spstate.procfd_ctl, &cur_proc->spstate.control_req,
                                 sizeof(cur_proc->spstate.control_req), 0x0) !=
                                sizeof(cur_proc->spstate.control_req))
                VGERR(0, INTERNAL_ERR, 0,ERR);
#endif
	return (OK);
}

vg_int	procfs_stop(vg_proc *p) { return(ERR); }


/*
 * API Default Selector
 *
 * Future versions will add a runtime switcher 
 */
vg_int  setup_proc_operations(vg_proc *p)
{
#if defined(__linux__) || defined(__BSD__)
	setup_ptrace_operations(p);
#elif defined(__Solaris__)
	setup_procfs_operations(p);
#endif
	return (OK);
}



/*
 * [ VG Wrappers ]
 * ------------------------------------------------------------------------+
 *                                                                         |
 *	1 proc_attach  4 vg_write  7 vg_proc_cont                          |
 *	2 proc_detach  5 vg_gregs  8 vg_proc_stop                          |
 *	3 vg_read      6 vg_sregs  9 vg_proc_step                          |
 *                                                                         |
 * ------------------------------------------------------------------------+
 */

/* 
 * proc_attach = gain control of a process and being tracing it
 * @pid: proc id
 */
vg_int	proc_attach(vg_pid pid)
{
	return (cur_proc->op.attach(pid));
}

/* 
 * proc_detach = stop tracing a process
 * @pid: proc id
 */
vg_int	proc_detach(vg_pid pid)
{
	return (cur_proc->op.detach(pid));
}

/*
 * vg_read/vg_write = read/write data from/to the process's address 
 * space
 *
 * @buf: buffer
 * @vaddr: virtual adress
 * @size: number of bytes to read
 * #ret: number of bytes read
 *
 * Both vg_read() and vg_write() works on protected mode:
 *
 * First: they check if the supplied virtual address 
 * is part of an object (text, data or heap). 
 *
 * Second: If a match is found they check if the object is mapped
 * or not. If it is not mapped vg_read does nothing and continue with
 * next steap but vg_write try to map it.
 * 
 * Third: If the object is mapped, the virtual address
 * (remote address) will be translated to get the correct
 * local virtual address of the mapped object. To finish,
 * we will read/write data using memcpy().
 *
 * Fourth: If no match is found, vg_read() reads data
 * directly from the remote process but vg_write does
 * nothing.
 *
 */
vg_size	vg_read(void *buf, vg_vaddr vaddr, vg_size size)
{
	vg_object	*obj;
	vg_vaddr	*local_addr = NULL;
        vg_uint		seg_type = 0;

	if (!buf || vaddr<=0 || size<=0)
		VGERR(VG_READ, NULL, E_INVAL, ERR);

	/* si la direccion de memoria corresponde a un objeto valido
 	 * comprueba si esta mapeado, en caso de estarlo, traduce la 
 	 * direccion remota dada en la direccion local correspondiente,
	 * y leera los datos del buffer local
	 */
	if ((obj = vg_obj_by_vaddr(vaddr, &seg_type))) {
		if (OBJ_IS_MAPPED(obj)) {
			/* convertir direccion remota en local */
			local_addr = (vg_vaddr*)RTOL((vg_vaddr)vaddr, obj);
			/* leer usando memcpy() */
			memcpy(buf, local_addr, size);
			return (size);
		}
	}

	/* si la vaddr no esta en el rango de un objeto conocido
 	 * se lee directamente 
 	 */
	return (cur_proc->op.read(buf, vaddr, size));
}

vg_size	vg_write(void *buf, vg_vaddr vaddr, vg_size size)
{
        vg_object       *obj;
        vg_vaddr        *local_addr = 0x0;
        u_int           seg_type = 0x0;

        if (!buf || vaddr<=0 || size<=0)
                VGERR(VG_WRITE, NULL, E_INVAL, ERR);

        /* first, search the correct object for vaddr */
        if (!(obj = vg_obj_by_vaddr(vaddr, &seg_type)))
                VGERR(VG_WRITE, "Incorrect virtual address", 0, ERR);

        /* second, object referenced by obj must be mapped, check it */
        if(vg_map_object(obj)<0)
                return (ERR);

        /* third, translate the traced process's virtual address
           to a valid local virtual address */
	local_addr = (vg_vaddr*)RTOL((vg_vaddr)vaddr, obj);

        /* last, copy data */
        memcpy(local_addr, buf, size);
        SETOBJCH(obj);
	return (size);
}

/*
 * vg_dread = read data directly from the process's adress space.
 *
 */
vg_size vg_dread(void *buf, vg_vaddr vaddr, vg_size size)
{
	return (cur_proc->op.read(buf,vaddr,size));
}

/*
 * vg_dwrite = write data directly to the proecss's address space
 *
 */
vg_size vg_dwrite(void *buf, vg_vaddr vaddr, vg_size size)
{
	return (cur_proc->op.write(buf,vaddr,size));
}

/*
 * vg_proc_stop = stop execuction of the process 
 * @p: proc context
 */
vg_int	vg_proc_stop(vg_proc *p)
{
	return (p->op.stop(p));
}

/*
 * vg_proc_continue = resume execution of the process
 * @p: proc context
 */
vg_int	vg_proc_cont(vg_proc *p)
{
	return (p->op.cont(p));
}

/*
 * vg_proc_step = resume execution of the process on stepping mode
 * @p: proc context
 */
vg_int	vg_proc_step(vg_proc *p)
{
	return (p->op.step(p));
}

/*
 * vg_proc_gregs = get the process's registers and
 * store them on the pctx structure 
 * @p: proc context
 */
vg_regs	*vg_gregs(vg_proc *p)
{
	return (p->op.gregs(p));
}

/*
 * vg_proc_sregs = change the process's registers stored
 * on the pctx structure
 * @p: proc context
 */
vg_regs	*vg_sregs(vg_proc *p, vg_regs *regs)
{
	return (p->op.sregs(p,regs));
}

