/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
#include "libvg.h"

vg_proc		*cur_proc = NULL;	/* ptr to current pctx structure */
vg_pid		cur_pid	= 0;		/* system process id */
vg_int		pctx_id	= 0;		/* internal process id */
vg_vaddr        proc_base = 0x00000000; /* base vaddr */

/*
 *	[I]	Checks if the process has been 
 *		dynamically linked or not
 */
vg_int	get_ptype(vg_proc *pctx, Elf32_Ehdr *hdr, Elf32_Phdr *phdr)
{
	vg_vaddr	addr;
		
	if (vg_dread(hdr, proc_base, sizeof(Elf32_Ehdr)) != 
				sizeof(Elf32_Ehdr)) 
		VGERR(VG_INIT_PCTX,"Problem with BASE addr", 0, ERR);
	
	addr = hdr->e_phoff + proc_base;
	while(hdr->e_phnum) {
		vg_dread(phdr, addr, sizeof(Elf32_Phdr));
		if (phdr->p_type == PT_DYNAMIC) {
			/* process dinamically linked */
			pctx->flags |= PROC_WITH_RTLD;
			return (OK);
		}
		addr += sizeof(Elf32_Phdr);
		--hdr->e_phnum;
	}
	/* process statically linked */
	pctx->flags |= PROC_WITHOUT_RTLD;
	return (OK);
}

/*
 * vg_init_pctx = Allocates and initializes a new struct vg_proc
 * , process context, for @pid
 *
 * @pid: a valid process id
 * @path: executable pathname, optional
 * #ret: pointer to the struct vg_proc created
 * 
 * This is the main function, first it attachs to a process specified by @pid
 * and then try to get all information about the process in order to fill 
 * the structure vg_proc.
 * 
 */
vg_proc	*vg_init_pctx(vg_pid pid, vg_char *path)
{
	Elf32_Ehdr	main_hdr;
	Elf32_Phdr	main_phdr;
	vg_proc		*pctx = NULL;

	/* initialize a new proc ctx */
	S_ALLOC(pctx, sizeof(vg_proc), NULL, VG_INIT_PCTX, CLEAN);

	/* setup all operations */
	setup_proc_operations(pctx);
	setup_stack_operations(pctx);

	/* set global vars */
	cur_pid = pid;
	cur_proc = pctx;

	/* set process running to be attached */
	pctx->pstate.status = PS_RUNNING;

	/* set signals handlers for this process */
	load_default_sighandlers();

	if (proc_attach(pid)<0)
		VGERR2(VG_INIT_PCTX, "", KEEP_LAST, NULL, pctx);

	/* set main info */
	pctx->pid = pid;
	pctx->id = ++pctx_id;
	if (path)
		pctx->execpath = strdup(path);

	/* (a) get process type, how it was linked */
	if (get_ptype(pctx, &main_hdr, &main_phdr)<0)
		return (NULL);

	/* (b) load process's objects descriptor table */
	if (pctx->flags & PROC_WITH_RTLD)
		if (load_odt(pctx, &main_hdr, &main_phdr)<0) 
			return(NULL);

	/* (c) load process's objects table */
	if (load_objects_table(pctx->obj_table, pctx->obj_desc_table)<0)
		return (NULL);

	/* save current registers from attached process */
	vg_gregs(pctx);

	/* get stack info */
	if (vg_get_stack()<0)
		return (NULL);

	/* set default values */
	pctx->flags |= PROC_INUSE;
	pctx->flags |= PROC_PROTECTED_MODE;

	/* return ptr */
	return ((vg_proc*)pctx);
}

/*
 * vg_fini_pctx = 
 *
 * @pctx:
 * #ret:
 *
 */
vg_int	vg_fini_pctx(vg_proc *pctx)
{
	/* (a) unload object descriptors table */
	if (pctx->flags & PROC_WITH_RTLD)
		unload_odt(pctx);

	/* (b) unload objects table */
	unload_objects_table(pctx->obj_table);

	/* (c) clean enabled breakpoints */
//	vg_bkpt_clear(&proc->init_bkpt);

	/* detaching */
	if (proc_detach(pctx->pid)<0)
		VGERR(VG_FINI_PCTX, 0, KEEP_LAST, ERR);

	/* freeing */
	if (pctx->execpath)
		free(pctx->execpath);
	free(pctx);

	return (OK);
}

/*
 * vg_save_pctx: Save changes on mapped objects, this change the orig
 * process
 *
 * @pctx:
 * #ret:
 *
 */  
vg_int	vg_save_pctx(vg_proc *p)
{
#if defined(__OpenBSD__) || defined(__linux__)
	vg_obj_desc	entry;
#else
	bsd_obj_ref	fentry;
#endif
	u_long		orig_addr;
	int		idx = 0;

	if (!p) 
		VGERR(VG_SAVE_PCTX, "Incorrect args", 0, ERR);

	/* (A) save ref list if it has changed */
	if (p->flags & PROC_WITH_RTLD && p->flags & PROC_OBJD_CHANGED) {
		orig_addr = p->reflist_addr;
#if defined(__OpenBSD__) || defined(__linux__)
		while(p->obj_desc_table[idx]) {
			/* get orig entry */
			if (vg_dread(&entry, orig_addr, sizeof(vg_obj_desc)) 
						!= sizeof(vg_obj_desc))
				return (ERR);				

			/* substitute it */
			if (p->obj_desc_table[idx]->od_base != INCORRECT_ENTRY ) {
				entry.od_base = p->obj_desc_table[idx]->od_base;
				entry.od_dyn = p->obj_desc_table[idx]->od_dyn;
				entry.od_next = p->obj_desc_table[idx]->od_next;
				entry.od_prev = p->obj_desc_table[idx]->od_prev;
				vg_dwrite(&entry, orig_addr, sizeof(vg_obj_desc));
				vg_dwrite(&p->obj_desc_table[idx]->od_name, (vg_vaddr)entry.od_name, 
						strlen(p->obj_desc_table[idx]->od_str)+1);
			}
			/* get next real remote entry */
			vg_dread(&entry, orig_addr, sizeof(vg_obj_desc));
			orig_addr = (u_long)entry.od_next;
			++idx;
		}				
#elif defined(__FreeBSD__) || defined(__NetBSD__)
		while(p->obj_desc_table[idx]) {
			/* pillar entrada original */
			if (vg_dread(&fentry, orig_addr, sizeof(bsd_obj_ref))
						!= sizeof(bsd_obj_ref))
				return (ERR);

			/* modificar fentry con los elem locales y volver a copiar */
			fentry.next = (u_long)p->obj_desc_table[idx]->od_next;
			fentry.mapbase = p->obj_desc_table[idx]->od_base;
			fentry.dynamic = (u_long)p->obj_desc_table[idx]->od_dyn;
			vg_dwrite(&fentry, orig_addr, sizeof(bsd_obj_ref));
			/* copiar local name */
			vg_dwrite(p->obj_desc_table[idx]->od_str, (u_long)fentry.path,
						strlen(p->obj_desc_table[idx]->od_str+1));

			/* next local and remote entry */
			vg_dread(&fentry, orig_addr, sizeof(bsd_obj_ref));
			orig_addr = (u_long)fentry.next;
			++idx;
		}
#endif
	}

	/* (B) saving changed objects */
	for (idx = 0; p->obj_table[idx]; idx++) {
		/* avoid non mapped objects */
		if (!(p->obj_table[idx]->flags & OBJ_MAPPED))
			continue;
		/* avoid non changed mapped objects, if exists */
	//	if (!(context->obj_table[idx]->flags & OBJ_CHANGED))
	//		continue;

		/* save text segment */
		if (vg_dwrite(p->obj_table[idx]->text.ptr, 
				p->obj_table[idx]->text.start, 
				p->obj_table[idx]->text.size) !=	
			p->obj_table[idx]->text.size)
			VGERR(VG_SAVE_PCTX , 0, 0, ERR);
		/* save data segment */
		if (vg_dwrite(p->obj_table[idx]->data.ptr,
				p->obj_table[idx]->data.start,
				p->obj_table[idx]->data.size) != 
			p->obj_table[idx]->data.size) 
			VGERR(VG_SAVE_PCTX, 0, 0, ERR);
	}
	/* (C) save regs if it has changed */
	if (p->flags & PROC_RG_CHANGED)
		vg_sregs(p, &p->regs);
	return (OK);
}


/*
 * vg_proc_get_id = get internal id for this process
 *
 */
vg_id	vg_proc_get_id(vg_proc *pctx)
{
	if (!pctx)
		VGERR(VG_PROC_GET_ID, NULL, E_INVAL, ERR);
	return (pctx->id);
}

/*
 * vg_proc_get_pid = get system id for this process
 *
 */
vg_pid	vg_proc_get_pid(vg_proc *pctx)
{
	if (!pctx)
		VGERR(VG_PROC_GET_PID, NULL, E_INVAL, ERR);
	return (pctx->pid);
}

/*
 * vg_proc_get_execpath = get the execpath or name of this process
 *
 */
vg_char *vg_proc_get_execpath(vg_proc *pctx)
{
	if (!pctx)
		VGERR(VG_PROC_GET_EXECPATH, NULL, E_INVAL, NULL);
	return (pctx->execpath);
}

/*
 * vg_proc_get_objnum = get the number of objects for this process
 *
 */
vg_idx	vg_proc_get_objnum(vg_proc *pctx)
{
	if (!pctx)
		VGERR(VG_PROC_GET_OBJNUM, NULL, E_INVAL, ERR);
	return (pctx->objnum);
}

/*
 * vg_proc_get_flags = get process's flags
 *
 */
vg_flags vg_proc_get_flags(vg_proc *pctx)
{
	if (!pctx)
		VGERR(VG_PROC_GET_FLAGS, NULL, E_INVAL, ERR);
	return (pctx->flags);
}

/* 
 * vg_proc_get_next/prev = get previous/next process
 * being traced
 */
vg_proc	*vg_proc_get_next(vg_proc *pctx)
{
	if (!pctx)
		VGERR(VG_PROC_GET_NEXT, NULL, E_INVAL, NULL);
	return (pctx->next);
}

vg_proc	*vg_proc_get_prev(vg_proc *pctx)
{
	if (!pctx)
		VGERR(VG_PROC_GET_PREV, NULL, E_INVAL, NULL);
	return (pctx->prev);
}

