/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
#include "libvg.h"

/*
 * vg_obj_get_relplt = load relplt section
 */
Elf32_Rel *vg_obj_get_relplt(vg_object *obj, vg_uint *entnum)
{
	Elf32_Dyn	*relplt, *relsz;

	CHECKDYN(VG_OBJ_GET_RELPLT, NULL);
	if (!obj)
		VGERR(VG_OBJ_GET_RELPLT, NULL, E_INVAL, NULL);

	if (obj->elf.relplt.data)
		goto end;

	if (!vg_obj_get_dyn(obj, NULL))
		return (NULL);

	if (!(relplt = vg_obj_dentry_by_type(obj, DT_JMPREL, 0)))
		VGERR(VG_OBJ_GET_RELPLT, ".relplt table not found", 0, NULL);
	if (!(relsz = vg_obj_dentry_by_type(obj, DT_PLTRELSZ, 0)))
		VGERR(VG_OBJ_GET_RELPLT, ".relplt table size not found", 0, NULL);

	SET_OBJECT_VADDR(relplt->d_un.d_ptr, obj);

	/* get data */
	if (OBJ_IS_MAPPED(obj)) {
		obj->elf.relplt.data = (Elf32_Rel*)RTOL((u_long)relplt->d_un.d_ptr, obj);
	}
	else {
		S_ALLOC(obj->elf.relplt.data, relsz->d_un.d_ptr, NULL, VG_OBJ_GET_RELPLT, CLEAN);
		if (vg_dread(obj->elf.relplt.data, relplt->d_un.d_ptr,
				 relsz->d_un.d_ptr) != relsz->d_un.d_ptr)
			VGERR2(VG_OBJ_GET_RELPLT, 0,0,NULL, obj->elf.relplt.data);
	}
	obj->elf.relplt.size = relsz->d_un.d_ptr;
	obj->elf.relplt.vaddr = relplt->d_un.d_ptr;
	obj->elf.relplt.offset = obj->elf.relplt.vaddr - obj->text.start;
	obj->elf.relplt.entsize = sizeof(Elf32_Rel);
	obj->elf.relplt.entnum = obj->elf.relplt.size / obj->elf.relplt.entsize;
end:
	if (entnum)
		*entnum = obj->elf.relplt.entnum;
	return (obj->elf.relplt.data);
}

/*
 * vg_obj_relplt_get/set_vaddr =
 */
vg_vaddr vg_obj_relplt_get_vaddr(vg_object *obj)
{
	CHECKDYN(VG_OBJ_RELPLT_GET_VADDR, ERR);
	if (!obj || !obj->elf.relplt.data)
		VGERR(VG_OBJ_RELPLT_GET_VADDR, NULL, E_INVAL, ERR);
	return (obj->elf.relplt.vaddr);
}
	
vg_vaddr vg_obj_relplt_set_vaddr(vg_object *obj, vg_vaddr vaddr)
{
	CHECKDYN(VG_OBJ_RELPLT_SET_VADDR, ERR);
	if (!obj || !obj->elf.relplt.data)
		VGERR(VG_OBJ_RELPLT_SET_VADDR, NULL, E_INVAL, ERR);
	obj->elf.relplt.vaddr = vaddr;
	return (obj->elf.relplt.vaddr);
}

/*
 * vg_obj_relplt_get/set_size = 
 */
vg_size	vg_obj_relplt_get_size(vg_object *obj)
{
	CHECKDYN(VG_OBJ_RELPLT_GET_SIZE, ERR);
	if (!obj || !obj->elf.relplt.data)
		VGERR(VG_OBJ_RELPLT_GET_SIZE, NULL, E_INVAL, ERR);
	return (obj->elf.relplt.size);
}

vg_size	vg_obj_relplt_set_size(vg_object *obj, vg_size size)
{
	CHECKDYN(VG_OBJ_RELPLT_SET_SIZE, ERR);
	if (!obj || !obj->elf.relplt.data)
		VGERR(VG_OBJ_RELPLT_SET_SIZE, NULL, E_INVAL, ERR);
	obj->elf.relplt.size = size;
	return (obj->elf.relplt.size);
}

/* 
 * vg_obj_relplt_get/set_offset = 
 */
vg_off vg_obj_relplt_get_offset(vg_object *obj)
{
	CHECKDYN(VG_OBJ_RELPLT_GET_OFFSET, ERR);
	if (!obj || !obj->elf.relplt.data)
		VGERR(VG_OBJ_RELPLT_GET_OFFSET, NULL, E_INVAL, ERR);
	return (obj->elf.relplt.offset);
}

vg_off	vg_obj_relplt_set_offset(vg_object *obj, vg_off offset)
{
	CHECKDYN(VG_OBJ_RELPLT_SET_OFFSET, ERR);
	if (!obj || !obj->elf.relplt.data)
		VGERR(VG_OBJ_RELPLT_SET_OFFSET, NULL, E_INVAL, ERR);
	obj->elf.relplt.offset = offset;
	return (obj->elf.relplt.offset);
}

/*
 * vg_obj_relplt_get/set_entsize = 
 */
vg_size	vg_obj_relplt_get_entsize(vg_object *obj)
{
	CHECKDYN(VG_OBJ_RELPLT_GET_ENTSIZE, ERR);
	if (!obj || !obj->elf.relplt.data)
		VGERR(VG_OBJ_RELPLT_GET_ENTSIZE, NULL, E_INVAL, ERR);
	return (obj->elf.relplt.entsize);
}

vg_size	vg_obj_relplt_set_entsize(vg_object *obj, vg_idx entsize)
{
	CHECKDYN(VG_OBJ_RELPLT_SET_ENTSIZE, ERR);
	if (!obj || !obj->elf.relplt.data)
		VGERR(VG_OBJ_RELPLT_SET_ENTSIZE, NULL, E_INVAL, ERR);
	obj->elf.relplt.entsize = entsize;
	return (obj->elf.relplt.entsize);
}

/*
 * vg_obj_relplt_get/set_entnum = 
 */
vg_idx	vg_obj_relplt_get_entnum(vg_object *obj)
{
	CHECKDYN(VG_OBJ_RELPLT_GET_ENTNUM, ERR);
	if (!obj || !obj->elf.relplt.data)
		VGERR(VG_OBJ_RELPLT_GET_ENTNUM, NULL, E_INVAL, ERR);
	return (obj->elf.relplt.entnum);
}

vg_idx	vg_obj_relplt_set_entnum(vg_object *obj, vg_idx entnum)
{
	CHECKDYN(VG_OBJ_RELPLT_SET_ENTNUM, ERR);
	if (!obj || !obj->elf.relplt.data)
		VGERR(VG_OBJ_RELPLT_SET_ENTNUM, NULL, E_INVAL, ERR);
	obj->elf.relplt.entnum = entnum;
	return (obj->elf.relplt.entnum);
}


/*
 * vg_obj_relplt_rentry_by_idx = return a relocation entry giving its index
 * into the relplt table
 */
Elf32_Rel *vg_obj_relplt_rentry_by_idx(vg_object *obj, vg_idx idx)
{
	CHECKDYN(VG_OBJ_RELPLT_RENTRY_BY_IDX, NULL);
	if (!obj)
		VGERR(VG_OBJ_RELPLT_RENTRY_BY_IDX, NULL, E_INVAL, NULL);
	
	if (!vg_obj_get_relplt(obj, 0))
		return (NULL);

	if (idx<0 || idx > obj->elf.relplt.entnum)
		VGERR(VG_OBJ_RELPLT_RENTRY_BY_IDX, "Incorrect index", 0, NULL);

	return ((idx? obj->elf.relplt.data+idx: obj->elf.relplt.data));
}

/*
 * vg_obj_relplt_rentry_by_sym = return a relocation entry giving 
 * a symbol name, it is usefull for searching dynamic/global symbols
 *
 */
Elf32_Rel *vg_obj_relplt_rentry_by_sym(vg_object *obj, vg_char *sname)
{
	Elf32_Rel	*rentry;
	Elf32_Sym	*dsym;
	vg_idx		idx;

	CHECKDYN(VG_OBJ_RELPLT_RENTRY_BY_SYM, NULL);
	if (!obj || !sname)
		VGERR(VG_OBJ_RELPLT_RENTRY_BY_SYM, NULL, E_INVAL, NULL);

	if (!vg_obj_get_relplt(obj, 0))
		return (NULL);

	for (idx = 0; idx < vg_obj_relplt_get_entnum(obj); idx++) {
		rentry = vg_obj_relplt_rentry_by_idx(obj, idx);
		if (!rentry)
			continue;

		dsym = vg_obj_dsym_by_idx(obj,
			vg_obj_rentry_get_info_symidx(obj, rentry));
		if (!dsym)
			continue;
		
		/* ok, this symbol has a correct relplt entry */
		return (rentry);
	}

	return (NULL);
}

/*
 * vg_obj_relplt_rentry_by_symidx = return a relocation entry giving its
 * symbol index
 */
Elf32_Rel *vg_obj_relplt_rentry_by_symidx(vg_object *obj, vg_idx symidx)
{
	Elf32_Rel	*rentry;
	vg_idx		idx;

	CHECKDYN(VG_OBJ_RELPLT_RENTRY_BY_SYMIDX, NULL);

	if (!obj || symidx<0)
		VGERR(VG_OBJ_RELPLT_RENTRY_BY_SYMIDX, NULL, E_INVAL, NULL);

	if (!vg_obj_get_relplt(obj,0))
		return (NULL);

	for (idx = 0, rentry = obj->elf.relplt.data; 
			idx < obj->elf.relplt.entnum;
				++idx, ++rentry) {
		if (vg_obj_rentry_get_info_symidx(obj, rentry) == symidx)
			return (rentry);
	}

	return (NULL);
}

/*
 * vg_obj_relplt_rentry_by_offset = return a relocation entry giving its offset
 * or virtual address
 */
Elf32_Rel *vg_obj_relplt_rentry_by_offset(vg_object *obj, vg_off offset)
{
	Elf32_Rel	*rentry;
	vg_idx		idx;

	CHECKDYN(VG_OBJ_RELPLT_RENTRY_BY_OFFSET, NULL);

	if (!obj || offset<0) 
		VGERR(VG_OBJ_RELPLT_RENTRY_BY_OFFSET, NULL, E_INVAL, NULL);

	if (!vg_obj_get_relplt(obj, 0))
		return (NULL);

	for(idx = 0, rentry = obj->elf.relplt.data;
			idx < obj->elf.relplt.entnum;
				++idx, ++rentry) {
		if (vg_obj_rentry_get_offset(obj, rentry) == offset)
			return (rentry);
	}

	return (NULL);
}

/*
 * vg_obj_rentry_get/set_offset = return/change the offset of the relplt
 * entry supplied
 */
Elf32_Addr vg_obj_rentry_get_offset(vg_object *obj, Elf32_Rel *rentry)
{
	CHECKDYN(VG_OBJ_RENTRY_GET_OFFSET, ERR);
	if (!obj || (!obj->elf.relplt.data && !obj->elf.reldyn.data) || !rentry)
		VGERR(VG_OBJ_RENTRY_GET_OFFSET, NULL, E_INVAL, ERR);
	return (rentry->r_offset);
}

Elf32_Addr vg_obj_rentry_set_offset(vg_object *obj, Elf32_Rel *rentry, Elf32_Addr offset)
{
	CHECKDYN(VG_OBJ_RENTRY_SET_OFFSET, ERR);
	if (!obj || (!obj->elf.relplt.data && !obj->elf.reldyn.data) || !rentry)
		VGERR(VG_OBJ_RENTRY_SET_OFFSET, NULL, E_INVAL, ERR);
	rentry->r_offset = offset;
	SETOBJCH(obj);
	return (rentry->r_offset);
}

/*
 * vg_obj_rentry_get/set_info = return/change info of the relplt entry
 * supplied
 */
Elf32_Word vg_obj_rentry_get_info(vg_object *obj, Elf32_Rel *rentry)
{
	CHECKDYN(VG_OBJ_RENTRY_GET_INFO, ERR);
	if (!obj || (!obj->elf.relplt.data && !obj->elf.reldyn.data) || !rentry)
		VGERR(VG_OBJ_RENTRY_GET_INFO, NULL, E_INVAL, ERR);
	return (rentry->r_info);
}

Elf32_Word vg_obj_rentry_set_info(vg_object *obj, Elf32_Rel *rentry, Elf32_Word info)
{
	CHECKDYN(VG_OBJ_RENTRY_SET_INFO, ERR);
	if (!obj || (!obj->elf.relplt.data && !obj->elf.reldyn.data) || !rentry)	
		VGERR(VG_OBJ_RENTRY_SET_INFO, NULL, E_INVAL, ERR);
	rentry->r_info = info;
	SETOBJCH(obj);
	return (rentry->r_info);
}

/*
 * vg_obj_rentry_get/set_info_symidx = return/change the symtab idx of the 
 * relplt entry supplied
 *
 */
Elf32_Word vg_obj_rentry_get_info_symidx(vg_object *obj, Elf32_Rel *rentry)
{
	CHECKDYN(VG_OBJ_RENTRY_GET_INFO_SYMIDX, ERR);
	if (!obj || (!obj->elf.relplt.data && !obj->elf.reldyn.data) || !rentry)
		VGERR(VG_OBJ_RENTRY_GET_INFO_SYMIDX, NULL, E_INVAL, ERR);
	return (ELF32_R_SYM(rentry->r_info));
}

Elf32_Word vg_obj_rentry_set_info_symidx(vg_object *obj, Elf32_Rel *rentry, vg_idx symidx)
{
	CHECKDYN(VG_OBJ_RENTRY_SET_INFO_SYMIDX, ERR);
	if (!obj || (!obj->elf.relplt.data && !obj->elf.reldyn.data) || !rentry)
		VGERR(VG_OBJ_RENTRY_SET_INFO_SYMIDX, NULL, E_INVAL, ERR);
	rentry->r_info = VG_SET_RSYM(rentry->r_info, symidx);
	SETOBJCH(obj);
	return (ELF32_R_SYM(rentry->r_info));
}

/*
 * vg_obj_rentry_get/set_info_type = return/change the relocation type of the 
 * relplt entry supplied
 *
 */
Elf32_Word vg_obj_rentry_get_info_type(vg_object *obj, Elf32_Rel *rentry)
{
	CHECKDYN(VG_OBJ_RENTRY_GET_INFO_TYPE, ERR);
	if (!obj || (!obj->elf.relplt.data && !obj->elf.reldyn.data) || !rentry)
		VGERR(VG_OBJ_RENTRY_GET_INFO_TYPE, NULL, E_INVAL, ERR);
	return (ELF32_R_TYPE(rentry->r_info));
}

Elf32_Word vg_obj_rentry_set_info_type(vg_object *obj, Elf32_Rel *rentry, vg_uchar type)
{
	CHECKDYN(VG_OBJ_RENTRY_SET_INFO_TYPE, ERR);
	if (!obj || (!obj->elf.relplt.data && !obj->elf.reldyn.data) || !rentry)
		VGERR(VG_OBJ_RENTRY_SET_INFO_TYPE, NULL, E_INVAL, ERR);
	rentry->r_info = VG_SET_RTYPE(rentry->r_info, type);
	SETOBJCH(obj);
	return (ELF32_R_TYPE(rentry->r_info));
}

