/*

 [=-- LIBVG --=] Runtime Process Manipulation Library [=------------- v0.3 -]
 [--------------------------------------------------------------------------]
 [-- Copyright (c) 2004 Pluf ---------------------------< 7a69ezine.org >---]

 THIS SOFTWARE COMES WITH ABSOLUTELY NO WARRANTY, USE AT YOUR OWN RISK

*/
#include "libvg.h"

extern  vg_proc *cur_proc;

#define	FUNC_LOCAL	1
#define	FUNC_GLOBAL	2
#define	VAR_LOCAL	3

#ifndef STT_COMMON
#define	STT_COMMON	STT_OBJECT
#endif

/* build the vg_sym structure for the symbol that has been located */
vg_int	build_vgsym(Elf32_Sym *elfs, vg_sym *sym, vg_object *obj, vg_idx symidx, vg_int type)
{
	memset(sym, 0, sizeof(vg_sym));

	sym->soname 	 = vg_obj_get_name(obj);
	sym->syname 	 = vg_obj_dsym_get_strname(obj, elfs);
	sym->obj  	 = obj;
	sym->size	 = elfs->st_size;	
	sym->stridx	 = elfs->st_name;
	sym->symtab_idx	 = symidx;
	sym->info	 = elfs->st_info;

	if (type == FUNC_LOCAL) {
		sym->vaddr	= GET_TEXTSEG_VADDR(elfs->st_value, obj);
		sym->offset	= GET_TEXTSEG_OFFSET(elfs->st_value, obj);
	}

	if (type == VAR_LOCAL) {
		sym->vaddr 	= GET_DATASEG_VADDR(elfs->st_value, obj);
		sym->offset 	= GET_DATASEG_OFFSET(elfs->st_value, obj);
	}
	
	return (1);
}

/*
 * vg_sym_func_local_by_idx = locate a valid local function giving its
 * index into the object.
 *
 * @obj: object
 * @sym: vg symbol
 * @idx: idx into symtab
 * #ret: 1 located, -1 not located
 */
vg_int	vg_sym_func_local_by_idx(vg_object *obj, vg_sym *sym, vg_idx idx)
{
	Elf32_Sym	*elfs;
	
	if (!obj || !sym)
		VGERR(VG_SYM_FUNC_LOCAL_BY_IDX, NULL, E_INVAL, ERR);

	if (idx<0 || idx>vg_obj_dynsym_get_entnum(obj))
		VGERR(VG_SYM_FUNC_LOCAL_BY_IDX, NULL, E_INVAL, ERR);

	if (!(elfs = vg_obj_dsym_by_idx(obj, idx)))
		return (ERR);

	/* 
	 * A valid local symbol need (for main object and shared object):
	 * st_value != 0 
	 * st_info = STT_FUNC
	 * no relplt entry 
	 */
	if (!elfs->st_value || ELF32_ST_TYPE(elfs->st_info) != STT_FUNC ||
				vg_obj_relplt_rentry_by_symidx(obj,idx))
		VGERR(VG_SYM_FUNC_LOCAL_BY_IDX, "The sym given is not a local function", 0, ERR);

	return (build_vgsym(elfs, sym, obj, idx, FUNC_LOCAL));
}

/*
 * vg_sym_func_local_by_name = this try to locate a valid local funcion 
 * giving its name, there are two ways to do it:
 * Local searching: if @obj is != 0 the symbol will be searched into this object only.
 * Global searching: if @obj = 0 the symbol will be searched into each object of the process
 * 
 * @obj: object structure or 0
 * @sym: vg symbol
 * @name: symbol name
 * #ret: 1 ok, -1 not ok
 */
vg_int	vg_sym_func_local_by_name(vg_object *obj, vg_sym *sym, vg_char *name)
{
	Elf32_Sym	*elfs;
	vg_idx		idx, hidx, sym_idx = 0;

	if (!sym || !name)
		VGERR(VG_SYM_FUNC_LOCAL_BY_NAME, NULL, E_INVAL, ERR);

	if (obj) {
		if ((elfs = vg_obj_dsym_by_name(obj, name)) && elfs->st_value &&
				     ELF32_ST_TYPE(elfs->st_info) == STT_FUNC &&
				     !vg_obj_relplt_rentry_by_sym(obj,name))
			return (build_vgsym(elfs, sym, obj, sym_idx, FUNC_LOCAL));
	}
	else {
		hidx = vg_symbol_hash(name);
		for (idx = 0; cur_proc->obj_table[idx]; ++idx) {
			if ((elfs = get_sym_in_hash(cur_proc->obj_table[idx], name, &sym_idx, hidx))
				&& elfs->st_value && ELF32_ST_TYPE(elfs->st_info) == STT_FUNC &&
				     !vg_obj_relplt_rentry_by_sym(obj,name)) {
				return (build_vgsym(elfs, sym, cur_proc->obj_table[idx], sym_idx, FUNC_LOCAL));
			}
		}
	}

	return (ERR);		
}

/* TODO 
 * vg_sym_func_local_by_vaddr = giving a valid local function giving its 
 * starting virtual address
 */
vg_int	vg_sym_func_local_by_vaddr(vg_object *obj, vg_sym *sym, vg_vaddr vaddr)
{
	return (ERR);
}

/*
 * vg_sym_func_global_by_idx = locate a valid global function giving its index
 * into the object
 * 
 */
vg_int	vg_sym_func_global_by_idx(vg_object *obj, vg_sym *sym, vg_idx idx)
{
	Elf32_Sym	*elfs;
	vg_idx		sym_idx = 0;

	if (!obj || !sym)
		VGERR(VG_SYM_FUNC_GLOBAL_BY_IDX, NULL, E_INVAL, ERR);		

	if (idx<0 || idx>vg_obj_dynsym_get_entnum(obj))
		VGERR(VG_SYM_FUNC_GLOBAL_BY_IDX, NULL, E_INVAL, ERR);

	if (!(elfs = vg_obj_dsym_by_idx(obj, idx)))
		return (ERR);

	/*
	 * A valid global function need (main object and shared object):
 	 * st_value = 0 or st_value != 0, this can change
	 * st_type = STT_FUNC or STT_COMMON, or STT_NOTYPE (bsd)
 	 * >> has relplt entry (main proof :P)
	 */
	if (!vg_obj_relplt_rentry_by_symidx(obj,idx) || 
			(ELF32_ST_TYPE(elfs->st_info) != STT_FUNC &&
			(ELF32_ST_TYPE(elfs->st_info) != STT_NOTYPE)))
		VGERR(VG_SYM_FUNC_GLOBAL_BY_IDX, "The sym is not a global function", 0, ERR);

	return (build_vgsym(elfs, sym, obj, sym_idx, FUNC_GLOBAL));
}

/*
 * vg_sym_func_global_by_name = locate a valid global function giving its name
 *
 */
vg_int	vg_sym_func_global_by_name(vg_object *obj, vg_sym *sym, vg_char *name)
{
	Elf32_Sym	*elfs;
	vg_idx		sym_idx = 0;

	if (!obj || !sym || !name)
		VGERR(VG_SYM_FUNC_GLOBAL_BY_NAME, NULL, E_INVAL, ERR);

	if (!(elfs = vg_obj_dsym_by_name(obj, name))) 
		return (ERR);

	if (!vg_obj_relplt_rentry_by_sym(obj,name) || 
			(ELF32_ST_TYPE(elfs->st_info) != STT_FUNC &&
			(ELF32_ST_TYPE(elfs->st_info) != STT_NOTYPE)))
		VGERR(VG_SYM_FUNC_GLOBAL_BY_NAME, "The sym is not a global function", 0, ERR);

	return (build_vgsym(elfs, sym, obj, sym_idx, FUNC_GLOBAL));
}

/* 
 * TODO
 * vg_sym_func_global_by_vaddr = locate a valid global function giving its
 * starting virtual address
 */
vg_int	vg_sym_func_global_by_vaddr(vg_object *obj, vg_sym *sym, vg_vaddr vaddr)
{
	return (0);
} 


/* 
 * vg_sym_var_local_by_idx = locate a local variable giving its index
 * into the dynamic symbol table
 *
 */
vg_int	vg_sym_var_local_by_idx(vg_object *obj, vg_sym *sym, vg_idx idx)
{
	Elf32_Sym	*elfs;
	vg_idx		sym_idx = 0;

	if (!obj || !sym)
		VGERR(VG_SYM_VAR_LOCAL_BY_IDX, NULL, E_INVAL, ERR);

	if (idx<0 || idx>vg_obj_dynsym_get_entnum(obj))
		VGERR(VG_SYM_FUNC_LOCAL_BY_IDX, NULL, E_INVAL, ERR);

	if (!(elfs = vg_obj_dsym_by_idx(obj, idx)))
		return (ERR);

	/*
 	 * A valid local variable need (for main object and shared object):
	 * st_value != 0
	 * st_info == STT_OBJECT or STT_COMMON
	 * no plt entry
	 */
	if (!elfs->st_value || (ELF32_ST_TYPE(elfs->st_info) != STT_OBJECT &&
				ELF32_ST_TYPE(elfs->st_info) != STT_COMMON) ||
			vg_obj_reldyn_rentry_by_symidx(obj,idx))
		VGERR(VG_SYM_VAR_LOCAL_BY_IDX, "The sym is not a local variable", 0, ERR);

	return (build_vgsym(elfs, sym, obj, sym_idx, VAR_LOCAL));
}

/*
 * vg_sym_var_local_by_name = locate a local variable giving its name
 *
 */
vg_int	vg_sym_var_local_by_name(vg_object *obj, vg_sym *sym, vg_char *name)
{
	Elf32_Sym	*elfs;
	vg_idx		idx, hidx, sym_idx = 0;

	if (!sym || !name)
		VGERR(VG_SYM_VAR_LOCAL_BY_NAME, NULL, E_INVAL, ERR);
	
	if (obj) {
		if ((elfs = vg_obj_dsym_by_name(obj, name)) && elfs->st_value &&
				(ELF32_ST_TYPE(elfs->st_info) == STT_OBJECT ||
				 ELF32_ST_TYPE(elfs->st_info) == STT_COMMON) &&
				!vg_obj_reldyn_rentry_by_sym(obj,name))
			return (build_vgsym(elfs, sym, obj, sym_idx, VAR_LOCAL));
	}
	else {
		hidx = vg_symbol_hash(name);
		for (idx = 0; cur_proc->obj_table[idx]; ++idx) {
			if ((elfs = get_sym_in_hash(cur_proc->obj_table[idx], name, &sym_idx, hidx)) &&
				elfs->st_value && (ELF32_ST_TYPE(elfs->st_info) == STT_FUNC || 
						   ELF32_ST_TYPE(elfs->st_info) == STT_COMMON) &&
					!vg_obj_reldyn_rentry_by_sym(obj,name))
				return (build_vgsym(elfs, sym, cur_proc->obj_table[idx], sym_idx, VAR_LOCAL));
		}
	}

	return (ERR);
}

/*
 * TODO
 * vg_sym_var_local_by_vaddr = locate a local variable giving its starting
 * virtual address
 */
vg_int	vg_sym_var_local_by_vaddr(vg_object *obj, vg_sym *sym, vg_vaddr vaddr)
{
	return (ERR);
}

/*
 * vg_sym_var_global_by_idx = locate a global variable giving its index into
 * the dynamic symbol table
 *
 */
vg_int	vg_sym_var_global_by_idx(vg_object *obj, vg_sym *sym, vg_idx idx)
{
	Elf32_Sym	*elfs;
	vg_idx		sym_idx = 0;

	if (!obj || !sym)
		VGERR(VG_SYM_VAR_GLOBAL_BY_IDX, NULL, E_INVAL, ERR);

	if (idx<0 || idx>vg_obj_dynsym_get_entnum(obj))
		VGERR(VG_SYM_VAR_GLOBAL_BY_IDX, NULL, E_INVAL, ERR);
	
	if (!(elfs = vg_obj_dsym_by_idx(obj, idx)))
		return (ERR);
	
	/*
 	 * A valid global variable need:
	 * has plt entry and type == STT_OBJECT or STT_COMMON
	 */
	if (!vg_obj_reldyn_rentry_by_symidx(obj,idx) ||
			      (ELF32_ST_TYPE(elfs->st_info) != STT_OBJECT &&
			       ELF32_ST_TYPE(elfs->st_info) != STT_COMMON))
		VGERR(VG_SYM_VAR_GLOBAL_BY_IDX, "The sym is not a global variable", 0, ERR);

	return (build_vgsym(elfs, sym, obj, sym_idx, 0));
}

/*
 * vg_sym_var_global_by_name = 
 *
 */
vg_int	vg_sym_var_global_by_name(vg_object *obj, vg_sym *sym, vg_char *name)
{
	Elf32_Sym	*elfs;
	vg_idx		sym_idx = 0;

	if (!obj || !sym || !name)
		VGERR(VG_SYM_VAR_GLOBAL_BY_NAME, NULL, E_INVAL, ERR);

	if (!(elfs = vg_obj_dsym_by_name(obj, name)))
		return (ERR);

	if (!vg_obj_reldyn_rentry_by_sym(obj,name) |
			      (ELF32_ST_TYPE(elfs->st_info) != STT_OBJECT &&
			       ELF32_ST_TYPE(elfs->st_info) != STT_COMMON))
		VGERR(VG_SYM_VAR_GLOBAL_BY_IDX, "The sym is not a global variable", 0, ERR);

	return (build_vgsym(elfs, sym, obj, sym_idx, 0));
} 

/* TODO
 * vg_sym_var_global_by_vaddr
 */
vg_int	vg_sym_var_global_by_vaddr(vg_object *obj, vg_sym *sym, vg_vaddr vaddr)
{
	return (ERR);
}

/*
 * vg_sym_section_by_name = locate a SECTION symbol giving its name into
 * the object
 *
 * @obj:
 * @sym:
 * @name:
 * #ret:
 */
vg_int	vg_sym_section_by_name(vg_object *obj, vg_sym *sym, vg_char *name)
{
	Elf32_Sym	*elfs;
	vg_idx		sym_idx = 0;

	if (!obj || !sym || !name)
		VGERR(VG_SYM_SECTION_BY_NAME, NULL, E_INVAL, ERR);

	if (!(elfs = vg_obj_dsym_by_name(obj, name)))
		return (ERR);

	if (!elfs->st_value || ELF32_ST_TYPE(elfs->st_info) != STT_SECTION)
		VGERR(VG_SYM_SECTION_BY_NAME, NULL, 0, ERR);

	return (build_vgsym(elfs, sym, obj, sym_idx, 0));
}

/* 
 * vg_sym_file_by_name =
 *
 * @obj:
 * @sym:
 * @name:
 * #ret:
 */
vg_int	vg_sym_file_by_name(vg_object *obj, vg_sym *sym, vg_char *name)
{
	Elf32_Sym	*elfs;
	vg_idx		sym_idx = 0;

	if (!obj || !sym || !name)
		VGERR(VG_SYM_FILE_BY_NAME, NULL, E_INVAL, ERR);

	if (!(elfs = vg_obj_dsym_by_name(obj, name)))
		return (ERR);

	if (!elfs->st_value || ELF32_ST_TYPE(elfs->st_info) != STT_FILE)
		VGERR(VG_SYM_FILE_BY_NAME, NULL, 0, ERR);

	return (build_vgsym(elfs, sym, obj, sym_idx, 0));
}


/* ret symbol's object name */
vg_char	*vg_symbol_get_soname(vg_sym *s) 
{	
	if (!s)
		VGERR(VG_SYM_GET_SONAME, NULL, E_INVAL, NULL);
	return (s->soname);
}

/* ret sym name */
vg_char *vg_symbol_get_name(vg_sym *s)
{
	if (!s)
		VGERR(VG_SYM_GET_NAME, NULL, E_INVAL, NULL);
	return (s->syname);
}

/* ret the starting adress of the sym */
vg_vaddr vg_symbol_get_vaddr(vg_sym *s)
{
	if (!s)
		VGERR(VG_SYM_GET_VADDR, NULL, E_INVAL, ERR);
	return (s->vaddr);
}

/* ret the offset of the sym into the object */
vg_off	vg_symbol_get_offset(vg_sym *s)
{
	if (!s)
		VGERR(VG_SYM_GET_OFFSET, NULL, E_INVAL, ERR);
	return (s->offset);
}

/* ret the sym size */
vg_size	vg_symbol_get_size(vg_sym *s)
{
	if (!s)
		VGERR(VG_SYM_GET_SIZE, NULL, E_INVAL, ERR);
	return (s->size);
}

/* ret the offset of the symbol string name into dynstr */
vg_idx	vg_symbol_get_stridx(vg_sym *s)
{
	if (!s)
		VGERR(VG_SYM_GET_STRIDX, NULL, E_INVAL, ERR);
	return (s->stridx);
}

vg_char	vg_symbol_get_info(vg_sym *s)
{
	if (!s)
		VGERR(VG_SYM_GET_INFO, NULL, E_INVAL, ERR);
	return (s->info);
}

vg_idx	vg_symbol_get_symtabidx(vg_sym *s)
{
	if (!s)
		VGERR(VG_SYM_GET_SYMTABIDX, NULL, E_INVAL, ERR);
	return (s->symtab_idx);
}

