//Please refer to http://dansguardian.org/?page=copyright2
//for the license for this code.
//Written by Daniel Barron (daniel@jadeb.com).
//For support go to http://groups.yahoo.com/group/dansguardian

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "autoconf/platform.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <iostream>
#include <arpa/inet.h>
#include <cstdio>
#include <string>
#include <unistd.h>
#include <cerrno>
#include "dansguardian.hpp"
#include "FatController.hpp"
#include <syslog.h>
#include <pwd.h>
#include <fstream>
#include <fcntl.h>
#include "Socket.hpp"
#include "SysV.hpp"
#include <sys/time.h>
#include <sys/resource.h>
#ifdef __BSD
	#include <sys/wait.h>
#else
	#include <wait.h>
#endif


OptionContainer o;

bool isDaemonised;

int main(int argc, char* argv[]) {
    isDaemonised = false;
    std::string configfile = __CONFFILE;
    std::string pidfile = PIDDIR;  // set at compile time
    pidfile += "dansguardian.pid";
    SysV sysv;

    if ((argc > 1) && (*argv[1] == '-')) {
        switch (argv[1][1]) {
            case 'q':
                return sysv.kill(pidfile);
            case 's':
                return sysv.showPID(pidfile);
            case 'r':
                return sysv.hup(pidfile);
            case 'v':
                std::cout << "DansGuardian 2.4.6-5" << std::endl;
                return 0;
            case 'c':
                configfile = argv[2];
                break;
            case 'h':
                std::cout << "Usage: " << argv[0] << " [{-v|-h|-c ConfigFileName|-q|-s|-r}]" << std::endl;
                std::cout << "  -v gives the version number." << std::endl;
                std::cout << "  -h gives this message." << std::endl;
                std::cout << "  -c allows you to specify a different configuration file location." << std::endl;
                std::cout << "  -q causes DansGuardian to kill any running copy." << std::endl;
                std::cout << "  -s shows the parent process PID" << std::endl;
                std::cout << "  -r closes all connections and reloads config files by issuing a HUP," << std::endl;
                std::cout << "     but this does not reset the maxchildren option." << std::endl;

                return 0;
            default:
                std::cout << "Usage: " << argv[0] << " [{-v|-h|-c ConfigFileName|-q|-s|-r}]" << std::endl;
                std::cout << "  -v gives the version number." << std::endl;
                std::cout << "  -h gives this message." << std::endl;
                std::cout << "  -c allows you to specify a different configuration file location." << std::endl;
                std::cout << "  -q causes DansGuardian to kill any running copy." << std::endl;
                std::cout << "  -s shows the parent process PID" << std::endl;
                std::cout << "  -r closes all connections and reloads config files by issuing a HUP," << std::endl;
                std::cout << "     but this does not reset the maxchildren option." << std::endl;
                return 1;
        }
    }

    if (sysv.amIRunning(pidfile)) {
        syslog(LOG_ERR, "%s","I seem to be running already!");
        std::cerr << "I seem to be running already!" << std::endl;
        return 1;  // can't have two copies running!!
    }

    o.loglocation = LOGLOCATION;  // this is set at make time
    int testopts = open(configfile.c_str(), 0, O_RDONLY);
    if (testopts < 0) {
        syslog(LOG_ERR, "%s","Error accessing dansguardian.conf");
	cerr << "Error accessing dansguardian.conf" << std::endl;
	return 1;  // could not open conf file for reading, exit with error
    }
    close (testopts);

    if (!o.read(configfile.c_str())) {
        syslog(LOG_ERR, "%s","Error parsing the dansguardian.conf file or other DansGuardian configuration files");
	cerr << "Error parsing the dansguardian.conf file or other DansGuardian configuration files" << std::endl;
	return 1;  // OptionContainer class had an error reading the conf or
	           // other files so exit with error
    }

     if (o.max_children > 999) {
        syslog(LOG_ERR, "%s","maxchildren option in dansguardian.conf has a value over 999.");
        std::cerr << "maxchildren option in dansguardian.conf has a value over 999." << std::endl;
        std::cerr << "Dammit Jim, I'm a filtering proxy, not a rabbit." << std::endl;
        return 1;  // we can't have rampant proccesses can we?
    }
    int rc;  // return condition variable

    #if defined(RLIMIT_NPROC)
        // The 'if defined' is for compiling on Solaris which does not have
        // an RLIMIT_NPROC.  The apache source gets round the problem in the
        // same way.  With a reasonable amount of research, it appears this
        // is the thing to do.
        struct rlimit rl;
        rl.rlim_cur = o.max_children + 2;
        rl.rlim_max = o.max_children + 2;
        rc = setrlimit(RLIMIT_NPROC, &rl);
        if (rc != 0) {
            syslog(LOG_ERR, "%s","Unable to set resource limit.");
            std::cerr << "Unable to set resource limit." << std::endl;
            return 1;  // exit with error
        }
    #endif

    int pidfilefd = sysv.openPIDFile(pidfile);  // we have to open/create
                                                // as root before drop privs
    if (pidfilefd < 0) {
        syslog(LOG_ERR, "%s","Error creating/opening pid file.");
        std::cerr << "Error creating/opening pid file:" << pidfile << std::endl;
        return 1;
    }

    unsigned int rootuid;  // prepare a struct for use later
    rootuid = geteuid();

    struct passwd *st;  // prepare a struct
    if ((st = getpwnam(PROXYUSER)) != 0) {  // find uid for proxy user
        rc = setgid(st->pw_gid);  // change to rights of proxy user group
		                     // i.e. low - for security
        if (rc == -1) {
            syslog(LOG_ERR, "%s","Unable to setgid()");
	    std::cerr << "Unable to setgid()" << std::endl;
	    return 1;  // setgid failed for some reason so exit with error
        }
	rc = seteuid(st->pw_uid);  // due to linuxthreads design, you
		// can not call setuid even before any thread creation
		// so we just use effective uid for security
		// However DG2 uses a fork pool...
	if (rc == -1) {
	    syslog(LOG_ERR, "%s","Unable to seteuid()");
	    std::cerr << "Unable to seteuid()" << std::endl;
	    return 1;  // seteuid failed for some reason so exit with error
        }
    }
    else {
        syslog(LOG_ERR, "%s","Unable to getpwnam() - does the proxy user exist?");
	cerr << "Unable to getpwnam() - does the proxy user exist?" << std::endl;
	cerr << "Proxy user looking for is '" << PROXYUSER << "'" << std::endl;
	return 1;  // was unable to lockup the user id from passwd
	           // for some reason, so exit with error
    }

    ofstream logfiletest(o.loglocation.c_str(), ios::app);
    if (logfiletest.fail()) {
        syslog(LOG_ERR, "%s","Error opening/creating log file. (check ownership and access rights).");
        std::cout << "Error opening/creating log file. (check ownership and access rights)." << std::endl;
        std::cout << "I am running as " << PROXYUSER << " and I am trying to open " << LOGLOCATION << std::endl;
        return 1;  // opening the log file for writing failed
    }
    logfiletest.close();


    FatController f;  // Thomas The Tank Engine



// *** Uncomment this section if you want to do benchmarking ***
//    std::cout << "Testing domain/url filter with 40000 queries." << std::endl;
//    system("date");
//    for(int k= 0; k < 10000; k++) {
//        o.banned_site_list.findEndsWith("dansguardian.org");
//        o.banned_url_list.findStartsWith("www.jadeb.com/porn/");
//        o.banned_site_list.findEndsWith("nakedchicks.com");
//        o.banned_url_list.findStartsWith("www.amazon.com/adulttoys/");
//    }
//    system("date");
//    std::cout << "Done." << std::endl;
// ************************************************************
    while (true) {
        if (!f.testProxy(o.proxy_ip, o.proxy_port, false)) {
            sleep(4);  // give the proxy more time (work around squid bug)
            if (!f.testProxy(o.proxy_ip, o.proxy_port, false)) {
                sleep(4);
                if (!f.testProxy(o.proxy_ip, o.proxy_port, true)) {
                    return 1;  // could not connect to the proxy so exit with error
                           // with no proxy we can not continue
                }
            }
        }

        rc = f.controlIt(pidfilefd);
                             // its a little messy, but I wanted to split
                             // all the ground work and non-daemon stuff
                             // away from the daemon class
                             // However the line is not so fine.
        if (rc == 2) {

            // In order to re-read the conf files and create cache files
            // we need to become root user again

            rc = seteuid(rootuid);
            if (rc == -1) {
                syslog(LOG_ERR, "%s","Unable to seteuid() to read conf files.");
                #ifdef DGDEBUG
                    std::cerr << "Unable to seteuid() to read conf files." << std::endl;
                #endif
                return 1;
            }

            #ifdef DGDEBUG
                std::cout << "About to re-read conf file." << std::endl;
            #endif
            o.reset();
            if (!o.read(configfile.c_str())) {
                syslog(LOG_ERR, "%s","Error re-parsing the dansguardian.conf file or other DansGuardian configuration files");
                #ifdef DGDEBUG
                    std::cerr << "Error re-parsing the dansguardian.conf file or other DansGuardian configuration files" << std::endl;
                #endif
                return 1;
                   // OptionContainer class had an error reading the conf or
	           // other files so exit with error
            }
            #ifdef DGDEBUG
                std::cout << "conf file read." << std::endl;
            #endif
            while(waitpid(-1, NULL, WNOHANG) > 0) {}  // mop up defunts

            rc = seteuid(st->pw_uid);  // become low priv again
            if (rc == -1) {
	        syslog(LOG_ERR, "%s","Unable to re-seteuid()");
	        #ifdef DGDEBUG
                    std::cerr << "Unable to re-seteuid()" << std::endl;
                #endif
	        return 1;  // seteuid failed for some reason so exit with error
            }
            continue;
        }

        if (rc > 0) {
            if (!isDaemonised) {
                std::cerr << "Exiting with error" << std::endl;
            }
            syslog(LOG_ERR, "%s","Exiting with error");
            return rc;  // exit returning the error number
        }
        return 0;  // exit without error
    }
}
