//Please refer to http://dansguardian.org/?page=copyright2
//for the license for this code.
//Written by Daniel Barron (daniel@jadeb//.com).
//For support go to http://groups.yahoo.com/group/dansguardian

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "autoconf/platform.h"
#include <syslog.h>
#include "FatController.hpp"
#include <csignal>
#ifdef __BSD  // A bodge to get round the problem of a truncated UDS file
	#include <sys/wait.h>
	#define __IPC "/tmp/.dguardianipcc"
	#define __URLIPC "/tmp/.dguardianurlipcc"
#else
	#include <wait.h>
	#define __IPC "/tmp/.dguardianipc"
	#define __URLIPC "/tmp/.dguardianurlipc"
#endif
#include <sys/stat.h>
#include "ConnectionHandler.hpp"
#include "DynamicURLList.hpp"
#include "String.hpp"
#include "Socket.hpp"
#include "UDSocket.hpp"
#include "SysV.hpp"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <pwd.h>
#include <cerrno>
#include <unistd.h>
#include <fstream>
#include <sys/time.h>

#ifdef __GCCVER3
    #include <istream>
#else
    #include <istream.h>
#endif

extern OptionContainer o;
extern bool isDaemonised;

int FatController::controlIt(int pidfilefd) {

    pid_t childpid;  // to hold the child process pid
    pid_t loggerpid;  // to hold the logging process pid
    pid_t urllistpid = 0;  // to hold the logging process pid (zero to stop
                           // warning
    int rc, serversockfd;

    Socket serversock;  // the socket we will listen on for connections
    UDSocket ipcsock;  // the unix domain socket to be used for ipc with
                       // the forked children
    UDSocket urllistsock;

    serversockfd = serversock.getFD();

    if (serversockfd < 0) {
        if (!isDaemonised) {
            std::cerr << "Error creating server socket" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error creating server socket");
        return 1;  // if the socket fd is not +ve then the socket
                   // creation failed
    }

    if (ipcsock.getFD() < 0) {
        if (!isDaemonised) {
            std::cerr << "Error creating ipc socket" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error creating ipc socket");
        return 1;  // if the socket fd is not +ve then the socket
                   // creation failed
    }

    if (o.filter_ip.length() > 6) {  // filter ip specified in conf
                       // listen/bind to a port and ip
        if (serversock.bind(o.filter_ip, o.filter_port)) {
            if (!isDaemonised) {
                std::cerr << "Error binding server socket (is something else running on the filter port and ip? [" << o.filter_port << " " << o.filter_ip << "])" << std::endl;
            }
            syslog(LOG_ERR, "%s","Error binding server socket (is something else running on the filter port and ip?");
            return 1;
        }
    }
    else {             // listen/bind to a port
        if (serversock.bind(o.filter_port)) {
            if (!isDaemonised) {
                std::cerr << "Error binding server socket (is something else running on the filter port? [" << o.filter_port << "])" << std::endl;
            }
            syslog(LOG_ERR, "%s","Error binding server socket (is something else running on the filter port?");
            return 1;
        }
    }
    string ipcfilename = "/tmp/.dguardianipc"; // Needs deleting if its there
    unlink(ipcfilename.c_str()); // this would normally be in a -r situation.
    ipcfilename = "/tmp/.dguardianurlipc"; // Needs deleting if its there
    unlink(ipcfilename.c_str()); // this would normally be in a -r situation.

    if (ipcsock.bind(__IPC)) {  // bind to file
        if (!isDaemonised) {
            std::cerr << "Error binding ipc server file (try using the SysV to stop DansGuardian then try starting it again or doing an 'rm /tmp/.dguardianipc')." << std::endl;
        }
        syslog(LOG_ERR, "%s","Error binding ipc server file (try using the SysV to stop DansGuardian then try starting it again or doing an 'rm /tmp/.dguardianipc').");
        return 1;
    }

    if (o.url_cache_number > 0) {
        if (urllistsock.bind(__URLIPC)) {  // bind to file
            if (!isDaemonised) {
                std::cerr << "Error binding urllistsock server file (try using the SysV to stop DansGuardian then try starting it again or doing an 'rm /tmp/.dguardianurlipc')." << std::endl;
            }
            syslog(LOG_ERR, "%s","Error binding urllistsock server file (try using the SysV to stop DansGuardian then try starting it again or doing an 'rm /tmp/.dguardianurlipc').");
            return 1;
        }
    }


    if (serversock.listen(256)) {  // set it to listen mode with a kernel
                                  // queue of 256 backlog connections
        if (!isDaemonised) {
            std::cerr << "Error listening to server socket" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error listening to server socket");
        return 1;
    }

    if (ipcsock.listen(256)) {  // set it to listen mode with a kernel
                                  // queue of 256 backlog connections
        if (!isDaemonised) {
            std::cerr << "Error listening to ipc server file" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error listening to ipc server file");
        return 1;
    }

    if (o.url_cache_number > 0) {
        if (urllistsock.listen(256)) {  // set it to listen mode with a kernel
                                  // queue of 256 backlog connections
            if (!isDaemonised) {
                std::cerr << "Error listening to url ipc server file" << std::endl;
            }
            syslog(LOG_ERR, "%s","Error listening to url ipc server file");
            return 1;
        }
    }

    Socket peersock;  // the socket which will contain the connection

    if (peersock.getFD() < 0) {
        if (!isDaemonised) {
            std::cerr << "Error creating peer socket" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error creating peer socket");
        return 1;  // if the socket fd is not +ve then the socket
                   // creation failed
    }
    peersock.close();  // We need to do this as we won't be using the
                       // fd created by the contructor as it will be
                       // set to the ones accepted().

    if (!daemonise(pidfilefd)) {  // become a detached daemon
        if (!isDaemonised) {
            std::cerr << "Error daemonising" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error daemonising");
        return 1;
    }

    // We are now a daemon so all errors need to go in the syslog, rather
    // than being reported on screen as we've detached from the console and
    // trying to write to stdout will not be nice.

    struct sigaction sa;
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    if (sigaction(SIGPIPE, &sa, NULL)) {  // ignore SIGPIPE so we can handle
                                          // premature disconections better
        syslog(LOG_ERR, "%s","Error ignoring SIGPIPE");
        return(1);
    }
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    if (sigaction(SIGHUP, &sa, NULL)) {  // ignore HUP
        syslog(LOG_ERR, "%s","Error ignoring HUP");
        return(1);
    }


    // Next thing we need to do is to split into two processes - one to
    // handle incoming TCP connections from the clients and one to handle
    // incoming UDS ipc from our forked children.  This helps reduce
    // bottlenecks by not having only one select() loop.

    loggerpid = fork(); // make a child processes copy of self to be logger

    if (loggerpid == 0) {  // ma ma!  i am the child
        serversock.close();  // we don't need our copy of this so close it
        urllistsock.close();  // we don't need our copy of this so close it
        if (o.url_cache_number > 0) {
            urllistsock.close();  // we don't need our copy of this so close it
        }
        logListener(ipcsock, o.loglocation, o.logconerror);
        #ifdef DGDEBUG
            std::cout << "Log listener exiting" << std::endl;
        #endif
        _exit(0); // is reccomended for child and daemons to use this instead
    }


    if (o.url_cache_number > 0) {
        urllistpid = fork(); // make a child processes copy of self to be logger

        if (urllistpid == 0) {  // ma ma!  i am the child
            serversock.close();  // we don't need our copy of this so close it
            ipcsock.close();  // we don't need our copy of this so close it

            urlListListener(urllistsock, o.logconerror);
            #ifdef DGDEBUG
                std::cout << "URL List listener exiting" << std::endl;
            #endif
            _exit(0); // is reccomended for child and daemons to use this instead
        }
    }

    // I am the parent process here onwards.

    if (o.url_cache_number > 0) {
        urllistsock.close();  // we don't need our copy of this so close it
    }
    ipcsock.close();  // we don't need our copy of this so close it

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = &sig_chld;  // register sig_chld as our handler
    if (sigaction(SIGCHLD, &sa, NULL)) {  // when a child process of ours
                                          // exits, the kernel sends us a
                                          // signal to which we must respond
                                          // or we get a lot of defunct
                                          // processes
        syslog(LOG_ERR, "%s","Error registering child reaper");
        return(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = &sig_term;  // register sig_term as our handler
    if (sigaction(SIGTERM, &sa, NULL)) {  // when the parent process gets a
                                          // sigterm we need to kill our
                                          // children which this will do,
                                          // then we need to exit
        syslog(LOG_ERR, "%s","Error registering SIGTERM handler");
        return(1);
    }

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = &sig_hup;  // register sig_hup as our handler
    if (sigaction(SIGHUP, &sa, NULL)) {  // when the parent process gets a
                                          // sigterm we need to kill our
                                          // children which this will do,
                                          // then we need to exit
        syslog(LOG_ERR, "%s","Error registering SIGHUP handler");
        return(1);
    }

    struct timeval quicksleep;  // used later on for a short sleep
    quicksleep.tv_sec = 0;
    quicksleep.tv_usec = 10000; // = 10ms = 0.01sec
    struct timeval qscopy;  // copy to use as select() can modify

    fd_set fdSet;  // our set of fds (only 1) that select monitors for us
    fd_set fdcpy;  // select modifes the set so we need to use a copy
    FD_ZERO(&fdSet);  // clear the set
    FD_SET(serversockfd, &fdSet);  // add sock to the set

    int pidchildren = 0;  // to keep count of our children

    ConnectionHandler h;  // the class that handles the connections

    int failurecount = 0;  // as we don't exit on an error with select()
                           // due to the fact that these errors do happen
                           // every so often on a fully working, but busy
                           // system, we just watch for too many errors
                           // consecutivly.

    while (failurecount < 9 && !ttg && !reloadconfig) {
                                // loop, essentially, for ever until 9
                                // consecutive errors in which case something
                                // is badly wrong. Nine is my lucky number.
                                // OR, its timetogo - got a sigterm
                                // OR, we need to exit to reread config

        fdcpy = fdSet;  // take a copy
        rc = select(serversockfd + 1, &fdcpy, NULL, NULL, NULL);  // block
                                                  // until something happens

        // Lets take the opportunity to clean up our dead babys if any
        while(waitpid(-1, NULL, WNOHANG) > 0) {  // mop up defunts
            pidchildren--;  // decrement our counter
        }

        if (rc < 0) {  // was an error
            if (errno == EINTR) {
                continue;  // was interupted by a signal so restart
            }
            failurecount++;  // log the error/failure
            continue;  // then continue with the looping
        }
        if (rc < 1) {
            failurecount++;  // log the error/failure
            continue;  // then continue with the looping
        }

        if (FD_ISSET(serversockfd, &fdcpy)) {
            failurecount = 0;  // something is clearly working so reset count
            while(pidchildren >= o.max_children) {
                if (o.logconerror == 1) {
                    syslog(LOG_ERR, "%s","Warning - at maximal connection number!");
                }
                // Lets take the opportunity to clean up our dead babys if any
                while(waitpid(-1, NULL, WNOHANG) > 0) {  // mop up defunts
                    pidchildren--;  // decrement our counter
                }
                // need to wait until we have a spare slot
                qscopy = quicksleep;  // use copy as select() can modify it
                select(0, NULL, NULL, NULL, &qscopy);  // is a very quick sleep()

            }

            peersock = serversock.accept(); // get the new socket which
                                        // describes the accepted connection

            if (peersock.getFD() < 1) {
                failurecount++;  // log the error/failure
                if (o.logconerror == 1) {
                    syslog(LOG_INFO, "%s","Error accepting. (Ignorable)");
                }
                continue; // if the fd of the new socket < 1 there was error
                          // but we ignore it as its not a problem
            }

            childpid = fork(); // make a child processes copy of self

            if (childpid != 0) {  // Eyem the Daddy nahh...
                peersock.close();  // child got copy of fd so we close ours

                if (childpid == -1) {  // fork failed, for example, if the
                                       // process is not allowed to create
                                       // any more
                    syslog(LOG_ERR, "%s","Unable to fork() any more.");
                    #ifdef DGDEBUG
                        std::cout << "Unable to fork() any more." << std::endl;
                    #endif
                    failurecount++;  // log the error/failure
                                     // A DoS attack on a server allocated
                                     // too many children in the conf will
                                     // kill the server.  But this is user
                                     // error.
                    sleep(1);  // need to wait until we have a spare slot
                    continue;  // Nothing doing, go back to listening
                }

                pidchildren++;  // something wonderful has happened...

                #ifdef DGDEBUG
                    std::cout << "Number of children: " << pidchildren << std::endl;
                #endif

                continue;  // daddy finished, go back to listening
            }

            // I am the child - I am alive!
            serversock.close();  // we don't need our copy of this so close it
            memset(&sa, 0, sizeof(sa));
            sa.sa_handler = SIG_DFL;
            if (sigaction(SIGCHLD, &sa, NULL)) {  // restore sig handler
                                                  // in child process
                #ifdef DGDEBUG
                    std::cerr << "Error resetting signal for SIGCHLD" << std::endl;
                #endif
                syslog(LOG_ERR, "%s","Error resetting signal for SIGCHLD");
            }
            memset(&sa, 0, sizeof(sa));
            sa.sa_handler = SIG_DFL;
            if (sigaction(SIGTERM, &sa, NULL)) {  // restore sig handler
                                                  // in child process
                #ifdef DGDEBUG
                    std::cerr << "Error resetting signal for SIGTERM" << std::endl;
                #endif
                syslog(LOG_ERR, "%s","Error resetting signal for SIGTERM");
            }
            memset(&sa, 0, sizeof(sa));
            sa.sa_handler = SIG_IGN;
            if (sigaction(SIGHUP, &sa, NULL)) {  // restore sig handler
                                                  // in child process
                #ifdef DGDEBUG
                    std::cerr << "Error resetting signal for SIGHUP" << std::endl;
                #endif
                syslog(LOG_ERR, "%s","Error resetting signal for SIGHUP");
            }

            h.handleConnection(peersock);  // deal with the connection
            peersock.close();  // finished with accepted connection so close it
            #ifdef DGDEBUG
                std::cout << "child about to exit" << std::endl;
            #endif

            _exit(0); // baby go bye bye
        }
        // should never get here
        syslog(LOG_ERR, "%s","Something wicked has happened.");
        failurecount++;  // log the error/failure
    }
    if (failurecount >= 9) {
        syslog(LOG_ERR, "%s","Exiting due to high failure count.");
        #ifdef DGDEBUG
            std::cout << "Exiting due to high failure count." << std::endl;
        #endif
    }

    #ifdef DGDEBUG
        std::cout << "Main parent process exiting." << std::endl;
    #endif
    serversock.close();  // be nice and neat

    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_DFL;
    if (sigaction(SIGCHLD, &sa, NULL)) {  // restore sig handler
                                          // in child process
        #ifdef DGDEBUG
            std::cerr << "Error resetting signal for SIGCHLD" << std::endl;
        #endif
        syslog(LOG_ERR, "%s","Error resetting signal for SIGCHLD");
    }
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_DFL;
    if (sigaction(SIGTERM, &sa, NULL)) {  // restore sig handler
                                          // in child process
        #ifdef DGDEBUG
            std::cerr << "Error resetting signal for SIGTERM" << std::endl;
        #endif
        syslog(LOG_ERR, "%s","Error resetting signal for SIGTERM");
    }
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_IGN;
    if (sigaction(SIGHUP, &sa, NULL)) {  // restore sig handler
                                          // in child process
        #ifdef DGDEBUG
            std::cerr << "Error resetting signal for SIGHUP" << std::endl;
        #endif
        syslog(LOG_ERR, "%s","Error resetting signal for SIGHUP");
    }
    memset(&sa, 0, sizeof(sa));
    sa.sa_handler = SIG_DFL;
    if (sigaction(SIGPIPE, &sa, NULL)) {  // restore sig handler
                                          // in child process
        #ifdef DGDEBUG
            std::cerr << "Error resetting signal for SIGPIPE" << std::endl;
        #endif
        syslog(LOG_ERR, "%s","Error resetting signal for SIGPIPE");
    }

    if (reloadconfig) {
        reloadconfig = false;
        ::kill(loggerpid, SIGTERM);  // get rid of logger
        if (o.url_cache_number > 0) {
            ::kill(urllistpid, SIGTERM);  // get rid of url cache
        }
        return 2;
    }
    if (ttg) {
        return 0;
    }
    if (o.logconerror == 1) {
        syslog(LOG_ERR, "%s","Main parent process exiting.");
    }
    return 1; // It is only possible to reach here with an error
}



// this is used by dansguardian.cpp, to, yep, test connection to the proxy
// If report is true we log the problem.  report as false allows the caller
// to test multiple times with no error
bool FatController::testProxy(std::string proxyip, int proxyport, bool report) {
    int sck_inet, conn;
    Socket sock;
    sck_inet = sock.getFD();
    if (sck_inet == -1) {
        if (report) {
            if (!isDaemonised) {
                std::cerr << "Error creating socket to test proxy connection" << std::endl;
            }
            syslog(LOG_ERR, "%s","Error creating socket to test proxy connection");
        }
        return false;
    }
    conn = sock.connect(proxyip, proxyport); // hmmm, I wonder what this do
    if (conn) {
        if (report) {
            if (!isDaemonised) {
                std::cerr << "Error creating connection to test proxy" << std::endl;
            }
            syslog(LOG_ERR, "%s","Error creating connection to test proxy");
        }
        return false;
    }
//    close(conn);  ** delete when confirmed not needed
    sock.close();
    return true;  // it worked!
}


bool FatController::daemonise(int pidfilefd) {

    #ifdef DGDEBUG
        return true;  // if debug mode is enabled we don't want to detach
    #endif

    if (isDaemonised) {
        return true;  // we are already daemonised so this must be a
                      // reload caused by a HUP
    }

    pid_t pid;
    if ((pid = fork()) < 0) {
        return false;
    }
    else if (pid != 0) {
        _exit(0);  // parent goes bye-bye
    }
    // child continues
    setsid();  // become session leader
    chdir("/"); // change working directory
    umask(0); // clear our file mode creation mask

    SysV sysv;
    int rc = sysv.writePIDFile(pidfilefd);  // also closes the fd
    if (rc != 0) {
        syslog(LOG_ERR, "%s","Error writing to the dansguardian.pid file.");
        return false;
    }
    isDaemonised = true;

    return true;
}



int FatController::logListener(UDSocket ipcsock, std::string loglocation, int logconerror) {
    UDSocket ipcpeersock;  // the socket which will contain the ipc connection
    ipcpeersock.close(); // we won't need its fd as we'll be accept()ing
    int rc, ipcsockfd;
    char* logline = new char[4096];

    ofstream logfile(loglocation.c_str(), ios::app);
    if (logfile.fail()) {
        syslog(LOG_ERR, "%s","Error opening/creating log file.");
        #ifdef DGDEBUG
            std::cout << "Error opening/creating log file: " << loglocation << std::endl;
        #endif
        return 1;  // return with error
    }

    ipcsockfd = ipcsock.getFD();

    fd_set fdSet;  // our set of fds (only 1) that select monitors for us
    fd_set fdcpy;  // select modifes the set so we need to use a copy
    FD_ZERO(&fdSet);  // clear the set
    FD_SET(ipcsockfd, &fdSet);  // add ipcsock to the set

    while (true) {  // loop, essentially, for ever

        fdcpy = fdSet;  // take a copy
        rc = select(ipcsockfd + 1, &fdcpy, NULL, NULL, NULL);  // block
                                                  // until something happens
        if (rc < 0) {  // was an error
            if (errno == EINTR) {
                continue;  // was interupted by a signal so restart
            }
            if (logconerror == 1) {
                syslog(LOG_ERR, "%s","ipc rc<0. (Ignorable)");
            }
            continue;
        }
        if (rc < 1) {
            if (logconerror == 1) {
                syslog(LOG_ERR, "%s","ipc rc<0. (Ignorable)");
            }
            continue;
        }
        if (FD_ISSET(ipcsockfd, &fdcpy)) {
            #ifdef DGDEBUG
                std::cout << "received a log request" << std::endl;
            #endif
            ipcpeersock = ipcsock.accept();
            if (ipcpeersock.getFD() < 0) {
                ipcpeersock.close();
                if (logconerror == 1) {
                    syslog(LOG_ERR, "%s","Error accepting ipc. (Ignorable)");
                }
                continue; // if the fd of the new socket < 0 there was error
                          // but we ignore it as its not a problem
            }
            try {
                rc = ipcpeersock.getline(logline, 4096, 3);  // throws on err
            } catch (exception& e) {
                ipcpeersock.close();
                if (logconerror == 1) {
                    syslog(LOG_ERR, "%s","Error reading ipc. (Ignorable)");
                }
                continue;
            }
            logfile << logline << std::endl;  // append the line
            #ifdef DGDEBUG
                std::cout << logline << std::endl;
            #endif
            ipcpeersock.close();  // close the connection
            #ifdef DGDEBUG
                std::cout << "logged" << std::endl;
            #endif
            continue;  // go back to listening
        }
        // should never get here
        syslog(LOG_ERR, "%s","Something wicked has ipc happened");

    }
    delete[] logline;
    logfile.close();  // close the file and
    ipcsock.close();  // be nice and neat
    return 1; // It is only possible to reach here with an error

}


extern "C" {  // The kernel knows nothing of objects so
              // we have to have a lump of c

    void sig_chld(int signo) {
//        timetoreap = true;  // No longer used
    }

    void sig_term(int signo) {
        if (ttg) {
            return; // this is to prevent an std::endless loop
        }
        kill(0, SIGTERM);  // send everyone in this process group a TERM
                           // which causes them to exit as the default action
                           // but it also seems to send itself a TERM
                           // which gets queued
        ttg = true;  // its time to go
    }

    void sig_hup(int signo) {
        reloadconfig = true;
        #ifdef DGDEBUG
            std::cout << "HUP received." << std::endl;
        #endif
    }
}

int FatController::urlListListener(UDSocket ipcsock, int logconerror) {
    #ifdef DGDEBUG
            std::cout << "url listener started" << std::endl;
    #endif
    UDSocket ipcpeersock;  // the socket which will contain the ipc connection
    ipcpeersock.close(); // we won't need its fd as we'll be accept()ing
    int rc, ipcsockfd;
    char* logline = new char[4096];
    String reply;
    DynamicURLList urllist;
    #ifdef DGDEBUG
            std::cout << "setting url list size-age:" << o.url_cache_number << "-" << o.url_cache_age << std::endl;
    #endif
    urllist.setListSize(o.url_cache_number, o.url_cache_age);
    ipcsockfd = ipcsock.getFD();
    #ifdef DGDEBUG
            std::cout << "url ipcsockfd:" << ipcsockfd << std::endl;
    #endif

    fd_set fdSet;  // our set of fds (only 1) that select monitors for us
    fd_set fdcpy;  // select modifes the set so we need to use a copy
    FD_ZERO(&fdSet);  // clear the set
    FD_SET(ipcsockfd, &fdSet);  // add ipcsock to the set

    #ifdef DGDEBUG
            std::cout << "url listener entering select()" << std::endl;
    #endif
    while (true) {  // loop, essentially, for ever

        fdcpy = fdSet;  // take a copy

        rc = select(ipcsockfd + 1, &fdcpy, NULL, NULL, NULL);  // block
                                                  // until something happens
        #ifdef DGDEBUG
            std::cout << "url listener select returned" << std::endl;
        #endif
        if (rc < 0) {  // was an error
            if (errno == EINTR) {
                continue;  // was interupted by a signal so restart
            }
            if (logconerror == 1) {
                syslog(LOG_ERR, "%s","url ipc rc<0. (Ignorable)");
            }
            continue;
        }
        if (FD_ISSET(ipcsockfd, &fdcpy)) {
            #ifdef DGDEBUG
                std::cout << "received an url request" << std::endl;
            #endif
            ipcpeersock = ipcsock.accept();
            if (ipcpeersock.getFD() < 0) {
                ipcpeersock.close();
                if (logconerror == 1) {
                    #ifdef DGDEBUG
                        std::cout << "Error accepting url ipc. (Ignorable)" << std::endl;
                    #endif
                    syslog(LOG_ERR, "%s","Error accepting url ipc. (Ignorable)");
                }
                continue; // if the fd of the new socket < 0 there was error
                          // but we ignore it as its not a problem
            }
            try {
                rc = ipcpeersock.getline(logline, 4096, 3);  // throws on err
            } catch (exception& e) {
                ipcpeersock.close();
                if (logconerror == 1) {
                    #ifdef DGDEBUG
                        std::cout << "Error reading ip ipc. (Ignorable)" << std::endl;
                    #endif
                    syslog(LOG_ERR, "%s","Error reading ip ipc. (Ignorable)");
                }
                continue;
            }
            if (logline[0] == 'A' && logline[1] == ' ') {
                ipcpeersock.close();  // close the connection
                urllist.addEntry(logline + 2);
                #ifdef DGDEBUG
                    std::cout << "url add request:" << logline << std::endl;
                #endif
                continue;
            }
            if (urllist.inURLList(logline)) {
                reply = "Y\n";
            }
            else {
                reply = "N\n";
            }
            try {
                ipcpeersock.writeString(reply.toCharArray());
            } catch (exception& e) {
                if (logconerror == 1) {
                    syslog(LOG_ERR, "%s","Error writing url ipc. (Ignorable)");
                }
                continue;
            }
            ipcpeersock.close();  // close the connection
            #ifdef DGDEBUG
                std::cout << "url list reply:" << reply << std::endl;
            #endif
            continue;  // go back to listening
        }
   }
    delete[] logline;
    ipcsock.close();  // be nice and neat
    return 1; // It is only possible to reach here with an error

}

