//Please refer to http://dansguardian.org/?page=copyright2
//for the license for this code.
//Written by Daniel Barron (daniel@jadeb.com).
//For support go to http://groups.yahoo.com/group/dansguardian

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <syslog.h>
#include "OptionContainer.hpp"
#include "RegExp.hpp"
#include <string>
#include <iostream>
#include <fstream>
#include <netdb.h>  // for gethostby
#include <netinet/in.h>  // for address structures
#include <arpa/inet.h>  // for inet_aton()
#include <sys/socket.h>

//using namespace std;
extern bool isDaemonised;

void OptionContainer::reset() {
    banned_phrase_list.reset();
    exception_phrase_list.reset();
    exception_site_list.reset();
    exception_user_list.reset();
    exception_ip_list.reset();
    exception_url_list.reset();
    banned_extension_list.reset();
    banned_mimetype_list.reset();
    banned_site_list.reset();
    banned_url_list.reset();
    banned_ip_list.reset();
    banned_user_list.reset();
    banned_regexpurl_list.reset();
    weighted_phrase_list.reset();
    html_template.reset();
    content_regexp_list.reset();
    language_list.reset();
    banned_phrase_list_index.clear();
    conffile.clear();
    banned_regexpurl_list_comp.clear();
    content_regexp_list_comp.clear();
    content_regexp_list_rep.clear();
}

bool OptionContainer::read(std::string filename) {
    try { // all sorts of exceptions could occur reading conf files
        std::string linebuffer;
        String temp;  // for tempory conversion and storage
        int j;  // counter
        ifstream conffiles(filename.c_str(), ios::in);  // dansguardian.conf
        if (!conffiles.good()) {
            if (!isDaemonised) {
                std::cerr << "error reading: " << filename << std::endl;
            }
            syslog(LOG_ERR, "%s","error reading dansguardian.conf");
            return false;
        }
        while (!conffiles.eof()) {
            getline(conffiles, linebuffer);
            if (!conffiles.eof() && linebuffer.length() != 0) {
                if (linebuffer[0] != '#') {  // i.e. not commented out
                    for(j = 0; j < (signed)linebuffer.length(); j++) {
                        linebuffer[j] = tolower(linebuffer[j]);
                    }
                    temp = (char*)linebuffer.c_str();
                    if (temp.contains("#")) {
                        temp = temp.before("#");
                    }
                    temp.removeWhiteSpace();  // get rid of spaces at end of line
                    linebuffer = temp.toCharArray();
                    conffile.push_back(linebuffer);  // stick option in deque
                }
            }
        }
        conffiles.close();

        // the dansguardian.conf and pics files get amalgamated into one
        // deque.  They are only seperate files for clarity.

        linebuffer = findoptionS("picsfile");
        ifstream picsfiles(linebuffer.c_str(), ios::in); // pics file
        if (!picsfiles.good()) {
            if (!isDaemonised) {
                std::cerr << "error reading: " << linebuffer << std::endl;
            }
            syslog(LOG_ERR, "%s","error reading pics file");
            return false;
        }
        while (!picsfiles.eof()) {
            getline(picsfiles, linebuffer);
            if (!picsfiles.eof() && linebuffer.length() != 0) {
                if (linebuffer[0] != '#') {  // i.e. not commented out
                    temp = (char*)linebuffer.c_str();
                    if (temp.contains("#")) {
                        temp = temp.before("#");
                    }
                    while (temp.endsWith(" ")) {
                        temp.chop();  // get rid of spaces at end of line
                    }
                    linebuffer = temp.toCharArray();
                    conffile.push_back(linebuffer);  // stick option in deque
                }
            }
        }
        picsfiles.close();
        max_children = findoptionI("maxchildren");
        if (!realitycheck(String(max_children), 1, 3, "maxchildren"))
            { return false; }  // check its a reasonable value
        max_upload_size = findoptionI("maxuploadsize") * 1024;
        if (!realitycheck(String(max_upload_size), 1, 8, "maxuploadsize"))
            { return false; }  // check its a reasonable value
        max_content_filter_size = findoptionI("maxcontentfiltersize") * 1024 * 1024;
        if (!realitycheck(String(max_content_filter_size), 1, 8, "maxcontentfiltersize"))
            { return false; }  // check its a reasonable value

        url_cache_number = findoptionI("urlcachenumber");
        if (!realitycheck(String(url_cache_number), 1, 5, "urlcachenumber"))
            { return false; }  // check its a reasonable value

        url_cache_age = findoptionI("urlcacheage");
        if (!realitycheck(String(url_cache_age), 1, 5, "urlcacheage"))
            { return false; }  // check its a reasonable value

        filter_port = findoptionI("filterport");
        if (!realitycheck(String(filter_port), 2, 6, "filterport"))
            { return false; }  // check its a reasonable value
        proxy_port = findoptionI("proxyport");
        if (!realitycheck(String(proxy_port), 2, 6, "proxyport"))
            { return false; }  // etc
        proxy_ip = findoptionS("proxyip");
        if (!realitycheck(String(proxy_ip.c_str()), 7, 15, "proxyip"))
            { return false; }
        filter_ip = findoptionS("filterip");
        if (!realitycheck(String(filter_ip.c_str()), 0, 15, "filterip"))
            { return false; }
        ll = findoptionI("loglevel");
        if (!realitycheck(ll, 1, 1, "loglevel"))
            { return false; }  // etc
        log_file_format = findoptionI("logfileformat");
        if (!realitycheck(log_file_format, 1, 1, "logfileformat"))
            { return false; }  // etc
        if (log_file_format < 1 || log_file_format > 4) {
            log_file_format = 1;
        }
        if (findoptionS("enablePICS") == "off") {
            enable_PICS = 0;
        }
        else {
            enable_PICS = 1;
        }
        if (findoptionS("showweightedfound") == "on") {
            show_weighted_found = 1;
        }
        else {
            show_weighted_found = 0;
        }
        weighted_phrase_mode = findoptionI("weightedphrasemode");
        if (!realitycheck(String(weighted_phrase_mode), 1, 1, "weightedphrasemode"))
            { return false; }
        naughtyness_limit = findoptionI("naughtynesslimit");
        if (!realitycheck(String(naughtyness_limit), 1, 4, "naughtynesslimit"))
            { return false; }
        reporting_level = findoptionI("reportinglevel");
        if (!realitycheck(String(reporting_level), 1, 2, "reportinglevel"))
            { return false; }
        html_template_location = findoptionS("htmltemplate");

        if (findoptionS("forwardedfor") == "on") {
           forwarded_for = 1;
        }
        else {
           forwarded_for = 0;
        }
        if (findoptionS("logexceptionhits") == "on") {
           log_exception_hits = 1;
        }
        else {
           log_exception_hits = 0;
        }
        if (findoptionS("createlistcachefiles") == "off") {
           createlistcachefiles = 0;
        }
        else {
           createlistcachefiles = 1;
        }
        if (findoptionS("logconnectionhandlingerrors") == "on") {
           logconerror = 1;
        }
        else {
           logconerror = 0;
        }
        if (findoptionS("usernameidmethodproxyauth") == "on") {
           uim_proxyauth = 1;
        }
        else {
           uim_proxyauth = 0;
        }
        if (findoptionS("usernameidmethodntlm") == "on") {
           uim_ntlm = 1;
        }
        else {
           uim_ntlm = 0;
        }
        if (findoptionS("usernameidmethodident") == "on") {
           uim_ident = 1;
        }
        else {
           uim_ident = 0;
        }
        if (findoptionS("reverseaddresslookups") == "on") {
           reverse_lookups = 1;
        }
        else {
           reverse_lookups = 0;
        }

        if (findoptionS("usexforwardedfor") == "on") {
	   use_xforwardedfor = 1;
	}
	else {
	   use_xforwardedfor = 0;
	}

        exception_phrase_list_location = findoptionS("exceptionphraselist");
        weighted_phrase_list_location = findoptionS("weightedphraselist");
        banned_ip_list_location = findoptionS("bannediplist");
        banned_user_list_location = findoptionS("banneduserlist");
        banned_phrase_list_location = findoptionS("bannedphraselist");
        banned_extension_list_location = findoptionS("bannedextensionlist");
        banned_mimetype_list_location = findoptionS("bannedmimetypelist");
        banned_site_list_location = findoptionS("bannedsitelist");
        banned_url_list_location = findoptionS("bannedurllist");
        banned_regexpurl_list_location = findoptionS("bannedregexpurllist");
        content_regexp_list_location = findoptionS("contentregexplist");
        exceptions_site_list_location = findoptionS("exceptionsitelist");
        exceptions_url_list_location = findoptionS("exceptionurllist");
        exceptions_user_list_location = findoptionS("exceptionuserlist");
        exceptions_ip_list_location = findoptionS("exceptioniplist");
        language_list_location = findoptionS("languagefile");
        access_denied_address = findoptionS("accessdeniedaddress");
        ada = access_denied_address.c_str();
        ada = ada.after("://");
        ada.removeWhiteSpace();
        ada = ada.before("/"); // ada now contains the FQ host name of the
                               // server that serves the accessdenied.html
                               // file
        if (ada.contains(":")) {
            ada = ada.before(":");  // chop off the port number if any
        }
        if (reporting_level == 1 || reporting_level == 2) {
            if (ada.length() < 4) {
                if (!isDaemonised) {
                    cerr << "accessdeniedaddress setting appears to be wrong." << endl;
                }
                syslog(LOG_ERR, "%s", "accessdeniedaddress setting appears to be wrong.");
                return false;
            }
        }
        pics_rsac_nudity = findoptionI("RSACnudity");
        pics_rsac_language = findoptionI("RSAClanguage");
        pics_rsac_sex = findoptionI("RSACsex");
        pics_rsac_violence = findoptionI("RSACviolence");
        pics_evaluweb_rating = findoptionI("evaluWEBrating");
        pics_cybernot_sex = findoptionI("CyberNOTsex");
        pics_cybernot_other = findoptionI("CyberNOTother");
        pics_safesurf_agerange = findoptionI("SafeSurfagerange");
        pics_safesurf_profanity = findoptionI("SafeSurfprofanity");
        pics_safesurf_heterosexualthemes = findoptionI("SafeSurfheterosexualthemes");
        pics_safesurf_homosexualthemes = findoptionI("SafeSurfhomosexualthemes");
        pics_safesurf_nudity = findoptionI("SafeSurfnudity");
        pics_safesurf_violence = findoptionI("SafeSurfviolence");
        pics_safesurf_sexviolenceandprofanity = findoptionI("SafeSurfsexviolenceandprofanity");
        pics_safesurf_intolerance = findoptionI("SafeSurfintolerance");
        pics_safesurf_druguse = findoptionI("SafeSurfdruguse");
        pics_safesurf_otheradultthemes = findoptionI("SafeSurfotheradultthemes");
        pics_safesurf_gambling = findoptionI("SafeSurfgambling");
        pics_icra_chat = findoptionI("ICRAchat");
        pics_icra_moderatedchat = findoptionI("ICRAmoderatedchat");
        pics_icra_languagesexual = findoptionI("ICRAlanguagesexual");
        pics_icra_languageprofanity = findoptionI("ICRAlanguageprofanity");
        pics_icra_languagemildexpletives = findoptionI("ICRAlanguagemildexpletives");
        pics_icra_nuditygraphic = findoptionI("ICRAnuditygraphic");
        pics_icra_nuditymalegraphic = findoptionI("ICRAnuditymalegraphic");
        pics_icra_nudityfemalegraphic = findoptionI("ICRAnudityfemalegraphic");
        pics_icra_nuditytopless = findoptionI("ICRAnuditytopless");
        pics_icra_nuditybottoms = findoptionI("ICRAnuditybottoms");
        pics_icra_nuditysexualacts = findoptionI("ICRAnuditysexualacts");
        pics_icra_nudityobscuredsexualacts = findoptionI("ICRAnudityobscuredsexualacts");
        pics_icra_nuditysexualtouching = findoptionI("ICRAnuditysexualtouching");
        pics_icra_nuditykissing = findoptionI("ICRAnuditykissing");
        pics_icra_nudityartistic = findoptionI("ICRAnudityartistic");
        pics_icra_nudityeducational = findoptionI("ICRAnudityeducational");
        pics_icra_nuditymedical = findoptionI("ICRAnuditymedical");
        pics_icra_drugstobacco = findoptionI("ICRAdrugstobacco");
        pics_icra_drugsalcohol = findoptionI("ICRAdrugsalcohol");
        pics_icra_drugsuse = findoptionI("ICRAdrugsuse");
        pics_icra_gambling = findoptionI("ICRAgambling");
        pics_icra_weaponuse = findoptionI("ICRAweaponuse");
        pics_icra_intolerance = findoptionI("ICRAintolerance");
        pics_icra_badexample = findoptionI("ICRAbadexample");
        pics_icra_pgmaterial = findoptionI("ICRApgmaterial");
        pics_icra_violencerape = findoptionI("ICRAviolencerape");
        pics_icra_violencetohumans = findoptionI("ICRAviolencetohumans");
        pics_icra_violencetoanimals = findoptionI("ICRAviolencetoanimals");
        pics_icra_violencetofantasy = findoptionI("ICRAviolencetofantasy");
        pics_icra_violencekillinghumans = findoptionI("ICRAviolencekillinghumans");
        pics_icra_violencekillinganimals = findoptionI("ICRAviolencekillinganimals");
        pics_icra_violencekillingfantasy = findoptionI("ICRAviolencekillingfantasy");
        pics_icra_violenceinjuryhumans = findoptionI("ICRAviolenceinjuryhumans");
        pics_icra_violenceinjuryanimals = findoptionI("ICRAviolenceinjuryanimals");
        pics_icra_violenceinjuryfantasy = findoptionI("ICRAviolenceinjuryfantasy");
        pics_icra_violenceartisitic = findoptionI("ICRAviolenceartisitic");
        pics_icra_violenceeducational = findoptionI("ICRAviolenceeducational");
        pics_icra_violencemedical = findoptionI("ICRAviolencemedical");
        pics_icra_violencesports = findoptionI("ICRAviolencesports");
        pics_weburbia_rating = findoptionI("Weburbiarating");
        pics_vancouver_multiculturalism = findoptionI("Vancouvermulticulturalism");
        pics_vancouver_educationalcontent = findoptionI("Vancouvereducationalcontent");
        pics_vancouver_environmentalawareness = findoptionI("Vancouverenvironmentalawareness");
        pics_vancouver_tolerance = findoptionI("Vancouvertolerance");
        pics_vancouver_violence = findoptionI("Vancouverviolence");
        pics_vancouver_sex = findoptionI("Vancouversex");
        pics_vancouver_profanity = findoptionI("Vancouverprofanity");
        pics_vancouver_safety = findoptionI("Vancouversafety");
        pics_vancouver_canadiancontent = findoptionI("Vancouvercanadiancontent");
        pics_vancouver_commercialcontent = findoptionI("Vancouvercommercialcontent");
        pics_vancouver_gambling = findoptionI("Vancouvergambling");


        // Most of the readfoofiles could be amalgamated into one fuction
        // and will be one day.  So it's a bit messy at the moment.

        if (!readbplfile(banned_phrase_list_location.c_str(), exception_phrase_list_location.c_str(), weighted_phrase_list_location.c_str())) {
            return false;
        }  // read banned, exception, weighted phrase list
        if (!readbilfile(banned_ip_list_location.c_str())) {
            return false;
        }  // read banned ip list
        if (!readbuslfile(banned_user_list_location.c_str())) {
            return false;
        }  // read banned user list
        if (!readeilfile(exceptions_ip_list_location.c_str())) {
            return false;
        }  // ip exceptions
        if (!readeslfile(exceptions_site_list_location.c_str())) {
            return false;
        }  // site exceptions
        if (!readeurllfile(exceptions_url_list_location.c_str())) {
            return false;
        }  // url exceptions
        if (!readeulfile(exceptions_user_list_location.c_str())) {
            return false;
        }  // site exceptions
        if (!readbelfile(banned_extension_list_location.c_str())) {
            return false;
        }  // file extensions
        if (!readbmlfile(banned_mimetype_list_location.c_str())) {
            return false;
        }  // mime types
        if (!readbslfile(banned_site_list_location.c_str())) {
            return false;
        }  // banned domains
        if (!readbulfile(banned_url_list_location.c_str())) {
            return false;
        }  // banned urls


        if (!readbreulfile(banned_regexpurl_list_location.c_str())) {
            return false;
        }  // banned reg exp urls

        if (!readcrelfile(content_regexp_list_location.c_str())) {
            return false;
        }  // content replacement regular expressions

        if (!precompileregexps()) {
            return false;
        }  // precompiled reg exps for speed

        if (!language_list.readLanguageList(language_list_location.c_str())) {
            return false;
        }  // messages language file

        if (banned_site_list.inList("**")) {
            blanketblock = 1;
        }
        else {
            blanketblock = 0;
        }
        if (banned_site_list.inList("*ip")) {
            blanket_ip_block = 1;
        }
        else {
            blanket_ip_block = 0;
        }
        if (reporting_level == 3) {  // only if reporting set to HTML templ
            if (!html_template.readTemplateFile(html_template_location.c_str())) {
                if (!isDaemonised) {
                    std::cerr << "Error reading HTML Template file:" << html_template_location << std::endl;
                }
                syslog(LOG_ERR, "%s", "Error reading HTML Template file.");
                return false;
                // HTML template file
            }
        }

    } catch (exception& e) {
        if (!isDaemonised) {
            std::cerr << e.what() << std::endl;  // when called the daemon has not
                                   // detached so we can do this
        }
	return false;
    }
    return true;
}

bool OptionContainer::readbplfile(const char* banned, const char* exception, const char* weighted) {

    bool result = banned_phrase_list.readPhraseList(exception, true);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening exceptionphraselist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening exceptionphraselist");
        return false;
    }
    result = banned_phrase_list.readPhraseList(banned, false);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening bannedphraselist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening bannedphraselist");
        return false;
    }
    if (weighted_phrase_mode > 0) {  // if zero wpl is deactivated
        result = banned_phrase_list.readPhraseList(weighted, false);
        if (!result) {
            if (!isDaemonised) {
                std::cerr << "Error opening weightedphraselist" << std::endl;
            }
            syslog(LOG_ERR, "%s","Error opening weightedphraselist");
            return false;
        }
    }
    banned_phrase_list.makeGraph();

    return true;

}

bool OptionContainer::readbilfile(const char* filename) {

    bool result = banned_ip_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening bannediplist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening bannediplist");
        return false;
    }
    banned_ip_list.startsWithSort();
    return true;
}

bool OptionContainer::readbuslfile(const char* filename) {

    bool result = banned_user_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening banneduserlist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening banneduserlist");
        return false;
    }
    banned_user_list.startsWithSort();
    return true;
}


bool OptionContainer::readeslfile(const char* filename) {

    bool result = exception_site_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening exceptionsitelist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening exceptionsitelist");
        return false;
    }
    exception_site_list.endsWithSort();
    return true;
}

bool OptionContainer::readeurllfile(const char* filename) {

    bool result = exception_url_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening exceptionurllist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening exceptionurllist");
        return false;
    }
    exception_url_list.startsWithSort();
    return true;
}

bool OptionContainer::readeulfile(const char* filename) {

    bool result = exception_user_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening exceptionuserlist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening exceptionuserlist");
        return false;
    }
    exception_user_list.endsWithSort();
    return true;
}

bool OptionContainer::readeilfile(const char* filename) {

    bool result = exception_ip_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening exceptioniplist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening exceptioniplist");
        return false;
    }
    exception_ip_list.endsWithSort();
    return true;
}


bool OptionContainer::readbelfile(const char* filename) {
    bool result = banned_extension_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening bannedextensionlist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening bannedextensionlist");
        return false;
    }
    banned_extension_list.endsWithSort();
    return true;
}


bool OptionContainer::readbslfile(const char* filename) {
    bool result = banned_site_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening bannedsitelist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening bannedsitelist");
        return false;
    }
    banned_site_list.endsWithSort();
    if (createlistcachefiles == 1) {
        bool result = banned_site_list.createCacheFile(filename);
        if (!result) {
            if (!isDaemonised) {
                std::cerr << "Error creating bannedsitelist cache file." << std::endl;
                std::cerr << "Do you have write access to this area:" << std::endl;
                std::cerr << filename << ".processed ?" << std::endl;
            }
            syslog(LOG_ERR, "%s","Error creating bannedsitelist cache file.");
            syslog(LOG_ERR, "%s","Do you have write access to this area:");
            syslog(LOG_ERR, "%s",filename);
            return false;
        }
    }
    return true;
}


bool OptionContainer::readbulfile(const char* filename) {
    bool result = banned_url_list.readItemList(filename, true, 1);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening bannedurllist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening bannedurllist");
        return false;
    }
    banned_url_list.startsWithSort();
    if (createlistcachefiles == 1) {
        bool result = banned_url_list.createCacheFile(filename);
        if (!result) {
            if (!isDaemonised) {
                std::cerr << "Error creating bannedurllist cache file." << std::endl;
                std::cerr << "Do you have write access to this area:" << std::endl;
                std::cerr << filename << ".processed ?" << std::endl;
            }
            syslog(LOG_ERR, "%s","Error creating bannedurllist cache file.");
            syslog(LOG_ERR, "%s","Do you have write access to this area:");
            syslog(LOG_ERR, "%s",filename);
            return false;
        }
    }
    return true;
}


bool OptionContainer::readbreulfile(const char* filename) {
    bool result = banned_regexpurl_list.readItemList(filename, true, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening bannedregexpurllist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening bannedregexpurllist");
        return false;
    }
    RegExp r;
    bool rv = true;
    int len = banned_regexpurl_list.getListLength();

    for(int i = 0; i < len; i++) {
        rv = r.comp(banned_regexpurl_list.getItemAt(i).c_str());
        if (rv == false) {
            if (!isDaemonised) {
                std::cerr << "Error compiling regexp:" << banned_regexpurl_list.getItemAt(i) << std::endl;
            }
            syslog(LOG_ERR, "%s", "Error compiling regexp:");
            syslog(LOG_ERR, "%s", banned_regexpurl_list.getItemAt(i).c_str());
        return false;
        }
        banned_regexpurl_list_comp.push_back(r);
    }
    return true;
}

bool OptionContainer::readcrelfile(const char* filename) {
    bool result = content_regexp_list.readItemList(filename, true, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening contentregexplist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening contentregexplist");
        return false;
    }
    content_regexp_list.startsWithSort();
    RegExp r;
    bool rv = true;
    String regexp;
    String replacement;
    for(int i = 0; i < content_regexp_list.getListLength(); i++) {
        regexp = content_regexp_list.getItemAt(i).c_str();
        replacement = regexp.after("\"->\"").before("\"");
        regexp = regexp.after("\"").before("\"->\"");
        if (replacement.length() < 1 || regexp.length() < 1) {
            continue;
        }
        rv = r.comp(regexp.toCharArray());
        if (rv == false) {
            if (!isDaemonised) {
                std::cerr << "Error compiling regexp:" << content_regexp_list.getItemAt(i) << std::endl;
            }
            syslog(LOG_ERR, "%s", "Error compiling regexp:");
            syslog(LOG_ERR, "%s", content_regexp_list.getItemAt(i).c_str());
        return false;
        }
        content_regexp_list_comp.push_back(r);
        content_regexp_list_rep.push_back(replacement);
    }
    return true;
}

bool OptionContainer::readbmlfile(const char* filename) {
    bool result = banned_mimetype_list.readItemList(filename, false, 0);
    if (!result) {
        if (!isDaemonised) {
            std::cerr << "Error opening bannedmimetypelist" << std::endl;
        }
        syslog(LOG_ERR, "%s","Error opening bannedmimetypelist");
        return false;
    }
    banned_mimetype_list.endsWithSort();
    return true;
}


bool OptionContainer::inexceptions(String url) {
    if (iswebserver(url)) {  // don't filter our web server
        return true;
    }
    url.removeWhiteSpace();  // just in case of weird browser crap
    url.toLower();
    url.removePTP();  // chop off the ht(f)tp(s)://
    if (url.contains("/")) {
        url = url.before("/");  // chop off any path after the domain
    }
    int i;
    while (url.contains(".")) {
        i = exception_site_list.findInList(url.toCharArray());
        if (i >= 0) {
            return true;  // exact match
        }
        url = url.after(".");  // check for being in higher level domains
    }
    return false;  // and our survey said "UUHH UURRGHH"
}

bool OptionContainer::iswebserver(String url) {
    url.removeWhiteSpace();  // just in case of weird browser crap
    url.toLower();
    url.removePTP();  // chop off the ht(f)tp(s)://
    if (url.contains("/")) {
        url = url.before("/");  // chop off any path after the domain
    }
    if (url.startsWith(ada.toCharArray())) { // don't filter our web
                                             // server
        return true;
    }
    return false;
}


bool OptionContainer::inurlexceptions(String url) {
    int i;
    url.removeWhiteSpace();  // just in case of weird browser crap
    url.toLower();
    url.removePTP();  // chop off the ht(f)tp(s)://
    if (url.endsWith("/")) {
        url.chop();  // chop off trailing / if any
    }
    while (url.before("/").contains(".")) {
        i = exception_url_list.findStartsWith(url.toCharArray());
        if (i >= 0) {
            return true;  // exact match
        }
        url = url.after(".");  // check for being in higher level domains
    }
    return false;
}

int OptionContainer::inBannedSiteList(String url) {
    url.removeWhiteSpace();  // just in case of weird browser crap
    url.toLower();
    url.removePTP();  // chop off the ht(f)tp(s)://
    if (url.contains("/")) {
        url = url.before("/");  // chop off any path after the domain
    }
    int i;
    bool isipurl = isIPHostname(url);
    if (blanket_ip_block == 1 && isipurl) {
        return -2;  // is an ip only address
    }
    if (reverse_lookups == 1 && isipurl) {  // change that ip into hostname
        std::deque<String> url2s = ipToHostname(url);
        String url2;
        unsigned int j;
        for (j = 0; j < url2s.size(); j++) {
            url2 = url2s[j];
            while (url2.contains(".")) {
                i = banned_site_list.findInList(url2.toCharArray());
                if (i >= 0) {
                    return i;  // exact match
                }
                url2 = url2.after(".");  // check for being in hld
            }
        }
    }
    while (url.contains(".")) {
        i = banned_site_list.findInList(url.toCharArray());
        if (i >= 0) {
            return i;  // exact match
        }
        url = url.after(".");  // check for being in higher level domains
    }
    return -1;  // and our survey said "UUHH UURRGHH"
}


int OptionContainer::inBannedExtensionList(String url) {
    url.removeWhiteSpace();  // just in case of weird browser crap
    url.toLower();
    url.removePTP();  // chop off the ht(f)tp(s)://
    url = url.after("/");  // chop off any domain before the path
    if (url.length() < 2) {  // will never match
        return -1;
    }
    return banned_extension_list.findEndsWith(url.toCharArray());
}


int OptionContainer::inBannedURLList(String url) {
    int i;
    url.removeWhiteSpace();  // just in case of weird browser crap
    url.toLower();
    url.removePTP();  // chop off the ht(f)tp(s)://
    url.removeMultiSlash();  // remove mulitple forward slashes
    if (url.endsWith("/")) {
        url.chop();  // chop off trailing / if any
    }
    if (reverse_lookups == 1 && url.after("/").length() > 0) {
        String hostname = url.before("/");
        if (isIPHostname(hostname)) {
            std::deque<String> url2s = ipToHostname(hostname);
            String url2;
            unsigned int j;
            for (j = 0; j < url2s.size(); j++) {
                url2 = url2s[j];
                url2 += "/";
                url2 += url.after("/");
                while (url2.before("/").contains(".")) {
                    i = banned_url_list.findStartsWith(url2.toCharArray());
                    if (i >= 0) {
                        return i;  // exact match
                    }
                    url2 = url2.after(".");  // check for being in hld
                }
            }
        }
    }
    while (url.before("/").contains(".")) {
        i = banned_url_list.findStartsWith(url.toCharArray());
        if (i >= 0) {
            return i;  // exact match
        }
        url = url.after(".");  // check for being in higher level domains
    }
    return -1;
}

int OptionContainer::inBannedRegExpURLList(String url) {
    url.removeWhiteSpace();  // just in case of weird browser crap
    url.toLower();
    url.removePTP();  // chop off the ht(f)tp(s)://
    if (url.endsWith("/")) {
        url.chop();  // chop off trailing / if any
    }
    unsigned int i;
    for (i = 0; i < banned_regexpurl_list_comp.size(); i++) {
        banned_regexpurl_list_comp[i].match(url.toCharArray());
        if (banned_regexpurl_list_comp[i].matched()) {
            return i;
        }
    }
    return -1;
}

bool OptionContainer::inuserexceptions(std::string user) {
    if (user.length() < 1) {
        return false;
    }
    return exception_user_list.inList((char*)user.c_str());
}


bool OptionContainer::inipexceptions(std::string ip) {
    return exception_ip_list.inList((char*)ip.c_str());
}

bool OptionContainer::inBannedUserList(std::string user) {
    if (user.length() < 1) {
        return false;
    }
    return banned_user_list.inList((char*)user.c_str());
}

bool OptionContainer::inBannedIPList(std::string ip) {
    return banned_ip_list.inList((char*)ip.c_str());
}

std::deque<String> OptionContainer::ipToHostname(String ip) {
    std::deque<String> result;
    struct in_addr address, ** addrptr;
    if (inet_aton(ip.toCharArray(), &address))  {  // convert to in_addr
        struct hostent* answer;
        answer = gethostbyaddr((char*) &address, sizeof(address), AF_INET);
        if (answer) {  // sucess in reverse dns
            result.push_back(String(answer->h_name));
            for (addrptr = (struct in_addr **) answer->h_addr_list;
                 *addrptr; addrptr++) {
                result.push_back(String(inet_ntoa(**addrptr)));
            }
        }
    }
    return result;
}

bool OptionContainer::isIPHostname(String url) {
//    RegExp re;
//    re.comp(".*[a-z|A-Z].*");
    if (!isiphost.match(url.toCharArray())) {
        return true;
    }
    return false;
}


int OptionContainer::findoptionI(const char* option) {
    int res = String(findoptionS(option).c_str()).toInteger();
    return res;
}


std::string OptionContainer::findoptionS(const char* option) {
      // findoptionS returns a found option stored in the deque
    String temp;
    String temp2;
    String o = option;
    for (int i = 0; i < (signed)conffile.size(); i++) {
        temp = conffile[i].c_str();
        temp2 = temp.before("=");
        while(temp2.endsWith(" ")) { // get rid of tailing spaces before =
            temp2.chop();
        }
        if (o == temp2) {
            temp = temp.after("=");
            while(temp.startsWith(" ")) { // get rid of heading spaces
                temp.lop();
            }
            if(temp.startsWith("'")) { // inverted commas
                temp.lop();
            }
            while(temp.endsWith(" ")) { // get rid of tailing spaces
                temp.chop();
            }
            if(temp.endsWith("'")) { // inverted commas
                temp.chop();
            }
            return temp.toCharArray();
        }
    }
    return "";
}

bool OptionContainer::realitycheck(String s, int minl, int maxl, char* emessage) {
      // realitycheck checks a String for certain expected criteria
      // so we can spot problems in the conf files easier
    if ((signed)s.length() < minl) {
        if (!isDaemonised) {
            std::cerr << emessage << std::endl;
                                   // when called we have not detached from
                                   // the console so we can write back an
                                   // error

            std::cerr << "Too short or missing." << std::endl;
        }
        syslog(LOG_ERR, "%s", emessage);
        syslog(LOG_ERR, "%s", "Too short or missing.");

        return false;
    }
    if ((signed)s.length() > maxl && maxl > 0) {
        if (!isDaemonised) {
            std::cerr << emessage << std::endl;
            std::cerr << "Too long or broken." << std::endl;
        }
        syslog(LOG_ERR, "%s", emessage);
        syslog(LOG_ERR, "%s", "Too long or broken.");
        return false;
    }
    return true;
}


bool OptionContainer::precompileregexps() {
    if (!pics1.comp("pics-label\"[ \t]*content=[\'\"]([^>]*)[\'\"]")) {
        if (!isDaemonised) {
            std::cerr << "Error compiling RegExp pics1." << std::endl;
        }
        syslog(LOG_ERR, "%s", "Error compiling RegExp pics1.");
        return false;
    }
    if (!pics2.comp("[r|{ratings}] *\\(([^\\)]*)\\)")) {
        if (!isDaemonised) {
            std::cerr << "Error compiling RegExp pics2." << std::endl;
        }
        syslog(LOG_ERR, "%s", "Error compiling RegExp pics2.");
        return false;
    }
    if (!isiphost.comp(".*[a-z|A-Z].*")) {
        if (!isDaemonised) {
            std::cerr << "Error compiling RegExp isiphost." << std::endl;
        }
        syslog(LOG_ERR, "%s", "Error compiling RegExp isiphost.");
        return false;
    }

    return true;
}

