/*=============================================================================
 *  Copyright 2002-2004 deny all - Asphanet S.A. (http://www.deny-all.com/)
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *=============================================================================
 *
 * NAME
 *	mod_securid.c - RSA ACE/SecurID authentication module for Apache
 *
 * AUTHORS
 *	Erwan Legrand <elegrand@deny-all.com>
 *      mod_securid 1.x was writen by Patrick Asty <pasty@deny-all.com>
 *
 * VERSION
 *	$Revision: 1.15 $
 *
 * DOCUMENTATION
 *	See doc/index.html...
 *
 * HOW DOES IT WORK?
 *	A (very) short explanation...
 *
 *	For each URIs, Apache calls securid_check_auth (to check that
 *	authentication is correct) and then securid_check_access (to check
 *	that user can access this URI):
 *
 *	securid_check_auth ()
 *	  if auth ko
 *	    Redirect, Location: URI/securid/auth?`ap_escape_uri (URI[?args])`
 *
 *	securid_check_access ()
 *	  if user ko
 *	    Redirect, Location: URI/securid/auth?`ap_escape_uri (URI[?args])`
 *
 *	"URI/securid/auth" is handled by securid_handler_auth to print the
 *	authentication form:
 *
 *	securid_handler_auth ()
 *	  referer = ap_unescape_url (r->args)
 *	  add header "Set-Cookie: AceHandle=NNNN"
 *	  securid_print_form (escape_html (referer))
 *
 *	securid_print_form (hreferer)
 *	  just print form, using some "INPUT TYPE=TEXT" (username), "PASSWORD"
 *	  (passcode) and "HIDDEN" (referer, VALUE="hreferer").
 *
 *	When user POST the form, ACTION is "URI/securid/check", and BODY like:
 *	  "username=_u_&passcode=_p_&referer=_r_
 *	with
 *	  _u_	`ap_escape_uri (username field)`;
 *	  _p_	`ap_escape_uri (passcode field)`;
 *	  _r_	`ap_escape_uri (hreferer)`;
 *	(note: all ap_escape_uri actions are browser's job).
 *
 *	"URI/securid/check" is handled by securid_handler_check to check
 *	username and passcode (using ACE API):
 *
 *	securid_handler_check:
 *	  username = ap_unescape_uri (_u_)
 *	  passcode = ap_unescape_uri (_p_)
 *	  referer  = ap_unescape_uri (_r_)
 *	  if sd_check (username, passcode) == ok
 *	    add header "Set-Cookie: webid2=xxx"
 *	    Redirect, Location: referer
 *	  else
 *	    securid_auth_failed (escape_html (referer))
 *
 *	securid_auth_failed (hreferer)
 *	  just print to error messages (auth failed, next code needed, ...).
 *
 *=============================================================================
 */
static char const rcsid [] = "$Id: mod_securid.c,v 1.15 2004/02/25 17:00:51 elegrand Exp $";

/*
 * "Standard" includes
 */
#define	CORE_PRIVATE
#include "httpd.h"
#include "http_config.h"
#include "http_protocol.h"
#include "http_log.h"
#include "util_script.h"
#include "http_main.h"
#include "http_request.h"
#include "http_core.h"
#include "util_md5.h"
#include "http_conf_globals.h"
#ifdef SECURID_USE_GDBM	/* { */
# include <gdbm.h>
  /*
   * "Compat"; as we already log/unlock .lck, we don't need gdbm to do it.
   */
# define XDBM_FILE	GDBM_FILE
# define xdbm_open	gdbm_open
# define xdbm_fetch	gdbm_fetch
# define xdbm_dirfno	gdbm_fdesc
# define xdbm_close	gdbm_close
# define XDBM_READER	(GDBM_READER | GDBM_NOLOCK)
# define XDBM_WRITER	(GDBM_WRITER | GDBM_NOLOCK)
# define xdbm_store	gdbm_store
# define XDBM_REPLACE	GDBM_REPLACE
# define XDBM_INSERT	GDBM_INSERT
# define xdbm_error(fd)	gdbm_errno
# define xdbm_delete	gdbm_delete
# define XDBM_NEWDB	(GDBM_NEWDB | GDBM_NOLOCK)
# define XDBM_BSIZE	, 0
# define XDBM_FFUNC	, 0
# define xdbm_firstkey	gdbm_firstkey
# define xdbm_nextkey(x,y)	gdbm_nextkey(x,y)
#else	/* }{ */
# if defined (__GLIBC__)					&&	\
     defined (__GLIBC_MINOR__)					&&	\
     __GLIBC__ >= 2						&&	\
     __GLIBC_MINOR__ >= 1
#   include <db1/ndbm.h>
# else
#   include <ndbm.h>
# endif
  /*
   * "Compat"
   */
# define XDBM_FILE	DBM	*
# define xdbm_open	dbm_open
# define xdbm_fetch	dbm_fetch
# define xdbm_dirfno	dbm_dirfno
# define xdbm_close	dbm_close
# define XDBM_READER	O_RDONLY
# define XDBM_WRITER	O_WRONLY
# define xdbm_store	dbm_store
# define XDBM_REPLACE	DBM_REPLACE
# define XDBM_INSERT	DBM_INSERT
# define xdbm_error(fd)	dbm_error(fd)
# define xdbm_delete	dbm_delete
# define XDBM_NEWDB	(O_WRONLY | O_CREAT | O_TRUNC)
# define XDBM_BSIZE	
# define XDBM_FFUNC	
# define xdbm_firstkey	dbm_firstkey
# define xdbm_nextkey(x,y)	dbm_nextkey(x)
#endif	/* } */

/*
 * Local includes and defines
 */
#include <sys/types.h>
#include <utime.h>
#ifndef FALSE
#define FALSE	0
#define TRUE	!FALSE
#endif

/*
 * SecurID includes
 */
#include "acexport.h"

/*
 * mod_securid local includes
 */
#include "securid_conf.h"
#include "securid_client.h"
#include "securid_webid.h"

#define ACM_NEW_PIN_GENERATED	-(ACM_NEW_PIN_REQUIRED)

/*
 * File locking (from mod_rewrite.h)
 */
#if defined(USE_FCNTL_SERIALIZED_ACCEPT)
#define USE_FCNTL 1
#include <fcntl.h>
#endif
#if defined(USE_FLOCK_SERIALIZED_ACCEPT)
#define USE_FLOCK 1
#include <sys/file.h>
#endif
#if !defined(USE_FCNTL) && !defined(USE_FLOCK)
#define USE_FLOCK 1
#if !defined(MPE) && !defined(WIN32) && !defined(__TANDEM)
#include <sys/file.h>
#endif
#ifndef LOCK_UN
#undef USE_FLOCK
#define USE_FCNTL 1
#include <fcntl.h>
#endif
#endif
#ifdef AIX
#undef USE_FLOCK
#define USE_FCNTL 1
#include <fcntl.h>
#endif
#ifdef WIN32
#undef USE_FCNTL
#define USE_LOCKING
#include <sys/locking.h>
#endif

/*
 * "./Configure" directives: the text used is between -START and -END.
 *
 * MODULE-DEFINITION-START
 * Name: securid_module
 * ConfigStart
    if [ "$SECURID_USE_GDBM" ]; then
      echo "      + using GDBM library" 2>&1
      CFLAGS="$CFLAGS -DSECURID_USE_GDBM"
      LIBS="$LIBS -lgdbm"
    else
      . ./helpers/find-dbm-lib
    fi
    test "$ACE"     &&     ace="$ACE"     ||     ace=/usr/ace
    test "$ACE_LIB" && ace_lib="$ACE_LIB" || ace_lib="$ace/examples"
    test "$ACE_INC" && ace_inc="$ACE_INC" || ace_inc="$ace/examples"
    if [ -f "$ace_lib/sdiclient.a" ]; then
      echo "      + using ACE library $ace_lib/sdiclient.a" 2>&1
      LIBS="$LIBS $ace_lib/sdiclient.a"
    else
      echo "      + no ACE library under $ace_lib" 2>&1
      echo "      + (from `/bin/pwd`)" 2>&1
      echo "      + use/adjust ACE or ACE_LIB environment variable..." 2>&1
      exit 1
    fi
    if [ -f "$ace_inc/sdi_athd.h" ]; then
      echo "      + using ACE include $ace_inc" 2>&1
      CFLAGS="$CFLAGS -I$ace_inc"
    else
      echo "      + no ACE include under $ace_inc" 2>&1
      echo "      + (from `/bin/pwd`)" 2>&1
      echo "      + use/adjust ACE or ACE_INC environment variable..." 2>&1
      exit 1
    fi
 * ConfigEnd
 * MODULE-DEFINITION-END
 */

/*
 * Pre-declaration of the module for the cross-references
 */
module securid_module;

/*
 * For "Copyright"
 */
#define SECURID_URL_HOME "http://www.deny-all.com/mod_securid/"

/*
 * We need 2 handlers: 1 for authentication form (auth-handler) and 1 to check
 * this form (check-handler).
 *
 * auth-handler will display the authentication form. This form uses
 * ACTION="URI/securid/check" (method=POST), to execute the handler carrying out
 * the authentication.
 *
 * If the authentication fails, the user is redirected (code 302) to
 * "URI/securid/auth".
 *
 * Note that it is important that the handler carrying out the authentication
 * (URI "/xxx-check") can, if authentication fails, redirect to another
 * URI ("/xxx-auth"), which is not the same. Else, a redirect to the same
 * URI is not a good idea for browsers...
 *
 * Also note that those handlers are "called" by securid_translate().
 */
#define SECURID_URL		"/securid/"
#define SECURID_URL_AUTH	SECURID_URL "auth"
#define SECURID_URL_CHECK	SECURID_URL "check"
#define SECURID_URL_STATUS	SECURID_URL "status"
#define SECURID_URL_AUTH_LEN	(sizeof (SECURID_URL_AUTH) - 1)
#define SECURID_URL_CHECK_LEN	(sizeof (SECURID_URL_CHECK) - 1)
#define SECURID_URL_STATUS_LEN	(sizeof (SECURID_URL_STATUS) - 1)

/*
 * Handler names
 */
#define	SECURID_HANDLER_AUTH	"securid-auth"
#define	SECURID_HANDLER_CHECK	"securid-check"
#define	SECURID_HANDLER_STATUS	"securid-status"

/*
 * Field names and value in authentication forms.
 */
#define SECURID_FORM_N_ACTION		"sd_action"
#define SECURID_FORM_V_PASSCODE		"passcode"
#define SECURID_FORM_V_NEXTCODE		"next_code"
#define SECURID_FORM_V_NEWPIN		"new_pin"
#define SECURID_FORM_V_NEWPINUSR	SECURID_FORM_V_NEWPIN "_usr"
#define SECURID_FORM_V_NEWPINSYS	SECURID_FORM_V_NEWPIN "_sys"
#define SECURID_FORM_N_USERNAME		"sd_username"
#define SECURID_FORM_N_PASSCODE		"sd_passcode"
#define SECURID_FORM_N_REFERER		"sd_referer"
#define SECURID_FORM_1_FOCUS		\
		"onLoad=\"document.forms[0].elements[1].focus ();\""
#define SECURID_FORM_2_FOCUS		\
		"onLoad=\"document.forms[0].elements[2].focus ();\""

/*
 * Default cookies name: handle (AceHandle) and cookie (webid2).
 */
#define SECURID_HANDLE_NAME	"AceHandle"
#define SECURID_COOKIE_NAME	"webid2"

/*
 * "Directory" config for the module
 */
typedef struct securid_dconf
{
  int			authoritative;	/* others auth after SecurID?	*/
} securid_dconf;

/*
 * Custom file prefix
 */
#define	SECURID_CUSTOM_AUTH		0
#define	SECURID_CUSTOM_CHECK		1
#define	SECURID_CUSTOM_STATUS		2
#define	SECURID_CUSTOM_AUTH_STR		"auth"
#define	SECURID_CUSTOM_CHECK_STR	"err"
#define	SECURID_CUSTOM_STATUS_STR	"status"

/*
 * mode_t for open()
 */
#ifdef WIN32
# define SECURID_OPEN_MODE	(_S_IREAD | _S_IWRITE)
#else
# define SECURID_OPEN_MODE	(S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)
#endif

/*
 * Macro to compute redirect location for auth. handler:
 * - if we are a standard-proxy:
 *   escape_uri ("uri") ("uri" is the full uri, *with* args);
 * - else (we are a web server or a reverse-proxy):
 *   escape_uri ("uri?args") or escape_uri ("uri"), depends on args/no arg.
 *
 * Values for r->proxyreq are:
 *	NOT_PROXY	we are a web-server
 *	STD_PROXY	we are a standard-proxy
 *	PROXY_PASS	we are a reverse-proxy
 */
#define SECURID_REFERER(r)						\
	ap_escape_uri (r->pool,						\
		       r->proxyreq == STD_PROXY				\
		       ? r->uri						\
		       : (r->args					\
		          ? ap_psprintf (r->pool, "%s?%s", r->uri, r->args)\
			  : r->uri))

/*
 * Macro to compute URL location for securid handlers
 */
#define SECURID_URL_FORM(r,priv_path)					\
	ap_pstrcat (r->pool, priv_path,					\
	            priv_path [strlen (priv_path) - 1]			\
		      == SECURID_URL [0]				\
		    ? SECURID_URL + 1 : SECURID_URL, NULL)

#define SECURID_URL_FORM_AUTH(r,priv_path)				\
	ap_pstrcat (r->pool, priv_path,					\
	            priv_path [strlen (priv_path) - 1]			\
		      == SECURID_URL_AUTH [0]				\
		    ? SECURID_URL_AUTH + 1 : SECURID_URL_AUTH, NULL)

#define SECURID_URL_FORM_CHECK(r,priv_path)				\
	ap_pstrcat (r->pool, priv_path,					\
	            priv_path [strlen (priv_path) - 1]			\
		      == SECURID_URL_CHECK [0]				\
		    ? SECURID_URL_CHECK + 1 : SECURID_URL_CHECK, NULL)

#define SECURID_URL_FORM_STATUS(r,priv_path)				\
	ap_pstrcat (r->pool, priv_path,					\
	            "?logout", NULL)

/*
 ************************************************************************
 * File locking and log time (see mod_rewrite.c)
 ************************************************************************
 */
#define	SECURID_LOCKFEXT	".lck"

#ifdef USE_FCNTL
static struct flock	lock_it;
static struct flock	unlock_it;
#endif

static int db_lock (const char *fname, const int mode, const int logerr)
{
  int	fd;
  int	rc;
#ifdef USE_LOCKING
  off_t	offset;
#endif

  /*
   * Try to open lock file
   */
  errno = 0;
  do
  {
    fd = open (fname, O_RDWR);
  } while (fd < 0 && errno == EINTR);

  if (fd < 0)
  {
    if (logerr)
    {
      ap_log_error (APLOG_MARK, APLOG_ERR, NULL,
		    "SecurID: could not open AuthSecurID_Cache-lock file `%s'",
		    fname);
    }
    return -1;
  }

  /*
   * Update the times of the cache lock file, to discourage users from
   * deleting it. Note that we can't just re-create the lock file here
   * (if necessary), as we may no longer have write permission to the
   * cache lock file's directory...
   */
  if (utime (fname, NULL) < 0)
  {
    ap_log_error (APLOG_MARK, APLOG_ERR, NULL,
		  "SecurID: could not touch AuthSecurID_Cache-lock file `%s'",
		  fname);
  }

  /*
   * Try to lock cache file
   */
  errno = 0;

#ifdef USE_FCNTL
  lock_it.l_whence = SEEK_SET;		/* from current point		*/
  lock_it.l_start  = 0;			/* -"-				*/
  lock_it.l_len    = 0;			/* until end of file		*/
  lock_it.l_type   = mode==XDBM_READER	/* set exclusive and		*/
		       ? F_RDLCK	/* read lock			*/
		       : F_WRLCK;	/* or write lock		*/
  lock_it.l_pid    = 0;			/* pid not actually interesting	*/

  do
  {
    rc = fcntl (fd, F_SETLKW, &lock_it);
  } while (rc < 0 && errno == EINTR);
#elif USE_FLOCK
  do
  {
    rc = flock (fd, LOCK_EX);
  } while (rc < 0 && errno == EINTR);
#elif USE_LOCKING
  /* Store offset, lock the first byte always, and restore offset */
  offset = lseek (fd, 0, SEEK_CUR);
  lseek (fd, 0, SEEK_SET);
  rc = _locking (fd, _LK_LOCK, 1);
  lseek (fd, offset, SEEK_SET);
#else
#error -Dxxx error: use USE_FCNTL, USE_FLOCK or USE_LOCKING
#endif

  if (rc < 0)
  {
    if (logerr)
    {
      ap_log_error (APLOG_MARK, APLOG_ERR, NULL,
		    "SecurID: could not lock AuthSecurID_Cache-lock file `%s'",
		    fname);
    }
    return -1;
  }

  return fd;
}

static void db_unlock (int fd)
{
  int	rc;
#ifdef USE_LOCKING
  off_t	offset;
#endif

  /*
   * We're just trying to be kosher here by explicitly unlocking the cache
   * lock descriptor.
   */
  errno = 0;

#ifdef USE_FCNTL
  unlock_it.l_whence = SEEK_SET;	/* from current point		*/
  unlock_it.l_start  = 0;		/* -"-				*/
  unlock_it.l_len    = 0;		/* until end of file		*/
  unlock_it.l_type   = F_UNLCK;		/* unlock			*/
  unlock_it.l_pid    = 0;		/* pid not actually interesting */

  do
  {
    rc = fcntl (fd, F_SETLKW, &unlock_it);
  } while (rc < 0 && errno == EINTR);
#elif USE_FLOCK
  do
  {
    rc = flock (fd, LOCK_UN);
  } while (rc < 0 && errno == EINTR);
#elif USE_LOCKING
  /* Store offset, unlock the first byte always, and restore offset */
  offset = lseek (fd, 0, SEEK_CUR);
  lseek (fd, 0, SEEK_SET);
  rc = _locking (fd, _LK_UNLOCK, 1);
  lseek (fd, offset, SEEK_SET);
#else
#error -Dxxx error: use USE_FCNTL, USE_FLOCK or USE_LOCKING
#endif

  if (rc < 0)
  {
    ap_log_error (APLOG_MARK, APLOG_ERR, NULL,
		  "SecurID: could not unlock AuthSecurID_Cache-lock file "
		  "`fd=%d'", fd);
  }

  /*
   * We also close(2) the descriptor at the end of this function, so that
   * will implicitly release all locks anyway.
   */
  errno = 0;
  do
  {
    rc = close (fd);
  } while (rc < 0 && errno == EINTR);

  if (rc < 0)
  {
    ap_log_error (APLOG_MARK, APLOG_ERR, NULL,
		  "SecurID: could not close AuthSecurID_Cache-lock file "
		  "`fd=%d'", fd);
  }
}

/*
 ************************************************************************
 * auth. cache: securid_auth_{open,close,get,put,del}
 ************************************************************************
 */
XDBM_FILE	securid_auth_open (request_rec *r, char *fname,
		                   char *lockfname, int flags, int *lockfd)
{
  XDBM_FILE	cachefd;

  *lockfd = db_lock (lockfname, flags, 1 /* logerr */);
  if (*lockfd < 0)
  {
    return NULL;
  }
  cachefd = xdbm_open (fname XDBM_BSIZE, flags, SECURID_OPEN_MODE XDBM_FFUNC);
  if (cachefd == NULL)
  {
    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
                   "SecurID: could not open AuthSecurID_Cache file `%s'",
		   fname);
    db_unlock (*lockfd);
  }
  return cachefd;
}

void securid_auth_close (request_rec *r, XDBM_FILE fdesc, int lockfd)
{
  xdbm_close (fdesc);
  db_unlock (lockfd);
}

/*
 * auth_get: get the key "handle" and put its value in "webid".
 * Return value
 *	 0	ok, handle exists
 *	 1	ko, handle unknown
 *	-1	on xdbm error
 */
int securid_auth_get (request_rec *r, char *fname, char *lockfname,
		      const char *handle, securid_webid *webid)
{
  datum		key;
  datum		val;
  XDBM_FILE	cachefd;
  int		lockfd;
  
  if ((cachefd = securid_auth_open (r, fname, lockfname, XDBM_READER,
                                    &lockfd)) == NULL)
  {
    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
                   "SecurID: could not open AuthSecurID_Cache file `%s'",
		   fname);
    return -1;
  }

  key.dptr  = (void *) handle;
  key.dsize = strlen (handle) + 1;
  val       = xdbm_fetch (cachefd, key);
  if (val.dptr != NULL)
  {
    memcpy (webid, val.dptr, sizeof (securid_webid));
#   ifdef SECURID_USE_GDBM
      free (val.dptr);
#   endif
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: auth_get: `%s' is found: "
		   "user = %s, first_time = %ld, last_time = %ld, "
		   "priv_path = %s",
		   (long) getpid (), handle, webid->username, webid->first_time,
		   webid->last_time, webid->priv_path);
#   endif
    securid_auth_close (r, cachefd, lockfd);
    return 0;
  }
# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: auth_get: `%s' is NOT found",
		 (long) getpid (), handle);
# endif
  securid_auth_close (r, cachefd, lockfd);
  return 1;
}

/*
 * auth_put: store the key "handle" with value "webid" in file "fname".
 * Return value
 *	 0	on success
 *	-1	on xdbm error
 */
int securid_auth_put (request_rec *r, char *fname, char *lockfname,
                      const char *handle, securid_webid *webid)
{
  datum		key;
  datum		val;
  XDBM_FILE	cachefd;
  int		lockfd;
  
  if ((cachefd = securid_auth_open (r, fname, lockfname, XDBM_WRITER,
                                    &lockfd)) == NULL)
  {
    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
                   "SecurID: could not open AuthSecurID_Cache file `%s'",
		   fname);
    return -1;
  }

  key.dptr  = (void *) handle;
  key.dsize = strlen (handle) + 1;
  val.dptr  = (void *) webid;
  val.dsize = sizeof (securid_webid);
  if (xdbm_store (cachefd, key, val, XDBM_REPLACE) != 0)
  {
    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
                   "SecurID: store error for AuthSecurID_Cache file "
		   "(dbm err #%d for key `%s')",
		   xdbm_error (cachefd), key.dptr);
    securid_auth_close (r, cachefd, lockfd);
    return -1;
  }
# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: auth_put: `%s' is stored",
		 (long) getpid (), handle);
# endif
  securid_auth_close (r, cachefd, lockfd);
  return 0;
}

/*
 * auth_del: delete the key "handle" with its value.
 * Return value
 *	 0	on success
 *	-1	on xdbm error
 */
int securid_auth_del (request_rec *r, char *fname, char *lockfname,
                      const char *handle)
{
  datum		key;
  XDBM_FILE	cachefd;
  int		lockfd;
  
  if ((cachefd = securid_auth_open (r, fname, lockfname, XDBM_WRITER,
                                    &lockfd)) == NULL)
  {
    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
                   "SecurID: could not open AuthSecurID_Cache file `%s'",
		   fname);
    return -1;
  }

  key.dptr  = (void *) handle;
  key.dsize = strlen (handle) + 1;
  if (xdbm_delete (cachefd, key) != 0)
  {
    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
                   "SecurID: delete error for AuthSecurID_Cache file "
		   "(dbm err #%d for key `%s')",
		   xdbm_error (cachefd), key.dptr);
    securid_auth_close (r, cachefd, lockfd);
    return -1;
  }
# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: auth_del: `%s' is deleted",
		 (long) getpid (), handle);
# endif
  securid_auth_close (r, cachefd, lockfd);
  return 0;
}

/*
 * auth_dos:
 * 1/ if maxcachesize != 0, check no more than maxcachesize entries in cache
 * 2/ if maxauthget != 0, count how many "from" are in cache and delete them
 *    if more than maxauthget.
 * Return value
 *	 0	ok
 *	 1	maxcachesize reached
 *	 2	maxauthget reached
 *	-1	on xdbm error
 */
int securid_auth_dos (request_rec *r, char *fname, char *lockfname,
		      const char *from, int maxcachesize, int maxauthget)
{
  datum		key;
  datum		nextkey;
  datum		val;
  XDBM_FILE	cachefd;
  int		lockfd;
  securid_webid	webid;
  int		hdlused = 0;
  int		getused = 0;

  /*
   * maxcachesize/maxauthget == 0 means no check.
   */
  if (maxcachesize == 0 && maxauthget == 0)
  {
    return 0;
  }

  if ((cachefd = securid_auth_open (r, fname, lockfname, XDBM_WRITER,
                                    &lockfd)) == NULL)
  {
    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
                   "SecurID: could not open AuthSecurID_Cache file `%s'",
		   fname);
    return -1;
  }

  /*
   * Get the first key and loop
   */
  key = xdbm_firstkey (cachefd);
  while (key.dptr)
  {
    /*
     * Total entries in cache
     */
    hdlused++;
    if (maxcachesize && hdlused > maxcachesize)
    {
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		     "SecurID: DoS check: too many entries in "
		     "AuthSecurID_Cache file `%s'", fname);
#     ifdef SECURID_USE_GDBM
        free (key.dptr);
#     endif
      securid_auth_close (r, cachefd, lockfd);
      return 1;
    }
    /*
     * Get auth. for this key
     */
    val = xdbm_fetch (cachefd, key);
    if (val.dptr != NULL)
    {
      /*
       * Get auth. value and free gdbm.
       */
      memcpy (&webid, val.dptr, sizeof (securid_webid));
#     ifdef SECURID_USE_GDBM
        free (val.dptr);
#     endif
      /*
       * Check FromAgent (for "null" username: we don't want to take care
       * of "real" authenticated users).
       */
      if (webid.username [0] == '\0' && strcmp (from, webid.from_agent) == 0)
      {
	/*
	 * One more...
	 */
        getused++;
      }
    }
    else
    {
      /*
       * No value for this key: could this happen???
       */
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		     "SecurID: NULL auth for `%s' (??\?)", key.dptr);
    }

    /*
     * Free gdbm key and get next.
     */
    nextkey = xdbm_nextkey (cachefd, key);
#   ifdef SECURID_USE_GDBM
      free (key.dptr);
#   endif
    memcpy (&key, &nextkey, sizeof (key));
  }

# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: auth_dos: %d entries in cache",
		 (long) getpid (), hdlused);
# endif

  /*
   * No more check?
   */
  if (maxauthget == 0)
  {
    securid_auth_close (r, cachefd, lockfd);
    return 0;
  }

# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: auth_dos: FromAgent `%s' is already used %d",
		 (long) getpid (), from, getused);
# endif

  if (getused >= maxauthget)
  {
    /*
     * One more time do delete those auth.
     */
    key = xdbm_firstkey (cachefd);
    while (key.dptr)
    {
      val = xdbm_fetch (cachefd, key);
      if (val.dptr != NULL)
      {
	memcpy (&webid, val.dptr, sizeof (securid_webid));
#       ifdef SECURID_USE_GDBM
	  free (val.dptr);
#       endif
	if (webid.username [0] == '\0' && strcmp (from, webid.from_agent) == 0)
	{
	  /*
	   * Delete it.
	   */
	  if (xdbm_delete (cachefd, key) != 0)
	  {
	    /*
	     * Delete KO, continue with "next" key (else we could loop forever).
	     */
	    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
			   "SecurID: delete error for AuthSecurID_Cache file "
			   "(dbm err #%d for key `%s')",
			   xdbm_error (cachefd), key.dptr);
	    nextkey = xdbm_nextkey (cachefd, key);
	  }
	  else
	  {
	    /*
	     * Delete ok, restart xdbm search (because of `hash table'...).
	     */
#           ifdef SECURID_DEBUG
	    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
			   "#%ld: auth_dos: `%s' is deleted",
			   (long) getpid (), key.dptr);
#           endif
	    nextkey = xdbm_firstkey (cachefd);
	  }
	}
	else
	{
	  nextkey = xdbm_nextkey (cachefd, key);
	}
      }
      else
      {
	nextkey = xdbm_nextkey (cachefd, key);
      }
#     ifdef SECURID_USE_GDBM
        free (key.dptr);
#     endif
      memcpy (&key, &nextkey, sizeof (key));
    }
    /*
     * Return "too much"
     */
    securid_auth_close (r, cachefd, lockfd);
    return 2;
  }
  else
  {
    /*
     * "Ok".
     */
    securid_auth_close (r, cachefd, lockfd);
    return 0;
  }
}

/*
 ************************************************************************
 * Custom stuff
 ************************************************************************
 */
/*
 * Extract "%." (%s, %c, %d, ...) from string.
 * Return the "..." extracted or "" of no "%." or NULL if too many "%.".
 *
 * Example: for "mod_securid v %d.%d.%d is %s", we'll return "ddds".
 */
static char	*securid_customcheck (pool *p, char *string)
{
#define	SECURID_CUSTOM_MAXPC	32
  char	format [SECURID_CUSTOM_MAXPC + 1];
  int	found = 0;
  char	*c;

  for (c = string; *c; c++)
  {
    if (c [0] == '%')
    {
      if (c [1] != '%')
      {
	if (found < SECURID_CUSTOM_MAXPC)
	{
	  format [found] = c [1];
	  found++;
	}
	else
	{
	  return NULL;
	}
      }
      c++;
    }
  }
  format [found] = '\0';
  return ap_pstrdup (p, format);
}

/*
 * Auth. form message, triggered by SECURID_URL_AUTH url.
 */
static char	SECURID_FMT_AUTH [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID PASSCODE Request</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\" " SECURID_FORM_1_FOCUS ">"
	"    <H1 ALIGN=CENTER>SecurID PASSCODE Request</H1>"
	"    <HR>"
	"    <P>"
	"      The page you are attempting to access requires that you"
	"      authenticate using your SecurID token."
	"    </P>"
	"    <P>"
	"      Please enter your Username and SecurID PASSCODE in"
	"      the following fields, then click the &quot;Send&quot;"
	"      button. If you make a mistake, use the &quot;Reset&quot;"
	"      button to clear the fields."
	"    </P>"
	"    <HR>"
	"    <FORM method=POST ACTION=\"%s\">"
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_ACTION ""
	"                         VALUE=" SECURID_FORM_V_PASSCODE ">"
	"      <CENTER><TABLE>"
	"        <TR>"
	"          <TD><B>Username:</B></TD>"
	"          <TD><INPUT TYPE=TEXT NAME=" SECURID_FORM_N_USERNAME ""
	"                               MAXLENGTH=32></TD>"
	"        </TR>"
	"        <TR>"
	"          <TD><B>PASSCODE:</B></TD>"
	"          <TD>"
	"            <INPUT TYPE=PASSWORD NAME=" SECURID_FORM_N_PASSCODE ""
	"                                 MAXLENGTH=32>"
	"          </TD>"
	"        </TR>"
	"      </TABLE></CENTER>"
	"      <HR>"
	"      <CENTER><P>"
	"        <INPUT TYPE=SUBMIT VALUE=\"Send\">"
	"        <INPUT TYPE=RESET VALUE=\"Reset\">"
	"      </P></CENTER>"
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_REFERER ""
	"                         VALUE=\"%s\">"
	"    </FORM>"
	"    <HR>"
	"  </BODY>"
	"</HTML>\n";

/*
 * Auth. Failed messages, triggered by SECURID_URL_CHECK url.
 */
static char	SECURID_FMT_CHECK_NOACM [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID Access Denied</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID Access Denied</H1>"
	"    <HR>"
	"    <P>"
	"      Authentication cannot be performed!!!."
	"      Please contact the ACE administrator."
	"      <br>"
	"      Use BACK to retry..."
	"    </P>"
	"  </BODY>"
	"</HTML>\n";
 
static char	SECURID_FMT_CHECK_NOHDL [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID Access Denied</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID Access Denied</H1>"
	"    <HR>"
	"    <P>"
	"      Authentication cannot be performed!!!."
	"      You should enable &quot;cookies&quot; on your browser."
	"      <br>"
	"      Use BACK to retry..."
	"    </P>"
	"  </BODY>"
	"</HTML>\n";
 
static char	SECURID_FMT_CHECK_NOGOOD [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID Access Denied</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID Access Denied</H1>"
	"    <HR>"
	"    <P>"
	"      Authentication cannot be performed!!!."
	"      Your browser sent some &quot;strange&quot; data."
	"      <br>"
	"      Use BACK to retry..."
	"    </P>"
	"  </BODY>"
	"</HTML>\n";
 
static char	SECURID_FMT_CHECK_MAXGREQ [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID Access Denied</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID Access Denied</H1>"
	"    <HR>"
	"    <P>"
	"      Authentication cannot be performed!!!."
	"      <b>Too many access failures</b> (get)."
	"    </P>"
	"  </BODY>"
	"</HTML>\n";
 
static char	SECURID_FMT_CHECK_MAXPREQ [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID Access Denied</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID Access Denied</H1>"
	"    <HR>"
	"    <P>"
	"      Authentication cannot be performed!!!."
	"      <b>Too many access failures</b> (post)."
	"    </P>"
	"  </BODY>"
	"</HTML>\n";
 
static char	SECURID_FMT_CHECK_ACCESS_DENIED [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID Access Denied</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID Access Denied</H1>"
	"    <HR>"
	"    <P>"
	"      The authenticate you provide is invalid!!!."
	"      Use BACK to retry..."
	"    </P>"
	"  </BODY>"
	"</HTML>\n";

static char	SECURID_FMT_CHECK_NEXT_CODE_REQUIRED [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID Next Tokencode Request</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\" " SECURID_FORM_2_FOCUS ">"
	"    <H1 ALIGN=CENTER>SecurID Next Tokencode Request</H1>"
	"    <HR>"
	"    <P>"
	"      You are required to enter the &quot;Next Tokencode&quot; from"
	"      your SecurID token. (The tokencode is the code that your token"
	"      generates <b>without</b> your PIN.)"
	"    </P>"
	"    <P>"
	"      Please wait for the code in the display of your token to"
	"      change, then enter the code in the &quot;Next Tokencode&quot;"
	"      field and press the &quot;Send&quot; button."
	"    </P>"
	"    <HR>"
	"    <FORM method=POST ACTION=\"%s\">"
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_ACTION ""
	"                         VALUE=" SECURID_FORM_V_NEXTCODE ">"
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_USERNAME ""
	"                         VALUE=\"%s\">"
	"      <CENTER><P>"
	"        Next Tokencode:"
	"        <INPUT TYPE=PASSWORD NAME=" SECURID_FORM_N_PASSCODE ""
	"                             MAXLENGTH=32>"
	"        <INPUT TYPE=SUBMIT VALUE=\"Send\">"
	"      </P></CENTER>"
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_REFERER ""
	"                         VALUE=\"%s\">"
	"    </FORM>"
	"    <HR>"
	"  </BODY>"
	"</HTML>\n";

static char	SECURID_FMT_CHECK_NEXT_CODE_BAD [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID Access Denied</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID Access Denied</H1>"
	"    <HR>"
	"    <P>"
	"      The Next Tokencode you provide is bad!!!."
	"      Use BACK to retry..."
	"    </P>"
	"  </BODY>"
	"</HTML>\n";

static char	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_CANNOT_CHOOSE [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID New PIN Request</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\" " SECURID_FORM_2_FOCUS ">"\
	"    <H1 ALIGN=CENTER>SecurID New PIN Request</H1>"
	"    <HR>"
	"    <P>"
	"      Your administrator has required to have a "
	"      system-generated PIN."
	"    </P>"
	"    <P>"
	"      Make sure nobody can see your screen, then click the"
	"      \"Receive System-Generated PIN\" button. The new PIN"
	"      will display for 10 seconds."
	"    </P>"
	"    <HR>"
	"    <FORM method=POST ACTION=\"%s\">"
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_ACTION ""
	"                         VALUE=" SECURID_FORM_V_NEWPINSYS ">"
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_USERNAME ""
	"                         VALUE=\"%s\">"
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_PASSCODE ""
	"                         VALUE=\"%s\">"
	"      <CENTER><P>"
	"        <INPUT TYPE=SUBMIT VALUE=\"Receive System-Generated PIN\">"
	"      </P></CENTER>"
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_REFERER ""
	"                         VALUE=\"%s\">"
	"    </FORM>"
	"    <HR>"
	"  </BODY>"
	"</HTML>\n";

#define		SECURID_FMT_ALPHA0	"digits."
#define		SECURID_FMT_ALPHA1	"characters and digits."

#define		SECURID_FMT_CHECK_NEW_PIN_REQUIRED_SELECTABLE_BEGIN\
	"<HTML>"\
	"  <HEAD><TITLE>SecurID New PIN Request</TITLE></HEAD>"\
	"  <BODY BGCOLOR=\"#FFFFFF\" " SECURID_FORM_2_FOCUS ">"\
	"    <H1 ALIGN=CENTER>SecurID New PIN Request</H1>"\
	"    <HR>"\
	"    <P>"\
	"      Your token is in New PIN mode."\
	"    </P>"\
	"    <P>"\
	"      You can either create your own PIN or have the system generate"\
	"      one for you."\
	"    </P>"\
	"    <P>"\
	"      To create your own PIN, type in your new PIN, then click the"\
	"      &quot;Send&quot; button. Use the &quot;Reset&quot; button to"\
	"      clear the PIN field if you make a mistake."\
	"    </P>"\
	"    <P>"\
	"      To receive a system-generated PIN, click the"\
	"      &quot;Receive System-Generated PIN&quot; button."\
	"      Your new PIN will be displayed in your browser for 10 seconds."\
	"    </P>"\
	"    <HR>"\
	"    <P>"\
	"      PINs must contain %d to %d "

#define		SECURID_FMT_CHECK_NEW_PIN_REQUIRED_SELECTABLE_END\
	"    </P>"\
	"    <CENTER><TABLE>"\
	"      <FORM method=POST ACTION=\"%s\">"\
	"        <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_ACTION ""\
	"                           VALUE=" SECURID_FORM_V_NEWPINUSR ">"\
	"        <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_USERNAME ""\
	"                           VALUE=\"%s\">"\
	"        <TR>"\
	"          <TD>"\
	"            <B>I will create my PIN</B>"\
	"          </TD>"\
	"          <TD>"\
	"            <INPUT TYPE=PASSWORD NAME=" SECURID_FORM_N_PASSCODE ""\
	"                                 MAXLENGTH=32>"\
	"          </TD>"\
	"          <TD>"\
	"            <INPUT TYPE=SUBMIT VALUE=\"Send\">"\
	"            <INPUT TYPE=RESET VALUE=\"Reset\">"\
	"          </TD>"\
	"        </TR>"\
	"        <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_REFERER ""\
	"                           VALUE=\"%s\">"\
	"      </FORM>"\
	"      <FORM method=POST ACTION=\"%s\">"\
	"        <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_ACTION ""\
	"                           VALUE=" SECURID_FORM_V_NEWPINSYS ">"\
	"        <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_USERNAME ""\
	"                           VALUE=\"%s\">"\
	"        <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_PASSCODE ""\
	"                           VALUE=\"%s\">"\
	"        <TR>"\
	"          <TD>"\
	"            <B>System-Generated PIN</B>"\
	"          </TD>"\
	"          <TD> </TD>"\
	"          <TD>"\
	"            <INPUT TYPE=SUBMIT"\
	"             VALUE=\"Receive System-Generated PIN\">"\
	"          </TD>"\
	"        </TR>"\
	"        <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_REFERER ""\
	"                           VALUE=\"%s\">"\
	"      </FORM>"\
	"    </TABLE></CENTER>"\
	"    <HR>"\
	"  </BODY>"\
	"</HTML>\n"

static char	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_SELECTABLE_ALPHA0 [] =
	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_SELECTABLE_BEGIN SECURID_FMT_ALPHA0
	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_SELECTABLE_END;

static char	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_SELECTABLE_ALPHA1 [] =
	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_SELECTABLE_BEGIN SECURID_FMT_ALPHA1
	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_SELECTABLE_END;

#define		SECURID_FMT_CHECK_NEW_PIN_REQUIRED_MUST_CHOOSE_BEGIN\
	"<HTML>"\
	"  <HEAD><TITLE>SecurID New PIN Request</TITLE></HEAD>"\
	"  <BODY BGCOLOR=\"#FFFFFF\" " SECURID_FORM_2_FOCUS ">"\
	"    <H1 ALIGN=CENTER>SecurID New PIN Request</H1>"\
	"    <HR>"\
	"    <P>"\
	"      Your token is in New PIN mode."\
	"    </P>"\
	"    <P>"\
	"      Enter a new PIN in the field provided, then click the"\
	"      &quot;Send&quot; button."\
	"      Use the &quot;Reset&quot; button to clear the PIN"\
	"      if you make a mistake."\
	"    </P>"\
	"    <P>"\
	"      <HR>"\
	"      PINs must contain %d to %d "

#define		SECURID_FMT_CHECK_NEW_PIN_REQUIRED_MUST_CHOOSE_END\
	"      <HR>"\
	"    </P>"\
	"    <P>"\
	"    <FORM method=POST ACTION=\"%s\">"\
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_ACTION ""\
	"                         VALUE=" SECURID_FORM_V_NEWPINUSR ">"\
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_USERNAME ""\
	"                         VALUE=\"%s\">"\
	"      <CENTER><TABLE>"\
	"        <TR>"\
	"          <TD><B>New PIN:</B></TD>"\
	"          <TD>"\
	"            <INPUT TYPE=PASSWORD NAME=" SECURID_FORM_N_PASSCODE ""\
	"                                 MAXLENGTH=32>"\
	"          </TD>"\
	"        </TR>"\
	"      </TABLE></CENTER>"\
	"      <CENTER><P>"\
	"        <INPUT TYPE=SUBMIT VALUE=\"Send\">"\
	"        <INPUT TYPE=RESET VALUE=\"Reset\">"\
	"      </P></CENTER>"\
	"      <INPUT TYPE=HIDDEN NAME=" SECURID_FORM_N_REFERER ""\
	"                         VALUE=\"%s\">"\
	"    </FORM>"\
	"    <HR>"\
	"  </BODY>"\
	"</HTML>\n"

static char	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_MUST_CHOOSE_ALPHA0 [] =
	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_MUST_CHOOSE_BEGIN SECURID_FMT_ALPHA0
	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_MUST_CHOOSE_END;

static char	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_MUST_CHOOSE_ALPHA1 [] =
	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_MUST_CHOOSE_BEGIN SECURID_FMT_ALPHA1
	SECURID_FMT_CHECK_NEW_PIN_REQUIRED_MUST_CHOOSE_END;

static char	SECURID_FMT_CHECK_NEW_PIN_GENERATED [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID System Generated PIN</TITLE>"
	"        <META HTTP-EQUIV=\"Refresh\" CONTENT=\"10; URL=%s\"></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID System Generated PIN</H1>"
	"    <HR>"
	"    <P>"
	"      Your New PIN is %s."
	"      Try to remember it and do not write it down."
	"    </P>"
	"    <P>"
	"      This page will change in 10 seconds. To proceed before"
	"      the 10 seconds elapse click the following link."
	"    </P>"
	"    <HR>"
	"    <P ALIGN=CENTER>"
	"      <A HREF=\"%s\">Continue</A>"
	"    </P>"
	"  </BODY>"
	"</HTML>\n";

static char	SECURID_FMT_CHECK_NEW_PIN_ACCEPTED [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID New PIN Accepted</TITLE>"
	"        <META HTTP-EQUIV=\"Refresh\" CONTENT=\"10; URL=%s\"></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID New PIN Accepted</H1>"
	"    <HR>"
	"    <P>"
	"      Your New PIN is valid."
	"    </P>"
	"    <P>"
	"      This page will change in 10 seconds. To proceed before"
	"      the 10 seconds elapse click the following link."
	"    </P>"
	"    <HR>"
	"    <P ALIGN=CENTER>"
	"      <A HREF=\"%s\">Continue</A>"
	"    </P>"
	"  </BODY>"
	"</HTML>\n";

static char	SECURID_FMT_CHECK_NEW_PIN_REJECTED [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID Access Denied</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID Access Denied</H1>"
	"    <HR>"
	"    <P>"
	"      The New PIN you provide is rejected!!!."
	"      Use BACK to retry..."
	"    </P>"
	"  </BODY>"
	"</HTML>\n";

static char	SECURID_FMT_CHECK_UNKNOWN_ERROR [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID Access Denied</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID Access Denied</H1>"
	"    <HR>"
	"    <P>"
	"      Please contact your administrator: unknown error %d!!!."
	"    </P>"
	"  </BODY>"
	"</HTML>\n";

/*
 * Status messages, triggered by SECURID_URL_STATUS url.
 */
static char	SECURID_FMT_STATUS_NOAUTH [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID module for Apache</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID module for Apache</H1>"
	"    <HR>"
	"    <P>"
	"      You are not authenticated with "
	"      <a href=" SECURID_URL_HOME ">mod_securid</a>..."
	"    </P>"
	"    <HR>"
	"  </BODY>"
	"</HTML>\n";

static char	SECURID_FMT_STATUS_AUTH0 [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID module for Apache</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID module for Apache</H1>"
	"    <HR>"
	"    <P>"
	"      You were authenticated as user <b>%s</b>:"
	"    </P>"
	"    <UL>"
	"      <TABLE>"
	"        <TR><TD><LI>username:</TD><TD><tt>%s</tt></TD><TR>"
	"        <TR><TD><LI>group(s):</TD><TD><tt>%s</tt></TD><TR>"
	"        <TR><TD><LI>first used:</TD><TD><tt>%ss ago</tt></TD><TR>"
	"      </TABLE>"
	"    </UL>"
	"    <P>"
	"      Will see you later..."
	"    </P>"
	"    <HR>"
	"  </BODY>"
	"</HTML>\n";

static char	SECURID_FMT_STATUS_AUTH1 [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID module for Apache</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID module for Apache</H1>"
	"    <HR>"
	"    <P>"
	"      You are authenticated as user <b>%s</b>:"
	"    </P>"
	"    <UL>"
	"      <TABLE>"
	"        <TR><TD><LI>username:</TD><TD><tt>%s</tt></TD><TR>"
	"        <TR><TD><LI>group(s):</TD><TD><tt>%s</tt></TD><TR>"
	"        <TR><TD><LI>first used:</TD><TD><tt>%ss ago</tt></TD><TR>"
	"        <TR><TD><LI>expiration:</TD><TD><tt>"
	"in %ss (even if you use your authentication)"
	"</tt></TD><TR>"
	"      </TABLE>"
	"    </UL>"
	"    <P>"
	"      Use this <A HREF=\"%s\">link</A>"
	"      to <B>logout</B>"
	"    </P>"
	"    <HR>"
	"  </BODY>"
	"</HTML>\n";

static char	SECURID_FMT_STATUS_AUTH2 [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID module for Apache</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID module for Apache</H1>"
	"    <HR>"
	"    <P>"
	"      You are authenticated as user <b>%s</b>:"
	"    </P>"
	"    <UL>"
	"      <TABLE>"
	"        <TR><TD><LI>username:</TD><TD><tt>%s</tt></TD><TR>"
	"        <TR><TD><LI>group(s):</TD><TD><tt>%s</tt></TD><TR>"
	"        <TR><TD><LI>first used:</TD><TD><tt>%ss ago</tt></TD><TR>"
	"        <TR><TD><LI>expiration:</TD><TD><tt>"
	"in %ss (if you do not use your authentication), or in %ss"
	"</tt></TD><TR>"
	"      </TABLE>"
	"    </UL>"
	"    <P>"
	"      Use this <A HREF=\"%s\">link</A>"
	"      to <B>logout</B>"
	"    </P>"
	"    <HR>"
	"  </BODY>"
	"</HTML>\n";

static char	SECURID_FMT_STATUS_AUTH3 [] =
	"<HTML>"
	"  <HEAD><TITLE>SecurID module for Apache</TITLE></HEAD>"
	"  <BODY BGCOLOR=\"#FFFFFF\">"
	"    <H1 ALIGN=CENTER>SecurID module for Apache</H1>"
	"    <HR>"
	"    <P>"
	"      You are authenticated as user <b>%s</b>:"
	"    </P>"
	"    <UL>"
	"      <TABLE>"
	"        <TR><TD><LI>username:</TD><TD><tt>%s</tt></TD><TR>"
	"        <TR><TD><LI>group(s):</TD><TD><tt>%s</tt></TD><TR>"
	"        <TR><TD><LI>first used:</TD><TD><tt>%ss ago</tt></TD><TR>"
	"        <TR><TD><LI>expiration:</TD><TD><tt>"
	"in %ss (if you do not use your authentication)"
	"</tt></TD><TR>"
	"      </TABLE>"
	"    </UL>"
	"    <P>"
	"      Use this <A HREF=\"%s\">link</A>"
	"      to <B>logout</B>"
	"    </P>"
	"    <HR>"
	"  </BODY>"
	"</HTML>\n";

/*
 * Compute custom error file:
 *
 * 1/ auth-handler (custompart == ..._AUTH)
 *     code1, code2 & sd are unused;
 *     return <customdir> + "auth." [ + <lang> ]
 *
 * 2/ check-handler (custompart == ..._CHECK)
 *     code1 is err_code (-6..-1), code2 is acm_code (-5, 1..7), sd is usefull;
 *     if code1 is not null
 *       code1 == -1 => code = 0.0,
 *       code1 == -2 => code = 0.1,
 *       code1 == -3 => code = 0.2, ...
 *     else
 *       code = code2
 *     return <customdir> + "err." + code [ + "." + <lang> ]
 *
 *     For NEW_PIN_..., also add sd->pin_params.Selectable (CANNOT_CHOOSE_PIN (0),
 *     USER_SELECTABLE (1) or MUST_CHOOSE_PIN (2)).
 *     For NEW_PIN_... *and not* CANNOT_CHOOSE_PIN, also add 0 for digit-only
 *     pin or 1 for alphanumeric pin.
 *
 *     For unknown error, change code1 to "x".
 *
 * 3/ status-handler (custompart == ..._STATUS)
 *     code1 is:
 *       -1: not-authenticated,
 *	  0: logout,
 *        1: authenticated, expiration == "even if you use ...",
 *        2: authenticated, expiration == "if you use ..., or ...",
 *        3: authenticated, expiration == "if you do not use ...",
 *     code2 & sd are unused;
 *     return <customdir> + "status." + code1 [ + "." + <lang> ]
 *
 * <customdir> is from "AuthSecurid_Custom".
 * <lang> is from "Accept-Language" HTTP header, if available; def. is "en".
 *
 * Then, we read the custom file and check "%.". If no problem, its content
 * is returned, else default (format) is returned.
 */
char *securid_customfile (request_rec *r, int custompart,
                          int code1, int code2, securid_webid *wid,
			  char *defautfmt)
{
  securid_sconf	*sconf =	/* server config (for customdir)	*/
    (securid_sconf *) ap_get_module_config (r->server->module_config,
					    &securid_module);
  const char	*customdirenv;	/* _CustomDir env. value		*/
  char		*customdir;	/* directive or env. value		*/
  char		*customfile;	/* custom filename			*/
  char		*customfmt;	/* content of custom file		*/
  struct stat	filestat;	/* buffer for file stats		*/
  const char	*langs;		/* Accept-Language header		*/
  char		*lang;		/* fr|en|..., from Accept-Language	*/
  char		*custompc;	/* "%." from custom file		*/
  char		*defautpc;	/* "%." from default format		*/

  /*
   * If no environment variable AuthSecurID_CustomDir, use the server
   * config value.
   */
  customdirenv = ap_table_get (r->subprocess_env, "AuthSecurID_CustomDir");
  if (customdirenv)
  {
    /*
     * Need to add "ServerRoot".
     */
    customdir = ap_server_root_relative (r->pool,
                                         ap_pstrdup (r->pool, customdirenv));
  }
  else
  {
    /*
     * "ServerRoot" has already been added (see cfg_customdir()).
     */
    customdir = sconf->customdir;
  }

  /*
   * If no custom dir, stop now.
   */
  if (!customdir)
  {
    return defautfmt;
  }

  /*
   * If invalid custom dir, stop now.
   */
  if ((sconf->customlink
       ? stat (customdir, &filestat) != 0
       : lstat (customdir, &filestat) != 0) ||
      !S_ISDIR (filestat.st_mode))
  {
    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
		   "SecurID: invalid AuthSecurID_CustomDir directory `%s'",
		   customdir);
    return defautfmt;
  }

  /*
   * Compute language
   */
  langs = ap_table_get (r->headers_in, "Accept-Language");
  if (langs == NULL)
  {
    /*
     * No user preference, choose en.
     */
    lang = "en";
  }
  else
  {
    /*
     * Get first token from header (delimited by semis and commas, 0 means
     * no whitespace).
     */
    lang = ap_get_token (r->pool, &langs, 0);
  }

  /*
   * Dispatch custom type
   */
  if (custompart == SECURID_CUSTOM_AUTH)
  {
    /*
     * Compute filename for auth-handler: "(customdir)/auth[.(lang)]"
     */
    customfile = ap_psprintf (r->pool, "%s/" SECURID_CUSTOM_AUTH_STR ".%s",
                              customdir, lang);
    if (sconf->customlink
        ? stat (customfile, &filestat) != 0
	: lstat (customfile, &filestat) != 0)
    {
      customfile = ap_psprintf (r->pool, "%s/" SECURID_CUSTOM_AUTH_STR,
                                customdir);
    }
  }
  else if (custompart == SECURID_CUSTOM_CHECK)
  {
    /*
     * Compute filename for check-handler
     */
    if (code1)
    {
      /*
       * -1 => 0.0,
       * -2 => 0.1,
       * -3 => 0.2, ...
       */
      code1 = -code1 - 1;
      /*
       * "(customdir)/err.0.(code1)[.(lang)]"
       */
      customfile = ap_psprintf (r->pool,
				"%s/" SECURID_CUSTOM_CHECK_STR ".0.%d.%s",
				customdir, code1, lang);
      if (sconf->customlink
	  ? stat (customfile, &filestat) != 0
	  : lstat (customfile, &filestat) != 0)
      {
	customfile = ap_psprintf (r->pool,
				  "%s/" SECURID_CUSTOM_CHECK_STR ".0.%d",
				  customdir, code1);
      }
    }
    else
    {
      if (code2 == ACM_NEW_PIN_REQUIRED)
      {
	if (wid->pin_params.Selectable == CANNOT_CHOOSE_PIN)
	{
	  /*
	   * "(customdir)/err.(code2).(sd->user_selectable)[.(lang)]"
	   */
	  customfile = ap_psprintf (r->pool,
				    "%s/" SECURID_CUSTOM_CHECK_STR ".%d.%d.%s",
				    customdir, code2,
				    wid->pin_params.Selectable, lang);
	  if (sconf->customlink
	      ? stat (customfile, &filestat) != 0
	      : lstat (customfile, &filestat) != 0)
	  {
	    customfile = ap_psprintf (r->pool,
				      "%s/" SECURID_CUSTOM_CHECK_STR ".%d.%d",
				      customdir, code2,
				      wid->pin_params.Selectable);
	  }
	}
	else
	{
	  /*
	   * "(customdir)/err.(code2).(sd->user_selectable)N[.(lang)]"
	   * N = "0" (digit pin) or "1" (alphanumeric pin).
	   */
	  customfile = ap_psprintf (r->pool,
				    "%s/" SECURID_CUSTOM_CHECK_STR
				    ".%d.%d%c.%s",
				    customdir, code2,
				    wid->pin_params.Selectable,
				    wid->pin_params.Alphanumeric ? '1' : '0', lang);
	  if (sconf->customlink
	      ? stat (customfile, &filestat) != 0
	      : lstat (customfile, &filestat) != 0)
	  {
	    customfile = ap_psprintf (r->pool,
				      "%s/" SECURID_CUSTOM_CHECK_STR ".%d.%d%c",
				      customdir, code2,
				      wid->pin_params.Selectable,
				      wid->pin_params.Alphanumeric ? '1' : '0');
	  }
	}
      }
      else if (code2 == ACM_NEW_PIN_GENERATED ||
	       (code2 >= 0 && code2 <= ACM_NEW_PIN_REJECTED))
      {
	/*
	 * Known errors: "(customdir)/err.(code2)[.(lang)]"
	 */
	customfile = ap_psprintf (r->pool,
				  "%s/" SECURID_CUSTOM_CHECK_STR ".%d.%s",
				  customdir, code2, lang);
	if (sconf->customlink
	    ? stat (customfile, &filestat) != 0
	    : lstat (customfile, &filestat) != 0)
	{
	  customfile = ap_psprintf (r->pool,
				    "%s/" SECURID_CUSTOM_CHECK_STR ".%d",
				    customdir, code2);
	}
      }
      else
      {
	/*
	 * Unknown errors: "(customdir)/err.x[.(lang)]"
	 */
	customfile = ap_psprintf (r->pool,
				  "%s/" SECURID_CUSTOM_CHECK_STR ".x.%s",
				  customdir, lang);
	if (sconf->customlink
	    ? stat (customfile, &filestat) != 0
	    : lstat (customfile, &filestat) != 0)
	{
	  customfile = ap_psprintf (r->pool,
				    "%s/" SECURID_CUSTOM_CHECK_STR ".x",
				    customdir);
	}
      }
    }
  }
  else
  {
    /*
     * Compute filename for status-handler: "(customdir)/err.(code1)[.(lang)]"
     */
    customfile = ap_psprintf (r->pool, "%s/" SECURID_CUSTOM_STATUS_STR ".%d.%s",
			      customdir, code1, lang);
    if (sconf->customlink
	? stat (customfile, &filestat) != 0
	: lstat (customfile, &filestat) != 0)
    {
      customfile = ap_psprintf (r->pool, "%s/" SECURID_CUSTOM_STATUS_STR ".%d",
				customdir, code1);
    }
  }

  /*
   * Check filename is ok (alloc from ap_psprintf).
   */
  if (customfile == NULL)
  {
    /*
     * Cannot alloc...
     */
    if (custompart == SECURID_CUSTOM_AUTH)
    {
      ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
		     "SecurID: could not compute custom filename `"
		     SECURID_CUSTOM_AUTH_STR "'");
    }
    else if (custompart == SECURID_CUSTOM_CHECK)
    {
      if (code1)
      {
	ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
		       "SecurID: could not compute custom filename `"
		       SECURID_CUSTOM_CHECK_STR ".0.%d'", -code1 - 1);
      }
      else
      {
	ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
		       "SecurID: could not compute custom filename `"
		       SECURID_CUSTOM_CHECK_STR ".%d'", code2);
      }
    }
    else
    {
      ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
		     "SecurID: could not compute custom filename `"
		     SECURID_CUSTOM_STATUS_STR ".%d'", code1);
    }
    return defautfmt;
  }
  else
  {
    /*
     * Alloc ok, check if file exists and is non-empty.
     */
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: customfile: file = %s",
		   (long) getpid (), customfile);
#   endif
    if ((sconf->customlink
	 ? stat (customfile, &filestat) == 0
	 : lstat (customfile, &filestat) == 0) &&
	S_ISREG (filestat.st_mode) && filestat.st_size != 0)
    {
      /*
       * File exists: just read it.
       */
      int		file;
      int		nread;

      if ((file = open (customfile, O_RDONLY)) == -1)
      {
	ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
	               "SecurID: could not open custom file `%s'",
		       customfile);
        return defautfmt;
      }
      else
      {
	/*
	 * Alloc buffer for file content.
	 */
	customfmt = (char *) ap_palloc (r->pool, filestat.st_size + 1);
	if (customfmt == NULL ||
	    ((nread = read (file, customfmt, filestat.st_size)) <= 0))
	{
	  /*
	   * Alloc or read failed.
	   */
	  ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
	                 "SecurID: could not read custom file `%s'",
			 customfile);
	  close (file);
	  return defautfmt;
	}
	else
	{
	  /*
	   * Alloc and read ok; add End Of String.
	   */
	  customfmt [nread] = '\0';
	  close (file);

	  /*
	   * And check "%." from file.
	   */
	  defautpc = securid_customcheck (r->pool, defautfmt);
	  custompc = securid_customcheck (r->pool, customfmt);
	  /*
	   * NULL means to many "%." in format; check this.
	   */
	  if (!defautpc)
	  {
	    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
			   "SecurID: NULL format for `%s' (??\?)", defautfmt);
	    return defautfmt;
	  }
	  if (!custompc)
	  {
	    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
			   "SecurID: bad %%. format for custom filename `%s'"
			   " (to many %%.)", customfile);
	    return defautfmt;
	  }
	  /*
	   * And check for diff.
	   */
	  if (strcmp (defautpc, custompc) != 0)
	  {
	    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
			   "SecurID: bad %%. format for custom filename `%s'"
			   " (`%s' instead of `%s')",
			   customfile, custompc, defautpc);
	    return defautfmt;
	  }
	  else
	  {
#           ifdef SECURID_DEBUG
	    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
			  "#%ld: valid format for custom filename `%s'"
			  " (`%s' == `%s')",
			  (long) getpid (), customfile, custompc, defautpc);
#           endif
	  }

	  /*
	   * Else, ok; use this custom file content.
	   */
	  return customfmt;
	}
      }
    }
    else
    {
      /*
       * File does not exists.
       */
      return defautfmt;
    }
  }
}

/*
 ************************************************************************
 * Expired authentication?
 ************************************************************************
 */
int securid_expired (server_rec *s, time_t first_time, time_t last_time)
{
  /*
   * Current server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (s->module_config, &securid_module);

  /*
   * Current time
   */
  time_t	this_time = time (NULL);

  /*
   * if using TTL & always_after,
   *   diff_time1 = current time - first time user has been authenticated;
   * if using TTL & if_not_used,
   *   diff_time1 = current time - last time user used this authentication;
   */
  time_t	diff_time1 = difftime (this_time,
                                       sconf->cachettltype == SECURID_TTL_ALWAYS
				       ? first_time : last_time);

  /*
   * if using MaxTTL,
   *   diff_time2 = current time - first time user has been authenticated;
   * else
   *   diff_time2 = 0;
   */
  time_t	diff_time2 = sconf->cachettlmax
                             ? difftime (this_time, first_time) : 0;

# ifdef SECURID_DEBUG
  ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		"#%ld: expired: delta1 = %ld & delta2 = %ld "
		"(first = %ld, last = %ld, current = %ld)",
		(long) getpid (), diff_time1, diff_time2,
		first_time, last_time, this_time);
# endif

  return (diff_time1 > sconf->cachettltime || diff_time2 > sconf->cachettlmax);
}

/*
 ************************************************************************
 * Server cleanup: called by module cleanup for each server.
 *
 * Note that if a virtual-server uses the same config as the main-server,
 * then its "sconf" was the same as the main and then all has already been
 * done.
 ************************************************************************
 */
void securid_server_cleanup (server_rec *s, int server_num,
                             server_rec *m /* main server */)
{
  /*
   * Current config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (s->module_config, &securid_module);
  /*
   * Main server config
   */
  securid_sconf	*mconf =
    (securid_sconf *) ap_get_module_config (m->module_config, &securid_module);

  /*
   * Lock file: do something if we are the main server or if the file of this
   * virtual server is not the same as the main.
   */
  if (server_num == 0 || strcmp (sconf->lockfile, mconf->lockfile) != 0)
  {
    if (unlink (sconf->lockfile) != 0)
    {
      ap_log_error (APLOG_MARK, APLOG_WARNING, s,
		    "SecurID: could not remove AuthSecurID_Cache-lock file "
		    "`%s'", sconf->lockfile);
    }
#   ifdef SECURID_DEBUG
    else
    {
      ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		    "#%ld: server_cleanup(%d): lock file removed",
		    (long) getpid (), server_num);
    }
#   endif
  }

  /*
   * Cache file(s): same as for lockfile, but we also have to check that
   * "noreset" is not used.
   */
  if (sconf->resetcache &&
      (server_num == 0 || strcmp (sconf->cachefile, mconf->cachefile) != 0))
  {
    if (unlink (sconf->cachefile1) != 0 ||
        (sconf->cachefile2 ? unlink (sconf->cachefile2) != 0 : 0))
    {
      ap_log_error (APLOG_MARK, APLOG_WARNING, s,
		    "SecurID: could not remove AuthSecurID_Cache file "
		    "`%s'", sconf->cachefile);
    }
#   ifdef SECURID_DEBUG
    else
    {
      ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		    "#%ld: server_cleanup(%d): cache file%s removed",
		    (long) getpid (), server_num, sconf->cachefile2 ? "s" : "");
    }
#   endif
  }

# ifdef SECURID_DEBUG
  ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		"#%ld: server_cleanup(%d): done",
		(long) getpid (), server_num);
# endif
}

/*
 ************************************************************************
 * Module cleanup: called on stop/restart.
 ************************************************************************
 */
void securid_module_cleanup (void *data)
{
  server_rec	*s = (server_rec *) data;	/* current server	*/
  server_rec	*m = s;				/* main server		*/
  int		i  = 0;				/* server #		*/

# ifdef SECURID_DEBUG
  ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		"#%ld: module_cleanup: starting", (long) getpid ());
# endif
  for (; s; s = s->next, i++)
  {
    securid_server_cleanup (s, i, m);
  }
# ifdef SECURID_DEBUG
  ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		"#%ld: module_cleanup: done", (long) getpid ());
# endif
}

/*
 * Create a directory to hold Unix domain sockets.
 */
const char *securid_cfg_make_dir(pool *p, char **path)
{
  struct stat finfo;
  
  /* If not an absolute path, append to server root directory */
  if (**path != '/') {
    *path = ap_server_root_relative (p, *path);
  }
  else {
    int i = strlen(*path) - 1;
      
    /* Strip trailing "/"s */
    while(i > 0 && (*path)[i] == '/') (*path)[i--] = '\0';
  }
    
  /* Does it exist? */
  if (stat(*path, &finfo) != 0) {
    /* No, but maybe we can create it */
    if (mkdir(*path, S_IRWXU) != 0)
      {
	return ap_psprintf(p, "can't be created: %s", strerror(errno));
      }
  }
  else {
    /* Yes, is it a directory? */
    if (!S_ISDIR(finfo.st_mode))
      {
	return "isn't a directory!";
      }
  }
    if (!geteuid()) {
      /* Apache was started by root. Change directory owner. */
      if (chown (*path, ap_user_id, ap_group_id) != 0) {
        return ap_psprintf(p, "can't change ownership of `%s' to uid(%lu) and "
                              "gid(%lu): %s", *path, (unsigned long)ap_user_id,
			   (unsigned long)ap_group_id, strerror(errno));
      }
    }
  return NULL;
}

int securid_proxy_start (void *param, child_info *info)
{
  char *proxy_argv[3], *proxy_envp[4], buf[256];
  server_rec *s = (server_rec *)param;
  /*
   * Current server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (s->module_config, &securid_module);

# ifdef SECURID_DEBUG
  ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		"#%ld: securid_proxy_start: starting", (long) getpid ());
# endif
  /*
   * Update server configuration
   */
  sconf->proxy_pid = getpid();
  snprintf (buf, sizeof(buf) - 1, "%s/%d", sconf->sockdir, sconf->proxy_pid);
  buf[sizeof(buf) - 1] = '\0';
  sconf->sockfn = buf;
  proxy_argv[0] = sconf->proxy;
  proxy_argv[1] = sconf->sockfn;
  proxy_argv[2] = NULL;
  proxy_envp[0] = "PATH=/bin";
  proxy_envp[1] = sconf->var_ace;
  proxy_envp[2] = sconf->ace_lang;
  proxy_envp[3] = NULL;

  /*
   * Redirect standard error to the error log
   */
  ap_error_log2stderr (s);

  /*
   * Set groups if we run as root
   */
    if (!geteuid()) {
        struct passwd *ent;

        /* Perform sanity check. */
        if ((ent = getpwuid(ap_user_id)) == NULL) {
          ap_log_error(APLOG_MARK, APLOG_ALERT, s,
                       "securid_proxy_start: couldn't determine user name "
                       "from uid %u, you probably need to modify the User "
                       "directive", (unsigned)ap_user_id);
                exit (EXIT_FAILURE);;
            }
                                                                                
        /* Set `main group' / GID */
        if (setgid(ap_group_id) == -1) {
            ap_log_error(APLOG_MARK, APLOG_ALERT, s, "securid_proxy_start: "
                         "unable to set group id to Group %u",
                         (unsigned)ap_group_id);
            exit (EXIT_FAILURE);;
        }
        /* Reset `groups' attributes. */

        if (initgroups(ap_user_name, ap_group_id) == -1) {
            ap_log_error(APLOG_MARK, APLOG_ALERT, s,
                        "securid_proxy_start: unable to set groups for User %s "
                        "and Group %u", ap_user_name, (unsigned)ap_group_id);
            exit (EXIT_FAILURE);;
        }
    }

  /*
   * Set UID if we run as root
   */
    if (!geteuid() && (
        setuid(ap_user_id) == -1)) {
        ap_log_error(APLOG_MARK, APLOG_ALERT, s,
                    "setuid: unable to change to uid: %ld", (long) ap_user_id);
        exit (EXIT_FAILURE);;
    }

  /*
   * Clean up before exec (close file descriptors...
   */
  ap_cleanup_for_exec();

  /*
   * Exec securid_proxy.
   */
  execve(sconf->proxy, proxy_argv, proxy_envp);

  /*
   * Exec failled! Log error.
   */
  ap_log_error (APLOG_MARK, APLOG_ERR, s,
		"SecurID: #%ld could not exec proxy %s - %s", (long) getpid (),
		sconf->proxy, strerror(errno));

  return errno;
}

/*
 ************************************************************************
 * Initialization of one server (see also module_init):
 *
 * Open auth cache and try to ACE init.
 *
 * Note that if a virtual-server uses the same config as the main-server,
 * then sconf == mconf (but s != m) and then we have nothing to do (except
 * for the main-server, iae #0).
 ************************************************************************
 */
static void securid_server_init (server_rec *s, pool *p, int server_num,
                                 server_rec *m /* main server */)
{
  /*
   * Current server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (s->module_config, &securid_module);
  /*
   * Main server config
   */
  securid_sconf	*mconf =
    (securid_sconf *) ap_get_module_config (m->module_config, &securid_module);

  const char *err;

  /*
   * To re-initialize the cache
   */
  int		need2create;
  XDBM_FILE	cachefd;
  int		lockfd, rc;

# ifdef SECURID_DEBUG
  ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		"#%ld: server_init(%d): starting", (long) getpid (),
		server_num);
# endif

  /*
   * Add "ServerRoot" to cache and lock filename.
   */
  sconf->cachefile = ap_server_root_relative (p, sconf->cachefile);
  sconf->lockfile  = ap_pstrcat (p, sconf->cachefile, SECURID_LOCKFEXT, NULL);

  /*
   * Deal with cache and lock files: if we are not server #0 (main server)
   * and if it is the same files as for the main server, then all has already
   * been done.
   */
  if (server_num == 0 || strcmp (sconf->cachefile, mconf->cachefile) != 0)
  {
    /*
     * First of all, create the cache lock file, and chown it if necessary...
     * We do not need to check the "noreset" in "AuthSecurID_Cache file noreset"
     * because lock file *must* exist.
     */
    errno = 0;
    do
    {
      lockfd = open (sconf->lockfile, O_WRONLY | O_CREAT, SECURID_OPEN_MODE);
    } while (lockfd < 0 && errno == EINTR);

    if (lockfd < 0)
    {
      ap_log_error (APLOG_MARK, APLOG_ERR, s,
		    "SecurID: could not create AuthSecurID_Cache-lock file "
		    "`%s'", sconf->lockfile);
      exit (EXIT_FAILURE);
    }

    errno = 0;
    do
    {
      rc = close (lockfd);
    } while (rc < 0 && errno == EINTR);

    if (geteuid () == 0)
    {
      if (chown (sconf->lockfile, ap_user_id, -1) != 0)
      {
	ap_log_error (APLOG_MARK, APLOG_ERR, s,
		      "SecurID: could not chown AuthSecurID_Cache-lock file "
		      "`%s'", sconf->lockfile);
      }
      /* Continue: this is not (yet) fatal... */
    }

    /*
     * If:
     * - there were no "noreset" ("AuthSecurID_Cache file noreset"),
     * - or there were "noreset" but cache does not exist,
     * then  init cache.
     *
     * We do not use securid_auth_open and securid_auth_close because this
     * is not a HTTP request and so (request_rec *r) is not available.
     * Additionally, it is here useless to lock file.
     *
     * Cache filename, "cachefile", has been first set by securid_create_sconf
     * (default value), and then possibly by the SecurIDAuthCache directive.
     *
     * To test if cache already exist, we have to know if we are using:
     *   - gdbm/all	=> "cachefile"
     *   - dbm/Linux	=> "cachefile".db
     *   - dbm/Solaris	=> "cachefile".pag & "cachefile".dir
     *
     * Also note that dbm/Linux #define DBM_SUFFIX. So:
     *   ifdef SECURID_USE_GDBM
     *     => "cachefile"
     *   else ifdef DBM_SUFFIX
     *     => "cachefile".db
     *   else
     *     => "cachefile".pag & "cachefile".dir
     */

    /*
     * Compute cache file(s) name
     */
#   ifdef SECURID_USE_GDBM /* { */
      sconf->cachefile1 = sconf->cachefile;
      sconf->cachefile2 = NULL;
#    else /* }{ */
#     ifdef DBM_SUFFIX /* { */
	sconf->cachefile1 = ap_pstrcat (p, sconf->cachefile, ".db", NULL);
	sconf->cachefile2 = NULL;
#     else /* }{ */
	sconf->cachefile1 = ap_pstrcat (p, sconf->cachefile, ".pag", NULL);
	sconf->cachefile2 = ap_pstrcat (p, sconf->cachefile, ".dir", NULL);
#     endif /* } */
#   endif /* } */

    /*
     * Then "compute" if we have to create cache
     */
    if (sconf->resetcache)
    {
      /*
       * No "noreset": we'll need to create a new cache.
       */
      need2create = 1;
    }
    else
    {
      /*
       * "noreset": check that cache file(s) exists.
       */
      struct stat	filestat;	/* buffer for file stats	*/

      if (stat (sconf->cachefile1, &filestat) != 0 ||
          !S_ISREG (filestat.st_mode) ||
          (sconf->cachefile2 &&
	   (stat (sconf->cachefile2, &filestat) != 0 ||
	    !S_ISREG (filestat.st_mode))))
      {
	need2create = 1;
	ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_NOTICE, s,
		      "SecurID: \"noreset\" ignored: need to create "
		      "AuthSecurID_Cache file `%s'", sconf->cachefile);
      }
      else
      {
	need2create = 0;
      }
    }

    /*
     * Reuse cache?
     */
    if (!need2create)
    {
      /*
       * Yes, reuse cache: check we can open it.
       */
      if ((cachefd = xdbm_open (sconf->cachefile XDBM_BSIZE, XDBM_WRITER,
				SECURID_OPEN_MODE XDBM_FFUNC)) != NULL)
      {
	/*
	 * Ok, that's all.
	 */
	xdbm_close (cachefd);
      }
      else
      {
	/*
	 * DB open failed.
	 */
	ap_log_error (APLOG_MARK, APLOG_WARNING, s,
		      "SecurID: could not reuse AuthSecurID_Cache file `%s'",
		      sconf->cachefile);
	/*
	 * Try to remove file(s) and reinitialize.
	 */
	if (unlink (sconf->cachefile1) == 0 &&
	    (sconf->cachefile2
	     ? unlink (sconf->cachefile2) == 0
	     : 1))
	{
	  need2create = 1;
	}
	else
	{
	  ap_log_error (APLOG_MARK, APLOG_ERR, s,
			"SecurID: could not remove AuthSecurID_Cache file `%s'",
			sconf->cachefile);
	  exit (EXIT_FAILURE);
	}
      }
    }

    /*
     * If we need to (re)create cache file or if reuse failed.
     */
    if (need2create)
    {
      /*
       * Create (and chown) new cache.
       */
      if ((cachefd = xdbm_open (sconf->cachefile XDBM_BSIZE, XDBM_NEWDB,
				SECURID_OPEN_MODE XDBM_FFUNC)) == NULL)
      {
	ap_log_error (APLOG_MARK, APLOG_ERR, s,
		      "SecurID: could not create AuthSecurID_Cache file `%s'",
		      sconf->cachefile);
	exit (EXIT_FAILURE);
      }
      xdbm_close (cachefd);

      /*
       * If root is starting Apache, then cache file must be chowned to "User".
       * -1 in chown means no gid change.
       */
      if (geteuid () == 0)
      {
	if (chown (sconf->cachefile1, ap_user_id, -1) != 0 ||
	    (sconf->cachefile2
	     ? chown (sconf->cachefile2, ap_user_id, -1) != 0
	     : 0))
	{
	  ap_log_error (APLOG_MARK, APLOG_ERR, s,
			"SecurID: could not chown AuthSecurID_Cache file `%s'",
			sconf->cachefile);
	}
      }
    }
  }
# ifdef SECURID_DEBUG
  else
  {
    ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		  "#%ld: server_init(%d): "
		  "inherit cache & lock work from main",
		  (long) getpid (), server_num);
  }
# endif

  /* If proxy executable path is relative, append to server root */
  if (sconf->proxy[0] != '/')
    {
      sconf->proxy = ap_server_root_relative (p, sconf->proxy);
    }

  /* Create directory to store Unix domain socket */
  err = securid_cfg_make_dir(p, &(sconf->sockdir));
  if (err)
    ap_log_error(APLOG_MARK, APLOG_ERR, s, "SecurID: %s", err);
  
  /* Start the SecurID proxy */
  sconf->proxy_pid = ap_spawn_child(p, securid_proxy_start, s,
				    kill_after_timeout, NULL, NULL, NULL);
  if (sconf->proxy_pid <= 0) {
    ap_log_error(APLOG_MARK, APLOG_ERR, s, "SecurID: can't start the SecurID proxy, ap_spawn_child() failed");
  }
  else
    {
      sconf->sockfn = ap_psprintf (p, "%s/%ld", sconf->sockdir, 
				   (long) sconf->proxy_pid);
# ifdef SECURID_DEBUG
      ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		    "SecurID: proxy %ld started for server %d, pid %ld",
		    (long) sconf->proxy_pid, server_num, (long) getpid ());
# endif
    }
  
  /*
   * It's all right...
   */
# ifdef SECURID_DEBUG
  ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		"#%ld: server_init(%d): done",
		(long) getpid (), server_num);
# endif
}

/*
 ************************************************************************
 * Initialization of the module:
 *
 * Quotting mod_ssl, "The initialization phase inside the Apache API is
 * totally bogus.
 *
 * 1. Under Unix the API does a 2-round initialization of modules while
 *    under Win32 it doesn't.
 *
 * 2. When Dynamic Shared Object (DSO) mechanism is used under Unix the
 *    module segment (code & data) gets unloaded and re-loaded between
 *    the first and the second round. This means no global data survives
 *    between first and the second init round.".
 *
 * Unix/static Unix/DSO          Win32     Action Required
 *             (-DSHARED_MODULE) (-DWIN32)
 * ----------- ----------------- --------- --------------------
 * -           load module       -         -
 * init        init              init      on Unix, nothing; on Win32, mod init
 * detach      detach            -         -
 * -           reload module     -         -
 * init        init              -         (on Unix,) mod init
 *
 * So this function is called 2 times on startup (1st & 2nd startup round)
 * and 1 time on restart (2nd startup round only); but we do not care
 * anymore...
 *
 * To init the module, we just call server_init() for *all* servers
 * (iae main and virtual(s)).
 ************************************************************************
 */
static void securid_module_init (server_rec *s, pool *p)
{
  int		i  = 0;				/* server #		*/
  server_rec	*m = s;				/* main server		*/

  /*
   * Let us cleanup on server-restart (arg#3) or before exec (arg#4).
   */
  ap_block_alarms();
  ap_register_cleanup (p, s, securid_module_cleanup, ap_null_cleanup);
  ap_unblock_alarms();

  /*
   * Init _all_ servers: main and virtual(s).
   */
  for (; s; s = s->next, i++)
  {
    securid_server_init (s, p, i, m);
  }
}

/*
 ************************************************************************
 * Childs init (rand() seeds)
 ************************************************************************
 */
static void securid_child_init (server_rec *s, pool *p)
{
  /*
   * All we have to do is to initialize rand() to make each handle AceHandle,
   * based on rand(), unique. ACE init will be done for the 1st auth.
   */
  srand ((unsigned int) getpid () * (unsigned int) time (NULL));

# ifdef SECURID_DEBUG
  ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		"#%ld: child_init: started", (long) getpid ());
# endif
  return;
}

/*
 ************************************************************************
 * Childs exit: call ACE close if needed and clean expired auth. in cache
 *
 * Note: on graceful restart (USR1), server_cleanup is called *before*
 * child_exit (and then cache & lock files have been removed), so we first
 * check that lock file does exist!
 ************************************************************************
 */
static void securid_child_exit (server_rec *s, pool *p)
{
  datum		key;			/* <id> of AceHandle(s)		*/
  datum		nextkey;		/* next <id> of AceHandle(s)	*/
  datum		val;			/* auth in cache		*/
  securid_webid	webid;			/* auth struct in cache		*/
  XDBM_FILE	cachefd;		/* cache file			*/
  int		lockfd;			/* lock file			*/
  int		del;			/* 1 if NULL key, 2 if expired	*/
  securid_sconf	*sconf =		/* server conf for cache name	*/
    (securid_sconf *) ap_get_module_config (s->module_config, &securid_module);

# ifdef SECURID_DEBUG
  ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		"#%ld: child_exit: starting", (long) getpid ());
# endif

  /*
   * Lock and open cache file. We cannot call securid_auth_open() because
   * we are not in a "request".
   */
  lockfd = db_lock (sconf->lockfile, XDBM_WRITER, 0 /* logerr */);
  if (lockfd < 0)
  {
#   ifdef SECURID_DEBUG
    ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		  "#%ld: child_exit: goodbye! (graceful restart?)",
		  (long) getpid ());
#   endif
    return;
  }
  if ((cachefd = xdbm_open (sconf->cachefile XDBM_BSIZE, XDBM_WRITER,
                            SECURID_OPEN_MODE XDBM_FFUNC)) == NULL)
  {
    ap_log_error (APLOG_MARK, APLOG_ERR, s,
                  "SecurID: could not open AuthSecurID_Cache file `%s'",
		  sconf->cachefile);
    db_unlock (lockfd);
    return;
  }

  /*
   * Get the first key and loop
   */
  key = xdbm_firstkey (cachefd);
  while (key.dptr)
  {
    /*
     * Get auth. for this key
     */
    val = xdbm_fetch (cachefd, key);
    del = 0;
    if (val.dptr != NULL)
    {
      /*
       * Get auth. value and free gdbm.
       */
      memcpy (&webid, val.dptr, sizeof (securid_webid));
#     ifdef SECURID_USE_GDBM
        free (val.dptr);
#     endif

      /*
       * And check if expired
       */
      if (securid_expired (s, webid.first_time, webid.last_time))
      {
	/*
	 * This key will be deleted (because it's expired).
	 */
        del = 1;
      }
    }
    else
    {
      /*
       * No value for this key: could this happen???
       */
      ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, s,
		    "SecurID: NULL auth for `%s' (??\?)", key.dptr);

      /*
       * Anyway, this key will be deleted (because it's NULL).
       */
      del = 2;
    }

    /*
     * If this auth. is to be deleted
     */
    if (del)
    {
      if (xdbm_delete (cachefd, key) != 0)
      {
	/*
	 * Delete KO, continue with "next" key (else we could loop forever).
	 */
	ap_log_error (APLOG_MARK, APLOG_ERR, s,
		      "SecurID: delete error for AuthSecurID_Cache file "
		      "(dbm err #%d for key `%s')",
		      xdbm_error (cachefd), key.dptr);
	nextkey = xdbm_nextkey (cachefd, key);
      }
      else
      {
	/*
	 * Delete ok, restart xdbm search (because of `hash table'...).
	 */
#       ifdef SECURID_DEBUG
	ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		      "#%ld: child_exit: %s `%s' deleted",
		      (long) getpid (),
		      del == 1 ? "expired" : "NULL", key.dptr);
#       endif
	nextkey = xdbm_firstkey (cachefd);
      }
    }
    else
    {
      nextkey = xdbm_nextkey (cachefd, key);
    }

    /*
     * Free gdbm key and get next.
     */
#   ifdef SECURID_USE_GDBM
      free (key.dptr);
#   endif
    memcpy (&key, &nextkey, sizeof (key));
  }

  /*
   * Unlock and close auth. cache file.
   */
  xdbm_close (cachefd);
  db_unlock (lockfd);

  /*
   * And that's all.
   */
# ifdef SECURID_DEBUG
  ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, s,
		"#%ld: child_exit: goodbye", (long) getpid ());
# endif
  return;
}

/*
 ************************************************************************
 * From mod_access.c::is_ip to check that a string is an IP address.
 ************************************************************************
 */
static int securid_isip (const char *s)
{
  while ((*s == '.') || ap_isdigit (*s))
  {
    s++;
  }
  return (*s == '\0');
}

/*
 ************************************************************************
 * Get "From-Agent", default is remote ip.
 ************************************************************************
 */
const char *securid_fromagent (request_rec *r)
{
  /*
   * Server config.
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (r->server->module_config,
					    &securid_module);
  /*
   * "From-Agent" value.
   */
  const char	*value = NULL;

  /*
   * Index for key(s)
   */
  int		i;

  /*
   * Loop on each key(s)
   */
  for (i = 0; i < sconf->fromagents.n; i++)
  {
    /*
     * Match "from"?
     */
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: fromagent: from_net[%d]=%lx, remote_addr=%x, "
		   "from_msk[%d]=%lx",
		   (long) getpid (), i, sconf->fromagents.k[i].from_net,
		   r->connection->remote_addr.sin_addr.s_addr,
		   i, sconf->fromagents.k[i].from_msk);
#   endif
    if (sconf->fromagents.k[i].from_net != INADDR_NONE &&
        (r->connection->remote_addr.sin_addr.s_addr &
	 sconf->fromagents.k[i].from_msk) == sconf->fromagents.k[i].from_net)
    {
      /*
       * Yes it does; dispatch type.
       */
#     ifdef SECURID_DEBUG
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		     "#%ld: fromagent: does match!", (long) getpid ());
#     endif
      if (sconf->fromagents.k[i].type == SECURID_FROMAGENT_NULL)
      {
	/*
	 * Special case: no "from-agent" check (but NULL is not welcome...).
	 */
	return "NULL";
      }
      else
      {
	if (sconf->fromagents.k[i].type == SECURID_FROMAGENT_KEY)
	{
	  /*
	   * Get this header
	   */
	  value = ap_table_get (r->headers_in, sconf->fromagents.k[i].key_val);
	}
	if (!value)
	{
	  /*
	   * If no header value, or if we want remote-ip, get it.
	   */
	  value = r->connection->remote_ip;
	}
	/*
	 * Need to apply a mask?
	 */
	if (sconf->fromagents.k[i].key_msk)
	{
	  unsigned long		ul_net;
	  struct in_addr	in_net;

	  /*
	   * Yes, we need to apply a mask to "value", check this is possible.
	   */
	  if (!securid_isip (value) ||
	      (ul_net = ap_inet_addr (value)) == INADDR_NONE)
	  {
	    /*
	     * This header is not a valid ip addr; use it but log a warning.
	     */
	    ap_log_rerror (APLOG_MARK, APLOG_WARNING, r,
			   "SecurID: FromAgent value is not a valid IP "
			   "address: `%s: %s'",
			   sconf->fromagents.k[i].key_val, value);
	    return value;
	  }
	  /*
	   * Return the masked value, converted to string.
	   */
	  in_net = inet_makeaddr (htonl (ul_net &
	                                 sconf->fromagents.k[i].key_msk), 0);
	  return inet_ntoa (in_net);
	}
	else
	{
	  /*
	   * No mask to apply
	   */
	  return value;
	}
      }
    }
#   ifdef SECURID_DEBUG
    else
    {
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		     "#%ld: fromagent: does NOT match!", (long) getpid ());
    }
#   endif
  }

  /*
   * Default is to get remote ip.
   */
  return r->connection->remote_ip;
}

/*
 ************************************************************************
 * Get "User-Agent" from HTTP header
 ************************************************************************
 */
const char *securid_useragent (request_rec *r)
{
  /*
   * Header value.
   */
  const char	*value = NULL;

  /*
   * Get header value, or "unknown" if none.
   */
  value = ap_table_get (r->headers_in, "User-Agent");
  if (!value)
  {
    value = "unknown";
  }

  /*
   * Return header. It's too late to try to truncate it because "alloc" has
   * already been made by ap_table_get()...
   */
  return value;
}

/*
 ************************************************************************
 * Extract AceHandle cookie (handle).
 *
 * In the HTTP header of the request, a cookie is a line like:
 *   Cookie: NAME1=OPAQUE_STRING1; NAME2=OPAQUE_STRING2 ...
 *
 * For SecurID, we have:
 *   Cookie: AceHandle=...; webid2=...
 *
 * As handle is a "%ld" string:
 *   - 0 < its size < 20
 *   - strtol() must not detect some errors
 ************************************************************************
 */
const char *securid_get_handle (request_rec *r, char *name)
{
  /*
   * Cookie name (AceHandle or whatever arg#2 says) + "=".
   */
  const char	*cookie = NULL;		/* cookie header line	*/
  char		*value  = NULL;		/* our cookie value	*/
  char		*name_;			/* cookie name + "="	*/
  char		*begin;
  char		*end;
  char		*err;
  int		len;
  
  /*
   * If custom cookie name is NULL, use default name. Compute (custom)
   * cookie name + "=".
   */
  name_  = ap_pstrcat (r->pool, name ? name : SECURID_HANDLE_NAME, "=", NULL);

  cookie = ap_table_get (r->headers_in, "Cookie");
  if (cookie)
  {
    begin = strstr (cookie, name_);
    if (begin)
    {

      begin += strlen (name_);
      value  = ap_pstrdup (r->pool, begin);
      end    = strchr (value, ';');
      if (end)
      {
        *end = '\0';
      }
      /*
       * Some checks on the value
       */
      len = strlen (value);
      if (len == 0 || len > 20)
      {
	ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		       "SecurID: ACE syntax error (bad AceHandle `%s')",
		       value);
        return NULL;
      }
      (void) strtol (value, &err, 10);
      if (*err != '\0')
      {
	ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		       "SecurID: ACE syntax error (bad AceHandle `%s')",
		       value);
        return NULL;
      }
    }
  }
  return (value);
}

/*
 ************************************************************************
 * Create authentication cookie (webid2) from securid_webid
 *
 * We build the cookie in the following way:
 *   username<X>first_time<X>shell
 * the character <X> will be used as separator during cookie extraction (see
 * securid_get_webid).
 *
 * As a cookie must be "isprint()"able and <X> may not appear in "username",
 * nor in "first_time" (of course...) nor in "shell", '!' has been choosed.
 ************************************************************************
 */
#define SECURID_COOKIE_SEP	'!'

char *securid_set_webid (request_rec *r, securid_webid *webid)
{
  char	*cookie;

  cookie = ap_psprintf (r->pool, "%s%c%ld%c%s",
			webid->username, SECURID_COOKIE_SEP,
			webid->first_time, SECURID_COOKIE_SEP,
			webid->shell);
  return cookie;
}

/*
 ************************************************************************
 * Extract authentication cookie (webid2)
 *
 * Note: see previous note (securid_set_webid)
 ************************************************************************
 */
int securid_get_webid (request_rec *r, char *name, securid_webid *webid)
{
  const char	*cookie = NULL;		/* cookie header line	*/
  char		*value  = NULL;		/* our cookie value	*/
  char		*name_;			/* cookie name + "="	*/
  char		*begin;
  char		*end;

  /*
   * If custom cookie name is NULL, use default name. Compute (custom)
   * cookie name + "=".
   */
  name_  = ap_pstrcat (r->pool, name ? name : SECURID_COOKIE_NAME, "=", NULL);

  cookie = ap_table_get (r->headers_in, "Cookie");
  if (cookie)
  {
    begin = strstr (cookie, name_);
    if (begin)
    {
      begin += strlen (name_);
      value  = ap_pstrdup (r->pool, begin);
      end    = strchr (value, ';');
      if (end)
      {
        *end = '\0';
      }
    }
  }

  /*
   * Recovered value is like "username:first_time:shell", where ':' are
   * SECURID_COOKIE_SEP characters.
   */
  if (value)
  {
    begin  = value;
    end    = strchr (begin, SECURID_COOKIE_SEP);
    if (end)
    {
      *end  = '\0';
      /*
       * Get username from Cookie: header.
       */
      strncpy (webid->username, begin, sizeof (webid->username) - 1);
      webid->username [sizeof (webid->username) - 1] = '\0';
      begin = end + 1;
      end   = strchr (begin, SECURID_COOKIE_SEP);
      if (end)
      {
	*end  = '\0';
	/*
	 * Get first time from Cookie: header.
	 */
	webid->first_time = (time_t) atol (begin);
	begin = end + 1;
	/*
	 * Get shell from Cookie: header.
	 */
	strncpy (webid->shell, begin, sizeof (webid->shell) - 1);
	webid->shell [sizeof (webid->shell) - 1] = '\0';
	/*
	 * Get "user-agent".
	 */
	strncpy (webid->user_agent, securid_useragent (r),
	         sizeof (webid->user_agent) - 1);
	webid->user_agent [sizeof (webid->user_agent) - 1] = '\0';
	/*
	 * Get "remote ip".
	 */
	strncpy (webid->from_agent, securid_fromagent (r),
	         sizeof (webid->from_agent) - 1);
	webid->from_agent [sizeof (webid->from_agent) - 1] = '\0';
	/*
	 * It's ok.
	 */
	return 1;
      }
    }
  }
  /*
   * It's KO.
   */
  return 0;
}

/*
 ************************************************************************
 * Get restricted <Location> of authentication. This may only be called
 * when we are handling a restricted location: for ex. securid_check_auth (),
 * but *not* securid_handler_auth () !
 *
 * Apache stores a var "d", that is the "directory" config (from <Location> or
 * from <Directory>) for which the auth. is needed.
 *
 * 1/ Example: a std web server with the config:
 *   DocumentRoot	/home/www/htdocs
 *   Alias /img/	/home/www/images
 *   ScriptAlias /bin/	/home/www/cgibin
 *   <Location	/priv/> ...
 *   <Directory	/home/www/images/priv/> ...
 *   <Location	/bin/priv/> ...
 *
 *   Apache will set "d" to:
 *   _URI_			"d"
 *     /priv/...		/home/www/htdocs/priv/
 *     /img/priv/...		/home/www/images/priv/
 *     /bin/priv/...		/home/www/cgibin/priv/
 *
 * 2/ Example: a rev proxy with the config:
 *   ProxyPass /	http[s]://_Www_[:_Port_]/
 *   <Location  /priv/> ...
 *   <Directory	proxy:http[s]://_Www_[:_Port_]/img/priv/> ...
 *   <Location	/bin/priv/> ...
 *
 *   Apache will set "d" to:
 *   _URI_			"d"
 *     /priv/...		proxy:http[s]://_Www_[:_Port_]/priv/
 *     /img/priv/...		proxy:http[s]://_Www_[:_Port_]/img/priv/
 *     /bin/priv/...		proxy:http[s]://_Www_[:_Port_]/bin/priv/
 *   or only:
 *	proxy:*
 *
 * 3/ Example: a std proxy with the config:
 *   ProxyRequests on
 *   <Location  /priv/> ...
 *   <Directory	proxy:http://_Www1_[:_Port_]/priv/> ...
 *   <Directory	proxy:http://_Www2_[:_Port_]/img/priv/> ...
 *   <Directory	proxy:ftp://_Ftp3_[:_Port_]/bin/priv/> ...
 *
 *   Apache will set "d" to the "<Directory>" value, or maybe only proxy:*.
 *
 * 4/ So?
 *   So when we are not a std web server, this is not too difficult to compute
 *   the "private path", we just have to skip "proxy:proto://server[:port]".
 *
 *   But when we are a std web, this is more difficult: we have to check for
 *   "Alias" & "ScriptAlias" directives if something matches the beginning
 *   of "d", or else to try with "DocumentRoot".
 *
 *   Also check that the PathCookie directive is not used to set the "private
 *   path"...
 ************************************************************************
 */
const char *securid_priv_path (request_rec *r)
{
  /*
   * Server and directory configs
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (r->server->module_config,
					    &securid_module);
  if (sconf->pathcookie)
  {
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: priv_path: PathCookie => \"%s\"",
		   (long) getpid (), sconf->pathcookie);
#   endif

    return sconf->pathcookie;
  }
  else
  {
    /*
     * Core & server configs, to get current restricted "directory" & docroot
     */
    core_dir_config	*dcore =
      (core_dir_config *) ap_get_module_config (r->per_dir_config,
						&core_module);
    core_server_config	*score =
      (core_server_config *) ap_get_module_config (r->server->module_config,
						   &core_module);
    /*
     * The restricted "directory" (in fact, "location", not "directory")
     */
    char			*rdir;

    /*
     * Safety check(s)...
     */
    if (!dcore || !dcore->d)
    {
#     ifdef SECURID_DEBUG
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		     "#%ld: priv_path: <null> core_dir_config (->d)? !!!",
		     (long) getpid ());
#     endif
      return "/";
    }

    /*
     * Are we a proxy?
     */
    if (r->proxyreq)
    {
      /*
       * Std or rev proxy; start with "d"
       */
      rdir = dcore->d;
      /*
       * Skip 'proxy:'
       */
      if (strncmp (rdir, "proxy:", 6) == 0)
      {
	rdir += 6;
      }
      /*
       * Skip 'scheme:'
       */
      for ( ; *rdir != ':' && *rdir != '\0'; rdir++)
	;
      /*
       * skip '://'.
       */
      if (*rdir != '\0')
      {
	rdir += 3;
      }
      /*
       * Skip host part
       */
      for ( ; *rdir != '/' && *rdir != '\0'; rdir++)
	;
    }
    else
    {
      /*
       * Std web server; try aliases, from mod_alias, if available.
       */
      module		*securid_alias_module =
			    ap_find_linked_module ("mod_alias.c");
      int			len;

      rdir = NULL;
      if (securid_alias_module)
      {
	/*
	 * mod_alias is available; just redefine some mod_alias' types/vars.
	 */
	typedef struct
	{
	  char		*real;
	  char		*fake;
	  char		*handler;
	  regex_t		*regexp;
	  int		redir_status;
	}			securid_mod_alias_entry;
	typedef struct
	{
	  array_header	*aliases;
	  array_header	*redirects;
	}			securid_mod_alias_sconf;
	securid_mod_alias_sconf
			  *alias_sconf = (securid_mod_alias_sconf *)
					   ap_get_module_config
					     (r->server->module_config,
					      securid_alias_module);
	securid_mod_alias_entry
			  *alias_entries = (securid_mod_alias_entry *)
					     alias_sconf->aliases->elts;
	int		i;

	for (i = 0; i < alias_sconf->aliases->nelts; ++i)
	{
	  securid_mod_alias_entry
			  *a = &alias_entries [i];
	  /*
	   * a->real == _DIR_ / a->fake == _URI_
	   */
#         ifdef SECURID_DEBUG
	  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
			 "#%ld: priv_path: alias \"%s\" -> \"%s\"",
			 (long) getpid (), a->real, a->fake);
#         endif
	  /*
	   * Does a->real match the beginning of "d"?
	   */
	  len = strlen (a->real);
	  if (strncmp (dcore->d, a->real, len) == 0)
	  {
	    /*
	     * Matched! Change a->real with a->fake
	     */
	    rdir = ap_psprintf (r->pool, "%s%s", a->fake, dcore->d + len);
	    break;
	  }
	}
      }
      /*
       * If no alias has matched, try DocumentRoot
       */
      if (rdir == NULL)
      {
#       ifdef SECURID_DEBUG
	ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		       "#%ld: priv_path: docroot -> \"%s\"",
		       (long) getpid (), score->ap_document_root);
#       endif
	len = strlen (score->ap_document_root);
	if (strncmp (dcore->d, score->ap_document_root, len) == 0)
	{
	  /*
	   * Matched! Skip docroot +- 1, if ending with "/" or not
	   */
	  if (score->ap_document_root [len - 1] == '/')
	  {
	    rdir = dcore->d + len - 1;
	  }
	  else
	  {
	    rdir = dcore->d + len;
	  }
	}
	else
	{
	  rdir = "/";
	}
      }
    }

    /*
     * If "null" rdir
     */
    if (*rdir == '\0')
    {
      rdir = "/";
    }

#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: priv_path: \"%s\" + \"%s\" => \"%s\"",
		   (long) getpid (), dcore->d,
		   r->proxyreq ? "PROXY" : "WEB", rdir);
#   endif

    return rdir;
  }
}

/*
 ************************************************************************
 * HTML escape, from Apache, util.c:ap_escape_html():
 * escape '<', '>' and '"'.
 ************************************************************************
 */
char *securid_hescape (request_rec *r, const char *s)
{
  int	i, j;
  char	*x;

  /* first, count the number of extra characters */
  for (i = 0, j = 0; s[i] != '\0'; i++)
  {
    if (s[i] == '<' || s[i] == '>')
    {
      /* '<' or '>' (1 char) => '&lt;' or '&gt' (4 chars) */
      j += 3;
    }
    else if (s[i] == '"')
    {
      /* '"' (1 char) => '&quot;' (6 chars) */
      j += 5;
    }
  }
  if (j == 0)
  {
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: hescape: \"%s\" => NO CHANGE",
		   (long) getpid (), s);
#   endif
    return ap_pstrndup (r->pool, s, i);
  }
  x = ap_palloc (r->pool, i + j + 1);
  for (i = 0, j = 0; s[i] != '\0'; i++, j++)
  {
    if (s[i] == '<')
    {
      memcpy (&x[j], "&lt;", 4);
      j += 3;
    }
    else if (s[i] == '>')
    {
      memcpy (&x[j], "&gt;", 4);
      j += 3;
    }
    else if (s[i] == '"')
    {
      memcpy (&x[j], "&quot;", 6);
      j += 5;
    }
    else
    {
      x[j] = s[i];
    }
  }
  x[j] = '\0';
# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: hescape: \"%s\" => \"%s\"",
		 (long) getpid (), s, x);
# endif
  return x;
}

/*
 ************************************************************************
 * URLs caching policy
 ************************************************************************
 */
#define	SECURID_NOCACHE_OFF	0
#define	SECURID_NOCACHE_ON	1
void securid_nocache (request_rec *r, int nocache)
{
  /*
   * 1/ Default controls
   *
   * It is really "r->no_cache = 1", checked in
   * http_protocol.c:ap_send_http_header (), that will add header:
   *   "Expires: _request_date_",
   *
   * But, "Expires: _request_date_" will only tell the client not to reuse this
   * object, it will not prevent from proxy caching (object may be stored,
   * even if it will never be used).
   * So a better idea is to set "Expires: 0":
   *   - as "Expires" header will already exist, ap_send_http_header () will
   *     not add a new one,
   *   - As "0" is not a valid date, proxies will not cache this object (see
   *     proxy_cache.c:ap_proxy_cache_update()).
   *
   * Note1: on "HTTP error" (including redirect, used by securid_handler_auth to
   *        print authentication form), only some of headers_out are used;
   *        Expires is one of those sent (see
   *        http_protocol.c:ap_send_error_response()).
   * Note2: you need a patch for mod_proxy to take care of "added" headers.
   */
  r->no_cache = 1;
  ap_table_set (r->headers_out, "Expires", "0");

  /*
   * 2/ Additional controls
   *
   * It's not a bad idea to add "Pragma: no-cache" (for HTTP/1.0) or
   * "Cache-Control: no-store" (for HTTP/1.1).
   *
   * So, if "nocache" is set to _NOCACHE_ON, we add the right "no-cache" header.
   *
   * Note: as IE is not always happy with "Pragma: no-cache" or
   *       "Cache-Control: no-cache" (...), default is _NOCACHE_OFF.
   */
  if (nocache != SECURID_NOCACHE_OFF)
  {
    if (r->proto_num >= HTTP_VERSION (1,1))
    {
      ap_table_add (r->headers_out, "Cache-Control", "no-cache");
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		     "#%ld: nocache: \"Cache-Control: no-cache\"",
		     (long) getpid ());
    }
    else
    {
      ap_table_add (r->headers_out, "Pragma", "no-cache");
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		     "#%ld: nocache: \"Pragma: no-cache\"",
		     (long) getpid ());
    }
  }
# ifdef SECURID_DEBUG
  else /* nocache == SECURID_NOCACHE_OFF */
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: nocache: OFF", (long) getpid ());
  }
# endif
}

/*
 ************************************************************************
 * Check authentication validity
 ************************************************************************
 */
static int securid_check_auth (request_rec *r)
{
  /*
   * Server and directory configs
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (r->server->module_config,
					    &securid_module);
  securid_dconf	*dconf =
    (securid_dconf *) ap_get_module_config (r->per_dir_config,
					    &securid_module);

  /*
   * SecurID handle
   */
  const char	*handle;

  /*
   * AuthType value; if we are here, it must have been set (else, it meens
   * mod_securid is the only mod_auth* compiled and there is no AuthType
   * directive in httpd.conf).
   */
  const char	*auth_type = ap_auth_type (r);

  /*
   * strlen (r->uri), skeeping '?.........' if needed
   */
  int		strlen_r_uri;

  /*
   * First first of all, check if AuthType is not NULL *and* == "SecurID"
   */
  if (auth_type == NULL)
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		   "SecurID: no AuthType! Please use one...");
    return DECLINED;			/* should be internal error?	*/
  }
  if (strcasecmp (auth_type, "SecurID"))
  {
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: check_auth: not for me (for %s)",
		   (long) getpid (), ap_auth_type (r));
#   endif
    return DECLINED;
  }

  /*
   * First of all, mod_securid URIs ("URI/securid/auth" & "URI/securid/check")
   * cannot need authentication (because it would be a recursive loop...).
   * This check is usefull when:
   * - mod_securid is running on a proxy (standard or reverse) which restrict
   *   access to proxy:* (and then mod_securid URIs are also restricted);
   *   see also securid_translate(),
   * - playing with some "strange" config...
   */
  if (strchr (r->uri, '?'))
  {
    strlen_r_uri = strchr (r->uri, '?') - r->uri;
  }
  else
  {
    strlen_r_uri = strlen (r->uri);
  }
  if (*r->uri != '\0' &&
      ((strlen_r_uri >= SECURID_URL_AUTH_LEN && 
	strncmp (r->uri + (strlen_r_uri - SECURID_URL_AUTH_LEN),
	         SECURID_URL_AUTH, SECURID_URL_AUTH_LEN) == 0)
       ||
       (strlen_r_uri >= SECURID_URL_CHECK_LEN &&
	strncmp (r->uri + (strlen_r_uri - SECURID_URL_CHECK_LEN),
	         SECURID_URL_CHECK, SECURID_URL_AUTH_LEN) == 0)
      ))
  {
    /*
     * We also need to "unset" all require directives, so check_access will
     * grant access.
     */
    core_dir_config	*dcore =
      (core_dir_config *) ap_get_module_config (r->per_dir_config,
						&core_module);
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: check_auth: %s: no recursive auth.!!!",
		   (long) getpid (), r->uri);
#   endif
    dcore->ap_requires = NULL;
    return OK;
  }

# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: check_auth: %s %s",
		 (long) getpid (), r->method, r->uri);
# endif

  /*
   * We first look for the SecurID handle
   */
  handle = securid_get_handle (r, sconf->hdlcookie);
  if (handle)
  {
    /*
     * For this handle, we extract the SecurID authentication cookie the
     * browser provides (_new) that we will compare to the one stored in
     * cache (_old).
     */
    securid_webid	cookie_old;
    securid_webid	cookie_new;

    /*
     * First, check if this handle is already in cache: handle must exists
     * *and* username != "".
     */
    if (securid_auth_get (r, sconf->cachefile, sconf->lockfile, handle,
                          &cookie_old) == 0 &&
        cookie_old.username [0])
    {
      /*
       * This handle is already in cache. Get browser cookie.
       */
      if (securid_get_webid (r, sconf->widcookie, &cookie_new))
      {
        /*
	 * Check if _old == _new
	 */
#       ifdef SECURID_DEBUG
	ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		       "#%ld: check_auth: "
		       "cookie_old.user=`%s', .time=%ld, .shell=`%s', "
				 ".from_agent=`%s', .user_agent=`%s', "
		       "cookie_new.user=`%s', .time=%ld, .shell=`%s', "
				 ".from_agent=`%s', .user_agent=`%s'",
		       (long) getpid (),
		       cookie_old.username, cookie_old.first_time,
		       cookie_old.shell,
		       cookie_old.from_agent, cookie_old.user_agent,
		       cookie_new.username, cookie_new.first_time,
		       cookie_new.shell,
		       cookie_new.from_agent, cookie_new.user_agent);
#       endif

	/*
	 * Check that cookies are the same, else authentication is KO.
	 */
	     if (strcmp (cookie_old.username, cookie_new.username))
	{
	  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
			 "SecurID: invalid webid cookie for user `%s' "
			 "(username; was `%s', now `%s')",
			 cookie_old.username,
			 cookie_old.username, cookie_new.username);
	}
	else if (cookie_old.first_time != cookie_new.first_time)
	{
	  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
			 "SecurID: invalid webid cookie for user `%s' "
			 "(first time; was `%ld', now `%ld')",
			 cookie_old.username,
			 cookie_old.first_time, cookie_new.first_time);
	}
	else if (strcmp (cookie_old.shell, cookie_new.shell))
	{
	  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
			 "SecurID: invalid webid cookie for user `%s' "
			 "(shell; was `%s', now `%s')",
			 cookie_old.username,
			 cookie_old.shell, cookie_new.shell);
	}
	else if (strcmp (cookie_old.from_agent, cookie_new.from_agent))
	{
	  /*
	   * Remote ip has changed!, authentication is then KO.
	   */
	  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
			 "SecurID: invalid From-Agent for user `%s' "
			 "(was `%s', now `%s')",
			 cookie_old.username,
			 cookie_old.from_agent, cookie_new.from_agent);
	}
	else if (sconf->useragent &&
	         strncmp (cookie_old.user_agent, cookie_new.user_agent,
		          sconf->useragent))
	{
	  /*
	   * User-Agent has changed!, authentication is then KO.
	   */
	  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
			 "SecurID: invalid User-Agent for user `%s' "
			 "(was `%s', now `%s')",
			 cookie_old.username,
			 cookie_old.user_agent, cookie_new.user_agent);
	}
	else
	{
	  /*
	   * Cookies, User-Agent & Remote IP are identical, so auth. is OK.
	   * We just have now to check if it is not expired...
	   */
#         ifdef SECURID_DEBUG
	  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
			 "#%ld: check_auth: valid cookie for "
			 "handle=`%s'", (long) getpid (), handle);
#         endif
	  /*
	   * Expired?
	   */
	  if (securid_expired (r->server, cookie_old.first_time,
	                       cookie_old.last_time))
	  {
	    char	*priv_path = cookie_old.priv_path;

	    /*
	     * It is expired. Delete this handle from cache and redirect
	     * to authentication URI (to execute auth. handler).
	     */
#           ifdef SECURID_DEBUG
	    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
			   "#%ld: check_auth: expired handle=`%s'",
			   (long) getpid (), handle);
#           endif

	    /*
	     * Remove this handle from the cache
	     */
	    (void) securid_auth_del (r, sconf->cachefile, sconf->lockfile,
	                             handle);

	    /*
	     * Redirect to the Authentication URL (to execute authentication
	     * handler (form).
	     * We do not need to check if we are authoritative or not because
	     * where cookie is expired we want to **force** user to
	     * re-authenticate.
	     *
	     * Note: with communicator-4.7/Linux, if the auth. expires while
	     * you are playing with HREF, all is fine. But it the auth. expires
	     * while you are playing with "Forward" button, the browser is not
	     * happy (Bus error...).
	     */
	    ap_table_setn (r->headers_out, "Location",
			   ap_psprintf (r->pool, "%s?%s&%s",
			                SECURID_URL_FORM_AUTH (r, priv_path),
			                SECURID_REFERER (r), priv_path));
	    return REDIRECT;
	  }
	  else
	  {
	    /*
	     * Cookie is not expired
	     */
#           ifdef SECURID_DEBUG
	    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
			   "#%ld: check_auth: non-expired handle=`%s'",
			   (long) getpid (), handle);
#           endif

	    /*
	     * Update last time if TTL type is not "always_after".
	     */
	    if (sconf->cachettltype != SECURID_TTL_ALWAYS)
	    {
	      cookie_old.last_time = time (NULL);
	      (void) securid_auth_put (r, sconf->cachefile, sconf->lockfile,
	                               handle, &cookie_old);
	    }

	    /*
	     * We have to set "REMOTE_USER", authentication name and the
	     * possible group(s) to which he belongs to.
	     *
	     * We duplicate "cookie_old.username" (local variable) to enable
	     * this information in the reste of Apache code.
	     *
	     * The user's groups, set by ACE in "shell" field (!!), are stored
	     * via r->notes table, in the "sd_groups" variable.
	     */
	    r->connection->user = ap_pstrdup (r->pool, cookie_old.username);
	    if (cookie_old.shell)
	    {
	      /*
	       * !!: table_set(not n) because cookie_old.shell is local.
	       */
	      ap_table_set (r->notes, "sd_groups", cookie_old.shell);
	    }
	    else
	    {
	      ap_table_unset (r->notes, "sd_groups");
	    }

	    /*
	     * We set ap_auth_type (and don't forget to use in config a
	     * 'AuthType "SecurID"' directive, else we get a "INTERNAL_ERROR"
	     * because a not-needed authentication would be returned...).
	     */
	    r->connection->ap_auth_type = "SecurID";

	    /*
	     * Last, a "private" URL protected with SecurID should not be
	     * cached.
	     */
	    securid_nocache (r, sconf->nocache);
	    return OK;
	  }
	}
      }
      else
      {
        /*
	 * Browser does not provide a authentication cookie; KO...
	 */
#       ifdef SECURID_DEBUG
	ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		       "#%ld: check_auth: NO browser-cookie for "
		       "handle=`%s'!!!", (long) getpid (), handle);
#       endif
      }
    }
    else
    {
      /*
       * This authentication is not in the cache file (or auth. is in the cache
       * file but username == ""), KO...
       */
#     ifdef SECURID_DEBUG
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		     "#%ld: check_auth: "
		     "NO cache-cookie for handle=`%s'",
		     (long) getpid (), handle);
#     endif
      /*
       * Delete any old "incomplete" auth.
       */
      if (securid_auth_get (r, sconf->cachefile, sconf->lockfile, handle,
                            &cookie_old) == 0)
      {
	(void) securid_auth_del (r, sconf->cachefile, sconf->lockfile, handle);
      }
    }
  }
# ifdef SECURID_DEBUG
  else
  {
    /*
     * Browser does not provide the SecurID handle, KO...
     */
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: check_auth: NO handle", (long) getpid ());
  }
# endif

  /*
   * If we are here, user is not allowed to access URL.
   * Then, if we are authoritative, we say no (AUTH_REQUIRED), else we decline
   * (DECLINED).
   * But for SecurID, we cannot say AUTH_REQUIRED; instead, we redirect user
   * to the authentication URL, with ?argument sets to the URL he asked for.
   */
  if (dconf->authoritative)
  {
    const char	*priv_path = securid_priv_path (r);

    ap_table_setn (r->headers_out, "Location",
		   ap_psprintf (r->pool, "%s?%s&%s",
		                SECURID_URL_FORM_AUTH (r, priv_path),
		                SECURID_REFERER (r), priv_path));
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: check_auth: we DO are autho! => redirect=%s",
		   (long) getpid (), ap_table_get (r->headers_out, "Location"));
#   endif
    return REDIRECT;
  }
  else
  {
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: check_auth: we are NOT autho! => declined",
		   (long) getpid ());
#   endif
    return DECLINED;
  }
}

/*
 ************************************************************************
 * Check access permission (from mod_auth.c:check_user_access).
 * I.e. check if this username is allowed to access to the URL.
 ************************************************************************
 */
static int securid_check_access (request_rec *r)
{
  /*
   * Directory config
   */
  securid_dconf		*dconf =
    (securid_dconf *) ap_get_module_config (r->per_dir_config, &securid_module);
  /*
   * "require" table
   */
  const array_header	*reqs_arr = ap_requires (r);
  require_line		*reqs     = reqs_arr
				      ? (require_line *) reqs_arr->elts
				      : NULL;
  /*
   * For the "require" loop
   */
  register int		x;
  /*
   * Boolean to know if asked method is "restricted"
   */
  int			method_restricted = 0;
  /*
   * Groups from "sd_group" in "notes" table (format: "grp1,grp2,...")
   * and the same but with a table (1:"grp1","in"; 2:"grp2","in"; ...).
   */
  const char		*groups_list;
  table			*groups_table;
  /*
   * Misc...
   */
  const char		*t;
  char			*w;

  /*
   * AuthType value; see comments in securid_check_auth();
   */
  const char		*auth_type = ap_auth_type (r);

  /*
   * The "private path", that needs our authentication
   */
  const char		*priv_path;

  /*
   * First first of all, check if AuthType is not NULL *and* == "SecurID"
   */
  if (auth_type == NULL)
  {
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "SecurID: no AuthType! Please use one...");
#   endif
    return DECLINED;			/* should be internal error?	*/
  }
  if (strcasecmp (auth_type, "SecurID"))
  {
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: check_access: not for me (for %s)",
		   (long) getpid (), ap_auth_type (r));
#   endif
    return DECLINED;
  }

# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: check_access: %s %s, user=%s",
		 (long) getpid (), r->method, r->uri, r->connection->user);
# endif

  /*
   * If no "require", it is OK...
   */
  if (!reqs_arr)
  {
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: check_access: no require => OK", (long) getpid ());
#   endif
    return OK;
  }

  /*
   * Recover user's group(s) list and put it into table.
   */
  groups_list = ap_table_get (r->notes, "sd_groups");
  if (groups_list && groups_list [0])
  {
    const char	*groups_temp = ap_pstrdup (r->pool, groups_list);
    const char	*group_temp;

    groups_table = ap_make_table (r->pool, 15);
    group_temp   = ap_getword (r->pool, &groups_temp, ',');
    while (group_temp [0])
    {
      ap_table_setn (groups_table, ap_pstrdup (r->pool, group_temp), "in");
      group_temp = ap_getword (r->pool, &groups_temp, ',');
    }
  }
  else
  {
    groups_table = NULL;
  }

  /*
   * Loop on each "require"
   */
  for (x = 0; x < reqs_arr->nelts; x++)
  {
    /*
     * If this "require" is not for this method, continue...
     */
    if (!(reqs [x].method_mask & (1 << r->method_number)))
    {
      continue;
    }

    /*
     * Else (this "require" is for the request method), the method is
     * "restricted"
     */
    method_restricted = 1;

    /*
     * The "require" keyword (are there other possibilities???)
     */
    t = reqs [x].requirement;

    /*
     * The word behind "require": "valid-user", "user", "group", ...
     */
    w = ap_getword_white (r->pool, &t);

    /*
     * If restriction is just for 1 authenticated user ("valid-user"), it is
     * OK because authentication has already been checked.
     */
    if (strcmp (w, "valid-user") == 0)
    {
#     ifdef SECURID_DEBUG
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		     "#%ld: check_access: require valid-user => OK",
		     (long) getpid ());
#     endif
      return OK;
    }

    /*
     * If restriction if for a specific "user"
     */
    if (strcmp (w, "user") == 0)
    {
      /*
       * For each username, ckeck if it is the one authenticated.
       */
      while (t [0])
      {
	w = ap_getword_conf (r->pool, &t);
	if (strcmp (r->connection->user, w) == 0)
	{
#         ifdef SECURID_DEBUG
	  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
			 "#%ld: check_access: require user `%s' => OK",
			 (long) getpid (), w);
#         endif
	  return OK;
	}
      }
    }
    /*
     * Restriction with "group"
     */
    else if (strcmp (w, "group") == 0)
    {
      /*
       * If the user belongs to a group, then for each group, check if it is
       * the one needed for the authentication.
       */
      if (groups_table)
      {
	while (t [0])
	{
	  w = ap_getword_conf (r->pool, &t);
	  if (ap_table_get (groups_table, w))
	  {
#           ifdef SECURID_DEBUG
	    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
			   "#%ld: check_access: require group `%s' => OK",
			   (long) getpid (), w);
#           endif
	    return OK;
	  }
	}
      }
      else
      {
	/*
	 * The user does not belong to any group.
	 * Check if this is what is wanted (require group "").
	 */
	while (t [0])
	{
	  w = ap_getword_conf (r->pool, &t);
	  if (w && w [0] == '\0')
	  {
#           ifdef SECURID_DEBUG
	    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
			   "#%ld: check_access: require group \"\" => OK",
			   (long) getpid ());
#           endif
	    return OK;
	  }
	}
      }
    }
    /*
     * "require" followed by an unknown directive.
     */
    else if (dconf->authoritative)
    {
      /*
       * If we are authoritative, we log a "error" but we don't say NO
       * immediately...
       */
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		      "access to %s failed, reason: unknown require directive "
		      "`%s'", r->uri, reqs[x].requirement);
    }
  }

  /*
   * If the method is not restricted, it's OK
   */
  if (!method_restricted)
  {
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: check_access: `%s' is unrestricted => OK",
		   (long) getpid (), r->method);
#   endif
    return OK;
  }

  /*
   * So method IS restricted.
   */
   
  /*
   * We decline if we are not authoritative.
   */
  if (!(dconf->authoritative))
  {
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: check_access: we are NOT autho! => declined",
		   (long) getpid ());
#   endif
    return DECLINED;
  }

  /*
   * Else, as we are authoritative, we must say that an AUTH is REQUIRED.
   * But for SecurID, this will be a redirect to the authentication URL, with
   * ?argument sets to the URL the user asked for.
   */
  priv_path = securid_priv_path (r);

  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		 "access to %s failed, reason: user `%s' not allowed access",
		 r->uri, r->connection->user);

  ap_table_setn (r->headers_out, "Location",
		 ap_psprintf (r->pool, "%s?%s&%s",
		              SECURID_URL_FORM_AUTH (r, priv_path),
		              SECURID_REFERER (r), priv_path));
# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: check_access: we DO are autho! => redirect=%s",
		 (long) getpid (), ap_table_get (r->headers_out, "Location"));
# endif
  return REDIRECT;
}

/*
 ************************************************************************
 * Just a little HTML "copyright"...
 ************************************************************************
 */
void securid_print_copyright (request_rec *r)
{
  r->content_type = "text/html";
  ap_send_http_header (r);
  ap_rputs
  (
    "<HTML>"
    "  <HEAD><TITLE>SecurID module for Apache</TITLE></HEAD>"
    "  <BODY BGCOLOR=\"#FFFFFF\">"
    "    <H1 ALIGN=CENTER>SecurID module for Apache</H1>"
    "    <HR>"
    "    <P>"
    "    Thank you for using <A HREF=" SECURID_URL_HOME ">mod_securid</A>..."
    "    </P>"
    "    <HR>"
    "  </BODY>"
    "</HTML>\n", r
  );
}

/*
 ************************************************************************
 * Auth. Failed dispatch:
 *
 * err_code ==	0			means ACE Server said no!, see acm_code
 *          !=	0			means auth. cannot be performed,
 *					because:
 *		SECURID_ACM_NOACM	ACE init is not done
 *		SECURID_ACM_NOHDL	no AceHandle (non-cookie browser?)
 *		SECURID_ACM_NOGOOD	unknown AceHandle or body error(s)
 *		SECURID_ACM_MAXGREQ	repeated passcode requests (get)
 *		SECURID_ACM_MAXPREQ	repeated passcode requests (post)
 *
 * acm_code are:
 *              ACM_ACCESS_DENIED,
 *              ACM_NEXT_CODE_REQUIRED,
 *              ACM_NEXT_CODE_OK: cannot occure,
 *              ACM_NEXT_CODE_BAD,
 *              ACM_NEW_PIN_REQUIRED,
 *              ACM_NEW_PIN_GENERATED (== -ACM_NEW_PIN_REQUIRED),
 *              ACM_NEW_PIN_ACCEPTED,
 *              ACM_NEW_PIN_REJECTED,
 *          else means unexpected error
 * see sdacmvls.h.
 ************************************************************************
 */
#define	SECURID_ACM_NOACM	-1
#define	SECURID_ACM_NOHDL	-2
#define	SECURID_ACM_NOGOOD	-3
#define	SECURID_ACM_MAXGREQ	-4
#define	SECURID_ACM_MAXPREQ	-5
void securid_auth_failed (request_rec *r, int err_code, int acm_code,
                          char *hreferer, securid_webid *wid, char *priv_path)
{
  char		*customfmt;	/* content of custom file		*/
  securid_sconf	*sconf =	/* server conf., just for customtype	*/
    (securid_sconf *) ap_get_module_config (r->server->module_config,
					    &securid_module);

# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
                 "#%ld: auth_failed: uri=%s, err=%d, acm=%d, href=%s, priv=%s",
                 (long) getpid (), r->uri, err_code, acm_code, hreferer,
		 priv_path);
# endif

  /*
   * First send HTTP Header response; For "Content-Type: text/html":
   * If no environment variable AuthSecurID_CustomType, use the server
   * config value.
   */
  r->content_type = ap_table_get (r->subprocess_env, "AuthSecurID_CustomType");
  if (!r->content_type)
  {
    r->content_type = sconf->customtype;
  }
  ap_send_http_header (r);

  /*
   * Dispatch error and print custom (using err_code and acm_code)
   * or default HTML.
   *
   * Be carefull when using sd->xxx, where xxx is a string: this could be NULL,
   * and so "printf" could core dump...
   */
  if (err_code)
  {
    if (err_code == SECURID_ACM_NOACM)
    {
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_NOACM);
      ap_rprintf (r, customfmt);
    }
    else if (err_code == SECURID_ACM_NOHDL)
    {
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_NOHDL);
      ap_rprintf (r, customfmt);
    }
    else if (err_code == SECURID_ACM_NOGOOD)
    {
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_NOGOOD);
      ap_rprintf (r, customfmt);
    }
    else if (err_code == SECURID_ACM_MAXGREQ)
    {
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_MAXGREQ);
      ap_rprintf (r, customfmt);
    }
    else if (err_code == SECURID_ACM_MAXPREQ)
    {
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_MAXPREQ);
      ap_rprintf (r, customfmt);
    }
    else
    {
      /*
       * Unknown error!!!
       */
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_UNKNOWN_ERROR);
      ap_rprintf (r, customfmt, err_code);
    }
  }
  else
  {
    if (acm_code == ACM_ACCESS_DENIED)
    {
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_ACCESS_DENIED);
      ap_rprintf (r, customfmt);
    }
    else if (acm_code == ACM_NEXT_CODE_REQUIRED)
    {
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_NEXT_CODE_REQUIRED);
      ap_rprintf (r, customfmt, SECURID_URL_FORM_CHECK (r, priv_path),
                  wid->username ? wid->username : "", hreferer);
    }
    else if (acm_code == ACM_NEXT_CODE_BAD)
    {
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_NEXT_CODE_BAD);
      ap_rprintf (r, customfmt);
    }
    else if (acm_code == ACM_NEW_PIN_REQUIRED)
    {
      switch (wid->pin_params.Selectable)
      {
	case CANNOT_CHOOSE_PIN:
	{
	  customfmt = securid_customfile
	                (r, SECURID_CUSTOM_CHECK, err_code, acm_code, wid,
			 SECURID_FMT_CHECK_NEW_PIN_REQUIRED_CANNOT_CHOOSE);
          ap_rprintf (r, customfmt, SECURID_URL_FORM_CHECK (r, priv_path),
                      wid->username ? wid->username : "",
		      wid->pin_params.System, hreferer);
          break;
	}
	case USER_SELECTABLE:
	{
	  customfmt = securid_customfile
		      (r, SECURID_CUSTOM_CHECK, err_code, acm_code, wid,
		       wid->pin_params.Alphanumeric
		       ? SECURID_FMT_CHECK_NEW_PIN_REQUIRED_SELECTABLE_ALPHA1
		       : SECURID_FMT_CHECK_NEW_PIN_REQUIRED_SELECTABLE_ALPHA0);
	  ap_rprintf (r, customfmt,
		      wid->pin_params.Min, wid->pin_params.Max,
                      SECURID_URL_FORM_CHECK (r, priv_path),
                      wid->username ? wid->username : "", hreferer,
                      SECURID_URL_FORM_CHECK (r, priv_path),
                      wid->username ? wid->username : "",
		      wid->pin_params.System, hreferer);
break;
	}
	case MUST_CHOOSE_PIN:
	{
	  customfmt = securid_customfile
		      (r, SECURID_CUSTOM_CHECK, err_code, acm_code, wid,
		       wid->pin_params.Alphanumeric
		       ? SECURID_FMT_CHECK_NEW_PIN_REQUIRED_MUST_CHOOSE_ALPHA1
		       : SECURID_FMT_CHECK_NEW_PIN_REQUIRED_MUST_CHOOSE_ALPHA0);
	  ap_rprintf (r, customfmt,
		      wid->pin_params.Min, wid->pin_params.Max,
                      SECURID_URL_FORM_CHECK (r, priv_path),
                      wid->username ? wid->username : "", hreferer);
          break;
	}
      }
    }
    else if (acm_code == ACM_NEW_PIN_GENERATED)
    {
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_NEW_PIN_GENERATED);
      ap_rprintf (r, customfmt, hreferer, wid->pin_params.System, hreferer);
      /*
       * Don't forget to clear wid->pin_params.System (search for
       * ACM_NEW_PIN_GENERATED in both this file and securid_client.c).
       */
      memset (wid->pin_params.System, 0, sizeof (wid->pin_params.System));
    }
    else if (acm_code == ACM_NEW_PIN_ACCEPTED)
    {
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_NEW_PIN_ACCEPTED);
      ap_rprintf (r, customfmt, hreferer, hreferer);
    }
    else if (acm_code == ACM_NEW_PIN_REJECTED)
    {
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_NEW_PIN_REJECTED);
      ap_rprintf (r, customfmt);
    }
    else
    {
      /*
       * Unknown acm error!!!
       */
      customfmt = securid_customfile (r, SECURID_CUSTOM_CHECK,
				      err_code, acm_code, wid,
				      SECURID_FMT_CHECK_UNKNOWN_ERROR);
      ap_rprintf (r, customfmt, acm_code);
    }
  }
}

/*
 ************************************************************************
 * HTML form for authenticate
 ************************************************************************
 */
void securid_print_form (request_rec *r, char *hreferer, char *priv_path)
{
  char		*customfmt;	/* content of custom file		*/

  /*
   * Don't send HTTP Header response: we have already did it...
   */

  /*
   * Compute custom or default format and print.
   */
  customfmt = securid_customfile (r, SECURID_CUSTOM_AUTH,
				  0, 0, NULL,
				  SECURID_FMT_AUTH);
  ap_rprintf (r, customfmt, SECURID_URL_FORM_CHECK (r, priv_path), hreferer);
}

/*
 ************************************************************************
 * Handler to display authenticate form
 ************************************************************************
 */
static int securid_handler_auth (request_rec *r)
{
  /*
   * This handler, triggered by SECURID_URL_AUTH, should be called only by
   * GET, without BODY and with 2 arguments:
   * - 1: the initial URL the user * wanted to reach (before being redirected
   *      to the authentication form),
   * - 2: the "private path", use for cookie "path"
   *
   * If this is not true, then maybe user is "playing";  we just print a
   * "copyright".
   *
   * Else, we print the username/passcode form; visible fields are username
   * and passcode; a hidden field: referer, is set to the request argument
   * (ie the initial URL the user wanted to reach).
   *
   * The form returns AceHandle=id cookie and triggers SECURID_URL_CHECK (see
   * securid_handler_check).
   *
   * Note: user is here because he has been "redirected" by access check
   * (securid_check_access). Unfortunately, this redirection makes unusable
   * "Referer" field, which is then null.
   * This is why the initial URL is sent as an argument (and not with Referer
   * as the "real" SecurID module...).
   *
   * Note2: user may also be here because he's playing with the "back". So, we
   * will check for SecurID cookie: if there is one, we just do the same,
   * *without* AceHandle=id cookie.
   *
   * Note3: the AceHandle=id cookie *must* use "path=/priv..." and not
   * something like "path=/securid..." to be RFC compliance.
   * So we use "path=/priv.../securid/" for this handler and for next URL
   * (SECURID_URL_CHECK), "path=/priv...".
   * We could "track" the private path, but it is safer to store it now...
   */
  /*
   * Referer & private path are in the URL arguments.
   */
  char		*referer   = NULL;
  char		*priv_path = NULL;
  /*
   * The authentication handle: something unique.
   */
  const char	*handleid;
  /*
   * The handle cookie text:
   *	AceHandle=<id>; path=URI/securid/ [; domain=...; [ secure ]]
   */
  char		*handletxt;
  /*
   * The webid cookie;
   */
  securid_webid	cookie;
  /*
   * True if user is playing "back" button.
   */
  int		back = 0;

  /*
   * Server config, just for auth_get for cache file.
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (r->server->module_config,
					    &securid_module);

  /*
   * The initial URL has been encoded (the REDIRECT at the end of
   * securid_check_auth).
   * But '&' are not escaped (?); so use a reverse-search.
   */
  if (r->args)
  {
    referer   = r->args;
    priv_path = strrchr (r->args, '&');
    if (priv_path)
    {
      *priv_path = '\0';
      priv_path++;
      ap_unescape_url (priv_path);
    }
    ap_unescape_url (referer);
  }

# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: handler_auth: %s %s, referer = %s, priv_path = %s",
		 (long) getpid (), r->method, r->uri, referer ? referer : "",
		 priv_path ? priv_path : "");
# endif

  /*
   * We should have been called by GET and with a "referer" & a "priv_path".
   */
  if (r->method_number != M_GET || !referer || !priv_path)
  {
    /*
     * Then user is "playing" with the URL...
     */
    securid_print_copyright (r);
    return 0;
  }

  /*
   * Here, we could check ACE init has been done. But if it has not been done
   * *and* we still cannot do it, what sort of error should we print?
   * So this check will be done in securid_handler_check (so error message
   * could be used...).
   */

  /*
   * Here we check for SecurID cookies: the handle *and* the webid.
   * If the browser provides those cookie, we then check if this authentication
   * is already in the cache.
   * If so, then the user is playing with the "back" button. He will be
   * immediatly redirected to the "referer" (as in securid_handler_check).
   */
  /*
   * Get browser handle cookie.
   */
  handleid = securid_get_handle (r, sconf->hdlcookie);
  if (handleid)
  {
    /*
     * Get browser webid cookie.
     */
    if (securid_get_webid (r, sconf->widcookie, &cookie))
    {
      /*
       * Check this handleid is in cache:
       * handleid must exists *and* username != "".
       */
      if (securid_auth_get (r, sconf->cachefile, sconf->lockfile, handleid,
                            &cookie) == 0 &&
          cookie.username [0])
      {
	/*
	 * Then the user is playing with the "back" button.
	 */
        back = 1;
#       ifdef SECURID_DEBUG
	ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		       "#%ld: handler_auth: handle=%s "
		       "(are you playing \"back\"?)",
		       (long) getpid (), handleid);
#       endif
      }
    }
  }

  /*
   * If the user is not playing "back", then we need to set the handle cookie.
   */
  if (!back)
  {
    int		dos;			/* ret. val. from auth_dos()	*/
    /*
     * To compute the <id> (AceHandle=<id>), we use rand(); this is not
     * perfect, but enough...
     * Moreover, each child has initialized rand() (see securid_child_init).
     *
     * If the handle is not set or not yet in cache, then we compute a new <id>,
     * else we re-use the old one.
     */
    if (!handleid ||
        (handleid &&
	 securid_auth_get (r, sconf->cachefile, sconf->lockfile, handleid,
	                   &cookie) != 0))
    {
      char		id [20];	/* "unique" id to compute	*/
      int		i = 5;		/* loop i times to compute id	*/
      int		ok;		/* if i random are not enough	*/

      /*
       * We will try "i" loops to compute this unique id.
       */
      while (i--)
      {
	ap_snprintf (id, sizeof (id), "%ld", (long int) rand ());
	ok = securid_auth_get (r, sconf->cachefile, sconf->lockfile, id,
	                       &cookie);
	if (ok > 0)
	{
	  /*
	   * ok != 0 means that this id does not already exist, so it's OK...
	   */
	  break;
	}
#       ifdef SECURID_DEBUG
	ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		       "#%ld: handler_auth: bad random!",
		       (long) getpid ());
#       endif
      }
      /*
       * ok == 0 means that id already exists, ok < 0 means xdbm error(s).
       */
      if (ok <= 0)
      {
	/*
	 * securid_auth_get always returned ok (id already exists):
	 * impossible (?)
	 */
	ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		      "SecurID: could not compute AceHandle");
	exit (EXIT_FAILURE);
      }
      handleid = ap_pstrdup (r->pool, id);
#     ifdef SECURID_DEBUG
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		     "#%ld: handler_auth: new handle `%s' computed",
		     (long) getpid (), handleid);
#     endif
    }
#   ifdef SECURID_DEBUG
    else
    {
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		     "#%ld: handler_auth: old handle `%s' re-used",
		     (long) getpid (), handleid);
    }
#   endif

    /*
     * Now we have an <id>, check for DoS.
     */
    strncpy (cookie.from_agent, securid_fromagent (r),
	     sizeof (cookie.from_agent) - 1);
    cookie.from_agent [sizeof (cookie.from_agent) - 1] = '\0';
    dos = securid_auth_dos (r, sconf->cachefile, sconf->lockfile,
                            cookie.from_agent, sconf->maxcachesize,
			    sconf->maxauthget);
    if (dos > 0)
    {
      /*
       * Stop it now. We use auth_failed().
       * returned value == 1 => maxauthget reached.
       * returned value == 2 => maxcachesize reached.
       *
       * Note that "returned value" could be -1 on xdbm error(s)...
       */
      if (dos == 1)
      {
	/* error msg already logged (see auth_dos())			*/
	securid_auth_failed (r, SECURID_ACM_NOACM, 0, "", &cookie, "");
      }
      else
      {
	ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		       "SecurID: DoS check: "
		       "repeated passcode requests (get) from `%s'",
		       cookie.from_agent);
	securid_auth_failed (r, SECURID_ACM_MAXGREQ, 0, "", &cookie, "");
      }
      return 0;
    }

    /*
     * With this <id>, compute handle cookie text.
     * DO use "path=/priv.../securid/", and not only "path=/priv..." (see
     * Note3 bellow).
     */
    handletxt = ap_psprintf (r->pool, "%s=%s; path=%s",
	                     sconf->hdlcookie
			     ? sconf->hdlcookie
			     : SECURID_HANDLE_NAME,
			     handleid, SECURID_URL_FORM (r, priv_path));
    if (sconf->domaincookie)
    {
      handletxt = ap_psprintf (r->pool,
			       "%s; domain=%s", handletxt, sconf->domaincookie);
    }
    if (sconf->securecookie)
    {
      handletxt = ap_psprintf (r->pool, "%s; secure", handletxt);
    }

    /*
     * Start storing in cache this auth (with an unknown cookie.sd, this will
     * set during the next handler: securid_handler_check, after ACE init).
     *
     * We also set an empty username: this meens user is not yet authenticated.
     */
    cookie.username [0]   = '\0';
    cookie.first_time     = time (NULL);
    cookie.shell [0]      = '\0';
    cookie.sd             = SDI_HANDLE_NONE;;
    cookie.last_time      = (sconf->cachettltype == SECURID_TTL_ALWAYS)
                              ? (time_t) -1 : time (NULL);
    /* cookie.from_agent already done...				*/
    strncpy (cookie.user_agent, securid_useragent (r),
	     sizeof (cookie.user_agent) - 1);
    cookie.user_agent [sizeof (cookie.user_agent) - 1] = '\0';
    cookie.acecalls       = 0;
    /* also store the private path					*/
    strncpy (cookie.priv_path, priv_path, sizeof (cookie.priv_path) - 1);
    cookie.priv_path [sizeof (cookie.priv_path) - 1] = '\0';
    (void) securid_auth_put (r, sconf->cachefile, sconf->lockfile, handleid,
                             &cookie);

    /*
     * And we put the cookie in the HTTP response header.
     *
     * !!: table_setn because handletxt is allocated from pool.
     */
    ap_table_addn (r->headers_out, "Set-Cookie", handletxt);
  }

  /*
   * Now the HTTP response header. We will have:
   *   Date: Mon, 06 Dec 1999 10:10:10 GMT
   *   Server: Apache/1.3.9 (Unix)
   *   [Set-Cookie: AceHandle=<id>; path=/priv...] (if !back)
   *   Pragma: no-cache
   *   Connection: close
   *   Content-Type: text/html
   *   Expires: Mon, 06 Dec 1999 10:10:10 GMT
   */
  securid_nocache (r, sconf->nocache);

  /*
   * For "Content-Type: text/html":
   * If no environment variable AuthSecurID_CustomType, use the server
   * config value.
   */
  r->content_type = ap_table_get (r->subprocess_env, "AuthSecurID_CustomType");
  if (!r->content_type)
  {
    r->content_type = sconf->customtype;
  }

  /*
   * Send those headers.
   */
  ap_send_http_header (r);

  /*
   * If it was in fact a HEAD method (and not GET), we do not need to print
   * any body.
   * (Is this really reasonable? When this could happen? Maybe when the form
   * is asked by a proxy; but there are some cache directives (see above) to
   * prevent this...).
   */
  if (r->header_only)
  {
    return 0;
  }

  /*
   * Print SecurID PASSCODE request form.
   */
  securid_print_form (r, securid_hescape (r, referer), priv_path);

  /*
   * It's OK, next time, securid_handler_check will be called...
   */
  return 0;
}

/*
 ************************************************************************
 * Handler to check authentication
 ************************************************************************
 */
static int securid_handler_check (request_rec *r)
{
  /*
   * This handler, triggered by SECURID_URL_CHECK, should be called by POST,
   * with a BODY.
   *
   * If this is not true, then maybe user is "playing";  we just print a
   * "copyright".
   *
   * Else (POST with a BODY), we should find in this body:
   *   sd_action=passcode&sd_username=_u_&sd_passcode=_passcode_&sd_referer=_r_
   * or
   *   sd_action=next_code&sd_username=_u_&sd_passcode=_code_&sd_referer=_r_
   * or
   *   sd_action=new_pin_usr&sd_username=_u_&sd_passcode=_pin_&sd_referer=_r_
   * or
   *   sd_action=new_pin_sys&sd_username=_u_&sd_passcode=_pin_&sd_referer=_r_
   *
   * (_u_ is the username and _r_ the referer).
   *
   * Finally, form ACTION that triggered this handler (see
   * securid_handler_auth), has set a AceHandle=_id_ cookie.
   *
   * If auth is OK:
   *   return a redirect to referer,
   *   return the cookie webid2=__username:first_time__; path=/priv...
   *
   * ("Pragma: no-cache" and "Expires: 0" will be set with by authentication
   * checker, just after redirect, each time a user access "private" pages.)
   *
   * If we need to return before all the body has been read, we have to
   * discard it.
   */
  const char	*handle;		/* SecurID session id		*/
  char		*cookie;		/* SecurID auth cookie		*/
  int		ace_result;		/* error code from ACE server	*/
  char		*referer   = NULL;	/* where do we come from	*/
  securid_webid	auth;			/* for sd_client struct		*/
  securid_sconf	*sconf     =		/* server conf (for cache file)	*/
    (securid_sconf *) ap_get_module_config (r->server->module_config,
					    &securid_module);
  char		body [HUGE_STRING_LEN];	/* body content & dispatch	*/
  char		trash [512];		/* to trash end of body		*/
  int		len;
  char		*begin;
  char		*end      = NULL;
  char		*action   = NULL;
  char		*username = NULL;
  char		*passcode = NULL;
  int		is_check;

# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: handler_check: %s %s",
		 (long) getpid (), r->method, r->uri);
# endif

  /*
   * We should have been called by POST
   */
  if (r->method_number != M_POST)
  {
    /*
     * The user is "playing"...
     */
    (void) ap_discard_request_body (r);
    securid_print_copyright (r);
    return 0;
  }

  /*
   * Check for AceHandle cookie. If not, this is maybe a browser that does
   * not support cookie...
   */
  handle = securid_get_handle (r, sconf->hdlcookie);
  if (handle == NULL)
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		   "SecurID: ACE syntax error (no AceHandle)");

    /*
     * "authentication" has failed, so access is denied.
     * "" is for referer, NULL for sd struct; unused here.
     */
    (void) ap_discard_request_body (r);
    securid_auth_failed (r, SECURID_ACM_NOHDL, 0, "", NULL, "");
    return 0;
  }

  /*
   * Get cache value for this handle. If this handle is not yet in cache,
   * then the user is really "playing"...
   */
  if (securid_auth_get (r, sconf->cachefile, sconf->lockfile, handle,
                        &auth) != 0)
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		   "SecurID: ACE syntax error (unknown AceHandle `%s')",
		   handle);

    /*
     * "authentication" has failed, so access is denied.
     * "" is for referer, NULL for sd struct; unused here.
     */
    (void) ap_discard_request_body (r);
    securid_auth_failed (r, SECURID_ACM_NOGOOD, 0, "", NULL, "");
    return 0;
  }

  /*
   * Check this user is still allowed to try authentication:
   * 1/ maxauthpost is not null
   * and
   * 2/ not too many access failures (acecalls >= maxauthpost)
   */
# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: handler_check: call #%d/%d for AceHandle %s",
		 (long) getpid (), auth.acecalls, sconf->maxauthpost, handle);
# endif
  if (sconf->maxauthpost && auth.acecalls >= sconf->maxauthpost)
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		   "SecurID: DoS check: "
		   "repeated passcode requests (post) from `%s'",
		   auth.from_agent);

    /*
     * Just delete this auth.
     */
    (void) securid_auth_del (r, sconf->cachefile, sconf->lockfile, handle);

    /*
     * "authentication" has failed, so access is denied.
     * "" is for referer; unused here.
     */
    (void) ap_discard_request_body (r);
    securid_auth_failed (r, SECURID_ACM_MAXPREQ, 0, "", &auth, "");
    return 0;
  }
  /*
   * Still ok, just update in cache this info right now.
   */
  auth.acecalls++;
  (void) securid_auth_put (r, sconf->cachefile, sconf->lockfile, handle, &auth);

  /*
   * We check for a "Content-length" field (and a body not chunked),
   * and then we check we should not block when reading the body.
   */
  if (ap_setup_client_block (r, REQUEST_CHUNKED_ERROR) ||
      !ap_should_client_block (r))
  {
    /*
     * No body... this is "strange"...
     */
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		   "SecurID: ACE syntax error (no body)");
    securid_auth_failed (r, SECURID_ACM_NOGOOD, 0, "", &auth, "");
    return 0;
  }

  /*
   * So a body is here, we should read it with a loop on
   * ap_get_client_block (). But as we know the body's length should be
   * small, we just use 1 ap_get_client_block() (HUGE_STRING_LEN, in
   * httpd.h, is set to 8192).
   */
  if ((len = ap_get_client_block (r, body, sizeof (body))) > 0)
  {
    /*
     * We have to string to the length of the body (... and we have to take
     * care to CR, LF or CR-LF: this differs between browsers, OS, year,
     * weather, ...).
     */
    if (body [len - 2] == '\r' && body [len - 1] == '\n')
    {
      len -= 2;
    }
    else if (body [len - 1] == '\r' || body [len - 1] == '\n')
    {
      len -= 1;
    }
    else if (len == sizeof (body))
    {
      len -= 1;
    }
    
    /*
     * Then discard the rest of the body, just in case
     */
    while (ap_get_client_block (r, trash, sizeof (trash)) > 0)
    {
      continue;
    }
  }
  else
  {
    /*
     * No body??? No error now: just dispatch variables and we'll then have
     * empty variables (=> error...).
     */
    len = 0;
  }
  body [len] = '\0';

# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: handler_check: BODY=`%s'",
		 (long) getpid (), body);
# endif

  /*
   * Just dispatch variables
   */
  begin = body;
  end   = NULL;
  if (begin && strncmp (begin, SECURID_FORM_N_ACTION "=",
                        strlen (SECURID_FORM_N_ACTION "=")) == 0)
  {
    begin += strlen (SECURID_FORM_N_ACTION "=");
    end   = strchr (begin, '&');
    if (!end) end = begin + strlen (begin);
    if (end > begin)
      action = ap_pstrndup (r->pool, begin, end - begin);
    else
      action = NULL;
    if (action) ap_unescape_url (action); else action = "";
  }
  else
  {
    action = "";
  }

  if (end && *end) begin = end + 1; else begin = NULL;
  end = NULL;
  if (begin && strncmp (begin, SECURID_FORM_N_USERNAME "=",
                        strlen (SECURID_FORM_N_USERNAME "=")) == 0)
  {
    begin += strlen (SECURID_FORM_N_USERNAME "=");
    end   = strchr (begin, '&');
    if (!end) end = begin + strlen (begin);
    if (end > begin)
      username = ap_pstrndup (r->pool, begin, end - begin);
    else
      username = NULL;
    if (username) ap_unescape_url (username); else username = "";
  }
  else
  {
    username = "";
  }

  if (end && *end) begin = end + 1; else begin = NULL;
  end = NULL;
  if (begin && strncmp (begin, SECURID_FORM_N_PASSCODE "=",
                        strlen (SECURID_FORM_N_PASSCODE "=")) == 0)
  {
    begin += strlen (SECURID_FORM_N_PASSCODE "=");
    end   = strchr (begin, '&');
    if (!end) end = begin + strlen (begin);
    if (end > begin)
      passcode = ap_pstrndup (r->pool, begin, end - begin);
    else
      passcode = NULL;
    if (passcode) ap_unescape_url (passcode); else passcode = "";
  }
  else
  {
    passcode = "";
  }

  if (end && *end) begin = end + 1; else begin = NULL;
  if (begin && strncmp (begin, SECURID_FORM_N_REFERER "=",
                        strlen (SECURID_FORM_N_REFERER "=")) == 0)
  {
    begin += strlen (SECURID_FORM_N_REFERER "=");
    referer = ap_pstrdup (r->pool, begin);
    if (referer) ap_unescape_url (referer); else referer = "";
  }
  else
  {
    referer = "";
  }

  /*
   * Done with the body. Zero it out, to minimize security exposure...
   */
  memset (body, 0, sizeof (body));

# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: handler_check: %s=`%s', "
		 "%s=`%s', %s=`%s', %s=`%s'",
		 (long) getpid (),
		 SECURID_FORM_N_ACTION, action,
		 SECURID_FORM_N_USERNAME, username,
		 SECURID_FORM_N_PASSCODE, passcode,
		 SECURID_FORM_N_REFERER, referer);
# endif

  /*
   * Check BODY fields are not "". Start with passcode (to know if it is ""),
   * so we will zero it out on error.
   */
  if (passcode [0] == '\0')
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		   "SecurID: ACE syntax error (no passcode)");
    securid_auth_failed (r, SECURID_ACM_NOGOOD, 0, "", &auth, "");
    return 0;
  }
  if (action [0] == '\0')
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		   "SecurID: ACE syntax error (no action)");
    memset (passcode, 0, strlen (passcode));
    securid_auth_failed (r, SECURID_ACM_NOGOOD, 0, "", &auth, "");
    return 0;
  }
  if (username [0] == '\0')
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		   "SecurID: ACE syntax error (no username)");
    memset (passcode, 0, strlen (passcode));
    securid_auth_failed (r, SECURID_ACM_NOGOOD, 0, "", &auth, "");
    return 0;
  }
  if (referer [0] == '\0')
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		   "SecurID: ACE syntax error (no referer)");
    memset (passcode, 0, strlen (passcode));
    securid_auth_failed (r, SECURID_ACM_NOGOOD, 0, "", &auth, "");
    return 0;
  }

  is_check = (strcmp (action, SECURID_FORM_V_PASSCODE) == 0);

  /*
   * Before calling any sd_xxx(), set the acm err code.
   */
  ace_result = SECURID_ACM_NOACM;

  /*
   * Then dispatch action and call sd_xxx().
   */
  if (is_check)
  {
    ace_result = securid_call_check (sconf, r, &auth, username, passcode);
  }
  else if (strcmp (action, SECURID_FORM_V_NEXTCODE) == 0)
  {
    ace_result = securid_call_next (sconf, r, &auth, passcode);
  }
  else if (strcmp (action, SECURID_FORM_V_NEWPINUSR) == 0)
  {
    ace_result = securid_call_pin (sconf, r, &auth, passcode);
  }
  else if (strcmp (action, SECURID_FORM_V_NEWPINSYS) == 0)
  {
    ace_result = securid_call_pin (sconf, r, &auth, passcode);
    /*
     * To print new PIN, just simulate an error so that
     * securid_auth_failed() will print it.
     * As sd_pin clears sd->system_pin, we need to re-store it (and to
     * re-clear it when we will have print it).
     */
    if (ace_result == ACM_NEW_PIN_ACCEPTED)
    {
      ace_result = ACM_NEW_PIN_GENERATED;
    }
  }
  else
  {
    /*
     * Unknown action... this is "very strange"...
     */
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		   "SecurID: ACE syntax error (unknown action `%s')", action);
    memset (passcode, 0, strlen (passcode));
    securid_auth_failed (r, SECURID_ACM_NOGOOD, 0, "", &auth, "");
    return 0;
  }

  /*
   * Done with the passcode. Zero it out, to minimize security exposure...
   */
  memset (passcode, 0, strlen (passcode));

  /*
   * After sd_xxx() call, we have to re-store sd_client struct if we
   * want to re-use it later.
   * We also have to reset acecalls if result is "ok".
   */
  if (ace_result == ACM_OK)
  {
    auth.acecalls = 0;
  }
  (void) securid_auth_put (r, sconf->cachefile, sconf->lockfile, handle, &auth);

  /*
   * For new_pin_* Requests, ace_result will be *_ACCEPTED or *_REJECTED,
   * so auth. will not be OK and securid_auth_failed() will print the
   * right HTML.
   *
   * Else, if ace_result is OK, then auth is OK and we have to deal with
   * the SecurID cookie.
   */
  if (strncmp (action, SECURID_FORM_V_NEWPIN,
	       strlen (SECURID_FORM_V_NEWPIN)) &&
      ace_result == ACM_OK)
  {
    /*
     * action != new_pin_* and ACE is happy...
     *
     * We will of course return auth cookie (webid2) but also handle cookie
     * (AceHandle=_id_); this is usefull when AceHandle has changed but as
     * the browser is still running, it sends the "old" handle.
     */
    cookie = ap_psprintf (r->pool, "%s=%s; path=%s",
			  sconf->hdlcookie
			  ? sconf->hdlcookie
			  : SECURID_HANDLE_NAME,
			  handle, auth.priv_path);
    if (sconf->domaincookie)
    {
      cookie = ap_psprintf (r->pool,
			    "%s; domain=%s", cookie, sconf->domaincookie);
    }
    if (sconf->securecookie)
    {
      cookie = ap_psprintf (r->pool, "%s; secure", cookie);
    }
    /*
     * !!: table_addn because cookie is allocated from pool.
     * !!: err_headers_out because see bellow next ap_table_setn.
     */
    ap_table_addn (r->err_headers_out, "Set-Cookie", cookie);

    cookie = ap_psprintf (r->pool, "%s=%s; path=%s",
			  sconf->widcookie
			  ? sconf->widcookie
			  : SECURID_COOKIE_NAME,
			  securid_set_webid (r, &auth), auth.priv_path);
    if (sconf->domaincookie)
    {
      cookie = ap_psprintf (r->pool,
			    "%s; domain=%s", cookie, sconf->domaincookie);
    }
    if (sconf->securecookie)
    {
      cookie = ap_psprintf (r->pool, "%s; secure", cookie);
    }
    /*
     * from_agent & user_agent are already stored, see
     * securid_handler_auth() (search for '!back' and look at end of `if').
     * Just re-use them.
     */
    /*
     * We store this auth in the cache.
     */
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: handler_check: ACE returned ACM_OK, "
		   "cookie=`%s'", (long) getpid (), cookie);
#   endif
    (void) securid_auth_put (r, sconf->cachefile, sconf->lockfile, handle,
                             &auth);
     
    /*
     * We put in the response headers the webid2 cookie.
     * Note: we have to use err_headers_out and not headers_out because as
     * this is a redirect (a sort of error), headers_out will be cleanned,
     * and cookie will never be returned.
     *
     * !!: table_addn because cookie is allocated from pool
     */
    ap_table_addn (r->err_headers_out, "Set-Cookie", cookie);

    /*
     * So authentication is OK. Just redirect use to the initial URL
     * asked for, before authentication form.
     *
     * !!: table_setn because referer is allocated from pool.
     */
    ap_table_setn (r->headers_out, "Location", referer);
    return REDIRECT;
  }
  else
  {
    /*
     * ACE is not happy... Or this is was just a Next PIN Request.
     * Access will be denied (see securid_auth_failed, at the end of this
     * function).
     */
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: handler_check: username=`%s', "
		   "ACE result=%d (%d, %d)",
		   (long) getpid (), username, ace_result,
		   ace_result == ACM_NEW_PIN_REQUIRED
		     ? auth.pin_params.Selectable : -1,
		   ace_result == ACM_NEW_PIN_REQUIRED
		     ? auth.pin_params.Alphanumeric : -1);
#   endif
    if (ace_result == ACM_ACCESS_DENIED)
    {
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		     "SecurID: ACE access denied for user `%s'",
		     username);
    }
    else if (ace_result == ACM_NEXT_CODE_BAD)
    {
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		     "SecurID: ACE next code bad for user `%s'",
		     username);
    }
    else if (ace_result == ACM_NEW_PIN_REJECTED)
    {
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		     "SecurID: ACE new pin rejected for user `%s'",
		     username);
    }
    else if (ace_result != ACM_OK			&&
	     ace_result != ACM_NEXT_CODE_REQUIRED	&&
	     ace_result != ACM_NEW_PIN_REQUIRED	&&
	     ace_result != ACM_NEW_PIN_ACCEPTED	&&
	     ace_result != ACM_NEW_PIN_GENERATED)
    {
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
		     "SecurID: ACE unknown error #%d for user `%s'",
		     ace_result, username);
    }
  }

  /*
   * Here, authentication has failed, so access is denied.
   */
  securid_auth_failed (r, 0, ace_result, securid_hescape (r, referer),
                       &auth, auth.priv_path);
  return 0;
}

/*
 ************************************************************************
 * Handler for authentication status
 *
 * This handler is triggered by securid_translate.
 *
 * Of course, access to this location should be restricted with SecurID...
 * With arg=logout, we logout user (iae delete auth. from cache).
 * Else, we print authentication status.
 *
 ************************************************************************
 */
static int securid_handler_status (request_rec *r)
{
  const char	*handle;		/* AceHandle			*/
  securid_webid	cookie;			/* SecurID auth cookie		*/
  securid_sconf	*sconf =		/* server conf (for cache file)	*/
    (securid_sconf *) ap_get_module_config (r->server->module_config,
					    &securid_module);
  const char	*auth_type =		/* AuthType value		*/
  		  ap_auth_type (r);
  time_t	this_time = time (NULL);/* current time			*/
  long		use_time;		/* time auth. has been used (s)	*/
  long		exp_time;		/* auth. expiration time (s)	*/
  char		*expire1 = NULL,	/* expiration text (1)		*/
  		*expire2 = NULL;	/* "             " (2)		*/
  int		customcode;		/* see securid_customfile()	*/
  char		*customfmt;		/* content of custom file	*/

# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: handler_status: %s %s",
		 (long) getpid (), r->method,
		 r->args
		   ? ap_psprintf (r->pool, "%s?%s", r->uri, r->args)
		   : r->uri);
# endif

  /*
   * Send HTTP headers; for "Content-Type: text/html":
   * If no environment variable AuthSecurID_CustomType, use the server
   * config value.
   */
  r->content_type = ap_table_get (r->subprocess_env, "AuthSecurID_CustomType");
  if (!r->content_type)
  {
    r->content_type = sconf->customtype;
  }
  ap_send_http_header (r);

  /*
   * If this <Location> is for SecurID, then
   *   - AuthType is not NULL *and* == "SecurID"
   * and then if user is authenticated:
   *   - "$REMOTE_USER" is not null *and* there is a webid cookie.
   *
   * Check this.
   */
  if (auth_type == NULL			||
      strcasecmp (auth_type, "SecurID")	||
      !r->connection->user		||
      !securid_get_webid (r, sconf->widcookie, &cookie))
  {
    /*
     * This is not for us... Compute custom or default format and print.
     */
    customfmt = securid_customfile (r, SECURID_CUSTOM_STATUS,
				    -1, 0, NULL,
				    SECURID_FMT_STATUS_NOAUTH);
    ap_rprintf (r, customfmt);

    /*
     * Stop request now.
     */
    return OK;
  }

  /*
   * Logout user?
   */
  if (r->args && strcasecmp (r->args, "logout") == 0)
  {
    /*
     * Delete this auth. from cache.
     */
    handle = securid_get_handle (r, sconf->hdlcookie);
    (void) securid_auth_del (r, sconf->cachefile, sconf->lockfile, handle);
    customcode = 0;			/* logout...			*/
  }
  else
  {
    /*
     * Compute expiration
     */
    if (sconf->cachettltype == SECURID_TTL_ALWAYS)
    {
      use_time = (long) difftime (this_time, cookie.first_time);
      exp_time = (long) sconf->cachettltime - use_time;

      customcode = 1;			/* even if you use...		*/
      expire1    = ap_psprintf (r->pool, "%ld", exp_time);
    }
    else
    {
      if (sconf->cachettlmax)
      {
	use_time = (long) difftime (this_time, cookie.first_time);
	exp_time = (long) sconf->cachettlmax - use_time;
	if (exp_time > sconf->cachettltime)
	{
	  customcode = 2;		/* if you do not use..., or...	*/
	  expire1    = ap_psprintf (r->pool, "%ld", sconf->cachettltime);
	  expire2    = ap_psprintf (r->pool, "%ld", exp_time);
	}
	else
	{
	  customcode = 1;		/* even if you use...		*/
	  expire1    = ap_psprintf (r->pool, "%ld", exp_time);
	}
      }
      else
      {
	customcode = 3;			/* if you do not use...		*/
	expire1    = ap_psprintf (r->pool, "%ld", sconf->cachettltime);
      }
    }
  }

  /*
   * Compute custom or default format and show status.
   */
       if (customcode == 1)
  {
#   define SECURID_FMT_STATUS_AUTH1_ARGS				\
	   r->connection->user, cookie.username,			\
	   cookie.shell [0] ? cookie.shell : "<i>-</i>",		\
	   ap_psprintf (r->pool, "%ld",					\
			(long) difftime (this_time, cookie.first_time)),\
	   expire1

    customfmt = securid_customfile (r, SECURID_CUSTOM_STATUS,
				    customcode, 0, NULL,
				    SECURID_FMT_STATUS_AUTH1);
    ap_rprintf (r, customfmt,
                SECURID_FMT_STATUS_AUTH1_ARGS,
		SECURID_URL_FORM_STATUS (r, r->uri));
  }
  else if (customcode == 2)
  {
#   define SECURID_FMT_STATUS_AUTH2_ARGS				\
	   SECURID_FMT_STATUS_AUTH1_ARGS, expire2

    customfmt = securid_customfile (r, SECURID_CUSTOM_STATUS,
				    customcode, 0, NULL,
				    SECURID_FMT_STATUS_AUTH2);
    ap_rprintf (r, customfmt,
                SECURID_FMT_STATUS_AUTH2_ARGS,
		SECURID_URL_FORM_STATUS (r, r->uri));
  }
  else if (customcode == 3)
  {
#   define SECURID_FMT_STATUS_AUTH3_ARGS				\
	   SECURID_FMT_STATUS_AUTH1_ARGS

    customfmt = securid_customfile (r, SECURID_CUSTOM_STATUS,
				    customcode, 0, NULL,
				    SECURID_FMT_STATUS_AUTH3);
    ap_rprintf (r, customfmt,
                SECURID_FMT_STATUS_AUTH3_ARGS,
		SECURID_URL_FORM_STATUS (r, r->uri));
  }
  else /* (customcode == 0) */
  {
#   define SECURID_FMT_STATUS_AUTH0_ARGS				\
	   r->connection->user, cookie.username,			\
	   cookie.shell [0] ? cookie.shell : "<i>-</i>",		\
	   ap_psprintf (r->pool, "%ld",					\
			(long) difftime (this_time, cookie.first_time))

    customfmt = securid_customfile (r, SECURID_CUSTOM_STATUS,
				    customcode, 0, NULL,
				    SECURID_FMT_STATUS_AUTH0);
    ap_rprintf (r, customfmt, SECURID_FMT_STATUS_AUTH0_ARGS);
  }
  /*
   * Stop request now.
   */
  return OK;
}

/*
 ************************************************************************
 * Handler to translate URI
 ************************************************************************
 */
static int securid_translate (request_rec *r)
{
  /*
   * This handler is used to "translate" URI and call auth or check handler.
   *
   * When not running as a (cache)proxy, "handler" requests are something like
   * "/URI/securid/(auth|check)?_referer_".
   *
   * When running as a cache-proxy, "handler" requests are something like
   * "http://www/URI/securid/(auth|check)?_referer_".
   *
   * So if we are a proxy, we first remove "scheme://host" from URI.
   *
   * Then we check the URI:
   * - if it is "/URI/securid/auth", we "call" the auth handler,
   * - if it is "/URI/securid/check", we "call" the check handler.
   *
   * To "call" another handler means to store the handler name in r->handler.
   * Why not?, this is what mod_rewrite does...
   */
  char	*curi;			/* to parse r->uri			*/
  char	*newhandler;
  int	strlen_curi;		/* strlen (), skeeping '?...' if needed	*/

  /*
   * First check if we are a "standard" (not reverse) proxy
   */
  if (r->proxyreq == STD_PROXY)
  {
    /*
     * Ok, we are a proxy, skip 'scheme:' from URI
     */
    for (curi = r->uri; *curi != ':' && *curi != '\0'; curi++)
      ;

    /*
     * skip '://'.
     */
    curi += 3;

    /*
     * skip host part
     */
    for ( ; *curi != '/' && *curi != '\0'; curi++)
      ;

  }
  else
  {
    /*
     * We are not a proxy, start with URI
     */
    curi = r->uri;
  }

  /*
   * If curi == ".../securid/auth"  then the new handler will be auth. handler,
   * ........................check" ............................ check ........
   */
  if (strchr (curi, '?'))
  {
    strlen_curi = strchr (curi, '?') - curi;
  }
  else
  {
    strlen_curi = strlen (curi);
  }
# ifdef SECURID_DEBUG
  if (strlen_curi >= SECURID_URL_AUTH_LEN)
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: translate: will strncmp %s and %s (n=%d)",
		   (long) getpid (),
		   curi + (strlen_curi - SECURID_URL_AUTH_LEN),
		   SECURID_URL_AUTH, SECURID_URL_AUTH_LEN);
  }
  else
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: translate: will not strncmp %s and %s (%d !>= %d)",
		   (long) getpid (),
		   curi, SECURID_URL_AUTH,
		   strlen_curi, SECURID_URL_AUTH_LEN);
  }
  if (strlen_curi >= SECURID_URL_CHECK_LEN)
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: translate: will strncmp %s and %s (n=%d)",
		   (long) getpid (),
		   curi + (strlen_curi - SECURID_URL_CHECK_LEN),
		   SECURID_URL_CHECK, SECURID_URL_CHECK_LEN);
  }
  else
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: translate: will not strncmp %s and %s (%d !>= %d)",
		   (long) getpid (),
		   curi, SECURID_URL_CHECK,
		   strlen_curi, SECURID_URL_CHECK_LEN);
  }
# endif
  if (*curi != '\0' && strlen_curi >= SECURID_URL_AUTH_LEN &&
      strncmp (curi + (strlen_curi - SECURID_URL_AUTH_LEN),
	       SECURID_URL_AUTH, SECURID_URL_AUTH_LEN) == 0)
  {
    newhandler = SECURID_HANDLER_AUTH;
  }
  else if (*curi != '\0' && strlen_curi >= SECURID_URL_CHECK_LEN &&
      strncmp (curi + (strlen_curi - SECURID_URL_CHECK_LEN),
	       SECURID_URL_CHECK, SECURID_URL_CHECK_LEN) == 0)
  {
    newhandler = SECURID_HANDLER_CHECK;
  }
  else
  {
    newhandler = NULL;
  }

  if (newhandler)
  {
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: translate: => %s (URI=%s, was %s)",
		   (long) getpid (), newhandler, curi, r->uri);
#   endif

    /*
     * Remove "scheme://host/" from URI and "redirect" to the new handler.
     */
    r->uri     = curi;
    r->handler = newhandler;
    return OK;
  }

  /*
   * No new handler...
   */
# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: translate: declined... (URI=%s)",
		 (long) getpid (), r->uri);
# endif
  return DECLINED;
}

/*
 ************************************************************************
 * Handler to detect securid-status handler
 ************************************************************************
 */
static int securid_mimetype (request_rec *r)
{
  /*
   * See notes for securid_translate...
   */
  char	*curi;			/* to parse r->uri			*/
  char	*newhandler;
  int	strlen_curi;		/* strlen (), skeeping '?...' if needed	*/

  /*
   * First check if we are a "standard" (not reverse) proxy
   */
  if (r->proxyreq == STD_PROXY)
  {
    /*
     * Ok, we are a proxy, skip 'scheme:' from URI
     */
    for (curi = r->uri; *curi != ':' && *curi != '\0'; curi++)
      ;

    /*
     * skip '://'.
     */
    curi += 3;

    /*
     * skip host part
     */
    for ( ; *curi != '/' && *curi != '\0'; curi++)
      ;

  }
  else
  {
    /*
     * We are not a proxy, start with URI
     */
    curi = r->uri;
  }

  /*
   * ........................status" ........................... status .......
   */
  if (strchr (curi, '?'))
  {
    strlen_curi = strchr (curi, '?') - curi;
  }
  else
  {
    strlen_curi = strlen (curi);
  }
# ifdef SECURID_DEBUG
  if (strlen_curi >= SECURID_URL_STATUS_LEN)
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: mimetype: will strncmp %s and %s (n=%d)",
		   (long) getpid (),
		   curi + (strlen_curi - SECURID_URL_STATUS_LEN),
		   SECURID_URL_STATUS, SECURID_URL_STATUS_LEN);
  }
  else
  {
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: mimetype: will not strncmp %s and %s (%d !>= %d)",
		   (long) getpid (),
		   curi, SECURID_URL_STATUS,
		   strlen_curi, SECURID_URL_STATUS_LEN);
  }
# endif
  if (*curi != '\0' && strlen_curi >= SECURID_URL_STATUS_LEN &&
      strncmp (curi + (strlen_curi - SECURID_URL_STATUS_LEN),
	       SECURID_URL_STATUS, SECURID_URL_STATUS_LEN) == 0)
  {
    newhandler = SECURID_HANDLER_STATUS;
  }
  else
  {
    newhandler = NULL;
  }

  if (newhandler)
  {
#   ifdef SECURID_DEBUG
    ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		   "#%ld: mimetype: => %s (URI=%s, was %s)",
		   (long) getpid (), newhandler, curi, r->uri);
#   endif

    /*
     * Remove "scheme://host/" from URI and "redirect" to the new handler.
     */
    r->uri     = curi;
    r->handler = newhandler;
    return OK;
  }

  /*
   * No new handler...
   */
# ifdef SECURID_DEBUG
  ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, r,
		 "#%ld: mimetype: declined... (URI=%s)",
		 (long) getpid (), r->uri);
# endif
  return DECLINED;
}

/*
 ************************************************************************
 * Config directives
 ************************************************************************
 */

/*
 * AuthSecurID_SockDir
 */
static const char *securid_cfg_proxy (cmd_parms *cmd, void *dummy, char *a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_proxy_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * We just store the filename (directive argument), "ServerRoot" will be
   * added later (see module init) if the path is not absolute.
   * Also note this directive has been used.
   */
  sconf->proxy = a1;
  sconf->directives |= SECURID_proxy_SET;

  /*
   * Return no error
   */
  return NULL;
}

static const char *securid_cfg_sockdir (cmd_parms *cmd, void *dummy, char *a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_sockdir_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * We just store the filename (directive argument), "ServerRoot" will be
   * added later (see module init) if the path is not absolute.
   * Also note this directive has been used.
   */
  sconf->sockdir = a1;
  sconf->directives |= SECURID_sockdir_SET;

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_Cache
 */
static const char *securid_cfg_cache (cmd_parms *cmd, void *dummy, char *a1,
                                      char *a2)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_cachefile_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * We just store the filename (directive argument), "ServerRoot" will be
   * added later (see module init).
   * Also note this directive has been used.
   */
  sconf->cachefile = a1;
  sconf->directives |= SECURID_cachefile_SET;

  /*
   * Check if a2 (2nd arg) == "noreset".
   */
  if (a2)
  {
    if (strcasecmp (a2, "noreset") == 0)
    {
      /*
       * Tell securid_init() to reuse existing cache file.
       */
      sconf->resetcache = 0;
    }
    else
    {
      /*
       * Return error message.
       */
      return "invalid optional argument: use \"noreset\" or nothing.";
    }
  }

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_TTL
 */
#define	SECURID_CFG_TTL_MAX	(LONG_MAX / 60L)
static const char *securid_cfg_ttl (cmd_parms *cmd, void *dummy,
                                    char *a1, char *a2)
{
  /*
   * String 2 Long error string.
   */
  char		*err;
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_cachettltime_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store TTL (specified in minutes with the directive), in seconds;
   * Also note this directive has been used.
   */
  errno = 0;
  sconf->cachettltime = strtol (a1, &err, 10);

  if (*err != '\0')
  {
    return "invalid value: cannot convert.";
  }
  if (errno ||
      sconf->cachettltime <= 0L ||
      sconf->cachettltime >= SECURID_CFG_TTL_MAX)
  {
    return ap_psprintf (cmd->pool, "invalid value: "
                        "TTL must be > 0 and < %ld.", SECURID_CFG_TTL_MAX);
  }

  sconf->cachettltime = sconf->cachettltime * 60L;
  sconf->directives |= SECURID_cachettltime_SET;

  /*
   * Check optional arg: "always_after" (default) or "if_not_used".
   */
  if (a2)
  {
    if (strncmp (a2, SECURID_TTL_ALWAYS_STR,
                 sizeof (SECURID_TTL_ALWAYS_STR)) == 0)
    {
      sconf->cachettltype = SECURID_TTL_ALWAYS;
    }
    else if (strncmp (a2, SECURID_TTL_UNUSED_STR,
                      sizeof (SECURID_TTL_UNUSED_STR)) == 0)
    {
      sconf->cachettltype = SECURID_TTL_UNUSED;
    }
    else
    {
      return "invalid optional argument: use \"" SECURID_TTL_ALWAYS_STR
             "\" or \"" SECURID_TTL_UNUSED_STR "\".";
    }
  }
  else
  {
    sconf->cachettltype = SECURID_TTL_ALWAYS;
  }

  /*
   * It's ok.
   */
  return NULL;
}

/*
 * AuthSecurID_MaxTTL
 */
#define	SECURID_CFG_MAXTTL_MAX	(LONG_MAX / 60L)
static const char *securid_cfg_maxttl (cmd_parms *cmd, void *dummy, char *a1)
{
  /*
   * String 2 Long error string.
   */
  char		*err;
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_cachettltype_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Check "AuthSecurID_TTL N always_after" is not used.
   */
  if (sconf->cachettltype == SECURID_TTL_ALWAYS)
  {
    return "\"AuthSecurID_MaxTTL\" is useless when not using "
           "\"AuthSecurID_TTL ... " SECURID_TTL_UNUSED_STR "\"";
  }

  /*
   * Store TTL (specified in minutes with the directive), in seconds;
   * Also note this directive has been used.
   */
  errno = 0;
  sconf->cachettlmax = strtol (a1, &err, 10);

  if (*err != '\0')
  {
    return "invalid value: cannot convert.";
  }
  if (errno ||
      sconf->cachettlmax <  0L ||
      sconf->cachettlmax >= SECURID_CFG_MAXTTL_MAX)
  {
    return ap_psprintf (cmd->pool, "invalid value: "
                        "max TTL must be >= 0 and < %ld.",
			SECURID_CFG_MAXTTL_MAX);
  }

  sconf->cachettlmax = sconf->cachettlmax * 60L;
  sconf->directives |= SECURID_cachettlmax_SET;

  /*
   * It's ok.
   */
  return NULL;
}

/*
 * AuthSecurID_VarAce
 */
static const char *securid_cfg_varace (cmd_parms *cmd, void *dummy, char *a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_var_ace_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store argument, which shoud be something like VAR_ACE=value.
   * Also note this directive has been used.
   */
  if (strncmp (a1, "VAR_ACE=", strlen ("VAR_ACE=")) == 0 &&
      strlen (a1) > strlen ("VAR_ACE="))
  {
    sconf->var_ace = a1;
    sconf->directives |= SECURID_var_ace_SET;
  }
  else
  {
    return "invalid value: format is \"VAR_ACE=directory\".";
  }

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_DomainCookie
 */
static const char *securid_cfg_domaincookie (cmd_parms *cmd, void *dummy,
                                             char *a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);
  /*
   * RE are used to check the directive value.
   */
# define	SECURID_RE_2DOM	"^(\\.[a-z0-9-]+)+\\.[a-z0-9-]+$"
# define	SECURID_RE_FLGS	REG_EXTENDED | REG_ICASE | REG_NOSUB
  regex_t	regexp;				/* compiled RE		*/
  size_t	dummy_nmatch = 0;			/* unused		*/
  regmatch_t	dummy_pmatch [1];		/* unused		*/

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_domaincookie_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store argument, which shoud be something like .toto.dom.
   * Also note this directive has been used.
   */
  if (regcomp (&regexp, SECURID_RE_2DOM, SECURID_RE_FLGS) != 0)
  {
    return ("internal error: could not compile SECURID_RE_2DOM.");
  }
  if (regexec (&regexp, a1, dummy_nmatch, dummy_pmatch, 0) == 0)
  {
    regfree (&regexp);
    sconf->domaincookie = a1;
    sconf->directives |= SECURID_domaincookie_SET;
  }
  else
  {
    regfree (&regexp);
    return "invalid value: format is \".xxx.yyy\".";
  }

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_SecureCookie
 */
static const char *securid_cfg_securecookie (cmd_parms *cmd, void *dummy,
                                             int a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_securecookie_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store argument flag (on | off)
   * Also note this directive has been used.
   */
  sconf->securecookie = a1;
  sconf->directives |= SECURID_securecookie_SET;

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_PathCookie
 */
static const char *securid_cfg_pathcookie (cmd_parms *cmd, void *dummy,
                                           char *a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_pathcookie_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store argument, which begin with a "/"...
   * Also note this directive has been used.
   */
  if (a1 [0] == '/')
  {
    sconf->pathcookie = a1;
    sconf->directives |= SECURID_pathcookie_SET;
  }
  else
  {
    return "invalid value: format is \"/...\".";
  }

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_FromAgent
 */
static const char *securid_cfg_fromagent (cmd_parms *cmd, void *dummy,
                                          const char *l)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_fromagent_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * # of word
   */
  sconf->fromagents.n = 0;

  /*
   * For each word of the list: "[from:]key[/mask]"
   */
  while (l [0])
  {
    /*
     * One word of the list l
     */
    char		*w = ap_getword_conf (cmd->pool, &l);
    /*
     * Save this word for error messages.
     */
    char		*a = ap_pstrdup (cmd->pool, w);
    /*
     * To scan inside the word (":" & "/").
     */
    char		*p = w;
    char		*s;
    /*
     * "from" ip addr & "from" mask.
     */
    unsigned long	from_net;
    unsigned long	from_msk;

#   ifdef SECURID_DEBUG
    ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, cmd->server,
		   "#%ld: cfg_fromagent: w = %s", (long) getpid (), w);
#   endif

    /*
     * To may keys?
     */
    if (sconf->fromagents.n == SECURID_FROMAGENT_MAX)
    {
      return ap_psprintf (cmd->pool, "stop!: "
			  "max #key reached (%d).", SECURID_FROMAGENT_MAX);
    }

    /*
     * "from": way be a.b.c. ==> a.b.c.0/24 for example.
     */
    from_net = 0;
    from_msk = 0;
    s = strchr (p, ':');
    if (s)
    {
      int	shift;
      char	*part = p;
      char	*t;
      int	octet;

      *s = '\0';
      p  = s + 1;

      /* parse components */
      shift = 24;
      while (*part)
      {
        t = part;
	if (!ap_isdigit (*t))
	{
	  return ap_psprintf (cmd->pool,
	                      "invalid ip address in \"from\" portion of `%s'.",
			      a);
	}
	while (ap_isdigit (*t))
	{
	  t++;
	}
	if (*t == '.')
	{
	  *t++ = 0;
	}
	else if (*t)
	{
	  return ap_psprintf (cmd->pool,
	                      "invalid ip address in \"from\" portion of `%s'.",
			      a);
	}
	if (shift < 0)
	{
	  return ap_psprintf (cmd->pool,
	                      "invalid ip address in \"from\" portion of `%s': "
			      "only 4 octets allowed.", a);
	}
	octet = atoi (part);
	if (octet < 0 || octet > 255)
	{
	  return ap_psprintf (cmd->pool,
	                      "invalid ip address in \"from\" portion of `%s': "
			      "each octet must be between 0 and 255 inclusive.",
			      a);
	}
	from_net |= octet << shift;
	from_msk |= 0xFFUL << shift;
	part = t;
	shift -= 8;
      }
    }
    sconf->fromagents.k[sconf->fromagents.n].from_net = ntohl (from_net);
    sconf->fromagents.k[sconf->fromagents.n].from_msk = ntohl (from_msk);

    /*
     * "key" & "mask"
     */
    s = strchr (p, '/');
    if (s)
    {
      *s = '\0';
      if (!securid_isip (s + 1) ||
          (sconf->fromagents.k[sconf->fromagents.n].key_msk =
	   ap_inet_addr (s + 1)) == INADDR_NONE)
      {
        return ap_psprintf (cmd->pool,
	                    "syntax error in \"mask\" portion of `%s'.", a);
      }
    }
    else
    {
      sconf->fromagents.k[sconf->fromagents.n].key_msk = 0;
    }
    sconf->fromagents.k[sconf->fromagents.n].key_val = p;

    /*
     * dispatch key: "null"
     */
    if (strcasecmp (p, "null") == 0)
    {
      sconf->fromagents.k[sconf->fromagents.n].type = SECURID_FROMAGENT_NULL;
    }
    /*
     * dispatch key: "client_ip"
     */
    else if (strcasecmp (p, "client_ip") == 0)
    {
      sconf->fromagents.k[sconf->fromagents.n].type = SECURID_FROMAGENT_IP;
    }
    else
    {
      sconf->fromagents.k[sconf->fromagents.n].type = SECURID_FROMAGENT_KEY;
    }

#   ifdef SECURID_DEBUG
    ap_log_error (APLOG_MARK, APLOG_NOERRNO | APLOG_DEBUG, cmd->server,
		   "#%ld: cfg_fromagent: k[%d].type = %s, "
		   "k[%d].from_net = %lx, k[%d].from_msk = %lx, "
		   "k[%d].key_val = %s, k[%d].key_msk = %lx",
		   (long) getpid (),
		   sconf->fromagents.n,
    sconf->fromagents.k[sconf->fromagents.n].type == SECURID_FROMAGENT_NULL
    ? "NULL" : (
    sconf->fromagents.k[sconf->fromagents.n].type == SECURID_FROMAGENT_IP
    ? "IP" : "KEY"),
                   sconf->fromagents.n,
		   sconf->fromagents.k[sconf->fromagents.n].from_net,
                   sconf->fromagents.n,
		   sconf->fromagents.k[sconf->fromagents.n].from_msk,
                   sconf->fromagents.n,
		   sconf->fromagents.k[sconf->fromagents.n].key_val,
                   sconf->fromagents.n,
		   sconf->fromagents.k[sconf->fromagents.n].key_msk);
#   endif

    /*
     * One more key.
     */
    sconf->fromagents.n++;
  }

  /*
   * Also note this directive has been used.
   */
  sconf->directives |= SECURID_fromagent_SET;

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_MaxAuthGet
 */
#define	SECURID_CFG_MAXAUTHGET_MAX	256
static const char *securid_cfg_maxauthget (cmd_parms *cmd, void *dummy,
                                           char *a1)
{
  /*
   * String 2 Long error string.
   */
  char		*err;
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_maxauthget_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store this number; check >= 0 && < max.
   * Also note this directive has been used.
   */
  errno = 0;
  sconf->maxauthget = strtol (a1, &err, 10);

  if (*err != '\0')
  {
    return "invalid value: cannot convert.";
  }
  if (errno ||
      sconf->maxauthget <  0 ||
      sconf->maxauthget >= SECURID_CFG_MAXAUTHGET_MAX)
  {
    return ap_psprintf (cmd->pool, "invalid value: "
                        "max authentication passcode requests "
			"(get) must be >= 0 and < %d.",
			SECURID_CFG_MAXAUTHGET_MAX);
  }

  sconf->directives |= SECURID_maxauthget_SET;

  /*
   * It's ok.
   */
  return NULL;
}

/*
 * AuthSecurID_MaxAuthPost
 */
#define	SECURID_CFG_MAXAUTHPOST_MAX	256
static const char *securid_cfg_maxauthpost (cmd_parms *cmd, void *dummy,
                                           char *a1)
{
  /*
   * String 2 Long error string.
   */
  char		*err;
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_maxauthpost_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Also note this directive has been used.
   * Store this number; check >= 0 && < MAX.
   */
  errno = 0;
  sconf->maxauthpost = strtol (a1, &err, 10);

  if (*err != '\0')
  {
    return "invalid value: cannot convert.";
  }
  if (errno ||
      sconf->maxauthpost <  0 ||
      sconf->maxauthpost >= SECURID_CFG_MAXAUTHPOST_MAX)
  {
    return ap_psprintf (cmd->pool, "invalid value: "
                        "max authentication passcode requests "
			"(post) must be >= 0 and < %d.",
			SECURID_CFG_MAXAUTHPOST_MAX);
  }

  sconf->directives |= SECURID_maxauthpost_SET;

  /*
   * It's ok.
   */
  return NULL;
}

/*
 * AuthSecurID_MaxCacheSize
 */
#define	SECURID_CFG_MAXCACHESIZE_MAX	INT_MAX
static const char *securid_cfg_maxcachesize (cmd_parms *cmd, void *dummy,
                                             char *a1)
{
  /*
   * String 2 Long error string.
   */
  char		*err;
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_maxcachesize_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store this number; check >= 0 && < max.
   * Also note this directive has been used.
   */
  errno = 0;
  sconf->maxcachesize = strtol (a1, &err, 10);

  if (*err != '\0')
  {
    return "invalid value: cannot convert.";
  }
  if (errno ||
      sconf->maxcachesize <  0 ||
      sconf->maxcachesize >= SECURID_CFG_MAXCACHESIZE_MAX)
  {
    return ap_psprintf (cmd->pool, "invalid value: "
                        "max cache size must be >= 0 and < %d.",
			SECURID_CFG_MAXCACHESIZE_MAX);
  }

  sconf->directives |= SECURID_maxcachesize_SET;

  /*
   * It's ok.
   */
  return NULL;
}

/*
 * AuthSecurID_UserAgent
 */
#define	SECURID_CFG_USERAGENT_MAX	(sizeof (webid.user_agent))
static const char *securid_cfg_useragent (cmd_parms *cmd, void *dummy, char *a1)
{
  /*
   * String 2 Long error string.
   */
  char		*err;
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);
  /*
   * Webid struct, just to get sizeof (user_agent)...
   */
  securid_webid	webid;

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_useragent_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store value; check >= 0 && < max.
   * Also note this directive has been used.
   */
  errno = 0;
  sconf->useragent = strtol (a1, &err, 10);

  if (*err != '\0')
  {
    return "invalid value: cannot convert.";
  }
  if (errno ||
      sconf->useragent <  0 ||
      sconf->useragent >= SECURID_CFG_USERAGENT_MAX)
  {
    return ap_psprintf (cmd->pool, "invalid value: "
			"User-Agent's length must be >= 0 and < %d.",
			SECURID_CFG_USERAGENT_MAX);
  }

  sconf->directives |= SECURID_useragent_SET;

  /*
   * It's ok.
   */
  return NULL;
}

/*
 * AuthSecurID_CustomDir
 */
static const char *securid_cfg_customdir (cmd_parms *cmd, void *dummy, char *a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_customdir_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * We just store the directory (directive argument), with "ServerRoot".
   * Also note this directive has been used.
   */
  sconf->customdir = ap_server_root_relative (cmd->pool, a1);
  sconf->directives |= SECURID_customdir_SET;

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_CustomType
 */
static const char *securid_cfg_customtype (cmd_parms *cmd, void *dummy,
                                           char *a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_customtype_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Check arg is "a sort of" Content-type (with a "/").
   * Also note this directive has been used.
   */
  if (strchr (a1, '/'))
  {
    sconf->customtype = a1;
    sconf->directives |= SECURID_customtype_SET;
  }
  else
  {
    return "invalid value: format is \"type/subtype\".";
  }

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_CustomSymLinks
 */
static const char *securid_cfg_customlink (cmd_parms *cmd, void *dummy,
                                           int a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_customlink_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store on/off.
   * Also note this directive has been used.
   */
  sconf->customlink = a1;
  sconf->directives |= SECURID_customlink_SET;

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_NoCache
 */
static const char *securid_cfg_nocache (cmd_parms *cmd, void *dummy, int a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_nocache_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store argument flag (on | off)
   * Also note this directive has been used.
   */
  sconf->nocache = a1;
  sconf->directives |= SECURID_nocache_SET;

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_HandleCookie
 */
static const char *securid_cfg_hdlcookie (cmd_parms *cmd, void *dummy, char *a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_hdlcookie_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store argument flag.
   * Also note this directive has been used.
   */
  sconf->hdlcookie = a1;
  sconf->directives |= SECURID_hdlcookie_SET;

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_WebidCookie
 */
static const char *securid_cfg_widcookie (cmd_parms *cmd, void *dummy, char *a1)
{
  /*
   * Server config
   */
  securid_sconf	*sconf =
    (securid_sconf *) ap_get_module_config (cmd->server->module_config,
                                            &securid_module);

  /*
   * Check this directive has not already been used.
   */
  if (sconf->directives & SECURID_widcookie_SET)
  {
    return "this directive has already been used.";
  }

  /*
   * Store argument flag.
   * Also note this directive has been used.
   */
  sconf->widcookie = a1;
  sconf->directives |= SECURID_widcookie_SET;

  /*
   * Return no error
   */
  return NULL;
}

/*
 * AuthSecurID_AceTTL: obsolete
 */
static const char *securid_cfg_acettltime (cmd_parms *cmd, void *dummy,
                                           char *a1)
{
  return "this directive is obsolete, please do not use it anymore.";
}

/*
 * Configuration directives
 */
command_rec securid_cmds [] =
{
  {
    "AuthSecurID_Proxy",
    securid_cfg_proxy, NULL, RSRC_CONF, TAKE1,
    "a path"
  },
  {
    "AuthSecurID_SockDir",
    securid_cfg_sockdir, NULL, RSRC_CONF, TAKE1,
    "a path"
  },
  {
    "AuthSecurID_Cache",
    securid_cfg_cache, NULL, RSRC_CONF, TAKE12,
    "a pathname, with an optional \"noreset\""
  },
  {
    "AuthSecurID_TTL",
    securid_cfg_ttl, NULL, RSRC_CONF, TAKE12,
    "a TTL (in minutes), with an optional \"" SECURID_TTL_ALWAYS_STR "\" or \""
    SECURID_TTL_UNUSED_STR "\""
  },
  {
    "AuthSecurID_MaxTTL",
    securid_cfg_maxttl, NULL, RSRC_CONF, TAKE1,
    "a TTL (in minutes)"
  },
  {
    "AuthSecurid_VarAce",
    securid_cfg_varace, NULL, RSRC_CONF, TAKE1,
    "VAR_ACE=directory"
  },
  {
    "AuthSecurID_Authoritative",
    ap_set_flag_slot, (void *) XtOffsetOf (securid_dconf, authoritative),
      OR_AUTHCFG, FLAG,
    "\"on\" or \"off\""
  },
  {
    "AuthSecurid_DomainCookie",
    securid_cfg_domaincookie, NULL, RSRC_CONF, TAKE1,
    "a domain name"
  },
  {
    "AuthSecurID_SecureCookie",
    securid_cfg_securecookie, NULL, RSRC_CONF, FLAG,
    "\"on\" or \"off\""
  },
  {
    "AuthSecurid_PathCookie",
    securid_cfg_pathcookie, NULL, RSRC_CONF, TAKE1,
    "a path name"
  },
  {
    "AuthSecurid_FromAgent",
    securid_cfg_fromagent, NULL, RSRC_CONF, RAW_ARGS,
    "a list of '[from:]key[/mask]'; see doc"
  },
  {
    "AuthSecurid_MaxAuthGet",
    securid_cfg_maxauthget, NULL, RSRC_CONF, TAKE1,
    "a number > 0, or 0 to bypass checks"
  },
  {
    "AuthSecurid_MaxAuthPost",
    securid_cfg_maxauthpost, NULL, RSRC_CONF, TAKE1,
    "a number > 0, or 0 to bypass checks"
  },
  {
    "AuthSecurid_MaxCacheSize",
    securid_cfg_maxcachesize, NULL, RSRC_CONF, TAKE1,
    "a number > 0, or 0 to bypass checks"
  },
  {
    "AuthSecurid_UserAgent",
    securid_cfg_useragent, NULL, RSRC_CONF, TAKE1,
    "a length for HTTP User-Agent header"
  },
  {
    "AuthSecurID_CustomDir",
    securid_cfg_customdir, NULL, RSRC_CONF, TAKE1,
    "a directory name"
  },
  {
    "AuthSecurID_CustomType",
    securid_cfg_customtype, NULL, RSRC_CONF, TAKE1,
    "a mime type"
  },
  {
    "AuthSecurID_CustomSymLinks",
    securid_cfg_customlink, NULL, RSRC_CONF, FLAG,
    "\"on\" or \"off\""
  },
  {
    "AuthSecurID_NoCache",
    securid_cfg_nocache, NULL, RSRC_CONF, FLAG,
    "\"on\" or \"off\""
  },
  {
    "AuthSecurID_HandleCookie",
    securid_cfg_hdlcookie, NULL, RSRC_CONF, TAKE1,
    "a custom AceHandle cookie name"
  },
  {
    "AuthSecurID_WebidCookie",
    securid_cfg_widcookie, NULL, RSRC_CONF, TAKE1,
    "a custom WebID cookie name"
  },
  {
    "AuthSecurID_AceTTL",
    securid_cfg_acettltime, NULL, RSRC_CONF, TAKE1,
    "obsolete; was a TTL (in minutes)"
  },
  {NULL}
};

/*
 ************************************************************************
 * How to create server and directory configs
 ************************************************************************
 */

/*
 * Defines for default values
 */
#define SECURID_var_ace_DEF		"VAR_ACE=/var/ace"
						/* $VAR_ACE/sdconf.rec	*/
#define SECURID_cachefile_DEF		"logs/securid_auth"
						/* ServerRoot with init	*/
#define SECURID_resetcache_DEF		1	/* don't reuse cache	*/
#define SECURID_cachettltime_DEF	30*60	/* 30 mn		*/
#define SECURID_cachettltype_DEF	SECURID_TTL_ALWAYS
						/* absolute TTL		*/
#define SECURID_cachettlmax_DEF		0	/* unlimited...		*/
#define SECURID_domaincookie_DEF	NULL	/* no domain= cookie	*/
#define SECURID_securecookie_DEF	1	/* secure=TRUE cookie	*/
#define SECURID_pathcookie_DEF		NULL	/* auto path= cookie	*/
#define	SECURID_fromagents_DEF_n	1	/* use remote IP	*/
#define	SECURID_fromagents_DEF_type	SECURID_FROMAGENT_IP
#define	SECURID_fromagents_DEF_from_net	0
#define	SECURID_fromagents_DEF_from_msk	0
#define	SECURID_fromagents_DEF_key_val	"client_ip"
#define	SECURID_fromagents_DEF_key_msk	0
#define SECURID_maxauthget_DEF		16	/* 16 req for auth form	*/
#define SECURID_maxauthpost_DEF		8	/* 8 req to ACE server	*/
#define SECURID_maxcachesize_DEF	1024	/* 1024 auth. users	*/
#define SECURID_useragent_DEF		32	/* check 1st 32 chars	*/
#define SECURID_customdir_DEF		NULL	/* no custom dir	*/
#define SECURID_customtype_DEF		"text/html"
						/* Content-type: ...	*/
#define SECURID_customlink_DEF		0	/* disable sym links	*/
#define SECURID_nocache_DEF		SECURID_NOCACHE_OFF
						/* URLs caching policy	*/
#define SECURID_hdlcookie_DEF		NULL	/* use def. cookie name	*/
#define SECURID_widcookie_DEF		NULL	/* idem	-^		*/
#define SECURID_ace_lang_DEF		"LANG=C"/* "en" error messages	*/
#define SECURID_sockdir_DEF             "logs/securid"
#define SECURID_proxy_DEF               "libexec/securid_proxy"

/*
 * Create server config.
 */
static void *securid_create_sconf (pool *p, server_rec *s)
{
  securid_sconf	*sconf;

  /*
   * We create a new config
   */
  sconf = (securid_sconf *) ap_pcalloc (p, sizeof (securid_sconf));

  /*
   * Init default values for config directives. We need to "strdup" variables
   * that will be used with "putenv" (unless it core dumps with some other
   * modules).
   */
  sconf->var_ace                  = ap_pstrdup (p, SECURID_var_ace_DEF);
  sconf->cachefile                = SECURID_cachefile_DEF;
  sconf->resetcache               = SECURID_resetcache_DEF;
  sconf->cachettltime             = SECURID_cachettltime_DEF;
  sconf->cachettltype             = SECURID_cachettltype_DEF;
  sconf->cachettlmax              = SECURID_cachettlmax_DEF;
  sconf->domaincookie             = SECURID_domaincookie_DEF;
  sconf->securecookie             = SECURID_securecookie_DEF;
  sconf->pathcookie               = SECURID_pathcookie_DEF;
  sconf->fromagents.n             = SECURID_fromagents_DEF_n;
  sconf->fromagents.k[0].type     = SECURID_fromagents_DEF_type;
  sconf->fromagents.k[0].from_net = SECURID_fromagents_DEF_from_net;
  sconf->fromagents.k[0].from_msk = SECURID_fromagents_DEF_from_msk;
  sconf->fromagents.k[0].key_val  = SECURID_fromagents_DEF_key_val;
  sconf->fromagents.k[0].key_msk  = SECURID_fromagents_DEF_key_msk;
  sconf->maxauthget               = SECURID_maxauthget_DEF;
  sconf->maxauthpost              = SECURID_maxauthpost_DEF;
  sconf->maxcachesize             = SECURID_maxcachesize_DEF;
  sconf->useragent                = SECURID_useragent_DEF;
  sconf->customdir                = SECURID_customdir_DEF;
  sconf->customtype               = SECURID_customtype_DEF;
  sconf->customlink               = SECURID_customlink_DEF;
  sconf->nocache                  = SECURID_nocache_DEF;
  sconf->hdlcookie                = SECURID_hdlcookie_DEF;
  sconf->widcookie                = SECURID_widcookie_DEF;
  sconf->sockdir                  = SECURID_sockdir_DEF;
  sconf->proxy                    = ap_pstrdup (p, SECURID_proxy_DEF);
  /*
   * Others default values.
   */
  sconf->directives    = 0;			/* no directive set	*/
  /*   ->lockfile      = will be set later (see server_init)		*/
  sconf->ace_lang      = ap_pstrdup (p, SECURID_ace_lang_DEF);
  /* no socket at the moment */
  sconf->sockfn        = NULL;
  sconf->proxy_pid     = -1;
  /*
   * And return this config
   */
  return (void *) sconf;
}

/*
 * Merge server configs.
 */
static void *securid_merge_sconf (pool *p, void *bconfv, void *aconfv)
{
  securid_sconf	*sconf =	/* resulting (merged) server config	*/
    (securid_sconf *) ap_pcalloc (p, sizeof (securid_sconf));
  securid_sconf	*bconf =	/* base (main server) config		*/
    (securid_sconf *) bconfv;
  securid_sconf	*aconf =	/* add (virtual server) config		*/
    (securid_sconf *) aconfv;

  /*
   * Merge value: "add (virtual server) config" *directives* overrides
   * "base (main server) config" (this is only for *directives*).
   */
  sconf->var_ace       = (aconf->directives & SECURID_var_ace_SET)
                         ?aconf->var_ace
			 :bconf->var_ace;
  sconf->cachefile     = (aconf->directives & SECURID_cachefile_SET)
                         ?aconf->cachefile
			 :bconf->cachefile;
  sconf->resetcache    = (aconf->directives & SECURID_cachefile_SET)
                         ?aconf->resetcache
			 :bconf->resetcache;
  sconf->cachettltime  = (aconf->directives & SECURID_cachettltime_SET)
                         ?aconf->cachettltime
			 :bconf->cachettltime;
  sconf->cachettltype  = (aconf->directives & SECURID_cachettltype_SET)
                         ?aconf->cachettltype
			 :bconf->cachettltype;
  sconf->cachettlmax   = (aconf->directives & SECURID_cachettlmax_SET)
                         ?aconf->cachettlmax
			 :bconf->cachettlmax;
  sconf->domaincookie  = (aconf->directives & SECURID_domaincookie_SET)
                         ?aconf->domaincookie
			 :bconf->domaincookie;
  sconf->securecookie  = (aconf->directives & SECURID_securecookie_SET)
                         ?aconf->securecookie
			 :bconf->securecookie;
  sconf->pathcookie    = (aconf->directives & SECURID_pathcookie_SET)
                         ?aconf->pathcookie
			 :bconf->pathcookie;
  memcpy (
  &(sconf->fromagents) , (aconf->directives & SECURID_fromagent_SET)
                         ?&(aconf->fromagents)
			 :&(bconf->fromagents), sizeof (sconf->fromagents));
  sconf->maxauthget    = (aconf->directives & SECURID_maxauthget_SET)
                         ?aconf->maxauthget
			 :bconf->maxauthget;
  sconf->maxauthpost   = (aconf->directives & SECURID_maxauthpost_SET)
                         ?aconf->maxauthpost
			 :bconf->maxauthpost;
  sconf->maxcachesize  = (aconf->directives & SECURID_maxcachesize_SET)
                         ?aconf->maxcachesize
			 :bconf->maxcachesize;
  sconf->useragent     = (aconf->directives & SECURID_useragent_SET)
                         ?aconf->useragent
			 :bconf->useragent;
  sconf->customdir     = (aconf->directives & SECURID_customdir_SET)
                         ?aconf->customdir
			 :bconf->customdir;
  sconf->customtype    = (aconf->directives & SECURID_customtype_SET)
                         ?aconf->customtype
			 :bconf->customtype;
  sconf->customlink    = (aconf->directives & SECURID_customlink_SET)
                         ?aconf->customlink
			 :bconf->customlink;
  sconf->nocache       = (aconf->directives & SECURID_nocache_SET)
                         ?aconf->nocache
			 :bconf->nocache;
  sconf->hdlcookie     = (aconf->directives & SECURID_hdlcookie_SET)
                         ?aconf->hdlcookie
			 :bconf->hdlcookie;
  sconf->widcookie     = (aconf->directives & SECURID_widcookie_SET)
                         ?aconf->widcookie
			 :bconf->widcookie;
  sconf->sockdir       = (aconf->directives & SECURID_sockdir_SET)
                         ?aconf->sockdir
			 :bconf->sockdir;
  sconf->proxy         = (aconf->directives & SECURID_proxy_SET)
                         ?aconf->proxy
			 :bconf->proxy;
  
  /*
   * Other members (not *directives*):
   *   ->directives	will be not be used any more (because we cannot
   *			include a virtual host into another virtual host...)
   *   ->lockfile	will be set later (see server_init)
   *   ->ace_lang	"force inherit" from Base (main) server
   */
  sconf->ace_lang = bconf->ace_lang;

  /*
   * And return this new (merged) config
   */
  return (void *) sconf;
}

static void *securid_create_dconf (pool *p, char *d)
{
  securid_dconf	*dconf;

  /*
   * We create a new config
   */
  dconf = (securid_dconf *) ap_pcalloc (p, sizeof (securid_dconf));

  /*
   * Init default values
   */
  dconf->authoritative = 1;	/* "keep the fortress secure by default"*/

  /*
   * And return this config
   */
  return (void *) dconf;
}

/*
 ************************************************************************
 * Module definition
 ************************************************************************
 */

static const handler_rec	securid_handlers [] =
{
  {SECURID_HANDLER_AUTH,	securid_handler_auth},
  {SECURID_HANDLER_CHECK,	securid_handler_check},
  {SECURID_HANDLER_STATUS,	securid_handler_status},
  {NULL}
};

module MODULE_VAR_EXPORT securid_module =
{
  STANDARD_MODULE_STUFF,
  securid_module_init,		/* module initializer			*/
  securid_create_dconf,		/* create per-dir    config structures	*/
  NULL,				/* merge  per-dir    config structures	*/
  securid_create_sconf,		/* create per-server config structures	*/
  securid_merge_sconf,		/* merge  per-server config structures	*/
  securid_cmds,			/* table of config file commands	*/
  securid_handlers,		/* [#8] MIME-typed-dispatched handlers	*/
  securid_translate,		/* [#1] URI to filename translation	*/
  securid_check_auth,		/* [#4] validate user id from request	*/
  securid_check_access,		/* [#5] check if the user is ok _here_	*/
  NULL,				/* [#3] check access by host address	*/
  securid_mimetype,		/* [#6] determine MIME type		*/
  NULL,				/* [#7] pre-run fixups			*/
  NULL,				/* [#9] log a transaction		*/
  NULL,				/* [#2] header parser			*/
  securid_child_init,		/* child_init				*/
  securid_child_exit,		/* child_exit				*/
  NULL				/* [#0] post read-request		*/
};
