/*=============================================================================
 *  Copyright 2002-2004 deny all - Asphanet S.A. (http://www.deny-all.com/)
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *=============================================================================
 *
 * NAME
 *      securid_client.c - Client functions to connect to a SecurID proxy
 *
 * AUTHORS
 *      Erwan Legrand <elegrand@deny-all.com>
 *
 * VERSION
 *      $Revision: 1.4 $
 *
 *=============================================================================
 */
static char const rcsid [] = "$Id: securid_client.c,v 1.4 2004/02/25 17:00:51 elegrand Exp $";

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>

#include "httpd.h"
#include "http_log.h"

#include "securid_client.h"
#include "securid_msg.h"

/*******************************************************************
 *
 * Function:
 * ---------
 * securid_client_connect
 * Creates a socket and connects to the SecurID proxy
 *
 * Parameters:
 * -----------
 * const securid_sconf *sconf    the server's configuration
 * const request_rec *r          the HTTP request being processed
 *
 * Return value:
 * -------------
 * Either a socket descriptor on success or -1 on faillure
 *
 ******************************************************************/

static int securid_client_connect (const securid_sconf *sconf,
				   const request_rec *r)
{
  int sock, res;
  struct sockaddr_un address;
  
  /* Create socket */
  sock = socket(AF_UNIX, SOCK_STREAM, 0);
  if (sock < 0)
    {
      ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
		     "SecurID: client could not create socket `%s'",
		     strerror(errno));
      return -1;
    }
#ifdef SECURID_DEBUG
  else
    {
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		    "SecurID: socket created by client %ld", (long) getpid());
    }
#endif

  /* Initialize address */
  address.sun_family = AF_UNIX;
  strncpy(address.sun_path, sconf->sockfn, sizeof(address.sun_path) - 1);
  address.sun_path[sizeof(address.sun_path) - 1] = '\0';

  /* Connect to SecurID proxy */
  res = connect(sock, (struct sockaddr *)&address, sizeof(address));
  if (res < 0)
    {
      ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
		    "SecurID: client could not connect to socket (%s) `%s'",
		    address.sun_path, strerror(errno));
      /*
       *  Connection failled: close socket!
       */
      res = close (sock);
      if (res < 0)
	{
	  ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
			 "SecurID: client could not close socket `%s'",
			 strerror(errno));
	}
      return -1;
    }
#ifdef SECURID_DEBUG
  else
    {
      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		    "SecurID: client with pid %ld connected to socket %s",
		    (long) getpid(), sconf->sockfn);
    }
#endif
  return sock;
}

/*******************************************************************
 *
 * Function:
 * ---------
 * securid_client_send_request
 * Sends request to the SecurID proxy. Must be called after a
 * successful call to securid_client_connect.
 *
 * Parameters:
 * -----------
 * int sd                        the socket descriptor returned
 *                               by securid_client_connect
 * const request_rec *r          the HTTP request being processed
 * const securid_request *req    the request to send to the proxy
 * securid_reply *rep            the reply sent by the proxy
 *
 * Return value:
 * -------------
 * Either a socket descriptor on success or -1 on faillure
 *
 ******************************************************************/

static int securid_client_send_request (int sd, /*socket descriptor */
					const request_rec *r,
					const securid_request *req,
					securid_reply *rep)
{
  ssize_t cnt;
  int ret;

  /* Send message to proxy */
  cnt = write (sd, req, sizeof (*req));
  if (cnt < 0)
    {
      ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
		     "SecurID: client could not write to socket");
    }
  else
    {
      if (cnt != sizeof (*req))
	{
	  ap_log_rerror (APLOG_MARK, APLOG_ERR | APLOG_NOERRNO, r,
			 "SecurID: client managed to send %d bytes while "
			 "length of request is %d", cnt, sizeof (*req));
	}
#ifdef SECURID_DEBUG
      else
	{
	  ap_log_rerror (APLOG_MARK, APLOG_DEBUG | APLOG_NOERRNO, r,
			 "SecurID: client sent %d bytes to proxy", cnt);

	}
#endif
    }

  /* Read reply */
  cnt = read (sd, rep, sizeof (*rep));
  if (cnt < 0)
    {
      ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
		     "SecurID: client could not read from socket");
      ret = SECURID_BAD_REQUEST;
    }
  else
    {
      if (cnt != sizeof (*rep))
	{
	  ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
			 "SecurID: client received %d bytes while "
			 "length of reply should be %d", cnt, sizeof (*rep));
	  ret = SECURID_BAD_REQUEST;
	}
      else
	{
#ifdef SECURID_DEBUG
	  ap_log_rerror (APLOG_MARK, APLOG_DEBUG | APLOG_NOERRNO, r,
			 "SecurID: client received %d bytes from proxy", cnt);
#endif
	  ret = rep->status;
	}
    }
  return ret;
}

static void webid_set_username (securid_webid *webid, const char *username)
{
      strncpy (webid->username, username, sizeof (webid->username) - 1);
      webid->username[sizeof (webid->username) - 1] = '\0';
}

static void webid_auth_ok(const securid_sconf *sconf,
			  securid_webid *webid,
			  const char *shell)
{
  /*
   * Update SecurID authentication cookie (webid2).
   */
  if (sconf->cachettltype != SECURID_TTL_ALWAYS)
    {
      webid->last_time  = time (NULL);
    }
  strncpy (webid->shell, shell, sizeof (webid->shell) - 1);
  webid->shell [sizeof (webid->shell) - 1] = '\0';
}

static void webid_set_pin_params(securid_webid *webid,
				 SD_PIN *pin_params)
{
  memcpy (&(webid->pin_params), pin_params, sizeof (SD_PIN));
  /* Clear system pin */
  memset (pin_params->System, 0, sizeof (pin_params->System));
}

int securid_call_check (const securid_sconf *sconf,
			const request_rec *r,
			securid_webid *wid,
			const char *username,
			char *passcode)
{
  securid_request req;
  securid_reply rep;
  int sock, ret;

  /* Connect to proxy */
  sock =  securid_client_connect (sconf, r);
  if (sock < 0)
    {
      ret = SECURID_BAD_REQUEST;
    }
  else
    {
      /* Build request message */
      req.cmd = securid_check;
      strncpy (req.bdy.req_check.username,
	       username,
	       sizeof(req.bdy.req_check.username) -1);
      req.bdy.req_check.username[sizeof(req.bdy.req_check.username) -1] = '\0';
      strncpy (req.bdy.req_check.passcode,
	       passcode,
	       sizeof(req.bdy.req_check.passcode) -1);
      req.bdy.req_check.username[sizeof(req.bdy.req_check.passcode) -1] = '\0';
      
      /* Send request */
      ret = securid_client_send_request (sock, r, &req, &rep);
    }
     
  /* Clear passcode */
  memset (passcode, 0, strlen(passcode));
  memset (req.bdy.req_check.passcode, 0, sizeof(req.bdy.req_check.passcode));

  switch (ret)
    {
    case ACM_OK:
      webid_set_username (wid, username);
      webid_auth_ok (sconf, wid, rep.bdy.auth_ok.shell);
      break;
    case ACM_NEXT_CODE_REQUIRED:
      webid_set_username (wid, username);
      /* Set handle so that authentication can continue */
      wid->sd = rep.bdy.next.sd;
      break;
    case ACM_NEW_PIN_REQUIRED:
      webid_set_username (wid, username);
      webid_set_pin_params (wid, &(rep.bdy.pin.pin_params));
      /* Set handle so that authentication can continue */
      wid->sd = rep.bdy.next.sd;
      break;
    default:
      /* Do nothing here */
      break;
    }
  return ret;
}

int securid_call_next (const securid_sconf *sconf,
		       const request_rec *r,
		       securid_webid *wid,
		       char *nextcode)
{
  securid_request req;
  securid_reply rep;
  int sock, ret;

  /* Connect to proxy */
  sock =  securid_client_connect (sconf, r);
  if (sock < 0)
    {
      ret = SECURID_BAD_REQUEST;
    }
  else
    {
      /* Build request message */
      req.cmd = securid_next;
#ifdef SECURID_DEBUG
      ap_log_rerror (APLOG_MARK, APLOG_DEBUG | APLOG_NOERRNO, r, "#%ld: ACE "
		    "library handle is %d", (long) getpid(), wid->sd);
#endif
      req.bdy.req_next.sd = wid->sd;
      strncpy (req.bdy.req_next.nextcode,
	       nextcode,
	       sizeof(req.bdy.req_next.nextcode) -1);
      req.bdy.req_next.nextcode[sizeof(req.bdy.req_next.nextcode) -1] = '\0';
      
      /* Send request */
      ret = securid_client_send_request (sock, r, &req, &rep);
    }
     
  /* Clear nextcode */
  memset (nextcode, 0, strlen(nextcode));
  memset (req.bdy.req_next.nextcode, 0, sizeof(req.bdy.req_next.nextcode));

  switch (ret)
    {
    case ACM_OK:
      webid_auth_ok (sconf, wid, rep.bdy.auth_ok.shell);
      break;
    case ACM_NEW_PIN_REQUIRED:
      webid_set_pin_params (wid, &(rep.bdy.pin.pin_params));
      break;
    default:
      /* Do nothing here */
      break;
    }
  return ret;
}

int securid_call_pin (const securid_sconf *sconf,
		      const request_rec *r,
		      securid_webid *wid,
		      char *newpin)
{
  securid_request req;
  securid_reply rep;
  int sock, ret;

  /* Connect to proxy */
  sock =  securid_client_connect (sconf, r);
  if (sock < 0)
    {
      ret = SECURID_BAD_REQUEST;
    }
  else
    {
      /* Build request message */
      req.cmd = securid_pin;
#ifdef SECURID_DEBUG
      ap_log_rerror (APLOG_MARK, APLOG_DEBUG | APLOG_NOERRNO, r, "#%ld: ACE "
		    "library handle is %d", (long) getpid(), wid->sd);
#endif
      req.bdy.req_pin.sd = wid->sd;
      strncpy (req.bdy.req_pin.newpin,
	       newpin,
	       sizeof(req.bdy.req_pin.newpin) -1);
      req.bdy.req_pin.newpin[sizeof(req.bdy.req_pin.newpin) -1] = '\0';
      
      /* Send request */
      ret = securid_client_send_request (sock, r, &req, &rep);
    }
     
  /* Clear newpin */
  memset (newpin, 0, strlen(newpin));
  memset (req.bdy.req_pin.newpin, 0, sizeof(req.bdy.req_pin.newpin));
  /* DON'T clear system pin in wid, so that we can display it! */
  /* See mod_securid.c: look for ACM_NEW_PIN_GENERATED */

  return ret;
}

