/*=============================================================================
 *  Copyright 2002-2004 deny all - Asphanet S.A. (http://www.deny-all.com/)
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *=============================================================================
 ************************************************************************
 *
 * NAME
 *      securid_proxy.c - Make requests to an ACE server on behalf of Apache.
 *
 * AUTHORS
 *      Erwan Legrand <elegrand@deny-all.com>
 *
 * VERSION
 *      $Revision: 1.13 $
 *
 *=============================================================================
 */
static char const rcsid [] = "$Id: securid_proxy.c,v 1.13 2004/02/25 17:00:51 elegrand Exp $";

/* standard includes */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <pthread.h>

/* SecurID includes */

#include <acexport.h>

/* local includes */

#include "securid_proxy_config.h"
#include "securid_proxy_signals.h"
#include "securid_msg.h"

typedef struct
{
  int sd; /* socket descriptor */
} thread_args;

extern char **environ;

pid_t pid;

static void proxy_reply_bad_request (const securid_request *req,
			      securid_reply *rep)
{
  rep->status = SECURID_BAD_REQUEST;
}

static void proxy_do_check (securid_request *req,
		     securid_reply *rep)
{
  SDI_HANDLE sd = SDI_HANDLE_NONE;

#ifdef SECURID_DEBUG
  fprintf (stderr, "SecurID: proxy_do_check entered\n");
#endif

  rep->status = SD_Init (&sd);
  if (rep->status != ACM_OK)
    {
#ifdef SECURID_DEBUG
      fputs ("SecurID: proxy_do_check failled to initialize connection with "
	     "ACE server\n", stderr);
#endif
      return;
    }
#ifdef SECURID_DEBUG
  fprintf (stderr, "SecurID proxy: proxy_do_check username is %s\n"
		   "SecurID proxy: proxy_do_check passcode is %s\n",
	   req->bdy.req_check.username, req->bdy.req_check.passcode);
#endif

  if (AceSetTimeout(sd, SECURID_PROXY_TIMEOUT, NULL) != ACE_SUCCESS)
  {
    fputs ("SecurID proxy: proxy_do_check failled to set timeout!\n", stderr);
  }

  /* Call SD_Lock here! */
  rep->status = SD_Lock(sd, req->bdy.req_check.username);
  if (rep->status != ACM_OK)
  {
#ifdef SECURID_DEBUG
    fprintf (stderr, "SecurID: proxy_do_check lock failed, status %d!\n",
             rep->status);
#endif
  }
  else
  {
    /* Actually perform the authentication */
    rep->status = SD_Check (sd,
			    req->bdy.req_check.passcode,
			    req->bdy.req_check.username);
  }

  /* Clear passcode */
  memset (req->bdy.req_check.passcode, 0, sizeof(req->bdy.req_check.passcode));

  switch (rep->status)
    {
    case ACM_OK:
      AceGetShell (sd, rep->bdy.auth_ok.shell);
      SD_Close (sd);
      break;
    case ACM_NEXT_CODE_REQUIRED:
      rep->bdy.next.sd = sd;
      break; /* Don't close! */
    case ACM_NEW_PIN_REQUIRED:
      rep->bdy.pin.sd = sd;
      AceGetPinParams (sd, &rep->bdy.pin.pin_params);
      break; /* Don't close! */
    default:
      SD_Close (sd);
    }
#ifdef SECURID_DEBUG
  fprintf (stderr, "SecurID: proxy_do_check ending, status %d\n",
	   rep->status);
#endif
}

static void proxy_do_next (securid_request *req, securid_reply *rep)
{
  SDI_HANDLE sd = req->bdy.req_next.sd;

  rep->status = SD_Next (sd,
			 req->bdy.req_next.nextcode);

  /* Clear nextcode */
  memset (req->bdy.req_next.nextcode, 0, sizeof (req->bdy.req_next.nextcode));

  switch (rep->status)
    {
    case ACM_OK:
      AceGetShell (sd, rep->bdy.auth_ok.shell);
      SD_Close (sd);
      break;
    case ACM_NEW_PIN_REQUIRED:
      rep->bdy.pin.sd = sd;
      AceGetPinParams (sd, &rep->bdy.pin.pin_params);
      break; /* Don't close! */
    default:
      SD_Close (sd);
    }
}

static void proxy_do_pin (securid_request *req, securid_reply *rep)
{
  SDI_HANDLE sd = req->bdy.req_pin.sd;

  rep->status = SD_Pin (sd,
			req->bdy.req_pin.newpin);

  /* Clear nextcode */
  memset (req->bdy.req_pin.newpin, 0, sizeof (req->bdy.req_pin.newpin));

  switch (rep->status)
    {
    case ACM_NEW_PIN_ACCEPTED:
      SD_Close (sd);
      break;
    default:
      /* Don't close! */
      break;
    }
}

static void *proxy_read_request (void *params)
{
  /*
   *  Processing of a new connection. This is started as
   *  new thread by main().
   */
  thread_args *args = params;
  securid_request req;
  securid_reply rep;
  ssize_t cnt;
  int res;

  cnt = read(args->sd, &req, sizeof(req));
  if (cnt < 0)
    {
      perror ("SecurID proxy: could not read from socket");
    }
  if (cnt != sizeof(req))
    {
      fprintf (stderr, "SecurID proxy: received %d byte while request size"
	       " should be %d\n", cnt, sizeof(rep));
      proxy_reply_bad_request (&req, &rep);
    }
  else
    {
#ifdef SECURID_DEBUG
	  fprintf (stderr, "SecurID proxy: received %d bytes\n", cnt);
#endif
      /*
       *  Request size OK, check type
       */
      switch (req.cmd)
	{
	case securid_check:
#ifdef SECURID_DEBUG
	  fprintf (stderr, "SecurID proxy: received a CHECK request\n");
#endif
	  proxy_do_check (&req, &rep);
	  break;
	case securid_next:
#ifdef SECURID_DEBUG
	  fprintf (stderr, "SecurID proxy: received a NEXT request\n");
#endif
	  proxy_do_next (&req, &rep);
	  break;
	case securid_pin:
#ifdef SECURID_DEBUG
	  fprintf (stderr, "SecurID proxy: received a PIN request\n");
#endif
	  proxy_do_pin (&req, &rep);
	  break;
	default:
	  proxy_reply_bad_request (&req, &rep);
	} /* switch */
    } /* else */
  cnt = write(args->sd, &rep, sizeof(rep));
  if (cnt < 0)
    {
      perror ("SecurID proxy: could not write to socket");
    }
  if (cnt != sizeof(rep))
    {
      fprintf (stderr, "SecurID proxy: sent %d bytes while reply size"
	       " should be %d\n", cnt, sizeof(rep));
    }

 /*
  *  Clean up prior to exiting
  */
  res = close (args->sd);
  if (res < 0)
    {
      perror ("SecurID proxy: could not close socket");
    }
  free (args);

  /* End of thread */
  return NULL;
}

static void proxy_check_varace (void)
{
  char *var_ace;

  var_ace = getenv ("VAR_ACE");
  if (var_ace)
    {
      fprintf (stderr, "SecurID proxy: %ld VAR_ACE set\n  %s\n", (long) pid,
	       var_ace);
    }
  else
    {
      fprintf (stderr, "SecurID proxy: %ld VAR_ACE not set\n", (long) pid);
    }
}

static void proxy_init_ace (void)
{
  SD_BOOL res;

  res = AceInitialize ();
  if (res == SD_FALSE)
    {
      fprintf (stderr, "SecurID proxy: could not initialize ACE library\n");
      exit (EXIT_FAILURE);
    }
#ifdef SECURID_DEBUG
  else
    {  
      fprintf (stderr, "SecurID proxy: ACE library initialized successfully\n");
    }
#endif
}

static int proxy_make_sock (char *filename)
{
  int sock;
  struct sockaddr_un address;

  sock = socket (AF_UNIX, SOCK_STREAM, 0);
  if (sock < 0)
    {
      perror ("SecurID proxy: could not create socket");
      return sock;
    }
#ifdef SECURID_DEBUG
  else
    {
      fprintf (stderr, "#%ld: proxy created socket with descriptor %d\n",
	       (long) pid, sock);
    }
#endif

  /* Initialize socket address */
  address.sun_family = AF_UNIX;
  strncpy(address.sun_path, filename, sizeof (address.sun_path));
  address.sun_path[sizeof (address.sun_path) - 1] = '\0';

  /* Try to unlink file in case it has already been created */
  unlink(filename);
  /* Bind socket to file */
  if (bind(sock, (struct sockaddr *)&address, sizeof (address)) < 0)
    {
      perror ("SecurID proxy: could not bind socket");
      fprintf (stderr, "SecurID proxy: socket path is `%s'\n",
	       address.sun_path);
      close (sock);
      return -1;
    }
#ifdef SECURID_DEBUG
  else
    {
      fprintf (stderr, "#%ld: proxy socket bound to %s\n",
	       (long) pid, filename);
    }
#endif

  /*
   *  Listen on our brand new socket
   */
  if (listen(sock, PROXY_QUEUE_LENGTH) < 0)
    {
      perror ("SecurID: proxy socket listen failled");
      close (sock);
      return -1;
    }
#ifdef SECURID_DEBUG
  else
    {
      fprintf (stderr, "#%ld: proxy socket now listening\n", (long) pid);
    }
#endif

  return sock;
}

int main (int argc, char **argv, char **envp)
{
  int sock, res;
  struct sockaddr_un address;
  socklen_t addrlen;
  thread_args *args;
  pthread_attr_t attrs;
  pthread_t thread;

  /*
   *  First, get our pid. We will use it for logging.
   */
  pid = getpid ();
  
  /*
   *  Then, check that we were called properly.
   */
  if ((argc != 2) || (argv[1][0] != '/'))
    {
      fprintf (stderr, "#%ld: USAGE -> %s [absolute path for socket]\n",
	       (long) pid, argv[0]);
      exit (EXIT_FAILURE);
    }

  /*
   * Also check that VAR_ACE has been set in our environment
   */
  proxy_check_varace ();

  /*
   * Block signals and start the signal handling thread
   * before we start any new thread!
   */
  proxy_init_signal_handler (argv[1]);

  /*
   * Initialize ACE client library. This does start new threads, hence it
   * MUST be called after proxy_init_signal_handler()! 
   */
  proxy_init_ace ();

  /*
   *  Create socket
   */
  sock = proxy_make_sock (argv[1]);
  if (sock < 0)
  {
    exit (EXIT_FAILURE);
  }

  /*
   *  Initialize POSIX thread attributes.
   */
  res = pthread_attr_init (&attrs);
  if (res)
    {
      perror ("SecurID proxy: could not initialize POSIX thread attributes");
    }
  pthread_attr_setdetachstate (&attrs, PTHREAD_CREATE_DETACHED);
  if (res)
    {
      perror ("SecurID proxy: could not set POSIX thread attributes");
    }
  
  /*
   *  Place the server in an infinite loop, waiting
   *  on connection requests to come from clients.
   *  In practice, there would need to be a clean
   *  way to terminate this process, but for now it
   *  will simply stay resident until terminated by
   *  the parent process or the super-user.
   */

  while (1) {
    /* Allocate memory for arguments passed to thread */
    args = malloc (sizeof(thread_args));
    if (args == NULL)
      {
	/* Out of memory! */
	fprintf (stderr, "SecurID: proxy pid %ld malloc failled!\n",
		 (long) pid);
	return ENOMEM;
      }
    addrlen = sizeof(struct sockaddr_un);
    args->sd = accept(sock, (struct sockaddr*)&address, &addrlen);
    if (args->sd < 0)
    {
      perror ("SecurID proxy: could not accept() connection on socket");
      return errno;
    }
  
    /*
     *  Create a new server thread to handle the connection.
     *  The main thread does no further processing -- it loops
     *  back to wait for another connection request.
     */
    res = pthread_create (&thread, &attrs, proxy_read_request, args);
    if (res)
      {
	/* free allocated memory */
	close (args->sd);
	free (args);
	perror ("SecurID proxy: could not create new POSIX thread");
      }
  }  /* while (1) */

  /* We should never get there. We are leaving. Remove socket file. */
  fputs ("SecurID proxy: end of infinite loop!??\n", stderr);
  unlink (argv[1]);

  return EXIT_FAILURE;
}  /* main () */

