/*=============================================================================
 *  Copyright 2002-2004 deny all - Asphanet S.A. (http://www.deny-all.com/)
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *=============================================================================
 *
 * NAME
 *      securid_proxy_signals.c - signal handling for the SecurID proxy
 *      program.
 *
 * AUTHORS
 *      Erwan Legrand <elegrand@deny-all.com>
 *
 * VERSION
 *      $Revision: 1.3 $
 *
 *=============================================================================
 */
static char const rcsid [] = "$Id: securid_proxy_signals.c,v 1.3 2004/02/25 17:00:51 elegrand Exp $";

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <pthread.h>
#include <signal.h>

/* SecurID includes */
#include <acexport.h>

/* Local includes */
#include "securid_proxy_config.h"
#include "securid_proxy_signals.h"

static void proxy_make_signal_set (sigset_t *set)
{
  sigemptyset (set);
  
  sigaddset (set, SIGHUP);
  sigaddset (set, SIGTERM);
  sigaddset (set, SIGINT);
  sigaddset (set, SIGUSR1);
}

static void *proxy_signal_handler (void *socket_filename)
{
	sigset_t set;
	int sig;
	char *signame;

	/* Make set of signals we wait for. */
	proxy_make_signal_set (&set);

	while (!sig)
	{
		/* Wait for signal... */
		sigwait (&set, &sig);
	}

	/* On BSD we could use psignal()... */
	switch (sig)
	{
		case SIGHUP:
			signame = "SIGHUP";
			break;
		case SIGINT:
			signame = "SIGINT";
			break;
		case SIGTERM:
			signame = "SIGTERM";
			break;
		case SIGUSR1:
			signame = "SIGUSR1";
			break;
		default:
			signame = "";
			break;
	}

	fprintf (stderr, "SecurID proxy: Signal caught - %s(%d) - exiting.\n",
		 signame, sig);
	/* Should be useless... */
	fflush (stderr);

	/* A signal was received! Shutdown cleanly. */
	AceShutdown (NULL);

	/* Remove Unix socket file. */
	unlink((char *)socket_filename);

	/* Exit! */
	exit (EXIT_SUCCESS);
}

static void proxy_block_signals (void)
{
  sigset_t set;

  proxy_make_signal_set (&set);

  if (pthread_sigmask (SIG_BLOCK, &set, NULL))
  {
    perror ("SecurID proxy: blocking signals failled");
    exit (EXIT_FAILURE);
  }
}
  
void proxy_init_signal_handler (const char *filename)
{
  /* Set signal handler up. */
  size_t len = strlen (filename) + 1;
  char *socket_filename;
  pthread_attr_t attrs;
  pthread_t thread;

  socket_filename = malloc (len);
  if (!socket_filename)
  {
    /* Out of memory! */
    fputs ("SecurID proxy: malloc failled\n", stderr);
    exit (EXIT_FAILURE);
  }
  strcpy (socket_filename, filename);

  /* Block signals before we start any new thread. */
  proxy_block_signals ();

  /*
   *  Initialize POSIX thread attributes.
   */
  if (pthread_attr_init (&attrs))
  {
    perror ("SecurID proxy: could not initialize POSIX threadattributes");
    exit (EXIT_FAILURE);
  }
  if (pthread_attr_setdetachstate (&attrs, PTHREAD_CREATE_DETACHED))
  {
    perror ("SecurID proxy: could not set POSIX thread attributes");
    exit (EXIT_FAILURE);
  }
  if (pthread_create (&thread, &attrs, proxy_signal_handler, socket_filename))
  {
    perror ("SecurID proxy: could not create POSIX thread");
    exit (EXIT_FAILURE);
  }
}

