/* sslclnt.cc */
/* Copyright (c) 2000-2003 Secure Bridge Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Secure Bridge
 *    Inc."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Secure Bridge
 *    Inc."
 *
 * THIS SOFTWARE IS PROVIDED BY THE SECURE BRIDGE INC. ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SECURE BRIDGE INC.
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/* Multi-threaded stress test tool for SSL based servers */

#include "sslclnt.h"

/* Global variables */
SSL_CTX *ctx = NULL;
struct timeb start, finish; 
double dt = 0;
BIO *logFile = NULL;

unsigned long BUFSIZE = 0;
unsigned long REQSIZE = 0;
unsigned long ntimes = 0;
int vfdepth = 0 ; /* max. verify depth */
bool bUseSSL = true ;
bool bJustConnect = false; 
bool bCacheSessionID = false;
char *pServerAddress = NULL;
int gPort = 0;
pthread_mutex_t mutex;
long nThreadsDone = 0;
int gRetries = 1; /* Connects are to be retried gRetries times */
long gMinRandomDelay = 0;
long gMaxRandomDelay = 0;
int gLogLevel = DEFAULT_LOG_LEVEL;
char *pScriptFile = NULL;
FILE* fpScript = NULL;
int gIterationsOfClient = 1;
char *pLogFile = NULL;
scripts gScript; /* The structure which contains scripts */
//Some variables for implemeting bandwidth throttling

//bandwidth requested by user
double bandwidth = 0;

int buf_size ;

//requested buffer size
int requestedBufSize = 0;

int
main(int argc, char* argv[])
{
	int nRetCode = 0;
	int index = 0;	
	char c; 

	int vfy = 0; /* server verification on */
	SSL_METHOD *meth ;

	char *cipher_list = DEFAULT_CIPHER_LIST;
	const char *CLIENT_CERT = DEFAULT_CERTIFICATE_FILE;
	const char *CAfile = DEFAULT_CA_CERTIFICATE_FILE;
	//const char *CApath = DEFAULT_CA_PATH;

	BIO *log = NULL; /* Log file handle */

	BIO *bioout, *bio_err;

	pthread_t *pThreadIDs = NULL;
	int NumThreads = 0;

	gScript.numScripts = 0;
	gScript.scriptBuf = NULL;

	/* register SIGPIPE handler */
	if(signal(SIGPIPE, sig_pipe) == SIG_ERR)
		DEBUG_MSG1("Can't catch SIGPIPE");

	/* Parse the command line arguments */
	if(argc < 2) {
		usage();
		return -1;
	}

	while((c = getopt(argc, argv, "c:d:e:f:hi:jl:n:p:a:s:t:vub:r")) != EOF) {
		switch(c){
		case 'r':
			bCacheSessionID = true;
			break;

		case 'd': /* Random delay */
			gMinRandomDelay = atoi(strtok(optarg,":"));
			gMaxRandomDelay = atoi(strtok(NULL,":"));
			break;
		
		case 'e':
			gLogLevel = atoi(optarg);
			break;

		case 'f': /* Script file */
			pScriptFile = optarg;
			break;

		case 'h': /* Help */
			showHelp();
			exit(1);

		case 'i': /* Iterations of this client */
			gIterationsOfClient = atoi(optarg);
			break;

		case 'j':
			bJustConnect = true;
			break;

		case 'l': /* Log file path */
			pLogFile = optarg;
			break;

		case 'n': /* Number of clients to be run */
			NumThreads = atoi(optarg);
			break;

		case 'p': /* Port Number */
			gPort = atoi(optarg);
			break;

		case 'a': /* Connects to be retried */
			gRetries = atoi(optarg);
			break;
		case 's': /* Server name */
		   	pServerAddress = optarg;
			break;

		case 't': /* Bandwidth throttling */
			bandwidth = atof(optarg);
                        bandwidth *= 1024; //kbps
                        break;
		
		case 'v':
			vfy = 1;
			break;

		case 'u':
			bUseSSL = false;
			break;
 		case 'b':
			requestedBufSize = atoi(optarg) * 1024;
      break;
		case 'c':
			cipher_list = optarg; 
			DEBUG_MSG2("Cipher used:",cipher_list);
			break;
		case '?': DEBUG_MSG2("Unrecognized option", optopt);
			usage();
			exit(0);
		}
	}

	/* Check if we have necessary arguments ready */
	if(pServerAddress == NULL)
		pServerAddress = DEFAULT_SERVER_ADDRESS;

	if(gPort == 0)
		gPort = DEFAULT_SERVER_PORT;

	if(bJustConnect == false) {
		if(pScriptFile == NULL) 
			pScriptFile = DEFAULT_SCRIPT_FILE;
	
		if((fpScript = fopen(pScriptFile,"rb"))	== NULL) {
			DEBUG_MSG2("Error opening script file", pScriptFile);
			return -1;
		}
		else
		{
			/* Read all the scripts into scripts data structure */
			if(ReadScriptFile(fpScript, &gScript) == -1)
			{ /* Error in reading scripts */
				DEBUG_MSG1("Problem in reading script file");
				return -1;
			}
		}
	}

	if(pLogFile == NULL)
		pLogFile = DEFAULT_LOG_FILE;
	
	if(NumThreads == 0) {
		NumThreads = MIN_CLIENTS;
	}
	else {
		NumThreads = (NumThreads > MAX_CLIENTS? MAX_CLIENTS: NumThreads);
	}
	
	bioout = BIO_new_fp(stdout,BIO_NOCLOSE);
	bio_err = BIO_new_fp(stderr, BIO_NOCLOSE);

	if(bUseSSL == true) {
		/* Do SSL setup */
		SSL_load_error_strings() ;
		if(!SSL_library_init()) {
			DEBUG_MSG1("Failure to initialize SSL library");
			exit(2);
		}
		MultiThreadSetup();
		
		if ((meth = SSLv3_client_method()) == NULL)
			return -1;

		if ((ctx = SSL_CTX_new(meth)) == NULL)
			return -1 ;
	
		if (cipher_list)
			SSL_CTX_set_cipher_list(ctx, cipher_list) ;
		
		if (vfy == 1) { 
			DEBUG_MSG1("Verification ON") ;
			SSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, SSLVerifyCallback) ;
		}

		if (SSL_CTX_use_certificate_file(ctx, CLIENT_CERT, 
			SSL_FILETYPE_PEM)<=0) {
			DEBUG_MSG1("invalid cert file") ;
			return -1 ;
		}
		if (SSL_CTX_use_PrivateKey_file(ctx, CLIENT_CERT, 
				SSL_FILETYPE_PEM)<=0) {
			DEBUG_MSG1("invalid pkey file") ;
			return -1 ;
		}
		
		if (!SSL_CTX_check_private_key(ctx)) {
			DEBUG_MSG1("private key mismatch") ;
			return -1 ;
		}

		if (!SSL_CTX_load_verify_locations(ctx, CAfile, NULL)  ||
			!SSL_CTX_set_default_verify_paths(ctx)) {
			DEBUG_MSG1("invalid CA file/path") ;
			return -1 ;
		}

		SSL_CTX_set_client_CA_list(ctx, SSL_load_client_CA_file(CAfile)) ;
		
		// SSL_CTX_set_info_callback(ctx, (void (*)(void))(SSLInfoCallback));
	}

	/* Create the threads and give them the log file to be used*/
	DEBUG_MSG3("Activating ",NumThreads, " client(s)");
	pThreadIDs = (pthread_t *) malloc(NumThreads * sizeof(pthread_t));

	log = BIO_new_file(pLogFile, "w+");

	if (pThreadIDs != NULL) {
		void * (*client_func) (void *) = (void *(*)(void *))(do_clconnect);

		/* Start timing */
		ftime(&start);
		for(index = 0;  index <NumThreads; index++) {
			nThreadsDone++;
				
			if(pthread_create(&pThreadIDs[index], NULL, 
					client_func, log) == -1) {
				DEBUG_MSG1("pthread_create error");
				exit(2);
			}
		}
		/* wait for all the threads to terminate */
		for(index = 0 ; index <NumThreads; index++)
			pthread_join(pThreadIDs[index], NULL);

		DEBUG_MSG1("All the clients terminated....");
		/* Record the total time taken */
		ftime(&finish);
	}
	else { 
		DEBUG_MSG1("Thread IDs could not be created: malloc failed");
		return -1;
	}
	dt = MILLTIME(finish, start) ;
	BIO_printf(log,"%3.5f", dt) ;

	/* Do cleanup */
	if(log){
		BIO_flush(log);
		BIO_free(log);
	}

	if(bUseSSL) {
		if (ctx != NULL) 
			SSL_CTX_free(ctx) ;
		MultiThreadCleanup();
		EVP_cleanup();
		ERR_remove_state(0);
		ERR_free_strings();
	}
	if(pThreadIDs)
		free(pThreadIDs);

	return nRetCode;
}

/* SIGPIPE handler */
static void sig_pipe(int signo)
{
	if(signo == SIGPIPE)
		DEBUG_MSG1("Received SIGPIPE");
	return;
}

/* Asynchronous connect to the server 
 * Return values: 0 - success, -1 failure 
 */
int DoAysncConnect(int& sock, int Port, char* ServerIP, BIO *log)
{
	sockaddr_in ServerAddress ;
	bool bConnectInProgress = false;
	bool bConnected = false;

	int flags,n;
	fd_set rset, wset;
	struct linger lingerinfo = {1, 0} ;

	pthread_t self = pthread_self();

	/* Socket initialization */
	sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP) ;
	if ( sock == -1) 
	{
		BIO_printf(log,"(%d)INVALSOCK\n",self) ;
		DEBUG_MSG2("Could not open socket : ",strerror(errno));
		return -1 ;
	}

	/* Fill in the server address details */
	memset(&ServerAddress, 0, sizeof(ServerAddress));
	ServerAddress.sin_family = AF_INET;
	ServerAddress.sin_port = htons(Port);

	if((ServerAddress.sin_addr.s_addr =  inet_addr(ServerIP)) ==INADDR_NONE){
		DEBUG_MSG1("Invalid Server IP address");
		return -1;
	}

	/* Set the socket to non-blocking */
	flags = fcntl(sock, F_GETFL, 0);
	fcntl(sock, F_SETFL, flags | O_NONBLOCK);

	/* Set the LINGER option so that connections are terminated immediately */
	setsockopt(sock, SOL_SOCKET, SO_LINGER, (char *)&lingerinfo,
				sizeof(lingerinfo));
	BIO_printf(log,"(%d)Connecting to %s on %d with socket %d \n",self, 
						ServerIP, Port, sock);

	/* Connects are retried in case of failure for specified number 
	 * of times before giving up.
	 */
	for(int i = 0; i < gRetries; i++) {
		if ((n = connect(sock,(sockaddr *)&ServerAddress, 
						sizeof(ServerAddress))) == -1){
			PrintSocketError(log, errno);
			if(errno == EINPROGRESS)
				bConnectInProgress = true;
			else return -1;
		}
		if(n == 0){
			bConnected = true;
			BIO_printf(log,"(%d)CONNECTED \n",self);
			fcntl(sock, F_SETFL,flags);//restore file status flags
			return 0;
		}
		if(bConnectInProgress == true){
			// BIO_printf(log,"(%d)EINPROGRESS error\n",self); 

			FD_ZERO(&rset);
			FD_ZERO(&wset);
			FD_SET(sock, &rset);
			FD_SET(sock, &wset);

			if((n = select(sock +1, &rset, &wset, NULL,NULL)) == -1){
				DEBUG_MSG2("SELECT error ",strerror(errno));
				break;
			} 

			/* Check if connection failed */
			if(FD_ISSET(sock, &rset)) { 
				if(errno == ECONNREFUSED){
					BIO_printf(log,"(%d)CONNECT_FAIL\n",self);
					DEBUG_MSG3("(",self,")CONNECT_FAIL");
				}
				/* Sleep for some time before retrying */
				BIO_printf(log,"(%d)CONNECT_FAIL\n",self);

				//RandomDelay(gMinRandomDelay,gMaxRandomDelay);
				continue;
				// return -1;
			}

			/* So, we must have got connected?? */ 
			if(FD_ISSET(sock, &wset)) { 
				BIO_printf(log,"(%d)CONNECTED \n",self);
				fcntl(sock, F_SETFL,flags); //restore file status flags
				return 0;

			}
		}
	}
	/* Make the socket blocking again */
	fcntl(sock, F_SETFL,flags); //restore file status flags
	/*
	*if(bConnected)
	*	return 0 ; 
	*else return -1;
	*/
	return -1;
}

int do_clconnect(void * lpvParam)
{
	int i , threadRetVal;
	int sock ;
	BIO * log = NULL;
	SSL* pSSL = NULL;
	SSL_SESSION *pSession = NULL;
	BIO *wbio = NULL, *rbio = NULL ; 
	int counter, len;
	char *pszRequest = NULL; /* Request to be fetched */
	int nNextRequest = 0;
        timeb prevReadTime,timeNow;

	pthread_t self = pthread_self();
	char *recvbuff = NULL;
        if(!requestedBufSize ) {
            if(!bandwidth)   
                buf_size = 64*1024;
            else
                buf_size = 2*1024;
        }
        else
            buf_size = requestedBufSize;
        //DEBUG_MSG3("Using Buffer of ", buf_size," bytes"); 
	recvbuff = (char *) malloc((sizeof(char)) * buf_size );

	if(recvbuff == NULL) {
		DEBUG_MSG1("Could not allocate memory for recvbuff");
		ERR_remove_state(0);
		pthread_exit((void* )&threadRetVal);
		return -1;
	}
	

	log = (BIO*) lpvParam;
	if (log == NULL) {
		DEBUG_MSG1("Out null in ");
		ERR_remove_state(0);
		pthread_exit((void* )&threadRetVal);
	}

	if(bUseSSL == true) {
		pSSL = SSL_new(ctx) ;
		if (pSSL == NULL) {
			BIO_printf(log, "(%d)sslmemfail\n",self) ;
			free (recvbuff);
			pthread_exit((void *)&threadRetVal);
		}
		SSL_set_app_data(pSSL, log);
	}
	for(counter = 0; counter < gIterationsOfClient; counter++) {
		/* Try fetching all the requests in the script file */
		nNextRequest = 0;
		
		bool bDoneItOnce = false;
		if(bCacheSessionID) 
		{
			pSession = NULL;
		}

		while(((pszRequest = getRequest(&gScript, nNextRequest++)) != NULL)
				|| ((bJustConnect == true) && (bDoneItOnce == false))) {
			/* Establish the connection with the server */
			if (DoAysncConnect(sock, gPort, pServerAddress,log) == -1) {
				DEBUG_MSG1("Failure in connecting....");
				/* Wait for some time before trying next script item */
				close(sock);
				//RandomDelay(gMinRandomDelay, gMaxRandomDelay);
				continue;
			}

			/* Connection established, now do I/O */
			if (bUseSSL == true) {
				wbio = BIO_new_socket(sock, BIO_NOCLOSE) ;
				rbio = wbio;
				if ( wbio == NULL || rbio == NULL) {
					BIO_printf(log, "(%d)BIOMEM_FAIL\n",self) ;
					close(sock);
					//RandomDelay(gMinRandomDelay, gMaxRandomDelay);
					continue;
				}
				/* Various call backs for more information */
				/*
		 		 * BIO_set_callback_arg(rbio, log);
		 		 * BIO_set_callback_arg(wbio, log);
		 		 * BIO_set_callback(rbio, bio_dump_cb);
		 		 * BIO_set_callback(wbio, bio_dump_cb);
				 */
				SSL_set_bio(pSSL, rbio, wbio) ;	
				if(bCacheSessionID) 
				{
					SSL_set_connect_state(pSSL) ;
				}
				SSL_set_session(pSSL,pSession);
				i = SSL_connect(pSSL) ;
				if (i <= 0) {
					BIO_printf(log, "(%d)SSL_CONN_FAIL\n",self);
					SSL_clear(pSSL);
					close(sock);
					//RandomDelay(gMinRandomDelay, gMaxRandomDelay);
					continue;
				}

				BIO_printf(log,"(%d)SSL_CONN_ESTABLISHED\n",self);
				
				if(bCacheSessionID) 
				{
					pSession = SSL_get_session(pSSL);
				}
				if(bJustConnect == true) {
					bDoneItOnce = true;
					SSL_clear(pSSL);
					close(sock);
					//RandomDelay(gMinRandomDelay, gMaxRandomDelay);
					continue;
				}
				/* Write the request */
				if(SSL_write(pSSL, pszRequest, 
							strlen(pszRequest)+1) <= 0) {
					DEBUG_MSG2("Error in requesting ",pszRequest);
					SSL_clear(pSSL);
					close(sock);
					//RandomDelay(gMinRandomDelay, gMaxRandomDelay);
					continue;
				}

				int nBytesRead = 0;
				int iRet = 0;
                                long totalDelay = 0; 
				/* Get the requested data now */
				while(1){ 
					ftime(&prevReadTime);
					iRet = SSL_read(pSSL, recvbuff, buf_size);
					if(iRet <= 0) {
						int iRet1 = SSL_get_error(pSSL,iRet);
						// PrintSSLError(iRet1);
						if(iRet1 == SSL_ERROR_WANT_READ)
						{
							DEBUG_MSG1("Want Read!!");
							continue;
						}
						/* Graceful termination */
						BIO_printf(log, "(%ld)SSL_CONN_FINISHED\n",self );
            BIO_printf(log, "(%ld)TOTAL_DELAY for request %d is %ld\n",
                       self,nNextRequest-1,totalDelay); 
						break;
					}
					len -= iRet;
					nBytesRead += iRet;
                                        
          if(bandwidth) {
            ftime(&timeNow);
            //calculate difference in time between 2 reads
            totalDelay += putDelay(prevReadTime,timeNow,iRet);
            //memcpy((void *)&prevReadTime, (void *)&timeNow, 
                    //sizeof(struct timeb));
          }    
				}
				DEBUG_MSG5("(",self,") Read ",nBytesRead, " bytes");
			}
			else { /* Plain sockets */

				BIO_printf(log,"(%ld)Retrieving %s \n",self, pszRequest);
				if(bJustConnect == true) {
					bDoneItOnce = true;
					close(sock);
					//RandomDelay(gMinRandomDelay, gMaxRandomDelay);
					continue;
				}
				if( send(sock, pszRequest,strlen(pszRequest), 0) == -1){
					DEBUG_MSG2("Failure in sending the request ", pszRequest);
					close(sock);
					//RandomDelay(gMinRandomDelay, gMaxRandomDelay);
					continue;
				}
				int iRet = 0;
				int nBytesRead = 0;
				int totalDelay = 0;
				while(1) {
					ftime(&prevReadTime);
					iRet = recv(sock, recvbuff, buf_size, 0);
					if(iRet == 0) {
						/* Graceful termination */
						BIO_printf(log,"(%ld)HTTP_CONN_FINISHED\n",self);
            BIO_printf(log, "(%ld)TOTAL_DELAY for request %d is %ld\n",
            							self,nNextRequest-1,totalDelay); 
						break;
					}
					if(iRet == -1) {
						DEBUG_MSG1("Socket Error ");
						if(errno == EAGAIN) {
							DEBUG_MSG3("(",self,")EAGAIN");
							continue;
						}
						break;
					}
					nBytesRead += iRet;
          if(bandwidth) {
          	ftime(&timeNow);
            totalDelay += putDelay(prevReadTime,timeNow,iRet);
          }
				}
				DEBUG_MSG5("(",self,")Read ",nBytesRead, " bytes");
			}

			if(sock) {
				//DEBUG_MSG3("(",self," Closing the connection");
				close(sock);
			}
			else DEBUG_MSG1("Invalid sock for close ");

			/* Clear the SSL structure before reuse */
			if (bUseSSL == true) {
				if(SSL_clear(pSSL) <= 0) {
					DEBUG_MSG1("Error in clearing SSL structure ");
					break;
				}
			}
			//RandomDelay(gMinRandomDelay, gMaxRandomDelay);
		}
	}
  DEBUG_MSG3("(",self,") Client finished");
	if (bUseSSL == true) {
		if (pSSL != NULL) {
			SSL_shutdown(pSSL);
			SSL_free(pSSL);
		}
	}
	/*
	if(sock)
		close(sock);
	*/
	ERR_remove_state(0);
	pthread_exit(&threadRetVal);
}

/* could do additional server authentication over here...*/
int SSLVerifyCallback(int ok, X509_STORE_CTX * ctx)
{
	/* We are not doing any verification stuff 
	 * We could just print the Certificate Info here
	 * but I have disabled those DEBUG_MSG's
	 */
	char buf[256] ;


	// DEBUG_MSG2("subject= ",buf);
	X509_NAME_oneline(X509_get_issuer_name(ctx->current_cert),buf,256);
	// DEBUG_MSG2("issuer= ",buf);
	ok = 1 ;
	return ok ;
}

long BIODumpCallback(BIO *bio, int cmd, const char *argp, 
					int argi, long argl, long ret)
{
	BIO *log;

	pthread_t self = pthread_self();

	log=(BIO *)BIO_get_callback_arg(bio);
	if (log == NULL) return(ret);

	if (cmd == (BIO_CB_READ|BIO_CB_RETURN))
		{
		BIO_printf(log,"(%d)read from %08X [%08lX] (%d bytes => %ld (0x%X))\n",
						self, bio,argp,argi,ret,ret);
		BIO_dump(log,argp,(int)ret);
		return(ret);
		}
	else if (cmd == (BIO_CB_WRITE|BIO_CB_RETURN))
		{
		BIO_printf(log,"(%d)write to %08X [%08lX] (%d bytes => %ld (0x%X))\n",
						self, bio,argp,argi,ret,ret);
		BIO_dump(log,argp,(int)ret);
		}
	return(ret);
}

void SSLInfoCallback(SSL *s, int where, int ret)
{
	char *str;
	int w;
	BIO *log = NULL;
	pthread_t self = pthread_self();
	
	w=where& ~SSL_ST_MASK;

	log = (BIO *)SSL_get_app_data(s);

	if (w & SSL_ST_CONNECT) str="SSL_connect";
	else if (w & SSL_ST_ACCEPT) str="SSL_accept";
	else str="undefined";

	if (where & SSL_CB_LOOP)
		{
		BIO_printf(log,"(%d)%s:%s\n",self,str,SSL_state_string_long(s));
		}
	else if (where & SSL_CB_ALERT)
		{
		str=(where & SSL_CB_READ)?"read":"write";
		BIO_printf(log,"(%d)SSL3 alert %s:%s:%s\n",
			self,
			str,
			SSL_alert_type_string_long(ret),
			SSL_alert_desc_string_long(ret));
		}
	else if (where & SSL_CB_EXIT)
		{
		if (ret == 0)
			BIO_printf(log,"(%d)%s:failed in %s\n",self,
				str,SSL_state_string_long(s));
		else if (ret < 0)
			{
			BIO_printf(log,"(%d)%s:error in %s\n",self,
				str,SSL_state_string_long(s));
			}
		}
}

/* Implementation of sleep using select() 
 * Returns 0  - on normal timeout
 *         -1 - on Error
 */
int
select_sleep(long millsec)
{
	struct timeval tv;
	tv.tv_sec = 0;
	tv.tv_usec = millsec*1000;
	//DEBUG_MSG3("sleeping for ",millsec, " millisec");
	return select(0, NULL, NULL, NULL,&tv);
}

int 
RandomDelay(long min, long max)
{
	long delay;

	if((max > 0) && (max > min)) {
		delay = (long int) min + ((max - min) * rand() / (RAND_MAX + 1.0));
		return select_sleep(delay); 
	}
	return -1; /* Error in min and max values */
}

/* Usage for the client */
void
usage(void)
{
	cout<<"Usage: client [options]... -f SCRIPTFILE ...\n";
	cout<<"Try -h option for more information\n";
}

void
showHelp(void)
{

	cout<<"SSL Stress Tool, "<< MAJORVERSION <<"."<< MINORVERSION <<
            " - (c) Secure Bridge Inc \n";
  cout <<"Usage: client [options]... -f SCRIPTFILE ...\n";
	cout <<"A client for stress testing both plain HTTP as well\n"
		"as SSL based HTTP web servers\n";
	cout <<"nInformation about the options is given below: \n"
		"-a retries \tNo. of times connects are retried before giving up\n"
        "-b buffer_size\tI/O buffer size in KB\n"
        "-c cipher_suite cipher suite to use\n"
		"-d MIN:MAX \tRandom delay in milliseconds between successive \n"
        "\t\trequests with min and max in milliseconds\n"
		"-e loglevel	Logging level : Default -None, 1 - data, 2 - SSL\n"
		"-f script_file	file containing URLs to be fetched\n"
		"-h \t\tPrints this help\n"
		"-i iterations	number of iterations per client\n"
		"-j \t\tjust connects without any data transfer\n"
		"-l log_file \tfile in which logs are to be dumped\n"
		"-n numClients	Number of concurrent clients to be activated\n"
		"-p port \tPort number of the server\n"
		"-r \t\tcache the session ID for one set of requests\n"
		"-s IPAddress \tIP address of the server\n"
		"-t bandwidth \tBandwidth throttling in KBPS\n"
		"-v \t\tturn the client verification ON\n"
		"-u \t\tPlain HTTP connection, SSL is not used\n"; 
	cout <<"\nReport bugs to <sbridge_pune@pspl.co.in>.\n";
}

void
printVersion()
{
	DEBUG_MSG4("client -", MAJORVERSION,".", MINORVERSION);
}

void PrintSocketError(BIO *log, int errval)
{
	pthread_t self = pthread_self();
	switch(errval) {
	case ECONNREFUSED: BIO_printf(log,"(%d)ECONNREFUSED\n",self); 
		break;
	case EFAULT: BIO_printf(log,"(%d)EFAULT\n",self); 
		break;
	case EBADF: BIO_printf(log,"(%d)EBADF\n",self); 
		break;
	case ENOTSOCK: BIO_printf(log,"(%d)ENOTSOCK\n",self); 
		break;
	case EISCONN: BIO_printf(log,"(%d)EISCONN\n",self); 
		break;
	case ETIMEDOUT: BIO_printf(log,"(%d)ETIMEDOUT\n",self); 
		break;
	case EADDRINUSE: BIO_printf(log,"(%d)EADDRINUSE\n",self); 
		break;
	case EINPROGRESS: // Non-blocking socket
		BIO_printf(log,"(%d)EINPROGRESS\n",self); 
		break;
	case EALREADY: BIO_printf(log,"(%d)EALREADY\n",self); 
		break;
	case ENETUNREACH: BIO_printf(log,"(%d)ENETUNREACH\n",self); 
		break;
	case EAFNOSUPPORT: BIO_printf(log,"(%d)EAFNOSUPPORT\n",self); 
		break;
	case EACCES: BIO_printf(log,"(%d)EACCES\n",self); 
		break;
	case EBADMSG: BIO_printf(log,"(%d)EBADMSG\n",self); 
		break;
	case EBUSY: BIO_printf(log,"(%d)EBUSY\n",self); 
		break;
	case ECHILD: BIO_printf(log,"(%d)ECHILD\n",self); 
		break;
	case EIO: BIO_printf(log,"(%d)EACCES\n",self); 
		break;
	case EINTR: BIO_printf(log,"(%d)EINTR\n",self);
		break;
	default: BIO_printf(log,"(%d)OTHER\n",self);
				break;
	}
}

int
ReadScriptFile(FILE *fp, scripts *http)
{
	int scriptLen = 0;
	char *fileBuf = NULL;
	#ifdef DEBUG
	pthread_t self = pthread_self();
  #endif

	http->numScripts = 0;
	http->scriptBuf = NULL;

	if(fp == NULL) {
		DEBUG_MSG3("(",self,")(ReadScriptFile): Trying to open NULL file pointer");
		return -1;
	}

	if(fseek(fp, 0, SEEK_END) == 0)
	{
		scriptLen = ftell(fp);

		if(scriptLen > 0)
		{
			/* Copy the entire file into a buffer */
			fileBuf = (char* ) malloc(scriptLen + 1);
			if(fileBuf == NULL)
			{
				DEBUG_MSG1("(ReadScriptFile):Error in (m)allocating buffer");
				return -1;
			}
			rewind(fp);
			int ret = fread(fileBuf, 1, scriptLen, fp);
			if(ret != scriptLen)
			{
				DEBUG_MSG1("Error reading script file");
				fclose(fp);
				free(fileBuf);
				return -1;
			}

			/* Determine number of scripts in the buffer */
			char *charPtr = fileBuf;

			for(int j = 0; j <scriptLen; j++)
			{
				if(*charPtr == '\n')
				http->numScripts++;
				charPtr++;
			}
		}
		else
		{
			DEBUG_MSG1("File contains no data");
			if(fp)
			fclose(fp);
			return -1;
		}
		/* Done with the script file */
		if(fp)
			fclose(fp);
	}

	http->scriptBuf = (char **) malloc(http->numScripts * sizeof(char*));
	for(int i = 0; i <http->numScripts; i++)
	{
		http->scriptBuf[i] = NULL;
	}

	/* Now we have the script file in fileBuf
	* Parse the requests and store it in the array
	*/
	char *scriptItem = strtok(fileBuf,"\n");
	
	int size = strlen(scriptItem);
	int validScriptItems = 0;
	if(size > 0)
	{
		http->scriptBuf[0] = (char *) malloc( (sizeof(char) * size) + 1);
		strcpy(http->scriptBuf[0], scriptItem);
		strcat(http->scriptBuf[0],"\n");
		validScriptItems++;
	}


	int index = 1;
	while((scriptItem = strtok(NULL, "\n")) != NULL )
	{
		size = strlen(scriptItem);
		if(size > 0)
		{
			http->scriptBuf[index] = (char* ) malloc((sizeof(char) * size) + 1);
			strcpy(http->scriptBuf[index], scriptItem);
			strcat(http->scriptBuf[index],"\n");
			index++;
			validScriptItems++;
		}
		else DEBUG_MSG1("Zero sized string");
	}
	http->numScripts = validScriptItems;
	free(fileBuf);
	#ifdef DEBUG
	cout<<"The URLs to be fetched are:\n";
	for(int i = 0; i <http->numScripts; i++)
	{
		cout<<i+1<<"> "<<http->scriptBuf[i];
	}
	#endif

	return 0; /* Success */
}

char* getRequest(scripts *http, int RequestNum)
{
	if(http->scriptBuf == NULL)
		return NULL;
	if(RequestNum < http->numScripts){
		return http->scriptBuf[RequestNum];
	}
	return NULL;
}

void PrintSSLError(int Err)
{
	switch(Err) {

	case SSL_ERROR_NONE:
		DEBUG_MSG1("SSL_ERROR_NONE");
		break;

	case SSL_ERROR_WANT_READ:
		DEBUG_MSG1("SSL_ERROR_WANT_READ");
		break;

	case SSL_ERROR_WANT_WRITE:
		DEBUG_MSG1("SSL_ERROR_WANT_WRITE");
		break;

	case SSL_ERROR_WANT_X509_LOOKUP:
		DEBUG_MSG1("SSL_ERROR_WANT_X509_LOOKUP");
		break;

	case SSL_ERROR_SYSCALL:
		DEBUG_MSG1("SSL_ERROR_SYSCALL");
		break;

	
	case SSL_ERROR_SSL:
		DEBUG_MSG1("SSL_ERROR_SSL");
		break;

	case SSL_ERROR_ZERO_RETURN:
		DEBUG_MSG1("SSL_ERROR_ZERO_RETURN");
		break;
	
	
	default:
		DEBUG_MSG1("Unhandled SSL error");
		break;
	}

}

long putDelay(timeb prevReadTime, timeb timeNow, int iRet) {
    long prevTime = ((prevReadTime.time) * 1000) + prevReadTime.millitm; 
    long nowTime  = ((timeNow.time) * 1000) + timeNow.millitm;
    long diff = nowTime - prevTime;
    DEBUG_MSG2("Bytes in current read =",iRet);
    DEBUG_MSG3("Diff is ",diff, " msec");
    //Find time to wait
    double delay = (iRet / bandwidth) * 1000;
    DEBUG_MSG2("Delay is ",delay);
    //Now see if elapsed time is greater or equal to delay time
    delay -= diff;
    if(delay) { //if sleep time has not elapsed
        select_sleep((long)delay);
        return (long)delay;
    } 
    return 0;
}

void
MultiThreadSetup(void)
{
	for(int i=0; i < CRYPTO_NUM_LOCKS; i++)
	{
		pthread_mutex_init(&cryptoLocks[i], NULL);
	}
	
	CRYPTO_set_locking_callback((void(*)(int, int, const char*, int))
										LockingCallback);
}

void
MultiThreadCleanup(void)
{
	CRYPTO_set_locking_callback(NULL);
}

static void
LockingCallback(int mode, int type, char *file, int line)
{
	if(mode & CRYPTO_LOCK)
	{
		pthread_mutex_lock(&cryptoLocks[type]);
	}
	else
	{
		pthread_mutex_unlock(&cryptoLocks[type]);
	}
}
 
