/*
        
        File:			AtheroJackCard.cpp
        Program:		AtheroJack
	Author:			Michael Rossberg
				mick@binaervarianz.de
	Description:		KisMAC is a wireless stumbler for MacOS X.
                
        This file is part of KisMAC.

        Parts of this file are based on Reyk Flters vt_ar5k driver.

    KisMAC is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    KisMAC is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KisMAC; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define KERNEL 1

#include <IOKit/IOWorkLoop.h>
#include <IOKit/IOLib.h>
#include <libkern/OSByteOrder.h>
#include "AtheroJackCard.h"
#include "AtheroJackLog.h"
#include "AtheroJackDriver.h"

#define roundup(n,b) (((n-1)|(b-1))+1)

AtheroJackCard::
AtheroJackCard(void* ioBase, void* klBase, IOService* parent, int alignSize): _ioBase(ioBase),
        			     _interrupts(0),
                                     _cardPresent(1),
                                     _isEnabled(false),
                                     _parent(parent) {    
    _alignSize = alignSize;
    
    _init();
    /*__resetChip(true);
    for(int i = 1; i<15; i++)
    _setChannel_5213(i);
    
    _setChannel_5213(3);
    _enable();
    */
    
    _workLoop = _parent->getWorkLoop();
    if (!_workLoop) {
        WLLogCrit("Failed to create workloop.\n");
        return;
    }
    
    _timedSendSource = IOTimerEventSource::timerEventSource(_parent, &AtheroJackCard::_myTimeoutHandler);
    if (!_timedSendSource) {
        WLLogCrit("Failed to create timer event source.\n");
        return;
    }

    if (_workLoop->addEventSource(_timedSendSource) != kIOReturnSuccess) {
        WLLogCrit("Failed to register timer event source.\n");
        return;
    }
}

AtheroJackCard::
~AtheroJackCard()
{
    if (_timedSendSource) {
        _timedSendSource->cancelTimeout();
        if (_workLoop) _workLoop->removeEventSource(_timedSendSource);
	_timedSendSource->release();
    }
    if (_cardPresent && _isEnabled) _disable();
}

#pragma mark -

IOReturn AtheroJackCard::getChannel(UInt16* channel) {    
    *channel = _channel;
    return kIOReturnSuccess;
}

IOReturn AtheroJackCard::setChannel(UInt16 channel) {
    /*if (_setChannel_5213(channel) != kIOReturnSuccess) {
        WLLogCrit("Error: could not set channel to: %u", channel);
        return kIOReturnError;
    }*/
    
    _channel = channel;
    return kIOReturnSuccess;
}

IOReturn AtheroJackCard::startCapture(IODataQueue* dq, UInt16 channel) {
    _packetQueue = dq;
    _channel = channel;
        
    if (_enable() != kIOReturnSuccess) {
        WLLogErr("Couldn't enable card\n");
        return kIOReturnError;
    }

    return kIOReturnSuccess;
}

IOReturn AtheroJackCard::stopCapture() {
    return _disable();
}

IOReturn AtheroJackCard::sendFrame(UInt8* data, UInt32 repeat) {
    //TODO  to implement
    return kIOReturnError;
}

IOReturn AtheroJackCard::stopSendingFrames() {
    if (_timedSendSource) _timedSendSource->cancelTimeout();
    _failures = 99;
    return kIOReturnSuccess;
}

IOReturn AtheroJackCard::cardGone() {
    _cardPresent = 0;
    return stopSendingFrames();
}


#pragma mark -

IOReturn AtheroJackCard::__rxBufInit(struct ar5213_buf *buf) {
    UInt32 rxBufSize;
    struct ar5213_desc *desc;

    //rxBufSize =  roundup(IEEE80211_MAX_LEN, _alignSize);
        
    if (buf->rxBuf==NULL) {
        buf->rxBuf = (void*)IOMallocContiguous(rxBufSize, PAGE_SIZE, 0);
        if (!buf->rxBuf) {
            WLLogCrit("Alloc Rx mbuf pointers failed!");
            return kIOReturnError;
        }
        bzero(buf->rxBuf, rxBufSize);	// clear out all the _rxBuf pointers

        buf->rxBufMemDesc = IOMemoryDescriptor::withAddress((vm_address_t)buf->rxBuf, rxBufSize, kIODirectionOutIn, kernel_task);
        if (!buf->rxBufMemDesc) {
            WLLogCrit("Could not get IOMemoryDescriptor for buffer!");
            return kIOReturnError;
        }
        
        buf->rxBufMemDesc->prepare(kIODirectionOutIn);
        
        buf->rxBufPhy = buf->rxBufMemDesc->getPhysicalAddress();
        if (!buf->rxBufPhy) {
            WLLogCrit("Bad Rx Buf!");
            return kIOReturnError;
        }
    }
  
    desc = buf->rxDesc;
    desc->ds_link = 0;
    desc->ds_data = buf->rxBufPhy;
    //__setupRxDesc(desc, rxBufSize, AR5K_RXDESC_INTREQ);
    
    return kIOReturnSuccess;
}


#pragma mark -

IOReturn AtheroJackCard::_setupRxRing() {
    UInt32 rxQueueSize;
    ar5213_desc *desc;
    
    WLEnter();
    /*
    //allocate the descriptors
    rxQueueSize = roundup(sizeof(struct ar5213_desc) * (RX_QUEUE_SIZE + 1), _alignSize);
    
    _rxFirstDesc = (ar5213_desc*)IOMallocContiguous(rxQueueSize, PAGE_SIZE, 0);
    if (!_rxFirstDesc) {
        WLLogCrit("Alloc Rx Desc failed!");
        return kIOReturnError;
    }
    bzero(_rxFirstDesc, rxQueueSize);	// clear out all the _rxBuf pointers

    _rxQueueMemDesc = IOMemoryDescriptor::withAddress((vm_address_t)_rxFirstDesc, rxQueueSize, kIODirectionOutIn, kernel_task);
    if (!_rxQueueMemDesc) {
        WLLogCrit("Could not get IOMemoryDescriptor for Desc Queue!");
        return kIOReturnError;
    }
    
    _rxQueueMemDesc->prepare(kIODirectionOutIn);
                                
    _rxQueuePhy = _rxQueueMemDesc->getPhysicalAddress();
    if (!_rxQueuePhy) {
        WLLogCrit("Bad Rx Ring!");
        return kIOReturnError;
    }
    
    desc = _rxFirstDesc;
    for (int i=0; i < RX_QUEUE_SIZE; i++) {
        _rxQueue[i].rxDesc = desc;
        _rxQueue[i].rxDescPhy = _rxQueuePhy + (desc - _rxFirstDesc);
        __rxBufInit(&_rxQueue[i]);
        desc++;
        _rxQueue[i].rxDesc->ds_link = _rxQueuePhy + (desc - _rxFirstDesc);
    }
    _rxQueue[RX_QUEUE_SIZE - 1].rxDesc->ds_link = _rxQueuePhy;
    
    setRegister(AR5K_RXDP, _rxQueuePhy); // set rx descriptor */
    WLLogDebug("set dma rx descriptor 0x%08x\n", (int)_rxQueuePhy);
    
    WLExit();
    return kIOReturnSuccess;    
}

IOReturn AtheroJackCard::_freeRxRing() {
    WLEnter();
    
    /* disable RX descriptors */
    //__disableRx();

    /* write null pointer to RXDP register (just to make full clearing) */
    //setRegister(AR5K_RXDP, 0);
/*
    UInt32 rxBufSize =  roundup(IEEE80211_MAX_LEN, _alignSize);
    
    for (int i=0; i < RX_QUEUE_SIZE; i++) {
        if (_rxQueue[i].rxBufMemDesc) {
            _rxQueue[i].rxBufMemDesc->complete(kIODirectionOutIn);
            _rxQueue[i].rxBufMemDesc->release();
            _rxQueue[i].rxBufMemDesc = NULL;
        }
        
        if (_rxQueue[i].rxBuf) {
            IOFreeContiguous(_rxQueue[i].rxBuf, rxBufSize);
            _rxQueue[i].rxBuf = NULL;
        }
    }
    
    
    if (_rxQueueMemDesc) {
        _rxQueueMemDesc->complete(kIODirectionOutIn);
        _rxQueueMemDesc->release();
        _rxQueueMemDesc = NULL;
    }
    
    UInt32 rxQueueSize = roundup(sizeof(struct ar5213_desc) * (RX_QUEUE_SIZE + 1), _alignSize);
    
    if (_rxFirstDesc) {
        IOFreeContiguous(_rxFirstDesc, rxQueueSize);
        _rxFirstDesc = NULL;
    }
    
    WLExit();
    
    return kIOReturnSuccess;
    */
}

#pragma mark -

IOReturn AtheroJackCard::_enable() {
    WLEnter();
    
    if (_isEnabled) _disable();
    
    // init rx ring
    if (_setupRxRing() != kIOReturnSuccess) {
        WLLogCrit("Could not setup RxRing!");
        return kIOReturnError;
    }
    
    //go into monitor mode
    //setRegister(AR5213_RX_FILTER, AR5K_RX_PROM | AR5K_RX_BEACON | AR5K_RX_CONTROL | AR5K_RX_PROBEREQ); /* set receive filter */
    /*setRegister(AR5213_PHY_FILTER, 0);
    setRegister(AR5K_RXCFG, getRegister(AR5K_RXCFG) & ~ AR5K_RXCFG_ZLFDMA);
    
    __startPCURecieve();
    */
    _isEnabled = true;
    
    enableInterrupts();

    WLExit();
    return kIOReturnSuccess;
}

IOReturn AtheroJackCard::_disable()
{
    //disable monitor moder
    //__stopPCURecieve();
    //setRegister(AR5K_RX_FILTER, AR5K_RX_FILTER_ALL);

    disableInterrupts();
    
    _freeRxRing();

    _isEnabled = false;

    return kIOReturnSuccess;
}

void AtheroJackCard::_myTimeoutHandler(OSObject *owner, IOTimerEventSource *sender) {
    AtheroJackDriver *me;
    AtheroJackCard *card;
    int backoff=0;

    WLLogInfo("AtheroJackCard::_myTimeoutHandler()\n");

    // Make sure that the owner of the timer is us.
    me = OSDynamicCast(AtheroJackDriver, owner);
    if (!me) return;
 
    card = me->getWLCard();
    
    if (!card) {
        WLLogErr("AtheroJackCard::_myTimeoutHandler: WARNING could not get WLCard reference.\n");
        return;
    };
    
    if (card->_failures>25) {
        WLLogErr("AtheroJackCard::_myTimeoutHandler: WARNING timed sendFrame terminated because of general failure.\n");
        return;
    };
    
    if (card->_sendFrame(card->_data,card->_dataSize)!=kIOReturnSuccess)  {
        card->_failures++;
        if (card->_failures%10) {
            //card->_reset(); //TODO reset
            IODelay(50);
        }
        backoff=10;
    } else card->_failures=0;

    // reset the timer
    sender->setTimeoutMS(card->_timeout + backoff);
}

IOReturn AtheroJackCard::
_reset(UInt32 val) {   
    UInt32 mask = val ? val : ~0, read;
    int i;
    IOReturn result = kIOReturnError;

    WLEnter();
    
    WLLogDebug("Intitialize the card\n");

    /* read added before writing, 'cause kernel hanged when reinserting module */
    /*read = getRegister(AR5K_RC);
    setRegister(AR5K_RC, val);
    
    // need to wait at least 128 clocks when reseting PCI before read 
        IODelay(15);
    
    val  &= AR5K_RC_PCU | AR5K_RC_DMA | AR5K_RC_MAC | AR5K_RC_PHY | AR5K_RC_PCI;
    mask &= AR5K_RC_PCU | AR5K_RC_DMA | AR5K_RC_MAC | AR5K_RC_PHY | AR5K_RC_PCI;
    
    for (i = 0; i < 10000; i++) {
        read = getRegister16(AR5K_RC);
        if ((read & mask) != val) 
            IODelay(50);
        else {
            result = kIOReturnSuccess;
            break;
        }
    }

    if (result != kIOReturnSuccess) {
        WLLogCrit("NIC reset failed 0x%x != 0x%x failed!\n", (int)val, (int)read);
    } else if (_isEnabled) _enable();
    */
    WLExit();
    return result;
}

IOReturn AtheroJackCard::
_sendFrame(UInt8* data, IOByteCount size)
{

    //TODO  to implement

    
    return kIOReturnSuccess;
}

#pragma mark -

void AtheroJackCard::
enableInterrupts() {
    /* enable rx */
    //__enableRx();
    //__startPCURecieve();
        
    /* enable interrupts */
    //__setInterrupts(DEFAULT_INTERRUPT_MASK);
    
    //this is for 5210
    //setRegister(AR5K_CR, AR5K_CR_RXE); /* enable receive bit */
    //setRegister(AR5K_DIAG_SW, getRegister(AR5K_DIAG_SW) & ~(AR5K_DIAG_RX_DIS)); 

    /* enable tx */
    //setRegister(AR5K_DIAG_SW, getRegister(AR5K_DIAG_SW) & ~(AR5K_DIAG_TX_DIS)); 
}

void AtheroJackCard::
disableInterrupts() {
    /* disable interrupts */
    //setRegister(AR5K_IER, AR5K_IER_DISABLE);
    
    /* reset interrupt mask */
    //__setInterrupts(0);
    
    /* disable rx */
    //__stopPCURecieve();
    //__disableRx();
    
    //this is for 5210
    //setRegister(AR5K_CR, (getRegister(AR5K_CR) | AR5K_CR_RXD)); /* disable receive bit */
    //setRegister(AR5K_DIAG_SW, AR5K_DIAG_RX_DIS); 

    /* disable tx */
    //setRegister(AR5K_DIAG_SW, AR5K_DIAG_TX_DIS); 
}

IOReturn AtheroJackCard::handleInterrupt() {
    UInt32 status;
    bool handled = false;
    
    WLEnter();

    /*status = getRegister(AR5213_ISR);
    
    if (status & AR5213_F2ERR) {
        setRegister(AR5K_IER, AR5K_IER_DISABLE);
        WLLogCrit("PhyError occured! Card disabled.\n");
        handled = true;
    } else if (status & AR5213_RXORN) {
        setRegister(AR5K_IER, AR5K_IER_DISABLE);
        WLLogCrit("Rx Overflow occured! Card disabled.\n");
        handled = true;
    } else {
        if (status & (AR5213_RX | AR5213_RXDESC | AR5213_RXERR)) { _handleRx(); handled = true; }
        if (status & (AR5213_TX | AR5213_TXDESC | AR5213_TXERR)) { _handleTx(); handled = true; }
    }

    if (status & AR5213_RXEOL) {
        WLLogCrit("The card disabled itself for the following reason: 0x%x.\n", ((int)status) & ~AR5213_RXEOL);
        if (_failures++ < 10) __enableRx();
    }
    
    if (!handled) {
        WLLogDebug("Unhandled interrupt occured! 0x%x.\n", (int)status);
    }*/
    return kIOReturnSuccess;
}

IOReturn AtheroJackCard::
_handleRx() {

    WLLogInfo("packet recieved\n");
    //TODO  to implement
    
    return kIOReturnSuccess;
}

IOReturn AtheroJackCard::
_handleTx() {
    WLLogInfo("packet sent\n");
    return kIOReturnSuccess;
}
