/*
        
        File:			AtheroJackUserClient.cpp
        Program:		AtheroJack
	Author:			Dino Dai Zovi, Michael Rossberg
				mick@binaervarianz.de
	Description:		KisMAC is a wireless stumbler for MacOS X.
                
        This file is part of KisMAC.

        This file is mostly taken from the Viha driver.

    KisMAC is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    KisMAC is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KisMAC; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * WLDriver UserClient implementation
 */

#include "AtheroJackUserClient.h"

#define super IOUserClient
OSDefineMetaClassAndStructors(AtheroJackUserClient, IOUserClient);

IODataQueue* AtheroJackUserClient::_packetQueue = NULL;

bool AtheroJackUserClient::
start(IOService* provider)
{
    WLLogInfo("AtheroJackUserClient::start()\n");
    
    if (!super::start(provider)) {
        WLLogErr("AtheroJackUserClient: start: super::start() failed\n");
        return false;
    }

    _provider = OSDynamicCast(AtheroJackDriver, provider);
    
    if (!_provider)
        return false;

    _wlCard = _provider->getWLCard();

    _userCommandGate = IOCommandGate::commandGate(this);
    if (!_userCommandGate) {
        WLLogErr("AtheroJackUserClient::start: Couldn't get CommandGate\n");
        return false;
    }

    IOWorkLoop* wl = _provider->getWorkLoop();
    if (wl->addEventSource(_userCommandGate) != kIOReturnSuccess) {
        WLLogErr("AtheroJackUserClient::start: Couldn't add gate to workloop\n");
        return false;
    }

    _packetQueue = _provider->getPacketQueue();
    
    return true;
}

void AtheroJackUserClient::stop(IOService* provider) {
    WLLogInfo("AtheroJackUserClient::stop()\n");
    if (_wlCard) _wlCard->stopSendingFrames();
}

bool AtheroJackUserClient::
initWithTask(task_t owningTask, void* securityToken, UInt32 type) {
    WLLogInfo("AtheroJackUserClient::initWithTask()\n");
    
    if (!super::initWithTask(owningTask, securityToken, type))
        return false;

    if (!owningTask)
	return false;

    _owningTask = owningTask;
    _securityToken = securityToken;
    _securityType = type;
    _provider = NULL;

    return true;
}

IOReturn AtheroJackUserClient::clientClose(void)
{
    WLLogInfo("AtheroJackUserClient::clientClose()\n");
    
    close();
    terminate();

    if (_owningTask)
        _owningTask = NULL;

    _provider = NULL;

    return kIOReturnSuccess;
}

IOReturn AtheroJackUserClient::clientDied(void)
{
    WLLogInfo("AtheroJackUserClient::clientDied()\n");

    return super::clientDied();
}

IOExternalMethod* AtheroJackUserClient::
getTargetAndMethodForIndex(IOService** target, UInt32 index) {
    static const IOExternalMethod sMethods[kAtheroJackUserClientLastMethod] = {
        {   // kAtheroJackUserClientOpen
            NULL,
            (IOMethod)&AtheroJackUserClient::open,
            kIOUCScalarIScalarO,
            0,
            0
        },
        {   // kAtheroJackUserClientClose
            NULL,
            (IOMethod)&AtheroJackUserClient::close,
            kIOUCScalarIScalarO,
            0,
            0
        },
        {
            // kAtheroJackUserClientGetChannel
            NULL,
            (IOMethod)&AtheroJackUserClient::_getChannel,
            kIOUCScalarIScalarO,
            0,
            1
        },
        {
            // kAtheroJackUserClientSetChannel
            NULL,
            (IOMethod)&AtheroJackUserClient::_setChannel,
            kIOUCScalarIScalarO,
            1,
            0
        },
        {
            // kAtheroJackUserClientStartCapture
            NULL,
            (IOMethod)&AtheroJackUserClient::_startCapture,
            kIOUCScalarIScalarO,
            1,
            0
        },
        {
            // kAtheroJackUserClientStopCapture
            NULL,
            (IOMethod)&AtheroJackUserClient::_stopCapture,
            kIOUCScalarIScalarO,
            0,
            0
        },
        {   // kAtheroJackUserClientSendFrame
            0,
            (IOMethod) &AtheroJackUserClient::sendFrame,
            kIOUCScalarIStructI,
            1,
            2364
        },
        {
            // kAtheroJackUserClientStopSendingFrames
            0,
            (IOMethod) &AtheroJackUserClient::stopSendingFrames,
            kIOUCScalarIScalarO,
            0,
            0
        },
    };

    if (index < (UInt32)kAtheroJackUserClientLastMethod) {
        *target = this;
        return (IOExternalMethod*)&sMethods[index];
    }

    return NULL;
}

IOReturn AtheroJackUserClient::
clientMemoryForType(UInt32 type, IOOptionBits* optionBits,
                    IOMemoryDescriptor** memoryDescriptor) {
    if (type == kAtheroJackUserClientMap) {
        /* Set memoryDescriptor to DataQueue memory descriptor */
        *memoryDescriptor = _packetQueue->getMemoryDescriptor();
        return kIOReturnSuccess;
    }
    else {
        WLLogCrit("AtheroJackUserClient::clientMemoryForType: bad type\n");
        return kIOReturnError;
    }
}

IOReturn AtheroJackUserClient::
registerNotificationPort(mach_port_t notificationPort,
                         UInt32 type, UInt32 refCon) {
    if (type == kAtheroJackUserClientNotify) {
        /* Set data queue's notification port */
        _packetQueue->setNotificationPort(notificationPort);
        return kIOReturnSuccess;
    }
    else {
        WLLogCrit("AtheroJackUserClient::registerNotificationPort: bad type\n");
        return kIOReturnError;
    }
}

/*********************************************************************
 *
 *********************************************************************/

IOReturn AtheroJackUserClient::open(void) {
    WLLogInfo("AtheroJackUserClient::open()\n");
    
    if (isInactive())
        return kIOReturnNotAttached;

    if (!_provider->open(this))
        return kIOReturnExclusiveAccess;

    return kIOReturnSuccess;
}

IOReturn AtheroJackUserClient::close(void) {
    WLLogInfo("AtheroJackUserClient::close()\n");

    if (!_provider)
        return kIOReturnNotAttached;

    if (_provider->isOpen(this)) {
        _stopCapture();
        _provider->close(this);
    }

    return kIOReturnSuccess;
}

IOReturn AtheroJackUserClient::_getChannel(UInt16* channel) {
    return _userCommandGate->runAction(
        (IOCommandGate::Action)&AtheroJackUserClient::getChannel,
        (void*)channel);
}

IOReturn AtheroJackUserClient::getChannel(OSObject* o, UInt16* channel) {
    WLLogInfo("AtheroJackUserClient::getChannel()\n");
    return kIOReturnError; //((AtheroJackUserClient*)o)->_wlCard->getChannel(channel);
}

IOReturn AtheroJackUserClient::_setChannel(UInt16 channel) {
    return _userCommandGate->runAction(
        (IOCommandGate::Action)&AtheroJackUserClient::setChannel,
        (void*)(UInt32)channel);
}

IOReturn AtheroJackUserClient::setChannel(OSObject* o, UInt16 channel) {
    WLLogInfo("AtheroJackUserClient::setChannel()\n");
    return kIOReturnError; //((AtheroJackUserClient*)o)->_wlCard->setChannel(channel);
}

IOReturn AtheroJackUserClient::_startCapture(UInt16 channel) {
    return _userCommandGate->runAction(
        (IOCommandGate::Action)&AtheroJackUserClient::startCapture,
        (void*)(UInt32)channel);
}
    
IOReturn AtheroJackUserClient::startCapture(OSObject* o, UInt16 channel) {
    WLLogInfo("AtheroJackUserClient::startCapture(%d)\n", channel);
    ((AtheroJackUserClient*)o)->_provider->getWorkLoop()->enableAllInterrupts();
    return ((AtheroJackUserClient*)o)->_wlCard->startCapture(_packetQueue, channel);
}

IOReturn AtheroJackUserClient::_stopCapture() {
    return _userCommandGate->runAction(
        (IOCommandGate::Action)&AtheroJackUserClient::stopCapture);
}

IOReturn AtheroJackUserClient::stopCapture(OSObject* o) {
    WLLogInfo("AtheroJackUserClient::stopCapture()\n");
    ((AtheroJackUserClient*)o)->_provider->getWorkLoop()->disableAllInterrupts();
    return ((AtheroJackUserClient*)o)->_wlCard->stopCapture();
}

IOReturn AtheroJackUserClient::sendFrame(UInt32 repeatTimer, void* pkt, IOByteCount size) {
    WLLogInfo("AtheroJackUserClient::sendFrame()\n");
    if (size != 2364) return kIOReturnBadArgument;
    return _wlCard->sendFrame((UInt8*)pkt, repeatTimer);
}

IOReturn AtheroJackUserClient::stopSendingFrames() {
    WLLogInfo("AtheroJackUserClient::stopSendingFrames()\n");
    return _wlCard->stopSendingFrames();
}


