/*
        
        File:			OpenHAL.cpp
        Author:			Michael Rossberg
				mick@binaervarianz.de
	Description:		OpenHAL is a free replacement for ther AtherosHALDriver.
                
        This file is part of OpenHAL.

    KisMAC is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    KisMAC is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KisMAC; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#define KERNEL 1

#include "OpenHAL.h"
#include "HALIDs.h"

OpenHAL::OpenHAL(void* ioBase, int alignSize) {
    _ioBase = ioBase;
    _alignSize = alignSize;
    _cardPresent = 1;
    
    _init();
}

#pragma mark -


void OpenHAL::enableRx() {
    setRegister(AR5K_CR, AR5K_CR_RXE);
}

IOReturn OpenHAL::disableRx() {
    setRegister(AR5K_CR, AR5K_CR_RXD);
    return _wait(AR5K_CR, AR5K_CR_RXE, 0);
}

void OpenHAL::setupRxDesc(struct ar5213_desc *desc, UInt32 size, UInt32 flags) {
    desc->ds_ctl0 = 0;
    desc->ds_ctl1 = size & 0x0FFF | flags;
    desc->ds_hw0 = 0;
    desc->ds_hw1 = 0;
}

void OpenHAL::startPCURecieve() {
    setRegister(AR5K_CFP_DUR, getRegister(AR5K_CFP_DUR) & ~AR5K_CFP_DUR_DISABLE_RX);
}

void OpenHAL::stopPCURecieve() {
    setRegister(AR5K_CFP_DUR, getRegister(AR5K_CFP_DUR) | AR5K_CFP_DUR_DISABLE_RX);
}

void OpenHAL::setRxDP(void* rxDP) {
    setRegister(AR5K_RXDP, (UInt32)rxDP);   /* set rx descriptor */
}

void OpenHAL::setInterrupts(UInt32 ints) {
    setRegister(AR5K_IER, AR5K_IER_DISABLE);
    setRegister(AR5K_IMR_5213, ints);
    if (ints) setRegister(AR5K_IER, AR5K_IER_ENABLE);
}

void OpenHAL::procRxDesc(struct ar5213_desc *x) {
    if (x->ds_hw1!=0) return;
    //not done yet
}

void OpenHAL::writeAssocID(UInt8* bssid, UInt16 assocId) {
    UInt32 addr_low, addr_high;
    
    memcpy(&addr_low, &(bssid[0]), 4);
    memcpy(&addr_high, &(bssid[4]), 2);
    
    addr_high = 0x0000ffff & OSSwapHostToLittleInt32(addr_high);
    addr_high |= (assocId & 0x3FFF) << 0x10;
    setRegister(AR5K_BSS_ID0, OSSwapHostToLittleInt32(addr_low));
    setRegister(AR5K_BSS_ID1, addr_high);
    
    if (assocId) {
        setRegister(AR5K_USEC, getRegister(AR5K_USEC) & ~0x007F0000);
    }

}

/*void OpenHAL::setOPMode() {
    UInt32 addr_low, addr_high;
    UInt8 mac[6];

    UInt32 pcu = AR5K_STA_ID1_STA_AP | 
            AR5K_STA_ID1_PSPOLLDIS | 
            AR5K_STA_ID1_DESC_ANT;
    UInt32 beacon = INIT_BCON_CNTRL_REG | 
            AR5K_BCR_APMODE;
    UInt32 led = AR5K_PCICFG_LED_ACT | AR5K_PCICFG_BCTL;
    
    _getMACAddress(mac);
    //
    memcpy(&addr_low, &(mac[0]), 4);
    memcpy(&addr_high, &(mac[4]), 2);
    addr_high = 0x0000ffff & OSSwapHostToLittleInt32(addr_high);
    setRegister(AR5K_STA_ID0, OSSwapHostToLittleInt32(addr_low));
    setRegister(AR5K_STA_ID1, addr_high);

    // set options
    setRegister(AR5K_STA_ID1, pcu | addr_high);
    setRegister(AR5K_BCR, beacon);
    setRegister(AR5K_PCICFG, led);
}*/

//function complete, 1:1 copy, only missing the last line
IOReturn OpenHAL::setPowerMode(HAL_POWER_MODE mode, int setChip, UInt16 sleepDuration) {
    int i;
    
    switch (mode) {
        case HAL_PM_NETWORK_SLEEP:
            setRegister(AR5K_STA_ID1, getRegister(AR5K_STA_ID1) | AR5K_STA_ID1_PWR_SAV);
            if (setChip) {
                setRegister(AR5K_SCR, (sleepDuration | AR5213_SCR_NET_SLEEP) & AR5K_SCR_SLDUR_MASK);
            }
            break;
        case HAL_PM_FULL_SLEEP:
            setRegister(AR5K_STA_ID1, getRegister(AR5K_STA_ID1) | AR5K_STA_ID1_PWR_SAV);
            if (setChip) {
                setRegister(AR5K_SCR, AR5213_SCR_COMPL_SLEEP);
            }
            break;
        case HAL_PM_AWAKE:
            if (setChip) {
                setRegister(AR5K_SCR, 0);
                IODelay(10);
                i = 100;
                while (getRegister(AR5K_PCICFG) & AR5K_PCICFG_SPWR_DN) {
                    IODelay(20);
                    setRegister(AR5K_SCR, 0);
                    if (i-- == 0) return kIOReturnError;
                }
            }
            //switch off powersave
            setRegister(AR5K_STA_ID1, getRegister(AR5K_STA_ID1) & ~AR5K_STA_ID1_PWR_SAV);
            break;
        default:
            return kIOReturnBadArgument;
            break;
    }
    
    //todo: we should save our state at the end
    return kIOReturnSuccess;
}

IOReturn OpenHAL::reset(HAL_OPMODE opmode, HAL_CHANNEL *channel, bool bChannelChange) {
    int register1140;
    int txMask;
    int pcfBit;
    int pciConfig;
    int channelFlags;
    struct initModeMatrix mset;
    struct initFreqMatrix fset;  
    struct initGeneralMatrix gset;
    enum ar5213_modi mode;
    enum ar5213_freq freq;
    int i;
    int waiter = 0;
    
    if (channel->channelFlags & CHANNEL_2GHZ) {
        if (channel->channelFlags & CHANNEL_5GHZ) return kIOReturnBadArgument;
    } else if (channel->channelFlags & CHANNEL_2GHZ) {
        if (channel->channelFlags & CHANNEL_2GHZ) return kIOReturnBadArgument;
    }

    if (channel->channelFlags & CHANNEL_OFDM) {
        if (channel->channelFlags & CHANNEL_CCK) return kIOReturnBadArgument;
    } else if (channel->channelFlags & CHANNEL_CCK) {
        if (channel->channelFlags & CHANNEL_OFDM) return kIOReturnBadArgument;
    }
    
    if (bChannelChange) {
        register1140 = 0;
    } else {
        register1140 = getRegister(0x1140);
    }
    
    txMask = getRegister(AR5K_TX_MASK0);
    if (txMask == 0) txMask = 1;
    
    pcfBit = getRegister(AR5K_STA_ID1) & AR5K_STA_ID1_PCF;
    pciConfig = getRegister(AR5K_PCICFG) & AR5K_PCICFG_LED_PEND &  AR5K_PCICFG_LED_ACT & AR5213_PCICFG_UNKNOWN2;
    
    if (_resetMacAndPhy(channel) != kIOReturnSuccess) return kIOReturnError;
    
    channelFlags = channel->channelFlags & (CHANNEL_5GHZ | CHANNEL_OFDM | CHANNEL_TURBO | CHANNEL_CCK | CHANNEL_2GHZ);    
   
    switch(channelFlags) {
        case (CHANNEL_5GHZ | CHANNEL_OFDM):
            mode = ANorm;
            break;
        case (CHANNEL_5GHZ | CHANNEL_OFDM | CHANNEL_TURBO):
            mode = ATurbo;
            break;
        case  (CHANNEL_2GHZ | CHANNEL_CCK):
            mode = BNorm;
            break;
        case (CHANNEL_2GHZ | CHANNEL_OFDM):
            mode = GNorm;
            break;
        case (CHANNEL_2GHZ | CHANNEL_OFDM | CHANNEL_TURBO):
            mode = GTurbo;
            break;
        default:
            return kIOReturnBadArgument;
    };
    
    freq = (channelFlags | CHANNEL_5GHZ) ? _5GHz : _2GHz;
    
    setRegister(PHY_BASE, PHY_BASE_RESET);

    //chip Reset depend on Mode
    for (i=0; i<=30; i++) {
        mset = modeInitMatrix[i];
        setRegister(mset.reg, mset.val[mode]);
        if (waiter++ % 64 == 0) IODelay(1);
    }
    
    //generic Chipreset
    for (i=0; i<=246; i++) {
        gset = genInitMatrix[i];
        if (bChannelChange || (gset.reg<0x9000)) {
            setRegister(gset.reg, gset.val);
            if (waiter++ % 64 == 0) IODelay(1);
        }
    }
    
    //Phy Chip Reset depends on version of the chip
    if ((_PHYVersion) & 0xF0 >= 0x30) {
        //Phy Chip Reset depended on Modes
        for (i=0; i<=6; i++) {
            mset = modeInitPhyMatrixNew[i];
            setRegister(mset.reg, mset.val[mode]);
            if (waiter++ % 64 == 0) IODelay(1);
        }
        
        //General Phy Chip Reset 
        for (i=0; i<=60; i++) {
            gset = genInitPhyMatrixNew[i];
            setRegister(gset.reg, gset.val);
            if (waiter++ % 64 == 0) IODelay(1);
        }
        
        //Frequency specfic setup 
        for (i=0; i<=63; i++) {
            fset = modeInitFreqMatrixNew[i];
            setRegister(fset.reg, fset.val[freq]);
            if (waiter++ % 64 == 0) IODelay(1);
        }
    } else {
        //Phy Chip Reset depended on Modes
        for (i=0; i<=8; i++) {
            mset = modeInitPhyMatrixOld[i];
            setRegister(mset.reg, mset.val[mode]);
            if (waiter++ % 64 == 0) IODelay(1);
        }
    
        //General Phy Chip Reset 
        for (i=0; i<=58; i++) {
            gset = genInitPhyMatrixOld[i];
            setRegister(gset.reg, gset.val);
            if (waiter++ % 64 == 0) IODelay(1);
        }

        //Frequency specfic setup 
        for (i=0; i<=63; i++) {
            fset = modeInitFreqMatrixOld[i];
            setRegister(fset.reg, fset.val[freq]);
            if (waiter++ % 64 == 0) IODelay(1);
        }
    }
    
    _setupTxTimer(channel);
    
    if (_PHYRevision < 65) {
        setRegister(0x982C, 0x02E002);
        setRegister(0x984C, 0x0C0C40);
        setRegister(0xA228, getRegister(0xA228) | 0x200);
        setRegister(0xA228, getRegister(0xA228) & ~0x01FC00 | 0x800);
    }
    
    //do country code stuff (not reverse engineered yet)
    
    if (channel->channelFlags & CHANNEL_OFDM == 0) __initCCK(channel);
    
    __messWithChannels(channel);
    
    //not done yet
    return kIOReturnSuccess;
}
#define align250(a) (((a)+249)&~249)
#define align125(a) (((a)+124)&~124)
//function complete, 1:1 copy
UInt32  OpenHAL::computeTxTime(const HAL_RATE_TABLE *rates, UInt32 frameLen, UInt16 rateix, bool shortPreamble) {
    int timeOffset;
    int bitlen;
    int time;
    int item = rates->rateCodeToIndex[rateix];
    
    bitlen = frameLen * 8;
    
    switch (rates->info[item].phy) {
        case 1: //CCK
            if (shortPreamble || (rates->info[item].shortPreamble!=0)) timeOffset = 0x60;
            else timeOffset = 0xC0;
            
            return (bitlen * 1000 / rates->info[item].rateKbps) + 10 + timeOffset;
        case 2: //OFDM
            return ((bitlen + 15) / (align250(rates->info[item].rateKbps) / 250)) * 4 + 0x28;
        case 3: //TURBO modulation
            return ((bitlen + 15) / (align125(rates->info[item].rateKbps) / 125)) * 4 + 0x1A;
        case 4: //another unknown modulation, DBPSK? DQPSK?
            time = ((bitlen + 15) / (align250(rates->info[item].rateKbps) / 250) + 1) / 4;
            if (time >= 1000) return time + 92;
            else return time + 189;
        default:
            return 0;
    };
}

//function complete, 1:1 copy
const HAL_RATE_TABLE* OpenHAL::getRateTable(UInt32 mode) {
    HAL_RATE_TABLE* table;
    int i;
    
    switch (mode) {
    case HAL_MODE_11A:
        table = &_rate_table[0];
        break;
    case HAL_MODE_TURBO:
        table = &_rate_table[1];
        break;
    case HAL_MODE_11B:
        table = &_rate_table[2];
        break;
    case HAL_MODE_11G:
        table = &_rate_table[3];
        break;
    case HAL_MODE_ALL:
        table = &_rate_table[4];
        break;
    default:
        return NULL;
    }
    
    if (table->rateCodeToIndex[0]) return kIOReturnSuccess;
    for (i = 0; i < 32; i++) table->rateCodeToIndex[i] = 0xFF;
    
    for (i = 0; i < table->rateCount; i++) {
        //this looks more reasonable
        table->rateCodeToIndex[table->info[i].rateCode] = i;
        table->rateCodeToIndex[table->info[i].shortPreamble] = i;
    }
    
    return table;    
}


#pragma mark -

void OpenHAL::__initCCK(HAL_CHANNEL *channel) {
    UInt32 num, j;
    int i;
    
    if (channel->channelFlags & CHANNEL_TURBO) num = 0;
    else num = 0x27;
    
    num = num * 5 + 400;
    num = num << 17;
    
    num = num / channel->channel
    
    for (i = 31; i > 0; i--) {
        if ((num >> i) & 0x1) break;
    }
    
    j = (32 - i);
    num = (num + (0x1 << i)) >> i;
    
    j = (j >> 13) & 0x0001E000;
    setRegister(0x9814, (getRegister(0x9814) & 0x0001FFFF | num >> 0x11));
    setRegister(0x9814, (getRegister(0x9814) & ~ 0x0001E000 | j));
}

const UInt32 oldFirmwareOffsets [] = { 0xC2, 0xC4, 0xC5, 0xD0, 0xDA,  0xE4 }
const UInt32 newFirmwareOffsets [] = { 0xC2, 0xC3, 0xD4, 0xF2, 0x10D, 0x128 }

inline void fillTable(UInt16 value, UInt16* channelArr) {
    channelArr[0] = (value >> 0x0F);        //6
    channelArr[1] = (value >> 0x03) & 0x01; //8
    channelArr[2] = (value >> 0x0E) & 0x01; //A
    channelArr[3] = (value >> 0x0B) & 0x07; //C
    channelArr[4] = (value >> 0x04) & 0x01; //E
    channelArr[7] = (value) & 0x01;         //16
    channelArr[8] = (value >> 0x01) & 0x01; //18
    channelArr[9] = (value >> 0x02) & 0x01; //1A
}

IOReturn OpenHAL::_doFirmwareSetup(UInt16 firmwareVersion) {
    UInt32 *offset;
    UInt16 val;
    UInt16 channelArr[5][10];
    
    if (firmwareVersion > 0x3002) offset = newFirmwareOffsets;
    else offset = oldFirmwareOffsets;
    
    if (_readEEPROM5213(offset[0], &val) != kIOReturnSuccess) return kIOReturnIOError;
    fillTable(val, channelArr[0]);
    
    
    //not done yet
}


IOReturn OpenHAL::__messWithChannels(HAL_CHANNEL *channel) {
    UInt32 mode;
    
    switch(channelFlags) {
        case (CHANNEL_5GHZ | CHANNEL_OFDM):
        case (CHANNEL_5GHZ | CHANNEL_OFDM | CHANNEL_TURBO):
            setRegister(0x9944, (getRegister(0x9944) & ~ 0x38) | ((4 << 3) & 0x38))
            mode = 0;
            break;
        case  (CHANNEL_2GHZ | CHANNEL_CCK):
            mode = 1;
            break;
        case (CHANNEL_2GHZ | CHANNEL_OFDM):
        case (CHANNEL_2GHZ | CHANNEL_OFDM | CHANNEL_TURBO):
            mode = 2;
            break;
        default:
            return kIOReturnBadArgument;
    };
    //not done yet
}

IOReturn OpenHAL::__setupModes(HAL_CHANNEL *channel, UInt32 mode) {
    switch(channelFlags) {
        case (CHANNEL_5GHZ | CHANNEL_OFDM):
        case (CHANNEL_5GHZ | CHANNEL_OFDM | CHANNEL_TURBO):
            mode = ATurbo;
            break;
        case  (CHANNEL_2GHZ | CHANNEL_CCK):
    
            break;
        case (CHANNEL_2GHZ | CHANNEL_OFDM):
        case (CHANNEL_2GHZ | CHANNEL_OFDM | CHANNEL_TURBO):
            break;
        default:
            return kIOReturnBadArgument;
    };
}
/*
IOReturn OpenHAL::__resetUnknown(HAL_CHANNEL *channel) {
    UInt32 var30[30];
    UInt32 var8 [8];
    UInt16 b0, ae;

    bzero(var30, 30);
    bzero(var8,  8);
    
    _resetUnknown2(var8, channel, 0x30000, &b0, &ae);
    //not done yet
}

IOReturn OpenHAL::__resetUnknown2(UInt8* var8, HAL_CHANNEL *channel, UInt32 i, UInt16 b0, UInt16 ae) {
    __resetChannel(channel);
    //not done yet
}

IOReturn OpenHAL::__resetChannel(HAL_CHANNEL *channel) {

}
*/


//function complete, 1:1 copy, init rate tables and buffer allocation not implemented
IOReturn OpenHAL::_init() {
    UInt16 val, firmwareVersion, crc;
    int i;
    
    //do some magic initialization
    setRegister8(0x0CD4, 0x10);
    setRegister (0x0CD0, 0x00);
    setRegister8(0x0CD5, 0x04);
    setRegister (0x0CD8, 0x00);
    setRegister (0x0CDC, 0x00);
    setRegister (0x0CE8, 0x00);
    setRegister (0x0CF4, 0x02);
    setRegister (0x0CF8, 0x00);
    setRegister (0x0CFC, 0x00);
    setRegister (0x0D00, 0x0A);
    setRegister (0x0D04, 0x0A);
    setRegister (0x0D08, 0x64);
    setRegister (0x0D0C, 0x00);
    setRegister (0x0D1C, 0x01);
    setRegister (0x0D20, 0x01);
    setRegister8(0x0D24, 0x00);

    if (_resetMacAndPhy(NULL)) return kIOReturnError;
    
    _MACRevision    = getRegister(AR5K_SREV) & 0xFF >> 4;
    _MACSubRevision = getRegister(AR5K_SREV) & 0x0F;
    
    if (_MACRevision != 5) return kIOReturnError;
    
    _PHYRevision = getRegister(PHY_CHIP_ID);
    
    if (_testReadWritePhyMac() != kIOReturnSuccess) return kIOReturnDeviceError;
    
    setRegister(PHY_BASE, PHY_BASE_RESET);
    
    _PHYVersion = _getPhyChipVersion();
    
    if (_PHYVersion & 0xF0 != 0x10) {
         HALLogEmerg("Unsupported PHY version!");
         return kIOReturnUnsupported;
    }
    
    if (_readEEPROM5213(EEPROM_VERSION, &firmwareVersion) != kIOReturnSuccess) {
         HALLogEmerg("Error reading EEPROM!");
         return kIOReturnIOError;
    }
    if (firmwareVersion < 0x3001) {
         HALLogEmerg("Unsupported EEPROM version!");
         return kIOReturnUnsupported;
    }
    
    if (((getRegister(AR5K_PCICFG) & AR5213_PCICFG_EEPROM_SIZE) >> 3) != AR5213_PCICFG_UNKNOWN1) {
         HALLogEmerg("Unsupported EEPROM size!");
         return kIOReturnUnsupported;
    }

    crc = 0;
    for (i = 0; i<=0x33F; i++) {
        if (_readEEPROM5213(EEPROM_BASE + i, &val) != kIOReturnSuccess) {
            HALLogEmerg("Error reading EEPROM for checksum!");
            return kIOReturnIOError;
        }
        crc ^= val;
    }
    
    if (crc != 0xFFFF) { 
         HALLogEmerg("EEPROM checksum failed!");
         return kIOReturnDeviceError;
    }

    for(i = 0; i < 9; i++) {
        _channelTable[i].size = sizeof(ar5213_channel_table_data);
    }
    
    //do more table setups
    _doFirmwareSetup(firmwareVersion);
    
    if (_PHYVersion & 0x0F != 0x10) {
        setRegister(PHY_BASE, PHY_BASE_RESET | PHY_BASE_MAGIC);
        IODelay(2000);
        _PHYSubRevision = _getPhyChipVersion();
        
        setRegister(PHY_BASE, PHY_BASE_RESET);
        IODelay(2000);
        
        if (_PHYSubRevision & 0x0F != 0x20) {
            HALLogEmerg("Unsupported Hardware version!");
            return kIOReturnDeviceError;
        }
    }
    
    return kIOReturnSuccess;
}

//needs work to detect _5Ghz based on mode
IOReturn OpenHAL::_setChannel_5213(UInt32 channel) {
    //lots of magic in this routine too
    UInt32 arg1, arg2, arg3, tmp, simple, val1, val2;
    struct channelsetting c;
    bool _5Ghz = false;
    
    if (channel > 14) _5Ghz = true;
    
    if (_5GHz) {
        c = channel_table_2400[channel];
        arg2 = (_bitswap(c.arg2, 8) & 0x0F) << 5;
        arg1 = c.arg1 << 4;
        arg1 = arg1 | arg2;
        arg3 = c.arg3;
        
        tmp=getRegister(AR5213_CHAN14_MODE);
        if (channel==14) {
            tmp |= AR5213_CHAN14_MODE_ENABLED;
        } else {
            tmp &= ~ AR5213_CHAN14_MODE_ENABLED;
        }
        setRegister(AR5213_CHAN14_MODE, tmp);
    } else {
        arg1 = 0;
        arg3 = channel;
    }
    
    if ((arg3 > 0x90) || (arg3 & 1)) {
        arg3 = arg3 - 0x18;
        arg3 = _bitswap(arg3, 8) & 0xFF;
        simple = 1;
    } else {
        arg3 = arg3 - 0x18;
        tmp = arg3 << 31;
        arg3 = (arg3 + tmp) >> 1;
        arg3 = _bitswap(arg3, 8) & 0xFF;
        simple = 0;
    }
    
    arg3 = (arg3 << 2) | (simple << 1) | 0x401;
    
    val1 = (arg1 & 0xFF) << 8 | (arg3 & 0xFF);
    val2 = (arg1 & 0xFF00)    | (arg3 & 0xFF00) >> 8;
    
    setRegister(PHY_RF_CHANREG0, val1);
    setRegister(PHY_RF_CHANREG1, val2);
    IOLog("set channel:%u register1:0x%x register2:0x%x\n", (int)channel, (int)val1, (int)val2 );
    
    return kIOReturnSuccess;
}

//function complete, 1:1 copy
UInt32 OpenHAL::_getPhyChipVersion() {
    int i;
    int rev;
    
    setRegister(PHY_MAGIC, 0x1C16);
    for (i = 0; i<=7; i++) {
        setRegister(PHY_MAGIC2, 0x10000);
    }
    
    rev = getRegister(PHY_CHIP_VERSION) >> 0x18;
    return _bitswap(rev >> 4 | (rev & 0x0F) << 4, 8);
}

//function complete, 1:1 copy
IOReturn OpenHAL::_testReadWritePhyMac() {
    int i, j;
    unsigned int tmp;
    int regist[] = { 0x8000, 0x9820 };
    int vals[2];
    
    for (i = 0; i < 2; i++) {
        vals[i] = getRegister(regist[i]);
        for (j = 0; j <= 0xFF; j++) {
            tmp = j << 16 | j;
            setRegister(regist[i], tmp);
            if (getRegister(regist[i]) != tmp) {
                HALLogEmerg("Read write test FAILED for register 0x%x!", regist[i]);
                return kIOReturnDeviceError;
            }
        }
        setRegister(regist[i], vals[i]);
    }
    
    return kIOReturnSuccess;
}

//function complete, 1:1 copy
IOReturn OpenHAL::_wait(UInt32 reg, UInt32 mask, UInt32 val) {
    int i;
    
    for (i = 0; i < 1000; i++) {
        if ((getRegister(reg) & mask) == val) return kIOReturnSuccess;
        IODelay(10);
    }
    
    HALLogCrit("timeout on reg 0x%x: 0x%08x & 0x%08x != 0x%08x\n", (int)reg, (int)getRegister(reg), (int)mask, (int)val);
    return kIOReturnError;
}

//function complete, 1:1 copy
UInt32 OpenHAL::_bitswap(UInt32 val, int bit_count) {
    UInt32 retval;
    UInt32 bit;
    int i;
    
    retval = 0;
    
    for (i = 0; i < bit_count; i++) {
        bit = (val >> i) & 1;
        retval = (retval << 1) | bit;
    }
    
    return retval;
}

//function complete, 1:1 copy
IOReturn OpenHAL::_setupTxTimer(HAL_CHANNEL *channel) {
    int mode;
    const HAL_RATE_TABLE* table;
    int txTime;
    int i;
    
    if (channel->channelFlags & CHANNEL_TURBO) mode = HAL_MODE_TURBO;
    else mode = HAL_MODE_ALL;
    
    table = getRateTable(mode);
    if (table) {
        for (i = 0; i < table->rateCount; i++) {
            txTime = computeTxTime(table, 14, table->info[i].controlRate, false);
            setRegister(AR5213_TXTIMER_BASE + (table->info[i].rateCode*4), txTime);
        }
    }
    
    if (mode == HAL_MODE_TURBO) return kIOReturnSuccess;
    
    table = getRateTable(HAL_MODE_11B);
    if (table) {
        for (i = 0; i < table->rateCount; i++) {
            txTime = computeTxTime(table, 14, table->info[i].controlRate, false);
            setRegister(AR5213_TXTIMER_BASE + (table->info[i].rateCode * 4), txTime);
            if (table->info[i].shortPreamble) {
                txTime = computeTxTime(table, 14, table->info[i].controlRate, true);
                setRegister(AR5213_TXTIMER_BASE + (table->info[i].rateCode * 4) + (table->info[i].shortPreamble * 4), txTime);
            }
        }
    }
    
    return kIOReturnSuccess;    
}


//function complete, 1:1 copy
IOReturn OpenHAL::_resetMacAndPhy(HAL_CHANNEL *channel) {
    
    if (_resetMAC(AR5K_RC_PCI | AR5K_RC_DMA | AR5K_RC_PCU) != kIOReturnSuccess) return kIOReturnError;
    
    if (setPowerMode(HAL_PM_AWAKE, 1, 0) != kIOReturnSuccess) return kIOReturnError;
    
    if (_resetMAC(0) != kIOReturnSuccess) return kIOReturnError;
    
    if (!channel) return kIOReturnSuccess;
    
    if (channel->channelFlags & CHANNEL_5GHZ) {
        setRegister(PHY_RF_MODE,PHY_RF_MODE_5GHz);
        IODelay(300);
        setRegister(AR5213_CHAN_WIFI_MODE, AR5213_CHAN_WIFI_MODE_A);
        
        //enable Turbo mode if needed
        if (channel->channelFlags & CHANNEL_TURBO) setRegister(PHY_FRAME_CONTROL, PHY_FC_TURBO_MODE | PHY_FC_TURBO_SHORT);
    } else if (channel->channelFlags & (CHANNEL_2GHZ | CHANNEL_CCK)) {
        //disable Turbo mode
        setRegister(PHY_FRAME_CONTROL, 0);
        
        setRegister(AR5213_CHAN_WIFI_MODE, AR5213_CHAN_WIFI_MODE_B);
        
        setRegister(PHY_RF_MODE,PHY_RF_MODE_2GHz);
        IODelay(300);
    } else if (channel->channelFlags & (CHANNEL_2GHZ | CHANNEL_OFDM)) {
        //enable Turbo mode if needed
        if (channel->channelFlags & CHANNEL_TURBO) setRegister(PHY_FRAME_CONTROL, PHY_FC_TURBO_MODE | PHY_FC_TURBO_SHORT);

        setRegister(AR5213_CHAN_WIFI_MODE, AR5213_CHAN_WIFI_MODE_G);
        
        setRegister(PHY_RF_MODE,PHY_RF_MODE_2GHz);
        IODelay(300);
    }
    
    return kIOReturnSuccess;
}

//function complete, 1:1 copy
IOReturn OpenHAL::_resetMAC(UInt32 flags) {
    UInt32 mask;
    
    if (flags) mask = flags & (AR5K_RC_PCU | AR5K_RC_DMA);
    else mask = (AR5K_RC_PCU | AR5K_RC_DMA);

    setRegister(AR5K_RC, flags);
    IODelay(15);
    
    return _wait(AR5K_RC, flags & (AR5K_RC_PCU | AR5K_RC_DMA), mask);
}

//function complete, 1:1 copy
IOReturn OpenHAL::_readEEPROM5213(UInt32 offset, UInt16 *data) {
    int timeout = 10000;   
    UInt32 status;

    // set address
    setRegister(AR5211_EEPROM_ADDR, offset);

    // enable eeprom read access
    setRegister(AR5211_EEPROM_COMD, AR5211_EEPROM_COMD_READ);
    
    while(timeout > 0) {
        IODelay(1);
        status = getRegister(AR5211_EEPROM_STATUS);
        if(status & AR5211_EEPROM_STAT_RDDONE) {
                if(status & AR5211_EEPROM_STAT_RDERR) {
                    HALLogErr("eeprom read access failed!\n");
                    return kIOReturnError;
                }
                status = getRegister(AR5211_EEPROM_DATA);
                *data = status & 0x0000ffff;
                return kIOReturnSuccess;
        }
        timeout--;
    }
	
    HALLogErr("eeprom read timeout!\n");
    return kIOReturnError;
}

