/*
        
        File:			OpenHAL.h
        Author:			Michael Rossberg
				mick@binaervarianz.de
	Description:		OpenHAL is a free replacement for ther AtherosHALDriver.
                
        This file is part of OpenHAL.

    KisMAC is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    KisMAC is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KisMAC; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <IOKit/IOService.h>
#include <IOKit/IOTypes.h>
#include <libkern/OSByteOrder.h>
#include <IOKit/IODataQueue.h>
#include <IOKit/IOTimerEventSource.h>
#include "ah_desc.h"
#include "HALStructures.h"

class OpenHAL {

public:
    OpenHAL(void* ioBase, int alignSize);
    
    void goSleep() { if(_cardPresent==1) _cardPresent=2; }
    void wakeUp()  { if(_cardPresent==2) _cardPresent=1; }
     
    //need verification
    void                    enableRx();
    IOReturn                disableRx();
    void                    setupRxDesc(struct ar5213_desc *desc, UInt32 size, UInt32 flags = 0);
    void                    stopPCURecieve();
    void                    startPCURecieve();
    void                    setRxDP(void* rxDP);
    void                    setInterrupts(UInt32 ints);
    
    void                    writeAssocID(UInt8* bssid, UInt16 assocId);
    //void                    setOPMode();
    
    //work in progress
    void                    procRxDesc(struct ar5213_desc *desc);
    IOReturn                reset(HAL_OPMODE opmode, HAL_CHANNEL *channel, bool bChannelChange);
    
    //completed
    const HAL_RATE_TABLE*   getRateTable(UInt32 mode);
    IOReturn                setPowerMode(HAL_POWER_MODE mode, int setChip, UInt16 sleepDuration);

    //generic functions
    UInt32                  computeTxTime(const HAL_RATE_TABLE *rates, UInt32 frameLen, UInt16 rateix, bool shortPreamble);
private:

    //work in progress
    IOReturn            _init();

    //completed
    IOReturn            _setChannel_5213(UInt32 channel);
    IOReturn            _setupTxTimer(HAL_CHANNEL *channel);
    IOReturn            _resetMacAndPhy(HAL_CHANNEL *channel);
    IOReturn            _resetMAC(UInt32 flags);
    IOReturn            _resetChip(bool didChannelChange);
    UInt32              _getPhyChipVersion();
    
    //generic
    IOReturn            _testReadWritePhyMac();
    IOReturn            _wait(UInt32 reg, UInt32 mask, UInt32 val);
    UInt32              _bitswap(UInt32 val, int bit_count);
    IOReturn            _readEEPROM5213(UInt32 offset, UInt16 *data);
  
    void*  _ioBase;
    UInt32 _alignSize;
    
    int             _cardPresent;
    
    UInt32          _MACRevision;
    UInt32          _PHYRevision;
    UInt32          _MACSubRevision;
    UInt32          _PHYVersion;
    UInt32          _PHYSubRevision;
    
    ar5213_channel_table _channelTable[10];
    /*
     * Register accessors (return/accept values in host byte order)
     */
    inline UInt32 getRegister(int r, bool littleEndian = true) {
        if (_cardPresent!=1) return 0xFFFFFFFF;
        if (littleEndian)
            return OSReadLittleInt32(_ioBase, r);
        else
            return OSReadBigInt32(_ioBase, r);
    }
    inline UInt16 getRegister16(int r, bool littleEndian = true) {
        if (_cardPresent!=1) return 0xFFFF;
        if (littleEndian)
            return OSReadLittleInt16(_ioBase, r);
        else
            return OSReadBigInt16(_ioBase, r);
    }
    
    inline void setRegister(int r, UInt32 v,
                            bool littleEndian = true) {
        if (_cardPresent!=1) return;
        if (littleEndian)
            OSWriteLittleInt32(_ioBase, r, v);
        else
            OSWriteBigInt32(_ioBase, r, v);
	
        OSSynchronizeIO();
    }    
    inline void setRegister8(int r, UInt8 v) {
        if (_cardPresent!=1) return;
        
        *(volatile UInt8*)((uintptr_t)_ioBase + r) = v;
	
        OSSynchronizeIO();
    }   
};
