/*
        
        File:			GPSController.m
        Program:		KisMAC
	Author:			Michael Rossberg
                                mick@binaervarianz.de
	Description:		KisMAC is a wireless stumbler for MacOS X.
                
        This file is part of KisMAC.
        
        Parts of this file are based on bsd airtools by h1kari.

    KisMAC is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    KisMAC is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KisMAC; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#import "GPSController.h"
#import "WaveHelper.h"

#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/uio.h>
#include <sys/vnode.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <netdb.h>
#include <string.h>
#include <errno.h>

#include <sys/termios.h>

struct termios ttyset;

#define MAX_GPSBUF_LEN 1024

@implementation GPSController

- (id)init {
    _gpsLock = [[NSLock alloc] init];
    _gpsThreadUp = NO;
    _gpsShallRun = NO;
    _debugEnabled = NO;
    _lastAdd = [[NSDate date] retain];
    _trace = [[NSMutableArray array] retain];
    return self;
}

- (bool)startForDevice:(NSString*) device {
    
    _reliable = NO;
    _ns.dir = 'N';
    _ns.coordinates = 0;
    _ew.dir = 'E';
    _ew.coordinates = 0;
    _elev.coordinates = 0;
    _elev.dir = 'm';

    
    [WaveHelper secureRelease:&_gpsDevice];
    _gpsDevice=[device retain];
    
        
    if ([_gpsDevice length]==0) {
        [self stop];
        NSLog(@"GPS integration disabled");
        return NO;
    }

    if ([_gpsDevice isEqualToString:@"GPSd"]) [NSThread detachNewThreadSelector:@selector(gpsThreadGPSd:) toTarget:self withObject:Nil];
    else [NSThread detachNewThreadSelector:@selector(gpsThreadSerial:) toTarget:self withObject:Nil];
    return YES;
}

#pragma mark -

- (bool)reliable {
    return _reliable;
}

- (bool)gpsRunning {
    return _gpsThreadUp;
}

- (NSString*) NSCoord {
    if (_ns.coordinates==0) return nil;
    return [NSString stringWithFormat:@"%f%c",_ns.coordinates, _ns.dir];
}

- (NSString*) EWCoord {
    if (_ew.coordinates==0) return nil;
    return [NSString stringWithFormat:@"%f%c",_ew.coordinates, _ew.dir];
}

- (NSString*) ElevCoord {
    if (_elev.coordinates==0) return [NSString stringWithFormat:@"No Elevation Data"];
    //NSLog([NSString stringWithFormat:@"%f",_elev.coordinates]);
    return [NSString stringWithFormat:@"%f%c",_elev.coordinates, _elev.dir]; //don't know if formatting stuff is correct
}

- (waypoint) currentPoint {
    waypoint w;
    
    w._lat =_ns.coordinates * ((_ns.dir=='N') ? 1.0 : -1.0);
    w._long=_ew.coordinates * ((_ew.dir=='E') ? 1.0 : -1.0);
    w._elevation=_elev.coordinates;
    
    return w;
}

- (void) resetTrace {
    [_trace removeAllObjects];
}

- (void)setTraceArray:(NSArray*)trace {
    [_trace release];
    _trace = [trace retain];
}

- (void)setTraceInterval:(int)interval {
    _traceInterval = interval;
}
- (void)setTripmateMode:(bool)mode {
    _tripmateMode = mode;
}

- (void) setCurrentPointNS:(double)ns EW:(double)ew ELV:(double)elv{  //need to add elevation support here
    _ns.dir = (ns<0 ? 'S' : 'N');
    _ew.dir = (ew<0 ? 'W' : 'E');
    
    _ns.coordinates = fabs(ns);
    _ew.coordinates = fabs(ew); 
    
    [WaveHelper secureReplace:&_lastUpdate withObject:[NSDate date]];
    [WaveHelper secureReplace:&_lastAdd withObject:[NSDate date]];
    
    if (abs(ns)>0 && abs(ns)<90 && abs(ew)>0 && abs(ew)<180 && _reliable) {
        [_trace addObject:[NSNumber numberWithDouble:ns]];
        [_trace addObject:[NSNumber numberWithDouble:ew]];
    }
}

- (void)setOnNoFix:(int)onNoFix {
    _onNoFix=onNoFix;
}

- (NSDate*) lastUpdate {
    return _lastUpdate;
}

- (NSArray*) traceArray {
    return _trace;
}

#pragma mark -

bool check_sum(char *s, char h, char l) {
  char checksum;
  unsigned char ref;      /* must be unsigned */

#ifdef PARANOIA
  if(!s)
    return NO;
  if(!*s)
    return NO;
#endif

  checksum = *s++;
  while(*s && *s !='*')
    checksum ^= *s++;

#ifdef PARANOIA
  if(!isxdigit(h))
    return NO;
  if(!isxdigit(l))
    return NO;
  h = (char)toupper(h);
  l = (char)toupper(l);
#endif

  ref =  ((h >= 'A') ? (h -'A' + 10):(h - '0'));
  ref <<= 4;
  ref &= ((l >= 'A') ? (l -'A' + 10):(l - '0'));

  if((char)ref == checksum)
    return YES;             /* ckecksum OK */
  
  return NO;              /* checksum error */
}

int ss(char* inp, char* outp) {
    int x=0;
    
    while(true) {
        if (inp[x]==0) return -1;
        if (inp[x]=='\n') {
            outp[x]=0;
            return x;
        }
        outp[x]=inp[x];
        x++;
    }
    
    return x;
}

- (void)gps_parse:(int) fd {
    int len, valid, x=0, y;
    static int q = 0;
    char cvalid;
    static char gpsin[MAX_GPSBUF_LEN];
    char gpsbuf[MAX_GPSBUF_LEN];
    int ewh, nsh;
    struct _position ns, ew, elev;
    bool updated;
    NSAutoreleasePool* subpool = [[NSAutoreleasePool alloc] init];

    if (_debugEnabled) NSLog(@"GPS read data");
    if (q>=1024) q = 0; //just in case something went wrong
    
    if((len = read(fd, &gpsin[q], MAX_GPSBUF_LEN-q-1)) <= 0) return;
    if (_debugEnabled) NSLog(@"GPS read data returned.");
    
    gpsin[q+len]=0;
    updated = NO;
    
    while (ss(&gpsin[x],gpsbuf)>0) {
        if (_debugEnabled) NSLog(@"GPS record: %s", gpsbuf);//uncommented
        if(_tripmateMode && (!strncmp(gpsbuf, "ASTRAL", 6))) {
            write(fd, "ASTRAL\r", 7);
        } else if(strncmp(gpsbuf, "$GPGGA", 6) == 0) {  //gpsbuf contains GPS fixed data (almost everything poss)
            if (sscanf(gpsbuf, "%*[^,],%*f,%2d%f,%c,%3d%f,%c,%d,%*d,%*f,%f",
		&nsh, &ns.coordinates, &ns.dir,
                &ewh, &ew.coordinates, &ew.dir,
	        &valid, &elev.coordinates)>=7) { // this probably should be == 10 not >= 7  more testing
                		
                if (valid) _reliable = YES;
                else _reliable = NO;
                
                if (_debugEnabled) NSLog(@"GPS data updated.");
                updated = YES;
            }
        } else if(strncmp(gpsbuf, "$GPRMC", 6) == 0) {  //gpsbuf contains Recommended minimum specific GPS/TRANSIT data !!does not include elevation
            if (sscanf(gpsbuf, "%*[^,],%*f,%c,%2d%f,%c,%3d%f,%c,",
                &cvalid, &nsh, &ns.coordinates, &ns.dir,
                &ewh, &ew.coordinates, &ew.dir)==7) {
            
                if (cvalid == 'A') _reliable = YES;
                else _reliable = NO;
                
                if (_debugEnabled) NSLog(@"GPS data updated.");  
                updated = YES;
            }
        } else if(strncmp(gpsbuf, "$GPGLL", 6) == 0) {  //gbsbuf contains Geographical postiion, latitude and longitude only  !!does not include elevation
            if (sscanf(gpsbuf, "%*[^,],%2d%f,%c,%3d%f,%c,%*f,%c",
                &nsh, &ns.coordinates, &ns.dir,
                &ewh, &ew.coordinates, &ew.dir, &cvalid)==7) {
            
                if (cvalid == 'A') _reliable = YES;
                else _reliable = NO;
                
                if (_debugEnabled) NSLog(@"GPS data updated.");  
                updated = YES;
            }
        }
        
        x+=strlen(gpsbuf)+1;
    }
    
    q+=len-x;
    memcpy(gpsbuf,&gpsin[x],q);
    memcpy(gpsin,gpsbuf,q);
    if (q>80) q=0;
    
    if (updated) {
        if ((_reliable)||(_onNoFix==0)) {
            if (ns.dir != 'S') _ns.dir = 'N';
            else _ns.dir = 'S';
            
            if (ew.dir != 'W') _ew.dir = 'E';
            else _ew.dir = 'W';
            
            _ns.coordinates   = nsh + ns.coordinates / 60.0;
            _ew.coordinates   = ewh + ew.coordinates / 60.0;
            _elev.coordinates = elev.coordinates;
            
            [WaveHelper secureReplace:&_lastUpdate withObject:[NSDate date]];
        
            if (([_lastUpdate timeIntervalSinceDate:_lastAdd]>_traceInterval) && (_traceInterval != 100)) {
                y = [_trace count];
                if (y==0 || (fabs([[_trace objectAtIndex:y-2] doubleValue])!=_ns.coordinates) || (fabs([[_trace objectAtIndex:y-1] doubleValue])!=_ew.coordinates)) {
                    [WaveHelper secureReplace:&_lastAdd withObject:[NSDate date]];
        
                    [_trace addObject:[NSNumber numberWithDouble:_ns.coordinates * ((_ns.dir=='N') ? 1.0 : -1.0)]];
                    [_trace addObject:[NSNumber numberWithDouble:_ew.coordinates * ((_ew.dir=='E') ? 1.0 : -1.0)]];
                }
            }
        } else if(_onNoFix==2) {
            _ns.dir = 'N';
            _ew.dir = 'E';
            
            _elev.coordinates = 0;
            _ns.coordinates = 0;
            _ew.coordinates = 0;
            
            [WaveHelper secureReplace:&_lastUpdate withObject:[NSDate date]];
        }
    }
    
    [subpool release];
}

- (void)gpsd_parse:(int) fd {
    int len, valid, q;
    char gpsbuf[MAX_GPSBUF_LEN];
    double ns, ew, elev;
    NSAutoreleasePool* subpool = [[NSAutoreleasePool alloc] init];

    if (_debugEnabled) NSLog(@"GPSd write command");
    
    if (write(fd, "PAS\r\n", 5) < 5) {
        NSLog(@"GPSd write failed");
        return;
    }
    
    if((len = read(fd, &gpsbuf[0], MAX_GPSBUF_LEN)) <= 0) {
        NSLog(@"GPSd read failed");
        return;
    }
    if (_debugEnabled) NSLog(@"GPSd read data returned.");
    
    gpsbuf[0+len]=0;
    
    if (sscanf(gpsbuf, "GPSD,P=%lg %lg,A=%lg,S=%d",
        &ns, &ew, &elev, &valid) ==4) {
                        
        if (valid >= 2) _reliable = YES;
        else _reliable = NO;
        
        if (_debugEnabled) NSLog(@"GPSd data updated.");

        if ((_reliable)||(_onNoFix==0)) {
            if (ns >= 0) _ns.dir = 'N';
            else _ns.dir = 'S';
            
            if (ew >= 0) _ew.dir = 'E';
            else _ew.dir = 'W';
            
            _ns.coordinates   = fabs(ns);
            _ew.coordinates   = fabs(ew);
            _elev.coordinates = elev;
            
            [WaveHelper secureReplace:&_lastUpdate withObject:[NSDate date]];
        
            if (([_lastUpdate timeIntervalSinceDate:_lastAdd]>_traceInterval) && (_traceInterval != 100)) {
                q = [_trace count];
                if (q==0 || (fabs([[_trace objectAtIndex:q-2] doubleValue])!=_ns.coordinates) || (fabs([[_trace objectAtIndex:q-1] doubleValue])!=_ew.coordinates)) {
                    [WaveHelper secureReplace:&_lastAdd withObject:[NSDate date]];
        
                    [_trace addObject:[NSNumber numberWithDouble:_ns.coordinates * ((_ns.dir=='N') ? 1.0 : -1.0)]];
                    [_trace addObject:[NSNumber numberWithDouble:_ew.coordinates * ((_ew.dir=='E') ? 1.0 : -1.0)]];
                }
            }
        } else if(_onNoFix==2) {
            _ns.dir = 'N';
            _ew.dir = 'E';
            
            _elev.coordinates = 0;
            _ns.coordinates = 0;
            _ew.coordinates = 0;
            
            [WaveHelper secureReplace:&_lastUpdate withObject:[NSDate date]];
        }
    } else {
        NSLog(@"GPSd parsing failure");
    }
    
    [subpool release];
}

- (void) continousParse:(int) fd {
    NSDate *date;
    
    while (_gpsShallRun) {
        [self gps_parse:fd];
        //actually once a sec should be enough, but sometimes we dont get any information. so do it more often.
        date = [[NSDate alloc] initWithTimeIntervalSinceNow:0.1];
        [NSThread sleepUntilDate:date];
        [date release];
    }
}

- (void) continousParseGPSd:(int) fd {
    NSDate *date;
    
    while (_gpsShallRun) {
        [self gpsd_parse:fd];
        //actually once a sec should be enough, but sometimes we dont get any information. so do it more often.
        date = [[NSDate alloc] initWithTimeIntervalSinceNow:0.5];
        [NSThread sleepUntilDate:date];
        [date release];
    }
}

- (void)gpsThreadSerial:(id)object {
    NSAutoreleasePool* subpool = [[NSAutoreleasePool alloc] init];
    int fd, handshake;
    struct termios backup;

    _gpsShallRun = NO;
    
    if ([_gpsLock lockBeforeDate:[NSDate dateWithTimeIntervalSinceNow:10]]) {
        _gpsThreadUp = YES;
        _gpsShallRun = YES;
        [_lastUpdate release];
        _lastUpdate = nil;
        
        //NSLog(@"Starting GPS device");
        if((fd = open([_gpsDevice cString], O_RDWR | O_NOCTTY | O_NONBLOCK )) < 0) {
            NSLog(@"error: unable to open gps device: %s", strerror(errno));
        } else if(!isatty(fd)) {
            NSLog(@"error: specified gps device is not a tty: %s", strerror(errno));
        } else if (ioctl(fd, TIOCEXCL) == -1) {
            NSLog(@"error: could not set exclusive flag: %s", strerror(errno));
        } else if (fcntl(fd, F_SETFL, 0) == -1) {
            NSLog(@"error: clearing O_NONBLOCK: %s(%d).\n", strerror(errno), errno);
        } else if(tcgetattr(fd, &backup) != 0) {
            NSLog(@"error: unable to set attributes for gps device: %s", strerror(errno));
        } else if(ioctl(fd, TIOCGETA, &ttyset) < 0) {
            NSLog(@"error: unable to ioctl gps device: %s", strerror(errno));
        } else {
            //NSLog(@"GPS device is open");
            ttyset.c_ispeed = B4800;
            ttyset.c_ospeed = B4800;
            /*
            ttyset.c_cflag &= ~(PARENB | CRTSCTS);
            ttyset.c_cflag |= (CSIZE & CS8) | CREAD | CLOCAL;
            ttyset.c_iflag = ttyset.c_oflag = ttyset.c_lflag = (tcflag_t) 0;
            ttyset.c_lflag |= ICANON;
            */
            ttyset.c_cflag |= CRTSCTS;      // hadware flow on
            ttyset.c_cflag  &= ~PARENB;    // no parity
            ttyset.c_cflag  &= ~CSTOPB;    // one stopbit
            ttyset.c_cflag  &= CSIZE;
            ttyset.c_cflag |= CS8;        // 8N1
            ttyset.c_cflag |= (CLOCAL | CREAD); //enable Localmode, receiver
            ttyset.c_cc[VMIN] = 0;    // set min read chars if 0  VTIME takes over
            ttyset.c_cc[VTIME] = 10;   // wait x ms for charakter

            //options.c_cflag &= ~ ICANON; // canonical input 
            ttyset.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG);

            
            if(ioctl(fd, TIOCSETAF, &ttyset) < 0) {
                NSLog(@"error: unable to ioctl gps device: %s", strerror(errno));
            } else {
                if (ioctl(fd, TIOCSDTR) == -1) { // Assert Data Terminal Ready (DTR)
                    NSLog(@"Error asserting DTR - %s(%d).\n", strerror(errno), errno);
                }
                
                if (ioctl(fd, TIOCCDTR) == -1) { // Clear Data Terminal Ready (DTR) 
                    NSLog(@"Error clearing DTR - %s(%d).\n", strerror(errno), errno);
                }
                
                handshake = TIOCM_DTR | TIOCM_RTS | TIOCM_CTS | TIOCM_DSR;
                if (ioctl(fd, TIOCMSET, &handshake) == -1) { // Set the modem lines depending on the bits set in handshake
                    NSLog(@"Error setting handshake lines - %s(%d).\n", strerror(errno), errno);
                }
                
                if (ioctl(fd, TIOCMGET, &handshake) == -1) { // Store the state of the modem lines in handshake
                    NSLog(@"Error getting handshake lines - %s(%d).\n", strerror(errno), errno);
                }

                NSLog(@"GPS started successfully in serial mode\n");
                [self continousParse:fd];
            }
        }
    
        close(fd);
    
        [_gpsLock unlock];
        _gpsThreadUp = NO;
    } else {
        NSLog(@"GPS LOCKING FAILURE!");
    }
    
    [subpool release];
    return;
}

- (void)gpsThreadGPSd:(id)object {
    int sockd;
    struct sockaddr_in serv_name;
    int status;
    struct hostent *hp;
    UInt32 ip;
    NSUserDefaults *sets;
    const char *hostname;
    NSAutoreleasePool* subpool = [[NSAutoreleasePool alloc] init];
    
    _gpsShallRun = NO;
    
    if ([_gpsLock lockBeforeDate:[NSDate dateWithTimeIntervalSinceNow:10]]) {
        _gpsThreadUp = YES;
        _gpsShallRun = YES;
        [_lastUpdate release];
        _lastUpdate = nil;
        
        sets = [NSUserDefaults standardUserDefaults];
        
        sockd  = socket(AF_INET, SOCK_STREAM, 0);
        if (sockd == -1) {
            NSLog(@"Socket creation failed!");
            goto err;
        }
        
        hostname = [[sets objectForKey:@"GPSDaemonHost"] cString];
        
        if (inet_addr(hostname) != INADDR_NONE) {
            ip = inet_addr(hostname);
        } else {
            hp = gethostbyname(hostname);
            if (hp == NULL) {
                NSLog(@"Could not resolve %s", hostname);
                goto err;
            }
            ip = *(int *)hp->h_addr_list[0];
        }
        
        /* server address */
        serv_name.sin_addr.s_addr = ip;
        serv_name.sin_family = AF_INET;
        serv_name.sin_port = htons([sets integerForKey:@"GPSDaemonPort"]);

        NSLog(@"Connecting to gpsd (%s)",inet_ntoa(serv_name.sin_addr));

        /* connect to the server */
        status = connect(sockd, (struct sockaddr*)&serv_name, sizeof(serv_name));
        
        if (status == -1) {
            NSLog(@"Could not connect to %s port %d", hostname, [sets integerForKey:@"GPSDaemonPort"]);
            goto err;
        }

        NSLog(@"GPS started successfully in GPSd mode.\n");
        // enable raw mode and parse ourselves, quick hack
        //write(sockd, "R\r\n", 3);
        [self continousParseGPSd: sockd];
        close(sockd);
    err:
        [_gpsLock unlock];
        _gpsThreadUp = NO;
    } else {
        NSLog(@"GPS LOCKING FAILURE!");
    }

    [subpool release];
    return;
}

#pragma mark -

- (void)writeDebugOutput:(BOOL)enable {
    _debugEnabled = enable;
}

#pragma mark -

- (void)stop {
    _gpsShallRun=NO;
}

- (void) dealloc {
    _gpsShallRun=NO;
    [_gpsLock release];
    [_gpsDevice release];
    [_lastAdd release];
    [_trace release];
}

@end
