/*
        
        File:			PacketDriver.mm
        Program:		KisMAC
	Author:			Dino Dai Zovi, Michael Roberg
				mick@binaervarianz.de
	Description:		KisMAC is a wireless stumbler for MacOS X.
                
        This file is part of KisMAC.

        Parts of this file has been taken from the viha driver.

    KisMAC is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    KisMAC is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KisMAC; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#import "PacketDriver.h"
#import "WavePacket.h"
#import "WaveHelper.h"
//#import <mach/mach_types.h>
//#import <mach/mach_error.h>
#import <IOKit/IOKitLib.h>
#import <IOKit/IODataQueueClient.h>
#import <Cocoa/Cocoa.h>

#define mach_error_string(x) ""

typedef enum WLUCMethods {
    kWLUserClientOpen,         // kIOUCScalarIScalarO, 0, 0
    kWLUserClientClose,        // kIOUCScalarIScalarO, 0, 0
    kWLUserClientGetChannel,   // kIOUCScalarIScalarO, 0, 1
    kWLUserClientSetChannel,   // kIOUCScalarIScalarO, 1, 0
    kWLUserClientStartCapture, // kIOUCScalarIScalarO, 1, 0
    kWLUserClientStopCapture,  // kIOUCScalarIScalarO, 0, 0
    kWLUserClientSendFrame,
    kWLUserClientStopSendingFrames, // kIOUCScalarIScalarO, 0, 0
    kWLUserClientSetSpecial,
    kWLUserClientLastMethod,
} WLUCMethod;

int kWLUserClientNotify[] = {0, 0xfeedbeef, 0xdeadbeef, 0xbeefc0de, 0 };
int kWLUserClientMap[] = {0, 0xfeedface, 0xdeadface, 0xdeadc0de, 0 };
char driverName[][20] = {"","WLanDriver", "MACJackDriver", "AiroJackDriver", "" };

@implementation PacketDriver

-(kern_return_t) _connect
{
    kern_return_t   kernResult;
    mach_port_t     masterPort;
    io_service_t    serviceObject;
    io_iterator_t   iterator;
    CFDictionaryRef classToMatch;

    kernResult = IOMasterPort(MACH_PORT_NULL, &masterPort);
    if (kernResult != KERN_SUCCESS) {
        printf("IOMasterPort returned %d\n", kernResult);
        return kernResult;
    }
    classToMatch = IOServiceMatching(driverName[_driverID]);

    if (classToMatch == NULL) {
        printf("IOServiceMatching returned a NULL dictionary.\n");
        return kernResult;
    }
    kernResult = IOServiceGetMatchingServices(masterPort,
                                              classToMatch,
                                              &iterator);

    if (kernResult != KERN_SUCCESS) {
        printf("IOServiceGetMatchingServices returned %d\n",
               kernResult);
        return kernResult;
    }

    serviceObject = IOIteratorNext(iterator);
    IOObjectRelease(iterator);
    if (serviceObject != NULL) {
        kernResult = IOServiceOpen(serviceObject, mach_task_self(), 0,
                                   &_userClientPort);

        IOObjectRelease(serviceObject);
        if (kernResult != KERN_SUCCESS) {
            printf("IOServiceOpen %s (%x)\n",
                   mach_error_string(kernResult), kernResult);
            return kernResult;
        }
    }

    
    kernResult = IOConnectMapMemory(_userClientPort,
                                    kWLUserClientMap[_driverID],
                                    mach_task_self(),
                                    (vm_address_t*)&_packetQueue,
                                    &_packetQueueSize,
                                    kIOMapAnywhere);
    if (kernResult != kIOReturnSuccess) {
        //printf("IOConnectMapMemory %s (%x)\n",
         //      mach_error_string(kernResult), kernResult);
        return kernResult;
    }

    _packetQueuePort = IODataQueueAllocateNotificationPort();
    kernResult = IOConnectSetNotificationPort(_userClientPort,
                                              kWLUserClientNotify[_driverID],
                                              _packetQueuePort,
                                              0);
    
    if (kernResult != kIOReturnSuccess) {
        printf("IOConnectSetNotificationPort %s (%x)\n",
               mach_error_string(kernResult), kernResult);
        return kernResult;
    }

    return kernResult;
}

-(kern_return_t) _disconnect {
    return IOServiceClose(_userClientPort);
}

-(id) initWithDriver:(int)driverID {
    if ((driverID>3)||(driverID<1)) return NULL;
    _driverID=driverID;
    
    self=[super init];
    if(self) { 
        kern_return_t kernResult;
    
        kernResult = [self _connect];
        if (kernResult != KERN_SUCCESS) return Nil;
        
        kernResult = IOConnectMethodScalarIScalarO(_userClientPort, kWLUserClientOpen, 0, 0);
        if (kernResult != KERN_SUCCESS) {
            NSLog(@"open: IOConnectMethodScalarIScalarO: %s (driver not loaded?)\n", mach_error_string(kernResult));
            [self release];
            return Nil;
        }
    }
    return self;
}

-(void) dealloc
{
    kern_return_t kernResult;
    kernResult = IOConnectMethodScalarIScalarO(_userClientPort,kWLUserClientClose,0, 0);
    if (kernResult != KERN_SUCCESS) NSLog(@"close: IOConnectMethodScalarIScalarO: %s\n", mach_error_string(kernResult));
    kernResult = [self _disconnect];
    [super dealloc];
}

-(unsigned short) getChannel {
    unsigned short channel = 0;
    kern_return_t kernResult;

    kernResult = IOConnectMethodScalarIScalarO(_userClientPort,
                                               kWLUserClientGetChannel,
                                               0, 1, &channel);
    if (kernResult != KERN_SUCCESS) {
        NSLog(@"getChannel: IOConnectMethodScalarIScalarO: %s\n", mach_error_string(kernResult));
    }

    return channel;
}

-(void) setChannel:(unsigned short) newChannel {
    kern_return_t kernResult;
    
    _channel=newChannel;
    
    kernResult = IOConnectMethodScalarIScalarO(_userClientPort,
                                               kWLUserClientSetChannel,
                                               1, 0, newChannel);
    if (kernResult != KERN_SUCCESS) {
        //NSLog(@"setChannel: IOConnectMethodScalarIScalarO: %s\n", mach_error_string(kernResult));
    }
}

-(void) startCapture:(unsigned short) newChannel {
    kern_return_t kernResult;

    _channel=newChannel;
    
    kernResult = IOConnectMethodScalarIScalarO(_userClientPort,
                                               kWLUserClientStartCapture,
                                               1, 0, newChannel);
    if (kernResult != KERN_SUCCESS) {
        NSLog(@"startCapture: IOConnectMethodScalarIScalarO: %s\n", mach_error_string(kernResult));
    }
}

-(WLFrame*) nextFrame {
    static UInt8  frame[2364];
    UInt32 frameSize = 2364;
    kern_return_t kernResult;

    while (!IODataQueueDataAvailable(_packetQueue)) {
        kernResult = IODataQueueWaitForAvailableData(_packetQueue,
                                                     _packetQueuePort);
        if (kernResult != KERN_SUCCESS) {
            NSLog(@"nextFrame: IODataQueueWaitForAvailableData: %s (%x)\n", mach_error_string(kernResult), kernResult);
            return NULL;
        }
    }

    kernResult = IODataQueueDequeue(_packetQueue, frame, &frameSize);
    if (kernResult != KERN_SUCCESS) {
        NSLog(@"nextFrame: IODataQueueDequeue: %s (%x)\n", mach_error_string(kernResult), kernResult);
        return NULL;
    }
    else 
        return (WLFrame*)frame;
}

-(void) stopCapture {
    kern_return_t kernResult;

    kernResult = IOConnectMethodScalarIScalarO(_userClientPort,
                                               kWLUserClientStopCapture,
                                               0, 0);
    if (kernResult != KERN_SUCCESS) {
        NSLog(@"IOConnectMethodScalarIScalarO: %s\n", mach_error_string(kernResult));
        return;
    }

    while (IODataQueueDataAvailable(_packetQueue))
        IODataQueueDequeue(_packetQueue, NULL, 0);

    return;
}

-(bool) sendFrame:(UInt8*)f withLength:(int) size atInterval:(int)interval {
    static UInt8  frame[2364];
    UInt32 frameSize = 2364;
    kern_return_t kernResult;

    if (_driverID!=2) {
        NSRunAlertPanel(ERROR_TITLE,
            NSLocalizedString(@"The driver that your are using is not able to send raw frames.", "error description"), 
            OK, Nil, Nil);
        return NO;
    }
    memcpy(frame,f,size);
    kernResult = IOConnectMethodScalarIStructureI(_userClientPort,
                                               kWLUserClientSendFrame,
                                               1, frameSize, interval, frame);
    if (kernResult != KERN_SUCCESS) {
        NSLog(@"sendFrame: IOConnectMethodScalarIScalarO: %s\n", mach_error_string(kernResult));
        return NO;
    }
    return YES;
}

-(void) stopSendingFrames {    
    kern_return_t kernResult;

    kernResult = IOConnectMethodScalarIScalarO(_userClientPort,
                                               kWLUserClientStopSendingFrames,
                                               0, 0);
    if (kernResult != KERN_SUCCESS) {
        NSLog(@"IOConnectMethodScalarIScalarO: %s\n", mach_error_string(kernResult));
        return;
    }
    
    return;
}

-(void) setSpecial:(special_set*)s {
    kern_return_t kernResult;

    if (_driverID!=2) return;
    
    kernResult = IOConnectMethodScalarIStructureI(_userClientPort,
                                               kWLUserClientSetSpecial,
                                               0, sizeof(special_set), s);
    if (kernResult != KERN_SUCCESS) {
        NSLog(@"IOConnectMethodScalarIScalarO: %s\n", mach_error_string(kernResult));
        return;
    }

    return;
}

@end
