/*
        
        File:			ScanController.m
        Program:		KisMAC
	Author:			Michael Roberg
				mick@binaervarianz.de
	Description:		KisMAC is a wireless stumbler for MacOS X.
                
        This file is part of KisMAC.

    KisMAC is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    KisMAC is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KisMAC; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#import "ScanController.h"
#import "ScanControllerPrivate.h"
#import <sys/sysctl.h>
#import "WaveHelper.h"
#import "GPSController.h"
#import "WaveClient.h"
#import "CrashReportController.h"

NSString *const KisMACViewItemChanged = @"KisMACViewItemChanged";
NSString *const KisMACCrackDone = @"KisMACCrackDone";
NSString *const KisMACAdvNetViewInvalid = @"KisMACAdvNetViewInvalid";
NSString *const KisMACModalDone = @"KisMACModalDone";
NSString *const KisMACFiltersChanged = @"KisMACFiltersChanged";
NSString *const KisMACStopScanForced = @"KisMACStopScanForced";
NSString *const KisMACNetworkAdded = @"KisMACNetworkAdded";

@implementation ScanController

+ (void)initialize {
    id registrationDict = nil ;

    registrationDict = [NSDictionary dictionaryWithObjectsAndKeys:
        [NSNumber numberWithFloat: 0.25], @"frequence",
        [NSNumber numberWithBool:NO], @"ScanAtStartUp",
        [NSNumber numberWithBool:NO], @"dontAskToSave",
        [NSNumber numberWithBool:NO], @"disableSleepMode",
        [NSNumber numberWithInt:250], @"GeigerSensity",
        [NSNumber numberWithInt:0], @"Voice",
        [WaveHelper colorToInt:[NSColor redColor]], @"CurrentPositionColor",
        [WaveHelper colorToInt:[NSColor redColor]], @"TraceColor",
        [WaveHelper colorToInt:[NSColor blueColor]], @"WayPointColor",
        [WaveHelper colorToInt:[[NSColor greenColor] colorWithAlphaComponent:0.5]], @"NetAreaColorGood",
        [WaveHelper colorToInt:[[NSColor redColor] colorWithAlphaComponent:0.5]], @"NetAreaColorBad",
        [NSNumber numberWithFloat:5.0], @"NetAreaQuality",
        [NSNumber numberWithInt:30], @"NetAreaSensitivity",
        @"None", @"WEPSound",
        @"None", @"noWEPSound",
        @"None", @"GeigerSound",
        @"", @"GPSDevice",
        [NSNumber numberWithInt:2], @"GPSTrace",
        [NSNumber numberWithInt:0], @"GPSNoFix",
        [NSNumber numberWithBool:NO], @"GPSTripmate",
        @"3", @"DownloadMapScale",
        @"<Select a Server>", @"DownloadMapServer",
        [NSNumber numberWithInt:1024], @"DownloadMapWidth",
        [NSNumber numberWithInt:768], @"DownloadMapHeight",
        [NSNumber numberWithFloat:0.0], @"DownloadMapLatitude",
        [NSNumber numberWithFloat:0.0], @"DownloadMapHLongitude",
        @"N", @"DownloadMapNS",
        @"E", @"DownloadMapEW",
        [NSNumber numberWithInt:1], @"TrafficViewShowSSID",
        [NSNumber numberWithInt:0], @"TrafficViewShowBSSID",
        [NSArray array], @"FilterBSSIDList",
        [NSNumber numberWithInt:2947], @"GPSDaemonPort",
        @"localhost", @"GPSDaemonHost",
        [NSNumber numberWithInt:0], @"DebugMode",
        [NSNumber numberWithInt:2], @"WaveNetAvgTime",
        [NSArray array], @"ActiveDrivers",
        nil];

    [[NSUserDefaults standardUserDefaults] registerDefaults:registrationDict];
}

-(id) init {
    self=[super init];
    if (self==Nil) return Nil;

    aNetHierarchVisible = NO;
    _visibleTab = tabNetworks;
    _isSaved = YES;
    _refreshGUI = YES;
    aMS = Nil;
    
    return self;
}

-(void)awakeFromNib {
    static BOOL alreadyAwake = NO;
    NSUserDefaults *sets;
    //NSString *md5;

    if(!alreadyAwake)
        alreadyAwake = YES;
    else
        return;

    [ScanHierach setContainer:_container];
    
    sets=[NSUserDefaults standardUserDefaults];
    
    [WaveHelper setMainWindow:aWindow];
    [WaveHelper setZoomPictureView:MapView];
    
    [aTabView removeTabViewItem:[aTabView tabViewItemAtIndex:[aTabView indexOfTabViewItemWithIdentifier:@"packets"]]];
    if ([[sets objectForKey:@"DebugMode"] intValue] != 1) [[_debugMenu menu] removeItem:_debugMenu];
    
    [LogTable setDoubleAction:@selector(showClient:)];
    
    aDetails=[[aTabView tabViewItemAtIndex:[aTabView indexOfTabViewItemWithIdentifier:@"details"]] retain];
    [aTabView removeTabViewItem:[aTabView tabViewItemAtIndex:[aTabView indexOfTabViewItemWithIdentifier:@"details"]]];
    
    [self menuSetEnabled:NO menu:aNetworkMenu];
    [[_showNetInMap menu] setAutoenablesItems:NO];
    [_showNetInMap setEnabled:NO];
    [trafficTimePopUp setTransparent:YES];
    [trafficModePopUp setTransparent:YES];

    [aWindow makeKeyAndOrderFront:self];
    /*md5=[[NSBundle mainBundle] resourcePath];
    [[NSFileManager defaultManager] changeCurrentDirectoryPath:[[NSBundle mainBundle] bundlePath]];
    if (system([[NSString stringWithFormat:@"\"%@/md5sum\" -b -c \"%@/checkfile\"", md5, md5] cString]))
        NSLog(@"WARNING Self-Verification failed! A system which verifies KisMACs application file  returned an error. You may ignore it, if you have \
        no problems, however if there are unexplained crashes, we recommend you a reinstallation of KisMAC.");
    */
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(updateViewItems:)     name:KisMACViewItemChanged      object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(modalDone:)           name:KisMACCrackDone            object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(advNetViewInvalid:)   name:KisMACAdvNetViewInvalid    object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(modalDone:)           name:KisMACModalDone            object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(stopScanForced:)      name:KisMACStopScanForced       object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(networkAdded:)        name:KisMACNetworkAdded         object:nil];

    NSLog(@"KiMAC startup done. Homedir is %@ NSAppKitVersionNumber: %f",[[NSBundle mainBundle] bundlePath], NSAppKitVersionNumber);
}

- (IBAction)updatePrefs:(id)sender {
    NSUserDefaults *sets;
    int x;
    NSString* key;
    [prefsWindow orderOut:nil];
    //[aWindow makeKeyAndOrderFront:nil];

    sets=[NSUserDefaults standardUserDefaults];

    if ([sets integerForKey:@"GeigerSensity"]<1) [sets setInteger:1 forKey:@"GeigerSensity"];
    key=[sets objectForKey:@"GeigerSound"];
    if ([key isEqualToString:@"None"]) key=Nil;
    [scanner setGeigerInterval:[sets integerForKey:@"GeigerSensity"] sound:key];

    if ([sets floatForKey:@"frequence"]<0.2) [sets setFloat:0.25 forKey:@"frequence"];
    [scanner setFrequency:[sets floatForKey:@"frequence"]];

    if ([sets integerForKey:@"disableSleepMode"]) {
        [WaveHelper runScript:@"nosleep_enable.sh"];
        aOurSleepMode = YES;
    } else if (([sets integerForKey:@"disableSleepMode"]==0) && [WaveHelper isServiceAvailable:"Insomnia"]) {
        [WaveHelper runScript:@"nosleep_disable.sh"];
        aOurSleepMode = NO;
    }
    
    [WaveHelper initGPSControllerWithDevice: [sets objectForKey:@"GPSDevice"]];
    
    switch ([[sets objectForKey:@"GPSTrace"] intValue]) {
        case 0: x = 100; break;
        case 1: x = 20;  break;
        case 2: x = 10;  break;
        case 3: x = 5;   break;
        case 4: x = 1;   break;
        case 5: x = 0;   break;
        default: x = 100;
    }
    
    [[WaveHelper gpsController] setTraceInterval:x];
    [[WaveHelper gpsController] setOnNoFix:[[sets objectForKey:@"GPSNoFix"] intValue]];
    [[WaveHelper gpsController] setTripmateMode:[[sets objectForKey:@"GPSTripMate"] boolValue]];
    
    [self updateChannelMenu];
}

#pragma mark -

//#define USEPASTEOUT
- (IBAction)updateLogTable:(id)sender complete:(bool)complete {
#ifndef USEPASTEOUT    
    int row;
    int i;
    [TrafficView setNeedsDisplay:YES];

    if ([_container count]!=[LogTable numberOfRows]) complete = YES;

    [_channelProg setChannel:[[WaveHelper driverWithName:_whichDriver] getChannel]];

    if (_visibleTab == tabNetworks) {
        if (aLastSorted) [_container sortWithShakerByColumn:aLastSorted order:aAscending];
        
        if (complete) {
            [LogTable reloadData];
            if (aDetailsPane) [aInfoController reloadData];
            if ([_container netAtIndex:_selectedRow] != aCurNet) { //we lost our selected network
                for (i = [_container count]; i>=0; i--)
                if ([_container netAtIndex:i] == aCurNet) {
                    _selectedRow = i;
                    [LogTable selectRow:i byExtendingSelection:NO];
                    break;
                }
            }
        }
        else {
            row = [_container nextChangedRow:0xFFFFFFFF];
            while (row != 0xFFFFFFFF) {
                if ([_container netAtIndex:row] == aCurNet) {
                    if (aDetailsPane) [aInfoController reloadData];
                    _selectedRow = row;
                    [LogTable selectRow:row byExtendingSelection:NO];
                }
                [LogTable displayRect:[LogTable rectOfRow:row]];
                row = [_container nextChangedRow:row];
            }
        }
    } else if (_visibleTab == tabDetails) {
        if (complete) {
            [aInfoController reloadData];
            if ([_container netAtIndex:_selectedRow] != aCurNet) { //we lost our selected network
                for (i = [_container count]; i>=0; i--)
                if ([_container netAtIndex:i] == aCurNet) {
                    _selectedRow = i;
                    [LogTable selectRow:i byExtendingSelection:NO];
                    break;
                }
            }
        } else {
            row = [_container nextChangedRow:0xFFFFFFFF];
            while (row != 0xFFFFFFFF) {
                if ([_container netAtIndex:row] == aCurNet) {
                    [aInfoController reloadData];
                    _selectedRow = row;
                    [LogTable selectRow:row byExtendingSelection:NO];
                }
                row = [_container nextChangedRow:row];
            }
        }
    }
#else
    NSPasteboard *board;
    NSMutableString *outString;
    NSString *lineString;
    WaveNet *n;
    WaveClient *cl;
    unsigned int i, d;
    NSString *type;
    NSString *wep;
    NSDictionary *c;
    NSArray *k;
    
    board = [NSPasteboard pasteboardWithName:NSGeneralPboard];
    outString = [NSMutableString string];

    for(i=0; i<[nets count]; i++) {
        n = [aNets objectForKey:[nets objectAtIndex:i]];
        switch ([n isWep]) {
            case 2: wep = @"YES";
            case 1: wep = @"NO";
            case 0: wep = @"NA";
        }
        switch ([n type]) {
            case 0: type =@"NA";
            case 1: type = @"ad-hoc";
            case 2: type = @"managed";
            case 3: type = @"tunnel";
            case 4: type = @"probe";
        }
    
        lineString = [NSString stringWithFormat:@"ap %i %@ %@ %@ %@ %@ %i %i %i %@ %@\n",[n channel],[n BSSID],[n SSID], type, wep, [n getVendor], [n curSignal],[n maxSignal], [n packets], [n data], [[n date] description]];
        [outString appendString:lineString];
        
        k = [n getClientKeys];
        c = [n getClients];
        for(d=0;d<[k count];d++) {
            wep = [k objectAtIndex:d];
            cl = [c objectForKey:wep];
            lineString = [NSString stringWithFormat:@"client %@ %@ %@ %@ %i %@\n", wep, [cl vendor], [cl sent], [cl recieved], [cl curSignal], [[cl date] description] ];
            [outString appendString:lineString];
        }
    }
    lineString = [board stringForType:NSStringPboardType];
    [board declareTypes:[NSArray arrayWithObject:NSStringPboardType] owner:nil];
    [board setString:outString forType:NSStringPboardType];
#endif    
}

- (void)updateViewItems:(NSNotification*)note {
    if (!_refreshGUI) return;
    [self performSelectorOnMainThread:@selector(doUpdateViewItems:) withObject:nil waitUntilDone:NO];
}

- (void)doUpdateViewItems:(id)anObject {
    [_container refreshView];
    if (aLastSorted) [_container sortWithShakerByColumn:aLastSorted order:aAscending];
    
    if (aNetHierarchVisible) {
        [ScanHierach updateTree];
        [aOutView reloadData];
    }
}

#pragma mark -

- (void)stopScanForced:(NSNotification*)note {
    [self stopScan];
}

-(void)updateGPSData:(NSTimer*)timer {
    GPSController *gps;
    
    gps = [WaveHelper gpsController];
    if ([gps lastUpdate])
        [aGPSStatus setStringValue:[NSString stringWithFormat:@"%@ %@, %@, %@: %@ @ %@%@", 
            NSLocalizedString(@"Last Position", "GPS status string."), 
            [gps NSCoord],[gps EWCoord],
            NSLocalizedString(@"Elevation", "GPS status string."), 
            [gps ElevCoord],[gps lastUpdate],
            [gps reliable] ? @"" : NSLocalizedString(@" Position is not reliable!", "GPS status string. Needs leading space")]];
    else if ([(NSString*)[[NSUserDefaults standardUserDefaults] objectForKey:@"GPSDevice"] length]) {
        if ([gps gpsRunning]) [aGPSStatus setStringValue:NSLocalizedString(@"GPS subsystem works, but there is no data.", "GPS status string")];
        else  [aGPSStatus setStringValue:
            NSLocalizedString(@"GPS not working", "LONG GPS status string with informations howto debug")
            //@"GPS subsystem is not working. See log file for more details."
            ];
    }
    else
        [aGPSStatus setStringValue:
            NSLocalizedString(@"GPS disabled", "LONG GPS status string with informations where to enable")
            //@"GPS subsystem is disabled. You have to select a device in the preferences window."
            ];
    [MapView setCurrentPoint:[[WaveHelper gpsController] currentPoint]];
}

- (void)handleGPSDeviceDeath:(NSNotification*)note {
    [aGPSTimer invalidate];
    aGPSTimer = Nil;
    [aGPSStatus setStringValue:NSLocalizedString(@"GPS device has been disconnected.", "GPS status string")];
    [WaveHelper initGPSControllerWithDevice:@""];
}

#pragma mark -

- (id) tableView:(NSTableView *) aTableView
objectValueForTableColumn:(NSTableColumn *) aTableColumn
             row:(int) rowIndex
{
    WaveNet *net = [_container netAtIndex:rowIndex];
    NSString *s = [aTableColumn identifier];
    
    if ([s isEqualToString:@"id"]) return [NSString stringWithFormat:@"%i", [net netID]];
    if ([s isEqualToString:@"ssid"]) return [net SSID];
    if ([s isEqualToString:@"bssid"]) return [net BSSID];
    if ([s isEqualToString:@"lastseen"]) return [net date];
    if ([s isEqualToString:@"signal"]) return [NSString stringWithFormat:@"%i", [net curSignal]];
    if ([s isEqualToString:@"avgsignal"]) return [NSString stringWithFormat:@"%i", [net avgSignal]];
    if ([s isEqualToString:@"maxsignal"]) return [NSString stringWithFormat:@"%i", [net maxSignal]];
    if ([s isEqualToString:@"channel"]) return [NSString stringWithFormat:@"%i", [net channel]];
    if ([s isEqualToString:@"packets"]) return [NSString stringWithFormat:@"%i", [net packets]];
    if ([s isEqualToString:@"data"]) return [net data];
    if ([s isEqualToString:@"wep"]) {
        switch ([net isWep]) {
            case 4: return NSLocalizedString(@"WPA", "table description");
            case 3: return NSLocalizedString(@"WEP-40", "table description");
            case 2: return NSLocalizedString(@"WEP", "table description");
            case 1: return NSLocalizedString(@"NO", "table description");
            case 0: return @"";
        }
    }
    if ([s isEqualToString:@"type"]) {
        switch ([net type]) {
            case 0: return @"";
            case 1: return NSLocalizedString(@"ad-hoc", "table description");
            case 2: return NSLocalizedString(@"managed", "table description");
            case 3: return NSLocalizedString(@"tunnel", "table description");
            case 4: return NSLocalizedString(@"probe", "table description");
        }
    }
    return @"unknown row";
}
- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
    return [_container count];
}

- (void)tableViewSelectionDidChange:(NSNotification *)aNotification {
    if ([LogTable selectedRow]<0) [self hideDetails]; 
    else [self selectNet:[_container netAtIndex:[LogTable selectedRow]]];
}

- (void)tableView:(NSTableView*)tableView mouseDownInHeaderOfTableColumn:(NSTableColumn *)tableColumn {
    NSString *ident = [tableColumn identifier];
    
    if(![tableView isEqualTo:LogTable]) return;

    if ((aLastSorted) && ([aLastSorted isEqualToString:ident])) {
        if (aAscending) aAscending=NO;
        else {
            [aLastSorted release];
            aLastSorted = Nil;
            [tableView setIndicatorImage:Nil inTableColumn:tableColumn];
            [tableView setHighlightedTableColumn:Nil];
            [tableView reloadData];
            return;
        }
    } else {
        aAscending=YES;
        if (aLastSorted) {
            [tableView setIndicatorImage:nil inTableColumn:[tableView tableColumnWithIdentifier:aLastSorted]];
            [aLastSorted release];
        }
        aLastSorted=[ident retain];
    }
    
    if (aAscending)
        [tableView setIndicatorImage:[NSImage imageNamed:@"NSAscendingSortIndicator"] inTableColumn:tableColumn];
    else 
        [tableView setIndicatorImage:[NSImage imageNamed:@"NSDescendingSortIndicator"] inTableColumn:tableColumn];
    [tableView setHighlightedTableColumn:tableColumn];
    //[tableView reloadData];
    //[_container sortByColumn:ident order:aAscending];
    
    [self updateLogTable:self complete:YES];
}

#pragma mark -

- (int)outlineView:(NSOutlineView *)outlineView numberOfChildrenOfItem:(id)item {
    return (item == nil) ? 3 : [item numberOfChildren];
}
- (BOOL)outlineView:(NSOutlineView *)outlineView isItemExpandable:(id)item {
    return ([item numberOfChildren] != -1);
}
- (id)outlineView:(NSOutlineView *)outlineView child:(int)index ofItem:(id)item {
    if (item!=Nil) return [item childAtIndex:index];
    else {
        return [ScanHierach rootItem:_container index:index];
    }
}
- (id)outlineView:(NSOutlineView *)outlineView objectValueForTableColumn:(NSTableColumn *)tableColumn byItem:(id)item {
    return (item == nil) ? nil : (id)[item nameString];
}
- (BOOL)outlineView:(NSOutlineView *)outlineView shouldEditTableColumn:(NSTableColumn *)tableColumn item:(id)item {
    return NO;
}
- (BOOL)outlineView:(NSOutlineView *)outlineView shouldSelectItem:(id)item {
    unsigned int tmpID[6], i;
    unsigned char ID[6];
    if (item==nil) return YES;
    
    switch ([(ScanHierach*)item type]) {
        case 99: //BSSID selector
            if (sscanf([[item identKey] cString], "%2X%2X%2X%2X%2X%2X", &tmpID[0], &tmpID[1], &tmpID[2], &tmpID[3], &tmpID[4], &tmpID[5])!=6) 
                NSLog(@"Error could not decode ID %@!", [item identKey]);
            for (i=0; i<6; i++) ID[i] = tmpID[i];
            
            [self showDetailsFor:[_container netForKey:ID]];
            break;
        case 1: //topitems
        case 2:
        case 36:
            [self hideDetails];
            [aTabView selectFirstTabViewItem:self];
            [_container setViewType:0 value:nil];
            if (aLastSorted) [_container sortByColumn:aLastSorted order:aAscending];
            break;
        case 3: //SSID selectors
            [self hideDetails];
            [aTabView selectFirstTabViewItem:self];
            [_container setViewType:2 value:[(ScanHierach*)item nameString]];
            if (aLastSorted) [_container sortByColumn:aLastSorted order:aAscending];
            break;
        case 37: //Crypto selectors
        case 38:
        case 40:
            [self hideDetails];
            [aTabView selectFirstTabViewItem:self];
            [_container setViewType:3 value:[NSNumber numberWithInt:[(ScanHierach*)item type]-36]];
            if (aLastSorted) [_container sortByColumn:aLastSorted order:aAscending];
            break;
        default: //channel selectors left
            [self hideDetails];
            [aTabView selectFirstTabViewItem:self];
            [_container setViewType:1 value:[NSNumber numberWithInt:[(ScanHierach*)item type]-20]];
            if (aLastSorted) [_container sortByColumn:aLastSorted order:aAscending];
    }
    [LogTable reloadData];
    return YES;
}

#pragma mark -

- (void)tabView:(NSTabView *)tabView willSelectTabViewItem:(NSTabViewItem *)tabViewItem {
    
    if (![[tabViewItem identifier] isEqualToString:@"details"]) {
        _visibleTab = tabNetworks;
        //[aTabView setDelegate:Nil];
        //[self hideDetails];
        //[aTabView selectTabViewItem:tabViewItem];
        //[aTabView setDelegate:self];
    }
    
    if([[tabViewItem identifier] isEqualToString:@"traffic"]) {
        _visibleTab = tabTraffic;
        [trafficTimePopUp setTransparent:NO];
        [trafficModePopUp setTransparent:NO];
        [trafficTimePopUp setEnabled:YES];
        [trafficModePopUp setEnabled:YES];
    } else {
        [trafficTimePopUp setTransparent:YES];
        [trafficModePopUp setTransparent:YES];
        [trafficTimePopUp setEnabled:NO];
        [trafficModePopUp setEnabled:NO];
    }
    
    if([[tabViewItem identifier] isEqualToString:@"map"]) {
        _visibleTab = tabMap;
        if (!aGPSTimer)
            aGPSTimer = [NSTimer scheduledTimerWithTimeInterval:0.25 target:self selector:@selector(updateGPSData:) userInfo:Nil repeats:TRUE];
        [MapView setVisible:YES];
    } else {
        [aGPSTimer invalidate];
        aGPSTimer = Nil;
        [MapView setVisible:NO];
    }
}
#pragma mark -


- (IBAction)showInfo:(id)sender {
    NSSize contentSize;
    
    if (_visibleTab == tabDetails) return;
    
    if (!aDetailsDrawer) {
        contentSize = NSMakeSize(200, 250);
        aDetailsDrawer = [[NSDrawer alloc] initWithContentSize:contentSize preferredEdge:NSMaxXEdge];
        [aDetailsDrawer setParentWindow:aWindow];
        [aDetailsDrawer setContentView:detailsView];
        [aDetailsDrawer setMinContentSize:NSMakeSize(50, 50)];

        [aDetailsDrawer setLeadingOffset:50];
        [aDetailsDrawer setTrailingOffset:[aWindow frame].size.height-300];

        [aDetailsDrawer setMaxContentSize:NSMakeSize(200, 300)];
    }
    aDetailsPane=YES;
    [aInfoController setDetails:YES];
    [aDetailsDrawer openOnEdge:NSMaxXEdge];
}

- (IBAction)showClient:(id)sender {
    if ([LogTable selectedRow]>=0) [self showDetailsFor:[_container netAtIndex:[LogTable selectedRow]]];
}

- (IBAction)showNetHierarch:(id)sender {
    if (!aNetHierarchVisible) {
        [sender setTitle: NSLocalizedString(@"Hide Hierarchy", "menu item")];
        [aNetHierarchDrawer openOnEdge:NSMinXEdge];
        
        [ScanHierach updateTree];
        [aOutView reloadData];
    } else {
        [sender setTitle: NSLocalizedString(@"Show Hierarchy", "menu item. needs to be same as MainMenu.nib")];
        [aNetHierarchDrawer close];
    }
    aNetHierarchVisible= !aNetHierarchVisible;
}

- (IBAction)restartGPS:(id)sender {
    NSString *lDevice;
    
    lDevice=[[NSUserDefaults standardUserDefaults] objectForKey:@"GPSDevice"];
    if ((lDevice!=nil)&&(![lDevice isEqualToString:@""])) {
        [aGPSStatus setStringValue:NSLocalizedString(@"Resetting GPS subsystem...", "GPS status string")];
        [WaveHelper initGPSControllerWithDevice: lDevice];
    } else {
        [aGPSStatus setStringValue:NSLocalizedString(@"GPS disabled", "LONG GPS status string with informations where to enable")];
    }
}

- (IBAction)scan:(id)sender {
    [LogTable reloadData];
    if ([[sender title] isEqualToString:@"Scan"]) [self startScan];
    else [self stopScan];
}


- (IBAction)injectPackets:(id)sender {
    bool res;
    //NSRunAlertPanel(@"Warning", @"I am not quiet happy with this function. It currently needs two cards. One Airport Card (needs to be set as default) and one PrismII Card. Orinoco cards will not work!\n\n What does packet reinjection?\n In this version KisMAC will re-send ARP-packets into an WEP-enabled network. This will cause a response. Basically it generates a lot of traffic, which will enable us to break into not busy networks. Do a deauthentication before injection, in order to make sure that there have been a couple of arp-packets.\n\nIf this works for you please drop me a mail.",
    //NULL, NULL, NULL);
    //return;    
    if ([aInjPacketsMenu state]==NSOffState && [self startActiveAttack]) {
        //either we are already active or we have to load the driver
        if (!aScanning) [self startScan];
        
        //[self startCrack];
        res=[scanner tryToInject:aCurNet];
        //[NSApp endModalSession:aMS];
        //[[aImportController window] close];
        //[aImportController release];
        //aImportController=Nil;
        if (res==NO) {
            NSBeginAlertSheet(NSLocalizedString(@"Reinjection unsuccessful", "Error dialog title"),
                OK, NULL, NULL, aWindow, self, NULL, NULL, NULL, 
                NSLocalizedString(@"Reinjection failure description", "LONG text about what might have gone wrong with the injection")
                //@"KisMAC was unable to start a reinjection attack. Possible reasons are:\n\n1. You do not use the MACJack driver.\n2. You did not collect enough data packets\n3. You selected injection attack on a non-WEP network.\n4. The network is not an managed network.\n5.You had bad luck."
                );
            [self stopActiveAttacks];
            return;
        }
        [aInjPacketsMenu setState:NSOnState];
        [aInjPacketsMenu setTitle:[NSLocalizedString(@"Reinjecting into ", "menu item") stringByAppendingString:[aCurNet BSSID]]];
    } else {
        [self stopActiveAttacks];
    }
}
- (IBAction)deautheticateNetwork:(id)sender {
    if ([aDeauthMenu state]==NSOffState && [self startActiveAttack] && [scanner deauthenticateNetwork:aCurNet]) {
        [aDeauthMenu setState:NSOnState];
        [aDeauthMenu setTitle:[NSLocalizedString(@"Deauthenticating ", "menu item") stringByAppendingString:[aCurNet BSSID]]];
    } else {
        [self stopActiveAttacks];
    }
}
- (IBAction)authFloodNetwork:(id)sender {
    if ([_authFloodMenu state]==NSOffState && [self startActiveAttack] && [scanner authFloodNetwork:aCurNet]) {
        [_authFloodMenu setState:NSOnState];
        [_authFloodMenu setTitle:[NSLocalizedString(@"Flooding ", "menu item") stringByAppendingString:[aCurNet BSSID]]];
    } else {
        [self stopActiveAttacks];
    }
}

- (IBAction)bruteASCILettersCrack40bit:(id)sender {
    if ([[aCurNet weakPacketsLog] count] < 5) {
        [self showNeedMorePacketsDialog];
        return;
    }
    if (aCurNet==Nil) return; 
    
    _crackType = 2;
    [self startCrack];
    [aImportController setMax:127];
    [aCurNet crack40WithBruteForce:aImportController charset:3];
    [self runCrack];
}
- (IBAction)bruteLowLettersCrack40bit:(id)sender {
    if ([[aCurNet weakPacketsLog] count] < 5) {
        [self showNeedMorePacketsDialog];
        return;
    }
    if (aCurNet==Nil) return;
    
    _crackType = 2;
    [self startCrack];
    [aImportController setMax:26];
    [aCurNet crack40WithBruteForce:aImportController charset:2];
    [self runCrack];
}
- (IBAction)bruteSelectedCrack40bit:(id)sender {
    if ([[aCurNet weakPacketsLog] count] < 5) {
        [self showNeedMorePacketsDialog];
        return;
    }
    if (aCurNet==Nil) return;
    
    _crackType = 2;
    [self startCrack];
    [aImportController setMax:62];
    [aCurNet crack40WithBruteForce:aImportController charset:1];
    [self runCrack];
}
- (IBAction)bruteCrack40bit:(id)sender {
    if ([[aCurNet weakPacketsLog] count] < 5) {
        [self showNeedMorePacketsDialog];
        return;
    }
    if (aCurNet==Nil) return;
    
    _crackType = 2;
    [self startCrack];
    [aImportController setMax:256];
    [aCurNet crack40WithBruteForce:aImportController charset:0];
    [self runCrack];
}
- (IBAction)weakCrackGeneric:(id)sender {
    if (([aCurNet isWep]==2)&&([aCurNet weakPackets]<5)) {
        [self showNeedMoreWeakPacketsDialog];
    } else {
        if (aCurNet==Nil) return;
        
        _crackType = 1;
        [self startCrack];
        [aImportController setMax:32];
        [aCurNet crackWithKeyByteLength:5 breath:4 import:aImportController];
        [aCurNet crackWithKeyByteLength:13 breath:2 import:aImportController];
        
        [self runCrack];
    }
}
- (IBAction)weakCrack40bit:(id)sender {
    if (([aCurNet isWep]==2)&&([aCurNet weakPackets]<5)) {
        [self showNeedMoreWeakPacketsDialog];
    } else {
        if (aCurNet==Nil) return;
        
        _crackType = 1;
        [self startCrack];
        [aImportController setMax:16];
        [aCurNet crackWithKeyByteLength:5 breath:4 import:aImportController];
        [self runCrack];
    }
}
- (IBAction)weakCrack104bit:(id)sender {
    if (([aCurNet isWep]==2)&&([aCurNet weakPackets]<5)) {
        [self showNeedMoreWeakPacketsDialog];
    } else {
        if (aCurNet==Nil) return;
        
        _crackType = 1;
        [self startCrack];
        [aImportController setMax:17];
        [aCurNet crackWithKeyByteLength:13 breath:2 import:aImportController];
        [self runCrack];
    }
}
- (IBAction)wordCrackApple40bit:(id)sender {
    if ([[aCurNet weakPacketsLog] count] < 5) {
        [self showNeedMorePacketsDialog];
        return;
    }
    if (aCurNet==Nil) return;
    
    _crackType = 2;
    [self startCrack];
    [aCurNet crackWithWordlistUseCipher:1 import:aImportController];
    [self runCrack];
}
- (IBAction)wordCrackApple104bit:(id)sender {
    if ([[aCurNet weakPacketsLog] count] < 5) {
        [self showNeedMorePacketsDialog];
        return;
    }
    if (aCurNet==Nil) return;
    
    _crackType = 2;
    [self startCrack];
    [aCurNet crackWithWordlistUseCipher:2 import:aImportController];
    [self runCrack];
}

- (IBAction)wordCrackMD5104bit:(id)sender {
    if ([[aCurNet weakPacketsLog] count] < 5) {
        [self showNeedMorePacketsDialog];
        return;
    }
    if (aCurNet==Nil) return;
    
    _crackType = 2;
    [self startCrack];
    [aCurNet crackWithWordlistUseCipher:3 import:aImportController];
    [self runCrack];
}

- (IBAction)selChannel:(id)sender {
    WaveDriver *wd;
    NSMutableDictionary *md;
    int y;
    int newChannel = [[[sender title] substringFromIndex:8] intValue];
    
    wd = [WaveHelper driverWithName:_whichDriver];
    if (!wd) {
        NSLog(@"Error: invalid driver selected");
        return;
    }
    
    md = [[wd configuration] mutableCopy];
    for(y=1; y<15; y++)
        [md setObject:[NSNumber numberWithInt:(y==newChannel) ? 1 : 0] forKey:[NSString stringWithFormat:@"useChannel%.2i",y]];
  
    [wd setConfiguration: md];
    [md release];

    [self updateChannelMenu];
}

- (IBAction)selChannelRange:(id)sender {
    WaveDriver *wd;
    NSMutableDictionary *md;
    int y;
    
    wd = [WaveHelper driverWithName:_whichDriver];
    if (!wd) {
        NSLog(@"Error: invalid driver selected");
        return;
    }
    
    md = [[wd configuration] mutableCopy];
    if ([[sender title] isEqualToString:NSLocalizedString(@"All FCC/IC Channels (1-11)", "menu item. needs to be the same as in MainMenu.nib")]) {
        for(y=1; y<=11; y++)
            [md setObject:[NSNumber numberWithInt:1] forKey:[NSString stringWithFormat:@"useChannel%.2i", y]];

        [md setObject:[NSNumber numberWithInt:0] forKey:[NSString stringWithFormat:@"useChannel%.2i", 12]];
        [md setObject:[NSNumber numberWithInt:0] forKey:[NSString stringWithFormat:@"useChannel%.2i", 13]];
     } else {
        for(y=1; y<=13; y++)
            [md setObject:[NSNumber numberWithInt:1] forKey:[NSString stringWithFormat:@"useChannel%.2i", y]];
    }
    
    [wd setConfiguration: md];
    [md release];
    
    [self updateChannelMenu];
}

- (IBAction)selDriver:(id)sender {
    NSUserDefaults *sets;

    sets = [NSUserDefaults standardUserDefaults];
    [sets setObject:[sender title] forKey:@"whichDriver"];
    [self updateChannelMenu];
}

- (IBAction)setAutoAdjustTimer:(id)sender {
    WaveDriver *wd;
    NSMutableDictionary *md;
    
    wd = [WaveHelper driverWithName:_whichDriver];
    if (!wd) {
        NSLog(@"Error: invalid driver selected");
        return;
    }
    
    md = [[wd configuration] mutableCopy];
    [md setObject:[NSNumber numberWithBool:(([sender state]==NSOffState) ? YES : NO)] forKey:@"autoAdjustTimer"];
 
    [wd setConfiguration: md];
    [md release];
    
    [self updateChannelMenu];

}

- (IBAction)showCurNetArea:(id)sender {
   if ([sender state] == NSOffState) {
        [self stopScan];
        aImportController = [[ImportController alloc] initWithWindowNibName:@"Crack"];
        [[aImportController window] setFrameUsingName:@"aKisMAC_Import"];
        [[aImportController window] setFrameAutosaveName:@"aKisMAC_Import"];
        [aImportController setTitle:NSLocalizedString(@"KisMAC - Caching Map", "Title of busy dialog")];
        [aImportController showWindow:self];
        [WaveHelper setImportController:aImportController];
        
        _doModal = YES;
        
        if ([_showAllNetsInMap state] == NSOnState) [self showAllNetArea:_showAllNetsInMap];
        [[WaveHelper zoomPictureView] showAdvNet:aCurNet];
        [sender setTitle:[NSLocalizedString(@"Show Net Area of ", "menu item") stringByAppendingString:[aCurNet BSSID]]];
        [sender setState: NSOnState];
        if (aMS) return;
        aMS=[NSApp beginModalSessionForWindow:[aImportController window]];
        
        for (;;) {
            if (([NSApp runModalSession:aMS] != NSRunContinuesResponse) || !_doModal) break;
        }
        [NSApp endModalSession:aMS];
        aMS=Nil;
        
        [[aImportController window] close];
        [aImportController stopAnimation];
        
        if ([aImportController canceled]) {
            [[WaveHelper zoomPictureView] showAdvNet:Nil];
            [sender setTitle:@"Show Net Area"];
            [sender setState: NSOffState];
        }
        [aImportController release];
        aImportController=Nil;

    } else {
        [self clearAreaMap];
    }
}

- (IBAction)showAllNetArea:(id)sender {
    NSMutableArray *a;
    unsigned int i;
    
    if ([sender state] == NSOffState) {
        [self stopScan];
        aImportController = [[ImportController alloc] initWithWindowNibName:@"Crack"];
        [[aImportController window] setFrameUsingName:@"aKisMAC_Import"];
        [[aImportController window] setFrameAutosaveName:@"aKisMAC_Import"];
        [aImportController setTitle: NSLocalizedString(@"KisMAC - Caching Map", "Title of busy dialog")];
        [aImportController showWindow:self];
        [WaveHelper setImportController:aImportController];
        
        _doModal = YES;
        
        if ([_showNetInMap state] == NSOnState) [self showCurNetArea:_showNetInMap];
    
        a = [[NSMutableArray alloc] init];
        for (i=0; i<[_container count]; i++) [a addObject:[_container netAtIndex:i]];
        [[WaveHelper zoomPictureView] showAdvNets:[NSArray arrayWithArray:a]];
        [a release];

        [sender setState: NSOnState];

        if (aMS) return;
        aMS=[NSApp beginModalSessionForWindow:[aImportController window]];
        
        for (;;) {
            if (([NSApp runModalSession:aMS] != NSRunContinuesResponse) || !_doModal) break;
        }
        [NSApp endModalSession:aMS];
        aMS=Nil;
        
        [[aImportController window] close];
        [aImportController stopAnimation];
        if ([aImportController canceled]) {
            [[WaveHelper zoomPictureView] showAdvNet:Nil];
            [sender setState: NSOffState];
        }
        [aImportController release];
        aImportController=Nil;
    } else {
        [self clearAreaMap];
    }
}

#pragma mark -

-(void) dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];

    [aImportController stopAnimation];
    [aImportController release];
    [aStatusItem release];
    [aStatusBar release];
    [_fileName release];
    [super dealloc];
}

#pragma mark -

- (BOOL)application:(NSApplication *)theApplication openFile:(NSString *)filename {
    NSModalSession lMS;
    
    if ([[[filename pathExtension] lowercaseString] isEqualToString:@"kismac"]) {
        [_fileName release];
        _fileName=[filename retain];
        [self clearAllNetworks:self];
        
        [self showBusy:@selector(performOpenFile:) withArg:_fileName];
        
        [self updateLogTable:self complete:YES];
        [self refreshScanHierarch];
        _isSaved = YES;
        return YES;
    } else if ([[[filename pathExtension] lowercaseString] isEqualToString:@"kismap"]) {
        aImportController = [[ImportController alloc] initWithWindowNibName:@"Import"];
        [[aImportController window] setFrameUsingName:@"aKisMAC_Import"];
        [[aImportController window] setFrameAutosaveName:@"aKisMAC_Import"];
        [aImportController showWindow:self];
        lMS=[NSApp beginModalSessionForWindow:[aImportController window]];
        [NSApp runModalSession:lMS];
        [[WaveHelper zoomPictureView] loadFromFile:filename];
        [NSApp endModalSession:lMS];
        [[aImportController window] close];
        [aImportController stopAnimation];
        [aImportController release];
        aImportController=Nil;
        return YES;
    } else return NO;
}

- (NSApplicationTerminateReply)applicationShouldTerminate:(NSApplication *)sender
{
    [self stopScan];
    [self stopActiveAttacks];
    
    if (![self isSaved]) {
        [self showWantToSaveDialog: @selector(reallyQuitDidEnd:returnCode:contextInfo:)];
        return NSTerminateLater;
    }
    
    if (aOurSleepMode) {
        [WaveHelper runScript:@"nosleep_disable.sh"];
    }
    [WaveHelper unloadAllDrivers];
    return NSTerminateNow;
}

- (void)applicationDidFinishLaunching:(NSNotification *)aNotification {
    NSString* crashPath;
    NSFileManager *mang;
    NSUserDefaults *sets;
 
    sets=[NSUserDefaults standardUserDefaults];
    [self updatePrefs:self];
    if ([sets integerForKey:@"ScanAtStartUp"]) [self scan:ScanButton];

    crashPath = [@"~/Library/Logs/CrashReporter/KisMAC.crash.log" stringByExpandingTildeInPath];
    mang = [NSFileManager defaultManager];
    
    if ([[sets objectForKey:@"SupressCrashReport"] intValue]!=1 && [mang fileExistsAtPath:crashPath] && [[[mang fileAttributesAtPath:crashPath traverseLink:NO] objectForKey:NSFileSize] intValue] != 0) {
        NSData *crashLog = [mang contentsAtPath:crashPath];
        CrashReportController* crc = [[CrashReportController alloc] initWithWindowNibName:@"CrashReporter"];
        [[crc window] setFrameUsingName:@"aKisMAC_CRC"];
        [[crc window] setFrameAutosaveName:@"aKisMAC_CRC"];
        
        [crc setReport:crashLog];
        [crc showWindow:self];
        [[crc window] makeKeyAndOrderFront:self];
        
        NSLog(@"crash occured the last time kismac started");
        [mang removeFileAtPath:crashPath handler:Nil];
        
        if (aMS) return;
        aMS=[NSApp beginModalSessionForWindow:[crc window]];
        for (;;) {
            if ([NSApp runModalSession:aMS] != NSRunContinuesResponse)
            break;
        }
        [NSApp endModalSession:aMS];
        aMS=Nil;
    }

}

- (void)reallyQuitDidEnd:(NSWindow *)sheet returnCode:(int)returnCode contextInfo:(void *)contextInfo {
    switch (returnCode) {
    case NSAlertOtherReturn:
        [NSApp replyToApplicationShouldTerminate:NO];
        break;
    case NSAlertDefaultReturn:
        [self saveFile:nil];
    case NSAlertAlternateReturn:
    default:
        if (aOurSleepMode) {
            [WaveHelper runScript:@"nosleep_disable.sh"];
        }
        [WaveHelper unloadAllDrivers];
        
        [NSApp replyToApplicationShouldTerminate:YES];
    }
}

- (NSRect)windowWillUseStandardFrame:(NSWindow *)sender defaultFrame:(NSRect)defaultFrame {
    if (aNetHierarchVisible) {
        defaultFrame.size.width-=[aNetHierarchDrawer contentSize].width+10;
        defaultFrame.origin.x+=[aNetHierarchDrawer contentSize].width+10;
    }
    if (aDetailsPane) {
        defaultFrame.size.width-=[aDetailsDrawer contentSize].width+10;
        [aWindow setFrame:[aWindow frame] display:YES animate:YES];
    }
    
    return defaultFrame;
}

- (NSSize)windowWillResize:(NSWindow *)sender toSize:(NSSize)proposedFrameSize {
    [aDetailsDrawer setTrailingOffset:proposedFrameSize.height-280];
    return proposedFrameSize;
}

- (void)reallyCloseDidEnd:(NSWindow *)sheet returnCode:(int)returnCode contextInfo:(void *)contextInfo {
    switch (returnCode) {
    case NSAlertDefaultReturn:
        [self saveFile:nil];
    case NSAlertOtherReturn:
        break;
    case NSAlertAlternateReturn:
    default:
        _isSaved = YES;
        [NSApp terminate:self];
    }
}

- (BOOL)windowShouldClose:(id)sender {
    if (![self isSaved]) {
        [self showWantToSaveDialog:@selector(reallyCloseDidEnd:returnCode:contextInfo:)];
        return NO;
    }
    
    [NSApp terminate:self];
    return YES;
}

@end
