/*	Copyright: 	 Copyright 2002 Apple Computer, Inc. All rights reserved.

	Disclaimer:	IMPORTANT:  This Apple software is supplied to you by Apple Computer, Inc.
			("Apple") in consideration of your agreement to the following terms, and your
			use, installation, modification or redistribution of this Apple software
			constitutes acceptance of these terms.  If you do not agree with these terms,
			please do not use, install, modify or redistribute this Apple software.

			In consideration of your agreement to abide by the following terms, and subject
			to these terms, Apple grants you a personal, non-exclusive license, under Apples
			copyrights in this original Apple software (the "Apple Software"), to use,
			reproduce, modify and redistribute the Apple Software, with or without
			modifications, in source and/or binary forms; provided that if you redistribute
			the Apple Software in its entirety and without modifications, you must retain
			this notice and the following text and disclaimers in all such redistributions of
			the Apple Software.  Neither the name, trademarks, service marks or logos of
			Apple Computer, Inc. may be used to endorse or promote products derived from the
			Apple Software without specific prior written permission from Apple.  Except as
			expressly stated in this notice, no other rights or licenses, express or implied,
			are granted by Apple herein, including but not limited to any patent rights that
			may be infringed by your derivative works or by other works in which the Apple
			Software may be incorporated.

			The Apple Software is provided by Apple on an "AS IS" basis.  APPLE MAKES NO
			WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
			WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
			PURPOSE, REGARDING THE APPLE SOFTWARE OR ITS USE AND OPERATION ALONE OR IN
			COMBINATION WITH YOUR PRODUCTS.

			IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
			CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
			GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
			ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION AND/OR DISTRIBUTION
			OF THE APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER THEORY OF CONTRACT, TORT
			(INCLUDING NEGLIGENCE), STRICT LIABILITY OR OTHERWISE, EVEN IF APPLE HAS BEEN
			ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
/* latency.c */

/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * "Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.0 (the 'License').  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License."
 * 
 * @APPLE_LICENSE_HEADER_END@
 */

/*
 * This code modified from the original to suit the purposes of CoreAudio's
 * kernel-scheduler-profiling-application 'MillionMonkeys'.
 */

#include "latency.h"

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <strings.h>
#include <nlist.h>
#include <fcntl.h>
#include <string.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>

#include <libc.h>
#include <termios.h>
#include <sys/ioctl.h>

#ifndef KERNEL_PRIVATE
#define KERNEL_PRIVATE
#include <sys/kdebug.h>
#undef KERNEL_PRIVATE
#else
#include <sys/kdebug.h>
#endif /*KERNEL_PRIVATE*/

#include <sys/sysctl.h>
#include <errno.h>
#include <err.h>

#include <mach/host_info.h>
#include <mach/mach_error.h>
#include <mach/mach_types.h>
#include <mach/message.h>
#include <mach/mach_syscalls.h>
#include <mach/clock_types.h>
#include <mach/mach_time.h>
#include <libkern/OSTypes.h>

extern mach_port_t clock_port;

#define KERN_KDPIDEX    14


int      s_too_slow;
int      s_max_latency;
int      s_min_latency = 0;
long long s_total_latency = 0;
int      s_total_samples;
int      s_exceeded_threshold = 0;

int      i_too_slow;
int      i_max_latency;
int      i_min_latency = 0;
long long i_total_latency = 0;
int      i_total_samples;
int      i_exceeded_threshold = 0;

long     start_time;
long     curr_time;
long     refresh_time;

char     *policy_name;
int      my_pri = -1;

char *kernelpath = (char *)0;
char *code_file = (char *)0;

typedef struct {
  u_long  k_sym_addr;       /* kernel symbol address from nm */
  u_int   k_sym_len;        /* length of kernel symbol string */
  char   *k_sym_name;       /* kernel symbol string from nm */
} kern_sym_t;

kern_sym_t *kern_sym_tbl;      /* pointer to the nm table       */
int        kern_sym_count;    /* number of entries in nm table */
char       pcstring[128];

#define UNKNOWN "Can't find symbol name"


double   divisor;
int      gotSIGWINCH = 0;
int      trace_enabled = 0;

#define SAMPLE_SIZE 300000

int mib[6];
size_t needed;
char  *my_buffer;

kbufinfo_t bufinfo = {0, 0, 0};

FILE *log_fp = (FILE *)0;
int num_of_codes = 0;
int need_new_map = 0;
int total_threads = 0;
kd_threadmap *mapptr = 0;

#define MAX_ENTRIES 1024
struct ct {
        int type;
        char name[32];
} codes_tab[MAX_ENTRIES];

/* If NUMPARMS changes from the kernel, then PATHLENGTH will also reflect the change */
#define NUMPARMS 23
#define PATHLENGTH (NUMPARMS*sizeof(long))

struct th_info {
        int  thread;
        int  type;
        int  child_thread;
        int  arg1;
        double stime;
        long *pathptr;
        char pathname[PATHLENGTH + 1];
};

#define MAX_THREADS 512
struct th_info th_state[MAX_THREADS];

int  cur_max = 0;

#define TRACE_DATA_NEWTHREAD   0x07000004
#define TRACE_STRING_NEWTHREAD 0x07010004
#define TRACE_STRING_EXEC      0x07010008

#define INTERRUPT         0x01050000
#define DECR_TRAP         0x01090000
#define DECR_SET          0x01090004
#define MACH_vmfault      0x01300000
#define MACH_sched        0x01400000
#define MACH_stkhandoff   0x01400008
#define VFS_LOOKUP        0x03010090
#define BSC_exit          0x040C0004
#define IES_action        0x050b0018
#define IES_filter        0x050b001c
#define TES_action        0x050c0010
#define CQ_action         0x050d0018


#define DBG_FUNC_ALL	(DBG_FUNC_START | DBG_FUNC_END)
#define DBG_FUNC_MASK	0xfffffffc

#define DBG_ZERO_FILL_FAULT   1
#define DBG_PAGEIN_FAULT      2
#define DBG_COW_FAULT         3
#define DBG_CACHE_HIT_FAULT   4

char *fault_name[5] = {
        "",
	"ZeroFill",
	"PageIn",
	"COW",
	"CacheHit",
};

char *pc_to_string();

int decrementer_val = 0;     /* Value used to reset decrementer */
int set_remove_flag = 1;     /* By default, remove trace buffer */

#define DEBUG_SAMPLER		0

int quit (char *s)
{
    void set_enable();
    void set_rtcdec();
    void set_remove();
    
    if (trace_enabled) set_enable(0);
    // This flag is turned off when calling quit() due to a set_remove() failure.
    if (set_remove_flag) set_remove();
    if (decrementer_val) set_rtcdec(0);
    
    printf("latency: ");
        if (s) printf("%s", s);
    exit(1);
}

void set_enable(int val) 
{
    mib[0] = CTL_KERN;
    mib[1] = KERN_KDEBUG;
    mib[2] = KERN_KDENABLE;		/* protocol */
    mib[3] = val;
    mib[4] = 0;
    mib[5] = 0;		        /* no flags */
    
    if (sysctl(mib, 4, NULL, &needed, NULL, 0) < 0)
        quit("trace facility failure, KERN_KDENABLE\n");
}

void set_numbufs (int nbufs) 
{
    mib[0] = CTL_KERN;
    mib[1] = KERN_KDEBUG;
    mib[2] = KERN_KDSETBUF;
    mib[3] = nbufs;
    mib[4] = 0;
    mib[5] = 0;		        /* no flags */
    if (sysctl(mib, 4, NULL, &needed, NULL, 0) < 0)
        quit("trace facility failure, KERN_KDSETBUF\n");
    
    mib[0] = CTL_KERN;
    mib[1] = KERN_KDEBUG;
    mib[2] = KERN_KDSETUP;		
    mib[3] = 0;
    mib[4] = 0;
    mib[5] = 0;		        /* no flags */
    if (sysctl(mib, 3, NULL, &needed, NULL, 0) < 0)
        quit("trace facility failure, KERN_KDSETUP\n");
}

void set_pidexclude (int pid, int on_off) 
{
    kd_regtype kr;
    
    kr.type = KDBG_TYPENONE;
    kr.value1 = pid;
    kr.value2 = on_off;
    needed = sizeof(kd_regtype);
    mib[0] = CTL_KERN;
    mib[1] = KERN_KDEBUG;
    mib[2] = KERN_KDPIDEX;
    mib[3] = 0;
    mib[4] = 0;
    mib[5] = 0;
    
    sysctl(mib, 3, &kr, &needed, NULL, 0);
}

void set_rtcdec (int decval)
{
    kd_regtype kr;
    int ret;
    extern int errno;
    
    kr.type = KDBG_TYPENONE;
    kr.value1 = decval;
    needed = sizeof(kd_regtype);
    mib[0] = CTL_KERN;
    mib[1] = KERN_KDEBUG;
    mib[2] = KERN_KDSETRTCDEC;		/* protocol */
    mib[3] = 0;				/* wildcard address family */
    mib[4] = 0;
    mib[5] = 0;				/* no flags */

    errno = 0;

    if ((ret=sysctl(mib, 3, &kr, &needed, NULL, 0)) < 0) {
        decrementer_val = 0;
        quit("trace facility failure, KERN_KDSETRTCDEC\n");
    }
}


void get_bufinfo(kbufinfo_t *val)
{
    needed = sizeof (*val);
    mib[0] = CTL_KERN;
    mib[1] = KERN_KDEBUG;
    mib[2] = KERN_KDGETBUF;		
    mib[3] = 0;
    mib[4] = 0;
    mib[5] = 0;		/* no flags */
    
    if (sysctl(mib, 3, val, &needed, 0, 0) < 0)
        quit("trace facility failure, KERN_KDGETBUF\n");
}

void set_remove() 
{
    extern int errno;
    
    mib[0] = CTL_KERN;
    mib[1] = KERN_KDEBUG;
    mib[2] = KERN_KDREMOVE;		/* protocol */
    mib[3] = 0;
    mib[4] = 0;
    mib[5] = 0;				/* no flags */
    
    errno = 0;
    
    if (sysctl(mib, 3, NULL, &needed, NULL, 0) < 0) {
        set_remove_flag = 0;
        if(errno == EBUSY) {
            quit("the trace facility is currently in use...\n         fs_usage, sc_usage, and latency use this feature.\n\n");
        } else {
            quit("trace facility failure, KERN_KDREMOVE\n");
        }
    }
}

void sigwinch()
{
    gotSIGWINCH = 1;
}

void sigintr()
{
    set_enable(0);
    set_pidexclude(getpid(), 0);
    set_rtcdec(0);
    set_remove();
    
    exit(1);
}

void sigquit()
{
    set_enable(0);
    set_pidexclude(getpid(), 0);
    set_rtcdec(0);
    set_remove();
    
    exit(1);
}

void sigterm()
{
    set_enable(0);
    set_pidexclude(getpid(), 0);
    set_rtcdec(0);
    set_remove();
    
    exit(1);
}

void initialize()
{
    int      		loop_cnt, sample_sc_now;
    int      		decrementer_usec = 0;
    kd_regtype		kr;
    void     init_code_file();
    void     do_kernel_nm();
    void     open_logfile();
    
    policy_name = "TIMESHARE";
    kernelpath = "/mach_kernel";
    code_file = "/usr/share/misc/trace.codes";
    
    do_kernel_nm();
    
    sample_sc_now = 25;
    
    getdivisor();
    decrementer_val = decrementer_usec * divisor;
    
    init_code_file();
    /* 
        When the decrementer isn't set in the options,
        decval will be zero and this call will reset
        the system default ...
    */
    set_rtcdec(decrementer_val);
    
    signal(SIGWINCH, sigwinch);
    signal(SIGINT, sigintr);
    signal(SIGQUIT, sigquit);
    signal(SIGTERM, sigterm);
    
    if ((my_buffer = malloc(SAMPLE_SIZE * sizeof(kd_buf))) == (char *)0)
        quit("can't allocate memory for tracing info\n");
    set_remove();
    set_numbufs(SAMPLE_SIZE);
    set_enable(0);
    
    // setup Kernel for trace logging
    kr.type = KDBG_RANGETYPE;
    kr.value1 = 0;	
    kr.value2 = -1;
    needed = sizeof(kd_regtype);
    mib[0] = CTL_KERN;
    mib[1] = KERN_KDEBUG;
    mib[2] = KERN_KDSETREG;		
    mib[3] = 0;
    mib[4] = 0;
    mib[5] = 0;		/* no flags */
    if (sysctl(mib, 3, &kr, &needed, NULL, 0) < 0)
        quit("trace facility failure, KERN_KDSETREG\n");
    
    mib[0] = CTL_KERN;
    mib[1] = KERN_KDEBUG;
    mib[2] = KERN_KDSETUP;		
    mib[3] = 0;
    mib[4] = 0;
    mib[5] = 0;		/* no flags */
    
    if (sysctl(mib, 3, NULL, &needed, NULL, 0) < 0)
        quit("trace facility failure, KERN_KDSETUP\n");

    // set_pidexclude(getpid(), 1);	// This line (if included) will remove this process from the trace log
    set_enable(1);
    trace_enabled = 1;
    need_new_map = 1;
    
    loop_cnt = 0;
    start_time = time((long *)0);
    refresh_time = start_time;
}

double getdivisor()
{
    mach_timebase_info_data_t info;
    mach_timebase_info (&info);

    divisor = ( (double)info.denom / (double)info.numer) * 1000;
	return divisor;
}
												  
void read_command_map()
{
    size_t size;
    int mib[6];
  
    if (mapptr) {
	free(mapptr);
	mapptr = 0;
    }
    total_threads = bufinfo.nkdthreads;
    size = bufinfo.nkdthreads * sizeof(kd_threadmap);
    if (size) {
        if (mapptr = (kd_threadmap *) malloc(size)) {
	     bzero (mapptr, size);
	} else {
	    printf("Thread map is not initialized -- this is not fatal\n");
	    return;
	}
    }
    
    // Now read the threadmap
    mib[0] = CTL_KERN;
    mib[1] = KERN_KDEBUG;
    mib[2] = KERN_KDTHRMAP;
    mib[3] = 0;
    mib[4] = 0;
    mib[5] = 0;		// no flags
    if (sysctl(mib, 3, mapptr, &size, NULL, 0) < 0) {
        // This is not fatal -- just means we can't map command strings
        printf("Can't read the thread map -- this is not fatal\n");
	free(mapptr);
	mapptr = 0;
	return;
    }
    return;
}

void create_map_entry(int thread, char *command)
{
    int			i, n;
    kd_threadmap	*map;
    
    if (!mapptr) return;
    
    for (i = 0, map = 0; !map && i < total_threads; i++) {
        if (mapptr[i].thread == thread )	
	    map = &mapptr[i];   // Reuse this entry, the thread has been reassigned
    }
    
    if (!map) {
        // look for invalid entries that I can reuse
        for (i = 0, map = 0; !map && i < total_threads; i++) {
	    if (mapptr[i].valid == 0 )	
	        map = &mapptr[i];   // Reuse this invalid entry
	}
    }
    
    if (!map) {
    /* If reach here, then this is a new thread and 
	 * there are no invalid entries to reuse
	 * Double the size of the thread map table.
	 */
        
        n = total_threads * 2;
	mapptr = (kd_threadmap *) realloc(mapptr, n * sizeof(kd_threadmap));
	bzero(&mapptr[total_threads], total_threads*sizeof(kd_threadmap));
	map = &mapptr[total_threads];
	total_threads = n;
    }

    map->valid = 1;
    map->thread = thread;
    (void)strncpy (map->command, command, sizeof(map->command));
    map->command[sizeof(map->command)-1] = '\0';
}

kd_threadmap *find_thread_map(int thread)
{
    int i;
    kd_threadmap *map;
    
    if (!mapptr)
        return((kd_threadmap *)0);
    
    for (i = 0; i < total_threads; i++) {
        map = &mapptr[i];
	if (map->valid && (map->thread == thread))
	    return(map);
    }
    return ((kd_threadmap *)0);
}

void kill_thread_map(int thread)
{
    kd_threadmap *map;
    
    if (map = find_thread_map(thread)) {
        map->valid = 0;
	map->thread = 0;
	map->command[0] = '\0';
    }
}

struct th_info *find_thread(int thread, int type1, int type2) {
    struct th_info *ti;
    
    for (ti = th_state; ti < &th_state[cur_max]; ti++) {
        if (ti->thread == thread) {
            if (type1 == 0)
                return(ti);
            if (type1 == ti->type)
                return(ti);
            if (type2 == ti->type)
                return(ti);
        }
    }
    return ((struct th_info *)0);
}

char *find_code(type)
{
    int i;
    
    for (i = 0; i < num_of_codes; i++) {
        if (codes_tab[i].type == type) 
            return(codes_tab[i].name);
    }
    
    return ((char *)0);
}

int check_for_thread_update (int thread, int type, kd_buf *kd)
{
    struct th_info *ti;
    void create_map_entry();
    
    switch (type) {
    
    case TRACE_DATA_NEWTHREAD:
        if ((ti = find_thread(thread, 0, 0)) == (struct th_info *)0) {
            if (cur_max >= MAX_THREADS) return 1;
            ti = &th_state[cur_max++];
            
            ti->thread = thread;
            ti->type   = -1;
            ti->pathptr = (long *)0;
        }
        ti->child_thread = kd->arg1;
        return 1;
    case TRACE_STRING_NEWTHREAD:
        if ((ti = find_thread(thread, 0, 0)) == (struct th_info *)0)
            return 1;
        if (ti->child_thread == 0)
            return 1;
        create_map_entry(ti->child_thread, (char *)&kd->arg1);
        
        ti->child_thread = 0;
        return 1;
    case TRACE_STRING_EXEC:
        create_map_entry(thread, (char *)&kd->arg1);
        return 1;
    }
    
    return 0;
}

double handle_decrementer(kd_buf *kd)
{
    double		latency;
    int			elapsed_usecs;
    
    if ((int)(kd->arg1) >= 0)
        latency = 1;
    else
        latency = (((double)(-1 - kd->arg1)) / divisor);
    
    elapsed_usecs = (int)latency;
    
    if (elapsed_usecs > i_max_latency)
        i_max_latency = elapsed_usecs;
    if (elapsed_usecs < i_min_latency || i_total_samples == 0)
        i_min_latency = elapsed_usecs;
    i_total_latency += elapsed_usecs;
    i_total_samples++;
    
    return (latency);
}

void init_code_file()
{
    FILE		*fp;
    int			i, n, cnt, code;
    char		name[128];
    
    if ((fp = fopen(code_file, "r")) == (FILE *)0) return;
    
    n = fscanf(fp, "%d\n", &cnt);
    
    if (n != 1) return;
    
    for (i = 0; i < MAX_ENTRIES; i++) {
        n = fscanf(fp, "%x%s\n", &code, name);
        
        if (n != 2) break;
        
        strncpy(codes_tab[i].name, name, 32);
        codes_tab[i].type = code;
    }
    num_of_codes = i;
    
    fclose(fp);
}

void do_kernel_nm()
{
    int i, len;
    FILE *fp = (FILE *)0;
    char tmp_nm_file[128];
    char tmpstr[1024];
    int inchr;
    
    bzero(tmp_nm_file, 128);
    bzero(tmpstr, 1024);
    
    // Build the temporary nm file path
    sprintf(tmp_nm_file, "/tmp/knm.out.%d", getpid());
    
    // Build the nm command and create a tmp file with the output
    sprintf (tmpstr, "/usr/bin/nm -f -n -s __TEXT __text %s > %s", kernelpath, tmp_nm_file);
    system (tmpstr);
    
    // Parse the output from the nm command 
    if ((fp=fopen(tmp_nm_file, "r")) == (FILE *)0)
    {
        // Hmmm, let's not treat this as fatal
        fprintf(stderr, "Failed to open nm symbol file [%s]\n", tmp_nm_file);
        return;
    }
    
    // Count the number of symbols in the nm symbol table
    kern_sym_count=0;
    while ( (inchr = getc(fp)) != -1) {
        if (inchr == '\n')
        kern_sym_count++;
    }
    
    rewind(fp);
    
    // Malloc the space for symbol table
    if (kern_sym_count > 0) {
        kern_sym_tbl = (kern_sym_t *)malloc(kern_sym_count * sizeof (kern_sym_t));
       if (!kern_sym_tbl) {
	   // Hmmm, lets not treat this as fatal
	   fprintf(stderr, "Can't allocate memory for kernel symbol table\n");
        } else {
            bzero(kern_sym_tbl, (kern_sym_count * sizeof(kern_sym_t)));
        }
    } else {
        // Hmmm, lets not treat this as fatal
        fprintf(stderr, "No kernel symbol table \n");
    }
    
    for (i = 0; i < kern_sym_count; i++) {
        bzero(tmpstr, 1024);
        if (fscanf(fp, "%x %c %s", &kern_sym_tbl[i].k_sym_addr, &inchr, tmpstr) != 3) {
            break;
        } else {
            len = strlen(tmpstr);
            kern_sym_tbl[i].k_sym_name = malloc(len + 1);
            
            if (kern_sym_tbl[i].k_sym_name == (char *)0) {
                fprintf(stderr, "Can't allocate memory for symbol name [%s]\n", tmpstr);
                kern_sym_tbl[i].k_sym_name = (char *)0;
                len = 0;
            } else {
                strcpy(kern_sym_tbl[i].k_sym_name, tmpstr);
            }
            
            kern_sym_tbl[i].k_sym_len = len;
        }
    }
    
    if (i != kern_sym_count) {
        /* Hmmm, didn't build up entire table from nm */
        /* scrap the entire thing */
        if (kern_sym_tbl) free (kern_sym_tbl);
        kern_sym_tbl = (kern_sym_t *)0;
        kern_sym_count = 0;
    }
    fclose(fp);
    
    // Remove the temporary nm file
    unlink(tmp_nm_file);
    
#if DEBUG_SAMPLER
    // Print out the WHOLE KERNEL SYMBOL TABLE
    for (i = 0; i < kern_sym_count; i++) {
        if (kern_sym_tbl[i].k_sym_name)
            printf ("[%d] 0x%x    %s\n", i, kern_sym_tbl[i].k_sym_addr, kern_sym_tbl[i].k_sym_name);
        else
            printf ("[%d] 0x%x    %s\n", i, kern_sym_tbl[i].k_sym_addr, "No symbol name");
    }
#endif
}

char *pc_to_string (unsigned int pc, int max_len, int mode)
{
    int ret, len;
    int binary_search();
    
    if (mode == 0) {
        sprintf(pcstring, "0x%-8x [usermode addr]", pc);
        return pcstring;
    }

    ret = 0;
    ret = binary_search(kern_sym_tbl, 0, kern_sym_count-1, pc);

    if (ret == -1) {
        sprintf(pcstring, "0x%x", pc);
        return pcstring;
    } else if (kern_sym_tbl[ret].k_sym_name == (char *)0) {
        sprintf(pcstring, "0x%x", pc);
        return pcstring;
    } else {
        if ((len = kern_sym_tbl[ret].k_sym_len) > (max_len - 8))
            len = max_len - 8;
    
        memcpy(pcstring, kern_sym_tbl[ret].k_sym_name, len);
        sprintf(&pcstring[len], "+0x%-5x", pc - kern_sym_tbl[ret].k_sym_addr);
        return pcstring;
    }
}

// find address in kernel symbol list
int binary_search (kern_sym_t *list, int low, int high, unsigned int addr)
{
    int mid;
    mid = (low + high) / 2;
    
    if (low > high) {
        return -1;   // failed
    }
    
    if (low + 1 == high) {
        if ( (list[low].k_sym_addr <= addr) && (addr < list[high].k_sym_addr) ) {
            return low;	// case for range match in bottom
	} else if (list[high].k_sym_addr <= addr) {
	  return high;	// case for range match in top
	} else {
            return -1;	// failed
        }
    } else if (addr < list[mid].k_sym_addr) {
        return (binary_search (list, low, mid, addr));	// address in lower-half of search space
    } else {
        return (binary_search (list, mid, high, addr));	// address in upper-half of search space
    }
}

/* 
 */
