/*
        
        File:			TrafficView.mm
        Program:		KisMAC
	Author:			Michael Thole
				mick@binaervarianz.de
	Description:		KisMAC is a wireless stumbler for MacOS X.
                
        This file is part of KisMAC.

    KisMAC is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    KisMAC is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KisMAC; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#import "TrafficView.h"

@implementation TrafficView

- (id)initWithFrame:(NSRect)frame {
    self = [super initWithFrame:frame];
    if (!self)
        return nil;

    justSwitchedDataType = NO;
    currentMode = trafficData;

    [self setBackgroundColor:[NSColor blackColor]];
    [self setTextColor:[NSColor greenColor]];
    [self setGridColor:[NSColor colorWithCalibratedRed:0 green:1 blue:0 alpha:0.5]];
    [self setGraphColor:[NSColor redColor]];

    zoomLock = [[NSLock alloc] init];
    
    vScale = 0;
    dvScale = 0;
    maxLength = 0;
    gridNeedsRedrawn = NO;
    
    /* color wheel permutation */
    NSMutableArray *tempColor = [NSMutableArray array];
    for (int x=0; x<=32; x++) {
        float hue=0.0;
        for (int i=2;i<=32;i=i*2) {
            if ( (x % i) >= (i/2) ) hue += 1.0/ (float)i;
        }
        [tempColor addObject:[NSColor colorWithDeviceHue:hue saturation:1 brightness:1 alpha:0.95]];
    }
    colorArray = [[NSArray arrayWithArray:tempColor] retain];
    
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(updateSettings:) name:NSUserDefaultsDidChangeNotification object:nil];
    [self updateSettings:nil];
    
    return self;
}

-(void)awakeFromNib {
    [_modeButton selectItemAtIndex:[[NSUserDefaults standardUserDefaults] integerForKey:@"GraphMode"]];
    [_intervalButton selectItemAtIndex:[[NSUserDefaults standardUserDefaults] integerForKey:@"GraphTimeInterval"]];

    // default to 30-second interval
    scanInterval = [scanner scanInterval];
    maxLength = (int)(30.0 / scanInterval);
    [self display];
    [self setTimeLength:_intervalButton];
    [self setCurrentMode:_modeButton];
}

- (void)setGraphColor:(NSColor *)newColor {
    if (graphColor != newColor) {
        [graphColor release];
        graphColor = [newColor retain];
    }
}

- (void)setBackgroundColor:(NSColor *)newColor {
    if (backgroundColor != newColor) {
        [backgroundColor release];
        backgroundColor = [newColor retain];
    }
}

- (void)setGridColor:(NSColor *)newColor {
    if (gridColor != newColor) {
        [gridColor release];
        gridColor = [newColor retain];
    }
}

- (void)setTextColor:(NSColor *)newColor {
    if (textColor != newColor) {
        [textColor release];
        textColor = [newColor retain];
    }
}

- (void)setGridPath:(NSBezierPath *)newPath {
    if (gridPath != newPath) {
        [gridPath release];
        gridPath = [newPath retain];
    }
}

- (IBAction)setTimeLength:(id)sender {
    [[NSUserDefaults standardUserDefaults] setInteger:[sender indexOfSelectedItem] forKey:@"GraphTimeInterval"];

    maxLength = (int)ceil([[sender selectedItem] tag] / scanInterval);
    gridNeedsRedrawn = YES;
    
    [self setNeedsDisplay:YES];
}

- (IBAction)setCurrentMode:(id)sender {
    [[NSUserDefaults standardUserDefaults] setInteger:[sender indexOfSelectedItem] forKey:@"GraphMode"];

    justSwitchedDataType = YES;
    currentMode = [sender indexOfSelectedItem];
    if(currentMode != trafficData && currentMode != packetData && currentMode != signalData)
        currentMode = trafficData;

    [self setNeedsDisplay:YES];
}

- (void)updateSettings:(NSNotification*)note {
    NSUserDefaults *sets = [NSUserDefaults standardUserDefaults];
    _legendMode = 0;
    
    if ([[sets objectForKey:@"TrafficViewShowSSID"]  intValue] == 1) _legendMode++;
    if ([[sets objectForKey:@"TrafficViewShowBSSID"] intValue] == 1) _legendMode+=2;
}

#pragma mark -

- (void)updateDataForRect:(NSRect)rect {
    int i, current;
    unsigned int j;
    
    [WaveHelper secureReplace:&allNets withObject:[_container allNets]];
    
    // order networks by signal value
    switch(currentMode) {
        case trafficData:
            [allNets sortUsingSelector:@selector(compareRecentTrafficTo:)];
            break;
        case packetData:
            [allNets sortUsingSelector:@selector(compareRecentPacketsTo:)];
            break;
        case signalData:
            [allNets sortUsingSelector:@selector(compareRecentSignalTo:)];
            break;
    }

    // setup graph rect with nice margins
    graphRect = rect;
    graphRect.origin.x = 30;
    graphRect.origin.y = 30;
    graphRect.size.width -= 60;
    graphRect.size.height -= 60;

    length = [scanner graphLength];
    if(length > maxLength) {
        offset = length - maxLength;
        length = maxLength;
    }
    else {
        offset = 0;
    }

    aMaximum=0;
    memset(buffer,0,MAX_YIELD_SIZE * sizeof(int));

    // find the biggest point on our graph
    for (i = 0 ; i < length ; i++) {
        current = 0;
        for(j = 0 ; j < [allNets count] ; j++) {
            switch(currentMode) {
                case trafficData:
                    current += [(WaveNet*)[allNets objectAtIndex:j] graphData].trafficData[i + offset];
                    break;
                case packetData:
                    current += [(WaveNet*)[allNets objectAtIndex:j] graphData].packetData[i + offset];
                    break;
                case signalData:
                    current += [(WaveNet*)[allNets objectAtIndex:j] graphData].signalData[i + offset];
                    break;
            }
        }
        buffer[i] = current;
        if (current > aMaximum)
            aMaximum = current;
    }
    

    // a horizontal line for every 5 seconds
    stepx = graphRect.size.width / maxLength / scanInterval * 5;

    dvScale = graphRect.size.height / (1.2 * aMaximum);
    if(!vScale)
        vScale = dvScale;
    if(dvScale != vScale) {
        if(justSwitchedDataType) {
            justSwitchedDataType = NO;
            vScale = dvScale;
        }
        else {
            [NSThread detachNewThreadSelector:@selector(zoomThread:) toTarget:self withObject:nil];
        }
    }

    // a vertical line for every 512 bytes
    stepy = 512 * vScale * scanInterval;
}


- (void)drawRect:(NSRect)rect {
    [backgroundColor set];
    NSRectFill(rect);

    // do some math...
    [self updateDataForRect:rect];
    
    // do the drawing...
    [self drawGridInRect:graphRect];
    [self drawGraphInRect:graphRect];
    [self drawGridLabelForRect:rect];
    [self drawLegendForRect:graphRect];
}

- (void)drawGridInRect:(NSRect)rect {
    static float lastVScale = 0.0;
    static NSRect lastRect = NSZeroRect;
    int i = 0;
    int count = 0;
    int multiple = 0;
    float curY, curX;
    
    [NSBezierPath setDefaultLineWidth:1];
    
    [self setFrameOrigin:NSMakePoint(rect.origin.x, rect.origin.y)];

    if(lastVScale == vScale && NSEqualRects(lastRect,rect)
       && !gridNeedsRedrawn) {
        [gridColor set];
        [gridPath stroke];
        [self setFrameOrigin:NSZeroPoint];
        gridNeedsRedrawn = NO;
        return;
    }

    // if we get here, then the grid needs to be redrawn
    lastVScale = vScale;
    lastRect = rect;
    [self setGridPath:[NSBezierPath bezierPath]];
    [gridColor set];

    count = (int)ceil(rect.size.height / stepy);
    if(count >= 20) {
        multiple = 2;		// show a line each 1kb
        if(count >= 100)
            multiple = 10;	// show a line very 5kb
        if(count >= 200)
            multiple = 20;	// show a line very 10kb
    }
    for(i = 0 ; i * stepy < rect.size.height ; i++) {
        if(multiple && i % multiple)
            continue;
        curY = (i * stepy);
        [gridPath moveToPoint:NSMakePoint(0.5, curY)];
        if (curY < rect.size.height) {
            [gridPath lineToPoint:NSMakePoint(rect.size.width, curY)];
        }
    }
    multiple = 0;

    count = (int)ceil(rect.size.width / stepx);
    if(count >= 60) {
        multiple = 12;		// show a line each minute
        if(count >= 720)
            multiple = 120;	// show a line very 5 minutes
    }
    for (i = 0 ; i < count ; i++) {
        if(multiple && i % multiple)
            continue;
        curX = (i * stepx);
        [gridPath moveToPoint:NSMakePoint(curX, 0.5)];
        if (curX < rect.size.width) {
            [gridPath lineToPoint:NSMakePoint(curX, rect.size.height)];
        }
    }
    [gridPath setLineWidth:0.5];
    [gridPath stroke];    
    [self setFrameOrigin:NSZeroPoint];
}

- (void)drawGraphInRect:(NSRect)rect {
    int i, *ptr;
    unsigned int n;
    NSBezierPath *graphPath;

    [self setFrameOrigin:NSMakePoint(rect.origin.x, rect.origin.y)];
          
    for( n = 0 ; n < [allNets count] ; n++) {
        NSPoint p;
        WaveNet* net = [allNets objectAtIndex:n];
        float width = rect.size.width;

        switch(currentMode) {
            case trafficData:
                ptr = [net graphData].trafficData;
                break;
            case packetData:
                ptr = [net graphData].packetData;
                break;
            case signalData:
                ptr = [net graphData].signalData;
                break;
            default:
                ptr = [net graphData].trafficData;
        }
        
        if([[NSDate date] timeIntervalSinceDate:[net lastSeenDate]] >= (maxLength * scanInterval)) {
            [allNets removeObjectAtIndex:n];
            n--;
            continue;
        }
        
        graphPath = [[NSBezierPath alloc] init];
        [graphPath moveToPoint:NSMakePoint(rect.size.width,0)];
        [graphPath setWindingRule:NSNonZeroWindingRule];
    
        stepx=(rect.size.width) / maxLength;

        [graphPath moveToPoint:NSMakePoint(width - (length * stepx), 0)];
        p = NSMakePoint(width - (length * stepx), buffer[0] * vScale);
        [graphPath appendBezierPathWithPoints:&p count:1];
        
        for(i = 1 ; i < length ; i++) {
            p = NSMakePoint(width - ((length - i) * stepx), buffer[i] * vScale);
            [graphPath appendBezierPathWithPoints:&p count:1];
        }
        i--;
        
        p = NSMakePoint(width, buffer[i] * vScale);
        [graphPath appendBezierPathWithPoints:&p count:1];
        p = NSMakePoint(width, 0);
        [graphPath appendBezierPathWithPoints:&p count:1];

        [graphPath closePath];

        if (![net graphColor]) {
            static int colorCount = 0;
            [net setGraphColor:[colorArray objectAtIndex:colorCount % [colorArray count]]];
            colorCount++;
        }
        [[net graphColor] set];
        [graphPath setLineWidth:2];
        [graphPath stroke];
        [graphPath fill];
        [graphPath release];

        for(i = 0 ; i < length ; i++) {
            buffer[i] -= ptr[i + offset];
        }
    }
    [self setFrameOrigin:NSZeroPoint];
}

- (void)drawGridLabelForRect:(NSRect)rect {
    // draws the text, giving a numerical value to the graph
    unsigned int j;
    int current = 0, max = 0;
    NSMutableDictionary* attrs = [[[NSMutableDictionary alloc] init] autorelease];
    NSFont* textFont = [NSFont fontWithName:@"Monaco" size:12];
    NSString *zeroStr, *currentStr, *maxStr;

    // this rect clears out the out-of-bounds spike in the graph which
    // sometimes happens when 'zooming' in
    // TODO: there is probably a better/cleaner way (clipping)
    [[NSColor blackColor] set];
    [NSBezierPath fillRect:NSMakeRect(0,
                                      graphRect.origin.y + graphRect.size.height,
                                      rect.size.width,
                                      rect.size.height)];
    [gridColor set];
    [NSBezierPath setDefaultLineWidth:2.0];
    [NSBezierPath strokeRect:graphRect];


    if(length) {
        for(j = 0 ; j < [allNets count] ; j++) {
            switch(currentMode) {
                case trafficData:
                    current += (int)([(WaveNet*)[allNets objectAtIndex:j] graphData].trafficData[length - 2 + offset]  / scanInterval);
                    break;
                case packetData:
                    current += (int)([(WaveNet*)[allNets objectAtIndex:j] graphData].packetData[length - 2 + offset]);
                    break;
                case signalData:
                    current += (int)([(WaveNet*)[allNets objectAtIndex:j] graphData].signalData[length - 2 + offset]);
                    break;
            }
        }
    }

    if (currentMode==trafficData)
        max = (int)(aMaximum * 1.1 / scanInterval);
    else
        max = (int)(aMaximum * 1.1);
    
    [attrs setObject:textFont forKey:NSFontAttributeName];
    [attrs setObject:[NSColor greenColor] forKey:NSForegroundColorAttributeName];

    switch(currentMode) {
        case trafficData:
            zeroStr = @"0 bps";
            currentStr = [self stringForBytes:current];
            maxStr = [self stringForBytes:max];
            break;
        case packetData:
            zeroStr = @"0 packets";
            currentStr = [self stringForPackets:current];
            maxStr = [self stringForPackets:max];
            break;            
        case signalData:
            zeroStr = @"0 signal";
            currentStr = [self stringForSignal:current];
            maxStr = [self stringForSignal:max];
            break;            
        default:
            zeroStr = @"0 bps";
            currentStr = [self stringForBytes:current];
            maxStr = [self stringForBytes:max];
            break;
    }
    
    [zeroStr drawAtPoint:NSMakePoint(15,8) withAttributes:attrs];
    [maxStr drawAtPoint:NSMakePoint(15,rect.size.height - 5 - [textFont boundingRectForFont].size.height) withAttributes:attrs];
    [currentStr drawAtPoint:NSMakePoint(rect.size.width - 15 - [textFont widthOfString:currentStr], 8) withAttributes:attrs];
}

- (void)drawLegendForRect:(NSRect)rect {
    unsigned int i;
    float width = 0, height = 0;
    NSBezierPath* legendPath = [[NSBezierPath alloc] init];
    NSMutableDictionary* attrs = [[[NSMutableDictionary alloc] init] autorelease];
    NSFont* textFont = [NSFont fontWithName:@"Monaco" size:12];
    
    if(_legendMode == 0 || ![allNets count])
        return;
    
    [self setFrameOrigin:NSMakePoint(rect.origin.x, rect.origin.y)];

    [attrs setObject:textFont forKey:NSFontAttributeName];

    for(i = 0 ; i < [allNets count] ; i++) {
        NSSize size = [[self stringForNetwork:[allNets objectAtIndex:i]] sizeWithAttributes:attrs];
        if(size.width > width) width = size.width;
        if(size.height > height) height = size.height;
    }
    width += 20;
    height = [allNets count] * (height + 5) + 10;
    
    [legendPath appendBezierPathWithRect:NSMakeRect(1, rect.size.height - height, width, height)];
    [[[NSColor blackColor] colorWithAlphaComponent:0.80] set];
    [legendPath fill];
    [[[NSColor whiteColor] colorWithAlphaComponent:0.10] set];
    [legendPath fill];
    [[[NSColor whiteColor] colorWithAlphaComponent:0.25] set];
    [NSBezierPath setDefaultLineWidth:2];
    [legendPath stroke];
    [legendPath release];

    for(i = 0 ; i < [allNets count] ; i++) {
        [attrs setObject:[(WaveNet*)[allNets objectAtIndex:i] graphColor] forKey:NSForegroundColorAttributeName];
        [[self stringForNetwork:[allNets objectAtIndex:i]] drawAtPoint:NSMakePoint(10,rect.size.height - ((i+1) * 20)) withAttributes:attrs];
    }
    
    [self setFrameOrigin:NSMakePoint(0,0)];
}

#pragma mark -

- (NSString*)stringForNetwork:(WaveNet*)net {
    switch (_legendMode) {
        case 1:
            return [net SSID];
        case 2:
            return [net BSSID];
        case 3:
            return [NSString stringWithFormat:@"%@, %@", [net BSSID],[net SSID]];
    }
    return nil;
}

- (NSString*)stringForBytes:(int)bytes {
    if(bytes < 1024)
        return [NSString stringWithFormat:@"%d bps",bytes];
    else
        return [NSString stringWithFormat:@"%.2f kbps",(float)bytes / 1024];
}

- (NSString*)stringForPackets:(int)bytes {
    return [NSString stringWithFormat:@"%d %@", bytes, NSLocalizedString(@"packets/sec", "label of traffic view")];
}

- (NSString*)stringForSignal:(int)bytes {
    return [NSString stringWithFormat:@"%d %@", bytes, NSLocalizedString(@"signal", "label of traffic view")];
}

#pragma mark -


- (void)zoomThread:(id)object {
    NSAutoreleasePool* subpool = [[NSAutoreleasePool alloc] init];

    int i;
    int fps = 30;
    int frames = (int)floor((float)fps * scanInterval);
    float delta = (dvScale - vScale) / (float)frames;

    if([zoomLock tryLock]) {
        //NSLog(@"ZOOMING: frames = %d, delta = %f",frames,delta);    
        for(i = 0 ; i < frames ; i++) {
            vScale += delta;
            [self setNeedsDisplay:YES];
            [NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:scanInterval / frames]];
        }
        vScale = dvScale;
        [zoomLock unlock];
    }
    else {
        //NSLog(@"ZOOM LOCK IS LOCKED!");
    }
    
    [subpool release];
}


#pragma mark -

- (void)dealloc {
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    [colorArray release];
    [allNets release];
    
    [backgroundColor release];
    [graphColor release];
    [gridColor release];
    [textColor release];

    [zoomLock release];
    [gridPath release];
    [super dealloc];
}

@end
