/*
        
        File:			WaveContainer.m
        Program:		KisMAC
	Author:			Michael Rossberg
				mick@binaervarianz.de
	Description:		KisMAC is a wireless stumbler for MacOS X.
                
        This file is part of KisMAC.

    KisMAC is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    KisMAC is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KisMAC; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#import "WaveContainer.h"
#import "WaveHelper.h"
#import "ScanController.h"
#import "quicksort.h"

//TODO make _idList binary search compatible 
// AVL trees?!
// hash - ring - structure?

typedef struct WaveSort {
    int ascend;
    int *sortedList;
    WaveNetEntry *idList;
} WaveSort;

inline UInt32 hashForMAC(UInt8* val) {
    UInt32 l, j, k;
    
    //add to hash table
#ifdef FASTLOOKUP
    memcpy(((char*)&l)+1, val, 3);
    memcpy(((char*)&j)+1, val+3, 3);
    l = (l ^ j) & 0x00FFFFFF;
#else
    memcpy(((char*)&l)+2, val,   2);
    memcpy(((char*)&j)+2, val+2, 2);
    memcpy(((char*)&k)+2, val+4, 2);
    l = (l ^ j ^ k) & 0x0000FFFF;
#endif

    return l;
}

@implementation WaveContainer

-(id) init {
    int i;
    self = [super init];
    if (!self) return Nil;
    
    _order = -1;
    _dropAll = NO;
    
    _viewType =  0;
    _netCount = 0;
    _filterCount = 0;
    _cacheSize = 0;
    _viewCrypto = 0;
    
    _sortLock = [[NSLock alloc] init];
    
    for(i=0; i<LOOKUPSIZE; i++) _lookup[i]=LOOKUPSIZE;
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(updateSettings:) name:KisMACFiltersChanged object:nil];
    [self updateSettings:nil];
    
    return self;
}

#pragma mark -

- (void)updateSettings:(NSNotification*)note {
    NSArray *filtered;
    unsigned int i, j, tmp[6];
    NSUserDefaults *sets = [NSUserDefaults standardUserDefaults];
    
    filtered = [sets objectForKey:@"FilterBSSIDList"];
    
    _filterCount = 0;
    for(i=0; i<[filtered count]; i++) {
        if (sscanf([[filtered objectAtIndex:i] cString],"%2X%2X%2X%2X%2X%2X", &tmp[0], &tmp[1], &tmp[2], &tmp[3], &tmp[4], &tmp[5]) != 6) continue;
    
        for (j=0; j<6; j++) 
            _filter[i][j] = tmp[j];
        
        _filterCount++;
    }
}


#pragma mark -

- (void) addNetToView:(unsigned int)entry {
    switch (_viewType) {
    case 0:
        _sortedList[_sortedCount] = entry;
        _sortedCount++;
        break;
    case 1:
        if ([_idList[entry].net packetsPerChannel][_viewChannel]!=0) {
            _sortedList[_sortedCount] = entry;
            _sortedCount++;
        }
        break;
    case 2:
        if ([[_idList[entry].net SSID] isEqualToString:_viewSSID]) {
            _sortedList[_sortedCount] = entry;
            _sortedCount++;
        }
        break;
    case 3:
        if ([_idList[entry].net isWep] == _viewCrypto) {
            _sortedList[_sortedCount] = entry;
            _sortedCount++;
        }
        break;
    default:
        NSLog(@"invalid view. this is a bug and shall never happen\n");
    }
    
}

- (void) refreshView {
    unsigned int i;
    
    _sortedCount = 0;
    for (i=0; i<_netCount; i++)
        [self addNetToView:i];
}

- (void) setViewType:(int)type value:(id)val {
    _viewType = type;
    
    switch(_viewType) {
    case 1:
        _viewChannel = [val intValue];
        break;
    case 2:
        [WaveHelper secureReplace:&_viewSSID withObject:val];
        break;
    case 3:
        _viewCrypto = [val intValue];
        break;
    default:
        NSLog(@"invalid viewtype. this is a bug and shall never happen!\n");
    }
    [self refreshView];
} 

#pragma mark -

- (bool) addNetwork:(WaveNet*)net {
    unsigned int entry, l;
    
    if (_netCount >= MAXNETS) return NO;
    
    _netCount++;
    entry = _netCount - 1;
    memcpy(&_idList[entry].ID, [net rawID], 6);
    _idList[entry].net=[net retain];
    
    //add to hash table
    l = hashForMAC(_idList[entry].ID);
    
    while (_lookup[l]!=LOOKUPSIZE) {
        l = (l + 1) % LOOKUPSIZE;
    }
    _lookup[l] = entry;
    
    [self addNetToView:entry];
    
    return YES;
}

- (bool) loadLegacyData:(NSDictionary*)data {
    NSEnumerator *e;
    id n;
    WaveNet *net;
        
    e = [data objectEnumerator];
    while (n = [e nextObject]) {
        if (![n isMemberOfClass:[WaveNet class]]) {
            NSLog(@"Could not load legacy data, because it is bad!");
            return NO;
        }
        
        if (_netCount == MAXNETS) {
            NSLog(@"Loaded more networks, but could not add them since you reached MAXNETS. Please recompile with a higher value");
            return YES;
        }
        
        net = n;
        
        [self addNetwork:net];
    }
    
    return YES;
}

- (bool) loadData:(NSArray*)data {
    id n;
    int i;
    WaveNet *net;
     
    for (i=0; i<[data count]; i++) {
        n = [data objectAtIndex:i];
        if (![n isMemberOfClass:[WaveNet class]]) {
            NSLog(@"Could not load data, because it is bad!");
            return NO;
        }
        
        if (_netCount == MAXNETS) {
            NSLog(@"Loaded more networks, but could not add them since you reached MAXNETS. Please recompile with a higher value");
            return YES;
        }
        
        net = n;
        
        [self addNetwork:net];
    }
    
    return YES;
}

- (bool) importLegacyData:(NSDictionary*)data {
    NSEnumerator *e;
    id n;
    WaveNet *net;
        
    e = [data objectEnumerator];
    while (n = [e nextObject]) {
        if (![n isMemberOfClass:[WaveNet class]]) {
            NSLog(@"Could not load legacy data, because it is bad!");
            return NO;
        }
        
        if (_netCount == MAXNETS) {
            NSLog(@"Loaded more networks, but could not add them since you reached MAXNETS. Please recompile with a higher value");
            return YES;
        }
        
        net = [self netForKey:[n rawID]];
        
        if (!net) [self addNetwork:n];
        else [net mergeWithNet:n];
    }
    
    return YES;
}

- (bool) importData:(NSArray*)data {
    id n;
    int i;
    WaveNet *net = nil;
     
    for (i=0; i<[data count]; i++) {
        n = [data objectAtIndex:i];
        if (![n isMemberOfClass:[WaveNet class]]) {
            NSLog(@"Could not load data, because it is bad!");
            return NO;
        }
        
        if (_netCount == MAXNETS) {
            NSLog(@"Loaded more networks, but could not add them since you reached MAXNETS. Please recompile with a higher value");
            return YES;
        }
        
        net = [self netForKey:[n rawID]];
        
        if (!net) [self addNetwork:n];
        else [net mergeWithNet:n];
    }
      
    return YES;
}

- (NSArray*) dataToSave {
    NSMutableArray *a;
    unsigned int i;
    
    a = [NSMutableArray arrayWithCapacity:_netCount];
    for (i=0; i<_netCount; i++)
        [a addObject:_idList[i].net];

    return a;
}

#pragma mark -


inline int compValues(int v1, int v2) {
    if (v1 < v2) return NSOrderedAscending;
    else if (v1 > v2) return NSOrderedDescending;
    else return NSOrderedSame;
}
int channelSort(WaveSort* p, const int *index1, const int *index2) {
    int v1 = [(*p).idList[*index1].net channel];
    int v2 = [(*p).idList[*index2].net channel];
    return (*p).ascend * compValues(v1,v2);
}
int idSort(WaveSort* p, const int *index1, const int *index2) {
    int v1 = [(*p).idList[*index1].net netID];
    int v2 = [(*p).idList[*index2].net netID];
    return (*p).ascend * compValues(v1,v2);
}

int bssidSort(WaveSort* p, const int *index1, const int *index2) {
    NSString *d1 = [(*p).idList[*index1].net BSSID];
    NSString *d2 = [(*p).idList[*index2].net BSSID];
    return (*p).ascend * [d1 compare:d2 options:NSLiteralSearch];
}

int ssidSort(WaveSort* p, const int *index1, const int *index2) {
    int i;
    NSString *d1 = [(*p).idList[*index1].net SSID];
    NSString *d2 = [(*p).idList[*index2].net SSID];
    i =  [d1 compare:d2 options:NSLiteralSearch|NSCaseInsensitiveSearch];
    return (*p).ascend * i;
}

int wepSort(WaveSort* p, const int *index1, const int *index2) {
    int v1 = [(*p).idList[*index1].net isWep];
    int v2 = [(*p).idList[*index2].net isWep];
    return (*p).ascend * compValues(v1,v2);
}

int typeSort(WaveSort* p, const int *index1, const int *index2) {
    int v1 = [(*p).idList[*index1].net type];
    int v2 = [(*p).idList[*index2].net type];
    return (*p).ascend * compValues(v1,v2);
}

int signalSort(WaveSort* p, const int *index1, const int *index2) {
    int v1 = [(*p).idList[*index1].net curSignal];
    int v2 = [(*p).idList[*index2].net curSignal];
    return (*p).ascend * compValues(v1,v2);
}

int maxSignalSort(WaveSort* p, const int *index1, const int *index2) {
    int v1 = [(*p).idList[*index1].net maxSignal];
    int v2 = [(*p).idList[*index2].net maxSignal];
    return (*p).ascend * compValues(v1,v2);
}

int avgSignalSort(WaveSort* p, const int *index1, const int *index2) {
    int v1 = [(*p).idList[*index1].net avgSignal];
    int v2 = [(*p).idList[*index2].net avgSignal];
    return (*p).ascend * compValues(v1,v2);
}

int packetsSort(WaveSort* p, const int *index1, const int *index2) {
    int v1 = [(*p).idList[*index1].net packets];
    int v2 = [(*p).idList[*index2].net packets];
    return (*p).ascend * compValues(v1,v2);
}

int dataSort(WaveSort* p, const int *index1, const int *index2) {
    float v1 = [(*p).idList[*index1].net dataCount];
    float v2 = [(*p).idList[*index2].net dataCount];
    if (v1 < v2) return (*p).ascend * NSOrderedAscending;
    else if (v1 > v2) return (*p).ascend * NSOrderedDescending;
    else return NSOrderedSame;
}

int lastSeenSort(WaveSort* p, const int *index1, const int *index2) {
    NSDate *d1 = [(*p).idList[*index1].net lastSeenDate];
    NSDate *d2 = [(*p).idList[*index2].net lastSeenDate];
    return (*p).ascend * [d1 compare:d2];
}

typedef int (*SORTFUNC)(void *, const void *, const void *);

- (void) sortByColumn:(NSString*)ident order:(bool)ascend {
    WaveSort ws;
    
    if (![_sortLock tryLock]) return;
    
    _ascend = ascend;
    ws.ascend = ascend ? 1 : -1;
    ws.idList = _idList;
    
    if ([ident isEqualToString:@"channel"])         qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)channelSort);
    else if ([ident isEqualToString:@"id"])         qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)idSort);
    else if ([ident isEqualToString:@"bssid"])      qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)bssidSort);
    else if ([ident isEqualToString:@"ssid"])       qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)ssidSort);
    else if ([ident isEqualToString:@"wep"])        qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)wepSort);
    else if ([ident isEqualToString:@"type"])       qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)typeSort);
    else if ([ident isEqualToString:@"signal"])     qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)signalSort);
    else if ([ident isEqualToString:@"maxsignal"])  qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)maxSignalSort);
    else if ([ident isEqualToString:@"avgsignal"])  qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)avgSignalSort);
    else if ([ident isEqualToString:@"packets"])    qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)packetsSort);
    else if ([ident isEqualToString:@"data"])       qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)dataSort);
    else if ([ident isEqualToString:@"lastseen"])   qsort_r(_sortedList, _sortedCount, sizeof(unsigned int), &ws, (SORTFUNC)lastSeenSort);
    else NSLog(@"Unknown sorting column. This is a bug and should never happen.");
    
    [_sortLock unlock];
}

- (void) sortWithShakerByColumn:(NSString*)ident order:(bool)ascend {
    SORTFUNC func;
    bool sorted = YES;
    int ret;
    unsigned int w, x, y, z;
    WaveSort ws;

    if (![_sortLock tryLock]) return;
    
    _ascend = ascend;
    ws.ascend = ascend ? 1 : -1;
    ws.idList = _idList;
    ws.sortedList = _sortedList;
    
    if ([ident isEqualToString:@"channel"]) func = (SORTFUNC)channelSort;
    else if ([ident isEqualToString:@"id"]) func = (SORTFUNC)idSort;
    else if ([ident isEqualToString:@"bssid"]) func = (SORTFUNC)bssidSort;
    else if ([ident isEqualToString:@"ssid"]) func = (SORTFUNC)ssidSort;
    else if ([ident isEqualToString:@"wep"]) func = (SORTFUNC)wepSort;
    else if ([ident isEqualToString:@"type"]) func = (SORTFUNC)typeSort;
    else if ([ident isEqualToString:@"signal"]) func = (SORTFUNC)signalSort;
    else if ([ident isEqualToString:@"maxsignal"]) func = (SORTFUNC)maxSignalSort;
    else if ([ident isEqualToString:@"avgsignal"]) func = (SORTFUNC)avgSignalSort;
    else if ([ident isEqualToString:@"packets"]) func = (SORTFUNC)packetsSort;
    else if ([ident isEqualToString:@"data"]) func = (SORTFUNC)dataSort;
    else if ([ident isEqualToString:@"lastseen"]) func = (SORTFUNC)lastSeenSort;
    else {
        [_sortLock unlock];
        NSLog(@"Unknown sorting column. This is a bug and should never happen.");
        return;
    }
    
    for (y = 1; y <= _sortedCount; y++) {

        for (x = y - 1; x < (_sortedCount - y); x++) {
            w = x + 1;
            ret = (*func)(&ws, &_sortedList[x], &_sortedList[w]);
            if (ret == NSOrderedDescending) {
                sorted = NO;
                
                //switch places
                z = _sortedList[x];
                _sortedList[x] = _sortedList[w];
                _sortedList[w] = z;
                
                _idList[_sortedList[x]].changed = YES;
                _idList[_sortedList[w]].changed = YES;
            }
        }
        
        if (sorted) break;
        sorted = YES;
        
        for (x = (_sortedCount - y); x >= y; x--) {
            w = x - 1;
            ret = (*func)(&ws, &_sortedList[w], &_sortedList[x]);
            if (ret == NSOrderedDescending) {
                sorted = NO;
                
                //switch places
                z = _sortedList[x];
                _sortedList[x] = _sortedList[w];
                _sortedList[w] = z;
                
                _idList[_sortedList[x]].changed = YES;
                _idList[_sortedList[w]].changed = YES;
            }
        }
        
        if (sorted) break;
        sorted = YES;
    }
    
    [_sortLock unlock];
}

#pragma mark -

- (unsigned int) findNetwork:(unsigned char*)ID {
    unsigned int i, lentry;
    unsigned int entry = 0xFFFFFFFF;
    unsigned int l = 0;
    
    
    //see if it is filtered
    for (i=0; i<_filterCount; i++)
        if (memcmp(ID, _filter[i], 6)==0) return 0xFFFFFFFF;

    
    //lookup the net in the hashtable
    l = hashForMAC(ID);
    lentry = l;
    
    i=_lookup[l];
    while (i!=LOOKUPSIZE) {
        if (memcmp(ID, _idList[i].ID, 6)==0) {
            entry = i;
            break;
        }
        l = (l + 1) % LOOKUPSIZE;
        i=_lookup[l];
    }
            
    if (entry==0xFFFFFFFF) {
        //the net does not exist - add it
        
        if (_netCount == MAXNETS) {
            NSLog(@"Found network, but could not add it since you reached MAXNETS. Please recompile with a higher value");
            return 0xFFFFFFFF;
        }
        
        _netCount++;
        entry = _netCount - 1;
        memcpy(&_idList[entry].ID, ID, 6);
        _idList[entry].net=[[WaveNet alloc] initWithID:entry];
        _lookup[l] = entry;

        [self addNetToView:entry];     
        
        [[NSNotificationCenter defaultCenter] postNotificationName:KisMACNetworkAdded object:self];   
    }
    
    if (l != lentry) {
        //optimize the hash table...
        i = _lookup[lentry];
        _lookup[lentry] = _lookup[l];
        _lookup[l] = i;
    }

    return entry;
}

- (bool) addPacket:(WavePacket*)p liveCapture:(bool)live {
    unsigned int entry;
    unsigned char ID[6];
    
    if (_dropAll) return YES;
    if (![p ID:ID]) return YES;
    
    entry = [self findNetwork:ID];
    if (entry == 0xFFFFFFF) return NO;                          //the object is filtered...
    
    [_idList[entry].net parsePacket:p withSound:live];		//add the packet to the network
    _idList[entry].changed = YES;
    return YES;
}

- (bool) addAppleAPIData:(WirelessNetworkInfo*)i {
    unsigned int entry;

    if (_dropAll) return YES;
    
    entry = [self findNetwork:i->macAddress];
    if (entry == 0xFFFFFFF) return NO;                          //the object is filtered...
    
    [_idList[entry].net parseAppleAPIData:i];                   //add the data to the network
    _idList[entry].changed = YES;
    
    return YES;
}

#pragma mark -

- (unsigned int) count {
    return _sortedCount;
}

- (WaveNet*) netAtIndex:(unsigned int)index {
    if (index >= _sortedCount) return 0;
    return _idList[_sortedList[index]].net;
}
- (WaveNet*) netForKey:(unsigned char*) ID {
    unsigned int i, l, entry = LOOKUPSIZE;
    
    //lookup the net in the hashtable
    l = hashForMAC(ID);
    
    i=_lookup[l];
    while (i!=LOOKUPSIZE) {
        if (memcmp(ID, _idList[i].ID, 6)==0) {
                entry = i;
                break;
        }
        l = (l + 1) % LOOKUPSIZE;
        i=_lookup[l];
    }
    
    if (entry==LOOKUPSIZE) return nil;
    
    return _idList[entry].net;
}
- (NSMutableArray*) allNets {
    NSMutableArray *a;
    unsigned int i;
    
    a = [NSMutableArray array];
    for (i=0; i<_sortedCount; i++)
        [a addObject:_idList[_sortedList[i]].net];
    
    return a;
}

- (void) scanUpdate:(int)graphLength {
    unsigned int i;
    
    for(i = 0 ; i < _netCount ; i++) {
        if ([_idList[i].net noteFinishedSweep:graphLength]) {
            //make sure this is going to be updated
            _idList[i].changed = YES;
        }
    }
}

- (void) ackChanges {
    unsigned int i;
    for(i = 0 ; i < _netCount ; i++) _idList[i].changed = NO;
}

- (unsigned int) nextChangedRow:(unsigned int)lastRow {
    unsigned int nxt;
    
    if (lastRow==0xFFFFFFFF) nxt=0;
    else nxt = lastRow + 1;
    
    while (nxt < _sortedCount) {
        if (_idList[_sortedList[nxt]].changed) return nxt;
        nxt++;
    }

    return 0xFFFFFFFF;
}
#pragma mark -

-(void) clearAllEntries {
    int i, oldcount;
    WaveNet *e;
    
    _dropAll = YES;
    [NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:0.2]];
    
    oldcount = _netCount;
    for(i=0; i<LOOKUPSIZE; i++) _lookup[i]=LOOKUPSIZE;
    _sortedCount = 0;
    _netCount = 0;

    for (i=0; i<oldcount; i++) {
        e = _idList[i].net;
        _idList[i].net = Nil;
        [e release];
    }
    
    _dropAll = NO;
}

- (void) clearEntry:(WaveNet*)net {
    unsigned char *ID = [net rawID];
    unsigned int i, l, firstentry, entry = LOOKUPSIZE, lentry = LOOKUPSIZE;
    
    _dropAll = YES;
    [NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:0.2]];
    
    //delete the net in the hashtable
    l = hashForMAC(ID);
    firstentry = l;
    
    i=_lookup[l];
    while (i!=LOOKUPSIZE) {
        if (memcmp(ID, _idList[i].ID, 6)==0) entry = i;
        else if (entry!=LOOKUPSIZE) {
            if (hashForMAC(_idList[i].ID)==l) lentry = l;
        }
        l = (l + 1) % LOOKUPSIZE;
        i=_lookup[l];
    }
    
    if (entry==LOOKUPSIZE) {
        NSAssert(i<_sortedCount, @"Error: net is not in hash table. This is prohibited!");
        return;
    }
  
    if (lentry!=LOOKUPSIZE) {
        _lookup[firstentry] = _lookup[lentry];
        _lookup[lentry] = LOOKUPSIZE;
    } else _lookup[firstentry] = LOOKUPSIZE;
    
    for (i=0; i<_sortedCount; i++)
        if (memcmp(ID, _idList[_sortedList[i]].ID, 6)==0) break;
    
    if(i<_sortedCount) {
        _sortedCount--;
        for (;i<_sortedCount; i++)
            _sortedList[i]=_sortedList[i+1];
    }
    
    net = _idList[entry].net;
    _netCount--;
    for (i=entry; i<_netCount; i++) {
        _idList[i]=_idList[i+1];
    }
    if (entry!=i) _idList[i+1].net = nil;
    else _idList[entry].net = nil;
    
    
    [net release];
    
    _dropAll = NO;
        
    return;
}

-(void) dealloc {
    [[NSNotificationCenter defaultCenter] removeObserver:self];

    [self clearAllEntries];
    [_sortLock release];
}

@end
