/*
        
        File:			WaveCracker.mm
        Program:		KisMAC
	Author:			Dylan Neild, Michael Roberg
				mick@binaervarianz.de
	Description:		KisMAC is a wireless stumbler for MacOS X.
                
        This file is part of KisMAC.
        
    KisMAC is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    KisMAC is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KisMAC; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#import "WaveCracker.h"
#import "WaveHelper.h"
#import "ScanController.h"

//#import "md5.h"

typedef unsigned long int UNS_32_BITS;

#define UPDC32(octet,crc) (crc_32_tab[((crc) ^ (octet)) & 0xff] ^ ((crc) >> 8))

static UNS_32_BITS crc_32_tab[] = { /* CRC polynomial 0xedb88320 */
    0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f,
    0xe963a535, 0x9e6495a3, 0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
    0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91, 0x1db71064, 0x6ab020f2,
    0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
    0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9,
    0xfa0f3d63, 0x8d080df5, 0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
    0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b, 0x35b5a8fa, 0x42b2986c,
    0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
    0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423,
    0xcfba9599, 0xb8bda50f, 0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
    0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d, 0x76dc4190, 0x01db7106,
    0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
    0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d,
    0x91646c97, 0xe6635c01, 0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
    0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457, 0x65b0d9c6, 0x12b7e950,
    0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
    0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7,
    0xa4d1c46d, 0xd3d6f4fb, 0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
    0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9, 0x5005713c, 0x270241aa,
    0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
    0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81,
    0xb7bd5c3b, 0xc0ba6cad, 0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
    0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683, 0xe3630b12, 0x94643b84,
    0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
    0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb,
    0x196c3671, 0x6e6b06e7, 0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
    0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5, 0xd6d6a3e8, 0xa1d1937e,
    0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
    0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55,
    0x316e8eef, 0x4669be79, 0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
    0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f, 0xc5ba3bbe, 0xb2bd0b28,
    0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
    0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f,
    0x72076785, 0x05005713, 0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
    0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21, 0x86d3d2d4, 0xf1d4e242,
    0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
    0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69,
    0x616bffd3, 0x166ccf45, 0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
    0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db, 0xaed16a4a, 0xd9d65adc,
    0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
    0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693,
    0x54de5729, 0x23d967bf, 0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
    0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};

@implementation WaveCracker

-(id) init {
    unsigned int i;
    self=[super init];
    
    setupIdentity();		//initialized the RC4 sboxes
    
    for (i = 0; i < 256; i++) {	    //precache crack data
        aMaster[i].score=0;
        aMaster[i].index=i;
    }
    aIsInit=NO;
    return self;
}

//check whether a certain key is right, by deciphering examples
- (int) checkKey:(NSArray*)packets key:(unsigned char*)k {
    int w;
    unsigned int i, h;
    unsigned char key[16];
    unsigned long crc;
    RC4 rc;
    
    //copy the guess without the iv
    memcpy(key+3, k, _keywidth);

    //go thought all examples
    for(i=0;i<[packets count];i++) {
        if (!aIsInit) {
            [(NSString*)[packets objectAtIndex:i] getCString:(char*)aData];
            aLength=[(NSString*)[packets objectAtIndex:i] length];
            memcpy(key, aData, 3);
            if (i==0) aIsInit=YES;
        }
        //decipher the packet
        RC4InitWithKey(&rc, key, _keywidth+3); 
        crc=0xffffffff;
        for(h = 4; h < aLength; h++) crc=UPDC32(aData[h]^step(&rc),crc);
        //check whether checksum is correct
        if (crc != 0xdebb20e3) return 0;
        aIsInit=NO;
    }

    //return key in hexadecimal
    aKey=[[NSMutableString stringWithFormat:@"%.2X", aCurGuess[3]] retain];
    for (w=4;w<(_keywidth+3);w++)
        [aKey appendString:[NSString stringWithFormat:@":%.2X", aCurGuess[w]]];
    return 1;
}

//sots all guesses based on their score
int score_compare(const void *a,const void *b) {
  return ((struct sScore*) b)->score-((struct sScore*) a)->score;
}

//does the actual cracking for byte "level"
-(int) performCrackWithDepth:(int)level brute:(bool)doBrute {
    struct sScore aScore[256];
    NSEnumerator* e;
    NSNumber* key;
    unsigned int iv,i;
    int h, j=0;
    NSDictionary *d;
    
    //if we hav the maximum size of the key check its correctness
    if (level == _keywidth) return [self checkKey:[_net weakPacketsLog] key:aCurGuess+3];
    
    //init
    memcpy(aScore,aMaster,sizeof(struct sScore)*256);
    
    d = [_net weakPacketsDict];
    //cycle through all weak packets, and give them a rating
    e=[[d objectForKey:[NSNumber numberWithInt:level]] keyEnumerator];
    while (key = [e nextObject]) {
        iv=[key unsignedIntValue];
        memcpy(aCurGuess,((char*)&iv)+1,3);
        //make a guess
        h = tryIVx(aCurGuess, level, [[[d objectForKey:[NSNumber numberWithInt:level]] objectForKey:key] unsignedCharValue],&j);
        aScore[j].score+=h;
    }

    //sort all possiblities based on their rating
    qsort(aScore, 256, sizeof(struct sScore), score_compare);
    for(i = 0; i < _breath; i++) {
        if (((level==1)&&(_keywidth==5))||((level==3)&&(_keywidth==13))) {
            if (doBrute) [_im increment];
            if ([_im canceled]) return 1;
        }
        
        if (aScore[i].score == 0) return 0;	//dont have any more weak packets for this byte
        aCurGuess[3+level] = aScore[i].index;	//set the current guess
        if ([self performCrackWithDepth:level + 1 brute:doBrute] == 1) return 1;
    }

    if (doBrute&&([[d objectForKey:[NSNumber numberWithInt:level]] count]<60)) for(i = 0; i < 256; i++) {
        aCurGuess[3+level] = aScore[i].index;	//set the current guess
        if ([self performCrackWithDepth:level + 1 brute:false] == 1) return 1;
    }
    
    return 0;
}

- (void)weakThread:(id)object {
    NSAutoreleasePool* subpool = [[NSAutoreleasePool alloc] init];

    [self performCrackWithDepth:0 brute:false];
    
    [[NSNotificationCenter defaultCenter] postNotificationName:KisMACCrackDone object:self];

    [subpool release];
}

//public function to crack based on weak packets
-(void) crackWithKeyByteLength:(unsigned int)a net:(WaveNet*)aNet breath:(unsigned int)b import:(ImportController*)i {
    for (int p=0;p<13;p++)
        NSLog(@"WeakPackets for %u. KeyByte: %u.\n",p+1,[[[aNet weakPacketsDict] objectForKey:[NSNumber numberWithInt:p]] count]);
    
    if (aKey==Nil) {
        _im = i;
        _net = aNet;
        _keywidth = a;
        _breath = b;
        
        [NSThread detachNewThreadSelector:@selector(weakThread:) toTarget:self withObject:nil];
    }
}

//public function - returns a cracked key, nil otherwise
-(NSString*) key {
    return aKey;
}

-(void) dealloc {
    [aKey release];
    [super dealloc];
}

// BRUTE FORCE FUNCTIONS
// EVERYTHING BELOW THIS LINE REPRESENTS AN ADAPTATION/PATCH BY DYLAN NEILD (dylann@mac.com)
// 
// 1. Floating point function is now VERY fast (almost 4x faster). There is no function calling (except to grab the packet). 
//    The key increment system is completely inlined and uses -no- recursion, so in -O3 mode, it's -very- fast. 
// 
// 2. CRC is calculated on the fly, per byte. We don't need to actually store the decrypted data when we're finished with it,
//    so rather then using lData[] to store data, we simply decrypt and CRC it as we go.  
// 
// 3. 3 seperate function are now included, for "All Characters", "Alpha Numeric Characters" and "Lower Case" characters.
//    This is so we don't need to use a function pointer for keyincrement (and the key increment itself is now inlined). 
//    This is a trivial performance boost, but a boost none the less.
//
// 4. The RC4 setup sequence (setting up the initial 0-256 array of unsigned chars) is no longer duplicate per key, as it's
//    a waste of 256+ cpu cycles. We build it once, then assign it in place per key. 

- (void)crackThread:(id)object {
    NSAutoreleasePool* subpool = [[NSAutoreleasePool alloc] init];
    unsigned int x;
    
    switch (_method) {
        case 0:
            switch (_chars) {
                case 1: 
                    [self performSelectedBruteForce40_alpha:_net];
                    break;
                case 2:
                    [self performSelectedBruteForce40_low:_net];
                    break;
                case 3:
                    [self performSelectedBruteForce40_ascii:_net];
                    break;
                default:
                    [self performSelectedBruteForce40_all:_net];
            }
            break;
        case 1:
            for (x=0; x<[_filenames count]; x++) {
                switch (_chars) {
                    case 1:
                        [self doWordlistAttack40Apple:_net withWordlist:[_filenames objectAtIndex:x]];
                        break;
                    case 2:
                        [self doWordlistAttack104Apple:_net withWordlist:[_filenames objectAtIndex:x]];
                        break;
                    case 3:
                        [self doWordlistAttack104MD5:_net withWordlist:[_filenames objectAtIndex:x]];
                        break;
                }
            }
            break;
    }
    
    [[NSNotificationCenter defaultCenter] postNotificationName:KisMACCrackDone object:self];
    
    [subpool release];
}

- (int) performSelectedBruteForce40:(WaveNet*)aNet import:(ImportController*)im charset:(int)chars {
    _im = im;
    _chars = chars;
    _net = aNet;
    _method = 0;
    
    [NSThread detachNewThreadSelector:@selector(crackThread:) toTarget:self withObject:nil];

    return 0;
}

-(void) crackWithWordlist:(NSArray*) filenames useCipher:(unsigned int)a net:(WaveNet*)aNet import:(ImportController*)i {
    _im = i;
    _chars = a;
    _net = aNet;
    _method = 1;
    
    [_filenames release];
    _filenames = [filenames retain];
    
    [NSThread detachNewThreadSelector:@selector(crackThread:) toTarget:self withObject:nil];
}

// this is a brute force attack based on an attack by Timothy Newsham
// more infos are at http://www.lava.net/~newsham/wlan/
// does 21 bit attack against all keys in a 40-bit network
#define KEYLENGTH 5
#define KEYNUM 4
- (int) performSelectedBruteForce40_ascii:(WaveNet*)aNet {	
    unsigned char key[KEYLENGTH * KEYNUM], skeletonStateArray[256], currentStateArray[256];
    unsigned int i, foundCRC, counter;
    unsigned char y, z, tmp, xov, l, j;
    unsigned int w, x, q, k;
    if (aKey!=Nil) return 1;
    aIsInit = false;
    
    memset(key,0,16);
    l = 0;
    q = 0;
    j = 0;
    k = 0;
    
    //if we want to do it against 2,3,4 key we have to modifiy this
    for(x = 0; x < KEYLENGTH * 1; x++) {
        q *= 0x343fd;
        q += 0x269ec3;
        key[x] = q >> 16;
    }
    w=0;
    
    for (counter = 0; counter < 256; counter++) 
        skeletonStateArray[counter] = counter;
    
    while(true) {
        for(i=0;i<[[aNet weakPacketsLog] count];i++) {

            if (!aIsInit) {	
                
                [(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] getCString:(char*)aData];
                aLength=[(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] length];
                
                memcpy(key, aData, 3);
                
                if (i==0) aIsInit=YES;
            }
            
            memcpy(currentStateArray, skeletonStateArray, 256);
            y = z = 0;
            
            for (counter = 0; counter < 256; counter++) {
                z = (key[y] + currentStateArray[counter] + z);
            
                tmp = currentStateArray[counter];
                currentStateArray[counter] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                y = (y + 1) % 8;
            }
            
            foundCRC = 0xFFFFFFFF;
            y = z = 0;
                        
            for (counter = 4; counter < aLength; counter++) {
                y++;
                z = currentStateArray[y] + z;
                
                tmp = currentStateArray[y];
                currentStateArray[y] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                xov = currentStateArray[y] + currentStateArray[z];

                foundCRC = UPDC32((aData[counter] ^ currentStateArray[xov]), foundCRC);
            }
            
            if (foundCRC == 0xdebb20e3) {
                memcpy(&aCurGuess, &key, 16);
                aIsInit=NO;
            }
            else 
                break;
        }

        if (i < 5) {
            if ((k++)==3) {
                k=0;
                while (++w & 0x80808080);
                if (w>0x1000000) return 0;

                if (((char*)(&w))[1]!=j) {
                    j=((char*)(&w))[1];
                    if ([_im canceled]) return 0;
                    [_im increment];
                }
                q = (w * 0x343fd) + 0x269ec3;
            } else
                q = (q * 0x343fd) + 0x269ec3;
            key[3] = q>>16;
            
            q = (q * 0x343fd) + 0x269ec3;
            key[4] = q >> 16;

            q = (q * 0x343fd) + 0x269ec3;
            key[5] = q >> 16;

            q = (q * 0x343fd) + 0x269ec3;
            key[6] = q >> 16;

            q = (q * 0x343fd) + 0x269ec3;
            key[7] = q >> 16;
        } else {
            aKey=[[NSMutableString stringWithFormat:@"%.2X", aCurGuess[3]] retain];
            for (i=4;i<(8);i++)
                [aKey appendString:[NSString stringWithFormat:@":%.2X", aCurGuess[i]]];

            return 1;
        }
    }
}

- (int) performSelectedBruteForce40_all:(WaveNet*)aNet {

    unsigned char key[16], skeletonStateArray[256], currentStateArray[256];
    unsigned int i, foundCRC, counter;
    unsigned char y, z, tmp, xov;
    NSString *packet;
    
    if (aKey!=Nil) return 1;
    aIsInit = false;
    
    memset(key,0,16);
    
    for (counter = 0; counter < 256; counter++) 
        skeletonStateArray[counter] = counter;
    
    while(true) {

        for(i=0;i<[[aNet weakPacketsLog] count];i++) {

            if (!aIsInit) {	
                
                packet = [[aNet weakPacketsLog] objectAtIndex:i];
                [packet getCString:(char*)aData];
                aLength=[packet length];
                
                memcpy(key, aData, 3);
                
                if (i==0) aIsInit=YES;
            }
            
            memcpy(currentStateArray, skeletonStateArray, 256);
            y = z = 0;
            
            for (counter = 0; counter < 256; counter++) {
                z = (key[y] + currentStateArray[counter] + z);
            
                tmp = currentStateArray[counter];
                currentStateArray[counter] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                y = (y + 1) % 8;
            }
            
            foundCRC = 0xFFFFFFFF;
            y = z = 0;
                        
            for (counter = 4; counter < aLength; counter++) {
                y++;
                z = currentStateArray[y] + z;
                
                tmp = currentStateArray[y];
                currentStateArray[y] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                xov = currentStateArray[y] + currentStateArray[z];

                foundCRC = UPDC32((aData[counter] ^ currentStateArray[xov]), foundCRC);
            }

            if (foundCRC == 0xdebb20e3) {
                memcpy(&aCurGuess, &key, 16);
                aIsInit=NO;
            }
            else 
                break;
        }

        if (i < 10) {
            key[3]++;

            if (key[3]==0) {
                key[4]++;

                if (key[4]==0) {
                    key[5]++;
                    if ([_im canceled]) return 0;
                        
                    if (key[5]==0) {
                        key[6]++;
                        
                        if (key[6]==0) {
                            key[7]++;

                            if (key[7]==0) 
                                return 0;
                            else {
                                [_im increment];
                            }
                        }
                    }
                }
            }                        
        }
        else {
            aKey=[[NSMutableString stringWithFormat:@"%.2X", aCurGuess[3]] retain];
            for (i=4;i<(8);i++)
                [aKey appendString:[NSString stringWithFormat:@":%.2X", aCurGuess[i]]];

            return 1;
        }
    }
}

- (int) performSelectedBruteForce40_alpha:(WaveNet*)aNet {

    unsigned int i, foundCRC, counter;
    unsigned char key[16], skeletonStateArray[256], currentStateArray[256];
    unsigned char y, z, tmp, xov;
    
    if (aKey!=Nil) return 1;
    aIsInit = false;
    
    memset(key,32,16);

    for (counter = 0; counter < 256; counter++) 
        skeletonStateArray[counter] = counter;
    
    while(true) {

        for(i=0;i<[[aNet weakPacketsLog] count];i++) {

            if (!aIsInit) {	
                
                [(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] getCString:(char*)aData];
                aLength=[(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] length];
                
                memcpy(key, aData, 3);
                
                if (i==0) aIsInit=YES;
            }
            
            memcpy(currentStateArray, skeletonStateArray, 256);
            y = z = 0;
            
            for (counter = 0; counter < 256; counter++) {
                z = (key[y] + currentStateArray[counter] + z);
            
                tmp = currentStateArray[counter];
                currentStateArray[counter] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                y = (y + 1) % 8;
            }
            
            foundCRC = 0xFFFFFFFF;
            y = z = 0;
                        
            for (counter = 4; counter < aLength; counter++) {
                y++;
                z = currentStateArray[y] + z;
                
                tmp = currentStateArray[y];
                currentStateArray[y] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                xov = currentStateArray[y] + currentStateArray[z];

                foundCRC = UPDC32((aData[counter] ^ currentStateArray[xov]), foundCRC);
            }

            if (foundCRC == 0xdebb20e3) {
                memcpy(&aCurGuess, &key, 16);
                aIsInit=NO;
            }
            else 
                break;
        }

        if (i < 10) {
            
            if (key[3]==32) key[3]=48;
            else if (key[3]==57) key[3]=65;
            else if (key[3]==90) key[3]=97;
            else key[3]++;
            
            if (key[3]>122) {
                key[3]=32;
        
                if (key[4]==32) key[4]=48;
                else if (key[4]==57) key[4]=65;
                else if (key[4]==90) key[4]=97;
                else key[4]++;
                
                if (key[4]>122) {
                    key[4]=32;
        
                    if (key[5]==32) key[5]=48;
                    else if (key[5]==57) key[5]=65;
                    else if (key[5]==90) key[5]=97;
                    else key[5]++;
                    
                    if (key[5]>122) {
                        key[5]=32;
                
                        if (key[6]==32) key[6]=48;
                        else if (key[6]==57) key[6]=65;
                        else if (key[6]==90) key[6]=97;
                        else key[6]++;
                        
                        if ([_im canceled]) return 0;
                        if (key[6]>122) {
                            key[6]=32;
        
                            if (key[7]==32) key[7]=48;
                            else if (key[7]==57) key[7]=65;
                            else if (key[7]==90) key[7]=97;
                            else key[7]++;
                            
                            if (key[7]>122) 
                                return 0;
                            else {
                                [_im increment];
                            }
                        }
                    }
                }
            }    
                                
        }
        else {
            aKey=[[NSMutableString stringWithFormat:@"%.2X", aCurGuess[3]] retain];
            for (i=4;i<(8);i++)
                [aKey appendString:[NSString stringWithFormat:@":%.2X", aCurGuess[i]]];

            return 1;
        }
    }
}

- (int) performSelectedBruteForce40_low:(WaveNet*)aNet {

    unsigned int i, foundCRC, counter;
    unsigned char key[16], skeletonStateArray[256], currentStateArray[256];
    unsigned char y, z, tmp, xov;
    
    if (aKey!=Nil) return 1;
    aIsInit = false;
    
    //select the right increment function for each character set

    memset(key,97,16);
    
    for (counter = 0; counter < 256; counter++) 
        skeletonStateArray[counter] = counter;
    
    while(true) {

        for(i=0;i<[[aNet weakPacketsLog] count];i++) {

            if (!aIsInit) {	
                
                [(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] getCString:(char*)aData];
                aLength=[(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] length];
                
                memcpy(key, aData, 3);
                
                if (i==0) aIsInit=YES;
            }
            
            memcpy(currentStateArray, skeletonStateArray, 256);
            y = z = 0;
            
            for (counter = 0; counter < 256; counter++) {
                z = (key[y] + currentStateArray[counter] + z);
            
                tmp = currentStateArray[counter];
                currentStateArray[counter] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                y = (y + 1) % 8;
            }
            
            foundCRC = 0xFFFFFFFF;
            y = z = 0;
                        
            for (counter = 4; counter < aLength; counter++) {
                y++;
                z = currentStateArray[y] + z;
                
                tmp = currentStateArray[y];
                currentStateArray[y] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                xov = currentStateArray[y] + currentStateArray[z];

                foundCRC = UPDC32((aData[counter] ^ currentStateArray[xov]), foundCRC);
            }

            if (foundCRC == 0xdebb20e3) {
                memcpy(&aCurGuess, &key, 16);
                aIsInit=NO;
            }
            else 
                break;
        }

        if (i < 10) {
            
            key[3]++;
            if (key[3]>122) {
                key[3]=97;
                key[4]++;
        
                if (key[4]>122) {
                    key[4]=97;    
                    key[5]++;
        
                    if (key[5]>122) {
                        key[5]=97;        
                        key[6]++;
                        if ([_im canceled]) return 0;
        
                        if (key[6]>122) {
                            key[6]=97;            
                            key[7]++;
                            
                            if (key[7]>122)
                                return 0;
                            else {
                                [_im increment];
                            }
                        }        
                    }
                }
            }
        }
        else {
            aKey=[[NSMutableString stringWithFormat:@"%.2X", aCurGuess[3]] retain];
            for (i=4;i<(8);i++)
                [aKey appendString:[NSString stringWithFormat:@":%.2X", aCurGuess[i]]];

            return 1;
        }
    }
}

-(int)doWordlistAttack40Apple:(WaveNet *)aNet withWordlist:(NSString*)filename {
    FILE* fptr;
    char wrd[1000];
    unsigned int i, words, foundCRC, counter;
    unsigned char key[16], skeletonStateArray[256], currentStateArray[256];
    unsigned char y, z, tmp, xov;
    
    if (aKey!=Nil) return 1;
    aIsInit = false;
    
    fptr = fopen([filename cString], "r");
    if (!fptr) return 0;    
    
    //select the right increment function for each character set
    
    for (counter = 0; counter < 256; counter++) 
        skeletonStateArray[counter] = counter;
    
    words = 0;
    wrd[990]=0;

    while(![_im canceled] && !feof(fptr)) {
        fgets(wrd, 990, fptr);
        i = strlen(wrd) - 1;
        wrd[i] = 0;
        if (wrd[i - 1]=='\r') wrd[--i] = 0;
        
        words++;
        
        WirelessEncrypt((CFStringRef)[NSString stringWithCString:wrd length:i],(WirelessKey*)(key+3),0);

        for(i=0;i<[[aNet weakPacketsLog] count];i++) {

            if (!aIsInit) {	
                
                [(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] getCString:(char*)aData];
                aLength=[(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] length];
                
                memcpy(key, aData, 3);
                
                if (i==0) aIsInit=YES;
            }
            
            memcpy(currentStateArray, skeletonStateArray, 256);
            y = z = 0;
            
            for (counter = 0; counter < 256; counter++) {
                z = (key[y] + currentStateArray[counter] + z);
            
                tmp = currentStateArray[counter];
                currentStateArray[counter] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                y = (y + 1) % 8;
            }
            
            foundCRC = 0xFFFFFFFF;
            y = z = 0;
                        
            for (counter = 4; counter < aLength; counter++) {
                y++;
                z = currentStateArray[y] + z;
                
                tmp = currentStateArray[y];
                currentStateArray[y] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                xov = currentStateArray[y] + currentStateArray[z];

                foundCRC = UPDC32((aData[counter] ^ currentStateArray[xov]), foundCRC);
            }

            if (foundCRC == 0xdebb20e3) {
                memcpy(&aCurGuess, &key, 16);
                aIsInit=NO;
            }
            else 
                break;
        }

        if (i >= 10) {
            aKey=[[NSMutableString stringWithFormat:@"%.2X", aCurGuess[3]] retain];
            for (i=4;i<(8);i++)
                [aKey appendString:[NSString stringWithFormat:@":%.2X", aCurGuess[i]]];
            fclose(fptr);
            NSLog(@"Cracking was successful. Password is <%s>", wrd);
            return 1;
        }
        if (words % 10000 == 0) {
            [_im setStatusField:[NSString stringWithFormat:@"%d words tested", words]];
        }
    }
    
    fclose(fptr);
    return 0;
}

-(int)doWordlistAttack104Apple:(WaveNet *)aNet withWordlist:(NSString*)filename {
    FILE* fptr;
    char wrd[1000];
    unsigned int i, words, foundCRC, counter;
    unsigned char key[16], skeletonStateArray[256], currentStateArray[256];
    unsigned char y, z, tmp, xov;
    
    if (aKey!=Nil) return 1;
    aIsInit = false;
    
    fptr = fopen([filename cString], "r");
    if (!fptr) return 0;    
    
    //select the right increment function for each character set
    
    for (counter = 0; counter < 256; counter++) 
        skeletonStateArray[counter] = counter;
    
    words = 0;
    wrd[990]=0;

    while(![_im canceled] && !feof(fptr)) {
        fgets(wrd, 990, fptr);
        i = strlen(wrd) - 1;
        wrd[i] = 0;
        if (wrd[i - 1]=='\r') wrd[--i] = 0;
        
        words++;
        
        WirelessEncrypt((CFStringRef)[NSString stringWithCString:wrd length:i],(WirelessKey*)(key+3),1);

        for(i=0;i<[[aNet weakPacketsLog] count];i++) {

            if (!aIsInit) {	
                
                [(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] getCString:(char*)aData];
                aLength=[(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] length];
                
                memcpy(key, aData, 3);
                
                if (i==0) aIsInit=YES;
            }
            
            memcpy(currentStateArray, skeletonStateArray, 256);
            y = z = 0;
            
            for (counter = 0; counter < 256; counter++) {
                z = (key[y] + currentStateArray[counter] + z);
            
                tmp = currentStateArray[counter];
                currentStateArray[counter] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                y = (y + 1) % 16;
            }
            
            foundCRC = 0xFFFFFFFF;
            y = z = 0;
                        
            for (counter = 4; counter < aLength; counter++) {
                y++;
                z = currentStateArray[y] + z;
                
                tmp = currentStateArray[y];
                currentStateArray[y] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                xov = currentStateArray[y] + currentStateArray[z];

                foundCRC = UPDC32((aData[counter] ^ currentStateArray[xov]), foundCRC);
            }

            if (foundCRC == 0xdebb20e3) {
                memcpy(&aCurGuess, &key, 16);
                aIsInit=NO;
            }
            else 
                break;
        }

        if (i >= 10) {
            aKey=[[NSMutableString stringWithFormat:@"%.2X", aCurGuess[3]] retain];
            for (i=4;i<(16);i++)
                [aKey appendString:[NSString stringWithFormat:@":%.2X", aCurGuess[i]]];
            fclose(fptr);
            NSLog(@"Cracking was successful. Password is <%s>", wrd);
            return 1;
        }
        if (words % 10000 == 0) {
            [_im setStatusField:[NSString stringWithFormat:@"%d words tested", words]];
        }
    }
    
    fclose(fptr);
    return 0;
}

-(int)doWordlistAttack104MD5:(WaveNet *)aNet withWordlist:(NSString*)filename {
    FILE* fptr;
    char wrd[1000];
    unsigned int i, words, foundCRC, counter;
    unsigned char key[16], skeletonStateArray[256], currentStateArray[256];
    unsigned char y, z, tmp, xov;
    
    if (aKey!=Nil) return 1;
    aIsInit = false;
    
    fptr = fopen([filename cString], "r");
    if (!fptr) return 0;
    
    //select the right increment function for each character set
    
    for (counter = 0; counter < 256; counter++) 
        skeletonStateArray[counter] = counter;
    
    words = 0;
    wrd[990]=0;

    while(![_im canceled] && !feof(fptr)) {
        fgets(wrd, 990, fptr);
        i = strlen(wrd) - 1;
        wrd[i--] = 0;
        if (wrd[i]=='\r') wrd[i] = 0;
        
        words++;
        
        WirelessCryptMD5(wrd, key+3);

        for(i=0;i<[[aNet weakPacketsLog] count];i++) {

            if (!aIsInit) {	
                
                [(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] getCString:(char*)aData];
                aLength=[(NSString*)[[aNet weakPacketsLog] objectAtIndex:i] length];
                
                memcpy(key, aData, 3);
                
                if (i==0) aIsInit=YES;
            }
            
            memcpy(currentStateArray, skeletonStateArray, 256);
            y = z = 0;
            
            for (counter = 0; counter < 256; counter++) {
                z = (key[y] + currentStateArray[counter] + z);
            
                tmp = currentStateArray[counter];
                currentStateArray[counter] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                y = (y + 1) % 16;
            }
            
            foundCRC = 0xFFFFFFFF;
            y = z = 0;
                        
            for (counter = 4; counter < aLength; counter++) {
                y++;
                z = currentStateArray[y] + z;
                
                tmp = currentStateArray[y];
                currentStateArray[y] = currentStateArray[z];
                currentStateArray[z] = tmp;
                
                xov = currentStateArray[y] + currentStateArray[z];

                foundCRC = UPDC32((aData[counter] ^ currentStateArray[xov]), foundCRC);
            }

            if (foundCRC == 0xdebb20e3) {
                memcpy(&aCurGuess, &key, 16);
                aIsInit=NO;
            }
            else 
                break;
        }

        if (i >= 10) {
            aKey=[[NSMutableString stringWithFormat:@"%.2X", aCurGuess[3]] retain];
            for (i=4;i<(16);i++)
                [aKey appendString:[NSString stringWithFormat:@":%.2X", aCurGuess[i]]];
            fclose(fptr);
            NSLog(@"Cracking was successful. Password is <%s>", wrd);
            return 1;
        }
        
        if (words % 10000 == 0) {
            [_im setStatusField:[NSString stringWithFormat:@"%d words tested", words]];
        }
    }
    
    fclose(fptr);
    return 0;
}

@end
