/*
        
        File:			WaveNet.mm
        Program:		KisMAC
	Author:			Michael Roberg
				mick@binaervarianz.de
	Description:		KisMAC is a wireless stumbler for MacOS X.
                
        This file is part of KisMAC.

    KisMAC is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    KisMAC is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KisMAC; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#import <AppKit/NSSound.h>
#import "WaveNet.h"
#import "WaveCracker.h"
#import "WaveClient.h"
#import "WaveHelper.h"
#import "80211b.h"
#import "ScanController.h"
#import "ValuePair.h"

#define AMOD(x, y) ((x) % (y) < 0 ? ((x) % (y)) + (y) : (x) % (y))
#define N 256

int lengthSort(id string1, id string2, void *context)
{
    int v1 = [(NSString*)string1 length];
    int v2 = [(NSString*)string2 length];
    if (v1 < v2)
        return NSOrderedAscending;
    else if (v1 > v2)
        return NSOrderedDescending;
    else
        return NSOrderedSame;
}

@implementation WaveNet

-(id)initWithID:(int)netID {
    self = [super init];
    
    if (!self) return nil;
    
    _dataLock = [[NSRecursiveLock alloc] init];
    [_dataLock lock];
    aNetView = [[NetView alloc] initWithFrame:NSMakeRect(0, 0, 200, 30)];
    aID = nil;
    
    aWeak=[[NSMutableDictionary dictionaryWithCapacity:13] retain];
    aPacketsLog=[[NSMutableArray arrayWithCapacity:20] retain];
    aARPLog=[[NSMutableArray arrayWithCapacity:20] retain];
    aACKLog=[[NSMutableArray arrayWithCapacity:20] retain];
    aClients=[[NSMutableDictionary dictionary] retain];
    aClientKeys=[[NSMutableArray array] retain];
    aComment=[[NSString stringWithString:@""] retain];
    aLat = [[NSString stringWithString:@""] retain];
    aLong = [[NSString stringWithString:@""] retain];
    aElev = [[NSString stringWithString:@""] retain];
    _coordinates = [[NSMutableDictionary dictionary] retain];
    aNetID=netID;

    _gotData = NO;
    recentTraffic = 0;
    curTraffic = 0;
    curPackets = 0;
    aCurSignal = 0;
    curTrafficData = 0;
    curPacketData = 0;
    memset(graphData.trafficData,0,(MAX_YIELD_SIZE + 1) * sizeof(int));
    memset(graphData.packetData,0,(MAX_YIELD_SIZE + 1) * sizeof(int));
    memset(graphData.signalData,0,(MAX_YIELD_SIZE + 1) * sizeof(int));
    
    aSSID = Nil;
    _firstPacket = YES;
    
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(updateSettings:) name:NSUserDefaultsDidChangeNotification object:nil];
    [self updateSettings:nil];
    [_dataLock unlock];
    return self;
}

- (id)initWithCoder:(NSCoder *)coder {
    waypoint wp;
    int bssid[6];
    
    if ( [coder allowsKeyedCoding] ) {
        if ([coder decodeObjectForKey:@"aFirstDate"] == Nil) {
            NSLog(@"Invalid net, dropping!");
            return Nil;
        }
        
        self = [self init];
        if (!self) return nil;
    
        _dataLock = [[NSRecursiveLock alloc] init];
        [_dataLock lock];
        aChannel=[coder decodeIntForKey:@"aChannel"];
        aNetID=[coder decodeIntForKey:@"aNetID"];
        _packets=[coder decodeIntForKey:@"aPackets"];
        aMaxSignal=[coder decodeIntForKey:@"aMaxSignal"];
        aCurSignal=[coder decodeIntForKey:@"aCurSignal"];
        aType=[coder decodeIntForKey:@"aType"];
        _isWep=[coder decodeIntForKey:@"aIsWep"];
        _weakPackets=[coder decodeIntForKey:@"aWeakPackets"];
        _dataPackets=[coder decodeIntForKey:@"aDataPackets"];
        
        for(int x=0; x<14; x++)
            _packetsPerChannel[x]=[coder decodeIntForKey:[NSString stringWithFormat:@"_packetsPerChannel%i",x]];
        
        aBytes=[coder decodeDoubleForKey:@"aBytes"];
        wp._lat =[coder decodeDoubleForKey:@"a_Lat"];
        wp._long=[coder decodeDoubleForKey:@"a_Long"];
	wp._elevation=[coder decodeDoubleForKey:@"a_Elev"];
        
        aLat = [[coder decodeObjectForKey:@"aLat"] retain];
        aLong = [[coder decodeObjectForKey:@"aLong"] retain];
	aElev = [[coder decodeObjectForKey:@"aElev"] retain];
        
        aID=[[coder decodeObjectForKey:@"aID"] retain];
        if (aID!=Nil && sscanf([aID cString], "%2X%2X%2X%2X%2X%2X", &bssid[0], &bssid[1], &bssid[2], &bssid[3], &bssid[4], &bssid[5])!=6) {
            NSLog(@"Error could not decode ID %@!", aID);
        }
        
        for (int x=0; x<6; x++)
            aRawID[x] = bssid[x];
        
        aSSID=[[coder decodeObjectForKey:@"aSSID"] retain];
        aBSSID=[[coder decodeObjectForKey:@"aBSSID"] retain];
        if (![aBSSID isEqualToString:@"<no bssid>"]) {
            if (aBSSID!=Nil && sscanf([aBSSID cString], "%2X:%2X:%2X:%2X:%2X:%2X", &bssid[0], &bssid[1], &bssid[2], &bssid[3], &bssid[4], &bssid[5])!=6) 
                NSLog(@"Error could not decode BSSID %@!", aBSSID);
            for (int x=0; x<6; x++)
                aRawBSSID[x] = bssid[x];
        } else {
             for (int x=0; x<6; x++)
                aRawBSSID[x] = bssid[0];
        }
        aDate=[[coder decodeObjectForKey:@"aDate"] retain];
        aFirstDate=[[coder decodeObjectForKey:@"aFirstDate"] retain];
        aWeak=[[coder decodeObjectForKey:@"aWeak"] retain];
        aPacketsLog=[[coder decodeObjectForKey:@"aPacketsLog"] retain];
        aARPLog=[[coder decodeObjectForKey:@"aARPLog"] retain];
        aACKLog=[[coder decodeObjectForKey:@"aACKLog"] retain];
        aPassword=[[coder decodeObjectForKey:@"aPassword"] retain];
        aComment=[[coder decodeObjectForKey:@"aComment"] retain];
        _coordinates=[[coder decodeObjectForKey:@"_coordinates"] retain];
        
        aClients=[[coder decodeObjectForKey:@"aClients"] retain];
        aClientKeys=[[coder decodeObjectForKey:@"aClientKeys"] retain];
        
        if (!aWeak) aWeak=[[NSMutableDictionary dictionaryWithCapacity:13] retain];
        if (!aPacketsLog) aPacketsLog=[[NSMutableArray arrayWithCapacity:20] retain];
        if (!aARPLog) aARPLog=[[NSMutableArray arrayWithCapacity:20] retain];
        if (!aACKLog) aACKLog=[[NSMutableArray arrayWithCapacity:20] retain];
        if (!aClients) aClients=[[NSMutableDictionary dictionary] retain];
        if (!aClientKeys) aClientKeys=[[NSMutableArray array] retain];
        if (!aComment) aComment=[[NSString stringWithString:@""] retain];
        if (!aLat) aLat = [[NSString stringWithString:@""] retain];
        if (!aLong) aLong = [[NSString stringWithString:@""] retain];
        if (!aElev) aElev = [[NSString stringWithString:@""] retain];
        if (!_coordinates) _coordinates = [[NSMutableDictionary dictionary] retain];
        
        _gotData = NO;
        
        aNetView = [[NetView alloc] initWithFrame:NSMakeRect(0, 0, 200, 30)];
        [aNetView setWep:_isWep];
        [aNetView setName:aSSID];
        [aNetView setCoord:wp];
        
        _firstPacket = NO;
        
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(updateSettings:) name:NSUserDefaultsDidChangeNotification object:nil];
        [self updateSettings:nil];
        [_dataLock unlock];
    } else {
        NSLog(@"Cannot decode this way");
        return nil;
    }
    return self;
}

- (id)initWithNetstumbler:(const char*)buf andDate:(NSString*)date {
    waypoint wp;
    char ns_dir, ew_dir;
    float ns_coord, ew_coord;
    char ssid[255], temp_bss[8];
    UInt hour, min, sec;
    UInt bssid[6], channelbits, flags;
    int interval = 0;
    
    self = [super init];
    
    if (!self) return nil;
    
    if(sscanf(buf, "%c %f %c %f (%*c%254[^)]) %7s "
    "( %2x:%2x:%2x:%2x:%2x:%2x ) %d:%d:%d (GMT) [ %d %*d %*d ] "
    "# %x %x %d",
    &ns_dir, &ns_coord, &ew_dir, &ew_coord, ssid, temp_bss,
    &bssid[0], &bssid[1], &bssid[2], &bssid[3], &bssid[4], &bssid[5],
    &hour, &min, &sec,
    &aMaxSignal,
    &flags, &channelbits, &interval) < 9) {
        NSLog(@"line in backup file is corrupt or not compatible");
        [self release];
        return Nil;
    }

    if(ssid[strlen(ssid) - 1] == ' ') ssid[strlen(ssid) - 1] = '\0';

    _dataLock = [[NSRecursiveLock alloc] init];
    [_dataLock lock];
    
    if (strcmp(temp_bss, "IBSS") == 0) aType = 1;
    else if (strcmp(temp_bss, "ad-hoc") == 0) aType = 1;
    else if (strcmp(temp_bss, "BSS") == 0) aType = 2;
    else if (strcmp(temp_bss, "TUNNEL") == 0) aType = 3;
    else if (strcmp(temp_bss, "PROBE") == 0) aType = 4;
    else aType = 0;

    _isWep = (flags & 0x0010) ? 2 : 1;

    aDate = [[NSDate dateWithString:[NSString stringWithFormat:@"%@ %.2d:%.2d:%.2d +0000", date, hour, min, sec]] retain];
    aFirstDate = [aDate retain];
    
    aLat  = [[NSString stringWithFormat:@"%f%c", ns_coord, ns_dir] retain];
    aLong = [[NSString stringWithFormat:@"%f%c", ew_coord, ew_dir] retain];
    aSSID = [[NSString stringWithCString: ssid] retain];

    aID = [[NSString stringWithFormat:@"%2X%2X%2X%2X%2X%2X", bssid[0], bssid[1], bssid[2], bssid[3], bssid[4], bssid[5]] retain];
    aBSSID = [[NSString stringWithFormat:@"%.2X:%.2X:%.2X:%.2X:%.2X:%.2X", bssid[0], bssid[1], bssid[2], bssid[3], bssid[4], bssid[5]] retain];
    for (int x=0; x<6; x++)
        aRawID[x] = bssid[x];
    
    wp._lat  = ns_coord * (ns_dir == 'N' ? 1.0 : -1.0);
    wp._long = ew_coord * (ew_dir == 'E' ? 1.0 : -1.0);
    wp._elevation = 0;

    aNetView = [[NetView alloc] initWithFrame:NSMakeRect(0, 0, 200, 30)];
    [aNetView setWep:_isWep];
    [aNetView setName:aSSID];
    [aNetView setCoord:wp];
    
    aWeak = [[NSMutableDictionary dictionaryWithCapacity:13] retain];
    aPacketsLog = [[NSMutableArray arrayWithCapacity:20] retain];
    aARPLog  = [[NSMutableArray arrayWithCapacity:20] retain];
    aACKLog  = [[NSMutableArray arrayWithCapacity:20] retain];
    aClients = [[NSMutableDictionary dictionary] retain];
    aClientKeys = [[NSMutableArray array] retain];
    aComment = [[NSString stringWithString:@""] retain];
    aElev = [[NSString stringWithString:@""] retain];
    _coordinates = [[NSMutableDictionary dictionary] retain];
    aNetID = 0;

    _gotData = NO;
    recentTraffic = 0;
    curTraffic = 0;
    curPackets = 0;
    aCurSignal = 0;
    curTrafficData = 0;
    curPacketData = 0;
    memset(graphData.trafficData,0,(MAX_YIELD_SIZE + 1) * sizeof(int));
    memset(graphData.packetData,0,(MAX_YIELD_SIZE + 1) * sizeof(int));
    memset(graphData.signalData,0,(MAX_YIELD_SIZE + 1) * sizeof(int));
        
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(updateSettings:) name:NSUserDefaultsDidChangeNotification object:nil];
    [self updateSettings:nil];
    [_dataLock unlock];
    return self;
}

- (void)encodeWithCoder:(NSCoder *)coder {
    waypoint wp;
    if ([coder allowsKeyedCoding]) {
        NS_DURING
            [_dataLock lock];
            [coder encodeInt:aMaxSignal forKey:@"aMaxSignal"];
            [coder encodeInt:aCurSignal forKey:@"aCurSignal"];
            [coder encodeInt:aType forKey:@"aType"];
            [coder encodeInt:_isWep forKey:@"aIsWep"];
            [coder encodeInt:_packets forKey:@"aPackets"];
            [coder encodeInt:_weakPackets forKey:@"aWeakPackets"];
            [coder encodeInt:_dataPackets forKey:@"aDataPackets"];
            [coder encodeInt:aChannel forKey:@"aChannel"];
            [coder encodeInt:aNetID forKey:@"aNetID"];
            
            for(int x=0;x<14;x++)
                [coder encodeInt:_packetsPerChannel[x] forKey:[NSString stringWithFormat:@"_packetsPerChannel%i",x]];
                
            [coder encodeDouble:aBytes forKey:@"aBytes"];
            
            wp = [aNetView coord];
            [coder encodeFloat:wp._lat forKey:@"a_Lat"];
            [coder encodeFloat:wp._long forKey:@"a_Long"];
	    [coder encodeFloat:wp._elevation forKey:@"a_Elev"];
            
            [coder encodeObject:aLat forKey:@"aLat"];
            [coder encodeObject:aLong forKey:@"aLong"];
	    [coder encodeObject:aElev forKey:@"aElev"];
            
            [coder encodeObject:aID forKey:@"aID"];
            [coder encodeObject:aSSID forKey:@"aSSID"];
            [coder encodeObject:aBSSID forKey:@"aBSSID"];
            [coder encodeObject:aDate forKey:@"aDate"];
            [coder encodeObject:aFirstDate forKey:@"aFirstDate"];
            [coder encodeObject:aWeak forKey:@"aWeak"];
            [coder encodeObject:aPacketsLog forKey:@"aPacketsLog"];
            [coder encodeObject:aARPLog forKey:@"aARPLog"];
            [coder encodeObject:aACKLog forKey:@"aACKLog"];
            [coder encodeObject:aPassword forKey:@"aPassword"];
            [coder encodeObject:aComment forKey:@"aComment"];
            [coder encodeObject:_coordinates forKey:@"_coordinates"];
            
            [coder encodeObject:aClients forKey:@"aClients"];
            [coder encodeObject:aClientKeys forKey:@"aClientKeys"];
            [_dataLock unlock];
        NS_HANDLER
            NSLog(@"Warning an exception was raised during save of aClients, please send the resulting kismac file to mick@binaervarianz.de");
        NS_ENDHANDLER
    } else {
        NSLog(@"Cannot encode this way");
    }
    return;
}

- (void)updateSettings:(NSNotification*)note {
    NSUserDefaults *sets = [NSUserDefaults standardUserDefaults];
    
    _avgTime = [[sets objectForKey:@"WaveNetAvgTime"]  intValue];
}

#pragma mark -

- (void)updateSSID:(NSString*)newSSID withSound:(bool)sound {
    int lVoice;
    NSString *lSentence;
    NSString *oc;
    const char *pc;
    unsigned int i;
    bool isHidden = YES;
    bool updatedSSID;
    
    if (newSSID==Nil || [newSSID isEqualToString:aSSID]) return;

    pc = [newSSID cString];
    for (i = 0; i < [newSSID length]; i++) {
        if (pc[i]) {
            isHidden = NO;
            break;
        }
    }
    if ([newSSID length]==1 && pc[i]==32) isHidden = YES;
    
    if (!aSSID) updatedSSID = NO;
    else updatedSSID = YES;
    
    if (isHidden) {
        if (aSSID!=Nil) return; //we might have the real bssid already
        [WaveHelper secureReplace:&aSSID withObject:@""];
    } else {
        [WaveHelper secureReplace:&aSSID withObject:newSSID];
    }
    
    [aNetView setName:aSSID];
    if (!_firstPacket) [[NSNotificationCenter defaultCenter] postNotificationName:KisMACViewItemChanged object:self];

    if (updatedSSID) return;
    
    lVoice=[[NSUserDefaults standardUserDefaults] integerForKey:@"Voice"];
    if (lVoice) {
        switch(_isWep) {
            case 1: oc = NSLocalizedString(@"open", "for speech");
                    break;
            case 2:
            case 3:
            case 4:
                    oc = NSLocalizedString(@"closed", "for speech");
                    break;
            default: oc=@"";
        }
        lSentence=[NSString stringWithFormat: NSLocalizedString(@"found %@ network. SSID is %@", "this is for speech output"),
            oc, isHidden ? NSLocalizedString(@"hidden", "for speech"): [aSSID uppercaseString]];
        NS_DURING
            [WaveHelper speakSentence:[lSentence cString] withVoice:lVoice];
        NS_HANDLER
        NS_ENDHANDLER
    }
}

- (void)generalEncounterStuff:(bool)onlineCapture {
    waypoint cp;
    GPSController *gpsc;
    ValuePair *pV;
    NSNumber *v;
    NSString *s;
    
    //after the first packet we should play some sound 
    if (aDate == Nil) {
        if (aSSID==Nil) [aNetView setName:aBSSID]; //draw BSSID into the map
        
        //lucent plays an extra role
        if ([aBSSID isEqualToString:@"00:00:00:00:00:00"]&&(aSSID==Nil))
            [self updateSSID:[NSString stringWithString:NSLocalizedString(@"<lucent tunnel>", "ssid for lucent tunnels")] withSound:onlineCapture];
        
        if (onlineCapture) { //sound?
            if (_isWep>=2) [[NSSound soundNamed:[[NSUserDefaults standardUserDefaults] objectForKey:@"WEPSound"]] play];
            else [[NSSound soundNamed:[[NSUserDefaults standardUserDefaults] objectForKey:@"noWEPSound"]] play];
        }
    }
    
    [aDate release];
    aDate = [[NSDate date] retain];
    if (!aFirstDate)
        aFirstDate = [[NSDate date] retain];


    if (onlineCapture) {
        gpsc = [WaveHelper gpsController];
        cp = [gpsc currentPoint];    
        if (cp._lat!=0 && cp._long!=0) {
            pV = [[ValuePair alloc] init];
            [pV setPairFromWaypoint:cp];
            v = [_coordinates objectForKey:pV];
            if ((v==Nil) || ([v intValue]<aCurSignal))
                [_coordinates setObject:[NSNumber numberWithInt:aCurSignal] forKey:pV];
            [pV release];
        }
    }
    
    if(aCurSignal>aMaxSignal) {
        aMaxSignal=aCurSignal;
        if (onlineCapture) {
            gpsc = [WaveHelper gpsController];
            s = [gpsc NSCoord];
            if (s) [WaveHelper secureReplace:&aLat withObject:s];
            s = [gpsc EWCoord];
            if (s) [WaveHelper secureReplace:&aLong withObject:s];
            s = [gpsc ElevCoord];
            if (s) [WaveHelper secureReplace:&aElev withObject:s];
            if (cp._lat!=0 && cp._long!=0) [aNetView setCoord:cp];
        }
    }
    
    _gotData = onlineCapture;
}

- (void) mergeWithNet:(WaveNet*)net {
    int temp;
    int* p;
    id key;
    NSDictionary *dict;
    NSMutableDictionary *mdict;
    NSEnumerator *e;
    
    temp = [net maxSignal];
    if (aMaxSignal < temp) {
        aMaxSignal = temp;
        [WaveHelper secureReplace:&aLat  withObject:[net latitude]];
        [WaveHelper secureReplace:&aLong withObject:[net longitude]];
	[WaveHelper secureReplace:&aElev withObject:[net elevation]];
    }
    
    if ([aDate compare:[net lastSeenDate]] == NSOrderedDescending) {
        aCurSignal = [net curSignal];
        
        temp = [net type];
        if (temp) aType = temp;
        
        temp = [net isWep];
        if (temp) _isWep = temp;
        
        temp = [net channel];
        if (temp) aChannel = temp;
        
        if (![[net SSID] isEqualToString: NSLocalizedString(@"<hidden ssid>", "hidden ssid")]) [self updateSSID:[net SSID] withSound:NO];
        [WaveHelper secureReplace:&aDate withObject:[net lastSeenDate]];
        if (![[net comment] isEqualToString:@""]) [WaveHelper secureReplace:&aComment withObject:[net comment]];
    }
    
    if ([aFirstDate compare:[net firstSeenDate]] == NSOrderedAscending)  [WaveHelper secureReplace:&aFirstDate withObject:[net firstSeenDate]];
        
    _packets +=     [net packets];
    _weakPackets += [net weakPackets];
    _dataPackets += [net dataPackets];
    
    p = [net packetsPerChannel];
    for(int x=0;x<14;x++) {
        _packetsPerChannel[x] += p[x];
        if (_packetsPerChannel[x] == p[x]) //the net we merge with has some channel, we did not know about
            [[NSNotificationCenter defaultCenter] postNotificationName:KisMACViewItemChanged object:self];
    }
    
    aBytes += [net dataCount];
    
    [_dataLock lock];
    
    [WaveHelper addDictionary:[net coordinates] toDictionary:_coordinates];
    
    //add all those weak packets to the log file
    dict = [net weakPacketsDict];
    e = [dict keyEnumerator];
    while(key = [e nextObject]) {
        mdict = [aWeak objectForKey:key];
        if (mdict)  [WaveHelper addDictionary:[dict objectForKey:key] toDictionary:mdict];
        else        [aWeak setObject:[dict objectForKey:key] forKey:key];
    }
    
    [aPacketsLog addObjectsFromArray:[net weakPacketsLog]];
    //sort them so that the smallest packet is in front of the array => faster cracking
    [aPacketsLog sortUsingFunction:lengthSort context:Nil];

    [_dataLock unlock];
}

- (void)parsePacket:(WavePacket*) w withSound:(bool)sound {
    NSString *clientid;
    WaveClient *lWCl;
    int lResolvType;
    unsigned int iv;
    NSNumber* num, *num2;
    NSMutableDictionary* x;
    int wep;
    unsigned int bodyLength;
    
    //int a, b;
    //UInt8 B;
    
    _packets++;
        
    if (!aID) {
        aID = [[w IDString] retain];
        [w ID:aRawID];
    }
    
    aCurSignal = [w signal];
    
    aChannel=[w channel];
    aBytes+=[w length];
    if ((_packetsPerChannel[aChannel]==0) && (!_firstPacket))
        [[NSNotificationCenter defaultCenter] postNotificationName:KisMACViewItemChanged object:self];
    _packetsPerChannel[aChannel]++;

    //statistical data for the traffic view
    if (sound) {
        graphData.trafficData[graphLength] += [w length];
        graphData.packetData[graphLength] += 1;
        curSignalData += aCurSignal;
        curPacketData++;
        curTrafficData += [w length];
    }
    
    if (aBSSID==Nil) {
        aBSSID=[[NSString stringWithString:[w BSSIDString]] retain];
        [w BSSID:aRawBSSID];
    }
    
    wep = [w wep];
    if (wep) {
        if (_isWep<wep || ([w type] == IEEE80211_TYPE_MGT && wep != 0 && _isWep!= wep)) {
            _isWep=wep;	//check if wep is enabled
            [aNetView setWep:_isWep];
        }
    }
    if ([w netType]) aType=[w netType];	//gets the type of network
    
    [_dataLock lock];
    
    //do some special parsing depending on the packet type
    switch ([w type]) {
        case IEEE80211_TYPE_DATA: //Data frame                        
            _dataPackets++;
            //is it WEP?
            if (_isWep>1) memcpy(aIV,[w framebody],3);	//sets the last IV thingy
            
            if (_isWep==2 || _isWep==3) {
                bodyLength = [w bodyLength];
                
                if (bodyLength>8) { //needs to have a fcs and an iv at least
                    
                    //this packet might be interesting for password checking, use the packet if we do not have enough, or f it is smaller than our smallest
                    if ([aPacketsLog count]<20 || [(NSString*)[aPacketsLog objectAtIndex:0] length] > bodyLength) {
                        [aPacketsLog addObject:[NSString stringWithCString:(const char*)([w framebody]) length:bodyLength]];
                        //sort them so that the smallest packet is in front of the array => faster cracking
                        [aPacketsLog sortUsingFunction:lengthSort context:Nil];
                    }

                    //log those packets for reinjection attack
                    if (([aARPLog count]<20)&&((bodyLength>=ARP_MIN_SIZE)&&(bodyLength<=ARP_MAX_SIZE))) {
                        if ([[w clientToID] isEqualToString:@"FF:FF:FF:FF:FF:FF"])
                            [aARPLog addObject:[NSString stringWithCString:(const char*)[w frame] length:[w length]]];
                    }
                    if (([aACKLog count]<20)&&((bodyLength>=TCPACK_MIN_SIZE)||(bodyLength<=TCPACK_MAX_SIZE))) {
                        [aACKLog addObject:[NSString stringWithCString:(const char*)[w frame] length:[w length]]];
                    }

                    /*
                    if (bodyLength>8) {
                    
                        a = (aIV[0] + aIV[1]) % N;
                        b = AMOD((aIV[0] + aIV[1]) - aIV[2], N);

                        for(B = 0; B < 13; B++) {
                            if((((0 <= a && a < B) ||
                                (a == B && b == (B + 1) * 2)) &&
                                (B % 2 ? a != (B + 1) / 2 : 1)) ||
                                (a == B + 1 && (B == 0 ? b == (B + 1) * 2 : 1)) ||
                                (aIV[0] == B + 3 && aIV[1] == N - 1) ||
                                (B != 0 && !(B % 2) ? (aIV[0] == 1 && aIV[1] == (B / 2) + 1) ||
                                (aIV[0] == (B / 2) + 2 && aIV[1] == (N - 1) - aIV[0]) : 0)) {

                                //if we dont have this type of packet make an array
                                num=[NSNumber numberWithInt:B];
                                x=[aWeak objectForKey:num];
                                if (x==Nil) {
                                    x=[[NSMutableDictionary dictionary] retain];
                                    [aWeak setObject:x forKey:num];
                                }
                                
                                //convert the iv to nextstep object
                                iv=aIV[0]*0x10000+aIV[1]*0x100+aIV[2];
                                num=[NSNumber numberWithUnsignedInt:iv];
                                num2=[x objectForKey:num];
                                if (num2==Nil) {
                                    //we dont have the iv => log it
                                    [x setObject:[NSNumber numberWithUnsignedChar:([w framebody][4] ^ 0xAA)] forKey:num];
                                    _weakPackets++;
                                }                            
                            }
                        }
                    }*/
                               
                    lResolvType = [w isResolved];	//check whether the packet is weak
                    if (lResolvType>-1) {
                        //if we dont have this type of packet make an array
                        num=[NSNumber numberWithInt:lResolvType];
                        x=[aWeak objectForKey:num];
                        if (x==Nil) {
                            x=[[NSMutableDictionary dictionary] retain];
                            [aWeak setObject:x forKey:num];
                        }
                        
                        //convert the iv to nextstep object
                        iv=aIV[0]*0x10000+aIV[1]*0x100+aIV[2];
                        num=[NSNumber numberWithUnsignedInt:iv];
                        num2=[x objectForKey:num];
                        if (num2==Nil) {
                            //we dont have the iv => log it
                            [x setObject:[NSNumber numberWithUnsignedChar:([w framebody][4] ^ 0xAA)] forKey:num];
                            _weakPackets++;
                        }
                    }
                }
            }
            break;
        case IEEE80211_TYPE_MGT:        //this is a management packet
            [self updateSSID:[w ssid] withSound:sound]; //might contain SSID infos
    }

    //update the clients to out client array
    //if they are not in add them
    clientid=[w clientToID];
    if (clientid!=Nil) {
        lWCl=[aClients objectForKey:clientid];
        if (lWCl==nil) {
            lWCl=[[WaveClient alloc] init];
            [aClients setObject:lWCl forKey:clientid];
            [aClientKeys addObject:clientid];  
            [lWCl release];        
        }
        [lWCl parseFrameAsIncoming:w];
    }
    clientid=[w clientFromID];
    if (clientid!=Nil) {
        lWCl=[aClients objectForKey:clientid];
        if (lWCl==nil) {
            lWCl=[[WaveClient alloc] init];
            [aClients setObject:lWCl forKey:clientid];
            [aClientKeys addObject:clientid];
            [lWCl release];              
        }
        [lWCl parseFrameAsOutgoing:w];
    }
    
    [self generalEncounterStuff:sound];
    
    if (_firstPacket) {
        [[NSNotificationCenter defaultCenter] postNotificationName:KisMACViewItemChanged object:self];
        _firstPacket = NO;
    }
    
    [_dataLock unlock];
}

- (void)parseAppleAPIData:(WirelessNetworkInfo*)info {
    int wep;
   
    if (!aID) {
        aID = [[NSString stringWithFormat:@"%.2X%.2X%.2X%.2X%.2X%.2X", info->macAddress[0], info->macAddress[1], info->macAddress[2],
                info->macAddress[3], info->macAddress[4], info->macAddress[5]] retain];
        memcpy(aRawID, info->macAddress, sizeof(info->macAddress));
    }
            
    aCurSignal = info->signal - info->noise;
    if (aCurSignal<0) aCurSignal = 0;
    
    aChannel = info->channel;
    if (_packetsPerChannel[aChannel]==0) {
        if (!_firstPacket) [[NSNotificationCenter defaultCenter] postNotificationName:KisMACViewItemChanged object:self];
        _packetsPerChannel[aChannel] = 1;
    }
    
    //statistical data for the traffic view
    //not much though
    curSignalData += aCurSignal;
    curPacketData++;
    
    if (aBSSID==Nil) {
        aBSSID = [[NSString stringWithFormat:@"%.2X:%.2X:%.2X:%.2X:%.2X:%.2X", info->macAddress[0], info->macAddress[1], info->macAddress[2],
                info->macAddress[3], info->macAddress[4], info->macAddress[5]] retain];
        memcpy(aRawBSSID, info->macAddress, sizeof(info->macAddress));
    }
    
    wep = (info->flags & IEEE80211_CAPINFO_PRIVACY_LE) ? 2 : 1;
    if (_isWep!=wep) {
        _isWep=wep;	//check if wep is enabled
        [aNetView setWep:_isWep];
    }
    
    if (info->flags & IEEE80211_CAPINFO_ESS_LE) {
        aType=2;
    } else if (info->flags & IEEE80211_CAPINFO_IBSS_LE) {
        aType=1;
    }

    [_dataLock lock];
    [self updateSSID:[NSString stringWithCString:(char*)info->name length:info->nameLen] withSound:YES];
    [self generalEncounterStuff:YES];
    
    if (_firstPacket) {
        [[NSNotificationCenter defaultCenter] postNotificationName:KisMACViewItemChanged object:self];
        _firstPacket = NO;
    }
    
    [_dataLock unlock];
}

#pragma mark -

- (bool)noteFinishedSweep:(int)num {
    // shuffle the values around in the aYield array
    bool ret;
    ValuePair *pV;
    waypoint cp;
    
    graphLength = num;

    if (curPacketData) {
        curSignalData/=curPacketData;
        ret = NO;
    } else if ([[NSDate date] timeIntervalSinceDate:aDate]>1 && _gotData) {
        cp = [[WaveHelper gpsController] currentPoint];
       
        if (cp._lat!=0 && cp._long!=0) {
            [_dataLock lock];
            pV = [[ValuePair alloc] init];
            [pV setPairFromWaypoint:cp];
            [_coordinates setObject:[NSNumber numberWithInt:0] forKey:pV];
            [pV release];
            [_dataLock unlock];
        }

        curSignalData=0;
        aCurSignal=0;
        ret = YES;	//the net needs an update
        _gotData = NO;
    } else {
        return NO;
    }
    
    // set the values we collected
    graphData.trafficData[graphLength] = curTrafficData;
    graphData.packetData[graphLength] = curPacketData;
    graphData.signalData[graphLength] = curSignalData;

    curTraffic = curTrafficData;
    curTrafficData = 0;
    curPackets = curPacketData;
    curPacketData = 0;
    curSignalData = 0;
    
    int x = num - 120;

    recentTraffic = 0;
    recentPackets = 0;
    recentSignal = 0;
    if(x < 0)
        x = 0;
    while(x < num) {
        recentTraffic += graphData.trafficData[x];
        recentPackets += graphData.packetData[x];
        recentSignal += graphData.signalData[x];
            x++;
    }
    
    if(graphLength >= MAX_YIELD_SIZE) {
        memcpy(graphData.trafficData,graphData.trafficData + 1, (MAX_YIELD_SIZE) * sizeof(int));
        graphData.trafficData[MAX_YIELD_SIZE] = 0;

        memcpy(graphData.packetData,graphData.packetData + 1, (MAX_YIELD_SIZE) * sizeof(int));
        graphData.packetData[MAX_YIELD_SIZE] = 0;

        memcpy(graphData.signalData,graphData.signalData + 1, (MAX_YIELD_SIZE) * sizeof(int));
        graphData.signalData[MAX_YIELD_SIZE] = 0;
    }
 
    return ret;
}

- (void)updatePassword {
    if ((aPassword==Nil)&&(aCracker!=Nil)) {
        aPassword=[[aCracker key] retain];
    }
}

- (int)crack40WithBruteForce:(ImportController*)i charset:(int) chars {
    int ret = 1;
    if (_isWep!=2) return 0;
    if (aCracker==Nil) aCracker=[[WaveCracker alloc] init];
    if (aPassword==Nil)
        ret = [aCracker performSelectedBruteForce40:self import:i charset:chars];
    
    return ret;
}

- (void)crackWithKeyByteLength:(unsigned int)a breath:(unsigned int)b import:(ImportController*)i {
    if (_isWep!=2) return;
    if (aCracker==Nil) aCracker=[[WaveCracker alloc] init];
    if (aPassword==Nil) {
        [aCracker crackWithKeyByteLength:a net:self breath:b import:i];
        aPassword=[[aCracker key] retain];
    }
}

- (void)crackWithWordlistUseCipher:(unsigned int)a import:(ImportController*)i {
    NSOpenPanel *aOP;
    
    if (_isWep!=2) goto error;
    
    if (aCracker==Nil) aCracker=[[WaveCracker alloc] init];
    if (aPassword==Nil) {
        aOP=[NSOpenPanel openPanel];
        [aOP setAllowsMultipleSelection:YES];
        [aOP setCanChooseFiles:YES];
        [aOP setCanChooseDirectories:NO];
        if ([aOP runModalForTypes:nil]==NSOKButton) {
            [aCracker crackWithWordlist:[aOP filenames] useCipher:a net:self import:i];
        } else goto error;
    } else goto error;
    
    return;
    
error:
    [i cancelAction:nil];
    [[NSNotificationCenter defaultCenter] postNotificationName:KisMACCrackDone object:self];
}

#pragma mark -

- (struct graphStruct)graphData {
    return graphData;
}
- (NSDictionary*)getClients {
    return aClients;
}
- (NSArray*)getClientKeys {
    return aClientKeys;
}
- (int)isWep { 
    return _isWep;
}
- (NSString *)ID {
    return aID;
}
- (NSString *)BSSID {
    if (aBSSID==Nil) return NSLocalizedString(@"<no bssid>", "for tunnels");
    return aBSSID;
}
- (NSString *)SSID {
    if (aSSID==Nil) return @"<no ssid>";
    if ([aSSID isEqualToString:@""]) return (aType == 4 ? 
        NSLocalizedString(@"<any ssid>", "the any ssid for probe nets") : 
        NSLocalizedString(@"<hidden ssid>", "hidden ssid")
        );
    return aSSID;
}
- (NSString *)date {
    return [NSString stringWithFormat:@"%@", aDate]; //return [aDate descriptionWithCalendarFormat:@"%H:%M %d-%m-%y" timeZone:nil locale:nil];
}
- (NSDate*)lastSeenDate {
    return aDate;
}
- (NSString *)firstDate {
    return [NSString stringWithFormat:@"%@", aFirstDate]; //[aFirstDate descriptionWithCalendarFormat:@"%H:%M %d-%m-%y" timeZone:nil locale:nil];
}
- (NSDate *)firstSeenDate {
    return aFirstDate;
}
- (NSString *)data {
    return [WaveHelper bytesToString: aBytes];
}
- (float)dataCount {
    return aBytes;
}
- (NSString *)getVendor {
    if (aVendor) return aVendor;
    aVendor=[[WaveHelper vendorForMAC:aBSSID] retain];
    return aVendor;
}
- (NSString*)comment {
    return aComment;
}
- (void)setComment:(NSString*)comment {
    [aComment release];
    aComment=[comment retain];
}
- (int)avgSignal {
    int sum = 0;
    int i, x, c;
    int max = (graphLength < _avgTime*4) ? graphLength : _avgTime*4;
    
    c=0;
    for (i=0; i<max; i++) {
        x = graphData.signalData[graphLength - i];
        if (x) {
            sum += x;
            c++;
        }
    }
    if (c==0) return 0;
    return sum / c;
}
- (int)curSignal {
    return aCurSignal;
}
- (int)curPackets {
    return curPackets;
}
- (int)curTraffic {
    return curTraffic;
}
- (int)recentTraffic {
    return recentTraffic;
}
- (int)recentPackets {
    return recentPackets;
}
- (int)recentSignal {
    return recentSignal;
}
- (int)maxSignal {
    return aMaxSignal;
}
- (int)channel {
    return aChannel;
}
- (int)type {
    return aType;
}
- (int)netID {
    return aNetID;
}
- (int)packets {
    return _packets;
}
- (int)weakPackets {
    return _weakPackets;
}
- (int)dataPackets {
    return _dataPackets;
}
- (int*)packetsPerChannel {
    return _packetsPerChannel;
}
- (NSDictionary*)weakPacketsDict {
    return aWeak;
}
- (NSArray*)weakPacketsLog {
    return aPacketsLog;
}
- (NSMutableArray*)arpPacketsLog {
    return aARPLog;
}
- (NSMutableArray*)ackPacketsLog {
    return aACKLog;
}
- (NSString*)key {
    if ((aPassword==Nil)&&(_isWep>1)) return NSLocalizedString(@"<unresolved>", "Unresolved password");
    return aPassword;
}
- (NSString*)lastIV {
    return [NSString stringWithFormat:@"%.2X:%.2X:%.2X", aIV[0], aIV[1], aIV[2]];
}
- (UInt8*)rawBSSID {
    return aRawBSSID;
}
- (UInt8*)rawID {
    return aRawID;
}
- (NSDictionary*)coordinates {
    return _coordinates;
}

#pragma mark -

- (NSString *)latitude {
    if (!aLat) return @"0.000000N";
    return aLat;
}
- (NSString *)longitude {
    if (!aLong) return @"0.000000E";
    return aLong;
}

- (NSString *)elevation {
    if (!aElev) return @"0";
    return aElev;
}


#pragma mark -

// for display color in TrafficView
- (NSColor*)graphColor {
    return _graphColor;
}
- (void)setGraphColor:(NSColor*)newColor {
    [_graphColor autorelease];
    _graphColor = [newColor retain];
}

// for easy sorting by TrafficView
- (NSComparisonResult)compareSignalTo:(id)aNet {
    if (aCurSignal == [aNet curSignal])
        return NSOrderedSame;
    if (aCurSignal > [aNet curSignal])
        return NSOrderedAscending;
    return NSOrderedDescending;
}

- (NSComparisonResult)comparePacketsTo:(id)aNet {
    if (curPackets == [aNet curPackets])
        return NSOrderedSame;
    if (curPackets > [aNet curPackets])
        return NSOrderedAscending;
    return NSOrderedDescending;
}

- (NSComparisonResult)compareTrafficTo:(id)aNet {
    if (curTraffic == [aNet curTraffic])
        return NSOrderedSame;
    if (curTraffic > [aNet curTraffic])
        return NSOrderedAscending;
    return NSOrderedDescending;
}
- (NSComparisonResult)compareRecentTrafficTo:(id)aNet {
    if (recentTraffic == [aNet recentTraffic])
        return NSOrderedSame;
    if (recentTraffic > [aNet recentTraffic])
        return NSOrderedAscending;
    return NSOrderedDescending;
}
- (NSComparisonResult)compareRecentPacketsTo:(id)aNet {
    if (recentPackets == [aNet recentPackets])
        return NSOrderedSame;
    if (recentPackets > [aNet recentPackets])
        return NSOrderedAscending;
    return NSOrderedDescending;
}
- (NSComparisonResult)compareRecentSignalTo:(id)aNet {
    if (recentSignal == [aNet recentSignal])
        return NSOrderedSame;
    if (recentSignal > [aNet recentSignal])
        return NSOrderedAscending;
    return NSOrderedDescending;
}

#pragma mark -

inline int compValues(int v1, int v2) {
    if (v1 < v2) return NSOrderedAscending;
    else if (v1 > v2) return NSOrderedDescending;
    else return NSOrderedSame;
}

inline int compFloatValues(float v1, float v2) {
    if (v1 < v2) return NSOrderedAscending;
    else if (v1 > v2) return NSOrderedDescending;
    else return NSOrderedSame;
}

int idSort(WaveClient* w1, WaveClient* w2, int ascend) {
    int v1 = [[w1 ID] intValue];
    int v2 = [[w2 ID] intValue];
    return ascend * compValues(v1,v2);
}

int clientSort(WaveClient* w1, WaveClient* w2, int ascend) {
    return ascend * [[w1 ID] compare:[w2 ID]];
}

int vendorSort(WaveClient* w1, WaveClient* w2, int ascend) {
    return ascend * [[w1 vendor] compare:[w2 vendor]];
}

int signalSort(WaveClient* w1, WaveClient* w2, int ascend) {
    return ascend * compValues( [w1 curSignal], [w2 curSignal]);
}

int recievedSort(WaveClient* w1, WaveClient* w2, int ascend) {
    return ascend * compFloatValues([w1 recievedBytes], [w2 recievedBytes]);
}

int sentSort(WaveClient* w1, WaveClient* w2, int ascend) {
    return ascend * compFloatValues([w1 sentBytes], [w2 sentBytes]);
}
int dateSort(WaveClient* w1, WaveClient* w2, int ascend) {
    return ascend * [[w1 rawDate] compare:[w2 rawDate]];
}


typedef int (*SORTFUNC)(id, id, void *);

- (void) sortByColumn:(NSString*)ident order:(bool)ascend {
    bool sorted = YES;
    SORTFUNC sf;
    int ret;
    unsigned int w, x, y, _sortedCount, a;
    
    if      ([ident isEqualToString:@"id"])     sf = (SORTFUNC)idSort;
    else if ([ident isEqualToString:@"client"]) sf = (SORTFUNC)clientSort;
    else if ([ident isEqualToString:@"vendor"]) sf = (SORTFUNC)vendorSort;
    else if ([ident isEqualToString:@"signal"]) sf = (SORTFUNC)signalSort;
    else if ([ident isEqualToString:@"recieved"]) sf=(SORTFUNC)recievedSort;
    else if ([ident isEqualToString:@"sent"])   sf = (SORTFUNC)sentSort;
    else if ([ident isEqualToString:@"lastseen"]) sf=(SORTFUNC)dateSort;
    else {
        NSLog(@"Unknown sorting column. This is a bug and should never happen.");
        return;
    }

    a = (ascend ? 1 : -1);
    
    [_dataLock lock];
    
    _sortedCount = [aClientKeys count];
    
    for (y = 1; y <= _sortedCount; y++) {
        for (x = y - 1; x < (_sortedCount - y); x++) {
            w = x + 1;
            ret = (*sf)([aClients objectForKey:[aClientKeys objectAtIndex:x]], [aClients objectForKey:[aClientKeys objectAtIndex:w]], (void*)a);
            if (ret == NSOrderedDescending) {
                sorted = NO;
                
                //switch places
                [aClientKeys exchangeObjectAtIndex:x withObjectAtIndex:w];
                [[aClients objectForKey:[aClientKeys objectAtIndex:x]] wasChanged];
                [[aClients objectForKey:[aClientKeys objectAtIndex:w]] wasChanged];
            }
        }
        
        if (sorted) break;
        sorted = YES;
        
        for (x = (_sortedCount - y); x >= y; x--) {
            w = x - 1;
            ret = (*sf)([aClients objectForKey:[aClientKeys objectAtIndex:w]], [aClients objectForKey:[aClientKeys objectAtIndex:x]], (void*)a);
            if (ret == NSOrderedDescending) {
                sorted = NO;
                
                //switch places
                [aClientKeys exchangeObjectAtIndex:x withObjectAtIndex:w];
                [[aClients objectForKey:[aClientKeys objectAtIndex:x]] wasChanged];
                [[aClients objectForKey:[aClientKeys objectAtIndex:w]] wasChanged];
            }
        }
        
        if (sorted) break;
        sorted = YES;
    }
        
    [_dataLock unlock];
}

#pragma mark -

- (void)dealloc {
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [aNetView removeFromSuperview];

    [_dataLock lock];
    [aID release];
    [aSSID release];
    [aBSSID release];
    [aDate release];
    [aFirstDate release];
    [aVendor release];
    [aCracker release];
    [aPassword release];
    [aPacketsLog release];
    [aARPLog release];
    [aACKLog release];
    [aWeak release];
    [aClients release];
    [aClientKeys release];
    [aComment release];
    [aLat release];
    [aLong release];
    [aElev release];
    [aNetView release];
    [_coordinates release];
    [_dataLock unlock];
    [_dataLock release];
    
    [super dealloc];
}

@end
