#  RFIDIOt.py - RFID IO tools for python
# 
#  Adam Laurie <adam@algroup.co.uk>
#  http://rfidiot.org/
# 
#  This code is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
# 
#  This code is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
# 

import serial
import string
import os

class rfidiot:
	"RFIDIOt - RFID I/O tools - http://rfidiot.org"
	#
	# open reader port
	#
	def __init__(self,reader,baud):
		self.ser = serial.Serial(reader, baud)
		self.ser.readline()
		self.ser.flushInput()
		self.ser.flushOutput()
	#
	# variables
	#
	# VERSION: RFIDIOt.py version number
	# errorcode: 1 letter errorcode returned by the reader
	#
	# MIFAREdata: ASCII HEX representation of data block after successful read
	# MIFAREbinary: data block converted back to binary
	# MIFAREBLOCKLEN: constant ASCII HEX block length
	# MIFAREVALUELEN: constant ASCII HEX value length
	# MIFAREserialnumber: Unique ID (UID) of card
	# MIFAREkeyA: KEYA from key block (will always be 000000000000)
	# MIFAREkeyB: KEYB from key block
	# MIFAREaccessconditions: access conditions field from Key Block
	# MIFAREC1: Access conditions bitfield C1
	# MIFAREC2: Access conditions bitfield C2
	# MIFAREC3: Access conditions bitfield C3
	# MIFAREblock0AC: Block 0 Access Conditions
	# MIFAREblock1AC: Block 1 Access Conditions
	# MIFAREblock2AC: Block 2 Access Conditions
	# MIFAREblock3AC: Block 3 Access Conditions
	# MIFAREACKB: Human readable Key Block Access Conditions
	# MIFAREACDB: Human readable Data Block Access ConditionsA
	#
	# MRPmrzu: Machine Readable Passport - Machine Readable Zone - Upper
	# MRPmrzl Machine Readable Passport - Machine Readable Zone - Lower
	VERSION= '0.1e'
	errorcode= ''
	binary= ''
	data= ''
	MIFAREdata=''
	MIFAREbinary=''
	MIFAREBLOCKLEN=32
	MIFAREVALUELEN=8
	MIFAREserialnumber= ''
	MIFAREcheckbyte= ''
	MIFAREmanufacturerdata= ''
	MIFAREkeyA= ''
	MIFAREkeyB= ''
	MIFAREaccessconditions= ''
	MIFAREaccessconditionsuserbyte= ' '
	MIFAREC1= 0
	MIFAREC2= 0
	MIFAREC3= 0
	MIFAREblock0AC= ''
	MIFAREblock1AC= ''
	MIFAREblock2AC= ''
	MIFAREblock3AC= ''
	MIFAREACKB= {'000':'Write KeyA: KEYA, Read Access bits: KEYA, Write Access bits: NONE, Read KeyB: KEYA, Write KeyB: KEYA (KEYB readable)',\
		     '010':'Write KeyA: NONE, Read Access bits: KEYA, Write Access bits: NONE, Read KeyB: KEYA, Write KeyB: NONE (KEYB readable)',\
		     '100':'Write KeyA: KEYB, Read Access bits: KEYA/B, Write Access bits: NONE, Read KeyB: NONE, Write KeyB: KEYB',\
		     '110':'Write KeyA: NONE, Read Access bits: KEYA/B, Write Access bits: NONE, Read KeyB: NONE, Write KeyB: NONE',\
		     '001':'Write KeyA: KEYA, Read Access bits: KEYA, Write Access bits: KEYA, Read KeyB: KEYA, Write KeyB: KEYA (KEYB readable, transport configuration)',\
		     '011':'Write KeyA: KEYB, Read Access bits: KEYA/B, Write Access bits: KEYB, Read KeyB: NONE, Write KeyB: KEYB',\
		     '101':'Write KeyA: NONE, Read Access bits: KEYA/B, Write Access bits: KEYB, Read KeyB: NONE, Write KeyB: NONE',\
		     '111':'Write KeyA: NONE, Read Access bits: KEYA/B, Write Access bits: NONE, Read KeyB: NONE, Write KeyB: NONE'}
	MIFAREACDB= {'000':'Read: KEYA/B, Write: KEYA/B, Increment: KEYA/B, Decrement/Tranfer/Restore: KEYA/B (transport configuration)',\
		     '010':'Read: KEYA/B, Write: NONE, Increment: NONE, Decrement/Tranfer/Restore: NONE',\
		     '100':'Read: KEYA/B, Write: KEYB, Increment: NONE, Decrement/Tranfer/Restore: NONE',\
		     '110':'Read: KEYA/B, Write: KEYB, Increment: KEYB, Decrement/Tranfer/Restore: KEYA/B',\
		     '001':'Read: KEYA/B, Write: NONE, Increment: NONE, Decrement/Tranfer/Restore: KEYA/B',\
		     '011':'Read: KEYB, Write: KEYB, Increment: NONE, Decrement/Tranfer/Restore: NONE',\
		     '101':'Read: KEYB, Write: NONE, Increment: NONE, Decrement/Tranfer/Restore: NONE',\
		     '111':'Read: NONE, Write: NONE, Increment: NONE, Decrement/Tranfer/Restore: NONE'}
	LFXTags= {'U':'EM 4x02           ',\
		  'Z':'EM 4x05 (ISO FXDB)',\
		  'T':'EM 4x50           ',\
		  'h':'Hitag 1 / Hitag S ',\
		  'H':'Hitag 2           ',\
		  'Q':'Q5',\
		  'R':'TI-RFID Systems'}
	EM4x05= 'Z'
	ISOTags= {'a':'ISO 14443 Type A  ',\
		  'b':'ISO 14443 Type B  ',\
		  's':'SR176             '}
	ISOTagsA= {'t':'All Supported Tags'}
	ISOAPDU=  {'ERASE BINARY':'0E',\
		   'VERIFY':'20',\
                   'MANAGE_CHANNEL':'70',\
                   'EXTERNAL_AUTHENTICATE':'82',\
                   'GET_CHALLENGE':'84',\
                   'INTERNAL_AUTHENTICATE':'88',\
                   'SELECT_FILE':'A4',\
                   'READ_BINARY':'B0',\
                   'READ_RECORD(S)':'B2',\
                   'GET_RESPONSE':'C0',\
                   'ENVELOPE':'C2',\
                   'GET_DATA':'CA',\
                   'WRITE_BINARY':'D0',\
                   'WRITE_RECORD':'D2',\
                   'UPDATE_BINARY':'D6',\
                   'PUT_DATA':'DA',\
                   'UPDATE_DATA':'DC',\
                   'APPEND_RECORD':'E2'}
	ISO7816ErrorCodes= {'9000':'No further qualification',\
			    '61':'SW2 indicates the number of response bytes still available',\
			    '6200':'No information given',\
			    '6281':'Part of returned data may be corrupted',\
			    '6282':'End of file/record reached before reading Le bytes',\
			    '6283':'Selected file invalidated',\
			    '6284':'FCI not formatted according to ISO7816-4 section 5.1.5',\
			    '6300':'No information given',\
			    '6381':'File filled up by the last write',\
			    '63C':'Counter provided by X (valued from 0 to 15) (exact meaning depending on the command)',\
			    '64':'State of non-volatile memory unchanged (SW2=00, other values are RFU)'}
	#
	# local/informational functions
	#
	def info(self,caller):
		self.reset()
		print caller + ' (using RFIDIOt v' + self.VERSION + ')'
		print 'reader: ' + self.version() + ' (serial no: ' + self.id() + ')'

	#
	# reader functions
	#
        def reset(self):
		if self.ser.inWaiting() > 0:
			self.ser.flushInput()
		self.ser.write('x')
		x= self.ser.readline()
		# check for 'constant read mode' and stop if true
		self.ser.flushInput()
		self.ser.write('!')
		if self.ser.readline()[0] == '!':
			self.ser.write(' ')
			self.ser.readline()
			self.ser.flushInput()
		return x[:-2]
	def version(self):
		self.ser.write('v')
		return self.ser.readline()[:-2]
	def id(self):
		return self.readEEPROM(0)[:2] + self.readEEPROM(1)[:2] + self.readEEPROM(2)[:2] + self.readEEPROM(3)[:2]
	def station(self):
		return self.readEEPROM(0x0a)[:2]
	def PCON(self):
		return self.readEEPROM(0x0b)[:2]
	def PCON2(self):
		return self.readEEPROM(0x13)[:2]
	def PCON3(self):
		return self.readEEPROM(0x1b)[:2]
	def BAUD(self):
		return self.readEEPROM(0x0c)[:2]
	def CGT(self):
		return self.readEEPROM(0x0d)[:2]
	def opmode(self):
		return self.readEEPROM(0x0e)[:2]
	def SST(self):
		return self.readEEPROM(0x0f)[:2]
	def ROT(self):
		return self.readEEPROM(0x14)[:2]
	def RRT(self):
		return self.readEEPROM(0x15)[:2]
	def AFI(self):
		return self.readEEPROM(0x16)[:2]
	def STOa(self):
		return self.readEEPROM(0x17)[:2]
	def STOb(self):
		return self.readEEPROM(0x18)[:2]
	def STOs(self):
		return self.readEEPROM(0x19)[:2]
	def readEEPROM(self,byte):
		self.ser.write('rp%02x' % byte)
		return self.ser.readline()[:2]
	def writeEEPROM(self,byte,value):
		self.ser.write('wp%02x%02x' % (byte,value))
		self.errorcode= self.ser.readline()[:-2]
		if eval(self.errorcode) == value:
			return True
		return False
	def settagtype(self,type):
		self.ser.write('o' + type)
		self.errorcode= self.ser.readline()[:-2]
		if self.errorcode == 'O' + string.upper(type):
			return True
		return False
	#
	# card functions
	#
	def select(self):
		self.ser.write('s')
		return self.ser.readline()[:-2]
	def hsselect(self,speed):
		"high speed select - 106 (speed= 01), 212 (speed= 02), 424 (speed= 04) or 848 (speed= 08) kBaud"
		self.ser.write('h'+speed)
		return self.ser.readline()[:-2]
	def iso_7816_fail(self,code):
		"print 7816 failure code and exit"
		print "Failed - reason code " + code + " (" + self.ISO7816ErrorCodes[code] + ")"
		print
		os._exit(False)
	def send_apdu(self,option,pcb,cid,nad,cla,ins,p1,p2,lc,data,le):
		"send iso-1786-4 apdu"
		if not option:
			option= '0f'
		if not pcb:
			pcb= '02'
		if not cla:
			cla= '00'
		if not p1:
			p1= '00'
		if not p2:
			p2= '00'
		dlength= 5
		command= pcb+cla+self.ISOAPDU[ins]+p1+p2+lc+data+le
		dlength += len(data) / 2
		dlength += len(lc) / 2
		dlength += len(le) / 2
#		print 'sending: ' + 't' + '%02x' % dlength + option + command
		self.ser.write('t' + '%02x' % dlength + option + command)
		# need check for 'le' length as well
		ret= self.ser.readline()[:-2] 
		self.errorcode= ret[len(ret) - 4:len(ret)]
		return ret[4:len(ret) - 4]
#		if not len(ret) / 2 == int(ret[0:2],16) + 1:
#			return False
#		return ret[4:int(le,16) * 2 + 4]
	def login(self,sector,keytype,key):
		"login to specified sector - returns True if successful, False if failed. If failure is due to an error, 'errorcode' will be set." 
		self.ser.write('l' + ('%02x' % sector) + keytype + key)
		if key == '':
			self.ser.write('\r')
		self.errorcode= self.ser.readline()[0]
		if self.errorcode == 'L':
			return True
		elif self.errorcode == 'X':
			self.errorcode= ''
		return False
	def readblock(self,block):
		self.ser.write('rb%02x' % block)
		self.data= self.ser.readline()[:-2]
		if len(self.data) == 1:
			self.errorcode= self.data
			self.data= ''
			return False
		count= 0
		while count * 2 < len(self.data):
			self.binary += chr(int(self.data[count * 2:(count * 2) + 2],16))
			count += 1
		return True	
	def readMIFAREblock(self,block):
		self.ser.write('rb%02x' % block)
		self.MIFAREdata= self.ser.readline()[:-2]
		if len(self.MIFAREdata) != self.MIFAREBLOCKLEN:
			self.errorcode= self.MIFAREdata
			self.MIFAREdata= ''
			return False
		count= 0
		while count * 2 < len(self.MIFAREdata):
			self.MIFAREbinary += chr(int(self.MIFAREdata[count * 2:(count * 2) + 2],16))
			count += 1
		return True
	def readvalueblock(self,block):
		self.ser.write('rv%02x' % block)
		self.MIFAREdata= self.ser.readline()[:-2]
		if len(self.MIFAREdata) != self.MIFAREVALUELEN:
			self.errorcode= self.MIFAREdata
			self.MIFAREdata= ''
			return False
		count= 0
		while count * 2 < len(self.MIFAREdata):
			self.MIFAREbinary += chr(int(self.MIFAREdata[count * 2:(count * 2) + 2],16))
			count += 1
		return True
	def writeblock(self,block,data):
		self.ser.write('wb%02x%s' % (block,data))
		x= self.ser.readline()[:-2]
		if x == string.upper(data):
			self.errorcode= ''
			return True
		self.errorcode= x
		return False
	def writevalueblock(self,block,data):
		self.ser.write('wv%02x%s' % (block,data))
                x= self.ser.readline()[:-2]
                if x == string.upper(data):
                        self.errorcode= ''
                        return True
                self.errorcode= x
                return False
	#
	# data manipulation
	#
	def MIFAREmfb(self,data):
		"Set variables from standard MIFARE manufacturer block (block 0 sector 0)"
		self.MIFAREserialnumber= data[0:8]
		self.MIFAREcheckbyte= data[8:10]
		self.MIFAREmanufacturerdata= data[10:32]
	def MIFAREkb(self,data):
		"Set variables from standard MIFARE key block (trailing sector)"
		self.MIFAREkeyA= data[0:12]
		self.MIFAREaccessconditions= data[12:18]
		self.MIFAREaccessconditionsuserbyte= data[18:20]
		self.MIFAREC1= int(data[14:16],16) >> 4
		self.MIFAREC2= int(data[16:18],16) & 0x0f
		self.MIFAREC3= (int(data[16:18],16) & 0xf0) >> 4
		self.MIFAREblock0AC= str(self.MIFAREC1 & 0x01) + str(self.MIFAREC2 & 0x01) + str(self.MIFAREC3 & 0x01)
		self.MIFAREblock1AC= str((self.MIFAREC1 & 0x02) >> 1) + str((self.MIFAREC2 & 0x02) >> 1) + str((self.MIFAREC3 & 0x02) >> 1)
		self.MIFAREblock2AC= str((self.MIFAREC1 & 0x04) >> 2) + str((self.MIFAREC2 & 0x04) >> 2) + str((self.MIFAREC3 & 0x04) >> 2)
		self.MIFAREblock3AC= str((self.MIFAREC1 & 0x08) >> 3) + str((self.MIFAREC2 & 0x08) >> 3) + str((self.MIFAREC3 & 0x08) >> 3)
		self.MIFAREkeyB= data[20:32]
	def MIFAREvb(self,data):
		"Set variables from standard MIFARE value block"
		self.MIFAREvalue= data[0:4]
		self.MIFAREvalueinv= data[4:8]
		self.MIFAREvalue2= data[8:12]
		self.MIFAREaddr= data[12]
		self.MIFAREaddrinv= data[13]
		self.MIFAREaddr2= data[14]
		self.MIFAREaddrinv2= data[15]
	def MRPmrzl(self,data):
		"Set variables from Machine Readable Zone (Lower)"
		self.MRPnumber= data[0:9]
		self.MRPnumbercd= data[9]
		self.MRPnationality= data[10:13]
		self.MRPdob= data[13:19]
		self.MRPdobcd= data[19]
		self.MRPsex= data[20]
		self.MRPexpiry= data[21:27]
		self.MRPexpirycd= data[27]
		self.MRPoptional= data[28:42]
		self.MRPoptionalcd= data[42]
		self.MRPcompsoitecd= data[43]
	def BitReverse(self,data):
		"Reverse bits - MSB to LSB"
		output= ''
		for y in reversed(range(len(data))):
			outchr= ''
			print y
			for x in range(8):
				outchr += str(ord(data[y]) >> x & 1)
			print outchr
			output += str(chr(int(outchr,2)))
		print len(output)
		return output
	def HexReverse(self,data):
		"Reverse HEX characters"
		output= ''
		for y in reversed(range(len(data))):
			output += data[y]
		return output
	def NibbleReverse(self,data):
		"Reverse Nibbles"
		output= ''
		for y in range(len(data)):
			leftnibble= ''
			rightnibble= ''
			for x in range(4):
				leftnibble += str(ord(data[y]) >> x & 1)
			for x in range(4,8):
				rightnibble += str(ord(data[y]) >> x & 1)
			output += str(chr(int(rightnibble + leftnibble,2)))
		return output
	def ToHex(self,data):
        	string= ''
        	for x in range(len(data)):
                	string += '%02x' % ord(data[x])
		return string
	def HexPrint(self,data):
        	print self.ToHex(data)
	def ToBinary(self,string):
        	output= ''
        	x= 0
        	while x < len(string):
                	output += chr(int(string[x:x + 2],16))
                	x += 2
        	return output
	def DESParity(self,data):
        	adjusted= ''
        	for x in range(len(data)):
                	y= ord(data[x]) & 0xfe
                	parity= 0
                	for z in range(8):
                        	parity += y >>  z & 1
                	adjusted += chr(y + (not parity % 2))
        	return adjusted
	def EM4x05ID(self,data):
		"Decode EM4x05 ID"
        	out= self.HexReverse(data)
        	hexout= self.ToHex(self.NibbleReverse(self.ToBinary(out)))
		# Application ID
        	self.EM4x05APP= hexout[:4]
		# Country Code
        	ccode= hexout[4:7]
        	self.EM4x05CCODE= int(ccode,16) >> 2
		# National ID
        	natid= hexout[7:16]
        	self.EM4x05NID= int(natid,16)
