#  RFIDIOt.py - RFID IO tools for python
# -*- coding: iso-8859-15 -*-
# 
#  Adam Laurie <adam@algroup.co.uk>
#  http://rfidiot.org/
# 
#  This code is copyright (c) Adam Laurie, 2006, All rights reserved.
#  For non-commercial use only, the following terms apply - for all other
#  uses, please contact the author:
#
#    This code is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This code is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#

# use Psyco compiler to speed things up if available
try:
	import psyco
	psyco.profile(0.01)
        psyco.full()
except ImportError:
        pass

import serial
import string
import os
from Crypto.Hash import SHA
from Crypto.Cipher import DES3
from Crypto.Cipher import DES
from operator import *

MASK_CCITT = 0x1021 # CRC-CCITT mask (ISO 3309, used in X25, HDLC)
MASK_11785 = 0x8408
MASK_CRC16 = 0xA001 # CRC16 mask (used in ARC files)


class rfidiot:
	"RFIDIOt - RFID I/O tools - http://rfidiot.org"
	#
	# open reader port
	#
	readertype= ''
	def __init__(self,reader,port,baud, to):

		self.readertype= reader
		# frosch always starts at 9600 baud - need to add code to test for selected rate and
		# switch if required. 
		self.ser = serial.Serial(port, baud, timeout=to)
		self.ser.readline()
		self.ser.flushInput()
		self.ser.flushOutput()
	#
	# variables
	#
	# VERSION: RFIDIOt.py version number
	# errorcode: 1 letter errorcode returned by the reader
	#
	# MIFAREdata: ASCII HEX representation of data block after successful read
	# MIFAREbinary: data block converted back to binary
	# MIFAREBLOCKLEN: constant ASCII HEX block length
	# MIFAREVALUELEN: constant ASCII HEX value length
	# MIFAREserialnumber: Unique ID (UID) of card
	# MIFAREkeyA: KEYA from key block (will always be 000000000000)
	# MIFAREkeyB: KEYB from key block
	# MIFAREaccessconditions: access conditions field from Key Block
	# MIFAREC1: Access conditions bitfield C1
	# MIFAREC2: Access conditions bitfield C2
	# MIFAREC3: Access conditions bitfield C3
	# MIFAREblock0AC: Block 0 Access Conditions
	# MIFAREblock1AC: Block 1 Access Conditions
	# MIFAREblock2AC: Block 2 Access Conditions
	# MIFAREblock3AC: Block 3 Access Conditions
	# MIFAREACKB: Human readable Key Block Access Conditions
	# MIFAREACDB: Human readable Data Block Access ConditionsA
	#
	# MRPmrzu: Machine Readable Passport - Machine Readable Zone - Upper
	# MRPmrzl Machine Readable Passport - Machine Readable Zone - Lower
	VERSION= '0.1n'
	# Reader types
	READER_ACG= 0x01
	READER_FROSCH= 0x02
	READER_DEMOTAG= 0x03
	errorcode= ''
	binary= ''
	data= ''
	tagtype= ''
	MIFAREdata=''
	MIFAREbinary=''
	MIFAREBLOCKLEN=32
	MIFAREVALUELEN=8
	MIFAREserialnumber= ''
	MIFAREcheckbyte= ''
	MIFAREmanufacturerdata= ''
	MIFAREkeyA= ''
	MIFAREkeyB= ''
	MIFAREaccessconditions= ''
	MIFAREaccessconditionsuserbyte= ' '
	MIFAREC1= 0
	MIFAREC2= 0
	MIFAREC3= 0
	MIFAREblock0AC= ''
	MIFAREblock1AC= ''
	MIFAREblock2AC= ''
	MIFAREblock3AC= ''
	MIFAREACKB= {'000':'Write KeyA: KEYA, Read Access bits: KEYA, Write Access bits: NONE, Read KeyB: KEYA, Write KeyB: KEYA (KEYB readable)',\
		     '010':'Write KeyA: NONE, Read Access bits: KEYA, Write Access bits: NONE, Read KeyB: KEYA, Write KeyB: NONE (KEYB readable)',\
		     '100':'Write KeyA: KEYB, Read Access bits: KEYA/B, Write Access bits: NONE, Read KeyB: NONE, Write KeyB: KEYB',\
		     '110':'Write KeyA: NONE, Read Access bits: KEYA/B, Write Access bits: NONE, Read KeyB: NONE, Write KeyB: NONE',\
		     '001':'Write KeyA: KEYA, Read Access bits: KEYA, Write Access bits: KEYA, Read KeyB: KEYA, Write KeyB: KEYA (KEYB readable, transport configuration)',\
		     '011':'Write KeyA: KEYB, Read Access bits: KEYA/B, Write Access bits: KEYB, Read KeyB: NONE, Write KeyB: KEYB',\
		     '101':'Write KeyA: NONE, Read Access bits: KEYA/B, Write Access bits: KEYB, Read KeyB: NONE, Write KeyB: NONE',\
		     '111':'Write KeyA: NONE, Read Access bits: KEYA/B, Write Access bits: NONE, Read KeyB: NONE, Write KeyB: NONE'}
	MIFAREACDB= {'000':'Read: KEYA/B, Write: KEYA/B, Increment: KEYA/B, Decrement/Transfer/Restore: KEYA/B (transport configuration)',\
		     '010':'Read: KEYA/B, Write: NONE, Increment: NONE, Decrement/Transfer/Restore: NONE',\
		     '100':'Read: KEYA/B, Write: KEYB, Increment: NONE, Decrement/Transfer/Restore: NONE',\
		     '110':'Read: KEYA/B, Write: KEYB, Increment: KEYB, Decrement/Transfer/Restore: KEYA/B',\
		     '001':'Read: KEYA/B, Write: NONE, Increment: NONE, Decrement/Transfer/Restore: KEYA/B',\
		     '011':'Read: KEYB, Write: KEYB, Increment: NONE, Decrement/Transfer/Restore: NONE',\
		     '101':'Read: KEYB, Write: NONE, Increment: NONE, Decrement/Transfer/Restore: NONE',\
		     '111':'Read: NONE, Write: NONE, Increment: NONE, Decrement/Transfer/Restore: NONE'}
	LFXTags= {'U':'EM 4x02 (Unique)   ',\
		  'Z':'EM 4x05 (ISO FDX-B)',\
		  'T':'EM 4x50           ',\
		  'h':'Hitag 1 / Hitag S ',\
		  'H':'Hitag 2           ',\
		  'Q':'Q5',\
		  'R':'TI-RFID Systems',\
		  'N':'No TAG present!'}
	# number of data blocks for each tag type (including block 0)
	LFXTagBlocks= {'U':0,\
		  'Z':0,\
		  'T':34,\
		  'h':64,\
		  'H':8,\
		  'Q':8,\
		  'R':18,\
		  'N':0}
	ALL= 'a'
	EM4x02= 'U'
	EM4x05= 'Z'
	Q5= 'Q'
	HITAG1= 'h'
	HITAG2= 'H'
	HITAG2_TRANSPORT_RWD='4D494B52'
	HITAG2_TRANSPORT_TAG='AA4854'
	HITAG2_TRANSPORT_HIGH='4F4E'
	HITAG2_PUBLIC_A= '02'
	HITAG2_PUBLIC_B= '00'
	HITAG2_PUBLIC_C= '04'
	HITAG2_PASSWORD= '06'
	HITAG2_CRYPTO= '0e'
	ISOTags= {'a':'ISO 14443 Type A  ',\
		  'b':'ISO 14443 Type B  ',\
		  'd':'ICODE UID         ',\
		  'e':'ICODE EPC         ',\
		  'i':'ICODE             ',\
		  's':'SR176             ',\
		  'v':'ISO 15693         '}
	ISOTagsA= {'t':'All Supported Tags'}
	ISO15693= 'v'
	# ISO 15693 Manufacturer codes (Listed in ISO/IEC 7816-6)
	ISO15693Manufacturer= {'02':'ST Microelectronics',\
			       '04':'Philips Semiconductors',\
			       '05':'Infineon Technologies AG',\
			       '07':'Texas Instrument',\
			       '16':'EM Microelectronic-Marin SA'}
	ISOAPDU=  {'ERASE BINARY':'0E',\
		   'VERIFY':'20',\
                   'MANAGE_CHANNEL':'70',\
                   'EXTERNAL_AUTHENTICATE':'82',\
                   'GET_CHALLENGE':'84',\
                   'INTERNAL_AUTHENTICATE':'88',\
                   'SELECT_FILE':'A4',\
                   'READ_BINARY':'B0',\
                   'READ_RECORD(S)':'B2',\
                   'GET_RESPONSE':'C0',\
                   'ENVELOPE':'C2',\
                   'GET_DATA':'CA',\
                   'WRITE_BINARY':'D0',\
                   'WRITE_RECORD':'D2',\
                   'UPDATE_BINARY':'D6',\
                   'PUT_DATA':'DA',\
                   'UPDATE_DATA':'DC',\
                   'APPEND_RECORD':'E2'}
	ISO7816ErrorCodes= {'9000':'No further qualification',\
			    '61':'SW2 indicates the number of response bytes still available',\
			    '6200':'No information given',\
			    '6281':'Part of returned data may be corrupted',\
			    '6282':'End of file/record reached before reading Le bytes',\
			    '6283':'Selected file invalidated',\
			    '6284':'FCI not formatted according to ISO7816-4 section 5.1.5',\
			    '6300':'No information given',\
			    '6381':'File filled up by the last write',\
			    '63C':'Counter provided by X (valued from 0 to 15) (exact meaning depending on the command)',\
			    '64':'State of non-volatile memory unchanged (SW2=00, other values are RFU)',\
			    '6500':'No information given',\
			    '6581':'Memory failure',\
			    '66':'Reserved for security-related issues (not defined in this part of ISO/IEC 7816)',\
			    '6700':'Wrong length',\
			    '6800':'No information given',\
			    '6881':'Logical channel not supported',\
			    '6882':'Secure messaging not supported',\
			    '6900':'No information given',\
			    '6981':'Command incompatible with file structure',\
			    '6982':'Security status not satisfied',\
			    '6983':'Authentication method blocked',\
			    '6984':'Referenced data invalidated',\
			    '6985':'Conditions of use not satisfied',\
			    '6986':'Command not allowed (no current EF)',\
			    '6987':'Expected SM data objects missing',\
			    '6988':'SM data objects incorrect',\
			    '6A00':'No information given',\
			    '6A80':'Incorrect parameters in the data field',\
			    '6A81':'Function not supported',\
			    '6A82':'File not found',\
			    '6A83':'Record not found',\
			    '6A84':'Not enough memory space in the file',\
			    '6A85':'Lc inconsistent with TLV structure',\
			    '6A86':'Incorrect parameters P1-P2',\
			    '6A87':'Lc inconsistent with P1-P2',\
			    '6A88':'Referenced data not found',\
			    '6B00':'Wrong parameter(s) P1-P2',\
			    '6C':'Wrong length Le: SW2 indicates the exact length',\
			    '6D00':'Instruction code not supported or invalid',\
			    '6E00':'Class not supported',\
			    '6F00':'No precise diagnosis'}
	# ISO-3166 alpha country codes
	ISO3166CountryCodesAlpha= { "ABW":"Aruba",\
				 "AFG":"Afghanistan",\
				 "AGO":"Angola",\
				 "AIA":"Anguilla",\
				 "ALA":"Åland Islands",\
				 "ALB":"Albania",\
				 "AND":"Andorra",\
				 "ANT":"Netherlands Antilles",\
				 "ARE":"United Arab Emirates",\
				 "ARG":"Argentina",\
				 "ARM":"Armenia",\
				 "ASM":"American Samoa",\
				 "ATA":"Antarctica",\
				 "ATF":"French Southern Territories",\
				 "ATG":"Antigua and Barbuda",\
				 "AUS":"Australia",\
				 "AUT":"Austria",\
				 "AZE":"Azerbaijan",\
				 "BDI":"Burundi",\
				 "BEL":"Belgium",\
				 "BEN":"Benin",\
				 "BFA":"Burkina Faso",\
				 "BGD":"Bangladesh",\
				 "BGR":"Bulgaria",\
				 "BHR":"Bahrain",\
				 "BHS":"Bahamas",\
				 "BIH":"Bosnia and Herzegovina",\
				 "BLR":"Belarus",\
				 "BLZ":"Belize",\
				 "BMU":"Bermuda",\
				 "BOL":"Bolivia",\
				 "BRA":"Brazil",\
				 "BRB":"Barbados",\
				 "BRN":"Brunei Darussalam",\
				 "BTN":"Bhutan",\
				 "BVT":"Bouvet Island",\
				 "BWA":"Botswana",\
				 "CAF":"Central African Republic",\
				 "CAN":"Canada",\
				 "CCK":"Cocos (Keeling) Islands",\
				 "CHE":"Switzerland",\
				 "CHL":"Chile",\
				 "CHN":"China",\
				 "CIV":"Côte d'Ivoire",\
				 "CMR":"Cameroon",\
				 "COD":"Congo, the Democratic Republic of the",\
				 "COG":"Congo",\
				 "COK":"Cook Islands",\
				 "COL":"Colombia",\
				 "COM":"Comoros",\
				 "CPV":"Cape Verde",\
				 "CRI":"Costa Rica",\
				 "CUB":"Cuba",\
				 "CXR":"Christmas Island",\
				 "CYM":"Cayman Islands",\
				 "CYP":"Cyprus",\
				 "CZE":"Czech Republic",\
				 "DEU":"Germany",\
				 "DJI":"Djibouti",\
				 "DMA":"Dominica",\
				 "DNK":"Denmark",\
				 "DOM":"Dominican Republic",\
				 "DZA":"Algeria",\
				 "ECU":"Ecuador",\
				 "EGY":"Egypt",\
				 "ERI":"Eritrea",\
				 "ESH":"Western Sahara",\
				 "ESP":"Spain",\
				 "EST":"Estonia",\
				 "ETH":"Ethiopia",\
				 "FIN":"Finland",\
				 "FJI":"Fiji",\
				 "FLK":"Falkland Islands (Malvinas)",\
				 "FRA":"France",\
				 "FRO":"Faroe Islands",\
				 "FSM":"Micronesia, Federated States of",\
				 "GAB":"Gabon",\
				 "GBR":"United Kingdom",\
				 "GEO":"Georgia",\
				 "GGY":"Guernsey",\
				 "GHA":"Ghana",\
				 "GIB":"Gibraltar",\
				 "GIN":"Guinea",\
				 "GLP":"Guadeloupe",\
				 "GMB":"Gambia",\
				 "GNB":"Guinea-Bissau",\
				 "GNQ":"Equatorial Guinea",\
				 "GRC":"Greece",\
				 "GRD":"Grenada",\
				 "GRL":"Greenland",\
				 "GTM":"Guatemala",\
				 "GUF":"French Guiana",\
				 "GUM":"Guam",\
				 "GUY":"Guyana",\
				 "HKG":"Hong Kong",\
				 "HMD":"Heard Island and McDonald Islands",\
				 "HND":"Honduras",\
				 "HRV":"Croatia",\
				 "HTI":"Haiti",\
				 "HUN":"Hungary",\
				 "IDN":"Indonesia",\
				 "IMN":"Isle of Man",\
				 "IND":"India",\
				 "IOT":"British Indian Ocean Territory",\
				 "IRL":"Ireland",\
				 "IRN":"Iran, Islamic Republic of",\
				 "IRQ":"Iraq",\
				 "ISL":"Iceland",\
				 "ISR":"Israel",\
				 "ITA":"Italy",\
				 "JAM":"Jamaica",\
				 "JEY":"Jersey",\
				 "JOR":"Jordan",\
				 "JPN":"Japan",\
				 "KAZ":"Kazakhstan",\
				 "KEN":"Kenya",\
				 "KGZ":"Kyrgyzstan",\
				 "KHM":"Cambodia",\
				 "KIR":"Kiribati",\
				 "KNA":"Saint Kitts and Nevis",\
				 "KOR":"Korea, Republic of",\
				 "KWT":"Kuwait",\
				 "LAO":"Lao People's Democratic Republic",\
				 "LBN":"Lebanon",\
				 "LBR":"Liberia",\
				 "LBY":"Libyan Arab Jamahiriya",\
				 "LCA":"Saint Lucia",\
				 "LIE":"Liechtenstein",\
				 "LKA":"Sri Lanka",\
				 "LSO":"Lesotho",\
				 "LTU":"Lithuania",\
				 "LUX":"Luxembourg",\
				 "LVA":"Latvia",\
				 "MAC":"Macao",\
				 "MAR":"Morocco",\
				 "MCO":"Monaco",\
				 "MDA":"Moldova, Republic of",\
				 "MDG":"Madagascar",\
				 "MDV":"Maldives",\
				 "MEX":"Mexico",\
				 "MHL":"Marshall Islands",\
				 "MKD":"Macedonia, the former Yugoslav Republic of",\
				 "MLI":"Mali",\
				 "MLT":"Malta",\
				 "MMR":"Myanmar",\
				 "MNE":"Montenegro",\
				 "MNG":"Mongolia",\
				 "MNP":"Northern Mariana Islands",\
				 "MOZ":"Mozambique",\
				 "MRT":"Mauritania",\
				 "MSR":"Montserrat",\
				 "MTQ":"Martinique",\
				 "MUS":"Mauritius",\
				 "MWI":"Malawi",\
				 "MYS":"Malaysia",\
				 "MYT":"Mayotte",\
				 "NAM":"Namibia",\
				 "NCL":"New Caledonia",\
				 "NER":"Niger",\
				 "NFK":"Norfolk Island",\
				 "NGA":"Nigeria",\
				 "NIC":"Nicaragua",\
				 "NIU":"Niue",\
				 "NLD":"Netherlands",\
				 "NOR":"Norway",\
				 "NPL":"Nepal",\
				 "NRU":"Nauru",\
				 "NZL":"New Zealand",\
				 "OMN":"Oman",\
				 "PAK":"Pakistan",\
				 "PAN":"Panama",\
				 "PCN":"Pitcairn",\
				 "PER":"Peru",\
				 "PHL":"Philippines",\
				 "PLW":"Palau",\
				 "PNG":"Papua New Guinea",\
				 "POL":"Poland",\
				 "PRI":"Puerto Rico",\
				 "PRK":"Korea, Democratic People's Republic of",\
				 "PRT":"Portugal",\
				 "PRY":"Paraguay",\
				 "PSE":"Palestinian Territory, Occupied",\
				 "PYF":"French Polynesia",\
				 "QAT":"Qatar",\
				 "REU":"Réunion",\
				 "ROU":"Romania",\
				 "RUS":"Russian Federation",\
				 "RWA":"Rwanda",\
				 "SAU":"Saudi Arabia",\
				 "SDN":"Sudan",\
				 "SEN":"Senegal",\
				 "SGP":"Singapore",\
				 "SGS":"South Georgia and the South Sandwich Islands",\
				 "SHN":"Saint Helena",\
				 "SJM":"Svalbard and Jan Mayen",\
				 "SLB":"Solomon Islands",\
				 "SLE":"Sierra Leone",\
				 "SLV":"El Salvador",\
				 "SMR":"San Marino",\
				 "SOM":"Somalia",\
				 "SPM":"Saint Pierre and Miquelon",\
				 "SRB":"Serbia",\
				 "STP":"Sao Tome and Principe",\
				 "SUR":"Suriname",\
				 "SVK":"Slovakia",\
				 "SVN":"Slovenia",\
				 "SWE":"Sweden",\
				 "SWZ":"Swaziland",\
				 "SYC":"Seychelles",\
				 "SYR":"Syrian Arab Republic",\
				 "TCA":"Turks and Caicos Islands",\
				 "TCD":"Chad",\
				 "TGO":"Togo",\
				 "THA":"Thailand",\
				 "TJK":"Tajikistan",\
				 "TKL":"Tokelau",\
				 "TKM":"Turkmenistan",\
				 "TLS":"Timor-Leste",\
				 "TON":"Tonga",\
				 "TTO":"Trinidad and Tobago",\
				 "TUN":"Tunisia",\
				 "TUR":"Turkey",\
				 "TUV":"Tuvalu",\
				 "TWN":"Taiwan, Province of China",\
				 "TZA":"Tanzania, United Republic of",\
				 "UGA":"Uganda",\
				 "UKR":"Ukraine",\
				 "UMI":"United States Minor Outlying Islands",\
				 "URY":"Uruguay",\
				 "USA":"United States",\
				 "UZB":"Uzbekistan",\
				 "VAT":"Holy See (Vatican City State)",\
				 "VCT":"Saint Vincent and the Grenadines",\
				 "VEN":"Venezuela",\
				 "VGB":"Virgin Islands, British",\
				 "VIR":"Virgin Islands, U.S.",\
				 "VNM":"Viet Nam",\
				 "VUT":"Vanuatu",\
				 "WLF":"Wallis and Futuna",\
				 "WSM":"Samoa",\
				 "YEM":"Yemen",\
				 "ZAF":"South Africa",\
				 "ZMB":"Zambia",\
				 "ZWE":"Zimbabwe"}
	# combined ISO-3166 country and icar.org manufacturer codes
	ISO3166CountryCodes= {'004':'Afghanistan',\
			      '248':'Åland Islands',\
			      '008':'Albania',\
			      '012':'Algeria',\
			      '016':'American Samoa',\
			      '020':'Andorra',\
			      '024':'Angola',\
			      '660':'Anguilla',\
			      '010':'Antarctica',\
			      '028':'Antigua and Barbuda',\
			      '032':'Argentina',\
			      '051':'Armenia',\
			      '533':'Aruba',\
			      '036':'Australia',\
			      '040':'Austria',\
			      '031':'Azerbaijan',\
			      '044':'Bahamas',\
			      '048':'Bahrain',\
			      '050':'Bangladesh',\
			      '052':'Barbados',\
			      '112':'Belarus',\
			      '056':'Belgium',\
			      '084':'Belize',\
			      '204':'Benin',\
			      '060':'Bermuda',\
			      '064':'Bhutan',\
			      '068':'Bolivia',\
			      '070':'Bosnia and Herzegovina',\
			      '072':'Botswana',\
			      '074':'Bouvet Island',\
			      '076':'Brazil',\
			      '086':'British Indian Ocean Territory',\
			      '096':'Brunei Darussalam',\
			      '100':'Bulgaria',\
			      '854':'Burkina Faso',\
			      '108':'Burundi',\
			      '116':'Cambodia',\
			      '120':'Cameroon',\
			      '124':'Canada',\
			      '132':'Cape Verde',\
			      '136':'Cayman Islands',\
			      '140':'Central African Republic',\
			      '148':'Chad',\
			      '152':'Chile',\
			      '156':'China',\
			      '162':'Christmas Island',\
			      '166':'Cocos (Keeling) Islands',\
			      '170':'Colombia',\
			      '174':'Comoros',\
			      '178':'Congo',\
			      '180':'Congo, the Democratic Republic of the',\
			      '184':'Cook Islands',\
			      '188':'Costa Rica',\
			      '384':'Côte d\'Ivoire',\
			      '191':'Croatia',\
			      '192':'Cuba',\
			      '196':'Cyprus',\
			      '203':'Czech Republic',\
			      '208':'Denmark',\
			      '262':'Djibouti',\
			      '212':'Dominica',\
			      '214':'Dominican Republic',\
			      '218':'Ecuador',\
			      '818':'Egypt',\
			      '222':'El Salvador',\
			      '226':'Equatorial Guinea',\
			      '232':'Eritrea',\
			      '233':'Estonia',\
			      '231':'Ethiopia',\
			      '238':'Falkland Islands (Malvinas)',\
			      '234':'Faroe Islands',\
			      '242':'Fiji',\
			      '246':'Finland',\
			      '250':'France',\
			      '254':'French Guiana',\
			      '258':'French Polynesia',\
			      '260':'French Southern Territories',\
			      '266':'Gabon',\
			      '270':'Gambia',\
			      '268':'Georgia',\
			      '276':'Germany',\
			      '288':'Ghana',\
			      '292':'Gibraltar',\
			      '300':'Greece',\
			      '304':'Greenland',\
			      '308':'Grenada',\
			      '312':'Guadeloupe',\
			      '316':'Guam',\
			      '320':'Guatemala',\
			      '831':'Guernsey',\
			      '324':'Guinea',\
			      '624':'Guinea-Bissau',\
			      '328':'Guyana',\
			      '332':'Haiti',\
			      '334':'Heard Island and McDonald Islands',\
			      '336':'Holy See (Vatican City State)',\
			      '340':'Honduras',\
			      '344':'Hong Kong',\
			      '348':'Hungary',\
			      '352':'Iceland',\
			      '356':'India',\
			      '360':'Indonesia',\
			      '364':'Iran, Islamic Republic of',\
			      '368':'Iraq',\
			      '372':'Ireland',\
			      '833':'Isle of Man',\
			      '376':'Israel',\
			      '380':'Italy',\
			      '388':'Jamaica',\
			      '392':'Japan',\
			      '832':'Jersey',\
			      '400':'Jordan',\
			      '398':'Kazakhstan',\
			      '404':'Kenya',\
			      '296':'Kiribati',\
			      '408':'Korea, Democratic People\'s Republic of',\
			      '410':'Korea, Republic of',\
			      '414':'Kuwait',\
			      '417':'Kyrgyzstan',\
			      '418':'Lao People\'s Democratic Republic',\
			      '428':'Latvia',\
			      '422':'Lebanon',\
			      '426':'Lesotho',\
			      '430':'Liberia',\
			      '434':'Libyan Arab Jamahiriya',\
			      '438':'Liechtenstein',\
			      '440':'Lithuania',\
			      '442':'Luxembourg',\
			      '446':'Macao',\
			      '807':'Macedonia, the former Yugoslav Republic of',\
			      '450':'Madagascar',\
			      '454':'Malawi',\
			      '458':'Malaysia',\
			      '462':'Maldives',\
			      '466':'Mali',\
			      '470':'Malta',\
			      '584':'Marshall Islands',\
			      '474':'Martinique',\
			      '478':'Mauritania',\
			      '480':'Mauritius',\
			      '175':'Mayotte',\
			      '484':'Mexico',\
			      '583':'Micronesia, Federated States of',\
			      '498':'Moldova, Republic of',\
			      '492':'Monaco',\
			      '496':'Mongolia',\
			      '499':'Montenegro',\
			      '500':'Montserrat',\
			      '504':'Morocco',\
			      '508':'Mozambique',\
			      '104':'Myanmar',\
			      '516':'Namibia',\
			      '520':'Nauru',\
			      '524':'Nepal',\
			      '528':'Netherlands',\
			      '530':'Netherlands Antilles',\
			      '540':'New Caledonia',\
			      '554':'New Zealand',\
			      '558':'Nicaragua',\
			      '562':'Niger',\
			      '566':'Nigeria',\
			      '570':'Niue',\
			      '574':'Norfolk Island',\
			      '580':'Northern Mariana Islands',\
			      '578':'Norway',\
			      '512':'Oman',\
			      '586':'Pakistan',\
			      '585':'Palau',\
			      '275':'Palestinian Territory, Occupied',\
			      '591':'Panama',\
			      '598':'Papua New Guinea',\
			      '600':'Paraguay',\
			      '604':'Peru',\
			      '608':'Philippines',\
			      '612':'Pitcairn',\
			      '616':'Poland',\
			      '620':'Portugal',\
			      '630':'Puerto Rico',\
			      '634':'Qatar',\
			      '638':'Réunion',\
			      '642':'Romania',\
			      '643':'Russian Federation',\
			      '646':'Rwanda',\
			      '654':'Saint Helena',\
			      '659':'Saint Kitts and Nevis',\
			      '662':'Saint Lucia',\
			      '666':'Saint Pierre and Miquelon',\
			      '670':'Saint Vincent and the Grenadines',\
			      '882':'Samoa',\
			      '674':'San Marino',\
			      '678':'Sao Tome and Principe',\
			      '682':'Saudi Arabia',\
			      '686':'Senegal',\
			      '688':'Serbia',\
			      '690':'Seychelles',\
			      '694':'Sierra Leone',\
			      '702':'Singapore',\
			      '703':'Slovakia',\
			      '705':'Slovenia',\
			      '090':'Solomon Islands',\
			      '706':'Somalia Somalia',\
			      '710':'South Africa',\
			      '239':'South Georgia and the South Sandwich Islands',\
			      '724':'Spain',\
			      '144':'Sri Lanka',\
			      '736':'Sudan',\
			      '740':'Suriname',\
			      '744':'Svalbard and Jan Mayen',\
			      '748':'Swaziland',\
			      '752':'Sweden',\
			      '756':'Switzerland',\
			      '760':'Syrian Arab Republic',\
			      '158':'Taiwan, Province of China',\
			      '762':'Tajikistan',\
			      '834':'Tanzania, United Republic of',\
			      '764':'Thailand',\
			      '626':'Timor-Leste',\
			      '768':'Togo',\
			      '772':'Tokelau',\
			      '776':'Tonga',\
			      '780':'Trinidad and Tobago',\
			      '788':'Tunisia',\
			      '792':'Turkey',\
			      '795':'Turkmenistan',\
			      '796':'Turks and Caicos Islands',\
			      '798':'Tuvalu',\
			      '800':'Uganda',\
			      '804':'Ukraine',\
			      '784':'United Arab Emirates',\
			      '826':'United Kingdom',\
			      '840':'United States',\
			      '581':'United States Minor Outlying Islands',\
			      '858':'Uruguay',\
			      '860':'Uzbekistan',\
			      '548':'Vanuatu',\
			      '862':'Venezuela',\
			      '704':'Viet Nam',\
			      '092':'Virgin Islands, British',\
			      '850':'Virgin Islands, U.S.',\
			      '876':'Wallis and Futuna',\
			      '732':'Western Sahara',\
			      '887':'Yemen',\
			      '894':'Zambia',\
			      '716':'Zimbabwe',\
			      '985':'MANUF: Destron Fearing / Digital Angel Corporation',\
			      '984':'MANUF: Nedap',\
			      '983':'MANUF: Texas Instruments',\
			      '982':'MANUF: Allflex',\
			      '981':'MANUF: Datamars',\
			      '980':'MANUF: AGRIDENT BV',\
			      '979':'MANUF: Earlsmere I.D.',\
			      '978':'MANUF: IER SA',\
			      '977':'MANUF: Avid',\
			      '976':'MANUF: Gemplus',\
			      '975':'MANUF: Sokymat',\
			      '974':'MANUF: Impro',\
			      '973':'MANUF: Fujihira',\
			      '972':'MANUF: Planet ID',\
			      '971':'MANUF: Alfa Laval Agri',\
			      '970':'MANUF: Amphenol',\
			      '969':'MANUF: Caisley',\
			      '968':'MANUF: AEG',\
			      '967':'MANUF: Rfdynamics',\
			      '966':'MANUF: PetCode',\
			      '965':'MANUF: 4D Technology Co. Ltd.',\
			      '964':'MANUF: Rumitag S.L.',\
			      '963':'MANUF: Korth Eletro Mecanica LTDA',\
			      '962':'MANUF: DigiTag A/S',\
			      '961':'MANUF: Mannings I.A.I.D.',\
			      '960':'MANUF: Chevillot',\
			      '959':'MANUF: Global ID Technologies',\
			      '958':'MANUF: Pet ID',\
			      '957':'MANUF: Innoceramics',\
			      '956':'MANUF: Trovan Ltd.',\
			      '955':'MANUF: Reseaumatique',\
			      '954':'MANUF: Ryeflex',\
			      '953':'MANUF: Cromasa',\
			      '952':'MANUF: JECTA',\
			      '951':'MANUF: Leader Products Pty Ltd',\
			      '950':'MANUF: SPLICE do Brasil Telecomunicacoes e Eletronica S.A.',\
			      '949':'MANUF: Y-Tex Corporation',\
			      '948':'MANUF: H. Hauptner und Richard Herberholz GmbH & Co. KG',\
			      '947':'MANUF: BELCAM. ID',\
			      '946':'MANUF: Advanced Ceramics Limited',\
			      '945':'MANUF: Business Inception Identification B.V.',\
			      '944':'MANUF: Net & Telligent SA',\
			      '943':'MANUF: E-Mark Technologie & Development',\
			      '942':'MANUF: Zee Tags',\
			      '941':'MANUF: Felixcan S.L.',\
			      '940':'MANUF: Shearwell Data Ltd.',\
			      '939':'MANUF: RealTrace',\
			      '938':'MANUF: INSVET',\
			      '937':'MANUF: ID & Traceback Systems AS',\
			      '936':'MANUF: CROVET, S.L.',\
			      '935':'MANUF: VeriLogik, Inc.',\
			      '900':'MANUF: Shared (see http://www.icar.org/manufacturer_codes.htm)',\
			      '1022':'UNREGISTERED MANUF: VeriChip Corporation'}
	DES_IV='\0\0\0\0\0\0\0\0'
	DES_PAD= [chr(0x80),chr(0),chr(0),chr(0),chr(0),chr(0),chr(0),chr(0)]
	KENC= '\0\0\0\1'
	KMAC= '\0\0\0\2'
	DO87= '870901'
	DO8E= '8E08'
	DO97= '9701'
	DO99= '99029000'
	#
	# frosch command set
	#
	#
	# Reader Stop
	FR_RWD_Stop_Cmd= chr(0xA6)
	# Reader Reset
	FR_RWD_HF_Reset= chr(0x68)
	# Reader Version
	FR_RWD_Get_Version= chr(0x56)
	# Hitag1 Get Serial Number
	FR_HT1_Get_Snr= chr(0x47)
	# Hitag1 Get Serial Number & set tag into Advanced Protocol Mode
	FR_HT1_Get_Snr_Adv= chr(0xA2)
	# Hitag1 Select Last Seen
	FR_HT1_Select_Last= chr(0x53)
	# Hitag1 Select Serial Number
	FR_HT1_Select_Snr= chr(0x53)
	# Hitag1 Read Page
	FR_HT1_Read_Page= chr(0x50)
	# Hitag2 Get Serial Number (password mode)
	FR_HT2_Get_Snr_PWD= chr(0x80) + chr(0x00)
	# Hitag2 Get Serial Number Reset (to reset for normal access when in public modes)
	FR_HT2_Get_Snr_Reset= chr(0x80)
	# Hitag2 Halt Selected
	FR_HT2_Halt_Selected= chr(0x81)
	# Hitag2 read page
	FR_HT2_Read_Page= chr(0x82)
	# Hitag2 Read Miro (Unique / Public Mode A)
	FR_HT2_Read_Miro= chr(0x4d)
	# Hitag2 Read Public B (FDX-B)
	FR_HT2_Read_PublicB= chr(0x9e)
	# Hitag2 Write Page
	FR_HT2_Write_Page= chr(0x84)
	#
	# frosch errors
	#
	FROSCH_Errors= { '00':'No Error',\
			 '02':'Error',\
			 '07':'No Error',\
			 'eb':'Antenna Overload',\
			 'f1':'EEPROM Read Protected',\
			 'f2':'EEPROM Write Protected',\
			 'f3':'EEPROM Wrong - Old Data',\
			 'f4':'EEPROM Error',\
			 'f5':'CryptoBlock not INIT',\
			 'f6':'Acknowledgement Error',\
			 'f9':'Authentication Error',\
			 'fa':'Incorrect Password TAG',\
			 'fb':'Incorrect Password RWD',\
			 'fc':'Timeout',\
			 'fd':'No Tag',\
			 'ff':'Serial port fail'}
	# 
	# frosch constants
	#
	FR_BAUD_RATE= {   9600:chr(0x01),\
		   	 14400:chr(0x02),\
		   	 19200:chr(0x03),\
		   	 38400:chr(0x04),\
		   	 57600:chr(0x05),\
		  	115200:chr(0x06)}
	FR_NO_ERROR= chr(0x00)
	FR_PLAIN= chr(0x00)
	FR_CRYPTO= chr(0x01)
	FR_TIMEOUT= 'fc'
	#
	# DemoTag command set
	#
	DT_SET_UID= 'u'
	#
	# DemoTag Errors
	#
	DT_ERROR= '?'
	#
	# local/informational functions
	#
	def info(self,caller):
		self.reset()
		print caller + ' (using RFIDIOt v' + self.VERSION + ')'
		self.version()
		print 'reader: ',
		if self.readertype == self.READER_ACG:
			print 'ACG ' + self.data,
			print ' (serial no: ' + self.id() + ')'
		if self.readertype == self.READER_FROSCH:
			print 'Frosch ' + self.ToBinary(self.data[:16]) + ' / ' + self.ToBinary(self.data[16:32]),
			print ' (serial no: ' + self.data[32:54] + ')'

	#
	# reader functions
	#
        def reset(self):
		if self.readertype == self.READER_ACG:
			# send a select to stop just in case it's in multi-select mode
			self.ser.write('s')
			self.ser.readline()
			self.ser.flushInput()
			self.ser.flushOutput()
			# now send a reset and read response
			self.ser.write('x')
			self.ser.readline()
			# now send a select and read remaining lines
			self.ser.write('s')
			self.ser.readline()
			self.ser.flushInput()
			self.ser.flushOutput()
			return True
		if self.readertype == self.READER_FROSCH:
			if self.frosch(self.FR_RWD_HF_Reset,''):
				return True
			else:
				print self.FROSCH_Errors[self.errorcode]
				os._exit(False)
	def version(self):
		if self.readertype == self.READER_ACG:
			self.ser.write('v')
			self.data= self.ser.readline()[:-2]
			return True
		if self.readertype == self.READER_FROSCH:
			if self.frosch(self.FR_RWD_Get_Version,''):
				return True
			else:
				print self.FROSCH_Errors[self.errorcode]
				os._exit(False)
	def id(self):
		return self.readEEPROM(0)[:2] + self.readEEPROM(1)[:2] + self.readEEPROM(2)[:2] + self.readEEPROM(3)[:2]
	def station(self):
		return self.readEEPROM(0x0a)[:2]
	def PCON(self):
		return self.readEEPROM(0x0b)[:2]
	def PCON2(self):
		return self.readEEPROM(0x13)[:2]
	def PCON3(self):
		return self.readEEPROM(0x1b)[:2]
	def BAUD(self):
		return self.readEEPROM(0x0c)[:2]
	def CGT(self):
		return self.readEEPROM(0x0d)[:2]
	def opmode(self):
		return self.readEEPROM(0x0e)[:2]
	def SST(self):
		return self.readEEPROM(0x0f)[:2]
	def ROT(self):
		return self.readEEPROM(0x14)[:2]
	def RRT(self):
		return self.readEEPROM(0x15)[:2]
	def AFI(self):
		return self.readEEPROM(0x16)[:2]
	def STOa(self):
		return self.readEEPROM(0x17)[:2]
	def STOb(self):
		return self.readEEPROM(0x18)[:2]
	def STOs(self):
		return self.readEEPROM(0x19)[:2]
	def readEEPROM(self,byte):
		self.ser.write('rp%02x' % byte)
		return self.ser.readline()[:2]
	def writeEEPROM(self,byte,value):
		self.ser.write('wp%02x%02x' % (byte,value))
		self.errorcode= self.ser.readline()[:-2]
		if eval(self.errorcode) == value:
			return True
		return False
	def settagtype(self,type):
		if self.readertype == self.READER_ACG:
			self.ser.write('o' + type)
			self.errorcode= self.ser.readline()[:-2]
			if self.errorcode == 'O' + string.upper(type):
				self.tagtype= type
				return True
		if self.readertype == self.READER_FROSCH:
			if type == self.EM4x02:
				return self.frosch(self.FR_HT2_Read_Miro,'')			
			if type == self.EM4x05:
				return self.frosch(self.FR_HT2_Read_PublicB,'')
		return False
	#
	# card functions
	#
	def waitfortag(self,message):
		print message
		while not self.select():
			# do nothing
			x= 0
		return True
	def select(self):
		# need to fix so ACG and FROSCH have same returns
		if self.readertype == self.READER_ACG:
			self.ser.write('s')
			self.data= self.ser.readline()[:-2]
			self.tagtype= self.data[:1]
			if self.tagtype == 'N':
				return False
			return True
		if self.readertype == self.READER_FROSCH:
			if self.frosch(self.FR_HT2_Get_Snr_PWD,''):
				# select retruns an extra byte on the serial number, so strip it
				self.data= self.data[:len(self.data) - 2]
				self.tagtype= self.HITAG2
				return True
			if self.frosch(self.FR_HT1_Get_Snr,''):
				# select returns an extra byte on the serial number, so strip it
				# and preserve for after select command
				serialno= self.data[:len(self.data) - 2]
				if self.frosch(self.FR_HT1_Select_Last,''):
					self.tagtype= self.HITAG1
					self.data= serialno
					return True
			return False
	def h2publicselect(self):
		"select Hitag2 from Public Mode A/B/C"
		if self.readertype == self.READER_FROSCH:
			if (self.frosch(self.FR_HT2_Get_Snr_Reset,self.FR_PLAIN + 'M')):
				self.tagtype= self.HITAG2
				self.data= self.data[:8]
				return True
		return False
	def hsselect(self,speed):
		"high speed select - 106 (speed= 01), 212 (speed= 02), 424 (speed= 04) or 848 (speed= 08) kBaud"
		self.ser.write('h'+speed)
		return self.ser.readline()[:-2]
	def iso_7816_fail(self,code):
		"print 7816 failure code and exit"
		if code == 'N':
			print "Application not implemented!"
			os._exit(False)
		print "Failed - reason code " + code + " (" + self.ISO7816ErrorCodes[code] + ")"
		print
		os._exit(False)
	def send_apdu(self,option,pcb,cid,nad,cla,ins,p1,p2,lc,data,le):
		"send iso-1786-4 apdu"
		if not option:
			option= '0f'
		if not pcb:
			pcb= '02'
		if not cla:
			cla= '00'
		if not p1:
			p1= '00'
		if not p2:
			p2= '00'
		dlength= 5
		command= pcb+cla+self.ISOAPDU[ins]+p1+p2+lc+data+le
		dlength += len(data) / 2
		dlength += len(lc) / 2
		dlength += len(le) / 2
#		print 'sending: ' + 't' + '%02x' % dlength + option + command
		self.ser.write('t' + '%02x' % dlength + option + command)
		# need check for 'le' length as well
		ret= self.ser.readline()[:-2] 
		self.errorcode= ret[len(ret) - 4:len(ret)]
		return ret[4:len(ret) - 4]
#		if not len(ret) / 2 == int(ret[0:2],16) + 1:
#			return False
#		return ret[4:int(le,16) * 2 + 4]
	def login(self,sector,keytype,key):
		"login to specified sector - returns True if successful, False if failed. If failure is due to an error, 'errorcode' will be set." 
		if self.readertype == self.READER_ACG:
			if not sector == '':
				self.ser.write('l' + ('%02x' % sector) + keytype + key)
			else:
				self.ser.write('l' + keytype + key)
			if key == '':
				self.ser.write('\r')
			self.errorcode= self.ser.readline()[0]
			if self.errorcode == 'L':
				return True
			elif self.errorcode == 'X':
				self.errorcode= ''
			return False
		if self.readertype == self.READER_FROSCH:
			return self.frosch(self.FR_HTS_TagAuthent,'')
	def readblock(self,block):
		if self.readertype == self.READER_FROSCH:
			if self.tagtype == self.HITAG1:
				return(self.frosch(self.FR_HT1_Read_Page,self.FR_PLAIN + chr(block))) 	
			if self.tagtype == self.HITAG2:
				return(self.frosch(self.FR_HT2_Read_Page,chr(block))) 	
		if self.readertype == self.READER_ACG:
			self.ser.write('r%02x' % block)
			self.data= self.ser.readline()[:-2]
			self.binary= ''
			if len(self.data) == 1:
				self.errorcode= self.data
				self.data= ''
				return False
			count= 0
			while count * 2 < len(self.data):
				self.binary += chr(int(self.data[count * 2:(count * 2) + 2],16))
				count += 1
			return True	
	def readMIFAREblock(self,block):
		self.ser.write('r%02x' % block)
		self.MIFAREdata= self.ser.readline()[:-2]
		if len(self.MIFAREdata) != self.MIFAREBLOCKLEN:
			self.errorcode= self.MIFAREdata
			self.MIFAREdata= ''
			return False
		count= 0
		while count * 2 < len(self.MIFAREdata):
			self.MIFAREbinary += chr(int(self.MIFAREdata[count * 2:(count * 2) + 2],16))
			count += 1
		return True
	def readvalueblock(self,block):
		self.ser.write('rv%02x' % block)
		self.MIFAREdata= self.ser.readline()[:-2]
		if len(self.MIFAREdata) != self.MIFAREVALUELEN:
			self.errorcode= self.MIFAREdata
			self.MIFAREdata= ''
			return False
		count= 0
		while count * 2 < len(self.MIFAREdata):
			self.MIFAREbinary += chr(int(self.MIFAREdata[count * 2:(count * 2) + 2],16))
			count += 1
		return True
	def writeblock(self,block,data):
		if self.readertype == self.READER_FROSCH:
			#if self.tagtype == self.HITAG1:
			#	return(self.frosch(self.FR_HT1_Read_Page,self.FR_PLAIN + chr(block))) 	
			if self.tagtype == self.HITAG2:
				return(self.frosch(self.FR_HT2_Write_Page,chr(block) + self.ToBinary(data))) 	
		if self.readertype == self.READER_ACG:
			self.ser.write('w%02x%s' % (block,data))
			x= self.ser.readline()[:-2]
			if x == string.upper(data):
				self.errorcode= ''
				return True
			self.errorcode= x
			return False
	def writevalueblock(self,block,data):
		self.ser.write('wv%02x%s' % (block,data))
                x= self.ser.readline()[:-2]
                if x == string.upper(data):
                        self.errorcode= ''
                        return True
                self.errorcode= x
                return False
	def frosch(self,command,data):
		"send frosch commands with check digit"
		command += data
		commandlen= len(command)
		bcc= commandlen + 1
		for x in range(commandlen):
			bcc= xor(bcc,ord(command[x]))
		# send length + command + checkdigit
		#print 'Sending: ', 
		#self.HexPrint(chr(commandlen + 1) + command + chr(bcc))
		self.ser.write(chr(commandlen + 1) + command + chr(bcc))
		ret= ''
		# perform a blocking read - returned byte is number of chars still to read
		ret += self.ser.read(1)
		# if read times out, reset may be required for normal read mode
		if len(ret) == 0:
			if command == self.FR_HT2_Read_PublicB or command == self.FR_HT2_Read_Miro:
				self.frosch(self.FR_RWD_Stop_Cmd,'')
			self.errorcode= self.FR_TIMEOUT
			return False
		# now read the rest
		ret += self.ser.read(ord(ret[0]))
		#print 'ret: %d ' % len(ret),
		#self.HexPrint(ret)
		# check integrity of return
		bcc= 0
		for x in range(len(ret) - 1):
			bcc= xor(bcc,ord(ret[x]))
		if not bcc == ord(ret[x + 1]):
			print 'Frosch error! Checksum error: ',
			self.HexPrint(ret)
			os._exit(False)
		status= ret[1]
		if status == self.FR_NO_ERROR:
			self.errorcode= ''
			# for consistency with ACG, data is converted to printable hex before return
			self.data= self.ToHex(ret[2:len(ret) - 1])
			return True
		else :
			self.errorcode= self.ToHex(status)
			self.data= ''
			# reader may need resetting to normal read mode
			if command == self.FR_HT2_Read_PublicB or command == self.FR_HT2_Read_Miro:
				self.frosch(self.FR_RWD_Stop_Cmd,'')
			return False
	def demotag(self,command,data):
		"send DemoTag commands"
		if self.ser.write(command + data):
                	x= self.ser.readline()[:-2]
			if x == self.DT_ERROR:
				self.errorcode= x
				return False
			self.data= x
			return True
		return False
	#
	# data manipulation
	#
	def Parity(self,data,parity):
		# return parity bit to make odd or even as required
		myparity= 0
		for x in range(len(data)):
			myparity += int(data[x],2)
		myparity %= 2
		return xor(myparity,parity)
	def Unique64Bit(self,data):
		"convert binary ID to Unique formatted 64 bit data block"
		# standard header == 9 bits of '1'
		out= '111111111'
		# break output into 4 bit chunks and add parity
		colparity= [0,0,0,0]
		for x in range(0,len(data),4):
			parity= 0
			chunk= data[x:x+4]
			for y in range(4):
				parity += int(chunk[y],2)
				colparity[y] += int(chunk[y],2)
			out += chunk + '%s' % (int(parity) % 2)
		# add column parity
		for x in range(4):
			out += '%s' % (int(colparity[x]) % 2)
		# add stop bit
		out += '0'
		return out
	def UniqueToEM(self,data):
		"convert Unique ID to raw EM4x02 ID"
		# swap words
		tmp= ''
		for x in range(5):
			tmp += data[x * 2 + 1] + data[x * 2]
		# reverse bits
		return self.ToBinaryString(self.ToBinary(tmp))[::-1]
	def EMToUnique(self,data):
		"convert raw EM4x02 ID to Unique"
		return self.ToHex(self.BitReverse(self.ToBinary(data)))
	def HexToQ5(self,data):
		"conver human readable HEX to Q5 ID"
		return self.ToBinaryString(self.ToBinary(data))
	def crcccitt(self,data):
		crcvalue= 0x0000
		for x in range(len(data)):
			crcvalue= self.crc(crcvalue,data[x],MASK_CCITT)
		return crcvalue
	def crc(self, crc, data, mask=MASK_CRC16):
		for char in data:
			c = ord(char)
			c = c << 8
		for j in xrange(8):
			if (crc ^ c) & 0x8000:
				crc = (crc << 1) ^ mask
			else:
				crc = crc << 1
			c = c << 1
		return crc & 0xffff
	def crc16(self,data):
		crcValue=0x0000
		crc16tab = (0x0000, 0xC0C1, 0xC181, 0x0140, 0xC301, 0x03C0, 0x0280,
		0xC241, 0xC601, 0x06C0, 0x0780, 0xC741, 0x0500, 0xC5C1, 0xC481,
		0x0440, 0xCC01, 0x0CC0, 0x0D80, 0xCD41, 0x0F00, 0xCFC1, 0xCE81,
		0x0E40, 0x0A00, 0xCAC1, 0xCB81, 0x0B40, 0xC901, 0x09C0, 0x0880,
		0xC841, 0xD801, 0x18C0, 0x1980, 0xD941, 0x1B00, 0xDBC1, 0xDA81,
		0x1A40, 0x1E00, 0xDEC1, 0xDF81, 0x1F40, 0xDD01, 0x1DC0, 0x1C80,
		0xDC41, 0x1400, 0xD4C1, 0xD581, 0x1540, 0xD701, 0x17C0, 0x1680,
		0xD641, 0xD201, 0x12C0, 0x1380, 0xD341, 0x1100, 0xD1C1, 0xD081,
		0x1040, 0xF001, 0x30C0, 0x3180, 0xF141, 0x3300, 0xF3C1, 0xF281,
		0x3240, 0x3600, 0xF6C1, 0xF781, 0x3740, 0xF501, 0x35C0, 0x3480,
		0xF441, 0x3C00, 0xFCC1, 0xFD81, 0x3D40, 0xFF01, 0x3FC0, 0x3E80,
		0xFE41, 0xFA01, 0x3AC0, 0x3B80, 0xFB41, 0x3900, 0xF9C1, 0xF881,
		0x3840, 0x2800, 0xE8C1, 0xE981, 0x2940, 0xEB01, 0x2BC0, 0x2A80,
		0xEA41, 0xEE01, 0x2EC0, 0x2F80, 0xEF41, 0x2D00, 0xEDC1, 0xEC81,
		0x2C40, 0xE401, 0x24C0, 0x2580, 0xE541, 0x2700, 0xE7C1, 0xE681,
		0x2640, 0x2200, 0xE2C1, 0xE381, 0x2340, 0xE101, 0x21C0, 0x2080,
		0xE041, 0xA001, 0x60C0, 0x6180, 0xA141, 0x6300, 0xA3C1, 0xA281,
		0x6240, 0x6600, 0xA6C1, 0xA781, 0x6740, 0xA501, 0x65C0, 0x6480,
		0xA441, 0x6C00, 0xACC1, 0xAD81, 0x6D40, 0xAF01, 0x6FC0, 0x6E80,
		0xAE41, 0xAA01, 0x6AC0, 0x6B80, 0xAB41, 0x6900, 0xA9C1, 0xA881,
		0x6840, 0x7800, 0xB8C1, 0xB981, 0x7940, 0xBB01, 0x7BC0, 0x7A80,
		0xBA41, 0xBE01, 0x7EC0, 0x7F80, 0xBF41, 0x7D00, 0xBDC1, 0xBC81,
		0x7C40, 0xB401, 0x74C0, 0x7580, 0xB541, 0x7700, 0xB7C1, 0xB681,
		0x7640, 0x7200, 0xB2C1, 0xB381, 0x7340, 0xB101, 0x71C0, 0x7080,
		0xB041, 0x5000, 0x90C1, 0x9181, 0x5140, 0x9301, 0x53C0, 0x5280,
		0x9241, 0x9601, 0x56C0, 0x5780, 0x9741, 0x5500, 0x95C1, 0x9481,
		0x5440, 0x9C01, 0x5CC0, 0x5D80, 0x9D41, 0x5F00, 0x9FC1, 0x9E81,
		0x5E40, 0x5A00, 0x9AC1, 0x9B81, 0x5B40, 0x9901, 0x59C0, 0x5880,
		0x9841, 0x8801, 0x48C0, 0x4980, 0x8941, 0x4B00, 0x8BC1, 0x8A81,
		0x4A40, 0x4E00, 0x8EC1, 0x8F81, 0x4F40, 0x8D01, 0x4DC0, 0x4C80,
		0x8C41, 0x4400, 0x84C1, 0x8581, 0x4540, 0x8701, 0x47C0, 0x4680,
		0x8641, 0x8201, 0x42C0, 0x4380, 0x8341, 0x4100, 0x81C1, 0x8081,
		0x4040)
		for ch in data:
			tmp=crcValue^(ord(ch))
			crcValue=(crcValue>> 8)^crc16tab[(tmp & 0xff)]
		return crcValue
	def MIFAREmfb(self,data):
		"Set variables from standard MIFARE manufacturer block (block 0 sector 0)"
		self.MIFAREserialnumber= data[0:8]
		self.MIFAREcheckbyte= data[8:10]
		self.MIFAREmanufacturerdata= data[10:32]
	def MIFAREkb(self,data):
		"Set variables from standard MIFARE key block (trailing sector)"
		self.MIFAREkeyA= data[0:12]
		self.MIFAREaccessconditions= data[12:18]
		self.MIFAREaccessconditionsuserbyte= data[18:20]
		self.MIFAREC1= int(data[14:16],16) >> 4
		self.MIFAREC2= int(data[16:18],16) & 0x0f
		self.MIFAREC3= (int(data[16:18],16) & 0xf0) >> 4
		self.MIFAREblock0AC= str(self.MIFAREC1 & 0x01) + str(self.MIFAREC2 & 0x01) + str(self.MIFAREC3 & 0x01)
		self.MIFAREblock1AC= str((self.MIFAREC1 & 0x02) >> 1) + str((self.MIFAREC2 & 0x02) >> 1) + str((self.MIFAREC3 & 0x02) >> 1)
		self.MIFAREblock2AC= str((self.MIFAREC1 & 0x04) >> 2) + str((self.MIFAREC2 & 0x04) >> 2) + str((self.MIFAREC3 & 0x04) >> 2)
		self.MIFAREblock3AC= str((self.MIFAREC1 & 0x08) >> 3) + str((self.MIFAREC2 & 0x08) >> 3) + str((self.MIFAREC3 & 0x08) >> 3)
		self.MIFAREkeyB= data[20:32]
	def MIFAREvb(self,data):
		"Set variables from standard MIFARE value block"
		self.MIFAREvalue= data[0:4]
		self.MIFAREvalueinv= data[4:8]
		self.MIFAREvalue2= data[8:12]
		self.MIFAREaddr= data[12]
		self.MIFAREaddrinv= data[13]
		self.MIFAREaddr2= data[14]
		self.MIFAREaddrinv2= data[15]
	def MRPmrzl(self,data):
		"Set variables from Machine Readable Zone (Lower)"
		self.MRPnumber= data[0:9]
		self.MRPnumbercd= data[9]
		self.MRPnationality= data[10:13]
		self.MRPdob= data[13:19]
		self.MRPdobcd= data[19]
		self.MRPsex= data[20]
		self.MRPexpiry= data[21:27]
		self.MRPexpirycd= data[27]
		self.MRPoptional= data[28:42]
		self.MRPoptionalcd= data[42]
		self.MRPcompsoitecd= data[43]
	def BitReverse(self,data):
		"Reverse bits - MSB to LSB"
		output= ''
		for y in range(len(data)):
			outchr= ''
			for x in range(8):
				outchr += str(ord(data[y]) >> x & 1)
			output += str(chr(int(outchr,2)))
		return output
	def HexReverse(self,data):
		"Reverse HEX characters"
		output= ''
		for y in reversed(range(len(data))):
			output += data[y]
		return output
	def NibbleReverse(self,data):
		"Reverse Nibbles"
		output= ''
		for y in range(len(data)):
			leftnibble= ''
			rightnibble= ''
			for x in range(4):
				leftnibble += str(ord(data[y]) >> x & 1)
			for x in range(4,8):
				rightnibble += str(ord(data[y]) >> x & 1)
			output += str(chr(int(rightnibble + leftnibble,2)))
		return output
	def ToHex(self,data):
        	string= ''
        	for x in range(len(data)):
                	string += '%02x' % ord(data[x])
		return string
	def HexPrint(self,data):
        	print self.ToHex(data)
	def ToBinary(self,string):
        	output= ''
        	x= 0
        	while x < len(string):
                	output += chr(int(string[x:x + 2],16))
                	x += 2
        	return output
	def BinaryPrint(self,data):
		"print binary representation"
		print self.ToBinaryString(data)
	def ToBinaryString(self,data):
		output= ''
		string= self.ToHex(data)
		for x in range(0,len(string),2):
			for y in range(7,-1,-1):
				output += '%s' % (int(string[x:x+2],16) >> y & 1)
		return output
	def DESParity(self,data):
        	adjusted= ''
        	for x in range(len(data)):
                	y= ord(data[x]) & 0xfe
                	parity= 0
                	for z in range(8):
                        	parity += y >>  z & 1
                	adjusted += chr(y + (not parity % 2))
        	return adjusted
	def DESKey(self,seed,type,length):
		d= seed + type	
		kencsha= SHA.new(d)
		k= kencsha.digest()
		kp= self.DESParity(k)
		return(kp[:length])
	def PADBlock(self,block):
        	for x in range(8 - (len(block) % 8)):
                	block += self.DES_PAD[x]
		return block
	def DESMAC(self,message,key,ssc):
	        tdesa= DES.new(key[0:8],DES.MODE_ECB,self.DES_IV)
        	tdesb= DES.new(key[8:16],DES.MODE_ECB,self.DES_IV)
        	if(ssc):
                	mac= tdesa.encrypt(self.ToBinary(ssc))
        	else:
                	mac= self.DES_IV
		message += self.PADBlock('')
        	for y in range(len(message) / 8):
                	current= message[y * 8:(y * 8) + 8]
                	left= ''
                	right= ''
                	for x in range(len(mac)):
                        	left += '%02x' % ord(mac[x])
                        	right += '%02x' % ord(current[x])
                	machex= '%016x' % xor(int(left,16),int(right,16))
                	mac= self.ToBinary(machex)
                	mac= tdesa.encrypt(mac)
        	mac= tdesb.decrypt(mac)
        	mac= tdesa.encrypt(mac)
        	return mac
	def MACVerify(self,message,key):
		mess= self.ToBinary(message[:len(message)- 16])
		mac= self.DESMAC(mess,key,'')
		if not mac == self.ToBinary(message[len(message) -16:]):
			print 'MAC Error!'
			print 'Expected MAC: ', message[len(message) -16:]
			print 'Actual MAC:   ',
			self.HexPrint(mac)
			return(False)
		return(True)
	def SSCIncrement(self,ssc):
		out= int(self.ToHex(ssc),16) + 1
		return self.ToBinary("%016x" % out)
	def FDXBID(self,data):
		"Decode FDX-B ID"
        	out= self.HexReverse(data)
        	hexout= self.ToHex(self.NibbleReverse(self.ToBinary(out)))
		# Application ID
        	self.FDXBAPP= hexout[:4]
		# Country Code
        	ccode= hexout[4:7]
        	self.FDXBCCODE= int(ccode,16) >> 2
		# Human Readable CCODE
		if "%d" % self.FDXBCCODE in self.ISO3166CountryCodes:
			self.FDXBCCODEHR= self.ISO3166CountryCodes["%d" % self.FDXBCCODE]
		else:
			self.FDXBCCODEHR= 'Undefined - see http://www.icar.org/manufacturer_codes.htm'
		# National ID
        	natid= hexout[7:16]
        	self.FDXBNID= int(natid,16)
	def FDXBIDEncode(self,appid,ccode,natid):
		"Encode FDX-B ID"
		hexccode= "%03x" % (int(ccode,10) << 2)
		hexid= "%09x" % int(natid,10)
		rawid= appid + hexccode + hexid
		nibbleid= self.NibbleReverse(self.ToBinary(rawid))
		hexout= self.HexReverse(self.ToHex(nibbleid))
		return hexout
	def FDXBIDPrint(self,data):
		self.FDXBID(data)
        	print 'Application Identifier: ', self.FDXBAPP
        	print 'Country Code: ',
        	print self.FDXBCCODE,
        	print  "(" + self.FDXBCCODEHR + ")"
        	print 'National ID: ',
        	print self.FDXBNID
	def FDXBID128Bit(self,data):
		"generate raw 128 bit FDX-B data from FDX-B ID"
		idbin= self.ToBinaryString(self.ToBinary(data))
		# construct FDX-B encoded blocks
		out= ''
		# header is ten zeros and a '1'
		header= '00000000001'
		out += header
		# break id into 8 bit chunks with a trailing '1' on each
		for x in range(0,len(idbin),8):
			out += idbin[x:x+8] + '1'
		# add 16 CRC-CCITT error detection bits
		crc= '%04x' % self.crcccitt(self.ToBinary(data))
		crcbin= self.ToBinaryString(self.ToBinary(crc))
		# crc is transmitted LSB first with trailing '1's
		out += crcbin[0:8] + '1'
		out += crcbin[8:16] + '1'
		# add 3 sets of trailer bits (RFU)
		trailer= '000000001'
		for x in range(3):
			out += trailer
		return out
	def FDXBID128BitDecode(self,data):
		"convert raw 128 bit FDX-B data to FDX-B ID"
		#strip off header
		y= data[11:]
		#strip off trailing '1' from the first 8 9-bit groups
		out= ''
		for x in range(0,72,9):
			out += y[x:x+8]
		# ignore the rest - CRC etc.
		return '%016x' % int(out,2)	
